# jats-generator

Generate JATS XML file from journal article data.

Using objects defined in `elifearticle` as data structure, this library generates JATS XML output for a research article.

No article body content is supported yet, though this library can output at least the following:

- Journal metadata (journal title, publisher name, issn, volume)
- Article metadata (title, doi, pub date, history dates, abstract, copyright, subject categories, research organisms)
- Authors, editors, group authors, author affiliations, correspondence, conflicts of interest
- Permissions (copyright statement, copyright year, copyright holder, license)
- Datasets
- Funding data
- Ethics

One way to populate the `elifearticle` objects with data, as is used by eLife from CSV files, relies on using the `ejpcsvparser` library by using `generate.build_article_from_csv()`, resulting in an `Article` object. The `Article` can then be used as input to `generate.build_xml()` or `generate.build_xml_to_disk()` which will produce the JATS XML output.

A configparser object can be populated from the `jatsgenerator.cfg` file, where a different config section per journal name can be included, and building it using the `jatsgenerator.config` module, 

Some sample CSV data input and JATS XML output files can be found in the `tests/test_data/` folder, which are the basis for the automated tests.

## Requirements and install

a) Install from `pypi` package index

```
pip install jatsgenerator
```

b) Install locally

Clone the git repo

    git clone https://github.com/elifesciences/jats-generator.git

Create a python virtual environment and activate it

    ./install.sh

## Configuration

The `jatsgenerator.cfg` configuration file provided in this repository can be changed in order to write slightly different JATS XML output, depending on the journal.

## Example usage

This library is meant to be integrated into another operational system, where the `elifearticle` objects can be populated with sufficient data to produce complete JATS XMl output, however the following is a simple example using interactive Python:


```python
>>> from elifearticle.article import Article, Contributor
>>> from jatsgenerator import generate
>>> from jatsgenerator.conf import raw_config, parse_raw_config
>>> article_object = Article("10.7554/eLife.00666", "Article title")
>>> article_object.abstract = "An abstract."
>>> article_object.add_contributor(Contributor("author", "Surname", "Given"))
>>> article_object.add_contributor(Contributor("editor", "Orr", "Edit"))
>>> jats_config = parse_raw_config(raw_config("elife"))
>>> article_xml_object = generate.build_xml(None, article_object, jats_config)
>>> print(article_xml_object.output_xml())
```

The JATS XML output, pretty formatted, will look something like this:

```xml
<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE article PUBLIC "-//NLM//DTD JATS (Z39.96) Journal Archiving and Interchange DTD v1.1d3 20150301//EN"  "JATS-archivearticle1.dtd">
<article article-type="research-article" dtd-version="1.1d3" xmlns:mml="http://www.w3.org/1998/Math/MathML" xmlns:xlink="http://www.w3.org/1999/xlink">
<!--generated by jats-generator at 2021-09-02 14:18:33 from version 6cd2bf35c814117f5e5966ec415e01cedbdcd7de-->
    <front>
        <journal-meta>
            <journal-id journal-id-type="nlm-ta">elife</journal-id>
            <journal-id journal-id-type="publisher-id">eLife</journal-id>
            <journal-title-group>
                <journal-title>eLife</journal-title>
            </journal-title-group>
            <issn publication-format="electronic">2050-084X</issn>
            <publisher>
                <publisher-name>eLife Sciences Publications, Ltd</publisher-name>
            </publisher>
        </journal-meta>
        <article-meta>
            <article-id pub-id-type="doi">10.7554/eLife.00666</article-id>
            <title-group>
                <article-title>Article title</article-title>
            </title-group>
            <contrib-group>
                <contrib contrib-type="author">
                    <name>
                        <surname>Surname</surname>
                        <given-names>Given</given-names>
                    </name>
                </contrib>
            </contrib-group>
            <contrib-group content-type="section">
                <contrib contrib-type="editor">
                    <name>
                        <surname>Orr</surname>
                        <given-names>Edit</given-names>
                    </name>
                    <role>Reviewing editor</role>
                </contrib>
            </contrib-group>
            <abstract>
                <p>An abstract.</p>
            </abstract>
        </article-meta>
    </front>
    <back>
        <sec id="s1" sec-type="additional-information">
            <title>Additional information</title>
        </sec>
    </back>
</article>
```

## Run code tests

Use `pytest` for testing, install it if missing:

```
pip install pytest
```

Run tests

```
pytest
```

Run tests with coverage

```
coverage run -m pytest
```

then report on code coverage

```
converage report -m
```

## License

Licensed under [MIT](https://opensource.org/licenses/mit-license.php).
