"""
Tests for naming convention validation system.

CRITICAL PREVENTION TESTING: Ensures zero tolerance enforcement for forbidden naming patterns.
"""

import pytest

from lib.validation.naming_conventions import (
    NamingConventionValidator,
    NamingViolation,
    naming_validator,
    validate_before_creation,
    validate_class_creation,
    validate_file_creation,
    validate_function_creation,
)


class TestNamingConventionValidator:
    """Test suite for the naming convention validator."""

    def setup_method(self):
        """Setup test environment."""
        self.validator = NamingConventionValidator()

    def test_forbidden_patterns_detection(self):
        """Test detection of all forbidden naming patterns."""
        forbidden_names = [
            "test_something_fixed.py",
            "function_enhanced",
            "ClassImproved",
            "variable_updated",
            "service_better.py",
            "new_implementation.py",
            "handler_v2.py",
            "processor_fix.py",
            "tool_optimized.py",
        ]

        for name in forbidden_names:
            is_valid, violations = self.validator.validate_name(name)
            assert not is_valid, f"Should detect violation in: {name}"
            assert len(violations) > 0, f"Should have violations for: {name}"
            assert all(v.severity == "CRITICAL" for v in violations), f"All violations should be CRITICAL for: {name}"

    def test_valid_names_pass_validation(self):
        """Test that purpose-based names pass validation."""
        valid_names = [
            "test_makefile_comprehensive.py",
            "user_authentication_service.py",
            "DatabaseConnectionManager",
            "api_response_handler",
            "configuration_loader.py",
            "payment_processor.py",
            "email_notification_system.py",
        ]

        for name in valid_names:
            is_valid, violations = self.validator.validate_name(name)
            assert is_valid, f"Should pass validation: {name}"
            assert len(violations) == 0, f"Should have no violations for: {name}"

    def test_historical_violations_prevention(self):
        """Test prevention of historically problematic names."""
        historical_violations = [
            "test_makefile_uninstall_enhanced.py",
            "test_makefile_uninstall_new.py",
            "test_makefile_uninstall_fixed.py",
            "test_makefile_uninstall_improved.py",
        ]

        for name in historical_violations:
            is_valid, violations = self.validator.validate_name(name)
            assert not is_valid, f"Should prevent historical violation: {name}"
            assert any("HISTORICAL_VIOLATION" in v.violation_type for v in violations), (
                f"Should detect historical pattern: {name}"
            )

    def test_file_path_validation(self):
        """Test file path validation including directory structure."""
        # Should fail - forbidden pattern in filename
        is_valid, violations = self.validator.validate_file_path("/path/to/service_enhanced.py")
        assert not is_valid
        assert len(violations) > 0

        # Should pass - clean purpose-based name
        is_valid, violations = self.validator.validate_file_path("/path/to/user_service.py")
        assert is_valid
        assert len(violations) == 0

    def test_function_name_validation(self):
        """Test function name validation."""
        # Should fail
        is_valid, violations = self.validator.validate_function_name("process_data_fixed")
        assert not is_valid
        assert len(violations) > 0

        # Should pass
        is_valid, violations = self.validator.validate_function_name("process_user_data")
        assert is_valid
        assert len(violations) == 0

    def test_class_name_validation(self):
        """Test class name validation."""
        # Should fail
        is_valid, violations = self.validator.validate_class_name("DatabaseManagerImproved")
        assert not is_valid
        assert len(violations) > 0

        # Should pass
        is_valid, violations = self.validator.validate_class_name("DatabaseConnectionManager")
        assert is_valid
        assert len(violations) == 0

    def test_variable_name_validation(self):
        """Test variable name validation."""
        # Should fail
        is_valid, violations = self.validator.validate_variable_name("config_updated")
        assert not is_valid
        assert len(violations) > 0

        # Should pass
        is_valid, violations = self.validator.validate_variable_name("database_config")
        assert is_valid
        assert len(violations) == 0

    def test_purpose_based_alternative_generation(self):
        """Test generation of purpose-based name alternatives."""
        test_cases = [
            ("test_makefile_enhanced.py", "test_makefile_comprehensive.py"),
            ("service_fixed.py", "service_operations.py"),
            ("config_updated.py", "config_settings.py"),
            ("util_improved.py", "util_operations.py"),
        ]

        for problematic_name, _expected_pattern in test_cases:
            alternative = self.validator.generate_purpose_based_alternative(problematic_name)
            assert "fixed" not in alternative.lower()
            assert "enhanced" not in alternative.lower()
            assert "improved" not in alternative.lower()
            assert "updated" not in alternative.lower()
            # Should contain purpose-based elements
            assert any(
                word in alternative
                for word in ["comprehensive", "operations", "settings", "validation", "implementation"]
            )

    def test_violation_report_generation(self):
        """Test detailed violation report generation."""
        violations = [
            NamingViolation(
                violation_type="FORBIDDEN_PATTERN_FILE",
                forbidden_pattern=r"\bfixed\b",
                suggested_alternative="Remove 'fixed' - describe what it actually does",
                severity="CRITICAL",
            )
        ]

        report = self.validator.get_violation_report(violations, "test_file_fixed.py")
        assert "NAMING CONVENTION VIOLATION" in report
        assert "test_file_fixed.py" in report
        assert "FORBIDDEN_PATTERN_FILE" in report
        assert "PURPOSE-BASED ALTERNATIVE" in report
        assert "CRITICAL" in report


class TestValidationHooks:
    """Test pre-creation validation hooks."""

    def test_validate_before_creation_success(self):
        """Test successful validation allows creation."""
        # Should not raise exception
        validate_before_creation("user_service.py", "file")
        validate_before_creation("process_data", "function")
        validate_before_creation("UserManager", "class")

    def test_validate_before_creation_failure(self):
        """Test validation failure prevents creation."""
        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_before_creation("service_fixed.py", "file")

        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_before_creation("process_data_enhanced", "function")

        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_before_creation("UserManagerImproved", "class")

    def test_validate_file_creation_hook(self):
        """Test file creation validation hook."""
        # Should not raise exception
        validate_file_creation("/path/to/user_service.py")

        # Should raise exception
        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_file_creation("/path/to/service_enhanced.py")

    def test_validate_function_creation_hook(self):
        """Test function creation validation hook."""
        # Should not raise exception
        validate_function_creation("authenticate_user")

        # Should raise exception
        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_function_creation("authenticate_user_fixed")

    def test_validate_class_creation_hook(self):
        """Test class creation validation hook."""
        # Should not raise exception
        validate_class_creation("AuthenticationService")

        # Should raise exception
        with pytest.raises(ValueError, match="NAMING CONVENTION VIOLATION PREVENTED"):
            validate_class_creation("AuthenticationServiceImproved")

    def test_error_message_includes_user_feedback(self):
        """Test that validation errors include original user feedback."""
        with pytest.raises(ValueError) as exc_info:
            validate_before_creation("test_enhanced.py", "file")

        error_message = str(exc_info.value)
        assert (
            "its completly forbidden, across all codebase, to write files and functionsm etc, with fixed, enhanced, etc"
            in error_message
        )
        assert "BEHAVIORAL LEARNING" in error_message
        assert "Zero tolerance" in error_message


class TestGlobalValidatorIntegration:
    """Test global validator instance integration."""

    def test_global_validator_accessible(self):
        """Test that global validator instance is accessible."""
        assert naming_validator is not None
        assert isinstance(naming_validator, NamingConventionValidator)

    def test_global_validator_functionality(self):
        """Test global validator functionality."""
        # Test through global instance
        is_valid, violations = naming_validator.validate_name("service_enhanced.py")
        assert not is_valid
        assert len(violations) > 0

        is_valid, violations = naming_validator.validate_name("user_service.py")
        assert is_valid
        assert len(violations) == 0


class TestEdgeCases:
    """Test edge cases and boundary conditions."""

    def test_empty_string_validation(self):
        """Test validation of empty strings."""
        validator = NamingConventionValidator()
        is_valid, violations = validator.validate_name("")
        assert is_valid  # Empty string should pass (no forbidden patterns)
        assert len(violations) == 0

    def test_case_insensitive_detection(self):
        """Test that pattern detection is case insensitive."""
        validator = NamingConventionValidator()

        test_cases = ["Service_FIXED.py", "ENHANCED_function", "Class_Improved", "VARIABLE_UPDATED"]

        for name in test_cases:
            is_valid, violations = validator.validate_name(name)
            assert not is_valid, f"Should detect violation in: {name}"
            assert len(violations) > 0, f"Should have violations for: {name}"

    def test_multiple_violations_in_single_name(self):
        """Test detection of multiple violations in a single name."""
        validator = NamingConventionValidator()

        # Name with multiple forbidden patterns
        is_valid, violations = validator.validate_name("service_fixed_enhanced_improved.py")
        assert not is_valid
        assert len(violations) >= 3  # Should detect multiple patterns

    def test_partial_pattern_matches(self):
        """Test that partial matches don't trigger false positives."""
        validator = NamingConventionValidator()

        # These should NOT trigger violations (partial matches)
        valid_names = [
            "configuration_manager.py",  # "config" contains "fix" but shouldn't match
            "newsletter_service.py",  # "new" in "newsletter" shouldn't match
            "optimizer_tool.py",  # "optimize" should pass (not "optimized")
            "enhancement_guide.py",  # "enhancement" as purpose is different from "enhanced"
        ]

        for name in valid_names:
            is_valid, violations = validator.validate_name(name)
            # Note: Some of these may still fail based on current patterns - this tests the boundary
            # The test documents expected behavior for refinement
            if not is_valid:
                pass
