# lazy_mysql

一个轻量级的Python库，为MySQL数据库操作提供简洁优雅的解决方案。

[![zread](https://img.shields.io/badge/Ask_Zread-_.svg?style=flat&color=00b0aa&labelColor=000000&logo=data%3Aimage%2Fsvg%2Bxml%3Bbase64%2CPHN2ZyB3aWR0aD0iMTYiIGhlaWdodD0iMTYiIHZpZXdCb3g9IjAgMCAxNiAxNiIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTQuOTYxNTYgMS42MDAxSDIuMjQxNTZDMS44ODgxIDEuNjAwMSAxLjYwMTU2IDEuODg2NjQgMS42MDE1NiAyLjI0MDFWNC45NjAxQzEuNjAxNTYgNS4zMTM1NiAxLjg4ODEgNS42MDAxIDIuMjQxNTYgNS42MDAxSDQuOTYxNTZDNS4zMTUwMiA1LjYwMDEgNS42MDE1NiA1LjMxMzU2IDUuNjAxNTYgNC45NjAxVjIuMjQwMUM1LjYwMTU2IDEuODg2NjQgNS4zMTUwMiAxLjYwMDEgNC45NjE1NiAxLjYwMDFaIiBmaWxsPSIjZmZmIi8%2BCjxwYXRoIGQ9Ik00Ljk2MTU2IDEwLjM5OTlIMi4yNDE1NkMxLjg4ODEgMTAuMzk5OSAxLjYwMTU2IDEwLjY4NjQgMS42MDE1NiAxMS4wMzk5VjEzLjc1OTlDMS42MDE1NiAxNC4xMTM0IDEuODg4MSAxNC4zOTk5IDIuMjQxNTYgMTQuMzk5OUg0Ljk2MTU2QzUuMzE1MDIgMTQuMzk5OSA1LjYwMTU2IDE0LjExMzQgNS42MDE1NiAxMy43NTk5VjExLjAzOTlDNS42MDE1NiAxMC42ODY0IDUuMzE1MDIgMTAuMzk5OSA0Ljk2MTU2IDEwLjM5OTlaIiBmaWxsPSIjZmZmIi8%2BCjxwYXRoIGQ9Ik0xMy43NTg0IDEuNjAwMUgxMS4wMzg0QzEwLjY4NSAxLjYwMDEgMTAuMzk4NCAxLjg4NjY0IDEwLjM5ODQgMi4yNDAxVjQuOTYwMUMxMC4zOTg0IDUuMzEzNTYgMTAuNjg1IDUuNjAwMSAxMS4wMzg0IDUuNjAwMUgxMy43NTg0QzE0LjExMTkgNS42MDAxIDE0LjM5ODQgNS4zMTM1NiAxNC4zOTg0IDQuOTYwMVYyLjI0MDFDMTQuMzk4NCAxLjg4NjY0IDE0LjExMTkgMS42MDAxIDEzLjc1ODQgMS42MDAxWiIgZmlsbD0iI2ZmZiIvPgo8cGF0aCBkPSJNNCAxMkwxMiA0TDQgMTJaIiBmaWxsPSIjZmZmIi8%2BCjxwYXRoIGQ9Ik00IDEyTDEyIDQiIHN0cm9rZT0iI2ZmZiIgc3Ryb2tlLXdpZHRoPSIxLjUiIHN0cm9rZS1saW5lY2FwPSJyb3VuZCIvPgo8L3N2Zz4K&logoColor=ffffff)](https://zread.ai/tinycen/lazy_mysql)

## ✨ 核心特性

- 🔌 **统一SQL执行接口** - 简化复杂的数据库操作流程
- 📊 **智能查询构建器** - 支持复杂条件、多表关联、排序限制
- 💾 **批量数据操作** - 自动优化策略，支持超大数量级数据处理
- 🔄 **Upsert支持** - 智能判断存在更新/不存在插入
- 🛡️ **安全防注入** - 参数化查询，自动SQL注入防护
- 📈 **结果格式化** - 支持DataFrame、字典、列表等多种格式输出
- 📝 **表结构导出** - 一键导出Markdown格式文档
- ⚡ **高性能优化** - LOAD DATA INFILE支持，百万级数据秒级处理

## 🚀 快速安装

```bash
pip install --upgrade lazy-mysql
```

## 🎯 快速开始

### 1. 数据库连接初始化

```python
from lazy_mysql.executor import SQLExecutor
from lazy_mysql.sql_config import MySQLConfig

# 创建数据库配置
config = MySQLConfig(
    host='localhost',
    user='your_username',
    passwd='your_password',
    default_database='your_database'
)

# 创建执行器实例
executor = SQLExecutor(config)
```

### 2. 智能查询操作

```python
# 基础查询
users = executor.select('users', ['id', 'name', 'email'])
print(users)

# 条件查询 + 排序限制
active_users = executor.select(
    'users',
    ['id', 'name', 'email'],
    where_conditions={'status': 'active', 'age': ('>', 18)},
    order_by='created_at DESC',
    limit=10
)

# 复杂条件查询
results = executor.select(
    'users',
    ['id', 'name', 'score'],
    where_conditions={
        'status': ('IN', ['active', 'premium']),
        'score': ('BETWEEN', [80, 100]),
        'name': ('LIKE', '%John%')
    },
    fetch_config={'output_format': 'df'}  # 返回DataFrame格式
)
```

### 3. 批量数据插入

```python
# 单条插入
executor.insert('users', {'name': '张三', 'email': 'zhang@example.com'}, commit=True)

# 批量插入（自动优化策略）
users_data = [
    {'name': '李四', 'email': 'li@example.com', 'age': 25},
    {'name': '王五', 'email': 'wang@example.com', 'age': 30},
    # ... 支持数千条记录
]
inserted_count = executor.insert('users', users_data, commit=True)

# Upsert操作（存在更新，不存在插入）
user_data = {
    'id': 1,
    'name': '张三',
    'email': 'new_email@example.com',
    'updated_at': '2024-01-15 10:00:00'
}
executor.upsert('users', user_data, commit=True)
```

### 4. 数据更新与删除

```python
# 条件更新
executor.update(
    'users',
    {'status': 'premium', 'updated_at': '2024-01-15 10:00:00'},
    where_conditions={'last_login': ('>=', '2024-01-01'), 'points': ('>=', 1000)},
    commit=True
)

# 安全删除（必须指定条件）
executor.delete('users', where_conditions={'status': 'inactive', 'last_login': ('<', '2023-01-01')}, commit=True)
```

### 5. 使用完毕后关闭连接

```python
executor.close()
```

### 📚 详细文档

### 🔗 连接与配置
- [数据库连接初始化](doc/CONNECTION.md) - 连接配置、错误处理、重试机制、最佳实践

### 🔍 查询操作
- [SELECT 查询操作](doc/SELECT.md) - 智能查询构建、复杂条件、多表关联、结果格式化

### 💾 数据修改
- [INSERT 插入操作](doc/INSERT.md) - 批量插入、Upsert、大数据优化、重复处理
- [UPDATE 更新操作](doc/UPDATE.md) - 条件更新、批量更新、SQL表达式、性能优化
- [DELETE 删除操作](doc/DELETE.md) - 安全删除、条件组合、错误处理、调试技巧


## 🔧 环境要求

- **Python**: 3.7+
- **MySQL**: 8.0.36+
- **依赖库**:
  - `mysql-connector-python>=9.4.0`
  - `pandas>=2.3.1`

> ⚠️ **兼容性说明**: 低于上述版本要求的兼容性尚未验证

## 🎯 适用场景

- ✅ **Web应用开发** - 快速原型开发、API后端服务
- ✅ **数据分析** - DataFrame集成、报表生成、数据清洗
- ✅ **批量数据处理** - 日志导入、数据迁移、ETL流程
- ✅ **企业级应用** - 事务处理、并发控制、错误重试

## 🏆 性能优势

| 操作类型 | 传统方式 | lazy_mysql优化 | 性能提升 |
|---------|---------|---------------|----------|
| 批量插入1万条 | 3-5秒 | 0.5-1秒 | **5-10倍** |
| 批量插入10万条 | 30-60秒 | 2-5秒 | **15-30倍** |
| 复杂条件查询 | 手动拼接SQL | 智能构建 | **开发效率3倍** |
| 大数据导出 | 内存占用高 | 流式处理 | **内存节省80%** |

## 📦 PyPI 项目

项目已发布到PyPI，可通过以下链接访问：
- **PyPI主页**: https://pypi.org/project/lazy-mysql/

## 📄 开源协议

本项目采用MIT开源协议 - 详见 [LICENSE](LICENSE) 文件

## 📄 License
This project is licensed under the MIT License.