from typing import List, Literal

from pydantic import BaseModel


class Usage(BaseModel):
    """Represents the token usage statistics for a request.

    Attributes:
        prompt_tokens (int): The number of tokens used by the prompt.
        total_tokens (int): The total number of tokens used by the request.
    """

    prompt_tokens: int
    """The number of tokens used by the prompt."""

    total_tokens: int
    """The total number of tokens used by the request."""


class Embedding(BaseModel):
    """Represents an embedding vector and metadata.

    Attributes:
        embedding (List[float]): The embedding vector as a list of floats. The length of the vector depends on the model.
        index (int): The index of the embedding in the list of embeddings.
        object (Literal["embedding"]): The object type, which is always "embedding".
    """

    embedding: List[float]
    """The embedding vector, which is a list of floats.

    The length of vector depends on the model as listed in the
    [embedding guide](https://platform.openai.com/docs/guides/embeddings).
    """

    index: int
    """The index of the embedding in the list of embeddings."""

    object: Literal["embedding"] = "embedding"
    """The object type, which is always "embedding"."""


class CreateEmbeddingResponse(BaseModel):
    """Represents the response from an embedding creation request.

    Attributes:
        data (List[Embedding]): The list of embeddings generated by the model.
        model (str): The name of the model used to generate the embeddings.
        object (Literal["list"]): The object type, which is always "list".
        usage (Usage): The usage information for the request.
    """

    data: List[Embedding]
    """The list of embeddings generated by the model."""

    model: str
    """The name of the model used to generate the embedding."""

    object: Literal["list"] = "list"
    """The object type, which is always "list"."""

    usage: Usage
    """The usage information for the request."""
