"""
function_call.py

Type definitions for function calling APIs used by LLM providers.
This file contains Pydantic models for use with OpenAI's chat-completion
and responses APIs. Types for additional providers (Anthropic, Gemini, etc.)
should be added to the corresponding sections below.

Sections:
    - OpenAI Types (Chat Completions & Responses)
    - Anthropic Types (TODO)
    - Google Gemini Types (TODO)
"""

from typing import Dict, List, Literal, Optional, TypeAlias, Union

from pydantic import BaseModel

# ======================================================================
# 1. OPENAI TYPES (CHAT COMPLETION & RESPONSES API)
# ======================================================================

# ===========================
# Chat Completion API SECTION
# ===========================


# --------- API INPUT ---------
class FunctionDefinitionCore(BaseModel):
    name: str
    """The name of the function to be called.

    Must be a-z, A-Z, 0-9, or contain underscores and dashes, with a maximum length
    of 64.
    """


class FunctionDefinition(FunctionDefinitionCore):
    description: Optional[str] = None
    """
    A description of what the function does, used by the model to choose when and
    how to call the function.
    """
    parameters: Optional[Dict[str, object]] = None
    """The parameters the functions accepts, described as a JSON Schema object.

    See the [guide](https://platform.openai.com/docs/guides/function-calling) for
    examples, and the
    [JSON Schema reference](https://json-schema.org/understanding-json-schema/) for
    documentation about the format.

    Omitting `parameters` defines a function with an empty parameter list.
    """
    strict: Optional[bool] = None
    """Whether to enable strict schema adherence when generating the function call.

    If set to true, the model will follow the exact schema defined in the
    `parameters` field. Only a subset of JSON Schema is supported when `strict` is
    `true`. Learn more about Structured Outputs in the
    [function calling guide](docs/guides/function-calling).
    """


# used in `tools`
class ChatCompletionToolParam(BaseModel):
    function: FunctionDefinition
    type: Literal["function"] = "function"
    """The type of the tool. Currently, only `function` is supported."""


# used in `tool_choice`
class ChatCompletionNamedToolChoiceParam(BaseModel):
    function: FunctionDefinitionCore
    type: Literal["function"] = "function"
    """The type of the tool. Currently, only `function` is supported."""


ChatCompletionToolChoiceOptionParam: TypeAlias = Union[
    Literal["none", "auto", "required"], ChatCompletionNamedToolChoiceParam
]


# --------- LLM OUTPUT ---------
class Function(BaseModel):
    arguments: str
    """
    The arguments to call the function with, as generated by the model in JSON
    format. Note that the model does not always generate valid JSON, and may
    hallucinate parameters not defined by your function schema. Validate the
    arguments in your code before calling your function.
    """
    name: str
    """The name of the function to call."""


# elements in `tool_calls`
class ChatCompletionMessageToolCall(BaseModel):
    id: str
    """The ID of the tool call."""
    function: Function
    """The function that the model called."""
    type: Literal["function"] = "function"
    """The type of the tool. Currently, only `function` is supported."""


# funtion definition in stream deltas
class ChoiceDeltaToolCallFunction(BaseModel):
    arguments: Optional[str] = None
    """
    The arguments to call the function with, as generated by the model in JSON
    format. Note that the model does not always generate valid JSON, and may
    hallucinate parameters not defined by your function schema. Validate the
    arguments in your code before calling your function.
    """
    name: Optional[str] = None
    """The name of the function to call."""


# used in `tool_calls` in stream deltas
class ChoiceDeltaToolCall(BaseModel):
    index: int

    id: Optional[str] = None
    """The ID of the tool call."""

    function: Optional[ChoiceDeltaToolCallFunction] = None

    type: Optional[Literal["function"]] = None
    """The type of the tool. Currently, only `function` is supported."""


# =====================
# Responses API SECTION
# =====================


# --------- API INPUT ---------
class FunctionTool(BaseModel):
    """API INPUT"""

    name: str
    """The name of the function to call."""
    parameters: Optional[Dict[str, object]] = None
    """A JSON schema object describing the parameters of the function."""
    strict: Optional[bool] = None
    """Whether to enforce strict parameter validation. Default `true`."""
    type: Literal["function"] = "function"
    """The type of the function tool. Always `function`."""
    description: Optional[str] = None
    """A description of the function.

    Used by the model to determine whether or not to call the function.
    """


class ToolChoiceFunctionParam(BaseModel):
    """API INPUT"""

    name: str
    """The name of the function to call."""
    type: Literal["function"] = "function"
    """For function calling, the type is always `function`."""


ToolChoice: TypeAlias = Union[
    Literal["none", "auto", "required"], ToolChoiceFunctionParam
]
# (API INPUT: as tool_choice argument in responses API)


# --------- LLM OUTPUT ---------
class ResponseFunctionToolCall(BaseModel):
    """LLM OUTPUT"""

    arguments: str
    """A JSON string of the arguments to pass to the function."""
    call_id: str
    """The unique ID of the function tool call generated by the model."""
    name: str
    """The name of the function to run."""
    type: Literal["function_call"] = "function_call"
    """The type of the function tool call. Always `function_call`."""
    id: Optional[str] = None
    """The unique ID of the function tool call."""
    status: Optional[Literal["in_progress", "completed", "incomplete"]] = None
    """The status of the item.

    One of `in_progress`, `completed`, or `incomplete`. Populated when items are
    returned via API.
    """


# ======================================================================
# 2. ANTHROPIC TYPES
# ======================================================================


# --------- API INPUT ---------
class InputSchemaTyped(BaseModel):
    type: Literal["object"]
    properties: Optional[object] = None
    required: Optional[List[str]] = None


InputSchema: TypeAlias = Union[InputSchemaTyped, Dict[str, object]]


class CacheControlEphemeralParam(BaseModel):
    type: Literal["ephemeral"] = "ephemeral"


class ToolParam(BaseModel):
    input_schema: InputSchema
    """[JSON schema](https://json-schema.org/draft/2020-12) for this tool's input.

    This defines the shape of the `input` that your tool accepts and that the model
    will produce.
    """

    name: str
    """Name of the tool.

    This is how the tool will be called by the model and in `tool_use` blocks.
    """

    cache_control: Optional[CacheControlEphemeralParam] = None
    """Create a cache control breakpoint at this content block."""

    description: str
    """Description of what this tool does.

    Tool descriptions should be as detailed as possible. The more information that
    the model has about what the tool is and how to use it, the better it will
    perform. You can use natural language descriptions to reinforce important
    aspects of the tool input JSON schema.
    """

    type: Optional[Literal["custom"]] = "custom"


# used in `tool_choice`
class ToolChoiceShared(BaseModel):
    disable_parallel_tool_use: bool = False
    """Whether to disable parallel tool use.

    Defaults to `false`. If set to `true`, the model will output exactly one tool use.
    """


class ToolChoiceAnyParam(ToolChoiceShared):
    type: Literal["any"] = "any"


class ToolChoiceAutoParam(ToolChoiceShared):
    type: Literal["auto"] = "auto"


class ToolChoiceNoneParam(BaseModel):
    type: Literal["none"] = "none"


class ToolChoiceToolParam(ToolChoiceShared):
    name: str
    """The name of the tool to use."""

    type: Literal["tool"] = "tool"


ToolChoiceParam: TypeAlias = Union[
    ToolChoiceAutoParam, ToolChoiceAnyParam, ToolChoiceToolParam, ToolChoiceNoneParam
]


# --------- LLM OUTPUT ---------
# elements in `tool_calls`
class ToolUseBlock(BaseModel):
    id: str

    input: object

    name: str

    type: Literal["tool_use"] = "tool_use"

    cache_control: Optional[CacheControlEphemeralParam] = None
    """Create a cache control breakpoint at this content block."""


# ======================================================================
# 3. GOOGLE GEMINI TYPES (TODO)
# ======================================================================
# Add Google Gemini-compatible function call types here...
