from typing import List, Optional, TypeAlias, Union

from pydantic import BaseModel
from typing_extensions import Literal

from .function_call import FunctionTool, ResponseFunctionToolCall

# ==================== Non-Streaming ====================


class ResponseError(BaseModel):
    code: Literal[
        "server_error",
        "rate_limit_exceeded",
        "invalid_prompt",
        "vector_store_timeout",
        "invalid_image",
        "invalid_image_format",
        "invalid_base64_image",
        "invalid_image_url",
        "image_too_large",
        "image_too_small",
        "image_parse_error",
        "image_content_policy_violation",
        "invalid_image_mode",
        "image_file_too_large",
        "unsupported_image_media_type",
        "empty_image_file",
        "failed_to_download_image",
        "image_file_not_found",
    ]
    """The error code for the response."""

    message: str
    """A human-readable description of the error."""


class ResponseUsage(BaseModel):
    input_tokens: int
    """The number of input tokens."""

    output_tokens: int
    """The number of output tokens."""

    total_tokens: int
    """The total number of tokens used."""


class Reasoning(BaseModel):
    effort: Optional[Optional[Literal["low", "medium", "high"]]] = None

    summary: Optional[Literal["auto", "concise", "detailed"]] = None


class ToolChoiceFunction(BaseModel):
    name: str
    """The name of the function to call."""

    type: Literal["function"] = "function"
    """For function calling, the type is always `function`."""


class ResponseOutputText(BaseModel):
    text: str
    """The text output from the model."""

    type: Literal["output_text"] = "output_text"
    """The type of the output text. Always `output_text`."""


class ResponseOutputMessage(BaseModel):
    id: str
    """The unique ID of the output message."""

    content: List[
        # Union[
        ResponseOutputText,
        # ResponseOutputRefusal,  # not sure when it could be triggered, disabled for now
        # ]
    ]
    """The content of the output message."""

    role: Literal["assistant"] = "assistant"
    """The role of the output message. Always `assistant`."""

    status: Literal["in_progress", "completed", "incomplete"]
    """The status of the message input.

    One of `in_progress`, `completed`, or `incomplete`. Populated when input items
    are returned via API.
    """

    type: Literal["message"] = "message"
    """The type of the output message. Always `message`."""


class Response(BaseModel):
    id: str
    """Unique identifier for this Response."""

    created_at: float
    """Unix timestamp (in seconds) of when this Response was created."""

    # error: Optional[ResponseError] = None
    # """An error object returned when the model fails to generate a Response."""

    instructions: Optional[str] = None
    """
    Inserts a system (or developer) message as the first item in the model's
    context.

    When using along with `previous_response_id`, the instructions from a previous
    response will not be carried over to the next response. This makes it simple to
    swap out system (or developer) messages in new responses.
    """

    model: str
    """Model ID used to generate the response, like `gpt-4o` or `o3`.

    OpenAI offers a wide range of models with different capabilities, performance
    characteristics, and price points. Refer to the
    [model guide](https://platform.openai.com/docs/models) to browse and compare
    available models.
    """

    object: Literal["response"] = "response"
    """The object type of this resource - always set to `response`."""

    output: List[
        Union[
            ResponseOutputMessage,
            ResponseFunctionToolCall,
            # ResponseReasoningItem,
            # ImageGenerationCall,
        ]
    ]
    """An array of content items generated by the model.

    - The length and order of items in the `output` array is dependent on the
      model's response.
    - Rather than accessing the first item in the `output` array and assuming it's
      an `assistant` message with the content generated by the model, you might
      consider using the `output_text` property where supported in SDKs.
    """

    # parallel_tool_calls: bool
    # """Whether to allow the model to run tool calls in parallel."""

    temperature: Optional[float] = None
    """What sampling temperature to use, between 0 and 2.

    Higher values like 0.8 will make the output more random, while lower values like
    0.2 will make it more focused and deterministic. We generally recommend altering
    this or `top_p` but not both.
    """

    tool_choice: Union[Literal["none", "auto", "required"], ToolChoiceFunction] = "auto"
    """
    How the model should select which tool (or tools) to use when generating a
    response. See the `tools` parameter to see how to specify which tools the model
    can call.
    """

    tools: List[FunctionTool] = []
    """An array of tools the model may call while generating a response.

    You can specify which tool to use by setting the `tool_choice` parameter.

    The one categories of tools you can provide the model are:

    - **Function calls (custom tools)**: Functions that are defined by you, enabling
      the model to call your own code. Learn more about
      [function calling](https://platform.openai.com/docs/guides/function-calling).
    """

    top_p: Optional[float] = None
    """
    An alternative to sampling with temperature, called nucleus sampling, where the
    model considers the results of the tokens with top_p probability mass. So 0.1
    means only the tokens comprising the top 10% probability mass are considered.

    We generally recommend altering this or `temperature` but not both.
    """

    background: Optional[bool] = False
    """Whether to run the model response in the background.

    [Learn more](https://platform.openai.com/docs/guides/background).
    """

    max_output_tokens: Optional[int] = None
    """
    An upper bound for the number of tokens that can be generated for a response,
    including visible output tokens and
    [reasoning tokens](https://platform.openai.com/docs/guides/reasoning).
    """

    reasoning: Optional[Reasoning] = None
    """**o-series models only**

    Configuration options for
    [reasoning models](https://platform.openai.com/docs/guides/reasoning).
    """

    status: Optional[
        Literal[
            "completed", "failed", "in_progress", "cancelled", "queued", "incomplete"
        ]
    ] = None
    """The status of the response generation.

    One of `completed`, `failed`, `in_progress`, `cancelled`, `queued`, or
    `incomplete`.
    """

    usage: Optional[ResponseUsage] = None
    """
    Represents token usage details including input tokens, output tokens, a
    breakdown of output tokens, and the total tokens used.
    """


# ==================== Streaming ====================


class ResponseCreatedEvent(BaseModel):
    response: Response
    """The response that was created."""

    sequence_number: int
    """The sequence number for this event."""

    type: Literal["response.created"] = "response.created"
    """The type of the event. Always `response.created`."""


class ResponseInProgressEvent(BaseModel):
    response: Response
    """The response that is in progress."""

    sequence_number: int
    """The sequence number of this event."""

    type: Literal["response.in_progress"] = "response.in_progress"
    """The type of the event. Always `response.in_progress`."""


ResponseOutputItem: TypeAlias = Union[
    ResponseOutputMessage,
    ResponseFunctionToolCall,
    # ResponseFileSearchToolCall,
    # ResponseFunctionWebSearch,
    # ResponseComputerToolCall,
    # ResponseReasoningItem,
    # ImageGenerationCall,
    # ResponseCodeInterpreterToolCall,
    # LocalShellCall,
    # McpCall,
    # McpListTools,
    # McpApprovalRequest,
]


class ResponseOutputItemAddedEvent(BaseModel):
    item: ResponseOutputItem
    """The output item that was added."""

    output_index: int
    """The index of the output item that was added."""

    sequence_number: int
    """The sequence number of this event."""

    type: Literal["response.output_item.added"] = "response.output_item.added"
    """The type of the event. Always `response.output_item.added`."""


class ResponseOutputItemDoneEvent(BaseModel):
    item: ResponseOutputItem
    """The output item that was marked done."""

    output_index: int
    """The index of the output item that was marked done."""

    sequence_number: int
    """The sequence number of this event."""

    type: Literal["response.output_item.done"] = "response.output_item.done"
    """The type of the event. Always `response.output_item.done`."""


Part: TypeAlias = Union[
    ResponseOutputText,
    ResponseOutputText,
    # ResponseOutputRefusal,
]


class ResponseContentPartAddedEvent(BaseModel):
    content_index: int
    """The index of the content part that was added."""

    item_id: str
    """The ID of the output item that the content part was added to."""

    output_index: int
    """The index of the output item that the content part was added to."""

    part: Part
    """The content part that was added."""

    sequence_number: int
    """The sequence number of this event."""

    type: Literal["response.content_part.added"] = "response.content_part.added"
    """The type of the event. Always `response.content_part.added`."""


class ResponseContentPartDoneEvent(BaseModel):
    content_index: int
    """The index of the content part that is done."""

    item_id: str
    """The ID of the output item that the content part was added to."""

    output_index: int
    """The index of the output item that the content part was added to."""

    part: Part
    """The content part that is done."""

    sequence_number: int
    """The sequence number of this event."""

    type: Literal["response.content_part.done"] = "response.content_part.done"
    """The type of the event. Always `response.content_part.done`."""


class ResponseTextDeltaEvent(BaseModel):
    content_index: int
    """The index of the content part that the text delta was added to."""

    delta: str
    """The text delta that was added."""

    item_id: str
    """The ID of the output item that the text delta was added to."""

    output_index: int
    """The index of the output item that the text delta was added to."""

    sequence_number: int
    """The sequence number for this event."""

    type: Literal["response.output_text.delta"] = "response.output_text.delta"
    """The type of the event. Always `response.output_text.delta`."""


class ResponseTextDoneEvent(BaseModel):
    content_index: int
    """The index of the content part that the text content is finalized."""

    item_id: str
    """The ID of the output item that the text content is finalized."""

    output_index: int
    """The index of the output item that the text content is finalized."""

    sequence_number: int
    """The sequence number for this event."""

    text: str
    """The text content that is finalized."""

    type: Literal["response.output_text.done"] = "response.output_text.done"
    """The type of the event. Always `response.output_text.done`."""


class ResponseCompletedEvent(BaseModel):
    response: Response
    """Properties of the completed response."""

    sequence_number: int
    """The sequence number for this event."""

    type: Literal["response.completed"] = "response.completed"
    """The type of the event. Always `response.completed`."""
