"""
This file tests 2 different config files other than slips' default config/slips.yaml
test/test.yaml and tests/test2.yaml
"""

import re
import shutil
from pathlib import PosixPath

import redis

from tests.common_test_utils import (
    create_output_dir,
    assert_no_errors,
    modify_yaml_config,
)
import pytest
import os
import subprocess
import time
import sys

alerts_file = "alerts.log"


def countdown(seconds, message):
    """
    counts down from the given number of seconds, printing a message each second.
    """
    while seconds > 0:
        sys.stdout.write(
            f"\rSending {message} in {seconds} "
        )  # overwrite the line
        sys.stdout.flush()  # ensures immediate output
        time.sleep(1)  # waits for 1 second
        seconds -= 1
    sys.stdout.write(f"\rSending {message} now!          \n")


def message_send(port, channel, message):
    # connect to redis database 0
    redis_client = redis.StrictRedis(host="localhost", port=port, db=0)

    # publish the message to the "network2fides" channel
    redis_client.publish(channel, message)

    print(f"Test message published to channel '{channel}'.")


message_alert_TL_NL = """{
    "type": "tl2nl_alert",
    "version": 1,
    "data": {
      "payload": "https://www.youtube.com/watch?v=dQw4w9WgXcQ"
    }
}"""


message_alert_NL_S = """{
    "type": "nl2tl_alert",
    "version": 1,
    "data":
        "sender": "<Metadata of peer who's alerting>"
        "payload": "https://www.youtube.com/watch?v=dQw4w9WgXcQ"
}"""


def check_strings_in_file(string_list, file_path):
    # Check if the file exists
    if not os.path.exists(file_path):
        print(f"File {file_path} does not exist.")
        return False

    # Open the file and read its content
    try:
        with open(file_path, "r") as file:
            file_content = file.read()

        # Check if all strings in the list are present in the file content
        for string in string_list:
            if string not in file_content:
                return False
        return True

    except Exception as e:
        print(f"Error reading file: {e}")
        return False


@pytest.mark.parametrize(
    "zeek_dir_path, output_dir, peer_output_dir, redis_port, peer_redis_port",
    [
        (
            "dataset/test13-malicious-dhcpscan-zeek-dir",
            "iris_integration_test/",
            "peer_iris_integration_test/",
            6644,
            6655,
        )
    ],
)
def test_messaging(
    zeek_dir_path, output_dir, peer_output_dir, redis_port, peer_redis_port
):
    """
    Tests whether Iris properly distributes an alert message generated by
    Slips to the network (~other peers).

    First Slips instance is a general node in the network, its connection
     string is generated and extracted from logs as a normal user would do,
    in a very standard use case.

    The second instance of Slips acts as a normal-user-peer that joins the
    network via the aforementioned Slips instance,
    which extends the standard use case of connecting to such P2P network.
    """
    # Two Slips instances are necessary to be run in this test.
    # Prepare output dir for the main Slips instance.
    # The logs of both beers will be clearly separated and kept intact.
    output_dir: PosixPath = create_output_dir(output_dir)
    output_file = os.path.join(output_dir, "slips_output.txt")

    # Prepare output dir for the main Slips instance.
    # The logs of both beers will be clearly separated and kept intact.
    output_dir_peer: PosixPath = create_output_dir(peer_output_dir)
    output_file_peer = os.path.join(output_dir_peer, "slips_output.txt")

    # this will be used for the extraction of the connection string form the
    # logs of iris running under the main Slips
    log_file_first_iris = output_dir / "iris/iris_logs.txt"
    log_file_second_iris = output_dir_peer / "iris/iris_logs.txt"

    # generate config of first peer
    slips_iris_main_config_file = (
        "tests/integration_tests/config/slips_iris_main.yaml"
    )
    modify_yaml_config(
        input_path="config/slips.yaml",
        output_dir=os.path.dirname(slips_iris_main_config_file),
        output_filename=os.path.basename(slips_iris_main_config_file),
        changes={
            "global_p2p": {"use_global_p2p": True},
            "modules": {"disable": ["template", "updatemanager"]},
        },
    )

    # that config file will be generated later to be able to add the first
    # peer's id to it
    iris_peer_config_file = (
        "tests/integration_tests/config/iris_peer_config.yaml"
    )

    # generate config of second peer
    iris_config_file = "tests/integration_tests/config/iris_config.yaml"
    modify_yaml_config(
        input_path="config/slips.yaml",
        output_dir=os.path.dirname(iris_config_file),
        output_filename=os.path.basename(iris_config_file),
        changes={
            "global_p2p": {
                "use_global_p2p": True,
                "iris_conf": iris_peer_config_file,
            },
            "modules": {"disable": ["template", "updatemanager"]},
        },
    )

    print("running slips ...")
    with open(output_file, "w") as log_file:
        with open(output_file_peer, "w") as iris_log_file:
            # Start the subprocess, redirecting stdout and stderr
            # to the same file
            # command for the main Slips instance
            command = [
                "./slips.py",
                "-t",
                "-g",
                "-e",
                "1",
                "-f",
                str(zeek_dir_path),
                "-o",
                str(output_dir),
                "-c",
                slips_iris_main_config_file,  # we're using the dafult peer
                # config here
                "-P",
                str(redis_port),
            ]
            process = subprocess.Popen(
                command,
                stdout=log_file,
                stderr=log_file,
            )

            # First peer (its Iris) needs to be ready and available for
            # connections when the second peer tries to reach out to it.
            countdown(20, "second peer")
            # get the connection string from the first peer and give it
            # to the second one so it is reachable
            with open(log_file_first_iris, "r") as log:
                for line in log:
                    match = re.search(r"connection string:\s+'(.+)'", line)
                    if match:
                        original_conn_string = match.group(1)
                        break
                else:
                    # if it comes here make sure that port 9010 used by
                    # iris is free
                    # sudo lsof -i :9010
                    # sudo kill -9 <PID>
                    print("No connection string found in log file.")
                    exit(1)

            # put the PeerID in the config file of the second peer's Iris
            # the goal is for the second iris to be able to find the first
            # iris/slips
            modify_yaml_config(
                input_path="config/iris_config.yaml",
                output_dir=os.path.dirname(iris_peer_config_file),
                output_filename=os.path.basename(iris_peer_config_file),
                changes={
                    "Redis": {"Port": 6655},
                    "Server": {"Port": 9006},
                    "PeerDiscovery": {
                        "ListOfMultiAddresses": [original_conn_string]
                    },
                    "Identity": {"KeyFile": "second.priv"}
                },
            )
            # generate a second command for the second peer
            peer_command = [
                "./slips.py",
                "-t",
                "-g",
                "-e",
                "1",
                "-f",
                str(zeek_dir_path),
                "-o",
                str(output_dir_peer),
                "-c",
                iris_config_file,  # we're not using the dafult peer config
                # here
                "-P",
                str(peer_redis_port),
            ]
            peer_process = subprocess.Popen(
                peer_command, stdout=iris_log_file, stderr=iris_log_file
            )

            print(
                f"Output and errors of first peer are logged in"
                f" {output_file}"
            )

            # let Slips properly and fully star with all of its parts and modules.
            countdown(80, "Sending msg in fides2network")
            # Sending a manual message to make sure there is an alert generated, because
            # is is highly probable that both slips have covered their network captures
            # before the infrastructure of P2P network was fully up and running
            message_send(
                redis_port,
                message=message_alert_TL_NL,
                channel="fides2network",
            )

            # these seconds are the time we give slips to process the msg
            countdown(30, "Sending SIGTERM to the 2 peers")
            # Kill em with kindness.
            os.kill(process.pid, 15)
            os.kill(peer_process.pid, 15)
            print("SIGTERM sent.")

            print("Sending SIGKILL to the 2 instances of Slips + iris")
            # Kill em. Without kindness.
            os.kill(process.pid, 9)
            print(f"Slips with PID {process.pid} was killed.")

            os.kill(peer_process.pid, 9)
            print(f"Slips peer with PID {peer_process.pid} was killed.")

    print("Slips is done, checking for errors in the 2 output dirs.")
    assert_no_errors(output_dir)
    assert_no_errors(output_dir_peer)

    print("Checking for iris expected logs in the generated log file")
    # make sure this string is there in the generated iris logs
    # this is how we ensure that iris ran correctly
    expected_log_entry = [
        "INFO	iris	protocols/alert.go:111	received p2p alert message"
    ]
    assert check_strings_in_file(expected_log_entry, log_file_second_iris)

    print("Deleting the output directories")
    shutil.rmtree(output_dir)
    shutil.rmtree(output_dir_peer)
    os.remove("modules/irisModule/second.priv")
    modify_yaml_config(
        input_path="config/iris_config.yaml",
        output_dir=os.path.dirname(iris_peer_config_file),
        output_filename=os.path.basename(iris_peer_config_file),
        changes={
            "Redis": {"Port": 6644},
            "Server": {"Port": 9010},
            "PeerDiscovery": {},
            "Identity": {"KeyFile": "private.key"}
        },
    )
