# LaTeX Examples from Mahonia

This document demonstrates the LaTeX output generated by the `mahonia.latex` module for various mathematical expressions.

> [!IMPORTANT]
> This file is auto-generated by `tests/test_latex_examples.py`
> Do not edit this file directly. Run the test to regenerate.

## Setup

All examples use the following setup:

```python
from typing import NamedTuple
from mahonia import Var, Const, PlusMinus, Percent, Approximately, Predicate
from mahonia.latex import latex

class Ctx(NamedTuple):
    x: float
    y: float
    # ... other fields as needed
```

## Basic Variables

### Simple Variables
- Variable `x`: `latex(Var[int, Ctx]('x'))` → $x$
- Variable `y`: `latex(Var[int, Ctx]('y'))` → $y$

### Variables with Subscripts

- Single character subscript: `latex(Var('x_1'))` → $x_1$
- Multiple character subscript: `latex(Var('x_max'))` → $x_{max}$
- Complex subscript: `latex(Var('velocity_initial'))` → $velocity_{initial}$

### Greek Letter Variables

- Alpha: `latex(Var('alpha'))` → $\alpha$
- Beta: `latex(Var('beta'))` → $\beta$
- Gamma: `latex(Var('gamma'))` → $\gamma$
- Delta: `latex(Var('delta'))` → $\delta$
- Epsilon: `latex(Var('epsilon'))` → $\epsilon$
- Theta: `latex(Var('theta'))` → $\theta$
- Pi: `latex(Var('pi'))` → $\pi$
- Sigma: `latex(Var('sigma'))` → $\sigma$
- Omega: `latex(Var('omega'))` → $\omega$


## Constants

### Numeric Constants

- Integer: `latex(Const(None, 42))` → $42$
- Float: `latex(Const(None, 3.14159))` → $3.14159$
- Negative: `latex(Const(None, -5))` → $-5$


### Named Constants

- Named constant Pi (becomes Greek π): `latex(Const('Pi', 3.14159))` → $\pi$
- Zero constant: `latex(Const('Zero', 0))` → $Zero$


## Arithmetic Operations

### Basic Operations

- Addition: `latex(x + y)` → $x + y$
- Subtraction: `latex(x - y)` → $x - y$
- Multiplication: `latex(x * y)` → $x \cdot y$
- Division: `latex(x / y)` → $\frac{x}{y}$

### Power Operations

- Square: `latex(x ** 2)` → $x^2$
- Variable exponent: `latex(x ** y)` → $x^{y}$
- Complex exponent: `latex(x ** (y + 1))` → $x^{y + 1}$
- Base with parentheses: `latex((x + y) ** 2)` → $(x + y)^2$

### Complex Arithmetic

```python
x = Var[float, Ctx]('x')
y = Var[float, Ctx]('y')
z = Var[float, Ctx]('z')
expr = ((x + y) * z) / (x - y)
latex(expr)  # '\frac{(x + y) \cdot z}{x - y}'
```
- Nested expression: $\frac{(x + y) \cdot z}{x - y}$

```python
x = Var[float, Ctx]('x')
two = Const('Two', 2)
one = Const('One', 1)
expr = x**3 + two*x**2 - x + one
latex(expr)  # 'x^3 + Two \cdot x^2 - x + One'
```
- Polynomial: $x^3 + Two \cdot x^2 - x + One$

## Comparison Operations

### Basic Comparisons

- Equal `latex(x == y)`: $x = y$
- Not equal `latex(x != y)`: $x \neq y$
- Less than `latex(x < y)`: $x < y$
- Less than or equal `latex(x <= y)`: $x \leq y$
- Greater than `latex(x > y)`: $x > y$
- Greater than or equal `latex(x >= y)`: $x \geq y$

### Complex Comparisons

- With arithmetic `latex((x + 5) > (y * 2))`: $x + 5 > y \cdot 2$

## Logical Operations

### Basic Logic

- AND `latex((x > 5) & (y < 10))`: $x > 5 \land y < 10$
- OR `latex((x > 5) | (y < 10))`: $x > 5 \lor y < 10$
- NOT `latex(~(x > 5))`: $\neg x > 5$

### Complex Logic

- Nested NOT: $\neg (x > 5 \land y < 10)$
- Mixed operations: $x + y > 10 \land x \cdot y < 50 \lor \frac{x}{y} > 2$

## Special Expressions

### Plus-Minus Expressions

- Named measurement: $measurement \pm 0.1$
- Unnamed measurement: $5.0 \pm 0.1$

### Percentage Expressions

- Named percentage: $measurement \pm 2.0\%$
- Unnamed percentage: $5.0 \pm 2.0\%$

### Approximately Expressions

- Basic approximation: $x \approx target \pm 0.1$
- Complex approximation: $x^{2.0} \approx target2 \pm 0.2$

### Predicate Expressions

- Named predicate: $\text{condition}: x > 5$
- Unnamed predicate: $x > 5$

## Real-World Examples

### Quadratic Formula Components

- Discriminant `b² - 4ac`: $b^2 - Four \cdot a \cdot c$

### Physics Equations

- Kinetic energy: $0.5 \cdot m \cdot v^2$
- Force equation: $F = m \cdot a$

### Engineering Tolerances

- Voltage specification: $V \approx 5.0 \pm 0.1$
- Resistance tolerance: $R \pm 5.0\%$
- Temperature range: $20 \leq T \land T \leq 25$

## Parentheses Logic Examples

### Precedence Demonstrations

- Multiplication precedence `latex(x + y * z)`: $x + y \cdot z$
- Addition needs parentheses `latex((x + y) * z)`: $(x + y) \cdot z$
- Subtraction on right `latex(x - (y + z))`: $x - (y + z)$
- Division complexity `latex(x / (y * z))`: $\frac{x}{y \cdot z}$

## Edge Cases

### Special Values

- Zero operations `latex(x + 0)`: $x + 0$
- Negative constants `latex(x + (-5))`: $x + -5$
- Identity operations `latex(x * 1)`: $x \cdot 1$

## Statistical Operations

Mahonia supports statistical operations on data collections with LaTeX mathematical notation:

```python
from mahonia.stats import Mean, StdDev, Median, Percentile, Range, Count

# Define context with sample data
class StatsCtx(NamedTuple):
    measurements: list[float] = [1.0, 2.0, 3.0, 4.0, 5.0]
    voltages: list[float] = [4.95, 5.05, 4.98, 5.02, 5.0]

# Variables for statistical data
measurements = Var[list[float], StatsCtx]('measurements')
voltages = Var[list[float], StatsCtx]('voltages')
```

### Basic Statistical Operations

- Mean (arithmetic average): `latex(Mean(measurements))` → $\bar{measurements}$
- Standard deviation: `latex(StdDev(measurements))` → $\sigma_{measurements}$
- Median: `latex(Median(measurements))` → $\text{median}(measurements)$
- 95th percentile: `latex(Percentile(95, measurements))` → $P_{95}(measurements)$
- Range (max - min): `latex(Range(measurements))` → $\text{range}(measurements)$
- Count: `latex(Count(measurements))` → $|measurements|$

### Statistical Operations with Context

When evaluated with context, statistical operations show data values and results:

- Mean with data: `latex(Mean(measurements), StatsCtx())` → $(\bar{measurements:5[1.0,..5.0]} \rightarrow 3.0)$
- Standard deviation with data: `latex(StdDev(measurements), StatsCtx())` → $(\sigma_{measurements:5[1.0,..5.0]} \rightarrow 1.5811388300841898)$
- Median with data: `latex(Median(measurements), StatsCtx())` → $(\text{median}(measurements:5[1.0,..5.0]) \rightarrow 3.0)$
- 95th percentile with data: `latex(Percentile(95, measurements), StatsCtx())` → $(P_{95}(measurements:5[1.0,..5.0]) \rightarrow 4.8)$
- Range with data: `latex(Range(measurements), StatsCtx())` → $(\text{range}(measurements:5[1.0,..5.0]) \rightarrow 4.0)$
- Count with data: `latex(Count(measurements), StatsCtx())` → $(|measurements:5[1.0,..5.0]| \rightarrow 5)$

### Complex Statistical Expressions

Statistical operations can be combined in complex mathematical expressions:

- Coefficient of variation: `σ/μ` → $\frac{\sigma_{voltages}}{\bar{voltages}}$
- Process capability index: `Cp = (μ - LSL)/(3σ)` → $\frac{\bar{voltages} - 4.9}{3 \cdot \sigma_{voltages}}$

With context evaluation:
- CV with data: $(\frac{(\sigma_{voltages:5[4.95,..5.0]} \rightarrow 0.038078865529319314)}{(\bar{voltages:5[4.95,..5.0]} \rightarrow 5.0)} \rightarrow 0.007615773105863863)$
- Cp with data: $(\frac{((\bar{voltages:5[4.95,..5.0]} \rightarrow 5.0) - 4.9 \rightarrow 0.09999999999999964)}{(3 \cdot (\sigma_{voltages:5[4.95,..5.0]} \rightarrow 0.038078865529319314) \rightarrow 0.11423659658795794)} \rightarrow 0.8753762190648192)$

### Statistical Operations in Quality Control

Statistical operations are commonly used in quality control and process monitoring:

- Quality threshold: `latex(Mean(voltages) > 4.95)` → $\bar{voltages} > 4.95$
- Batch acceptance: `latex((Mean(voltages) > 4.9) & (StdDev(voltages) < 0.1))` → $\bar{voltages} > 4.9 \land \sigma_{voltages} < 0.1$
- Quality predicate: `latex(Predicate('Quality Check', condition))` → $\text{Quality Check}: \bar{voltages} > 4.9 \land \sigma_{voltages} < 0.1$

With context evaluation:
- Quality threshold result: $((\bar{voltages:5[4.95,..5.0]} \rightarrow 5.0) > 4.95 \rightarrow \text{True})$
- Batch acceptance result: $(((\bar{voltages:5[4.95,..5.0]} \rightarrow 5.0) > 4.9 \rightarrow \text{True}) \land ((\sigma_{voltages:5[4.95,..5.0]} \rightarrow 0.038078865529319314) < 0.1 \rightarrow \text{True}) \rightarrow \text{True})$
- Quality predicate result: $(\text{Quality Check}: (((\bar{voltages:5[4.95,..5.0]} \rightarrow 5.0) > 4.9 \rightarrow \text{True}) \land ((\sigma_{voltages:5[4.95,..5.0]} \rightarrow 0.038078865529319314) < 0.1 \rightarrow \text{True}) \rightarrow \text{True}) \rightarrow \text{True})$

### Statistical Operations with Greek Variables

Variable names are automatically converted to Greek letters in LaTeX output:

- Mean of sigma data: `latex(Mean(Var('sigma')))` → $\bar{\sigma}$
- Standard deviation of alpha: `latex(StdDev(Var('alpha')))` → $\sigma_{\alpha}$

With context (using measurements data):
- Mean sigma with data: $(\bar{\sigma:5[1.0,..5.0]} \rightarrow 3.0)$
- StdDev alpha with data: $(\sigma_{\alpha:5[1.0,..5.0]} \rightarrow 1.5811388300841898)$

## Expressions with Context

The `latex()` function supports an optional context parameter that evaluates variables and shows their values:

```python
# Define context with concrete values
ctx = Ctx(x=2.0, y=3.0, z=1.5, w=4.0)

# Without context - shows expression structure
latex(expr)  # Shows symbolic form

# With context - shows values
latex(expr, ctx)  # Shows evaluated values
```

### Variables with Context

- Variable without context: `latex(x)` → $x$
- Variable with context: `latex(x, LatexCtx(ctx))` → $(x:2.0 \rightarrow 2.0)$

### Arithmetic with Context

- Addition: `latex(x + y)` → $x + y$
- Addition with context: `latex(x + y, ctx)` → $(x:2.0 + y:3.0 \rightarrow 5.0)$

- Multiplication: `latex(x * y)` → $x \cdot y$
- Multiplication with context: `latex(x * y, ctx)` → $(x:2.0 \cdot y:3.0 \rightarrow 6.0)$

- Division: `latex(x / y)` → $\frac{x}{y}$
- Division with context: `latex(x / y, ctx)` → $(\frac{x:2.0}{y:3.0} \rightarrow 0.6666666666666666)$

- Power: `latex(x ** 2)` → $x^2$
- Power with context: `latex(x ** 2, ctx)` → $(x:2.0^2 \rightarrow 4.0)$

### Comparisons with Context

- Less than: `latex(x < y)` → $x < y$
- Less than with context: `latex(x < y, ctx)` → $(x:2.0 < y:3.0 \rightarrow \text{True})$

- Equality: `latex(x == 2.0)` → $x = 2.0$
- Equality with context: `latex(x == 2.0, ctx)` → $(x:2.0 = 2.0 \rightarrow \text{True})$

- Complex logic: `latex((x > 1.0) & (y < 5.0))` → $x > 1.0 \land y < 5.0$
- Complex logic with context: `latex((x > 1.0) & (y < 5.0), ctx)` → $((x:2.0 > 1.0 \rightarrow \text{True}) \land (y:3.0 < 5.0 \rightarrow \text{True}) \rightarrow \text{True})$

### Special Expressions with Context

- Approximately: `latex(x ≈ target ± 0.1)` → $x \approx target \pm 0.1$
- Approximately with context: `latex(x ≈ target ± 0.1, ctx)` → $(x:2.0 \approx target \pm 0.1 \rightarrow \text{True})$

- Predicate: `latex(condition: x > 1.5)` → $\text{condition}: x > 1.5$
- Predicate with context: `latex(condition: x > 1.5, ctx)` → $(\text{condition}: (x:2.0 > 1.5 \rightarrow \text{True}) \rightarrow \text{True})$

### Show Flags - Controlling Display Options

The `latex()` function supports Show flags that control what information is displayed:

- `Show.VALUES`: Shows variable values as `x:5`
- `Show.WORK`: Shows the evaluation result as `expr → result`
- `Show.VALUES | Show.WORK` (default): Shows both values and result
- `Show(0)`: Shows only structure (no values or work)

- Structure only: `latex(x + y, LatexCtx(ctx, Show(0)))` → $(x + y \rightarrow 5.0)$
- VALUES only: `latex(x + y, LatexCtx(ctx, Show.VALUES))` → $(x:2.0 + y:3.0 \rightarrow 5.0)$
- WORK only: `latex(x + y, LatexCtx(ctx, Show.WORK))` → $(x + y \rightarrow 5.0)$
- Both (default): `latex(x + y, LatexCtx(ctx))` → $(x:2.0 + y:3.0 \rightarrow 5.0)$

**Boolean Expressions:**
- Structure: `latex(x < y, LatexCtx(ctx, Show(0)))` → $(x < y \rightarrow \text{True})$
- VALUES: `latex(x < y, LatexCtx(ctx, Show.VALUES))` → $(x:2.0 < y:3.0 \rightarrow \text{True})$
- WORK: `latex(x < y, LatexCtx(ctx, Show.WORK))` → $(x < y \rightarrow \text{True})$
- Both: `latex(x < y, LatexCtx(ctx))` → $(x:2.0 < y:3.0 \rightarrow \text{True})$

**Complex Expressions:**
- Structure: `latex((x + z) * y, LatexCtx(ctx, Show(0)))` → $((x + z) \cdot y \rightarrow 10.5)$
- VALUES: `latex((x + z) * y, LatexCtx(ctx, Show.VALUES))` → $((x:2.0 + z:1.5) \cdot y:3.0 \rightarrow 10.5)$
- Both: `latex((x + z) * y, LatexCtx(ctx))` → $(((x:2.0 + z:1.5 \rightarrow 3.5)) \cdot y:3.0 \rightarrow 10.5)$

---

*Generated by the mahonia LaTeX module test suite*

To regenerate this file, run: `python -m pytest tests/test_latex_examples.py::test_generate_latex_examples`