# Copyright (c) 2025 JP Hutchins
# SPDX-License-Identifier: MIT

"""
Test that generates LaTeX examples documentation.

This test generates the latex_examples.md file by executing actual Python code
and capturing the LaTeX output. This ensures the documentation stays in sync
with the implementation.
"""

from pathlib import Path
from typing import Any, NamedTuple, cast

from mahonia import (
	Approximately,
	Const,
	Expr,
	Percent,
	PlusMinus,
	Predicate,
	Var,
)
from mahonia.latex import LatexCtx, Show, latex
from mahonia.stats import Count, Mean, Median, Percentile, Range, SizedIterable, StdDev


def test_generate_latex_examples() -> None:
	"""Generate the LaTeX examples markdown file."""

	# Define context for examples
	class Ctx(NamedTuple):
		x: float
		y: float
		z: float
		w: float
		a: float
		b: float
		c: float
		v: float

	# Define statistical context for statistical operations
	class StatsCtx(NamedTuple):
		measurements: SizedIterable[float] = [1.0, 2.0, 3.0, 4.0, 5.0]
		voltages: SizedIterable[float] = [4.95, 5.05, 4.98, 5.02, 5.0]
		batch_sizes: SizedIterable[int] = [10, 12, 8, 15, 11]

	# Start building the markdown content
	lines = [
		"# LaTeX Examples from Mahonia",
		"",
		"This document demonstrates the LaTeX output generated by the `mahonia.latex` module for various mathematical expressions.",
		"",
		"> [!IMPORTANT]",
		"> This file is auto-generated by `tests/test_latex_examples.py`",
		"> Do not edit this file directly. Run the test to regenerate.",
		"",
		"## Setup",
		"",
		"All examples use the following setup:",
		"",
		"```python",
		"from typing import NamedTuple",
		"from mahonia import Var, Const, PlusMinus, Percent, Approximately, Predicate",
		"from mahonia.latex import latex",
		"",
		"class Ctx(NamedTuple):",
		"    x: float",
		"    y: float",
		"    # ... other fields as needed",
		"```",
		"",
		"## Basic Variables",
		"",
		"### Simple Variables",
	]

	# Basic variables
	x = Var[int, Ctx]("x")
	y = Var[int, Ctx]("y")
	lines.extend(
		[
			f"- Variable `x`: `latex(Var[int, Ctx]('x'))` → ${latex(x)}$",
			f"- Variable `y`: `latex(Var[int, Ctx]('y'))` → ${latex(y)}$",
			"",
		]
	)

	# Subscripts
	lines.extend(
		[
			"### Variables with Subscripts",
			"",
		]
	)

	x1 = Var[int, Ctx]("x_1")
	x_max = Var[int, Ctx]("x_max")
	velocity_initial = Var[float, Ctx]("velocity_initial")

	lines.extend(
		[
			f"- Single character subscript: `latex(Var('x_1'))` → ${latex(x1)}$",
			f"- Multiple character subscript: `latex(Var('x_max'))` → ${latex(x_max)}$",
			f"- Complex subscript: `latex(Var('velocity_initial'))` → ${latex(velocity_initial)}$",
			"",
		]
	)

	# Greek letters
	lines.extend(
		[
			"### Greek Letter Variables",
			"",
		]
	)

	greek_vars = [
		("alpha", "α"),
		("beta", "β"),
		("gamma", "γ"),
		("delta", "δ"),
		("epsilon", "ε"),
		("theta", "θ"),
		("pi", "π"),
		("sigma", "σ"),
		("omega", "ω"),
	]

	for name, _symbol in greek_vars:
		var = Var[float, Ctx](name)
		lines.append(f"- {name.capitalize()}: `latex(Var('{name}'))` → ${latex(var)}$")
	lines.extend(["", ""])

	# Constants
	lines.extend(
		[
			"## Constants",
			"",
			"### Numeric Constants",
			"",
		]
	)

	const_examples = [
		(Const(None, 42), "Const(None, 42)", "Integer"),
		(Const(None, 3.14159), "Const(None, 3.14159)", "Float"),
		(Const(None, -5), "Const(None, -5)", "Negative"),
	]

	for const, code, desc in const_examples:
		lines.append(f"- {desc}: `latex({code})` → ${latex(cast(Expr[Any, Any], const))}$")
	lines.extend(["", ""])

	# Named constants
	lines.extend(
		[
			"### Named Constants",
			"",
		]
	)

	pi_const = Const("Pi", 3.14159)  # This becomes \pi due to Greek letter conversion
	zero_const = Const("Zero", 0)

	named_examples = [
		(pi_const, "Const('Pi', 3.14159)", "Named constant Pi (becomes Greek π)"),
		(zero_const, "Const('Zero', 0)", "Zero constant"),
	]

	for const, code, desc in named_examples:
		lines.append(f"- {desc}: `latex({code})` → ${latex(cast(Expr[Any, Any], const))}$")
	lines.extend(["", ""])

	# Arithmetic Operations
	lines.extend(
		[
			"## Arithmetic Operations",
			"",
			"### Basic Operations",
			"",
		]
	)

	x_float = Var[float, Ctx]("x")
	y_float = Var[float, Ctx]("y")

	lines.extend(
		[
			f"- Addition: `latex(x + y)` → ${latex(x + y)}$",
			f"- Subtraction: `latex(x - y)` → ${latex(x - y)}$",
			f"- Multiplication: `latex(x * y)` → ${latex(x_float * y_float)}$",
			f"- Division: `latex(x / y)` → ${latex(x_float / y_float)}$",
			"",
		]
	)

	# Power Operations
	lines.extend(
		[
			"### Power Operations",
			"",
			f"- Square: `latex(x ** 2)` → ${latex(x**2)}$",
			f"- Variable exponent: `latex(x ** y)` → ${latex(x**y)}$",
			f"- Complex exponent: `latex(x ** (y + 1))` → ${latex(x ** (y + 1))}$",
			f"- Base with parentheses: `latex((x + y) ** 2)` → ${latex((x + y) ** 2)}$",
			"",
		]
	)

	# Complex Arithmetic
	lines.extend(
		[
			"### Complex Arithmetic",
			"",
		]
	)

	z_float = Var[float, Ctx]("z")
	complex_examples = [
		(
			((x_float + y_float) * z_float) / (x_float - y_float),
			"((x + y) * z) / (x - y)",
			"Nested expression",
		),
		(
			x_float**3 + Const("Two", 2) * x_float**2 - x_float + Const("One", 1),
			"x**3 + 2*x**2 - x + 1",
			"Polynomial",
		),
	]

	for expr, code, desc in complex_examples:
		if "polynomial" in desc.lower():
			lines.extend(
				[
					"```python",
					"x = Var[float, Ctx]('x')",
					"two = Const('Two', 2)",
					"one = Const('One', 1)",
					"expr = x**3 + two*x**2 - x + one",
					f"latex(expr)  # '{latex(cast(Expr[Any, Any], expr))}'",
					"```",
				]
			)
		else:
			lines.extend(
				[
					"```python",
					"x = Var[float, Ctx]('x')",
					"y = Var[float, Ctx]('y')",
					"z = Var[float, Ctx]('z')",
					f"expr = {code}",
					f"latex(expr)  # '{latex(cast(Expr[Any, Any], expr))}'",
					"```",
				]
			)
		lines.extend(
			[
				f"- {desc}: ${latex(cast(Expr[Any, Any], expr))}$",
				"",
			]
		)

	# Comparison Operations
	lines.extend(
		[
			"## Comparison Operations",
			"",
			"### Basic Comparisons",
			"",
			f"- Equal `latex(x == y)`: ${latex(x == y)}$",
			f"- Not equal `latex(x != y)`: ${latex(x != y)}$",
			f"- Less than `latex(x < y)`: ${latex(x < y)}$",
			f"- Less than or equal `latex(x <= y)`: ${latex(x <= y)}$",
			f"- Greater than `latex(x > y)`: ${latex(x > y)}$",
			f"- Greater than or equal `latex(x >= y)`: ${latex(x >= y)}$",
			"",
			"### Complex Comparisons",
			"",
			f"- With arithmetic `latex((x + 5) > (y * 2))`: ${latex((x + 5) > (y * 2))}$",
			"",
		]
	)

	# Logical Operations
	lines.extend(
		[
			"## Logical Operations",
			"",
			"### Basic Logic",
			"",
			f"- AND `latex((x > 5) & (y < 10))`: ${latex((x > 5) & (y < 10))}$",
			f"- OR `latex((x > 5) | (y < 10))`: ${latex((x > 5) | (y < 10))}$",
			f"- NOT `latex(~(x > 5))`: ${latex(~(x > 5))}$",
			"",
			"### Complex Logic",
			"",
			f"- Nested NOT: ${latex(~((x > 5) & (y < 10)))}$",
			f"- Mixed operations: ${latex(((x_float + y_float) > 10) & ((x_float * y_float) < 50) | (x_float / y_float > 2))}$",
			"",
		]
	)

	# Special Expressions
	pm1 = PlusMinus("measurement", 5.0, 0.1)
	pm2 = PlusMinus(None, 5.0, 0.1)
	pct1 = Percent("measurement", 5.0, 2.0)
	pct2 = Percent(None, 5.0, 2.0)

	x_approx = Var[float, Ctx]("x")
	target1 = PlusMinus("target", 5.0, 0.1)
	target2 = PlusMinus("target2", 10.0, 0.2)
	approx1 = Approximately(x_approx, target1)
	approx2 = Approximately(x_approx ** Const(None, 2.0), target2)

	pred1 = Predicate("condition", x > 5)
	pred2 = Predicate(None, x > 5)

	lines.extend(
		[
			"## Special Expressions",
			"",
			"### Plus-Minus Expressions",
			"",
			f"- Named measurement: ${latex(pm1)}$",
			f"- Unnamed measurement: ${latex(pm2)}$",
			"",
			"### Percentage Expressions",
			"",
			f"- Named percentage: ${latex(pct1)}$",
			f"- Unnamed percentage: ${latex(pct2)}$",
			"",
			"### Approximately Expressions",
			"",
			f"- Basic approximation: ${latex(approx1)}$",
			f"- Complex approximation: ${latex(approx2)}$",
			"",
			"### Predicate Expressions",
			"",
			f"- Named predicate: ${latex(pred1)}$",
			f"- Unnamed predicate: ${latex(pred2)}$",
			"",
		]
	)

	# Real-World Examples
	a = Var[float, Ctx]("a")
	b = Var[float, Ctx]("b")
	c = Var[float, Ctx]("c")
	discriminant = b**2 - Const("Four", 4) * a * c

	m = Var[float, Ctx]("m")
	v = Var[float, Ctx]("v")
	F = Var[float, Ctx]("F")
	half = Const(None, 0.5)
	kinetic_energy = half * m * v**2
	force_eq = F == m * a

	V = Var[float, Ctx]("V")
	voltage_spec = Approximately(V, PlusMinus(None, 5.0, 0.1))
	resistance_tol = Percent("R", 100.0, 5.0)
	T = Var[float, Ctx]("T")
	temp_range = (Const(None, 20) <= T) & (T <= Const(None, 25))

	lines.extend(
		[
			"## Real-World Examples",
			"",
			"### Quadratic Formula Components",
			"",
			f"- Discriminant `b² - 4ac`: ${latex(discriminant)}$",
			"",
			"### Physics Equations",
			"",
			f"- Kinetic energy: ${latex(kinetic_energy)}$",
			f"- Force equation: ${latex(force_eq)}$",
			"",
			"### Engineering Tolerances",
			"",
			f"- Voltage specification: ${latex(voltage_spec)}$",
			f"- Resistance tolerance: ${latex(resistance_tol)}$",
			f"- Temperature range: ${latex(temp_range)}$",
			"",
		]
	)

	# Parentheses Logic Examples
	z_int = Var[int, Ctx]("z")

	lines.extend(
		[
			"## Parentheses Logic Examples",
			"",
			"### Precedence Demonstrations",
			"",
			f"- Multiplication precedence `latex(x + y * z)`: ${latex(x + y * z_int)}$",
			f"- Addition needs parentheses `latex((x + y) * z)`: ${latex((x + y) * z_int)}$",
			f"- Subtraction on right `latex(x - (y + z))`: ${latex(x - (y + z_int))}$",
			f"- Division complexity `latex(x / (y * z))`: ${latex(x_float / (y_float * z_float))}$",
			"",
		]
	)

	# Edge Cases
	zero = Const(None, 0)
	one = Const(None, 1)
	neg_five = Const(None, -5)

	lines.extend(
		[
			"## Edge Cases",
			"",
			"### Special Values",
			"",
			f"- Zero operations `latex(x + 0)`: ${latex(x + zero)}$",
			f"- Negative constants `latex(x + (-5))`: ${latex(x + neg_five)}$",
			f"- Identity operations `latex(x * 1)`: ${latex(x * one)}$",
			"",
		]
	)

	# Statistical Operations Examples - NEW SECTION
	stats_ctx = StatsCtx()

	lines.extend(
		[
			"## Statistical Operations",
			"",
			"Mahonia supports statistical operations on data collections with LaTeX mathematical notation:",
			"",
			"```python",
			"from mahonia.stats import Mean, StdDev, Median, Percentile, Range, Count",
			"",
			"# Define context with sample data",
			"class StatsCtx(NamedTuple):",
			"    measurements: list[float] = [1.0, 2.0, 3.0, 4.0, 5.0]",
			"    voltages: list[float] = [4.95, 5.05, 4.98, 5.02, 5.0]",
			"",
			"# Variables for statistical data",
			"measurements = Var[list[float], StatsCtx]('measurements')",
			"voltages = Var[list[float], StatsCtx]('voltages')",
			"```",
			"",
		]
	)

	# Statistical operations examples
	measurements = Var[SizedIterable[float], StatsCtx]("measurements")
	voltages = Var[SizedIterable[float], StatsCtx]("voltages")

	# Individual operations
	mean_measurements = Mean(measurements)
	stddev_measurements = StdDev(measurements)
	median_measurements = Median(measurements)
	p95_measurements = Percentile(95, measurements)
	range_measurements = Range(measurements)
	count_measurements = Count(measurements)

	lines.extend(
		[
			"### Basic Statistical Operations",
			"",
			f"- Mean (arithmetic average): `latex(Mean(measurements))` → ${latex(mean_measurements)}$",
			f"- Standard deviation: `latex(StdDev(measurements))` → ${latex(stddev_measurements)}$",
			f"- Median: `latex(Median(measurements))` → ${latex(median_measurements)}$",
			f"- 95th percentile: `latex(Percentile(95, measurements))` → ${latex(p95_measurements)}$",
			f"- Range (max - min): `latex(Range(measurements))` → ${latex(range_measurements)}$",
			f"- Count: `latex(Count(measurements))` → ${latex(count_measurements)}$",
			"",
		]
	)

	lines.extend(
		[
			"### Statistical Operations with Context",
			"",
			"When evaluated with context, statistical operations show data values and results:",
			"",
			f"- Mean with data: `latex(Mean(measurements), StatsCtx())` → ${latex(mean_measurements, LatexCtx(stats_ctx))}$",
			f"- Standard deviation with data: `latex(StdDev(measurements), StatsCtx())` → ${latex(stddev_measurements, LatexCtx(stats_ctx))}$",
			f"- Median with data: `latex(Median(measurements), StatsCtx())` → ${latex(median_measurements, LatexCtx(stats_ctx))}$",
			f"- 95th percentile with data: `latex(Percentile(95, measurements), StatsCtx())` → ${latex(p95_measurements, LatexCtx(stats_ctx))}$",
			f"- Range with data: `latex(Range(measurements), StatsCtx())` → ${latex(range_measurements, LatexCtx(stats_ctx))}$",
			f"- Count with data: `latex(Count(measurements), StatsCtx())` → ${latex(count_measurements, LatexCtx(stats_ctx))}$",
			"",
		]
	)

	# Complex statistical expressions
	mean_voltages = Mean(voltages)
	stddev_voltages = StdDev(voltages)
	coefficient_of_variation = stddev_voltages / mean_voltages
	three = Const(None, 3)
	four_nine = Const(None, 4.9)
	process_capability = (mean_voltages - four_nine) / (three * stddev_voltages)

	lines.extend(
		[
			"### Complex Statistical Expressions",
			"",
			"Statistical operations can be combined in complex mathematical expressions:",
			"",
			f"- Coefficient of variation: `σ/μ` → ${latex(coefficient_of_variation)}$",
			f"- Process capability index: `Cp = (μ - LSL)/(3σ)` → ${latex(process_capability)}$",
			"",
			"With context evaluation:",
			f"- CV with data: ${latex(coefficient_of_variation, LatexCtx(stats_ctx))}$",
			f"- Cp with data: ${latex(process_capability, LatexCtx(stats_ctx))}$",
			"",
		]
	)

	# Statistical operations in comparisons and predicates
	four_ninefive = Const(None, 4.95)
	zero_one = Const(None, 0.1)
	quality_check = mean_voltages > four_ninefive
	batch_acceptable = (mean_voltages > four_nine) & (stddev_voltages < zero_one)
	quality_predicate = Predicate("Quality Check", batch_acceptable)

	lines.extend(
		[
			"### Statistical Operations in Quality Control",
			"",
			"Statistical operations are commonly used in quality control and process monitoring:",
			"",
			f"- Quality threshold: `latex(Mean(voltages) > 4.95)` → ${latex(quality_check)}$",
			f"- Batch acceptance: `latex((Mean(voltages) > 4.9) & (StdDev(voltages) < 0.1))` → ${latex(batch_acceptable)}$",
			f"- Quality predicate: `latex(Predicate('Quality Check', condition))` → ${latex(quality_predicate)}$",
			"",
			"With context evaluation:",
			f"- Quality threshold result: ${latex(quality_check, LatexCtx(stats_ctx))}$",
			f"- Batch acceptance result: ${latex(batch_acceptable, LatexCtx(stats_ctx))}$",
			f"- Quality predicate result: ${latex(quality_predicate, LatexCtx(stats_ctx))}$",
			"",
		]
	)

	# Statistical operations with Greek variables - create a context with Greek names
	class GreekStatsCtx(NamedTuple):
		sigma: SizedIterable[float] = [1.0, 2.0, 3.0, 4.0, 5.0]
		alpha: SizedIterable[float] = [1.0, 2.0, 3.0, 4.0, 5.0]

	greek_stats_ctx = GreekStatsCtx()
	sigma_data = Var[SizedIterable[float], GreekStatsCtx]("sigma")
	alpha_data = Var[SizedIterable[float], GreekStatsCtx]("alpha")
	mean_sigma = Mean(sigma_data)
	stddev_alpha = StdDev(alpha_data)

	lines.extend(
		[
			"### Statistical Operations with Greek Variables",
			"",
			"Variable names are automatically converted to Greek letters in LaTeX output:",
			"",
			f"- Mean of sigma data: `latex(Mean(Var('sigma')))` → ${latex(mean_sigma)}$",
			f"- Standard deviation of alpha: `latex(StdDev(Var('alpha')))` → ${latex(stddev_alpha)}$",
			"",
			"With context (using measurements data):",
			f"- Mean sigma with data: ${latex(mean_sigma, LatexCtx(greek_stats_ctx))}$",
			f"- StdDev alpha with data: ${latex(stddev_alpha, LatexCtx(greek_stats_ctx))}$",
			"",
		]
	)

	# Context Examples - NEW SECTION
	# Define a concrete context for examples
	ctx = Ctx(x=2.0, y=3.0, z=1.5, w=4.0, a=1.0, b=2.0, c=3.0, v=5.0)

	lines.extend(
		[
			"## Expressions with Context",
			"",
			"The `latex()` function supports an optional context parameter that evaluates variables and shows their values:",
			"",
			"```python",
			"# Define context with concrete values",
			"ctx = Ctx(x=2.0, y=3.0, z=1.5, w=4.0)",
			"",
			"# Without context - shows expression structure",
			"latex(expr)  # Shows symbolic form",
			"",
			"# With context - shows values",
			"latex(expr, ctx)  # Shows evaluated values",
			"```",
			"",
		]
	)

	# Context examples for variables
	x_ctx = Var[float, Ctx]("x")
	y_ctx = Var[float, Ctx]("y")
	z_ctx = Var[float, Ctx]("z")

	lines.extend(
		[
			"### Variables with Context",
			"",
			f"- Variable without context: `latex(x)` → ${latex(x_ctx)}$",
			f"- Variable with context: `latex(x, LatexCtx(ctx))` → ${latex(x_ctx, LatexCtx(ctx))}$",
			"",
		]
	)

	# Context examples for arithmetic
	lines.extend(
		[
			"### Arithmetic with Context",
			"",
		]
	)

	add_ctx = x_ctx + y_ctx
	mul_ctx = x_ctx * y_ctx
	div_ctx = x_ctx / y_ctx
	pow_ctx = x_ctx**2

	lines.extend(
		[
			f"- Addition: `latex(x + y)` → ${latex(add_ctx)}$",
			f"- Addition with context: `latex(x + y, ctx)` → ${latex(add_ctx, LatexCtx(ctx))}$",
			"",
			f"- Multiplication: `latex(x * y)` → ${latex(mul_ctx)}$",
			f"- Multiplication with context: `latex(x * y, ctx)` → ${latex(mul_ctx, LatexCtx(ctx))}$",
			"",
			f"- Division: `latex(x / y)` → ${latex(div_ctx)}$",
			f"- Division with context: `latex(x / y, ctx)` → ${latex(div_ctx, LatexCtx(ctx))}$",
			"",
			f"- Power: `latex(x ** 2)` → ${latex(pow_ctx)}$",
			f"- Power with context: `latex(x ** 2, ctx)` → ${latex(pow_ctx, LatexCtx(ctx))}$",
			"",
		]
	)

	# Context examples for comparisons and logic
	lines.extend(
		[
			"### Comparisons with Context",
			"",
		]
	)

	lt_ctx = x_ctx < y_ctx
	eq_ctx = x_ctx == 2.0
	complex_logic = (x_ctx > 1.0) & (y_ctx < 5.0)

	lines.extend(
		[
			f"- Less than: `latex(x < y)` → ${latex(lt_ctx)}$",
			f"- Less than with context: `latex(x < y, ctx)` → ${latex(lt_ctx, LatexCtx(ctx))}$",
			"",
			f"- Equality: `latex(x == 2.0)` → ${latex(eq_ctx)}$",
			f"- Equality with context: `latex(x == 2.0, ctx)` → ${latex(eq_ctx, LatexCtx(ctx))}$",
			"",
			f"- Complex logic: `latex((x > 1.0) & (y < 5.0))` → ${latex(complex_logic)}$",
			f"- Complex logic with context: `latex((x > 1.0) & (y < 5.0), ctx)` → ${latex(complex_logic, LatexCtx(ctx))}$",
			"",
		]
	)

	# Context examples for special expressions
	lines.extend(
		[
			"### Special Expressions with Context",
			"",
		]
	)

	approx_ctx = Approximately(x_ctx, PlusMinus("target", 2.0, 0.1))
	pred_ctx = Predicate("condition", x_ctx > 1.5)

	lines.extend(
		[
			f"- Approximately: `latex(x ≈ target ± 0.1)` → ${latex(approx_ctx)}$",
			f"- Approximately with context: `latex(x ≈ target ± 0.1, ctx)` → ${latex(approx_ctx, LatexCtx(ctx))}$",
			"",
			f"- Predicate: `latex(condition: x > 1.5)` → ${latex(pred_ctx)}$",
			f"- Predicate with context: `latex(condition: x > 1.5, ctx)` → ${latex(pred_ctx, LatexCtx(ctx))}$",
			"",
		]
	)

	# NEW SECTION: Show flags functionality
	lines.extend(
		[
			"### Show Flags - Controlling Display Options",
			"",
			"The `latex()` function supports Show flags that control what information is displayed:",
			"",
			"- `Show.VALUES`: Shows variable values as `x:5`",
			"- `Show.WORK`: Shows the evaluation result as `expr → result`",
			"- `Show.VALUES | Show.WORK` (default): Shows both values and result",
			"- `Show(0)`: Shows only structure (no values or work)",
			"",
		]
	)

	# Examples of show_values
	add_show = x_ctx + y_ctx
	comp_show = x_ctx < y_ctx
	complex_show = (x_ctx + z_ctx) * y_ctx

	lines.extend(
		[
			f"- Structure only: `latex(x + y, LatexCtx(ctx, Show(0)))` → ${latex(add_show, LatexCtx(ctx, Show(0)))}$",
			f"- VALUES only: `latex(x + y, LatexCtx(ctx, Show.VALUES))` → ${latex(add_show, LatexCtx(ctx, Show.VALUES))}$",
			f"- WORK only: `latex(x + y, LatexCtx(ctx, Show.WORK))` → ${latex(add_show, LatexCtx(ctx, Show.WORK))}$",
			f"- Both (default): `latex(x + y, LatexCtx(ctx))` → ${latex(add_show, LatexCtx(ctx))}$",
			"",
			"**Boolean Expressions:**",
			f"- Structure: `latex(x < y, LatexCtx(ctx, Show(0)))` → ${latex(comp_show, LatexCtx(ctx, Show(0)))}$",
			f"- VALUES: `latex(x < y, LatexCtx(ctx, Show.VALUES))` → ${latex(comp_show, LatexCtx(ctx, Show.VALUES))}$",
			f"- WORK: `latex(x < y, LatexCtx(ctx, Show.WORK))` → ${latex(comp_show, LatexCtx(ctx, Show.WORK))}$",
			f"- Both: `latex(x < y, LatexCtx(ctx))` → ${latex(comp_show, LatexCtx(ctx))}$",
			"",
			"**Complex Expressions:**",
			f"- Structure: `latex((x + z) * y, LatexCtx(ctx, Show(0)))` → ${latex(complex_show, LatexCtx(ctx, Show(0)))}$",
			f"- VALUES: `latex((x + z) * y, LatexCtx(ctx, Show.VALUES))` → ${latex(complex_show, LatexCtx(ctx, Show.VALUES))}$",
			f"- Both: `latex((x + z) * y, LatexCtx(ctx))` → ${latex(complex_show, LatexCtx(ctx))}$",
			"",
		]
	)

	# Footer
	lines.extend(
		[
			"---",
			"",
			"*Generated by the mahonia LaTeX module test suite*",
			"",
			"To regenerate this file, run: `python -m pytest tests/test_latex_examples.py::test_generate_latex_examples`",
		]
	)

	# Write the markdown file
	output_path = Path(__file__).parent / "latex_examples.md"
	with open(output_path, "w", encoding="utf-8") as f:
		f.write("\n".join(lines))

	print(f"Generated LaTeX examples at {output_path}")


if __name__ == "__main__":
	test_generate_latex_examples()
