"""
Mock dashboard module for CLI migrations.

Smart auto-mocking: ANY import from dashboard.* will return a valid mock.
No need to manually list every possible import!

These mocks are ONLY for makemigrations - they don't need to be functional.
"""

import sys
from types import ModuleType


class UniversalMock:
    """
    A TRULY universal mock that supports ANY operation Python can throw at it.
    
    Supports:
    - Calling (functions/constructors)
    - Decorating (@decorator)
    - Inheriting (base classes)
    - Attribute access
    - Iteration
    - Length
    - Boolean context
    - String operations
    - Arithmetic operations
    - Comparison operations
    - Context managers
    - And more!
    """
    
    def __init__(self, name="UniversalMock"):
        self._name = name
    
    def __call__(self, *args, **kwargs):
        # Handle decorator usage: @mock or @mock()
        if len(args) == 1 and callable(args[0]) and not kwargs:
            return args[0]  # Pass through decorated function/class
        return UniversalMock(f"{self._name}()")
    
    def __getattr__(self, name):
        # Any attribute access returns another mock
        return UniversalMock(f"{self._name}.{name}")
    
    def __getitem__(self, key):
        # Support indexing: mock[0], mock['key']
        return UniversalMock(f"{self._name}[{key}]")
    
    def __setitem__(self, key, value):
        # Support assignment: mock[key] = value
        pass
    
    def __delitem__(self, key):
        # Support deletion: del mock[key]
        pass
    
    def __len__(self):
        # Support len(): len(mock)
        return 0
    
    def __bool__(self):
        # Support truthiness: if mock:
        return True
    
    def __iter__(self):
        # Support iteration: for x in mock:
        return iter([])
    
    def __contains__(self, item):
        # Support membership: item in mock
        return False
    
    def __enter__(self):
        # Support context manager: with mock:
        return self
    
    def __exit__(self, *args):
        # Support context manager cleanup
        return False
    
    def __str__(self):
        return f"<Mock: {self._name}>"
    
    def __repr__(self):
        return f"<UniversalMock: {self._name}>"
    
    def __mro_entries__(self, bases):
        # Support being used as a base class
        # Return empty tuple = skip this base entirely (no-op mixin)
        return ()
    
    # Arithmetic operations
    def __add__(self, other):
        return UniversalMock(f"{self._name}+{other}")
    
    def __sub__(self, other):
        return UniversalMock(f"{self._name}-{other}")
    
    def __mul__(self, other):
        return UniversalMock(f"{self._name}*{other}")
    
    def __truediv__(self, other):
        return UniversalMock(f"{self._name}/{other}")
    
    # Comparison operations
    def __eq__(self, other):
        return False
    
    def __ne__(self, other):
        return True
    
    def __lt__(self, other):
        return False
    
    def __le__(self, other):
        return False
    
    def __gt__(self, other):
        return False
    
    def __ge__(self, other):
        return False


class SmartModule(ModuleType):
    """Module that auto-generates mocks for any attribute"""
    
    def __getattr__(self, name):
        # Return a universal mock for any import
        mock = UniversalMock(f"{self.__name__}.{name}")
        # Cache it so repeated imports get the same object
        setattr(self, name, mock)
        return mock


# Create smart modules
dashboard = SmartModule('dashboard')
dashboard.veloz_sdk = SmartModule('dashboard.veloz_sdk')
dashboard.models = SmartModule('dashboard.models')
dashboard.views = SmartModule('dashboard.views')

# Pre-populate common SDK objects (for better behavior)
dashboard.veloz_sdk.plugin = UniversalMock('plugin')
dashboard.veloz_sdk.agent = UniversalMock('agent')
dashboard.veloz_sdk.route = UniversalMock('route')
dashboard.veloz_sdk.menu = UniversalMock('menu')
dashboard.veloz_sdk.function_tool = UniversalMock('function_tool')
dashboard.veloz_sdk.ai = UniversalMock('ai')
dashboard.veloz_sdk.log = UniversalMock('log')
dashboard.veloz_sdk.SingletonMixin = UniversalMock('SingletonMixin')

# Create flowpress_dashboard mocks (settings module)
flowpress_dashboard = SmartModule('flowpress_dashboard')
flowpress_dashboard.settings = SmartModule('flowpress_dashboard.settings')
flowpress_dashboard.settings.base = SmartModule('flowpress_dashboard.settings.base')
flowpress_dashboard.urls = SmartModule('flowpress_dashboard.urls')

# Register in sys.modules
sys.modules['dashboard'] = dashboard
sys.modules['dashboard.veloz_sdk'] = dashboard.veloz_sdk
sys.modules['dashboard.models'] = dashboard.models
sys.modules['dashboard.views'] = dashboard.views
sys.modules['flowpress_dashboard'] = flowpress_dashboard
sys.modules['flowpress_dashboard.settings'] = flowpress_dashboard.settings
sys.modules['flowpress_dashboard.settings.base'] = flowpress_dashboard.settings.base
sys.modules['flowpress_dashboard.urls'] = flowpress_dashboard.urls

