#!/usr/bin/env python3
"""
leakify - aggressive downloader for leakedzone.com

Goal: If the script "found N videos" we actually attempt to download ALL N with
multiple fallbacks and aggressive retries.

Usage:
  leakify -u username --videos
  leakify -u username --photos
  leakify -u username --photos --videos
  leakify -u username --videos --cookies cookies.txt

Notes:
- Requires ffmpeg in PATH.
- yt-dlp if present will be used as an additional fallback; install it for best coverage.
- If site requires auth, provide cookies file with --cookies path (Netscape / curl style).
"""

import argparse
import asyncio
import os
import re
import sys
import uuid
import base64
import shutil
import time
from pathlib import Path
from urllib.parse import urljoin, urlparse
from datetime import datetime

import aiohttp
from bs4 import BeautifulSoup
from subprocess import PIPE
from tabulate import tabulate
from colorama import Fore, Style, init as colorama_init

# Initialize colorama
colorama_init(autoreset=True)

API_BASE = "https://leakedzone.com"
DEFAULT_HEADERS = {
    "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36",
    "X-Requested-With": "XMLHttpRequest",
}

MIN_VALID_BYTES = 30_000  # minimal size to consider a file "real" (30KB)
MAX_RETRIES = 6
FFMPEG_TIMEOUT = 600  # seconds
YTDLP_TIMEOUT = 600
AIO_TIMEOUT = 300

def ensure_ffmpeg():
    if shutil.which('ffmpeg') is None:
        print(f"{Fore.RED}Error: 'ffmpeg' not found in PATH. Install ffmpeg and re-run.{Style.RESET_ALL}")
        sys.exit(1)

def find_yt_dlp_binary():
    for name in ("yt-dlp", "yt-dlp.exe", "youtube-dl", "youtube-dl.exe"):
        if shutil.which(name):
            return name
    return None

def update_table(model: str, mode: str, status: str):
    """Clear screen and display status table."""
    os.system('cls' if os.name == 'nt' else 'clear')
    title = f"{Fore.GREEN}{Style.BRIGHT}{'Leakify V2.1'.center(70)}{Style.RESET_ALL}"
    print(title, "\n")
    headers = [
        f"{Style.BRIGHT}Model{Style.RESET_ALL}",
        f"{Style.BRIGHT}Mode{Style.RESET_ALL}",
        f"{Style.BRIGHT}Status{Style.RESET_ALL}"
    ]
    row = [model, mode, status]
    table = tabulate(
        [row],
        headers=headers,
        tablefmt="fancy_grid",
        stralign="center",
        numalign="right"
    )
    print(f"{Fore.CYAN}{table}{Style.RESET_ALL}")

async def fetch_photo_urls(session: aiohttp.ClientSession, model: str):
    """Fetch full-size photo URLs for a model via JSON pages."""
    thumb_re = re.compile(r"https://image-cdn\.leakedzone\.com/.+_300\.(jpg|webp)$")
    urls, page = [], 1
    while True:
        url = f"{API_BASE}/{model}/photo?page={page}&type=photos&order=0"
        update_table(model, "photos", f"GET {url}")
        try:
            async with session.get(url, timeout=30) as resp:
                if resp.status != 200:
                    break
                data = await resp.json()
        except Exception:
            break
        if not data:
            break
        for item in data:
            for v in item.values():
                if isinstance(v, str) and thumb_re.match(v):
                    urls.append(v.replace("_300.", "."))
        page += 1
    return list(dict.fromkeys(urls))

async def fetch_video_urls(session: aiohttp.ClientSession, model: str, batch: int):
    """
    Fetch all likely video URLs. For each item:
    - Try to decode stream_url_play token.
    - Collect detail page URLs for additional scraping.
    """
    page_urls, hls_or_direct_urls, page = [], [], 1
    while True:
        url = f"{API_BASE}/{model}/video?page={page}&type=videos&order=0"
        update_table(model, "videos", f"GET {url}")
        try:
            async with session.get(url, timeout=30) as resp:
                if resp.status != 200:
                    break
                data = await resp.json()
        except Exception:
            break
        if not data:
            break
        for item in data:
            # decode stream_url_play token if present
            tok = item.get("stream_url_play", "")
            if tok and len(tok) > 32:
                core = tok[16:-16][::-1]
                try:
                    dec = base64.b64decode(core).decode(errors="ignore")
                    link = dec if dec.startswith("http") else f"https://cdn32.leakedzone.com/{dec}"
                    hls_or_direct_urls.append(link)
                except Exception:
                    pass
            # collect detail pages
            for v in item.values():
                if isinstance(v, str) and v.startswith(f"/{model}/video/"):
                    page_urls.append(urljoin(API_BASE, v))
        page += 1

    # Scrape detail pages using shared session (bounded concurrency)
    sem = asyncio.Semaphore(max(5, batch // 2))

    async def scrape_detail(u):
        async with sem:
            try:
                async with session.get(u, timeout=40) as resp:
                    if resp.status != 200:
                        return None
                    text = await resp.text()
            except Exception:
                return None
            soup = BeautifulSoup(text, "html.parser")
            # Try <video><source src=...>
            vid = soup.find("video")
            if vid:
                src_tag = vid.find("source")
                if src_tag and src_tag.has_attr("src"):
                    return src_tag["src"].strip()
                # sometimes video tag has src attribute directly
                if vid.has_attr("src"):
                    return vid["src"].strip()
            # meta og:video
            og = soup.find("meta", {"property": "og:video"})
            if og and og.has_attr("content"):
                return og["content"].strip()
            # any script-containing "m3u8" or ".mp4"
            for script in soup.find_all("script"):
                text = script.string or ""
                if "m3u8" in text or ".mp4" in text:
                    m = re.search(r"(https?://[^\s'\"\\]+(?:\.m3u8|\.mp4)[^\s'\"\\]*)", text)
                    if m:
                        return m.group(1)
            # fallback: look for links
            for a in soup.find_all("a", href=True):
                href = a['href']
                if href.endswith(".m3u8") or href.endswith(".mp4"):
                    return href
            return None

    tasks = [scrape_detail(u) for u in page_urls]
    for coro in asyncio.as_completed(tasks):
        try:
            src = await coro
        except Exception:
            src = None
        if src:
            # normalize relative
            if src.startswith("/"):
                src = urljoin(API_BASE, src)
            hls_or_direct_urls.append(src)

    return list(dict.fromkeys(hls_or_direct_urls))

async def run_ffmpeg_download(out_path: str, url: str, headers=None, extra_flags=None, timeout=FFMPEG_TIMEOUT):
    """
    Run ffmpeg to download url -> out_path. Returns (ok:bool, message:str).
    """
    cmd = ["ffmpeg", "-hide_banner", "-loglevel", "error", "-y"]
    if headers:
        header_str = "".join(h.rstrip() + "\r\n" for h in headers)
        cmd.extend(["-headers", header_str])
    if extra_flags:
        cmd.extend(extra_flags)
    cmd.extend(["-i", url, "-c", "copy", out_path])
    try:
        proc = await asyncio.create_subprocess_exec(*cmd, stdout=PIPE, stderr=PIPE)
        try:
            stdout, stderr = await asyncio.wait_for(proc.communicate(), timeout=timeout)
        except asyncio.TimeoutError:
            proc.kill()
            await proc.communicate()
            return False, "ffmpeg timeout"
        rc = proc.returncode
        stderr_text = (stderr.decode(errors="ignore") if stderr else "").strip()
        if rc == 0:
            return True, "ffmpeg ok"
        else:
            return False, f"ffmpeg rc={rc} stderr={stderr_text[:2000]}"
    except FileNotFoundError:
        return False, "ffmpeg not installed"
    except Exception as e:
        return False, f"ffmpeg exception: {e}"

async def run_yt_dlp(out_path: str, url: str, headers=None, timeout=YTDLP_TIMEOUT):
    """
    Try yt-dlp as a fallback. Returns (ok, message).
    """
    binname = find_yt_dlp_binary()
    if not binname:
        return False, "yt-dlp not found"
    # create parent dir
    os.makedirs(os.path.dirname(out_path), exist_ok=True)
    # yt-dlp -f best -o out_path url
    cmd = [binname, "-f", "best", "-o", out_path, url]
    # yt-dlp header support: --add-header "Header: value"
    if headers:
        for h in headers:
            cmd.extend(["--add-header", h])
    try:
        proc = await asyncio.create_subprocess_exec(*cmd, stdout=PIPE, stderr=PIPE)
        try:
            stdout, stderr = await asyncio.wait_for(proc.communicate(), timeout=timeout)
        except asyncio.TimeoutError:
            proc.kill()
            await proc.communicate()
            return False, "yt-dlp timeout"
        rc = proc.returncode
        stderr_text = (stderr.decode(errors="ignore") if stderr else "").strip()
        if rc == 0:
            return True, "yt-dlp ok"
        else:
            return False, f"yt-dlp rc={rc} stderr={stderr_text[:2000]}"
    except Exception as e:
        return False, f"yt-dlp exception: {e}"

async def aiohttp_stream_download(session: aiohttp.ClientSession, url: str, out_path: str, headers=None, timeout=AIO_TIMEOUT):
    """
    Stream file using aiohttp. Headers should be a list of "Key: val".
    """
    hdrs = {}
    if headers:
        for h in headers:
            if ":" in h:
                k, v = h.split(":", 1)
                hdrs[k.strip()] = v.strip()
    try:
        async with session.get(url, headers=hdrs or None, timeout=timeout) as resp:
            if resp.status != 200:
                return False, f"HTTP {resp.status}"
            tmp = out_path + ".part"
            with open(tmp, "wb") as fh:
                async for chunk in resp.content.iter_chunked(64 * 1024):
                    if not chunk:
                        break
                    fh.write(chunk)
            os.replace(tmp, out_path)
            return True, "aiohttp ok"
    except Exception as e:
        return False, f"aio error: {e}"

async def download_m3u8_and_merge(session: aiohttp.ClientSession, url: str, out_path: str, headers=None):
    """
    Manual m3u8: fetch playlist, if master pick best stream, download .ts segments then use ffmpeg to concat into out_path.
    Returns (ok, message).
    """
    try:
        hdrs = {}
        if headers:
            for h in headers:
                if ":" in h:
                    k, v = h.split(":", 1)
                    hdrs[k.strip()] = v.strip()
        # get playlist
        async with session.get(url, headers=hdrs or None, timeout=60) as resp:
            if resp.status != 200:
                return False, f"m3u8 HTTP {resp.status}"
            text = await resp.text()
    except Exception as e:
        return False, f"m3u8 fetch error: {e}"

    # If master playlist with EXT-X-STREAM-INF choose highest BANDWIDTH
    if any("EXT-X-STREAM-INF" in l for l in text.splitlines()):
        # parse pairs
        variants = []
        raw_lines = text.splitlines()
        for i, ln in enumerate(raw_lines):
            if ln.startswith("#EXT-X-STREAM-INF"):
                # next non-empty line is variant URL
                j = i + 1
                while j < len(raw_lines) and raw_lines[j].strip() == "":
                    j += 1
                if j < len(raw_lines):
                    variants.append((ln, raw_lines[j].strip()))
        # pick variant with highest BANDWIDTH
        best_url = None
        best_bw = 0
        for info, link in variants:
            m = re.search(r"BANDWIDTH=(\d+)", info)
            bw = int(m.group(1)) if m else 0
            if bw >= best_bw:
                best_bw = bw
                best_url = link
        if best_url:
            if best_url.startswith("/"):
                base = f"{urlparse(url).scheme}://{urlparse(url).netloc}"
                best_url = urljoin(base, best_url)
            elif not best_url.startswith("http"):
                best_url = urljoin(url, best_url)
            # recursive call to fetch variant playlist
            return await download_m3u8_and_merge(session, best_url, out_path, headers=headers)

    # At this point gather segment URIs
    segments = []
    for ln in text.splitlines():
        ln = ln.strip()
        if ln and not ln.startswith("#"):
            if ln.endswith(".ts") or ln.endswith(".aac") or ln.endswith(".mp4") or ln.endswith(".m4s"):
                segments.append(ln)
            elif ".ts?" in ln or ".m3u8" in ln or ln.endswith(".bin"):
                segments.append(ln)
    if not segments:
        return False, "no segments found in m3u8"

    tmpdir = Path(out_path + ".parts")
    tmpdir.mkdir(parents=True, exist_ok=True)
    seg_files = []
    for idx, seg in enumerate(segments, 1):
        seg_url = seg
        if seg_url.startswith("/"):
            base = f"{urlparse(url).scheme}://{urlparse(url).netloc}"
            seg_url = urljoin(base, seg_url)
        elif not seg_url.startswith("http"):
            seg_url = urljoin(url, seg_url)
        seg_file = tmpdir / f"{idx:05d}.ts"
        try:
            async with session.get(seg_url, timeout=60) as resp:
                if resp.status != 200:
                    # cleanup and fail
                    for f in seg_files:
                        try:
                            f.unlink()
                        except Exception:
                            pass
                    return False, f"segment HTTP {resp.status} for {seg_url}"
                with open(seg_file, "wb") as fh:
                    async for chunk in resp.content.iter_chunked(128 * 1024):
                        fh.write(chunk)
            seg_files.append(seg_file)
        except Exception as e:
            # cleanup
            for f in seg_files:
                try:
                    f.unlink()
                except Exception:
                    pass
            return False, f"segment download error: {e}"

    # create file list for ffmpeg concat (safe quoting)
    listfile = tmpdir / "list.txt"
    try:
        with open(listfile, "w", encoding="utf-8") as fh:
            for f in seg_files:
                # write file lines safely (escape single quotes)
                safe_path = f.as_posix().replace("'", "\\'")
                fh.write("file '{}'\n".format(safe_path))
    except Exception as e:
        return False, f"write listfile error: {e}"

    # try ffmpeg concat using the list file
    cmd = ["ffmpeg", "-hide_banner", "-loglevel", "error", "-y", "-f", "concat", "-safe", "0", "-i", str(listfile), "-c", "copy", out_path]
    try:
        proc = await asyncio.create_subprocess_exec(*cmd, stdout=PIPE, stderr=PIPE)
        await proc.communicate()
        ok = (proc.returncode == 0)
        if not ok:
            msg = f"ffmpeg concat rc={proc.returncode}"
        else:
            msg = "ffmpeg concat ok"
    except Exception as e:
        ok = False
        msg = f"ffmpeg concat exception: {e}"

    # cleanup temp parts only on success
    if ok:
        for f in seg_files:
            try:
                f.unlink()
            except Exception:
                pass
        try:
            listfile.unlink()
            tmpdir.rmdir()
        except Exception:
            pass
        return True, msg
    else:
        # keep parts for inspection
        return False, msg

async def append_to_log(log_path: Path, content: str):
    """Append content to central log (create parent dir if needed)."""
    try:
        log_path.parent.mkdir(parents=True, exist_ok=True)
        timestamp = datetime.utcnow().isoformat() + "Z"
        with open(log_path, "a", encoding="utf-8") as fh:
            fh.write(f"[{timestamp}] {content}\n\n")
    except Exception:
        # best-effort; don't crash if logging fails
        pass

async def download_worker(session: aiohttp.ClientSession, model: str, url: str, idx: int, total: int, folder: str, ext: str, headers=None, central_log: Path = None):
    """
    Try multiple strategies for each URL until success. Returns (ok:bool, msg:str, out_path:str).
    """
    unique = uuid.uuid4().hex
    fname = f"leakify_{model}_{unique}.{ext}"
    dest_dir = Path(model) / folder
    dest_dir.mkdir(parents=True, exist_ok=True)
    out_path = str(dest_dir / fname)

    update_table(model, folder, f"Downloading {idx}/{total}: {fname}")

    is_m3u8 = ".m3u8" in url or url.endswith(".m3u8") or "playlist" in url.lower() or "/hls/" in url.lower()
    ff_headers = headers

    last_err = ""
    for attempt in range(1, MAX_RETRIES + 1):
        update_table(model, folder, f"{idx}/{total} attempt {attempt}: {fname}")

        # 1) ffmpeg primary
        extra_flags = None
        if is_m3u8:
            extra_flags = ["-allowed_extensions", "ALL", "-protocol_whitelist", "file,http,https,tcp,tls,crypto"]
        ok, msg = await run_ffmpeg_download(out_path, url, headers=ff_headers, extra_flags=extra_flags)
        if ok and Path(out_path).exists() and Path(out_path).stat().st_size >= MIN_VALID_BYTES:
            await append_to_log(central_log, f"SUCCESS ffmpeg | URL: {url} | FILE: {out_path} | MSG: {msg}")
            return True, f"ffmpeg success (attempt {attempt})", out_path
        last_err = f"ffmpeg: {msg}"

        # 2) yt-dlp fallback (if available)
        yt_ok, yt_msg = await run_yt_dlp(out_path, url, headers=ff_headers)
        if yt_ok and Path(out_path).exists() and Path(out_path).stat().st_size >= MIN_VALID_BYTES:
            await append_to_log(central_log, f"SUCCESS yt-dlp | URL: {url} | FILE: {out_path} | MSG: {yt_msg}")
            return True, f"yt-dlp success (attempt {attempt})", out_path
        last_err += f"\nyt-dlp: {yt_msg}"

        # 3) aiohttp stream for direct files (not ideal for m3u8)
        if not is_m3u8:
            try:
                aio_ok, aio_msg = await aiohttp_stream_download(session, url, out_path, headers=ff_headers)
            except Exception as e:
                aio_ok, aio_msg = False, f"aio exception: {e}"
            if aio_ok and Path(out_path).exists() and Path(out_path).stat().st_size >= MIN_VALID_BYTES:
                await append_to_log(central_log, f"SUCCESS aiohttp | URL: {url} | FILE: {out_path} | MSG: {aio_msg}")
                return True, f"aiohttp success (attempt {attempt})", out_path
            last_err += f"\naio: {aio_msg}"

        # 4) If it's an m3u8 or ffmpeg/yt-dlp failed, try manual m3u8 segment download
        if is_m3u8 or (".m3u8" in msg or "m3u8" in yt_msg.lower()):
            m3_ok, m3_msg = await download_m3u8_and_merge(session, url, out_path, headers=ff_headers)
            if m3_ok and Path(out_path).exists() and Path(out_path).stat().st_size >= MIN_VALID_BYTES:
                await append_to_log(central_log, f"SUCCESS m3u8_manual | URL: {url} | FILE: {out_path} | MSG: {m3_msg}")
                return True, f"m3u8 manual success (attempt {attempt})", out_path
            last_err += f"\nm3u8: {m3_msg}"

        # If file exists but too small, delete and retry
        try:
            if Path(out_path).exists() and Path(out_path).stat().st_size < MIN_VALID_BYTES:
                Path(out_path).unlink()
        except Exception:
            pass

        # slow backoff
        await asyncio.sleep(1 + attempt)

    # if reached here, failed all attempts -> log centrally
    log_content = f"FAILED | URL: {url} | FILE: {out_path} | LAST_ERRORS: {last_err}"
    if central_log:
        await append_to_log(central_log, log_content)
    update_table(model, folder, f"Failed {idx}/{total}: {fname}")
    return False, last_err, out_path

async def download_bulk(session: aiohttp.ClientSession, model: str, urls: list, folder: str, ext: str, batch: int, extra_headers=None, central_log: Path = None):
    """Download all URLs with concurrency."""
    os.makedirs(f"./{model}/{folder}", exist_ok=True)
    sem = asyncio.Semaphore(max(1, batch))
    total = len(urls)
    tasks = []

    async def sem_task(i, u):
        async with sem:
            try:
                return await download_worker(session, model, u, i, total, folder, ext, headers=extra_headers, central_log=central_log)
            except Exception as e:
                # log exception centrally
                if central_log:
                    await append_to_log(central_log, f"EXCEPTION in worker | URL: {u} | EXC: {e}")
                return False, f"exception: {e}", ""

    for i, u in enumerate(urls, 1):
        tasks.append(asyncio.create_task(sem_task(i, u)))

    results = await asyncio.gather(*tasks, return_exceptions=True)
    successes = []
    failures = []
    for i, r in enumerate(results, 1):
        if isinstance(r, Exception):
            failures.append((i, urls[i-1], str(r)))
            continue
        ok, msg, out = r
        if ok:
            successes.append((urls[i-1], out, msg))
        else:
            failures.append((i, urls[i-1], msg))
    return successes, failures

async def main():
    parser = argparse.ArgumentParser(prog="leakify", description="Download photos/videos from leakedzone.com (aggressive)")
    parser.add_argument("-u", "--user", required=True, help="Model username to scrape")
    group = parser.add_mutually_exclusive_group()
    group.add_argument("-p", "--photos", action="store_true", help="Only download photos")
    group.add_argument("-v", "--videos", action="store_true", help="Only download videos")
    parser.add_argument("-b", "--batch", type=int, default=5, help="concurrent downloads (default 5)")
    args = parser.parse_args()

    ensure_ffmpeg()

    # Build headers list and session
    headers = DEFAULT_HEADERS.copy()
    extra_headers_list = []
    
    # also include Referer etc in header list
    extra_headers_list.append("Referer: https://leakedzone.com/")
    extra_headers_list.append("User-Agent: " + DEFAULT_HEADERS["User-Agent"])

    conn = aiohttp.TCPConnector(limit_per_host=20, ssl=False)
    async with aiohttp.ClientSession(headers=headers, connector=conn) as session:
        model = args.user
        mode = ("photos only" if args.photos else "videos only" if args.videos else "photos + videos")
        update_table(model, mode, "Starting…")

        # central log path (per model)
        central_log = Path(model) / "logs" / "leakify_errors.log"

        photo_count = video_count = 0
        found_count = 0
        successes = []
        failures = []

        if args.photos or not args.videos:
            photos = await fetch_photo_urls(session, model)
            photo_count = len(photos)
            update_table(model, "photos", f"Found {photo_count} photos")
            if photos:
                s, f = await download_bulk(session, model, photos, 'photos', 'jpg', max(3, args.batch//2), extra_headers_list, central_log=central_log)
                # We don't strictly count photo successes here, but could
            else:
                update_table(model, "photos", "⚠️ No photos found")

        if args.videos or not args.photos:
            vids = await fetch_video_urls(session, model, args.batch)
            found_count = len(vids)
            update_table(model, "videos", f"Found {found_count} videos")
            if vids:
                successes, failures = await download_bulk(session, model, vids, 'videos', 'mp4', args.batch, extra_headers_list, central_log=central_log)
                video_count = len(successes)
            else:
                update_table(model, "videos", "⚠️ No videos found")
                successes = []
                failures = []

        update_table(model, mode, "Complete ✅")
        print()
        print(f"Found {found_count if (args.videos or not args.photos) else 0} videos total.")
        print(f"Successfully downloaded {video_count} videos and {photo_count} photos.")
        if failures:
            print(f"\nFailed downloads: {len(failures)} (details appended to {str(central_log)})")
            for i, url, reason in failures[:50]:
                print(f" {i}. {url} -> {reason}")
        else:
            print(f"\nAll videos downloaded successfully (log file: {str(central_log)})")

        try:
            input("Press any key to continue...")
        except Exception:
            pass

def cli():
    asyncio.run(main())

if __name__ == "__main__":
    cli()
