from math import *  # type: ignore
from pathlib import Path
import readline
import sys

from natlog.parser import *
from natlog.prolog_parser import parse_program_with_varnames, parse_goal
from natlog.unify import *  # unify, lazy_unify, activate, extractTerm, Var
from natlog.tools import *
from natlog.db import Db

sys.setrecursionlimit(1 << 24)


def my_path():
    return str(Path(__file__).parent) + "/"


def natprogs():
    return my_path() + "natprogs/"


def to_python(x):
    return x


def from_python(x):
    return x


# eng X (between 1 5 X) E,`next E R, #print R, fail?


class Eng:
    def __init__(self, interp, css, g, db, callables):
        self.interp = interp
        self.css = css
        self.db = db
        self.g = g
        self.callables = callables
        self.runner = None
        self.stopped = False

    def start(self):
        if self.runner is None and not self.stopped:
            self.runner = interp(self.css, self.g, db=self.db, callables=self.callables)

    # eng X (between 1 5 X) E, stop E.
    def stop(self):

        if self.runner is not None and not self.stopped:
            self.runner.close()
        self.stopped = True

    def __next__(self):
        if self.stopped:
            return None
        self.start()
        return next(self.runner)  # type: ignore

    def __call__(self):
        self.start()
        if not self.stopped:
            yield from self.runner  # type: ignore

    def __repr__(self):
        mes = ""
        if self.stopped:
            mes = "stopped_"
        return mes + "eng_" + str(id(self))


def undo(trail):
    while trail:
        trail.pop().unbind()


def unfold1(g, gs, h, bs, trail):
    d = dict()
    if not lazy_unify(h, g, trail, d):
        undo(trail)
        return None  # FAILURE

    for b in reversed(bs):
        b = activate(b, d)
        gs = (b, gs)
    return gs  # SUCCESS


nat_builtins = {
    "call",
    "~",
    "`",
    "``",
    "^",
    "#",
    "$",
    "@",
    "if",
    "eng",
    "ask",
    "unify_with_occurs_check",
}


def interp(css, goals0, db=None, callables=dict()):
    """
    main interpreter
    """

    def to_callable(name):
        """
        associates string names  to callables
        """
        if callable(name):
            return name
        f = callables.get(name, None)
        if f is not None:
            return f
        return eval(name)

    def dispatch_call(op, g, goals, trail):
        """
        dispatches several types of calls to Python
        """

        # yields facts matching g in Db
        def db_call(g):
            for ok in db.unify_with_fact(  # type: ignore
                g, trail
            ):  # pyright: ignore[reportOptionalMemberAccess]
                if not ok:  # FAILURE
                    undo(trail)
                    continue
                yield from step(goals)  # SUCCESS
                undo(trail)

        def python_call(g):
            """
            simple call to Python (e.g., print, no return expected)
            """
            f = to_callable(g[0])
            args = to_python(g[1:])
            f(*args)

        def python_fun(g):
            """
            function call to Python, last arg unified with result
            """
            f = to_callable(g[0])
            g = g[1:]
            v = g[-1]
            args = to_python(g[:-1])
            r = f(*args)
            r = from_python(r)
            if not unify(v, r, trail):
                undo(trail)
            else:
                yield from step(goals)

        def python_var(g):
            """
            query value of Python var last arg unified with result
            """
            r = to_callable(g[0])
            v = g[-1]
            r = from_python(r)
            if not unify(v, r, trail):
                undo(trail)
            else:
                yield from step(goals)

        def dcg_terminals(g):
            assert len(g) >= 3
            args = g[0:-2]
            s1 = g[-2]
            s2 = g[-1]
            xs = to_dif_list(args, s2)
            if not unify(xs, s1, trail):
                undo(trail)
            else:
                yield from step(goals)

        def eng(xge):
            x, g, e = xge
            (x, g) = copy_term((x, g))  # type: ignore
            g = (("the", x, g), ())
            assert isinstance(e, Var)
            r = Eng(interp, css, g, db, callables)
            e.bind(r, trail)
            yield from step(goals)

        # eng X (between 1 5 X) E, ask E R, #print R, fail?
        def ask(ex):
            e, x = ex
            a = next(e, None)
            # print('RAW ask next:',a)
            if a is None:
                r = "no"
                e.stop()
            elif len(a) == 1:  # a ^ operation
                r = ("the", copy_term(a[0]))
            else:
                ((the, r, g), ()) = a
                r = (the, copy_term(r))
            if not unify(x, r, trail):
                undo(trail)
            else:
                yield from step(goals)

        def gen_call(g):
            """
            unifies with last arg yield from a generator
            and first args, assumed ground, passed to it
            """
            gen = to_callable(g[0])
            g = g[1:]
            v = g[-1]
            args = to_python(g[:-1])
            for r in gen(*args):  # type: ignore
                r = from_python(r)

                if unify(v, r, trail):
                    yield from step(goals)
                    undo(trail)

        def if_op(g):
            cond, yes, no = g
            cond = extractTerm(cond)

            if next(step((cond, ())), None) is not None:
                yield from step((yes, goals))
            else:
                yield from step((no, goals))

        def unify_with_occurs_check_op(g):
            t1, t2 = g
            if not unify(t1, t2, trail, occ=True):
                undo(trail)
            else:
                yield from step(goals)

        if op == "eng":
            yield from eng(g)

        elif op == "ask":
            yield from ask(g)

        elif op == "call":
            yield from step((g[0] + g[1:], goals))

        elif op == "if":
            yield from if_op(g)

        elif op == "unify_with_occurs_check":
            yield from unify_with_occurs_check_op(g)

        elif op == "~":  # matches against database of facts
            yield from db_call(g)

        elif op == "^":  # yield g as an answer directly
            yield extractTerm(g)
            yield from step(goals)

        elif op == "`":  # function call, last arg unified
            yield from python_fun(g)

        elif op == "``":  # generator call, last arg unified
            yield from gen_call(g)

        elif op == "#":  # simple call, no return
            python_call(g)
            yield from step(goals)

        elif op == "@":  # DCG terminal(s)
            yield from dcg_terminals(g)

        else:  # op == '$' find value of variable
            yield from python_var(g)

        undo(trail)

    def step(goals):
        """
        recursive inner function
        """
        trail = []
        if goals == ():
            yield extractTerm(goals0)
            undo(trail)
        else:
            g, goals = goals
            op = g[0] if g else None
            if op in nat_builtins:
                g = extractTerm(g[1:])
                yield from dispatch_call(op, g, goals, trail)
            else:
                for h, bs in css:
                    bsgs = unfold1(g, goals, h, bs, trail)
                    if bsgs is not None:
                        yield from step(bsgs)
                        undo(trail)

    done = False
    while not done:
        done = True
        for a in step(goals0):
            assert not isinstance(a, Var)
            if a is not None and len(a) >= 2 and a[0] == "trust":
                newg = a[1:], ()
                goals0 = newg
                done = False
                break
            yield a


LIB = "../natprogs/lib.nat"


class Natlog:
    def __init__(
        self,
        text=None,
        file_name=None,
        clauses=None,
        syntax="natlog",
        db_name=None,
        with_lib=None,
        callables=dict(),
    ):
        if file_name is not None:
            with open(file_name, "r") as f:
                self.text = f.read()
            self.file_name = file_name
        elif text is not None:
            self.text = text
            self.file_name = None
        elif with_lib is not None:
            self.text = "ok."
            self.file_name = None
        elif clauses is not None:
            self.file_name = None
            self.text = "ok."
        else:
            raise ValueError("Natlog: text or file_name or with_lib must be provided")

        self.syntax = syntax
        self.callables = callables
        self.gsyms = dict()
        self.gixs = dict()

        self.css, self.ixss = self.add_lib(self.text, clauses, with_lib)

        # print('GIXSS in natlog:', self.gixs)

        if db_name is not None:
            self.db_init()
            self.db.load(db_name)  # type: ignore
        else:
            self.db = None

    def parse_prog(self, syntax):

        if syntax == "prolog":
            css_ixss = parse_program_with_varnames(self.text)
            css, ixss = zip(*css_ixss)
            css = tuple((h[0], tuple(b)) for (h, b) in css)
        else:
            css, ixss = self.parse_natlog_program(self.text)
        # print(f"\nCLAUSES in {self.syntax}")
        # for cs in css:
        #    print(cs)
        return css, ixss

    def guess_syntax(self):
        if self.file_name is not None:
            if self.file_name.endswith(".pl") or self.file_name.endswith(".pro"):
                self.syntax = "prolog"
            elif self.file_name.endswith(".nat"):
                self.syntax = "natlog"
            else:
                raise ValueError("Unknown file type:" + self.file_name)

    # overridable
    def add_lib(self, text, clauses, with_lib):
        """
        add library to program
        """
        if with_lib:
            with open(with_lib, "r") as f:
                lib = f.read()

                lib_css, _ = self.parse_natlog_program(lib)
        else:
            lib_css = []

        self.guess_syntax()

        css, ixss = self.parse_prog(self.syntax)

        if clauses is not None:
            css = tuple(clauses) + tuple(css)

        return css + tuple(lib_css), ixss

    # overridable
    def parse_natlog_program(self, text):
        """
        parse text and return css and ixss
        """
        css, ixss = zip(
            *parse(text, gsyms=self.gsyms, gixs=self.gixs, ground=False, rule=True)
        )
        return css, ixss

    def db_init(self):
        """
        overridable database initializer
        sets the type of the database (default or neuro-symbolic)
        """
        self.db = Db()

    def parse_query(self, quest, syntax):
        if syntax == "natlog":
            goals0, ixs = next(
                parse(quest, gsyms=self.gsyms, gixs=self.gixs, ground=False, rule=False)
            )
        else:

            body, ixs = parse_goal(quest)

            goals0 = to_cons_list(body)
        # print(f"#PARSED GOAL with {self.syntax}:", goals0, ixs)
        return goals0, ixs

    def solve(self, quest):
        """
        answer generator for given question
        """
        # goals0, ixs = next(
        #    parse(quest, gsyms=self.gsyms, gixs=self.gixs, ground=False, rule=False)
        # )

        goals0, ixs = self.parse_query(quest, self.syntax)

        vs = dict()
        goals0 = activate(goals0, vs)

        if ixs is not None:  # we have var names
            ns = dict(zip(vs, ixs))

            for k, v in self.gixs.items():
                ns[k] = v
        else:
            # print("!!! VS:", vs)
            ns = dict(zip(vs.keys(), vs.keys()))

        print("!!! SOLVE GOALS0:", goals0)
        for answer in interp(self.css, goals0, self.db, self.callables):

            if answer and len(answer) == 1:
                sols = {"_": answer[0]}
            else:
                sols = dict((ns[v], deref(r)) for (v, r) in vs.items())
            yield sols

    def count(self, quest):
        """
        answer counter
        """
        c = 0
        for _ in self.solve(quest):
            c += 1
        return floor(c)

    def query(self, quest, in_repl=False):
        """
        show answers for given query
        """
        if not in_repl:
            print("QUERY:", quest)
        success = False
        for answer in self.solve(quest):
            success = True
            print("ANSWER:", answer)
        if not success:
            print("No ANSWER!")
        print("")

    def repl(self):
        """
        read-eval-print-loop
        """

        print("Type ENTER to quit.")
        while True:
            q = input("?- ")
            if not q:
                return
            if q == "listing.":
                for cs in self.css:
                    print(cs, "\n")

            try:
                self.query(q, in_repl=True)
            except Exception as e:
                print("EXCEPTION:", type(e).__name__, e.args)
                # raise e

    # shows tuples of Natlog rule base
    def __repr__(self):
        xs = [str(cs) + "\n" for cs in self.css]
        return " ".join(xs)


# built-ins, callable with ` notation


def numlist(n, m):
    return to_cons_list(range(n, m + 1))


def consult(natfile=natprogs() + "family.nat"):
    n = Natlog(file_name=natfile, with_lib=natprogs() + "lib.nat")
    n.repl()


def load(natfile):
    Natlog(file_name=natprogs() + natfile + ".nat").repl()


# tests


def test_natlog():
    n = Natlog(file_name="natprogs/tc.nat")
    print(n)
    n.query("tc Who is animal ?")

    # n = Natlog(file_name="../natprogs/queens.nat")
    # n.query("goal8 Queens?")

    n = Natlog(file_name="natprogs/perm.nat")
    # print(n)
    n.query("perm (1 (2 (3 ())))  X ?")

    n = Natlog(file_name="natprogs/py_call.nat")
    # print(n)
    n.query("goal X?")
    # n.repl()

    n = Natlog(file_name="natprogs/family.nat")
    # print(n)
    n.query("cousin of X C, male C?")
    # n.repl()

    # n = Natlog(file_name="../natprogs/queens.nat")

    # print(n.count("goal8  X ?"))

    n = Natlog(file_name="natprogs/lib.nat")
    print(n)
    n.repl()


def pconsult(prolog_code_file, prolog_ground_db_file=None):
    # consults a prolog_code_file and optionally a prolog_ground_db_file
    n = Natlog(
        file_name=prolog_code_file,
        db_name=prolog_ground_db_file,
        with_lib=natprogs() + "lib.nat",
        syntax="prolog",
        callables=globals(),
    )
    n.repl()


def tconsult(fname):
    nname = natprogs() + fname + ".nat"
    dname = natprogs() + fname + ".tsv"
    n = Natlog(file_name=nname, db_name=dname)
    n.repl()


def natrun(fname, natgoal, callables=globals()):
    fname = fname + ".nat"
    n = Natlog(file_name=fname, with_lib=natprogs() + "lib.nat", callables=callables)
    # n.repl()
    return list(n.solve(natgoal))


def natlog(file_name=None, db_name=None, goal=None, syntax="natlog"):
    n = Natlog(
        file_name=file_name,
        db_name=db_name,
        syntax=syntax,
        with_lib=natprogs() + "lib.nat",
        callables=globals(),
    )
    if goal:
        goal = " ".join(goal)
        if goal[-1] not in "?.":
            goal = goal + "."
        n.query(goal)
    n.repl()


def natpro(file_name=None, goal=None):
    n = Natlog(
        file_name=file_name,
        with_lib=natprogs() + "lib.nat",
        syntax="prolog",
        callables=globals(),
    )
    # print("syntax:", n.syntax)

    if goal:
        if goal[-1] not in ".":
            goal = goal + "."
        n.query(goal)
    n.repl()
