# -*- coding: UTF-8 -*-
# Copyright 2011-2024 Rumma & Ko Ltd
# License: GNU Affero General Public License v3 (see file COPYING for details)

import datetime

from django.conf import settings
from django.db import models
from django.core.exceptions import ValidationError
from django.utils import translation
from django.utils.translation import gettext_lazy as _
from django.contrib.contenttypes.models import ContentType
# from django.contrib.humanize.templatetags.humanize import ordinal
# Django's ordinal() works only for English
from django.utils.encoding import force_str
from lino.utils.html import E, tostring, escape

from lino import mixins
from lino.api import dd, rt
from lino.mixins import Registrable
from lino.mixins.periods import Started
from lino.mixins.human import Born
from lino.mixins.periods import DateRangeObservable
from lino.core.exceptions import ChangedAPI
from lino.core.workflows import Workflow
from lino.modlib.office.roles import OfficeStaff, OfficeOperator
from lino.modlib.uploads.mixins import UploadController
from lino.modlib.users.mixins import UserAuthored
from lino.modlib.gfks.mixins import Controllable
from lino.modlib.notify.mixins import ChangeNotifier
from lino.modlib.system.mixins import RecurrenceSet
from lino.modlib.system.choicelists import Recurrences, Weekdays
from lino_xl.lib.xl.choicelists import Priorities

from .choicelists import EntryStates, DisplayColors
# from .choicelists import PositioningTypes
from .utils import day_and_month, day_and_weekday
from .actions import UpdateAllGuests
# from .roles import CalendarOperator

from lino.utils.format_date import fdmy


def daterange_text(a, b):
    if a == b:
        return a.strftime(settings.SITE.date_format_strftime)
    d = dict(min="...", max="...")
    if a:
        d.update(min=a.strftime(settings.SITE.date_format_strftime))
    if b:
        d.update(max=b.strftime(settings.SITE.date_format_strftime))
    return _("Dates %(min)s to %(max)s") % d


class MoveEntryNext(dd.MultipleRowAction):
    label = _('Move down')
    button_text = _('▽')  # 25BD White down-pointing triangle
    custom_handler = True
    # icon_name = 'date_next'
    show_in_workflow = True
    show_in_toolbar = False
    help_text = _("Move this event to next available date")
    readonly = False

    def get_action_permission(self, ar, obj, state):
        if obj.auto_type is None:
            return False
        if state.fixed:
            return False
        return super().get_action_permission(ar, obj, state)

    def run_on_row(self, obj, ar):
        obj.owner.move_event_next(obj, ar)
        return 1


class UpdateEntries(dd.MultipleRowAction):
    label = _('Update calendar')
    button_text = ' ⚡ '  # 26A1
    # help_text = _('Create or update the automatic calendar entries '
    #               'controlled by this generator.')

    # icon_name = 'lightning'
    readonly = False

    # required_roles = dd.login_required(CalendarOperator)

    # We want permission to update calendar entries even when it is clear that
    # there won't be any events because we might to clean up existing
    # suggestions that were generated by a previous run.

    # def get_action_permission(self, ar, obj, state):
    #     if not obj.has_auto_events():
    #         # raise Exception("20231109 {} no permission to {}".format(ar, obj))
    #         return False
    #     return super().get_action_permission(ar, obj, state)
    #     # if not rv:
    #     #     print("20231109-b no permission to", ar, obj)
    #     # return rv

    def run_on_row(self, obj, ar):
        return obj.update_reminders(ar)


class UpdateEntriesByEvent(UpdateEntries):

    def get_action_permission(self, ar, obj, state):
        if obj.auto_type is None:
            return False
        if obj.owner is None:
            return False
        # if not obj.owner.has_auto_events():
        #     return False
        # Do NOT inherit from UpdateEntries:
        return super(UpdateEntries, self).get_action_permission(ar, obj, state)

    def run_on_row(self, obj, ar):
        return obj.owner.update_reminders(ar)


class EventGenerator(dd.Model):

    class Meta:
        abstract = True

    do_update_events = UpdateEntries()
    update_all_guests = UpdateAllGuests()

    @classmethod
    def get_registrable_fields(cls, site):
        for f in super().get_registrable_fields(site):
            yield f
        yield "user"

    # def get_event_generators(self):
    #     """
    #
    #     The default behaviour just returns itself, but a single database object
    #     can potentially generate multiple series of calendar entries. Example is
    #     a Client in Lino Amici, which can generate birthdays as well as coaching
    #     meetings.
    #
    #     """
    #     return [self]

    def disable_delete(self, ar=None):
        for ee in self.get_existing_auto_events():
            if ee.is_user_modified():
                return _(
                    "Cannot delete {} because it has user-modified calendar entries."
                ).format(self)
        return super().disable_delete(ar)

    def delete(self):
        """
        Delete all unmodified generated events, including their guests, before
        deleting self.
        """
        # print("20210705 EventGenerator.delete()")
        for ee in self.get_existing_auto_events():
            assert not ee.is_user_modified()
            ee.delete()
        super().delete()

    def get_recurrence_set(self):
        """Return the *reccurrency set* to be used when generating events for
        this reservation.

        """
        raise NotImplementedError()

    def update_cal_from(self, ar):
        """

        Return the date of the first :term:`automatic calendar entry` to be
        generated.

        Return `None` if no Events should be generated.

        """
        return dd.today()

    def update_reminders(self, ar):
        return self.update_auto_events(ar)

    def update_auto_events(self, ar):
        """Generate automatic calendar entries owned by this generator.

        """
        if settings.SITE.loading_from_dump:
            # ~ print "20111014 loading_from_dump"
            return 0
        # ar.info("20231108 Update automatic calendar entries for %s", self)
        # for sa in SchedulingAreas.get_list_items():
        # for eg in self.get_event_generators():

        rset = self.get_recurrence_set()
        wanted, unwanted = self.get_wanted_auto_events(ar)
        # ar.debug(
        #     "20231109 get_wanted_auto_events() returned %s, %s",
        #     wanted, unwanted)
        count = len(wanted)
        # current = 0

        # msg = dd.obj2str(self)
        # msg += ", qs=" + str([e.auto_type for e in qs])
        # msg += ", wanted=" + ', '.join([
        #     "{}:{}".format(e.auto_type, dd.fds(e.start_date))
        #     for e in wanted.values()])
        # dd.logger.info('20161015 ' + msg)

        # ar.debug("%d wanted and %d unwanted events", count, len(unwanted))

        for ee in unwanted:
            assert not ee.is_user_modified()
            ee.delete()
            count += 1

        # import pdb ; pdb.set_trace()

        # create new Events for remaining wanted
        for we in wanted:
            if not we.is_user_modified():
                rset.before_auto_event_save(we)
            we.save()
            we.after_ui_save(ar, None)
        # ~ logger.info("20130528 update_auto_events done")
        return count

    def has_auto_events(self):
        rset = self.get_recurrence_set()
        if rset is None:
            # print("20231109 No recurrency set")
            return

        # ~ ar.debug("20131020 rset %s",rset)
        # ~ if rset and rset.every > 0 and rset.every_unit:
        if not rset.every_unit:
            # print("20231109 No every_unit")
            return
        return rset

    def move_event_next(self, we, ar):
        """Move the specified event to the next date in this series."""

        if we.auto_type is None:
            raise Exception("Cannot move uncontrolled event")
        if we.owner is not self:
            raise Exception("%s cannot move event controlled by %s" %
                            (self, we.owner))

        def doit(ar):
            if we.state == EntryStates.suggested:
                we.state = EntryStates.draft
            rset = self.get_recurrence_set()
            date = rset.get_next_alt_date(ar, we.start_date)
            if date is None:
                return
            until = self.update_cal_until() \
                or dd.plugins.cal.ignore_dates_after
            we.start_date = date
            if self.resolve_conflicts(we, ar, rset, until) is None:
                return
            we.save()

            # update all following events:
            self.update_auto_events(ar)

            # report success and tell the client to refresh
            ar.success(refresh=True)

        ar.confirm(doit, _("Move {} to next available date?").format(we))

    def care_about_conflicts(self, we):
        return True

    def resolve_conflicts(self, we, ar, rset, until):

        date = we.start_date
        if rset == Recurrences.once:
            return date
        if not self.care_about_conflicts(we):
            return date
        # if date.day == 9 and date.month == 3:
        #     ar.debug("20171130 resolve_conflicts() %s",
        #             we.has_conflicting_events())
        # ar.debug("20140310 resolve_conflicts %s", we.start_date)
        while we.has_conflicting_events():
            qs = we.get_conflicting_events()
            date = rset.get_next_alt_date(ar, date)
            ar.debug("%s wants %s but conflicts with %s, moving to %s. ",
                     we.summary, we.start_date, qs, date)
            if date is None or date > until:
                ar.debug("Failed to get next date for %s (%s > %s).", we, date,
                         until)
                conflicts = [tostring(ar.obj2html(o)) for o in qs]
                msg = ', '.join(conflicts)
                ar.warning("%s conflicts with %s. ", we, msg)
                return None

            rset.move_event_to(we, date)
        return date

    def get_existing_auto_events(self):
        ot = ContentType.objects.get_for_model(self.__class__)
        qs = rt.models.cal.Event.objects.filter(owner_type=ot,
                                                owner_id=self.pk,
                                                auto_type__isnull=False)
        # noauto_states = set([x for x in EntryStates.objects() if x.noauto])
        # if noauto_states:
        #     qs = qs.exclude(state__in=noauto_states)
        return qs

    def suggest_cal_guests(self, event):
        """Yield or return a list of (unsaved) :class:`Guest
        <lino_xl.lib.cal.Guest>` objects representing the
        participants to invite to the given event. Called on
        every event generated by this generator.

        """

        return []

    def get_date_formatter(self):
        rset = self.get_recurrence_set()
        if rset and rset.every_unit:
            return rset.every_unit.get_date_formatter()
        return day_and_month

    # def format_cal_entry(self, evt, fmt, ar):
    #     """
    #     Yield a list of etree elements to represent the given calendar
    #     entry `evt`.
    #     """
    #     if evt.auto_type:
    #         # elems.append("({}) ".format(evt.auto_type))
    #         yield "{}: ".format(evt.auto_type)

    #     lbl = fmt(evt.start_date)
    #     if evt.state.button_text:
    #         lbl = "{0}{1}".format(lbl, evt.state.button_text)
    #     yield ar.obj2html(evt, lbl)

    def update_cal_until(self):
        """
        Return the limit date until which to generate events.

        Returning `None` means "no limit", which de facto becomes
        :attr:`lino_xl.lib.cal.Plugin.ignore_dates_after`

        """

        return None

    def update_cal_event_type(self):
        """

        Return the :attr:`Event.event_type` for the calendar entries to
        generate.

        Returning `None` means: don't generate any entries.

        """
        # raise NotImplementedError()
        return None

    def get_events_language(self):
        """Return the language to activate while events are being
        generated.

        """
        user = self.get_events_user()
        if user is None:
            return settings.SITE.get_default_language()
        return user.language

    def update_cal_room(self, i):
        return None

    def get_events_user(self):
        """Returns the user who is responsible for generated events.

        In :mod:`lino_avanti` this is not the author of the course but
        the teacher.

        """
        return self.get_author()

    # def update_cal_summary(self, i):
    #     ep = self.exam_policy
    #     if ep is not None and ep.event_type is not None:
    #         if ep.event_type.event_label:
    #             return ep.event_type.event_label + " " + str(i)
    #     return _("Evaluation %d") % i

    def update_cal_summary(self, event_type, i):
        label = dd.babelattr(event_type, 'event_label')
        return _("{} {}").format(label, i)

    def setup_auto_event(self, obj):
        pass

    def get_wanted_auto_events(self, ar=None):
        existing = dict()
        wanted = []
        unwanted = []
        rset = self.get_recurrence_set()
        # rset = self.has_auto_events()
        if rset is None:
            return wanted, unwanted

        qs = self.get_existing_auto_events()
        qs = qs.order_by('start_date', 'start_time', 'auto_type')

        # Find the date and number at which to start generating events.
        # Find the existing event before the first unmodified event.

        # If the series contains at least one user-modified calendar entry, then
        # this entry serves as the start point.

        # We reuse existing events to avoid re-creating guests

        event_no = 0
        date = self.update_cal_from(ar)
        date = rset.find_start_date(date)
        # if qs.count():
        #     raise Exception("20180321 {}".format(qs.count()))

        # loop over existing events in chronological order
        # print("20231123", qs, date, existing)
        for ee in qs:
            if date is None and ee.is_user_modified():
                next_date = rset.get_next_suggested_date(
                    ee.start_date, ar.logger)
                date = next_date
                event_no = ee.auto_type
            if ee.auto_type in existing:
                # Duplicate auto_type. Keep only one of them.
                if ee.is_user_modified():
                    oe = existing[ee.auto_type]
                    if oe.is_user_modified():
                        ar.debug("Oops: duplicate auto_type {}".format(
                            ee.auto_type))
                        return wanted, unwanted
                    unwanted.append(oe)
                    existing[ee.auto_type] = ee
                else:
                    # it's unmodified and duplicate
                    unwanted.append(ee)
            else:
                existing[ee.auto_type] = ee

        def nothing():
            for ee in existing.values():
                if not ee.is_user_modified():
                    unwanted.append(ee)
            return wanted, unwanted

        if date is None:
            ar.debug("No calendar entries because no start date")
            return nothing()

        event_type = self.update_cal_event_type()
        if event_type is None:
            # raise Exception("20170731")
            ar.warning(
                _("No calendar entries because no entry type is configured"))
            return nothing()

        # ar.debug("20140310a %s", date)
        date = rset.find_start_date(date)
        # ar.debug("20140310b %s", date)
        if date is None:
            ar.debug("No available start date.")
            return nothing()
        until = self.update_cal_until() \
            or dd.plugins.cal.ignore_dates_after
        if until is None:
            raise Exception("ignore_dates_after may not be None")
        # don't take rset.max_events == 0 as False
        if rset.max_events is None:
            max_events = settings.SITE.get_config_value('max_auto_events')
        else:
            max_events = rset.max_events
        Event = settings.SITE.models.cal.Event
        ar.debug("Generating events between %s and %s (max. %s).", date, until,
                 max_events)
        ignore_before = dd.plugins.cal.ignore_dates_before
        user = self.get_events_user()
        # if max_events is not None and event_no >= max_events:
        #     raise Exception("20180321")
        last_date = None
        with translation.override(self.get_events_language()):
            # print("20221129 max_events", max_events)
            while max_events is None or event_no < max_events:
                if date > until:
                    ar.debug("Reached upper date limit %s for %s", until,
                             event_no)
                    break
                event_no += 1
                # print("20221129 event_no", event_no)
                if ignore_before and date < ignore_before:
                    ar.debug("Ignore %d because it is before %s", event_no,
                             ignore_before)
                else:
                    we = Event(auto_type=event_no,
                               user=user,
                               start_date=date,
                               summary=self.update_cal_summary(
                                   event_type, event_no),
                               room=self.update_cal_room(event_no),
                               owner=self,
                               event_type=event_type,
                               start_time=rset.start_time,
                               end_time=rset.end_time)
                    self.setup_auto_event(we)
                    date = self.resolve_conflicts(we, ar, rset, until)
                    if date is None:
                        ar.debug("Could not resolve conflicts for %s",
                                 event_no)
                        return wanted, unwanted
                    ee = existing.pop(event_no, None)
                    if ee is None:
                        wanted.append(we)
                        # print("20231109 wanted event", ee)
                    elif ee.is_user_modified():
                        # print("20221129 is_user_modified", ee)
                        if last_date and ee.start_date < last_date:
                            msg = _(
                                "Automatic entry {} ({}) goes back in time.")
                            raise Warning(
                                msg.format(ee.auto_type, ee.start_date))
                        # wanted.append(ee)
                        ar.debug("%s has been moved from %s to %s." %
                                 (ee.summary, date, ee.start_date))
                        date = ee.start_date
                    else:
                        # print("20221129 compare_auto_event", ee, we)
                        rset.compare_auto_event(ee, we)
                        # we don't need to append it to wanted because
                        # compare_auto_event() saves any changes immediately.

                last_date = date
                date = rset.get_next_suggested_date(date, ar.logger)
                date = rset.find_start_date(date)
                if date is None:
                    ar.debug("Could not find next date after %s (%s).",
                             last_date, event_no)
                    break

        # if there is something left in existing, keep only user-modified
        # entries
        for ee in existing.values():
            if ee.is_user_modified():
                wanted.append(ee)
            else:
                unwanted.append(ee)

        # print("20221124", wanted, unwanted, existing)
        return wanted, unwanted


#
# class RecurByRule(dd.Model):
#     class Meta:
#         abstract = dd.is_abstract_model(__name__, 'RecurByRule')
#         app_label = 'cal'
#         verbose_name = _("Recur by rule")
#         verbose_name_plural = _("Recur by rules")
#
#     name = RecurBy.field(_("Recur by"), default='byday')
#     positions = models.CharField(_("Positions"), blank=True, max_length=500)
#
#     monday = models.BooleanField(Weekdays.monday.text, default=False)
#     tuesday = models.BooleanField(Weekdays.tuesday.text, default=False)
#     wednesday = models.BooleanField(Weekdays.wednesday.text, default=False)
#     thursday = models.BooleanField(Weekdays.thursday.text, default=False)
#     friday = models.BooleanField(Weekdays.friday.text, default=False)
#     saturday = models.BooleanField(Weekdays.saturday.text, default=False)
#     sunday = models.BooleanField(Weekdays.sunday.text, default=False)
#
#
# class EventRecurBy(dd.Model):
#     class Meta:
#         abstract = dd.is_abstract_model(__name__, 'EventRecurBy')
#         app_label = 'cal'
#         verbose_name = _("Event recursion by")
#         verbose_name_plural = _("Event recursions by")
#
#     event = dd.ForeignKey('cal.RecurrentEvent', related_name='recur_by_rules')
#     recurby = dd.ForeignKey('cal.RecurByRule')


class ReservationStates(Workflow):
    is_editable = models.BooleanField(_("Editable"), default=True)


class Reservation(RecurrenceSet, EventGenerator, Registrable, UserAuthored):

    class Meta:
        abstract = True

    workflow_state_field = 'state'

    room = dd.ForeignKey('cal.Room', blank=True, null=True)
    max_date = models.DateField(blank=True,
                                null=True,
                                verbose_name=_("Generate events until"))

    @classmethod
    def on_analyze(cls, site):
        if cls.workflow_state_field is None:
            raise Exception("{} has no workflow_state_field".format(cls))
        super().on_analyze(site)
        ic = cls.workflow_state_field.choicelist
        k = 'auto_update_calendar'
        if not hasattr(ic, k):
            raise ChangedAPI("The workflow state field for {} uses {} which "
                             "has no attribute {}".format(cls, ic, k))

    @classmethod
    def get_simple_parameters(cls):
        s = list(super().get_simple_parameters())
        s.append('room')
        return s

    def update_cal_until(self):
        return self.max_date

    def get_recurrence_set(self):
        return self

    def update_cal_room(self, i):
        return self.room

    @classmethod
    def get_registrable_fields(cls, site):
        for f in super().get_registrable_fields(site):
            yield f
        yield 'room'
        yield 'max_date'

    def after_state_change(self, ar, old, target_state):
        super().after_state_change(ar, old, target_state)
        if target_state.auto_update_calendar:
            self.update_reminders(ar)


class Component(UserAuthored, Started, DateRangeObservable, mixins.ProjectRelated,
                Controllable, UploadController, ChangeNotifier,
                mixins.CreatedModified):

    # MRO notes: get_time_zone() is implemented by UserAuthored and by Started,
    # we want the former

    workflow_state_field = 'state'
    manager_roles_required = dd.login_required((OfficeStaff, OfficeOperator))

    class Meta:
        abstract = True

    summary = models.CharField(_("Short description"),
                               blank=True,
                               max_length=dd.get_plugin_setting(
                                   'cal', 'summary_length', 200))
    description = dd.RichTextField(_("Description"),
                                   blank=True,
                                   format='plain')
    # format='html')

    #     access_class = AccessClasses.field(blank=True, help_text=_("""\
    # Whether this is private, public or between."""))  # iCal:CLASS
    # sequence = models.IntegerField(_("Revision"), default=0)
    auto_type = models.IntegerField(_("No."), null=True, blank=True)
    priority = Priorities.field(default='normal')

    # def save(self, *args, **kw):
    #     if self.user is not None and self.access_class is None:
    #         self.access_class = self.user.access_class
    #     super().save(*args, **kw)

    def on_duplicate(self, ar, master):
        self.auto_type = None
        super().on_duplicate(ar, master)

    def disabled_fields(self, ar):
        rv = super().disabled_fields(ar)
        if self.auto_type:
            rv |= self.DISABLED_AUTO_FIELDS
        return rv

    def get_uid(self):
        """
        This is going to be used when sending
        locally created components to a remote calendar.
        """
        if not settings.SITE.uid:
            raise Exception(
                'Cannot create local calendar components because settings.SITE.uid is empty.'
            )
        return "%s@%s" % (self.pk, settings.SITE.uid)

    def unused_as_paragraph(self, ar):
        # dd.logger.info("20120217 Component.summary_row() %s", ar.renderer)
        text = self._meta.verbose_name
        if self.start_time:
            text += " "
            text += _("at {time}").format(time=dd.strftime(self.start_time))
        if self.state:
            text += ' [%s]' % escape(self.state)
        if self.summary:
            text += ': %s' % escape(self.summary)
        return tostring(self.as_summary_item(ar, text))

    # def get_change_owner(self):
    #     return self.project


Component.set_widget_options('auto_type', hide_sum=True)

# ~ Component.owner.verbose_name = _("Automatically created by")


class Colored(dd.Model):
    """
    Base class for models that define a color.

    """

    class Meta(object):
        abstract = True

    display_color = DisplayColors.field("Color", default='blue')


# class BirthdayGenerator(Born, EventGenerator):
#
#     class Meta:
#         abstract = True
#
#     def get_recurrence_set(self):
#         if self.birth_date:
#             bd = self.birth_date
#             if bd.month and bd.day:
#                 y = dd.plugins.cal.beginning_of_time.year
#                 sd = datetime.date(y, bd.month, bd.day)
#                 ep = rt.models.cal.EventPolicy(
#                     every_unit='yearly', every=1,
#                     name=str(self),
#                     start_date=sd)
#                 ep.full_clean()
#                 # print("20231109 find_start_date is", ep.find_start_date())
#                 return ep
#         return None
#
#     def update_cal_event_type(self):
#         # birthdays_ref = dd.get_plugin_setting('cal', 'birthdays_ref')
#         # if birthdays_ref:
#         return rt.models.cal.EventType.get_by_ref('birthdays')
#
#     def get_events_user(self):
#         return None
