# -*- coding: UTF-8 -*-
# Copyright 2025 Rumma & Ko Ltd
# License: GNU Affero General Public License v3 (see file COPYING for details)
# Developer docs: https://dev.lino-framework.org/plugins/peppol.html

from django.conf import settings
from django.db import models
from lino.api import dd, rt, _
# from lino import logger
from lino.core.signals import database_prepared
# from lino.mixins import Contactable, Phonable
from lino.modlib.checkdata.choicelists import Checker
# from lino.mixins.periods import DateRange
from lino_xl.lib.accounting.roles import LedgerStaff
# from lino_xl.lib.contacts.mixins import ContactRelated
# from lino_xl.lib.countries.mixins import AddressLocation
from .utils import supplier_attrs

DATA_PY_TEMPLATE = """\
# -*- coding: UTF-8 -*-
# This file was generated by `lino_xl.lib.peppol` when `database_prepared`.
# See also /docs/plugins/peppol/suppliers.rst
from collections import namedtuple
Supplier = namedtuple('Supplier', ['vat_id', 'names', 'supplier_id'])
suppliers = {demo_suppliers}

"""


if dd.get_plugin_setting('peppol', 'credentials', False):

    def write_data_py(**kwargs):
        from collections import namedtuple
        Supplier = namedtuple('Supplier', ['vat_id', 'names', 'supplier_id'])
        pth = settings.SITE.site_dir / "data.py"
        # settings.SITE.logger.info(f"Write Peppol demo data to %s", pth)
        # qs = rt.models.peppol.Supplier.objects.filter(supplier_id__isnull=False)
        # When no credentials are activated, supplier_id is empty. We don't want the list to be empty in that case.

        qs = rt.models.peppol.Supplier.objects.all()
        qs = qs.order_by('company__vat_id')
        demo_suppliers = [
            Supplier(obj.company.vat_id, obj.names, obj.supplier_id) for obj in qs]
        txt = DATA_PY_TEMPLATE.format(**locals())
        # print(txt)
        pth.write_text(txt)

    database_prepared.connect(write_data_py)


# TODO: move babelfilter to lino core.utils.mldbc


def babelfilter(name, value, **kwargs):
    flt = models.Q()
    for k, v in dd.str2kw(name, value).items():
        flt |= models.Q(**{k: v})
    if kwargs:
        flt &= models.Q(**kwargs)
    return flt


with_suppliers = dd.get_plugin_setting("peppol", "with_suppliers", False)
# outbound_model = dd.get_plugin_setting("peppol", "outbound_model", None)
# inbound_model = dd.get_plugin_setting("peppol", "inbound_model", None)
# supplier_id = dd.get_plugin_setting("peppol", "supplier_id", None)

peppol = dd.plugins.peppol


def clean_phone_number(phone):
    phone = phone.replace("/", "")
    phone = phone.replace(" ", "")
    phone = phone.replace(".", "")
    phone = phone.replace("(", "")
    phone = phone.replace(")", "")
    return phone


class OnboardingStates(dd.ChoiceList):
    verbose_name = _("Onboarding state")
    verbose_name_plural = _("Onboarding states")
    required_roles = dd.login_required(LedgerStaff)


add = OnboardingStates.add_item
add('10', _("Draft"), 'draft')
add('20', _("Created"), 'created')
add('30', _("Approved"), 'approved')
add('40', _("Rejected"), 'rejected')
add('50', _("Onboarded"), 'onboarded')
add('60', _("Offboarded"), 'offboarded')

# add('10', _("Active"), 'active')
# add('20', _("Potential"), 'potential')
# add('30', _("Unreachable"), 'unreachable')

# COMPANY_TO_SUPPLIER = ('vat_id', 'country', 'city', 'zip_code',
#                        'street', 'email', 'street_no', 'phone', 'url')

# class Supplier(AddressLocation, ContactRelated, Contactable, Phonable):


# class Supplier(AddressLocation, Contactable, Phonable):
class Supplier(dd.Model):

    class Meta:
        app_label = 'peppol'
        verbose_name = _("Ibanity supplier")
        verbose_name_plural = _("Ibanity suppliers")

    quickfix_checkdata_label = _("Sync to Ibanity")
    # preferred_foreignkey_width = 10

    supplier_id = models.CharField(
        _("Supplier ID"), max_length=50, blank=True, null=True, unique=True)
    if False:  # Should work but doesn't. Does lino_react support OneToOneField?
        company = dd.OneToOneField("contacts.Company")
    else:
        company = dd.ForeignKey(
            "contacts.Company",
            related_name="peppol_suppliers_by_company",
            verbose_name=_("Organization"))
    names = models.CharField(_('Names'), max_length=200, blank=True)
    ibans = models.CharField(_('IBANs'), max_length=200, blank=True)
    # vat_id = models.CharField(_("VAT id"), max_length=200, blank=True)
    onboarding_state = OnboardingStates.field(default='draft')
    peppol_receiver = models.BooleanField(_("Peppol receiver"), default=True)
    last_sync = models.DateTimeField(_("Last sync"), editable=False, null=True)

    @classmethod
    def get_simple_parameters(cls):
        yield super().get_simple_parameters()
        yield "onboarding_state"

    def full_clean(self):
        super().full_clean()
        # if not self.supplier_id and self.company_id:
        if not self.names:
            self.names = self.company.name
        if not self.ibans:
            Account = rt.models.sepa.Account
            qs = Account.objects.filter(partner=self.company, primary=True)
            self.ibans = "; ".join([o.iban for o in qs])
            # for k in COMPANY_TO_SUPPLIER:
            #     if not getattr(self, k):
            #         setattr(self, k, getattr(self.company, k))
        # self.phone = clean_phone_number(self.company.phone)
        # if self.vat_id:
        #     assert self.country_id
        #     mod = self.country.get_stdnum_module("vat")
        #     self.vat_id = self.country.isocode + mod.compact(self.vat_id)

    def remote_supplier_data(self):
        d = dict()
        d["names"] = [{"value": name.strip()} for name in self.names.split(";")]
        d["ibans"] = [{"value": i.strip()} for i in self.ibans.split(";")]
        d["city"] = str(self.company.city)
        d["country"] = str(self.company.country)
        d["homepage"] = self.company.url or "https://"
        d["phoneNumber"] = clean_phone_number(self.company.phone) or "+012345678"
        d["street"] = self.company.street
        d["streetNumber"] = self.company.street_no
        d["contactEmail"] = self.company.email or "info@example.com"
        d["email"] = self.company.email or "info@example.com"
        # if (site_company := settings.SITE.site_config.site_company):
        sc = settings.SITE.get_plugin_setting('contacts', 'site_owner', None)
        if sc is not None:
            d["supportEmail"] = sc.email
            d["supportPhone"] = sc.phone
            d["supportUrl"] = sc.url
        d["zip"] = self.company.zip_code
        d["peppolReceiver"] = self.peppol_receiver
        return d

    def __str__(self):
        if self.company_id is not None:
            return str(self.company)
        return super().__str__()


class SupplierDetail(dd.DetailLayout):
    main = """
    supplier_id company id
    names
    peppol_receiver onboarding_state last_sync
    ibans
    checkdata.MessagesByOwner
    """


class Suppliers(dd.Table):
    required_roles = dd.login_required(LedgerStaff)
    model = "peppol.Supplier"
    column_names = "supplier_id company company__vat_id onboarding_state last_sync *"
    insert_layout = """
    company
    supplier_id
    """
    detail_layout = "peppol.SupplierDetail"


# class SuppliersListChecker(Checker):
#     verbose_name = _("Check for unknown suppliers on Ibanity")
#     model = None
#
#     def get_checkdata_problems(self, ar, obj, fix=False):
#         if not peppol.credentials:
#             logger.info("No Ibanity credentials")
#             return
#
#         ses = peppol.get_ibanity_session(ar)
#         for sup_data in ses.list_suppliers():
#             sup_id = sup_data['id']
#             if Supplier.objects.filter(supplier_id=sup_id).exists():
#                 continue
#             yield True, _("Unknown supplier on Ibanity: {}").format(sup_data)
#             if fix:
#                 ses.delete_supplier(sup_id)


# if peppol.credentials:
# SuppliersListChecker.activate(no_auto=True)
# SuppliersListChecker.activate()


def update_id_list(oldlist, value, sep=";"):
    """
    Return a list of dicts to send to Ibanity API in order to update
    the fields :attr:`names` and :attr:`ibans`.

    - `value` is the current Lino value, a string containing one or more names
      or ibans separated by `sep`.

    - `oldlist` is the list returned by get_supplier().

    Each item of oldlist is a list of `{'id': x, 'value': y }`.

    """
    newlist = []
    values = {v.strip() for v in value.split(sep) if v}
    for item in oldlist:
        if (v := item['value']) in values:
            newlist.append(item)
            values.remove(v)
    for v in values:
        newlist.append({'value': v})
    newlist.sort(key=lambda x: x['value'])
    return newlist


class SupplierChecker(Checker):
    verbose_name = _("Check for differences between our copy and Ibanity")
    model = Supplier
    msg_needs_update = _("Some fields need update: {changed}")

    def get_checkdata_problems(self, ar, obj, fix=False):

        if obj.company.name not in obj.names:
            msg = _("{model} name {obj.company.name} missing in supplier names")
            ctx = dict(model=obj.company._meta.verbose_name, obj=obj)
            yield (False, msg.format(**ctx))

            # mod = obj.country.get_stdnum_module("vat")
            # vat_id = obj.country.isocode + mod.compact(obj.company.vat_id)
            # if vat_id != obj.vat_id:
            #     msg = _("{model} VAT id {obj.company.vat_id} ≠ {obj.vat_id}")
            #     yield (False, msg.format(**ctx))

        if not peppol.credentials:
            ar.logger.info("No Ibanity credentials")
            return

        # Country = rt.models.countries.Country
        # Place = rt.models.countries.Place

        def save():
            obj.last_sync = dd.now()
            obj.full_clean()
            obj.save()

        ses = peppol.get_ibanity_session(ar)

        p2l = dict()  # Peppol to Lino
        l2p = dict()  # Lino to Peppol

        if obj.supplier_id:
            sup_data, errmsg = ses.get_supplier(obj.supplier_id)
            if sup_data is None:
                yield True, _("Supplier {} doesn't exist on Ibanity.").format(
                    obj.supplier_id)
                if fix:
                    obj.supplier_id = None
                    save()
                else:
                    return
            else:
                assert sup_data['id'] == obj.supplier_id
                # attrs = sup_data['attributes']

                # obj.onboarding_state = OnboardingStates.get_by_name(
                #     attrs['onboardingStatus'].lower())

            # if sup_id != obj.supplier_id:
            #     yield False, _("Supplier {} doesn't exist on Ibanity.").format(sup_id)
            #     save()
            #     return
        epi = obj.company.endpoint_id

        if not obj.supplier_id:
            yield True, _("Must find or create supplier on Ibanity")
            if fix:
                # To avoid lino_xl.lib.peppol.utils.PeppolFailure: POST
                # https://api.ibanity.com/einvoicing/suppliers/ () returned 409
                # {"errors":[{"code":"alreadyRegistered","detail":"A supplier
                # already exists with this enterprise identification"}]}), we need
                # to check whether a suppliser with this VAT id has been created and
                # then deleted (onboarded) earlier (e.g. by by previous test runs)
                if (sup_data := ses.find_supplier_by_eid(epi)) is None:
                    attrs = supplier_attrs(epi)
                    attrs.update(obj.remote_supplier_data())
                    sup_data, errmsg = ses.create_supplier(**attrs)
                    if sup_data is None:
                        yield False, _(
                            "Error {!r} while trying to create supplier from {}").format(
                                errmsg, attrs)
                        return
                    obj.supplier_id = sup_data['id']
                    # print(f"20250613 created new supplier {obj.supplier_id}")
                else:
                    supplier_id = sup_data['id']
                    # print(f"20250613 reuse existing supplier {supplier_id}")
                    sup_data['attributes']['onboardingStatus'] = 'ONBOARDED'
                    ses.update_supplier(supplier_id, **sup_data)
                    obj.supplier_id = supplier_id
                    # obj.onboarding_state = OnboardingStates.onboarded
            else:
                return

        assert sup_data
        assert obj.supplier_id == sup_data['id']

        # Synchronize supplier data between Ibanity (remote) and our (local) data

        attrs = sup_data['attributes']

        p2l['onboarding_state'] = OnboardingStates.get_by_name(
            attrs['onboardingStatus'].lower())

        remote_vat_id = attrs['enterpriseIdentification']['vatNumber']
        local_vat_id = epi.country_code + epi.vat_id
        if local_vat_id != remote_vat_id:
            yield False, _("Differing VAT IDs for {obj.supplier_id}: {} != {}").format(
                local_vat_id, remote_vat_id)
            # p2l['vat_id'] = attrs['enterpriseIdentification']['vatNumber']
            return

        lattrs = obj.remote_supplier_data()
        for k in ('street', 'streetNumber', 'phoneNumber', 'email', 'zip',
                  'homepage', 'country', 'city', 'peppolReceiver'):
            if lattrs[k] != attrs[k]:
                l2p[k] = lattrs[k]

        if obj.names != "; ".join([i['value'] for i in attrs['names']]):
            l2p['names'] = update_id_list(attrs['names'], obj.names)
        if obj.ibans != "; ".join([i['value'] for i in attrs['ibans']]):
            l2p['ibans'] = update_id_list(attrs['ibans'], obj.ibans)
        if obj.peppol_receiver != attrs['peppolReceiver']:
            l2p['peppolReceiver'] = obj.peppol_receiver
        # if obj.company.street != attrs['street']:
        #     l2p['street'] = obj.company.street
        # if obj.company.street_no != attrs['streetNumber']:
        #     l2p['streetNumber'] = obj.company.street_no
        # if clean_phone_number(obj.company.phone) != attrs['phoneNumber']:
        #     l2p['phoneNumber'] = obj.company.phone
        # if obj.company.email != attrs['email']:
        #     l2p['email'] = obj.company.email
        # if obj.company.zip_code != attrs['zip']:
        #     l2p['zip'] = obj.company.zip_code
        # if attrs['homepage'] != obj.company.url:
        #     l2p['homepage'] = obj.company.url
        # if str(obj.company.country) != attrs['country']:
        #     l2p['country'] = str(obj.company.country)
        # if str(obj.company.city) != attrs['city']:
        #     l2p['city'] = str(obj.company.city)
        # else:
        #     try:
        #         country = Country.objects.get(babelfilter('name', attrs['country']))
        #     except Country.DoesNotExist:
        #         yield (False, _("Unknown country {country}").format(**attrs))
        #     else:
        #         p2l['country'] = country

        # if obj.vat_id != attrs['enterpriseIdentification']['vatNumber']:
        #     p2l['vat_id'] = attrs['enterpriseIdentification']['vatNumber']

        # if obj.company.city_id:
        #     l2p['city'] = str(obj.company.city)
        # else:
        #     try:
        #         city = Place.objects.get(
        #             babelfilter('name', attrs['city'], country=country))
        #     except Place.DoesNotExist:
        #         yield (True, _("Unknown city {city} ((in {country})").format(**attrs))
        #         if fix:
        #             city = Place(name=attrs['city'], country=country)
        #             city.full_clean()
        #             city.save()
        #             p2l['city'] = city
        #     else:
        #         p2l['city'] = city

        if len(l2p):
            # print(f"20250613 update remote supplier {obj.supplier_id} {l2p}")
            ses.update_supplier(obj.supplier_id, **l2p)

        changed = []
        for k, v in p2l.items():
            if getattr(obj, k) != v:
                changed.append(k)
        changed.sort()
        if len(changed):
            msg = self.msg_needs_update.format(changed=", ".join(changed))
            yield (True, msg)
            if fix:
                # print(f"20250613 update local supplier {obj.supplier_id} {p2l}")
                for k in changed:
                    setattr(obj, k, p2l[k])

        # We call save() even when there are no changes because we want
        # last_sync to reflect each online check.
        save()

# On a site without Ibanity credentials we don't need to activate this
# checker. SupplierChecker.activate() sets no_auto to True because we do not
# want this checker to run automatically (i.e. during checkdata). It should
# run only when called manually because it requires Ibanity credentials,
# which are not available e.g. on GitLab.


# if peppol.credentials:
# SupplierChecker.activate(no_auto=True)
SupplierChecker.activate()
