# Copyright 1999-2025 Alibaba Group Holding Ltd.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import numpy as np

from ... import opcodes
from ..utils import inject_dtype
from .core import TensorBinOp
from .utils import arithmetic_operator


@arithmetic_operator(sparse_mode="binary_and")
class TensorLessEqual(TensorBinOp):
    _op_type_ = opcodes.LE
    _func_name = "less_equal"


@inject_dtype(np.bool_)
def less_equal(x1, x2, out=None, where=None, **kwargs):
    """
    Return the truth value of (x1 =< x2) element-wise.

    Parameters
    ----------
    x1, x2 : array_like
        Input tensors.  If ``x1.shape != x2.shape``, they must be
        broadcastable to a common shape (which may be the shape of one or
        the other).
    out : Tensor, None, or tuple of Tensor and None, optional
        A location into which the result is stored. If provided, it must have
        a shape that the inputs broadcast to. If not provided or `None`,
        a freshly-allocated tensor is returned. A tuple (possible only as a
        keyword argument) must have length equal to the number of outputs.
    where : array_like, optional
        Values of True indicate to calculate the ufunc at that position, values
        of False indicate to leave the value in the output alone.
    **kwargs

    Returns
    -------
    out : bool or tensor of bool
        Array of bools, or a single bool if `x1` and `x2` are scalars.

    See Also
    --------
    greater, less, greater_equal, equal, not_equal

    Examples
    --------
    >>> import maxframe.tensor as mt

    >>> mt.less_equal([4, 2, 1], [2, 2, 2]).execute()
    array([False,  True,  True])
    """
    op = TensorLessEqual(**kwargs)
    return op(x1, x2, out=out, where=where)
