import pytest
from lxml import etree

from verbecc import Conjugator
from verbecc.tense_template import TenseTemplate

cg = Conjugator(lang="ca")


def test_all_verbs_have_templates():
    verbs = cg.get_verbs()
    assert len(verbs) == 8616
    template_names = cg.get_template_names()
    assert len(template_names) == 66
    missing_verbs = []
    missing_templates = set()
    for verb in verbs:
        if verb.template not in template_names:
            missing_templates.add(verb.template)
            missing_verbs.append(verb)
    assert len(missing_templates) == len(missing_verbs) == 0


def test_find_verb_by_infinitive():
    v = cg.find_verb_by_infinitive("abandonar")
    assert v.infinitive == "abandonar"
    assert v.template == "cant:ar"


@pytest.mark.parametrize(
    "infinitive,mood,tense,alternate,gender,expected_result",
    [
        (
            "ser",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo sóc",
                "tu ets",
                "ell és",
                "nosaltres som",
                "vosaltres sou",
                "ells són",
            ],
        ),
        (
            "ser",
            "indicatiu",
            "imperfet",
            False,
            "m",
            [
                "jo era",
                "tu eres",
                "ell era",
                "nosaltres érem",
                "vosaltres éreu",
                "ells eren",
            ],
        ),
        (
            "ser",
            "indicatiu",
            "passat-simple",
            False,
            "m",
            [
                "jo fui",
                "tu fores",
                "ell fou",
                "nosaltres fórem",
                "vosaltres fóreu",
                "ells foren",
            ],
        ),
        (
            "ser",
            "indicatiu",
            "futur",
            False,
            "m",
            [
                "jo seré",
                "tu seràs",
                "ell serà",
                "nosaltres serem",
                "vosaltres sereu",
                "ells seran",
            ],
        ),
        (
            "ser",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo sigui",
                "tu siguis",
                "ell sigui",
                "nosaltres siguem",
                "vosaltres sigueu",
                "ells siguin",
            ],
        ),
        (
            "ser",
            "subjuntiu",
            "imperfet",
            False,
            "m",
            [
                "jo fos",
                "tu fossis",
                "ell fos",
                "nosaltres fóssim",
                "vosaltres fóssiu",
                "ells fossin",
            ],
        ),
        (
            "ser",
            "condicional",
            "present",
            False,
            "m",
            [
                "jo seria",
                "tu series",
                "ell seria",
                "nosaltres seríem",
                "vosaltres seríeu",
                "ells serien",
            ],
        ),
        (
            "ser",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["sigues", "sigui", "siguem", "sigueu", "siguin"],
        ),
        (
            "parlar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo parlo",
                "tu parles",
                "ell parla",
                "nosaltres parlem",
                "vosaltres parleu",
                "ells parlen",
            ],
        ),
        (
            "parlar",
            "indicatiu",
            "imperfet",
            False,
            "m",
            [
                "jo parlava",
                "tu parlaves",
                "ell parlava",
                "nosaltres parlàvem",
                "vosaltres parlàveu",
                "ells parlaven",
            ],
        ),
        (
            "parlar",
            "indicatiu",
            "passat-simple",
            False,
            "m",
            [
                "jo parlí",
                "tu parlares",
                "ell parlà",
                "nosaltres parlàrem",
                "vosaltres parlàreu",
                "ells parlaren",
            ],
        ),
        (
            "parlar",
            "indicatiu",
            "futur",
            False,
            "m",
            [
                "jo parlaré",
                "tu parlaràs",
                "ell parlarà",
                "nosaltres parlarem",
                "vosaltres parlareu",
                "ells parlaran",
            ],
        ),
        (
            "parlar",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo parli",
                "tu parlis",
                "ell parli",
                "nosaltres parlem",
                "vosaltres parleu",
                "ells parlin",
            ],
        ),
        (
            "parlar",
            "subjuntiu",
            "imperfet",
            False,
            "m",
            [
                "jo parlés",
                "tu parlessis",
                "ell parlés",
                "nosaltres parléssim",
                "vosaltres parléssiu",
                "ells parlessin",
            ],
        ),
        (
            "parlar",
            "condicional",
            "present",
            False,
            "m",
            [
                "jo parlaria",
                "tu parlaries",
                "ell parlaria",
                "nosaltres parlaríem",
                "vosaltres parlaríeu",
                "ells parlarien",
            ],
        ),
        (
            "estar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo estic",
                "tu estàs",
                "ell està",
                "nosaltres estem",
                "vosaltres esteu",
                "ells estan",
            ],
        ),
        (
            "estar",
            "indicatiu",
            "imperfet",
            False,
            "m",
            [
                "jo estava",
                "tu estaves",
                "ell estava",
                "nosaltres estàvem",
                "vosaltres estàveu",
                "ells estaven",
            ],
        ),
        (
            "estar",
            "indicatiu",
            "passat-simple",
            False,
            "m",
            [
                "jo estiguí",
                "tu estigueres",
                "ell estigué",
                "nosaltres estiguérem",
                "vosaltres estiguéreu",
                "ells estigueren",
            ],
        ),
        (
            "estar",
            "indicatiu",
            "futur",
            False,
            "m",
            [
                "jo estaré",
                "tu estaràs",
                "ell estarà",
                "nosaltres estarem",
                "vosaltres estareu",
                "ells estaran",
            ],
        ),
        (
            "estar",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo estigui",
                "tu estiguis",
                "ell estigui",
                "nosaltres estiguem",
                "vosaltres estigueu",
                "ells estiguin",
            ],
        ),
        (
            "estar",
            "subjuntiu",
            "imperfet",
            False,
            "m",
            [
                "jo estigués",
                "tu estiguessis",
                "ell estigués",
                "nosaltres estiguéssim",
                "vosaltres estiguéssiu",
                "ells estiguessin",
            ],
        ),
        (
            "estar",
            "condicional",
            "present",
            False,
            "m",
            [
                "jo estaria",
                "tu estaries",
                "ell estaria",
                "nosaltres estaríem",
                "vosaltres estaríeu",
                "ells estarien",
            ],
        ),
        (
            "haver",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo he",
                "tu has",
                "ell ha",
                "nosaltres havem",
                "vosaltres haveu",
                "ells han",
            ],
        ),
        (
            "haver",
            "indicatiu",
            "imperfet",
            False,
            "m",
            [
                "jo havia",
                "tu havies",
                "ell havia",
                "nosaltres havíem",
                "vosaltres havíeu",
                "ells havien",
            ],
        ),
        (
            "haver",
            "indicatiu",
            "passat-simple",
            False,
            "m",
            [
                "jo haguí",
                "tu hagueres",
                "ell hagué",
                "nosaltres haguérem",
                "vosaltres haguéreu",
                "ells hagueren",
            ],
        ),
        (
            "haver",
            "indicatiu",
            "futur",
            False,
            "m",
            [
                "jo hauré",
                "tu hauràs",
                "ell haurà",
                "nosaltres haurem",
                "vosaltres haureu",
                "ells hauran",
            ],
        ),
        (
            "haver",
            "condicional",
            "present",
            False,
            "m",
            [
                "jo hauria",
                "tu hauries",
                "ell hauria",
                "nosaltres hauríem",
                "vosaltres hauríeu",
                "ells haurien",
            ],
        ),
        (
            "haver",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo hagi",
                "tu hagis",
                "ell hagi",
                "nosaltres hàgim",
                "vosaltres hàgiu",
                "ells hagin",
            ],
        ),
        (
            "tenir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo tinc",
                "tu tens",
                "ell té",
                "nosaltres tenim",
                "vosaltres teniu",
                "ells tenen",
            ],
        ),
        (
            "fer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo faig",
                "tu fas",
                "ell fa",
                "nosaltres fem",
                "vosaltres feu",
                "ells fan",
            ],
        ),
        (
            "fer",
            "indicatiu",
            "imperfet",
            False,
            "m",
            [
                "jo feia",
                "tu feies",
                "ell feia",
                "nosaltres fèiem",
                "vosaltres fèieu",
                "ells feien",
            ],
        ),
        (
            "servir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo serveixo",
                "tu serveixes",
                "ell serveix",
                "nosaltres servim",
                "vosaltres serviu",
                "ells serveixen",
            ],
        ),
        (
            "veure",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo veig",
                "tu veus",
                "ell veu",
                "nosaltres veiem",
                "vosaltres veieu",
                "ells veuen",
            ],
        ),
        (
            "abandonar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo abandono",
                "tu abandones",
                "ell abandona",
                "nosaltres abandonem",
                "vosaltres abandoneu",
                "ells abandonen",
            ],
        ),
        (
            "rebre",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo rebo",
                "tu reps",
                "ell rep",
                "nosaltres rebem",
                "vosaltres rebeu",
                "ells reben",
            ],
        ),
        (
            "cabre",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo cabo",
                "tu caps",
                "ell cap",
                "nosaltres cabem",
                "vosaltres cabeu",
                "ells caben",
            ],
        ),
        (
            "començar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo començo",
                "tu comences",
                "ell comença",
                "nosaltres comencem",
                "vosaltres comenceu",
                "ells comencen",
            ],
        ),
        (
            "enaiguar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo enaiguo",
                "tu enaigües",
                "ell enaigua",
                "nosaltres enaigüem",
                "vosaltres enaigüeu",
                "ells enaigüen",
            ],
        ),
        (
            "envejar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo envejo",
                "tu enveges",
                "ell enveja",
                "nosaltres envegem",
                "vosaltres envegeu",
                "ells envegen",
            ],
        ),
        (
            "esglaiar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo esglaio",
                "tu esglaies",
                "ell esglaia",
                "nosaltres esglaiem",
                "vosaltres esglaieu",
                "ells esglaien",
            ],
        ),
        (
            "trencar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo trenco",
                "tu trenques",
                "ell trenca",
                "nosaltres trenquem",
                "vosaltres trenqueu",
                "ells trenquen",
            ],
        ),
        (
            "perdre",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo perdo",
                "tu perds",
                "ell perd",
                "nosaltres perdem",
                "vosaltres perdeu",
                "ells perden",
            ],
        ),
        (
            "caure",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo caic",
                "tu caus",
                "ell cau",
                "nosaltres caiem",
                "vosaltres caieu",
                "ells cauen",
            ],
        ),
        (
            "collar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo collo",
                "tu colles",
                "ell colla",
                "nosaltres collem",
                "vosaltres colleu",
                "ells collen",
            ],
        ),
        (
            "collir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo cullo",
                "tu culls",
                "ell cull",
                "nosaltres collim",
                "vosaltres colliu",
                "ells cullen",
            ],
        ),
        (
            "dormir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo dormo",
                "tu dorms",
                "ell dorm",
                "nosaltres dormim",
                "vosaltres dormiu",
                "ells dormen",
            ],
        ),
        (
            "tossir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo tusso",
                "tu tusses",
                "ell tus",
                "nosaltres tossim",
                "vosaltres tossiu",
                "ells tussen",
            ],
        ),
        (
            "collir",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo culli",
                "tu cullis",
                "ell culli",
                "nosaltres collim",
                "vosaltres colliu",
                "ells cullin",
            ],
        ),
        (
            "dormir",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo dormi",
                "tu dormis",
                "ell dormi",
                "nosaltres dormim",
                "vosaltres dormiu",
                "ells dormin",
            ],
        ),
        (
            "tossir",
            "subjuntiu",
            "present",
            False,
            "m",
            [
                "jo tussi",
                "tu tussis",
                "ell tussi",
                "nosaltres tossim",
                "vosaltres tossiu",
                "ells tussin",
            ],
        ),
        (
            "dir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo dic",
                "tu dius",
                "ell diu",
                "nosaltres diem",
                "vosaltres dieu",
                "ells diuen",
            ],
        ),
        (
            "anar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo vaig",
                "tu vas",
                "ell va",
                "nosaltres anem",
                "vosaltres aneu",
                "ells van",
            ],
        ),
        (
            "poder",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo puc",
                "tu pots",
                "ell pot",
                "nosaltres podem",
                "vosaltres podeu",
                "ells poden",
            ],
        ),
        (
            "arribar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo arribo",
                "tu arribes",
                "ell arriba",
                "nosaltres arribem",
                "vosaltres arribeu",
                "ells arriben",
            ],
        ),
        (
            "trobar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo trobo",
                "tu trobes",
                "ell troba",
                "nosaltres trobem",
                "vosaltres trobeu",
                "ells troben",
            ],
        ),
        (
            "partir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo parteixo",
                "tu parteixes",
                "ell parteix",
                "nosaltres partim",
                "vosaltres partiu",
                "ells parteixen",
            ],
        ),
        (
            "morir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo moro",
                "tu mors",
                "ell mor",
                "nosaltres morim",
                "vosaltres moriu",
                "ells moren",
            ],
        ),
        (
            "eixir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo ixo",
                "tu ixes",
                "ell ix",
                "nosaltres eixim",
                "vosaltres eixiu",
                "ells ixen",
            ],
        ),
        (
            "jaure",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo jec",
                "tu jeus",
                "ell jeu",
                "nosaltres jaiem",
                "vosaltres jaieu",
                "ells jeuen",
            ],
        ),
        (
            "jeure",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo jec",
                "tu jeus",
                "ell jeu",
                "nosaltres jaiem",
                "vosaltres jaieu",
                "ells jeuen",
            ],
        ),
        (
            "pagar",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo pago",
                "tu pagues",
                "ell paga",
                "nosaltres paguem",
                "vosaltres pagueu",
                "ells paguen",
            ],
        ),
        (
            "mentir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo menteixo",
                "tu menteixes",
                "ell menteix",
                "nosaltres mentim",
                "vosaltres mentiu",
                "ells menteixen",
            ],
        ),
        (
            "desmentir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo desmenteixo",
                "tu desmenteixes",
                "ell desmenteix",
                "nosaltres desmentim",
                "vosaltres desmentiu",
                "ells desmenteixen",
            ],
        ),
        (
            "nàixer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo naixo",
                "tu naixes",
                "ell naix",
                "nosaltres naixem",
                "vosaltres naixeu",
                "ells naixen",
            ],
        ),
        (
            "néixer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo naixo",
                "tu naixes",
                "ell naix",
                "nosaltres naixem",
                "vosaltres naixeu",
                "ells naixen",
            ],
        ),
        (
            "obrir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo obro",
                "tu obres",
                "ell obre",
                "nosaltres obrim",
                "vosaltres obriu",
                "ells obren",
            ],
        ),
        (
            "riure",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo ric",
                "tu rius",
                "ell riu",
                "nosaltres riem",
                "vosaltres rieu",
                "ells riuen",
            ],
        ),
        (
            "pertànyer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo pertanyo",
                "tu pertanys",
                "ell pertany",
                "nosaltres pertanyem",
                "vosaltres pertanyeu",
                "ells pertanyen",
            ],
        ),
        (
            "pertànyer",
            "indicatiu",
            "passat-simple",
            False,
            "m",
            [
                "jo pertanyí",
                "tu pertanyeres",
                "ell pertanyé",
                "nosaltres pertanyérem",
                "vosaltres pertanyéreu",
                "ells pertanyeren",
            ],
        ),
        (
            "pertànyer",
            "indicatiu",
            "passat-simple",
            True,
            "m",
            [
                "jo pertanguí",
                "tu pertangueres",
                "ell pertangué",
                "nosaltres pertanguérem",
                "vosaltres pertanguéreu",
                "ells pertanguéren",
            ],
        ),
        (
            "fènyer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo fenyo",
                "tu fenys",
                "ell feny",
                "nosaltres fenyem",
                "vosaltres fenyeu",
                "ells fenyen",
            ],
        ),
        (
            "córrer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo corro",
                "tu corres",
                "ell corre",
                "nosaltres correm",
                "vosaltres correu",
                "ells corren",
            ],
        ),
        (
            "transcórrer",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo transcorro",
                "tu transcorres",
                "ell transcorre",
                "nosaltres transcorrem",
                "vosaltres transcorreu",
                "ells transcorren",
            ],
        ),
        (
            "imprimir",
            "indicatiu",
            "present",
            False,
            "m",
            [
                "jo imprimeixo",
                "tu imprimeixes",
                "ell imprimeix",
                "nosaltres imprimim",
                "vosaltres imprimiu",
                "ells imprimeixen",
            ],
        ),
        (
            "imprimir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo imprimeixo",
                "tu imprimeixes",
                "ella imprimeix",
                "nosaltres imprimim",
                "vosaltres imprimiu",
                "elles imprimeixen",
            ],
        ),
        (
            "vindre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo vindo",
                "tu vinds",
                "ella vind",
                "nosaltres vindem",
                "vosaltres vindeu",
                "elles vinden",
            ],
        ),
        (
            "tòrcer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo torço",
                "tu torces",
                "ella torça",
                "nosaltres torcem",
                "vosaltres torceu",
                "elles torcen",
            ],
        ),
        (
            "caldre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "-",
                "-",
                "ella cal",
                "-",
                "-",
                "elles calen",
            ],
        ),
        (
            "obeir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo obeeixo",
                "tu obeeixes",
                "ella obeeix",
                "nosaltres obeïm",
                "vosaltres obeïu",
                "elles obeeixen",
            ],
        ),
        (
            "beneir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo beneeixo",
                "tu beneeixes",
                "ella beneeix",
                "nosaltres beneïm",
                "vosaltres beneïu",
                "elles beneeixen",
            ],
        ),
        (
            "coldre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo colc",
                "tu cols",
                "ella col",
                "nosaltres colem",
                "vosaltres coleu",
                "elles colen",
            ],
        ),
        (
            "dissoldre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo dissolc",
                "tu dissols",
                "ella dissol",
                "nosaltres dissolem",
                "vosaltres dissoleu",
                "elles dissolen",
            ],
        ),
        (
            "percudir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo percudeixo",
                "tu percudeixes",
                "ella percudeix",
                "nosaltres percudim",
                "vosaltres percudiu",
                "elles percudeixen",
            ],
        ),
        (
            "percudir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo percudo",
                "tu percuts",
                "ella percut",
                "nosaltres percudim",
                "vosaltres percudiu",
                "elles percuden",
            ],
        ),
        (
            "permetre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo permeto",
                "tu permets",
                "ella permet",
                "nosaltres permetem",
                "vosaltres permeteu",
                "elles permeten",
            ],
        ),
        (
            "desconèixer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo desconec",
                "tu desconeixes",
                "ella desconeix",
                "nosaltres desconeixem",
                "vosaltres desconeixeu",
                "elles desconeixen",
            ],
        ),
        (
            "desconèixer",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo desconeguí",
                "tu desconegueres",
                "ella desconegué",
                "nosaltres desconeguérem",
                "vosaltres desconeguéreu",
                "elles desconegueren",
            ],
        ),
        (
            "parèixer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo parec",
                "tu pareixes",
                "ella pareix",
                "nosaltres pareixem",
                "vosaltres pareixeu",
                "elles pareixen",
            ],
        ),
        (
            "expendre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo expenc",
                "tu expens",
                "ella expèn",
                "nosaltres expenem",
                "vosaltres expeneu",
                "elles expenen",
            ],
        ),
        (
            "contendre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo contenc",
                "tu contens",
                "ella contén",
                "nosaltres contenem",
                "vosaltres conteneu",
                "elles contenen",
            ],
        ),
        (
            "atènyer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo atenyo",
                "tu atenys",
                "ella ateny",
                "nosaltres atenyem",
                "vosaltres atenyeu",
                "elles atenyen",
            ],
        ),
        (
            "descloure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo descloc",
                "tu desclous",
                "ella desclou",
                "nosaltres descloem",
                "vosaltres descloeu",
                "elles desclouen",
            ],
        ),
        (
            "descloure",
            "particip",
            "particip",
            False,
            "f",
            ["desclòs", "desclosa", "desclosos", "descloses"],
        ),
        (
            "cloure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo cloc",
                "tu clous",
                "ella clou",
                "nosaltres cloem",
                "vosaltres cloeu",
                "elles clouen",
            ],
        ),
        (
            "abreviar",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo abrevio",
                "tu abrevies",
                "ella abrevia",
                "nosaltres abreviem",
                "vosaltres abrevieu",
                "elles abrevien",
            ],
        ),
        (
            "abreviar",
            "subjuntiu",
            "present",
            False,
            "f",
            [
                "jo abreviï",
                "tu abreviïs",
                "ella abreviï",
                "nosaltres abreviem",
                "vosaltres abrevieu",
                "elles abreviïn",
            ],
        ),
        (
            "abreviar",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["abrevia", "abreviï", "abreviem", "abrevieu", "abreviïn"],
        ),
        (
            "compondre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo componc",
                "tu compons",
                "ella compon",
                "nosaltres componem",
                "vosaltres componeu",
                "elles componen",
            ],
        ),
        (
            "contendre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo contenc",
                "tu contens",
                "ella contén",
                "nosaltres contenem",
                "vosaltres conteneu",
                "elles contenen",
            ],
        ),
        (
            "complir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo compleixo",
                "tu compleixes",
                "ella compleix",
                "nosaltres complim",
                "vosaltres compliu",
                "elles compleixen",
            ],
        ),
        (
            "establir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo estableixo",
                "tu estableixes",
                "ella estableix",
                "nosaltres establim",
                "vosaltres establiu",
                "elles estableixen",
            ],
        ),
        (
            "complaure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo complac",
                "tu complaus",
                "ella complau",
                "nosaltres complaem",
                "vosaltres complaeu",
                "elles complauen",
            ],
        ),
        (
            "desplaure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo desplac",
                "tu desplaus",
                "ella desplau",
                "nosaltres desplaem",
                "vosaltres desplaeu",
                "elles desplauen",
            ],
        ),
        (
            "complaure",
            "particip",
            "particip",
            False,
            "f",
            ["complagut", "complaguda", "complaguts", "complagudes"],
        ),
        (
            "acréixer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo acreixo",
                "tu acreixes",
                "ella acreix",
                "nosaltres acreixem",
                "vosaltres acreixeu",
                "elles acreixen",
            ],
        ),
        (
            "acréixer",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo acreixí",
                "tu acreixeres",
                "ella acreixé",
                "nosaltres acreixérem",
                "vosaltres acreixéreu",
                "elles acreixeren",
            ],
        ),
        (
            "acréixer",
            "indicatiu",
            "passat-simple",
            True,
            "f",
            [
                "jo acresquí",
                "tu acresqueres",
                "ella acresqué",
                "nosaltres acresquérem",
                "vosaltres acresquéreu",
                "elles acresqueren",
            ],
        ),
        (
            "acréixer",
            "subjuntiu",
            "imperfet",
            False,
            "f",
            [
                "jo acreixés",
                "tu acreixessis",
                "ella acreixés",
                "nosaltres acreixéssim",
                "vosaltres acreixéssiu",
                "elles acreixessin",
            ],
        ),
        (
            "acréixer",
            "subjuntiu",
            "imperfet",
            True,
            "f",
            [
                "jo acresqués",
                "tu acresquessis",
                "ella acresqués",
                "nosaltres acresquéssim",
                "vosaltres acresquéssiu",
                "elles acresquessin",
            ],
        ),
        (
            "merèixer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo mereixo",
                "tu mereixes",
                "ella mereix",
                "nosaltres mereixem",
                "vosaltres mereixeu",
                "elles mereixen",
            ],
        ),
        (
            "confondre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo confonc",
                "tu confons",
                "ella confon",
                "nosaltres confonem",
                "vosaltres confoneu",
                "elles confonen",
            ],
        ),
        (
            "consternar",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo consterno",
                "tu consternes",
                "ella consterna",
                "nosaltres consternem",
                "vosaltres consterneu",
                "elles consternen",
            ],
        ),
        (
            "contradir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo contradic",
                "tu contradius",
                "ella contradiu",
                "nosaltres contradiem",
                "vosaltres contradieu",
                "elles contradiuen",
            ],
        ),
        (
            "contradir",
            "particip",
            "particip",
            False,
            "f",
            ["contradit", "contradida", "contradits", "contradides"],
        ),
        (
            "dir",
            "particip",
            "particip",
            False,
            "f",
            ["dit", "dita", "dits", "dites"],
        ),
        (
            "creure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo crec",
                "tu creus",
                "ella creu",
                "nosaltres creiem",
                "vosaltres creieu",
                "elles creuen",
            ],
        ),
        (
            "creure",
            "indicatiu",
            "futur",
            False,
            "f",
            [
                "jo creuré",
                "tu creuràs",
                "ella creurà",
                "nosaltres creurem",
                "vosaltres creureu",
                "elles creuran",
            ],
        ),
        (
            "dur",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo duc",
                "tu duus",
                "ella duu",
                "nosaltres duem",
                "vosaltres dueu",
                "elles duen",
            ],
        ),
        (
            "dur",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo duc",
                "tu dus",
                "ella du",
                "nosaltres duem",
                "vosaltres dueu",
                "elles duen",
            ],
        ),
        (
            "dur",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["duu", "dugui", "duguem", "dueu", "duguin"],
        ),
        (
            "dur",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["du", "dugui", "duguem", "dueu", "duguin"],
        ),
        (
            "descosir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo descuso",
                "tu descuses",
                "ella descús",
                "nosaltres descosim",
                "vosaltres descosiu",
                "elles descusen",
            ],
        ),
        (
            "descosir",
            "subjuntiu",
            "present",
            True,
            "f",
            [
                "jo descusi",
                "tu descusis",
                "ella descusi",
                "nosaltres descosim",
                "vosaltres descosiu",
                "elles descusin",
            ],
        ),
        (
            "descosir",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["descús", "descusi", "descosim", "descosiu", "descusin"],
        ),
        (
            "valer",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo valc",
                "tu vals",
                "ella val",
                "nosaltres valem",
                "vosaltres valeu",
                "elles valen",
            ],
        ),
        (
            "valer",
            "indicatiu",
            "passat-simple",
            True,
            "f",
            [
                "jo valguí",
                "tu valgueres",
                "ella valgué",
                "nosaltres valguérem",
                "vosaltres valguéreu",
                "elles valgueren",
            ],
        ),
        (
            "doldre",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo dolc",
                "tu dols",
                "ella dol",
                "nosaltres dolem",
                "vosaltres doleu",
                "elles dolen",
            ],
        ),
        (
            "entrelluir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo entrelluo",
                "tu entrelluus",
                "ella entrelluu",
                "nosaltres entrelluïm",
                "vosaltres entrelluïu",
                "elles entrelluen",
            ],
        ),
        (
            "entrelluir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo entrelluo",
                "tu entrelluus",
                "ella entrellú",
                "nosaltres entrelluïm",
                "vosaltres entrelluïu",
                "elles entrelluen",
            ],
        ),
        (
            "entrelluir",
            "indicatiu",
            "imperfet",
            True,
            "f",
            [
                "jo entrelluïa",
                "tu entrelluïes",
                "ella entrelluïa",
                "nosaltres entrelluíem",
                "vosaltres entrelluíeu",
                "elles entrelluïen",
            ],
        ),
        (
            "entrelluir",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["entrelluu", "entrelluï", "entrelluïm", "entrelluïu", "entrelluïn"],
        ),
        (
            "entrelluir",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["entrellú", "entrelluï", "entrelluïm", "entrelluïu", "entrelluïn"],
        ),
        (
            "esglaiar",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo esglaio",
                "tu esglaies",
                "ella esglaia",
                "nosaltres esglaiem",
                "vosaltres esglaieu",
                "elles esglaien",
            ],
        ),
        (
            "esglaiar",
            "indicatiu",
            "passat-simple",
            True,
            "f",
            [
                "jo esglaí",
                "tu esglaiares",
                "ella esglaià",
                "nosaltres esglaiàrem",
                "vosaltres esglaiàreu",
                "elles esglaiaren",
            ],
        ),
        (
            "esglaiar",
            "subjuntiu",
            "present",
            True,
            "f",
            [
                "jo esglaï",
                "tu esglaïs",
                "ella esglaï",
                "nosaltres esglaiem",
                "vosaltres esglaieu",
                "elles esglaïn",
            ],
        ),
        (
            "esglaiar",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["esglaia", "esglaï", "esglaiem", "esglaieu", "esglaïn"],
        ),
        (
            "esglaiar",
            "particip",
            "particip",
            False,
            "f",
            ["esglaiat", "esglaiada", "esglaiats", "esglaiades"],
        ),
        (
            "evacuar",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo evacuo",
                "tu evacues",
                "ella evacua",
                "nosaltres evacuem",
                "vosaltres evacueu",
                "elles evacuen",
            ],
        ),
        (
            "evacuar",
            "subjuntiu",
            "present",
            True,
            "f",
            [
                "jo evacuï",
                "tu evacuïs",
                "ella evacuï",
                "nosaltres evacuem",
                "vosaltres evacueu",
                "elles evacuïn",
            ],
        ),
        (
            "evacuar",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["evacua", "evacuï", "evacuem", "evacueu", "evacuïn"],
        ),
        (
            "tondre",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo tonc",
                "tu tons",
                "ella ton",
                "nosaltres tonem",
                "vosaltres toneu",
                "elles tonen",
            ],
        ),
        (
            "fugir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo fujo",
                "tu fuges",
                "ella fuig",
                "nosaltres fugim",
                "vosaltres fugiu",
                "elles fugen",
            ],
        ),
        (
            "fugir",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["fuig", "fugi", "fugim", "fugiu", "fugin"],
        ),
        (
            "imprimir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo imprimeixo",
                "tu imprimeixes",
                "ella imprimeix",
                "nosaltres imprimim",
                "vosaltres imprimiu",
                "elles imprimeixen",
            ],
        ),
        (
            "imprimir",
            "particip",
            "particip",
            False,
            "f",
            ["imprès", "impresa", "impresos", "impreses"],
        ),
        (
            "ajeure",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo ajec",
                "tu ajeus",
                "ella ajeu",
                "nosaltres ajaiem",
                "vosaltres ajaieu",
                "elles ajeuen",
            ],
        ),
        (
            "moldre",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo molc",
                "tu mols",
                "ella mol",
                "nosaltres molem",
                "vosaltres moleu",
                "elles molen",
            ],
        ),
        (
            "malvendre",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo malvenc",
                "tu malvens",
                "ella malvèn",
                "nosaltres malvenem",
                "vosaltres malveneu",
                "elles malvenen",
            ],
        ),
        (
            "malvendre",
            "indicatiu",
            "imperfet",
            True,
            "f",
            [
                "jo malvenia",
                "tu malvenies",
                "ella malvenia",
                "nosaltres malveníem",
                "vosaltres malveníeu",
                "elles malvenien",
            ],
        ),
        (
            "malvendre",
            "indicatiu",
            "passat-simple",
            True,
            "f",
            [
                "jo malvenguí",
                "tu malvengueres",
                "ella malvengué",
                "nosaltres malvenguérem",
                "vosaltres malvenguéreu",
                "elles malvengueren",
            ],
        ),
        (
            "malvendre",
            "subjuntiu",
            "present",
            True,
            "f",
            [
                "jo malvengui",
                "tu malvenguis",
                "ella malvengui",
                "nosaltres malvenguem",
                "vosaltres malvengueu",
                "elles malvenguin",
            ],
        ),
        (
            "malvendre",
            "subjuntiu",
            "imperfet",
            True,
            "f",
            [
                "jo malvengués",
                "tu malvenguessis",
                "ella malvengués",
                "nosaltres malvenguéssim",
                "vosaltres malvenguéssiu",
                "elles malvenguessin",
            ],
        ),
        (
            "omplir",
            "indicatiu",
            "present",
            True,
            "f",
            [
                "jo omplo",
                "tu omples",
                "ella omple",
                "nosaltres omplim",
                "vosaltres ompliu",
                "elles omplen",
            ],
        ),
        (
            "omplir",
            "particip",
            "particip",
            False,
            "f",
            ["omplert", "omplerta", "omplerts", "omplertes"],
        ),
        (
            "omplir",
            "particip",
            "particip",
            True,
            "f",
            ["omplit", "omplida", "omplits", "omplides"],
        ),
        (
            "omplir",
            "imperatiu",
            "imperatiu-present",
            True,
            "m",
            ["omple", "ompli", "omplim", "ompliu", "omplin"],
        ),
        (
            "moure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo moc",
                "tu mous",
                "ella mou",
                "nosaltres movem",
                "vosaltres moveu",
                "elles mouen",
            ],
        ),
        (
            "moure",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo moguí",
                "tu mogueres",
                "ella mogué",
                "nosaltres moguérem",
                "vosaltres moguéreu",
                "elles mogueren",
            ],
        ),
        (
            "moure",
            "indicatiu",
            "imperfet",
            False,
            "f",
            [
                "jo movia",
                "tu movies",
                "ella movia",
                "nosaltres movíem",
                "vosaltres movíeu",
                "elles movien",
            ],
        ),
        (
            "absoldre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo absolc",
                "tu absols",
                "ella absol",
                "nosaltres absolem",
                "vosaltres absoleu",
                "elles absolen",
            ],
        ),
        (
            "absoldre",
            "particip",
            "particip",
            True,
            "f",
            ["absolt", "absolta", "absolts", "absoltes"],
        ),
        (
            "obrir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo obro",
                "tu obres",
                "ella obre",
                "nosaltres obrim",
                "vosaltres obriu",
                "elles obren",
            ],
        ),
        (
            "obrir",
            "particip",
            "particip",
            False,
            "f",
            ["obert", "oberta", "oberts", "obertes"],
        ),
        (
            "prevenir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo previnc",
                "tu prevens",
                "ella prevé",
                "nosaltres prevenim",
                "vosaltres preveniu",
                "elles prevenen",
            ],
        ),
        (
            "prevenir",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo previnguí",
                "tu previngueres",
                "ella previngué",
                "nosaltres previnguérem",
                "vosaltres previnguéreu",
                "elles previngueren",
            ],
        ),
        (
            "prevenir",
            "indicatiu",
            "imperfet",
            False,
            "f",
            [
                "jo prevenia",
                "tu prevenies",
                "ella prevenia",
                "nosaltres preveníem",
                "vosaltres preveníeu",
                "elles prevenien",
            ],
        ),
        (
            "prevenir",
            "indicatiu",
            "futur",
            False,
            "f",
            [
                "jo previndré",
                "tu previndràs",
                "ella previndrà",
                "nosaltres previndrem",
                "vosaltres previndreu",
                "elles previndran",
            ],
        ),
        (
            "prevenir",
            "subjuntiu",
            "present",
            False,
            "f",
            [
                "jo previngui",
                "tu previnguis",
                "ella previngui",
                "nosaltres previnguem",
                "vosaltres previngueu",
                "elles previnguin",
            ],
        ),
        (
            "prevenir",
            "subjuntiu",
            "imperfet",
            False,
            "f",
            [
                "jo previngués",
                "tu previnguessis",
                "ella previngués",
                "nosaltres previnguéssim",
                "vosaltres previnguéssiu",
                "elles previnguessin",
            ],
        ),
        (
            "prevenir",
            "condicional",
            "present",
            False,
            "f",
            [
                "jo previndria",
                "tu previndries",
                "ella previndria",
                "nosaltres previndríem",
                "vosaltres previndríeu",
                "elles previndrien",
            ],
        ),
        (
            "prevenir",
            "particip",
            "particip",
            False,
            "f",
            ["previngut", "previnguda", "previnguts", "previngudes"],
        ),
        (
            "prevenir",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["prevén", "previngui", "previnguem", "preveniu", "previnguin"],
        ),
        (
            "sortir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo surto",
                "tu surts",
                "ella surt",
                "nosaltres sortim",
                "vosaltres sortiu",
                "elles surten",
            ],
        ),
        (
            "sortir",
            "subjuntiu",
            "present",
            False,
            "f",
            [
                "jo surti",
                "tu surtis",
                "ella surti",
                "nosaltres sortim",
                "vosaltres sortiu",
                "elles surtin",
            ],
        ),
        (
            "sortir",
            "particip",
            "particip",
            False,
            "f",
            ["sortit", "sortida", "sortits", "sortides"],
        ),
        (
            "sortir",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["surt", "surti", "sortim", "sortiu", "surtin"],
        ),
        (
            "témer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo temo",
                "tu tems",
                "ella tem",
                "nosaltres temem",
                "vosaltres temeu",
                "elles temen",
            ],
        ),
        (
            "témer",
            "infinitiu",
            "infinitiu-present",
            False,
            "m",
            ["témer"],
        ),
        (
            "témer",
            "gerundi",
            "gerundi",
            False,
            "m",
            ["tement"],
        ),
        (
            "témer",
            "particip",
            "particip",
            False,
            "f",
            ["temut", "temuda", "temuts", "temudes"],
        ),
        (
            "retraure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo retrec",
                "tu retreus",
                "ella retreu",
                "nosaltres retraiem",
                "vosaltres retraieu",
                "elles retreuen",
            ],
        ),
        (
            "retraure",
            "indicatiu",
            "imperfet",
            False,
            "f",
            [
                "jo retreia",
                "tu retreies",
                "ella retreia",
                "nosaltres retrèiem",
                "vosaltres retrèieu",
                "elles retreien",
            ],
        ),
        (
            "retraure",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo retraguí",
                "tu retragueres",
                "ella retragué",
                "nosaltres retraguérem",
                "vosaltres retraguéreu",
                "elles retragueren",
            ],
        ),
        (
            "retraure",
            "indicatiu",
            "futur",
            False,
            "f",
            [
                "jo retraué",
                "tu retrauàs",
                "ella retrauà",
                "nosaltres retrauem",
                "vosaltres retraueu",
                "elles retrauan",
            ],
        ),
        (
            "retraure",
            "condicional",
            "present",
            False,
            "f",
            [
                "jo retrauia",
                "tu retrauies",
                "ella retrauia",
                "nosaltres retrauíem",
                "vosaltres retrauíeu",
                "elles retrauien",
            ],
        ),
        (
            "retraure",
            "subjuntiu",
            "present",
            False,
            "f",
            [
                "jo retregui",
                "tu retreguis",
                "ella retregui",
                "nosaltres retreguem",
                "vosaltres retregueu",
                "elles retreguin",
            ],
        ),
        (
            "premorir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo premoro",
                "tu premors",
                "ella premor",
                "nosaltres premorim",
                "vosaltres premoriu",
                "elles premoren",
            ],
        ),
        (
            "conviure",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo convisc",
                "tu convius",
                "ella conviu",
                "nosaltres convivim",
                "vosaltres conviviu",
                "elles conviuen",
            ],
        ),
        (
            "conviure",
            "indicatiu",
            "imperfet",
            False,
            "f",
            [
                "jo convivia",
                "tu convivies",
                "ella convivia",
                "nosaltres convivíem",
                "vosaltres convivíeu",
                "elles convivien",
            ],
        ),
        (
            "conviure",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo convisquí",
                "tu convisqueres",
                "ella convisqué",
                "nosaltres convisquérem",
                "vosaltres convisquéreu",
                "elles convisqueren",
            ],
        ),
        (
            "vèncer",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo venço",
                "tu vences",
                "ella venç",
                "nosaltres vencem",
                "vosaltres venceu",
                "elles vencen",
            ],
        ),
        (
            "vèncer",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["venç", "venci", "vencem", "venceu", "vencin"],
        ),
        (
            "vèncer",
            "particip",
            "particip",
            False,
            "f",
            ["vençut", "vençuda", "vençuts", "vençudes"],
        ),
        (
            "vèncer",
            "infinitiu",
            "infinitiu-present",
            False,
            "m",
            ["vèncer"],
        ),
        (
            "vèncer",
            "gerundi",
            "gerundi",
            False,
            "m",
            ["vencent"],
        ),
        (
            "cerndre",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo cerno",
                "tu cerns",
                "ella cern",
                "nosaltres cernem",
                "vosaltres cerneu",
                "elles cernen",
            ],
        ),
        (
            "oir",
            "indicatiu",
            "present",
            False,
            "f",
            [
                "jo oeixo",
                "tu oeixes",
                "ella oeix",
                "nosaltres oïm",
                "vosaltres oïu",
                "elles oeixen",
            ],
        ),
        (
            "oir",
            "indicatiu",
            "imperfet",
            False,
            "f",
            [
                "jo oïa",
                "tu oïes",
                "ella oïa",
                "nosaltres oíem",
                "vosaltres oíeu",
                "elles oïen",
            ],
        ),
        (
            "oir",
            "indicatiu",
            "passat-simple",
            False,
            "f",
            [
                "jo oí",
                "tu oïres",
                "ella oí",
                "nosaltres oírem",
                "vosaltres oíreu",
                "elles oïren",
            ],
        ),
        (
            "oir",
            "indicatiu",
            "futur",
            False,
            "f",
            [
                "jo oiré",
                "tu oiràs",
                "ella oirà",
                "nosaltres oirem",
                "vosaltres oireu",
                "elles oiran",
            ],
        ),
        (
            "oir",
            "subjuntiu",
            "present",
            False,
            "f",
            [
                "jo oeixi",
                "tu oeixis",
                "ella oeixi",
                "nosaltres oïm",
                "vosaltres oïu",
                "elles oeixin",
            ],
        ),
        (
            "oir",
            "subjuntiu",
            "imperfet",
            False,
            "f",
            [
                "jo oís",
                "tu oïssis",
                "ella oís",
                "nosaltres oíssim",
                "vosaltres oíssiu",
                "elles oïssin",
            ],
        ),
        (
            "oir",
            "condicional",
            "present",
            False,
            "f",
            [
                "jo oiria",
                "tu oiries",
                "ella oiria",
                "nosaltres oiríem",
                "vosaltres oiríeu",
                "elles oirien",
            ],
        ),
        (
            "oir",
            "particip",
            "particip",
            False,
            "f",
            ["oït", "oïda", "oïts", "oïdes"],
        ),
        (
            "oir",
            "infinitiu",
            "infinitiu-present",
            False,
            "m",
            ["oir"],
        ),
        (
            "oir",
            "gerundi",
            "gerundi",
            False,
            "m",
            ["oint"],
        ),
        (
            "oir",
            "imperatiu",
            "imperatiu-present",
            False,
            "m",
            ["oeix", "oeixi", "oïm", "oïu", "oeixin"],
        ),
    ],
)
def test_inflector_ca_conjugate_mood_tense(
    infinitive, mood, tense, alternate, gender, expected_result
):
    co = cg._inflector._get_conj_obs(infinitive)
    print("template:" + co.template.name)
    assert (
        cg.conjugate_mood_tense(infinitive, mood, tense, alternate, gender)
        == expected_result
    )


def test_inflector_ca_get_conj_obs():
    co = cg._inflector._get_conj_obs("parlar")
    assert co.verb.infinitive == "parlar"
    assert co.verb_stem == "parl"
    assert co.template.name == "cant:ar"


def test_inflector_ca_get_conj_obs_2():
    co = cg._inflector._get_conj_obs("abandonar")
    assert co.verb.infinitive == "abandonar"
    assert co.verb_stem == "abandon"
    assert co.template.name == "cant:ar"


def test_inflector_ca_get_conj_obs_3():
    co = cg._inflector._get_conj_obs("pertànyer")
    assert co.verb.infinitive == "pertànyer"
    assert co.verb_stem == "pertàny"
    assert co.template.name == "pertàny:er"
    assert co.template.modify_stem == "strip-accents"


def test_inflector_ca_get_verb_stem_from_template_name():
    verb_stem = cg._inflector._get_verb_stem_from_template_name("parlar", "cant:ar")
    assert verb_stem == "parl"


def test_inflector_ca_get_verb_stem_from_template_name_2():
    verb_stem = cg._inflector._get_verb_stem_from_template_name("abandonar", "cant:ar")
    assert verb_stem == "abandon"


def test_inflector_ca_conjugate_simple_mood_tense():
    verb_stem = "parl"
    tense_elem = etree.fromstring(
        """<present>
            <p><i>o</i></p>
            <p><i>es</i></p>
            <p><i>a</i></p>
            <p><i>em</i></p>
            <p><i>eu</i></p>
            <p><i>en</i></p>
        </present>""",
        parser=None,
    )
    tense_name = "present"
    tense_template = TenseTemplate(tense_elem)
    out = cg._inflector._conjugate_simple_mood_tense(
        verb_stem, "indicatiu", tense_template
    )
    assert len(out) == 6
    assert out == [
        "jo parlo",
        "tu parles",
        "ell parla",
        "nosaltres parlem",
        "vosaltres parleu",
        "ells parlen",
    ]


@pytest.mark.parametrize(
    "person,gender,is_reflexive,expected_result",
    [
        ("1s", "m", False, "jo"),
        ("1s", "m", True, "jo me"),
        ("2s", "m", False, "tu"),
        ("2s", "m", True, "tu te"),
        ("3s", "m", False, "ell"),
        ("3s", "m", True, "ell se"),
        ("3s", "f", False, "ella"),
        ("3s", "f", True, "ella se"),
        ("1p", "m", False, "nosaltres"),
        ("1p", "m", True, "nosaltres nos"),
        ("2p", "m", False, "vosaltres"),
        ("2p", "m", True, "vosaltres os"),
        ("3p", "m", False, "ells"),
        ("3p", "m", True, "ells se"),
        ("3p", "f", False, "elles"),
        ("3p", "f", True, "elles se"),
    ],
)
def test_inflector_ca_get_default_pronoun(
    person, gender, is_reflexive, expected_result
):
    assert (
        cg._inflector._get_default_pronoun(person, gender, is_reflexive=is_reflexive)
        == expected_result
    )


def test_inflector_conjugate():
    assert cg.conjugate("ser") == {
        "verb": {
            "infinitive": "ser",
            "predicted": False,
            "pred_score": 1.0,
            "template": "és:ser",
            "translation_en": "be",
            "stem": "",
        },
        "moods": {
            "indicatiu": {
                "present": [
                    "jo sóc",
                    "tu ets",
                    "ell és",
                    "nosaltres som",
                    "vosaltres sou",
                    "ells són",
                ],
                "imperfet": [
                    "jo era",
                    "tu eres",
                    "ell era",
                    "nosaltres érem",
                    "vosaltres éreu",
                    "ells eren",
                ],
                "passat-simple": [
                    "jo fui",
                    "tu fores",
                    "ell fou",
                    "nosaltres fórem",
                    "vosaltres fóreu",
                    "ells foren",
                ],
                "futur": [
                    "jo seré",
                    "tu seràs",
                    "ell serà",
                    "nosaltres serem",
                    "vosaltres sereu",
                    "ells seran",
                ],
            },
            "subjuntiu": {
                "present": [
                    "jo sigui",
                    "tu siguis",
                    "ell sigui",
                    "nosaltres siguem",
                    "vosaltres sigueu",
                    "ells siguin",
                ],
                "imperfet": [
                    "jo fos",
                    "tu fossis",
                    "ell fos",
                    "nosaltres fóssim",
                    "vosaltres fóssiu",
                    "ells fossin",
                ],
            },
            "imperatiu": {
                "imperatiu-present": ["sigues", "sigui", "siguem", "sigueu", "siguin"]
            },
            "condicional": {
                "present": [
                    "jo seria",
                    "tu series",
                    "ell seria",
                    "nosaltres seríem",
                    "vosaltres seríeu",
                    "ells serien",
                ]
            },
            "infinitiu": {"infinitiu-present": ["ser"]},
            "gerundi": {"gerundi": ["sent"]},
            "particip": {"particip": ["estat", "estada", "estats", "estades"]},
        },
    }


def test_inflector_conjugate_with_alternates():
    assert cg.conjugate("ser", include_alternates=True) == {
        "verb": {
            "infinitive": "ser",
            "predicted": False,
            "pred_score": 1.0,
            "template": "és:ser",
            "translation_en": "be",
            "stem": "",
        },
        "moods": {
            "indicatiu": {
                "present": [
                    ["jo sóc"],
                    ["tu ets"],
                    ["ell és"],
                    ["nosaltres som"],
                    ["vosaltres sou"],
                    ["ells són"],
                ],
                "imperfet": [
                    ["jo era"],
                    ["tu eres"],
                    ["ell era"],
                    ["nosaltres érem"],
                    ["vosaltres éreu"],
                    ["ells eren"],
                ],
                "passat-simple": [
                    ["jo fui"],
                    ["tu fores"],
                    ["ell fou"],
                    ["nosaltres fórem"],
                    ["vosaltres fóreu"],
                    ["ells foren"],
                ],
                "futur": [
                    ["jo seré"],
                    ["tu seràs"],
                    ["ell serà"],
                    ["nosaltres serem"],
                    ["vosaltres sereu"],
                    ["ells seran"],
                ],
            },
            "subjuntiu": {
                "present": [
                    ["jo sigui"],
                    ["tu siguis"],
                    ["ell sigui"],
                    ["nosaltres siguem"],
                    ["vosaltres sigueu"],
                    ["ells siguin"],
                ],
                "imperfet": [
                    ["jo fos"],
                    ["tu fossis"],
                    ["ell fos"],
                    ["nosaltres fóssim"],
                    ["vosaltres fóssiu"],
                    ["ells fossin"],
                ],
            },
            "imperatiu": {
                "imperatiu-present": [
                    ["sigues"],
                    ["sigui"],
                    ["siguem"],
                    ["sigueu"],
                    ["siguin"],
                ]
            },
            "condicional": {
                "present": [
                    ["jo seria", "jo fora"],
                    ["tu series", "tu fores"],
                    ["ell seria", "ell fora"],
                    ["nosaltres seríem", "nosaltres fórem"],
                    ["vosaltres seríeu", "vosaltres fóreu"],
                    ["ells serien", "ells foren"],
                ]
            },
            "infinitiu": {"infinitiu-present": [["ser", "ésser"]]},
            "gerundi": {"gerundi": [["sent", "essent"]]},
            "particip": {
                "particip": [
                    ["estat", "sigut"],
                    ["estada", "siguda"],
                    ["estats", "siguts"],
                    ["estades", "sigudes"],
                ]
            },
        },
    }

def test_inflector_conjugate_with_alternates_noconjpronouns():
    assert cg.conjugate("ser", include_alternates=True, conjugate_pronouns=False) == {
        "verb": {
            "infinitive": "ser",
            "predicted": False,
            "pred_score": 1.0,
            "template": "és:ser",
            "translation_en": "be",
            "stem": "",
        },
        "moods": {
            "indicatiu": {
                "present": [
                    ["sóc"],
                    ["ets"],
                    ["és"],
                    ["som"],
                    ["sou"],
                    ["són"],
                ],
                "imperfet": [
                    ["era"],
                    ["eres"],
                    ["era"],
                    ["érem"],
                    ["éreu"],
                    ["eren"],
                ],
                "passat-simple": [
                    ["fui"],
                    ["fores"],
                    ["fou"],
                    ["fórem"],
                    ["fóreu"],
                    ["foren"],
                ],
                "futur": [
                    ["seré"],
                    ["seràs"],
                    ["serà"],
                    ["serem"],
                    ["sereu"],
                    ["seran"],
                ],
            },
            "subjuntiu": {
                "present": [
                    ["sigui"],
                    ["siguis"],
                    ["sigui"],
                    ["siguem"],
                    ["sigueu"],
                    ["siguin"],
                ],
                "imperfet": [
                    ["fos"],
                    ["fossis"],
                    ["fos"],
                    ["fóssim"],
                    ["fóssiu"],
                    ["fossin"],
                ],
            },
            "imperatiu": {
                "imperatiu-present": [
                    ["sigues"],
                    ["sigui"],
                    ["siguem"],
                    ["sigueu"],
                    ["siguin"],
                ]
            },
            "condicional": {
                "present": [
                    ["seria", "fora"],
                    ["series", "fores"],
                    ["seria", "fora"],
                    ["seríem", "fórem"],
                    ["seríeu", "fóreu"],
                    ["serien", "foren"],
                ]
            },
            "infinitiu": {"infinitiu-present": [["ser", "ésser"]]},
            "gerundi": {"gerundi": [["sent", "essent"]]},
            "particip": {
                "particip": [
                    ["estat", "sigut"],
                    ["estada", "siguda"],
                    ["estats", "siguts"],
                    ["estades", "sigudes"],
                ]
            },
        },
    }
