import os
from pathlib import Path

import bokeh.embed
import numpy as np
import pandas as pd
from bokeh.models import HoverTool, LogColorMapper

from pydatalab.blocks.base import DataBlock
from pydatalab.bokeh_plots import DATALAB_BOKEH_THEME, selectable_axes_plot
from pydatalab.file_utils import get_file_info_by_id
from pydatalab.logger import LOGGER


class FTIRBlock(DataBlock):
    accepted_file_extensions: tuple[str, ...] = (".asp",)
    blocktype = "ftir"
    name = "FTIR"
    description = (
        "This block can plot FTIR data from .asp files generated by an Agilent Spectrometer"
    )

    @property
    def plot_functions(self):
        return (self.generate_ftir_plot,)

    @classmethod
    def parse_ftir_asp(cls, filename: Path) -> pd.DataFrame:
        """Parses .asp FTIR data generated by an Agilent Spectrometer

        The file consists of a header with the number of points, the start and end wavenumber, a few lines of metadata,
        and then the absorbance data. This function reads the file, extracts the absorbance data, and returns it as a
        pandas DataFrame.

        Args:
            filename: Path to the .asp file

        Returns:
            FTIR dataframe with columns "Wavenumber" and "Absorbance"" with the wavenumber in cm^-1
        """

        ftir = pd.read_csv(filename, header=None)
        number_of_points = int(ftir[0].iloc[0])
        start_wavenumber = ftir[0].iloc[1]
        end_wavenumber = ftir[0].iloc[2]
        x_range = np.linspace(start_wavenumber, end_wavenumber, number_of_points)
        y = ftir[0].iloc[-number_of_points:]
        ftir = pd.DataFrame.from_dict({"Wavenumber": x_range, "Absorbance": y})
        return ftir

    @classmethod
    def _format_ftir_plot(self, ftir_data: pd.DataFrame) -> bokeh.layouts.layout:
        """Formats FTIR data for plotting in Bokeh, inverted x-axis with a buffer of 50 cm^-1 on either side

        Args:
            ftir_data (pd.DataFrame): FTIR data with columns "Wavenumber" and "Absorbance"

        Returns:
            bokeh.layouts.layout: Bokeh layout with FTIR data plotted
        """
        layout = selectable_axes_plot(
            ftir_data,
            x_options=["Wavenumber"],
            y_options=["Absorbance"],
            x_range=(ftir_data["Wavenumber"].max() + 50, ftir_data["Wavenumber"].min() - 50),
            # color_options=["Frequency [Hz]"],
            color_mapper=LogColorMapper("Cividis256"),
            plot_points=False,
            plot_line=True,
            tools=HoverTool(
                tooltips=[
                    ("Wavenumber / cm⁻¹", "@Wavenumber{0.00}"),
                    ("Absorbance", "@Absorbance{0.0000}"),
                ],  # Display x and y values to specified decimal places
                mode="vline",  # Ensures hover follows the x-axis
            ),
        )
        # Adding cm^-1 to the x-axis label using unicode characters - might be a more logical way
        layout.children[1].xaxis.axis_label = "Wavenumber / cm⁻¹"
        return layout

    def generate_ftir_plot(self):
        file_info = None
        # all_files = None
        ftir_data = None

        if "file_id" not in self.data:
            LOGGER.warning("No file set in the DataBlock")
            return
        else:
            file_info = get_file_info_by_id(self.data["file_id"], update_if_live=True)
            ext = os.path.splitext(file_info["location"].split("/")[-1])[-1].lower()
            if ext not in self.accepted_file_extensions:
                LOGGER.warning(
                    "Unsupported file extension (must be one of %s, not %s)",
                    self.accepted_file_extensions,
                    ext,
                )
                return

            ftir_data = self.parse_ftir_asp(Path(file_info["location"]))

        if ftir_data is not None:
            layout = self._format_ftir_plot(ftir_data)
            self.data["bokeh_plot_data"] = bokeh.embed.json_item(layout, theme=DATALAB_BOKEH_THEME)
