from __future__ import annotations

from typing_extensions import Literal

from abqpy.decorators import abaqus_class_doc, abaqus_method_doc

from ..UtilityAndView.abaqusConstants import AXIS_3, ROTATION_NONE, SymbolicConstant
from ..UtilityAndView.abaqusConstants import abaqusConstants as C


@abaqus_class_doc
class SectionLayer:
    """The SectionLayer object defines the material layer in a composite shell.

    .. note::
        This object can be accessed by::

            import section
            mdb.models[name].parts[name].compositeLayups[i].section.layup[i]
            mdb.models[name].sections[name].layup[i]
            import odbSection
            session.odbs[name].sections[name].layup[i]

        The corresponding analysis keywords are:

        - SHELL SECTION
        - SHELL GENERAL SECTION

    .. versionchanged:: 2017
        The ``thicknessType`` attribute and ``thicknessField`` attribute were removed.
    """

    #: A Float specifying the thickness of the section layer.
    thickness: float

    #: A String specifying the name of the section layer material.
    material: str

    #: A Float or a String specifying the relative orientation of the section layer. A Float
    #: specifies the angular orientation; a String specifies a user-subroutine orientation
    #: name. If a String is specified, a user-subroutine orientation is used, otherwise the
    #: Float value is used as an angular orientation. The default value is 0.0.
    orientAngle: float = 0

    #: An Int specifying the number of integration points to be used through the section. This
    #: argument is valid only if the **preIntegrate** argument on the parent
    #: CompositeShellSection object is set to ON. The default value is 3.
    numIntPts: int = 3

    #: A SymbolicConstant specifying the axis of a cylindrical or spherical datum coordinate
    #: system about which an additional rotation is applied. For shells this axis is also the
    #: shell normal. This only applies if a valid reference is provided for the orientation.
    #: Possible values are AXIS_1, AXIS_2, and AXIS_3. The default value is AXIS_3.
    axis: SymbolicConstant = AXIS_3

    #: A Float specifying the angle of the additional rotation. This only applies if a valid
    #: reference is provided for the orientation. The default value is 0.0.
    angle: float = 0

    #: A SymbolicConstant specifying the method used to describe the additional rotation when a
    #: valid orientation is specified. Possible values are ROTATION_NONE, ROTATION_ANGLE, and
    #: ROTATION_FIELD. The default value is ROTATION_NONE.
    additionalRotationType: SymbolicConstant = ROTATION_NONE

    #: A String specifying the ply identifier for this section layer. The default value is "".
    plyName: str = ""

    #: The SymbolicConstant None or a DatumCsys object specifying a coordinate system reference
    #: for the relative orientation of this layer. If this reference is valid it is used as the
    #: relative orientation of the layer, otherwise the **orientAngle** is used as described. The
    #: default value is None.
    orientation: SymbolicConstant

    #: A String specifying the name of the field specifying the additional rotation. The
    #: default value is "".
    additionalRotationField: str = ""

    @abaqus_method_doc
    def __init__(
        self,
        thickness: float,
        material: str,
        orientAngle: float = 0,
        numIntPts: int = 3,
        axis: Literal[C.AXIS_1, C.AXIS_3, C.AXIS_2] = AXIS_3,
        angle: float = 0,
        additionalRotationType: Literal[C.ROTATION_NONE, C.ROTATION_FIELD, C.ROTATION_ANGLE] = ROTATION_NONE,
        plyName: str = "",
        orientation: SymbolicConstant | None = None,
        additionalRotationField: str = "",
    ):
        """This method creates a SectionLayer object.

        .. note::
            This function can be accessed by::

                section.SectionLayer
                odbSection.SectionLayer

        .. versionchanged:: 2017
            The ``thicknessType`` argument and ``thicknessField`` argument were removed.

        Parameters
        ----------
        thickness
            A Float specifying the thickness of the section layer.
        material
            A String specifying the name of the section layer material.
        orientAngle
            A Float or a String specifying the relative orientation of the section layer. A Float
            specifies the angular orientation; a String specifies a user-subroutine orientation
            name. If a String is specified, a user-subroutine orientation is used, otherwise the
            Float value is used as an angular orientation. The default value is 0.0.
        numIntPts
            An Int specifying the number of integration points to be used through the section. This
            argument is valid only if the **preIntegrate** argument on the parent
            CompositeShellSection object is set to ON. The default value is 3.
        axis
            A SymbolicConstant specifying the axis of a cylindrical or spherical datum coordinate
            system about which an additional rotation is applied. For shells this axis is also the
            shell normal. This only applies if a valid reference is provided for the orientation.
            Possible values are AXIS_1, AXIS_2, and AXIS_3. The default value is AXIS_3.
        angle
            A Float specifying the angle of the additional rotation. This only applies if a valid
            reference is provided for the orientation. The default value is 0.0.
        additionalRotationType
            A SymbolicConstant specifying the method used to describe the additional rotation when a
            valid orientation is specified. Possible values are ROTATION_NONE, ROTATION_ANGLE, and
            ROTATION_FIELD. The default value is ROTATION_NONE.
        plyName
            A String specifying the ply identifier for this section layer. The default value is "".
        orientation
            The SymbolicConstant None or a DatumCsys object specifying a coordinate system reference
            for the relative orientation of this layer. If this reference is valid it is used as the
            relative orientation of the layer, otherwise the **orientAngle** is used as described. The
            default value is None.
        additionalRotationField
            A String specifying the name of the field specifying the additional rotation. The
            default value is "".

        Returns
        -------
        SectionLayer
            A SectionLayer object.
        """
        ...
