from __future__ import annotations

from typing_extensions import Literal

from abqpy.decorators import abaqus_class_doc, abaqus_method_doc

from ..Adaptivity.AdaptiveMeshConstraintState import AdaptiveMeshConstraintState
from ..Adaptivity.AdaptiveMeshDomain import AdaptiveMeshDomain
from ..BoundaryCondition.BoundaryConditionState import BoundaryConditionState
from ..Load.LoadCase import LoadCase
from ..Load.LoadState import LoadState
from ..PredefinedField.PredefinedFieldState import PredefinedFieldState
from ..StepMiscellaneous.Control import Control
from ..StepMiscellaneous.SolverControl import SolverControl
from ..StepOutput.DiagnosticPrint import DiagnosticPrint
from ..StepOutput.FieldOutputRequestState import FieldOutputRequestState
from ..StepOutput.HistoryOutputRequestState import HistoryOutputRequestState
from ..StepOutput.Monitor import Monitor
from ..StepOutput.Restart import Restart
from ..UtilityAndView.abaqusConstants import (
    DIRECT,
    OFF,
    SOLVER_DEFAULT,
    Boolean,
    SymbolicConstant,
)
from ..UtilityAndView.abaqusConstants import abaqusConstants as C
from .AnalysisStep import AnalysisStep


@abaqus_class_doc
class StaticLinearPerturbationStep(AnalysisStep):
    """The StaticLinearPerturbationStep object is used to indicate that the static step should be analyzed as a
    linear perturbation load step. The StaticLinearPerturbationStep object is derived from the AnalysisStep
    object.

    .. note::
        This object can be accessed by::

            import step
            mdb.models[name].steps[name]

        The corresponding analysis keywords are:

        - STATIC
        - STEP
    """

    #: A String specifying the repository key.
    name: str = ""

    #: A SymbolicConstant specifying the type of solver. Possible values are DIRECT and
    #: ITERATIVE. The default value is DIRECT.
    matrixSolver: SymbolicConstant = DIRECT

    #: A SymbolicConstant specifying the type of matrix storage. Possible values are SYMMETRIC,
    #: UNSYMMETRIC, and SOLVER_DEFAULT. The default value is SOLVER_DEFAULT.
    matrixStorage: SymbolicConstant = SOLVER_DEFAULT

    #: A SymbolicConstant specifying the technique used for solving nonlinear equations. Possible values are
    #: FULL_NEWTON and LCP_CONTACT.
    #:
    #: .. versionadded:: 2024
    #:
    #:     The attribute ``solutionTechnique`` was added.
    solutionTechnique: SymbolicConstant = C.FULL_NEWTON

    #: A Float specifying the zone within which potential contact constraints are exposed to the LCP solver,
    #: applicable when solutionTechnique is LCP_CONTACT.
    #:
    #: .. versionadded:: 2024
    #:
    #:     The attribute ``gapDistance`` was added.
    gapDistance: float = 0.0

    #: A Float specifying the scaling factor for the initial gap distance specified by gapDistance or the
    #: internally computed (contact) pair-wise gap distances, applicable when solutionTechnique is LCP_CONTACT.
    #:
    #: .. versionadded:: 2024
    #:
    #:     The attribute ``scaleFactor`` was added.
    scaleFactor: float = 1.0

    #: A String specifying the name of the previous step. The new step appears after this step
    #: in the list of analysis steps.
    previous: str = ""

    #: A String specifying a description of the new step. The default value is an empty string.
    description: str = ""

    #: A SymbolicConstant specifying whether the step has an explicit procedure type
    #: (*procedureType* = ANNEAL, DYNAMIC_EXPLICIT, or DYNAMIC_TEMP_DISPLACEMENT).
    explicit: SymbolicConstant

    #: A Boolean specifying whether the step has a perturbation procedure type.
    perturbation: Boolean = OFF

    #: A Boolean specifying whether the step has a mechanical procedure type.
    nonmechanical: Boolean = OFF

    #: A SymbolicConstant specifying the Abaqus procedure. Possible values are:
    #:
    #: - ANNEAL
    #: - BUCKLE
    #: - COMPLEX_FREQUENCY
    #: - COUPLED_TEMP_DISPLACEMENT
    #: - COUPLED_THERMAL_ELECTRIC
    #: - DIRECT_CYCLIC
    #: - DYNAMIC_IMPLICIT
    #: - DYNAMIC_EXPLICIT
    #: - DYNAMIC_SUBSPACE
    #: - DYNAMIC_TEMP_DISPLACEMENT
    #: - COUPLED_THERMAL_ELECTRICAL_STRUCTURAL
    #: - FREQUENCY
    #: - GEOSTATIC
    #: - HEAT_TRANSFER
    #: - MASS_DIFFUSION
    #: - MODAL_DYNAMICS
    #: - RANDOM_RESPONSE
    #: - RESPONSE_SPECTRUM
    #: - SOILS
    #: - STATIC_GENERAL
    #: - STATIC_LINEAR_PERTURBATION
    #: - STATIC_RIKS
    #: - STEADY_STATE_DIRECT
    #: - STEADY_STATE_MODAL
    #: - STEADY_STATE_SUBSPACE
    #: - VISCO
    procedureType: SymbolicConstant

    #: A Boolean specifying whether the step is suppressed or not. The default value is OFF.
    suppressed: Boolean = OFF

    #: A repository of FieldOutputRequestState objects.
    fieldOutputRequestState: dict[str, FieldOutputRequestState] = {}

    #: A repository of HistoryOutputRequestState objects.
    historyOutputRequestState: dict[str, HistoryOutputRequestState] = {}

    #: A DiagnosticPrint object.
    diagnosticPrint: DiagnosticPrint = DiagnosticPrint()

    #: A Monitor object.
    monitor: Monitor | None = None

    #: A Restart object.
    restart: Restart = Restart()

    #: A repository of AdaptiveMeshConstraintState objects.
    adaptiveMeshConstraintStates: dict[str, AdaptiveMeshConstraintState] = {}

    #: A repository of AdaptiveMeshDomain objects.
    adaptiveMeshDomains: dict[str, AdaptiveMeshDomain] = {}

    #: A Control object.
    control: Control = Control()

    #: A SolverControl object.
    solverControl: SolverControl = SolverControl()

    #: A repository of BoundaryConditionState objects.
    boundaryConditionStates: dict[str, BoundaryConditionState] = {}

    #: A repository of InteractionState objects.
    interactionStates: int | None = None

    #: A repository of LoadState objects.
    loadStates: dict[str, LoadState] = {}

    #: A repository of LoadCase objects.
    loadCases: dict[str, LoadCase] = {}

    #: A repository of PredefinedFieldState objects.
    predefinedFieldStates: dict[str, PredefinedFieldState] = {}

    @abaqus_method_doc
    def __init__(
        self,
        name: str,
        previous: str,
        description: str = "",
        matrixSolver: Literal[C.DIRECT, C.ITERATIVE] = DIRECT,
        matrixStorage: Literal[C.SYMMETRIC, C.SOLVER_DEFAULT, C.UNSYMMETRIC] = SOLVER_DEFAULT,
        maintainAttributes: Boolean = False,
        solutionTechnique: Literal[C.FULL_NEWTON, C.LCP_CONTACT] = C.FULL_NEWTON,
        gapDistance: float = 0.0,
        scaleFactor: float = 1.0,
    ):
        """This method creates a StaticLinearPerturbationStep object.

        .. note::
            This function can be accessed by::

                mdb.models[name].StaticLinearPerturbationStep

        Parameters
        ----------
        name
            A String specifying the repository key.
        previous
            A String specifying the name of the previous step. The new step appears after this step
            in the list of analysis steps.
        description
            A String specifying a description of the new step. The default value is an empty string.
        matrixSolver
            A SymbolicConstant specifying the type of solver. Possible values are DIRECT and
            ITERATIVE. The default value is DIRECT.
        matrixStorage
            A SymbolicConstant specifying the type of matrix storage. Possible values are SYMMETRIC,
            UNSYMMETRIC, and SOLVER_DEFAULT. The default value is SOLVER_DEFAULT.
        maintainAttributes
            A Boolean specifying whether to retain attributes from an existing step with the same
            name. The default value is False.
        solutionTechnique
            A SymbolicConstant specifying the technique used for solving nonlinear equations. Possible values are
            FULL_NEWTON and LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``solutionTechnique`` was added.
        gapDistance
            A Float specifying the zone within which potential contact constraints are exposed to the LCP solver,
            applicable when solutionTechnique is LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``gapDistance`` was added.
        scaleFactor
            A Float specifying the scaling factor for the initial gap distance specified by gapDistance or the
            internally computed (contact) pair-wise gap distances, applicable when solutionTechnique is LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``scaleFactor`` was added.
        Returns
        -------
        StaticLinearPerturbationStep
            A StaticLinearPerturbationStep object.

        Raises
        ------
        RangeError
        """
        super().__init__()

    @abaqus_method_doc
    def setValues(
        self,
        description: str = "",
        matrixSolver: Literal[C.DIRECT, C.ITERATIVE] = DIRECT,
        matrixStorage: Literal[C.SYMMETRIC, C.SOLVER_DEFAULT, C.UNSYMMETRIC] = SOLVER_DEFAULT,
        solutionTechnique: Literal[C.FULL_NEWTON, C.LCP_CONTACT] = C.FULL_NEWTON,
        gapDistance: float = 0.0,
        scaleFactor: float = 1.0,
    ):
        """This method modifies the StaticLinearPerturbationStep object.

        Parameters
        ----------
        description
            A String specifying a description of the new step. The default value is an empty string.
        matrixSolver
            A SymbolicConstant specifying the type of solver. Possible values are DIRECT and
            ITERATIVE. The default value is DIRECT.
        matrixStorage
            A SymbolicConstant specifying the type of matrix storage. Possible values are SYMMETRIC,
            UNSYMMETRIC, and SOLVER_DEFAULT. The default value is SOLVER_DEFAULT.
        solutionTechnique
            A SymbolicConstant specifying the technique used for solving nonlinear equations. Possible values are
            FULL_NEWTON and LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``solutionTechnique`` was added.
        gapDistance
            A Float specifying the zone within which potential contact constraints are exposed to the LCP solver,
            applicable when solutionTechnique is LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``gapDistance`` was added.
        scaleFactor
            A Float specifying the scaling factor for the initial gap distance specified by gapDistance or the
            internally computed (contact) pair-wise gap distances, applicable when solutionTechnique is LCP_CONTACT.

            .. versionadded:: 2024

                The argument ``scaleFactor`` was added.

        Raises
        ------
        RangeError
        """
        ...
