from galaxy.api.types import GameTime

from dataclasses import dataclass
from time import time
from typing import Dict, Optional
import pickle


@dataclass
class _RunningGameInfo(object):
    game_id: str = None
    start_time: float = None

    def update_start_time(self):
        self.start_time = time()


class TimeTracker:
    """
    Create a new ``TimeTracker`` object to begin tracking time. The parameter upon creation accepts a former cache
    generated by a previous run of this module.

    :param game_time_cache: The dictionary returned by this module from a previous session. This should be the actual
        dictionary itself, rather than a hex representation of it.

    Example:
        .. code-block:: python

            def handshake_complete(self):
                if "game_time_cache" in self.persistent_cache:  # This assumes that the game time cache is stored in the
                                                                # persistent cache.
                    self.game_time_cache = pickle.loads(bytes.fromhex(self.persistent_cache["game_time_cache"]))
                    self.game_time_tracker = TimeTracker(game_time_cache=self.game_time_cache)

    """
    def __init__(self, game_time_cache: Optional[Dict[str, Dict[str, float]]] = None):
        self._running_games_dict: Dict[str, _RunningGameInfo] = {}
        self._game_time_cache = game_time_cache if game_time_cache else {}

    def start_tracking_game(self, game_id: str, start_time: Optional[float] = None) -> None:
        """
        Notify the module to begin keeping track of the user's played time for the specified game. If this is the first
        time that the played time has ever been tracked for this game, then a new entry is created in the game time
        cache and the played time is set to 0. The last played time for the game is always updated to the time when
        either the method was called or, if specified, to the value set by the ``start_time`` parameter.

        :param game_id: The ID of the game to begin tracking play time for.
        :param start_time: A specified time from which to begin counting. This parameter can be used to deal with
            troublesome timeouts.

        Example:
            .. code-block:: python

                async def launch_game(self, game_id):
                    await self._local_client.launch_game(game_id)
                    self.game_time_tracker.start_tracking_game(game_id)

        """
        if not start_time:
            start_time = time()
        if game_id not in self._game_time_cache:
            self._game_time_cache[game_id] = dict()
            self._game_time_cache[game_id]['time_played'] = 0
        self._game_time_cache[game_id]['last_played'] = start_time
        self._running_games_dict[game_id] = (_RunningGameInfo(game_id=game_id, start_time=(time() if not start_time else
                                                                                           start_time)))

    def stop_tracking_game(self, game_id: str) -> None:
        """
        Notify the module to finish keeping track of the user's played time for the specified game. The already tracked
        total play time and last played time will be retained. Both the total play time and the last played time are
        updated and saved immediately before the game time tracking stops.

        :param game_id: The ID of the game to cease tracking play time for.
        :raises GameNotTrackedException: This exception is thrown if the specified game_id is not currently being
            tracked.

        Example:
            .. code-block:: python

                def check_game_status(self, game_id):
                    state = LocalGameState.None_

                    if self._local_client.get_path_to_game(game_id):
                        state |= LocalGameState.Installed

                    if game_id in self.running_games_list:
                        if self._local_client.check_if_process_exists(game_id):
                            state |= LocalGameState.Running
                        else:
                            del self.running_games_list[game_id]
                            self.game_time_tracker.stop_tracking_game(game_id)

                    return LocalGame(game_id, state)

        """
        self._update_tracked_time(game_id)
        del self._running_games_dict[game_id]

    def _update_tracked_time(self, game_id: str) -> None:
        if game_id not in self._running_games_dict:
            raise GameNotTrackedException
        start_time = self._game_time_cache[game_id]['last_played']
        current_time = time()
        self._game_time_cache[game_id]['last_played'] = current_time
        minutes_passed = (current_time - start_time) / 60
        self._game_time_cache[game_id]['time_played'] += minutes_passed

    def get_tracked_time(self, game_id: str) -> GameTime:
        """
        Returns a ``GameTime`` object for the specified game. This object contains the latest total time played and last
        time played values for the game.

        :param game_id: The ID of the game to create a ``GameTime`` object for.
        :raises GameNotTrackedException: This exception is thrown if the specified ``game_id`` has no game time data
            in the game time cache (i.e., it has not been tracked at least once).

        Example:
            .. code-block:: python

                async def get_game_time(self, game_id, context):
                    return self.game_time_tracker.get_tracked_time(game_id)

        """
        if game_id not in self._game_time_cache:
            raise GameNotTrackedException
        # The tracked time should only be updated if the game is running; otherwise, a GameNotTrackedException is
        # erroneously thrown.
        if game_id in self._running_games_dict:
            self._update_tracked_time(game_id)
        time_played = self._game_time_cache[game_id]['time_played']
        current_time = self._game_time_cache[game_id]['last_played']
        return GameTime(game_id, time_played=int(time_played), last_played_time=int(current_time))

    def get_time_cache(self, ignore_tracking: Optional[bool] = False) -> Dict[str, Dict[str, float]]:
        """
        Returns the current game time cache as a dictionary. Each key in this dictionary is a ``game_id`` which has been
        tracked at least once, and each key's value is another dictionary. Here is an example of a returned dictionary
        (make note of the keys contained within the nested dictionary, as these are used to access relevant
        information):

            .. code-block:: python

                self.game_time_cache = {
                    "game_id": {
                        "time_played": 60,  # This is the total minutes played for the game (60 in this case).
                        "last_played": 1574865787  # This is the epoch time (in seconds) that the game was last played.
                    }
                }

        :param ignore_tracking: If set to true, a GamesStillBeingTrackedException will not be thrown when it normally
            would be.
        :raises GamesStillBeingTrackedException: This exception is thrown if there is at least one game which is still
            being tracked.

        Example:
            .. code-block:: python

                def update_game_time_cache(self):
                    try:
                        self.game_time_cache = self.game_time_tracker.get_time_cache()
                    except GamesStillBeingTrackedException:
                        for game_id in self.running_games_list:
                            self.game_time_tracker.stop_tracking_game(game_id)
                        self.game_time_cache = self.game_time_tracker.get_time_cache()

        """
        if len(self._running_games_dict) > 0 and not ignore_tracking:
            raise GamesStillBeingTrackedException
        return self._game_time_cache

    def get_time_cache_hex(self, ignore_tracking: Optional[bool] = False) -> str:
        """
        Returns the current game time cache in hex form as a string.

        :param ignore_tracking: If set to true, a GamesStillBeingTrackedException will not be thrown when it normally
            would be.
        :raises GamesStillBeingTrackedException: This exception is thrown if there is at least one game which is still
            being tracked.

        Example:
            .. code-block:: python

                async def shutdown(self):
                    if self.game_time_cache:
                        file = open("PlayTimeCache.txt", "w+")
                        file.write(str(self.game_time_tracker.get_time_cache_hex()))
                        file.close()
                    await self._http_client.close()
                    await super().shutdown()

        """
        if len(self._running_games_dict) > 0 and not ignore_tracking:
            raise GamesStillBeingTrackedException
        return pickle.dumps(self._game_time_cache).hex()


class GameNotTrackedException(Exception):
    pass


class GamesStillBeingTrackedException(Exception):
    pass
