"""
Data version manager.

Based on the data configuration, data is versionned. Version ids can come from the file or directory names,
or can be generated by the database.

The DataVersion object is the main class that managed the data version. The data version inclued two main feature, 
the data version values and the data version tables.

The values are object representing the values extract from the files or the directories, or generated from the database,
and then added to dataversion tables and data.

The data version columns are part of the tables.
"""
from __future__ import annotations
from typing import Dict,List,Union

import os
import re
import datetime
import pandas

from smb.SMBConnection import SMBConnection

from datablender.base import (
    DataConfiguration,
    Connection,
    Data,
    AsyncConnection,
    AsyncDataConfiguration,
    File
)
from datablender.database import (
    Table,
    Database,
    AsyncDatabase,
    AsyncTable,
    AsyncView
)

class DataVersionValue:
    """Data version value.

    Attributes:
    ----------
        name (str): Value name.
        type (str): Value type.
        detect (str): Value to detect.
        start (int): Start position to extract value.
        end (int): End position to extract value.
        add_to_data (bool): Add the value to data or not.
        format (dict): Params to format the value.
        value (str): Value.

    Methods:
    ----------
        setValue(self,directory_name,file_name) -> None: Set value from directory name and file name.
        configuration(self) -> dict : Get the configuration.

    Examples:
    ----------
        >>> import datablender

    """
    def __init__(
        self,
        name:str,
        type:str=None,
        detect:str=None,
        start:int=None,
        end:int=None,
        add_to_data:Union[bool,dict]=False,
        format:dict=None,
        value:Union[str,int]=None,
        contain:str=None,
        elements:List[Union[str,List[str],List[Dict[str,any]]]] = None,
        file_contains:str = None,
        data_conditions:dict = None,
        directory_name_contains:str = None
    ):
        """Initate the data version value.

        Args:
            name (str): Value name.
            type (str): Value type.
            detect (str, optional): Value to detect. Defaults to None.
            start (int, optional): Start position to extract value. Defaults to None.
            end (int, optional): End position to extract value. Defaults to None.
            add_to_data (bool, optional): Add the value to data or not. Defaults to False.
            format (dict, optional): Params to format the value. Defaults to None.
        """
        self.name = name
        self.type = type
        self.detect = detect
        self.start = start
        self.end = end
        self.add_to_data = add_to_data
        self.format = format
        self.value = value
        self.contain = contain
        self.elements = elements
        self.file_contains = file_contains
        self.data_conditions = data_conditions
        self.directory_name_contains = directory_name_contains

    @property
    def configuration(self) -> dict: 
        """Get the configuration.

        Returns:
            dict: Configuration.
        """
        return {
            'name':self.name,
            'type':self.type,
            'detect':self.detect,
            'start':self.start,
            'end':self.end,
            'add_to_data':self.add_to_data,
            'format':self.format,
            'contain':self.contain,
            'file_contains':self.file_contains,
            'data_conditions':self.data_conditions
        }

    def setValue(
        self,
        file_name:str,
        directory_name:str,
        main_file_name:str,
        data:Data = None,
        **kwargs
    ) -> None:
        """Set the value from directory name or file name.

        Args:
            directory_name (str): Directory name.
            file_name (str): File name.
        """
        # If detect in file
        if self.type=='file':
            value = re.search(self.detect, file_name)
            if value:
                self.value = value.group()[self.start:self.end]

        elif self.type=='main_file':
            value = re.search(self.detect, main_file_name)
            if value:
                self.value = value.group()[self.start:self.end]

        # If detect in directory
        elif self.type=='directory':
            if self.contain:
                self.value = self.contain in directory_name
            else:

                
                self.value = re.search(
                    self.detect,
                    directory_name
                )
                
                if self.value:

                    self.value = self.value.group()
                    
                    if self.end is None:
                        self.end = len(self.value)

                    self.value = self.value[
                        self.start:self.end
                    ]
 
        # If detect in data
        elif self.type=='data':
            if self.file_contains in file_name:
                self.value = data.frame.iloc[0][self.name]

        # Format ID value and save it in config
        if self.format:
            if self.value and hasattr(self,'format'):
                if self.format['type']=='date':
                    self.value = datetime.datetime.strptime(self.value,self.format['format']).date()
                elif self.format['type']=='datetime':
                    self.value = datetime.datetime.strptime(self.value,self.format['format'])
                elif self.format['type']=='int':
                    self.value = int(self.value)
          
class DataVersionColumn:
    """Represents a column of a data version table.

    Attributes:
    ----------
        file_name (str): The file associated with the table.
        data_version_values (dict): Data version values detected from the file name or the the directory name.
        name (str): Column name.
        type (str): Column type.
        extract_mode (str): Indicate how the value is extract.
        is_in_id_columns (str): Indicate if the column is part of the columns to identify the version.
        is_unique_id_column (int): Indicate if the column is the unique id of the column.
        values (str): value of the column.

    Methods:
    ----------
        setValue(self,file_name:str,file_modification_time:datetime.datetime,main_file_name:str,data:Data): Set the value
            of the column based on the file and its content.
        configuration(self) -> dict : Get the configuration.

    Examples:
    ----------
        >>> import datablender
        >>> data_version_column = datablender.DataVersionColumn(
            
        )

    """
    def __init__(
        self,
        file_name:str,
        data_version_values:List[
            DataVersionValue
        ],
        name:str,
        type:str,
        extract_mode:str,
        is_in_id_columns:str,
        is_unique_id_column:int,
        serial:bool=False,
        default:bool=None
    ):
        """Initiate the column by initiating the value.

        Args:
            file_name (str): The file associated with the table.
            data_version_values (dict): Data version values detected from the file name or the the directory name.
            name (str): Column name.
            type (str): Column type.
            extract_mode (str): Indicate how the value is extract.
            is_in_id_columns (str): Indicate if the column is part of the columns to identify the version.
            is_unique_id_column (int): Indicate if the column is the unique id of the column.
            serial (int): Indicate if the column is the unique id of the column.
        """
        
        self.file_name = file_name
        self.data_version_values = data_version_values
        self.name = name
        self.type = type
        self.extract_mode = extract_mode
        self.is_in_id_columns = is_in_id_columns
        self.is_unique_id_column = is_unique_id_column
        self.serial = serial
        self.default = default

        self.value = None
    
    @property
    def configuration(self):
        """Get the configuration.

        Returns:
            dict: Configuration.
        """
        return {
            'name':self.name,
            'type':self.type,
            'is_in_id_columns':self.is_in_id_columns,
            'is_unique_id_column':self.is_unique_id_column,
            'extract_mode':self.extract_mode,
            'serial':self.serial,
            'default':self.default
        }

    @property
    def informations(self):
        """Get the informations.

        Returns:
            dict: informations.
        """
        return {
            'name':self.name,
            'type':self.type,
            'is_in_id_columns':self.is_in_id_columns,
            'is_unique_id_column':self.is_unique_id_column,
            'extract_mode':self.extract_mode,
            'serial':self.serial,
            'value':self.value
        }
    
    def setValue(
        self,
        file_name:str,
        file_modification_time:datetime.datetime,
        main_file_name:str,
        data:Data
    ) -> DataVersionColumn:
        """Set the value of the column based on the file and its content.

        Args:
            file_name (str): File name.
            file_modification_time (datetime.datetime): File modification time.
            main_file_name (str): Main file name.
            data (Data): Data from the file.
        """
        #self.value=None

        # If the value come from the data and file continaining the data is associated with the data version table.
        if (
            self.extract_mode == 'data'
            and file_name == self.file_name
        ):
            self.save_data=False
            if self.name in list(data.columns):
                self.value = data.frame[self.name].iloc[0]

        elif self.extract_mode == 'given':
            if self.name == 'main_file':
                self.value = main_file_name
            if self.name == 'created':
                self.value = file_modification_time

        elif self.extract_mode == 'detected':
            self.value = [
                x.value
                for x in self.data_version_values
                if x.name == self.name
            ][0]
        
        return self

class DataVersionTable(Table):
    """Represent

    Attributes:
    ----------
        connection (Connection): Connection to a database.
        data_version_values (dict): Data version values.
        schema_name (str): Schema name.
        data_configuration (DataConfiguration): Data configuration.
        name (str): Table name.
        columns (list[dict]): Table columns attributes.
        data_version_columns (Dict[str, DataVersionColumn]) : Data version table columns.
        is_deleting (str): Indicate if the data will be deleted.
        file_name (str): File name associated with the table.
        version_exists (bool): Indicate if the version exists in the table.

    Methods:
    ----------
        getDataVersionColumns(self,columns:list) -> None: Create data version columns.
        setColumns(self,file_name:str,file_modification_time:datetime.datetime,main_file_name:str,data:Data) -> None: Set values of the data version columns.
        checkUniqueID(self) -> None: Check if the id of the version exists in the table version.
        updateTableVersion(self) -> None: Update the table version with the values.

    Examples:
    ----------
        >>> import datablender
        >>> data_version_column = datablender.DataVersionTable()

    """
    def __init__(
        self,
        connection:Connection,
        data_configuration:DataConfiguration,
        schema_name:str,
        data_version_values:Dict[str, DataVersionValue],
        name:str,
        columns:List[dict]=[],
        is_deleting:bool=False,
        file_name:str=None,
        constraints:list=[],
        schema_id:int=None
    ):
        """Initiate the data version by creating it and creating the data version columns.

        Args:
            connection (Connection): Connection to a database.
            data_version_values (dict): Data version values.
            schema_name (str): Schema name.
            data_configuration (DataConfiguration): Data configuration.
            name (str): Table name.
            columns (list[dict]): Table columns attributes.
            is_deleting (str): Indicate if the data will be deleted.
            file_name (str): File name associated with the table.
        """
        
        self.is_deleting = is_deleting
        self.file_name = file_name
        self.data_version_values = data_version_values

        self.data_version_columns: Dict[str, DataVersionColumn] = {}
        self.getDataVersionColumns(columns)

        super(DataVersionTable,self).__init__(
            connection,
            name,
            schema_name,
            data_configuration=data_configuration,
            constraints=constraints,
            columns=[
                self.data_version_columns[column_name].configuration
                for column_name in self.data_version_columns
            ],
            schema_id=schema_id
        )

        self.manage()
    
    def getDataVersionColumns(
        self,
        columns:List[dict]
    ) -> None:
        """Create data version columns.

        Args:
            columns (list): Columns configuration.
        """
        
        self.data_version_columns = {
            column.get('name'):DataVersionColumn(
                self.file_name,
                self.data_version_values,
                **column
            ) for column in columns
        }
    
    def setColumns(
        self,
        file_name:str,
        file_modification_time:datetime.datetime,
        main_file_name:str,
        data:Data
    ) -> None:
        """Set values of the data version columns.

        Args:
            file_name (str): File name.
            file_modification_time (datetime.datetime): File modification time.
            main_file_name (str): Main file name.
            data (Data): Data from the file.
        """
        
        for column in self.data_version_columns:
            self.data_version_columns[column].setValue(
                file_name,
                file_modification_time,
                main_file_name,
                data
            )
    
    def checkUniqueID(self) -> None:
        """Check if the id of the version exists in the table version.
        """
        columns = self.data_version_columns
        
        self.select(
            [columns[column_name].informations for column_name in columns if columns[column_name].is_in_id_columns],
            [columns[column_name].informations for column_name in columns if columns[column_name].is_unique_id_column]
        ) 
        self.version_exists = True if self.data.frame.shape[0] else False
            
    def updateTableVersion(
        self,
        file_name:str,
        file_modification_time:datetime.datetime,
        main_files:list,
        data:Data
    ) -> bool:
        """Update the table version with the values.
        """
        self.setColumns(
            file_name,
            file_modification_time,
            main_files,
            data
        )
        self.checkUniqueID()
        
        has_genereted_column = True if [
            self.data_version_columns[name] for name in self.data_version_columns
            if self.data_version_columns[name].extract_mode == 'generated'
        ] else False

        if self.version_exists:
            values = self.data.frame.loc[[0]].to_dict('records')[0]
            
            if has_genereted_column:
                
                [
                    setattr(self.data_version_columns[name], 'value', values[name])
                    for name in self.data_version_columns
                    if self.data_version_columns[name].is_unique_id_column
                ]
                [
                    setattr(self.data_version_values[name], 'value', values[name])
                    for name in self.data_version_values
                    if self.data_version_values[name].type == 'generated'
                ]

            # Si la version existe et il y a une colonne 'processed', mettre 
            # la prossus à non procédé
            if [
                name for name in self.data_version_columns
                if name == 'processed'
            ]:
                self.update({'processed':False},where_statement=values)

            return self.is_deleting

        else:
            if has_genereted_column:
                genereted_values = self.insert(
                    {
                        name:self.data_version_columns[name].value
                        for name in self.data_version_columns
                        if self.data_version_columns[name].is_in_id_columns
                    },
                    [
                        self.data_version_columns[name].name
                        for name in self.data_version_columns
                        if self.data_version_columns[name].extract_mode == 'generated'
                    ]
                )
                
                [
                    setattr(self.data_version_columns[name], 'value', genereted_values[0][name])
                    for name in self.data_version_columns
                    if self.data_version_columns[name].extract_mode == 'generated'
                ]
                [
                    setattr(self.data_version_values[name], 'value', genereted_values[0][name])
                    for name in self.data_version_values
                    if self.data_version_values[name].type == 'generated'
                ]
            else:
                self.insert({
                    self.data_version_columns[name].name:self.data_version_columns[name].value
                    for name in self.data_version_columns
                })

        return False
    
    @property
    def version_configuration(self) -> dict:
    
        return {
            'id':self.id,
            'schema_id':self.schema_id,
            'columns':[self.data_version_columns[column].configuration for column in self.data_version_columns],
            'is_deleting':self.is_deleting,
            'file_name':self.file_name,
            'constraints':self.constraints
        }

    def getNonProccesedVersions(self) -> List[dict]:
        self.select()
        return self.data.frame.loc[
            ~self.data.frame['processed'],
            [
                column for column in self.data_version_columns
                if self.data_version_columns[column].is_in_id_columns
            ]
        ].to_dict('records')

class DataVersion:
    """Data version manager.

    When files are stored in a directory, files name and directories name can contain values to
    versioned data. This class, based on data configuration, can detect those values and add it to
    data. 

    Also, when data is selected from database to be used in a process, the data version can managed 
    the versions.

    A version is defined by a serie of variables

    Attributes:
    ----------
        connection (Connection): Connection to a database.
        schema_name (str): Schema name.
        data_configuration (DataConfiguration): Data configuration.
        values_config (list): Data version values.
        tables_config (list): Data version tables.
        active (bool): Indicate if data version is active or not.

    Methods:
    ----------
        getTables(self,tables_config) -> None: Create tables containing data version.
        getValues(self,values_config) -> None: Initiate data version values.
        setValues(self,directory_name,file_name) -> None: Set data version values.
        checkDataSaving(self,file_name) -> bool: Check if data from a file is going to be saved.
        updateData(self,data) -> Data: Add columns to data with data version values.

    Examples:
    ----------
        >>> import datablender
        >>> data_version = datablender.DataVersion()

    """
    def __init__(
        self,
        connection:Connection,
        schema_name:str,
        data_configuration:DataConfiguration,
        values:list=[],
        tables:list=[],
        active:bool = True,
        schema_id:int =None,
        database:Database =None
    ):
        """Initiate the data version by creating the data version tables and the values.

        Args:
            connection (Connection): Connection to a database.
            schema_name (str): Schema name.
            data_configuration (DataConfiguration): Data configuration.
            values_config (list, optional): Data version values. Defaults to None.
            tables_config (list, optional): Data version tables. Defaults to None.
            active (bool, optional): Indicate if data version is active or not. Defaults to True.
        """
        self.connection = connection
        self.database = database
        self.schema_name = schema_name
        self.schema_id = schema_id
        self.data_configuration = data_configuration
        self.active = active

        self.values: Dict[str, DataVersionValue] = {}
        self.tables: List[DataVersionTable] = []

        self.versions = None

        if self.active:
            self.getValues(values)
            self.getTables(tables)

    @property
    def configuration(self):
        
        return {
            'values':[
                self.values[value].configuration
                for value in self.values
            ],
            'tables':[
                table.version_configuration
                for table in self.tables
            ]
        }

    def getTables(
        self,
        tables_config:List[dict]
    ) -> None:
        """Create data version tables from configuration.

        Args:
            tables_config (list): Tables configuration.
        """
        
        self.tables = [
            DataVersionTable(
                self.connection,
                self.data_configuration,
                configuration.pop(
                    'schema_name',
                    self.schema_name
                ),
                self.values,
                schema_id=configuration.pop(
                    'schema_id',
                    self.schema_id
                ),
                **configuration
            )
            for configuration in tables_config
        ]
    
    def getValues(
        self,
        values_config:List[dict]
    ) -> None:
        """Create data version tables from configuration.

        Args:
            values_config (list): Columns configuration.
        """
        
        self.values = {
            value.get('name'):DataVersionValue(**value)
            for value in values_config
        }
    
    def setValues(
        self,
        directory_name:str,
        name:str,
        **kwargs
    ) -> None:
        """Get columns values from directory name and file name.

        Args:
            directory_name (str): Directory name.
            file_name (str): File name.
        """
        
        if self.active:
            {
                self.values[value_name].setValue(
                    directory_name,
                    name
                ) 
                for value_name in self.values
            }
  
    def checkDataSaving(
        self,
        file_name:str
    ) -> bool:
        """Check if data from a file is going to be saved. If the file is
        linked to a data version table, it won't be saved.

        Args:
            file_name (str): File name containing the data.

        Returns:
            bool: Is data will be saved.
        """
        
        if self.active:
            return False if [
                table for table in self.tables
                if table.file_name == file_name
            ] else True
        
        return True

    def updateData(
        self,
        data:Data,
        file_name:str=None
    ) -> None:
        """Add columns to data with data version values.

        Args:
            data (Data): Data to update.

        Returns:
            Data: Data with new data version columns.
        """
        
        if self.active:
            
            for value_name in self.values:
                value = self.values[value_name]
            
                if value.add_to_data:
            
                    if (
                        isinstance(value.add_to_data,dict)
                        and value.add_to_data.get(file_name)
                    ):
                        data.frame[value.name] =value.value
            
                    elif isinstance(value.add_to_data,bool):
                        data.frame[value.name] =value.value
        
    def getTablesAssociatedWithFiles(self) -> List[Dict]:
        if self.active:
            return [
                {
                    'name':table.name,
                    'file_name':table.file_name,
                    'priority':1,
                    'order':index
                }
                for index,table in enumerate(self.tables)
                if table.file_name
            ]
        return []

    def getNonProccesedVersions(self) -> None:
        return [
            table.getNonProccesedVersions()
            for table in self.tables
            if table.is_deleting
        ][0]

    def checkProccessedVersion(
        self,
        version:dict
    ) -> None:
        [
            table for table in self.tables if table.is_deleting
        ][0].update({'processed':True},where_statement=version)

class AsyncDataVersionTable(AsyncTable):
    """Represent

    Attributes:
    ----------
        connection (Connection): Connection to a database.
        data_version_values (dict): Data version values.
        schema_name (str): Schema name.
        data_configuration (DataConfiguration): Data configuration.
        name (str): Table name.
        columns (list[dict]): Table columns attributes.
        data_version_columns (Dict[str, DataVersionColumn]) : Data version table columns.
        is_deleting (str): Indicate if the data will be deleted.
        file_name (str): File name associated with the table.
        version_exists (bool): Indicate if the version exists in the table.

    Methods:
    ----------
        getDataVersionColumns(self,columns:list) -> None: Create data version columns.
        setColumns(self,file_name:str,file_modification_time:datetime.datetime,main_file_name:str,data:Data) -> None: Set values of the data version columns.
        checkUniqueID(self) -> None: Check if the id of the version exists in the table version.
        updateTableVersion(self) -> None: Update the table version with the values.

    Examples:
    ----------
        >>> import datablender
        >>> data_version_column = datablender.DataVersionTable()

    """
    def __init__(
        self,
        connection:AsyncConnection,
        data_configuration:AsyncDataConfiguration,
        schema_name:str,
        data_version_values:List[DataVersionValue],
        name:str,
        columns:List[dict]=[],
        is_deleting:bool=False,
        file_name:str=None,
        constraints:list=[],
        schema_id:int=None
    ):
        """Initiate the data version by creating it and creating the data version columns.

        Args:
            connection (Connection): Connection to a database.
            data_version_values (dict): Data version values.
            schema_name (str): Schema name.
            data_configuration (DataConfiguration): Data configuration.
            name (str): Table name.
            columns (list[dict]): Table columns attributes.
            is_deleting (str): Indicate if the data will be deleted.
            file_name (str): File name associated with the table.
        """
        
        self.is_deleting = is_deleting
        self.file_name = file_name
        self.data_version_values = data_version_values

        self.data_version_columns: Dict[str, DataVersionColumn] = {}
        self.getDataVersionColumns(columns)

        super(AsyncDataVersionTable,self).__init__(
            connection,
            name,
            schema_name,
            data_configuration=data_configuration,
            constraints=constraints,
            columns=[
                self.data_version_columns[column_name].configuration
                for column_name in self.data_version_columns
            ],
            schema_id=schema_id
        )
    
    def getDataVersionColumns(
        self,
        columns:List[dict]
    ) -> None:
        """Create data version columns.

        Args:
            columns (list): Columns configuration.
        """
        
        self.data_version_columns = {
            column.get('name'):DataVersionColumn(
                self.file_name,
                self.data_version_values,
                **column
            ) for column in columns
        }
    
    def setColumns(
        self,
        file_name:str,
        file_modification_time:datetime.datetime,
        main_file_name:str,
        data:Data
    ) -> None:
        """Set values of the data version columns.

        Args:
            file_name (str): File name.
            file_modification_time (datetime.datetime): File modification time.
            main_file_name (str): Main file name.
            data (Data): Data from the file.
        """
        
        for column in self.data_version_columns:
            self.data_version_columns[column].setValue(
                file_name,
                file_modification_time,
                main_file_name,
                data
            )
    
    async def checkUniqueID(self) -> None:
        """Check if the id of the version exists in the table version.
        """
        columns = self.data_version_columns
        
        await self.select(
            [columns[column_name].informations for column_name in columns if columns[column_name].is_in_id_columns],
            [columns[column_name].informations for column_name in columns if columns[column_name].is_unique_id_column]
        ) 
        self.version_exists = True if self.data is not None and self.data.frame.shape[0] else False
            
    async def updateTableVersion(
        self,
        file_name:str,
        file_modification_time:datetime.datetime,
        main_files:list,
        data:Data
    ) -> bool:
        """Update the table version with the values.
        """
        self.setColumns(
            file_name,
            file_modification_time,
            main_files,
            data
        )
        await self.checkUniqueID()
        
        has_genereted_column = True if [
            self.data_version_columns[name] for name in self.data_version_columns
            if self.data_version_columns[name].extract_mode == 'generated'
        ] else False

        if self.version_exists:
            values = self.data.frame.loc[[0]].to_dict('records')[0]
            
            if has_genereted_column:
                
                [
                    setattr(self.data_version_columns[name], 'value', values[name])
                    for name in self.data_version_columns
                    if self.data_version_columns[name].is_unique_id_column
                ]
                [
                    setattr(value, 'value', values[value.name])
                    for value in self.data_version_values
                    if value.type == 'generated'
                ]

            # Si la version existe et il y a une colonne 'processed', mettre 
            # la prossus à non procédé
            if [
                name for name in self.data_version_columns
                if name == 'processed'
            ]:
                await self.update({'processed':False},where_statement=values)

            return self.is_deleting

        else:
            if has_genereted_column:
                genereted_values = await self.insert(
                    {
                        name:self.data_version_columns[name].value
                        for name in self.data_version_columns
                        if self.data_version_columns[name].is_in_id_columns
                    },
                    [
                        self.data_version_columns[name].name
                        for name in self.data_version_columns
                        if self.data_version_columns[name].extract_mode == 'generated'
                    ]
                )
                
                [
                    setattr(self.data_version_columns[name], 'value', genereted_values[0][name])
                    for name in self.data_version_columns
                    if self.data_version_columns[name].extract_mode == 'generated'
                ]
                [
                    setattr(value, 'value', genereted_values[0][value.name])
                    for value in self.data_version_values
                    if value.type == 'generated'
                ]
            else:
                await self.insert({
                    self.data_version_columns[name].name:self.data_version_columns[name].value
                    for name in self.data_version_columns
                })

        return False
    
    @property
    def version_configuration(self) -> dict:
    
        return {
            'id':self.id,
            'schema_id':self.schema_id,
            'columns':[self.data_version_columns[column].configuration for column in self.data_version_columns],
            'is_deleting':self.is_deleting,
            'file_name':self.file_name,
            'constraints':self.constraints
        }

    async def getNonProccesedVersions(self) -> List[dict]:
        await self.select()
        return self.data.frame.loc[
            ~self.data.frame['processed'],
            [
                column for column in self.data_version_columns
                if self.data_version_columns[column].is_in_id_columns
            ]
        ].to_dict('records')

class AsyncDataVersionView(AsyncView):
    """Represent

    Attributes:
    ----------
        connection (Connection): Connection to a database.
        data_version_values (dict): Data version values.
        schema_name (str): Schema name.
        data_configuration (DataConfiguration): Data configuration.
        name (str): Table name.
        columns (list[dict]): Table columns attributes.
        data_version_columns (Dict[str, DataVersionColumn]) : Data version table columns.
        is_deleting (str): Indicate if the data will be deleted.
        file_name (str): File name associated with the table.
        version_exists (bool): Indicate if the version exists in the table.

    Methods:
    ----------
        getDataVersionColumns(self,columns:list) -> None: Create data version columns.
        setColumns(self,file_name:str,file_modification_time:datetime.datetime,main_file_name:str,data:Data) -> None: Set values of the data version columns.
        checkUniqueID(self) -> None: Check if the id of the version exists in the table version.
        updateTableVersion(self) -> None: Update the table version with the values.

    Examples:
    ----------
        >>> import datablender
        >>> data_version_column = datablender.DataVersionTable()

    """
    def __init__(
        self,
        connection:AsyncConnection,
        data_configuration:AsyncDataConfiguration,
        schema_name:str,
        data_version_values:List[DataVersionValue],
        name:str,
        schema_id:int=None,
        query:Union[str,dict] = None,
        is_materialized:bool = False,
        is_database_saved:bool  = False,
        directory_name:str = None,
        dependences:List[dict] = [],
        file_server:SMBConnection = None
    ):
        """Initiate the data version by creating it and creating the data version columns.

        Args:
            connection (Connection): Connection to a database.
            data_version_values (dict): Data version values.
            schema_name (str): Schema name.
            data_configuration (DataConfiguration): Data configuration.
            name (str): Table name.
            columns (list[dict]): Table columns attributes.
            is_deleting (str): Indicate if the data will be deleted.
            file_name (str): File name associated with the table.
        """
        
        super(AsyncDataVersionView,self).__init__(
            connection,
            name,
            query,
            schema_name=schema_name,
            data_configuration=data_configuration,
            # columns=[
            #     self.data_version_columns[column_name].configuration
            #     for column_name in self.data_version_columns
            # ],
            schema_id=schema_id,
            is_database_saved=is_database_saved,
            is_materialized=is_materialized
        )
        self.data_version_values = data_version_values
        self.dependences = dependences

        self.file = File(
            directory_name,
            '{}.csv'.format(self.name),
            file_server=file_server
        )
            
    async def saveVersions(
        self,
        database:AsyncDatabase
    ) -> None:

        await self.manage(
            refresh = True,
            dependance_exists=self.checkDependance(database)
        )
        if self.db_element:
            await self.select()

        if self.file.exists:
            
            if hasattr(self,'data'):
                data = Data(self.file.read().content)

                if self.data_version_values:

                    for value in self.data_version_values:
                        if hasattr(value,'format') and value.format and value.format.get('type') == 'date':
                            data.toDateTime(
                                value.name,
                                "%Y-%m-%d %H:%M:%S",
                                value.format.get('type')
                            )

                            self.data.toDateTime(
                                value.name,
                                "%Y-%m-%d %H:%M:%S",
                                value.format.get('type')
                            )

                    columns = [
                        c.get('name') for c in self.data.columns
                        if c.get('name') not in [v.name for v in self.data_version_values]
                    ]

                    self.data.rename({c:'{}_new'.format(c) for c in columns})

                    data.extra_data[self.name] = self.data.frame.copy()

                    data.mergeExtraData(
                        self.name,
                        how = 'outer',
                        on = list(set([v.name for v in self.data_version_values]))
                    )
                    data.frame[columns] = data.frame.apply(
                        lambda x:[x['{}_new'.format(c)] if pandas.isna(x[c]) else x[c] for c in columns],
                        axis=1,
                        result_type='expand'
                    )

                    data.frame.drop(
                        ['{}_new'.format(c) for c in columns],
                        axis=1,
                        inplace=True
                    )
                    
                else:
                    data = self.data
                
                await data.write(file=self.file)

        elif hasattr(self,'data'):
            await self.data.write(file=self.file)

    def checkDataSaving(
        self,
        data_version_value:dict = None,
        file_name:str = None
    ) -> None:
        
        if self.file.exists:
            data:pandas.DataFrame = self.file.read().content


            if file_name:
                data = data[data['main_file_name']==file_name]
            else:

                for value in self.data_version_values:
                    
                    if value.value:
                        if value.format:
                            if value.format['type']=='date':
                                value_= value.value.strftime('%Y-%m-%d')
                        else:
                            value_ = value.value

                        data = data[data[value.name]==value_]


            if data.shape[0] and data_version_value:
                if data_version_value.get('type') == 'date':
                    return datetime.datetime.strptime(
                        data.to_dict('records')[0].get('start_date'),
                        '%Y-%m-%d'
                    ) <= datetime.datetime.strptime(
                        data_version_value.get('value'),
                        '%Y-%m-%d'
                    ) <=datetime.datetime.strptime(
                        data.to_dict('records')[0].get('end_date'),
                        '%Y-%m-%d'
                    )
                
        return True
    
    def checkDependance(
        self,
        database:AsyncDatabase
    ) -> None:
        
        if isinstance(self.query,dict):

            schema = next((s for s in database.schema if s.name == self.query.get('from')['schema_name']))
            element_from = next(
                (
                    s for s in getattr(schema,self.query.get('from')['element_type'])
                    if s.name == self.query.get('from')['name'] and s.db_element
                ),
                None
            )
            return element_from is not None
    
        return True

class DataVersionFile(File):

    def __init__(
        self,
        directory_name: str = None,
        file_name: str = None,
        path: str = None,
        content = None,
        values:List[dict] = []
    ):
        super(DataVersionFile,self).__init__(
            directory_name,
            file_name,
            path,
            content
        )
        self.values = values
        self.data = Data(self.read().content if self.exists else None) 

    def checkDataSaving(
        self,
        data_version_value:dict = None,
        file_name:str = None
    ) -> None:
        if self.exists:

            return file_name in self.data.frame.loc[
                self.data.frame['date']==data_version_value.get('value'),
                'file_name'
            ].to_list()
        
class AsyncDataVersion:
    """Data version manager.

    When files are stored in a directory, files name and directories name can contain values to
    versioned data. This class, based on data configuration, can detect those values and add it to
    data. 

    Also, when data is selected from database to be used in a process, the data version can managed 
    the versions.

    A version is defined by a serie of variables

    Attributes:
    ----------
        connection (Connection): Connection to a database.
        schema_name (str): Schema name.
        data_configuration (DataConfiguration): Data configuration.
        values_config (list): Data version values.
        tables_config (list): Data version tables.
        active (bool): Indicate if data version is active or not.

    Methods:
    ----------
        getTables(self,tables_config) -> None: Create tables containing data version.
        getValues(self,values_config) -> None: Initiate data version values.
        setValues(self,directory_name,file_name) -> None: Set data version values.
        checkDataSaving(self,file_name) -> bool: Check if data from a file is going to be saved.
        updateData(self,data) -> Data: Add columns to data with data version values.

    Examples:
    ----------
        >>> import datablender
        >>> data_version = datablender.DataVersion()

    """
    def __init__(
        self,
        connection:AsyncConnection,
        schema_name:str,
        data_configuration:AsyncDataConfiguration,
        values:List[dict]=[],
        tables:List[dict]=[],
        views:List[dict]=[],
        files:List[dict]=[],
        active:bool = True,
        schema_id:int =None,
        database:AsyncDatabase =None,
        directory_name:str = None
    ):
        """Initiate the data version by creating the data version tables and the values.

        Args:
            connection (Connection): Connection to a database.
            schema_name (str): Schema name.
            data_configuration (DataConfiguration): Data configuration.
            values_config (list, optional): Data version values. Defaults to None.
            tables_config (list, optional): Data version tables. Defaults to None.
            active (bool, optional): Indicate if data version is active or not. Defaults to True.
        """
        self.connection = connection
        self.database = database
        self.schema_name = schema_name
        self.schema_id = schema_id
        self.data_configuration = data_configuration
        self.active = active


        self.files = [
            DataVersionFile(
                directory_name,
                **f
            ) for f in files
        ]

        self.views_config = views

        self.values: List[DataVersionValue] = {}
        self.tables: List[AsyncDataVersionTable] = []
        self.views: List[AsyncDataVersionView] = []

        self.versions = None

        if self.active:
            self.getValues(values)
            self.getTables(tables)

    @property
    def configuration(self):
        
        return {
            'values':[
                value.configuration
                for value in self.values
            ],
            'tables':[
                table.version_configuration
                for table in self.tables
            ]
        }

    def getTables(
        self,
        tables_config:List[dict]
    ) -> None:
        """Create data version tables from configuration.

        Args:
            tables_config (list): Tables configuration.
        """
        
        self.tables = [
            AsyncDataVersionTable(
                self.connection,
                self.data_configuration,
                configuration.pop(
                    'schema_name',
                    self.schema_name
                ),
                self.values,
                schema_id=configuration.pop(
                    'schema_id',
                    self.schema_id
                ),
                **configuration
            )
            for configuration in tables_config
        ]
    
    async def getViews(
        self,
        directory_name:str=None,
        file_server:SMBConnection = None
    ) -> None:
        """Create data version views from configuration.
        """
        
        for configuration in self.views_config:
            
            view = AsyncDataVersionView(
                self.connection,
                self.data_configuration,
                configuration.pop(
                    'schema_name',
                    self.schema_name
                ),
                self.values,
                schema_id=configuration.pop(
                    'schema_id',
                    self.schema_id
                ),
                directory_name=directory_name,
                file_server=file_server,
                **configuration
            )

            await view.initiate()
            await view.manage(dependance_exists=view.checkDependance(self.database))

            self.views.append(view)

    def getValues(
        self,
        values_config:List[dict]
    ) -> None:
        """Create data version tables from configuration.

        Args:
            values_config (list): Columns configuration.
        """
        
        self.values = [
            DataVersionValue(**value)
            for value in values_config
        ]
     
    def checkDataSaving(
        self,
        file_name:str,
        directory_name:str,
        data_version_value:dict = None,
        update_version = False,
        **kwargs
    ) -> bool:
        """Check if data from a file is going to be saved. If the file is
        linked to a data version table, it won't be saved.

        Args:
            file_name (str): File name containing the data.

        Returns:
            bool: Is data will be saved.
        """
        
        if self.active:
            if (
                self.views and
                data_version_value and
                not update_version
            ):
                for view in self.views:
                    return view.checkDataSaving(
                        data_version_value,
                        file_name
                    )
                
            if (
                self.files and
                data_version_value and
                not update_version
            ):
                for file in self.files:
                    return file.checkDataSaving(
                        data_version_value,
                        file_name
                    )
                

            if self.views:
                for view in self.views:
                    return file_name != view.file.name
                
            if self.files:
                for file in self.files:
                    return file_name != file.name

            if self.tables:
                return False if [
                    table for table in self.tables
                    if table.file_name == file_name
                ] else True

        return True
        
    def getTablesAssociatedWithFiles(self) -> List[Dict]:
        if self.active:
            return [
                {
                    'name':table.name,
                    'file_name':table.file_name,
                    'priority':1,
                    'order':index
                }
                for index,table in enumerate(self.tables)
                if table.file_name
            ]
        return []

    def getNonProccesedVersions(self) -> None:
        return [
            table.getNonProccesedVersions()
            for table in self.tables
            if table.is_deleting
        ][0]

    async def checkProccessedVersion(
        self,
        version:dict
    ) -> None:
        await [
            table for table in self.tables if table.is_deleting
        ][0].update({'processed':True},where_statement=version)

