## Info

[Reply](https://forums.manning.com/posts/reply/0/44893.page) to ["Improvement to Reconstruction Error"](https://forums.manning.com/posts/list/44893.page) question by [user 57816](https://forums.manning.com/user/profile/28088.page)

## Reply

Wow! That's a lot of useful insight and some tricky questions! ;)

Regarding "industrial strength" and memory-efficient SVD, we'll definitely attempt to address that in a second edition of the book. In the mean time you may have to do some more research yourself. The `gensim` source code can be a great source of ideas and patterns for "out of core" processing, incremental optimizers/solvers, and sparse matrix multiplication. However they implemented their algorithms is probably a good way to go, because it minimizes the memory footprint on my machine whenever I'm careful about not instantiating my entire corpus or bags of words in memory, but rather generating them as-needed.

Regarding your three "researching" questions, please post your discoveries here whenever you have an update on your research. Here are my thoughts:

1. `numpy` can handle sparse matrix multiplication just fine. Scipy just routes it's linear algebra operators there:

[code]
>>> import numpy as np
>>> import scipy
>>> id(scipy.dot) == id(np.dot)
True
>>> A = scipy.sparse.csr_matrix([[1, 2, 0], [0, 0, 3], [4, 0, 5]])
>>> v = scipy.sparse.csr_matrix([[1], [0], [-1]])
>>> A.dot(v)
<3x1 sparse matrix of type '<class 'numpy.int64'>'
    with 3 stored elements in Compressed Sparse Row format>
>>> scipy.dot(A, v)
<3x1 sparse matrix of type '<class 'numpy.int64'>'
    with 3 stored elements in Compressed Sparse Row format>
>>> np.dot(A, v)
<3x1 sparse matrix of type '<class 'numpy.int64'>'
    with 3 stored elements in Compressed Sparse Row format>
[/code]

2. TruncatedSVD and it's explained_variance method are implemented in `sklearn.decomposition.TruncatedSVD`. Scipy's equivalent is in `scipy.sparse.linalg.svds`. Gensim's is in `gensim.models.lsimodel.stochastic_svd`. You can calculate explained variance yourself from the gensim SVD results (or any SVD) with:

[code]
>>> import numpy as np
>>> import sklearn.decomposition
>>> svd = sklearn.decomposition.TruncatedSVD(2)
>>> A = np.random.randn(1000,100)
>>> svd.fit(A)
TruncatedSVD(algorithm='randomized', n_components=2, n_iter=5,
       random_state=None, tol=0.0)
>>> A_2D = svd.transform(A)
>>> np.var(A_2D, axis=0)
array([1.7039018 , 1.65362273])
>>> var = A.shape[1] * np.var(A_2D, axis=0) / np.var(A, axis=0).sum()
>>> var
array([1.6955373 , 1.64550506])
>>> np.abs(var - svd.explained_variance_).round(2)
array([0., 0.])
[/code]

3. I think you just need to correct the math you're using to compute the total variance. You should be squaring, then summing then dividing by `shape[1] - 1` not `shape[1]`. But `np.var` will do all that efficiently on a sparse matrix for you. However, it won't do it incrementally (also called "out-of-core processing" ). So if you have a severe RAM restriction you should research techniques for incremental computation of things like sum() and var(), or just copy the implementations in the gensim source code.

In general I'd recommend using a computational graph framework like Spark, TensorFlow, Keras, Hadoop, whenever you can't get things done with server that has large RAM. This is the "industrial strength" out-of-core processing that you are searching for.



