# mcp_cli/commands/provider.py - Final optimized version
"""
Provider command with all fixes applied and optimized.
This version incorporates the diagnostic fixes with your existing architecture.
"""
from __future__ import annotations
import subprocess
from typing import Dict, List, Tuple, Any
from rich.table import Table

from mcp_cli.model_manager import ModelManager
from mcp_cli.utils.rich_helpers import get_console

console = get_console()


def _check_ollama_running() -> tuple[bool, int]:
    """
    Check if Ollama is running and return status with model count.
    Returns (is_running, model_count)
    """
    try:
        result = subprocess.run(['ollama', 'list'], 
                              capture_output=True, 
                              text=True, 
                              timeout=5)
        if result.returncode == 0:
            # Count actual models (skip header line and empty lines)
            lines = result.stdout.strip().split('\n')
            model_lines = [line for line in lines[1:] if line.strip()]
            return True, len(model_lines)
        return False, 0
    except (FileNotFoundError, subprocess.TimeoutExpired, Exception):
        return False, 0


def _get_provider_status_enhanced(provider_name: str, info: Dict[str, Any]) -> tuple[str, str, str]:
    """
    Enhanced status logic that handles all provider types correctly.
    Returns (status_icon, status_text, status_reason)
    """
    # Handle Ollama specially - it doesn't need API keys
    if provider_name.lower() == "ollama":
        is_running, model_count = _check_ollama_running()
        if is_running:
            return "✅", "Ready", f"Running ({model_count} models)"
        else:
            return "❌", "Not Running", "Ollama service not accessible"
    
    # For API-based providers, check configuration
    has_api_key = info.get("has_api_key", False)
    
    if not has_api_key:
        return "❌", "Not Configured", "No API key"
    
    # If has API key, check model availability
    models = info.get("models", info.get("available_models", []))
    model_count = len(models) if isinstance(models, list) else 0
    
    if model_count == 0:
        return "⚠️", "Partial Setup", "API key set but no models found"
    
    return "✅", "Ready", f"Configured ({model_count} models)"


def _get_model_count_display_enhanced(provider_name: str, info: Dict[str, Any]) -> str:
    """
    Enhanced model count display that handles Ollama and chuk-llm 0.7+ correctly.
    """
    # For Ollama, get live count from ollama command
    if provider_name.lower() == "ollama":
        is_running, live_count = _check_ollama_running()
        if is_running:
            return f"{live_count} models"
        else:
            return "Ollama not running"
    
    # For other providers, use chuk-llm data with proper key handling
    # chuk-llm 0.7+ uses "models" key, but we'll check both for compatibility
    models = info.get("models", info.get("available_models", []))
    
    if not isinstance(models, list):
        return "Unknown"
    
    count = len(models)
    if count == 0:
        return "No models found"
    elif count == 1:
        return "1 model"
    else:
        return f"{count} models"


def _get_features_display_enhanced(info: Dict[str, Any]) -> str:
    """Enhanced feature display with more comprehensive icons."""
    baseline_features = info.get("baseline_features", [])
    
    feature_icons = []
    if "streaming" in baseline_features:
        feature_icons.append("📡")
    if "tools" in baseline_features or "parallel_calls" in baseline_features:
        feature_icons.append("🔧")
    if "vision" in baseline_features:
        feature_icons.append("👁️")
    if "reasoning" in baseline_features:
        feature_icons.append("🧠")
    if "json_mode" in baseline_features:
        feature_icons.append("📝")
    
    return "".join(feature_icons) if feature_icons else "📄"


def _render_list_optimized(model_manager: ModelManager) -> None:
    """
    Optimized provider list that handles all the edge cases correctly.
    """
    tbl = Table(title="Available Providers")
    tbl.add_column("Provider", style="green", width=12)
    tbl.add_column("Status", style="cyan", width=15)
    tbl.add_column("Default Model", style="yellow", width=25)
    tbl.add_column("Models Available", style="blue", width=18)
    tbl.add_column("Features", style="magenta", width=10)

    current_provider = model_manager.get_active_provider()
    
    try:
        # Get provider info using the working method
        all_providers_info = model_manager.list_available_providers()
        
        if not all_providers_info:
            console.print("[red]No providers found. Check chuk-llm installation.[/red]")
            return
            
    except Exception as e:
        console.print(f"[red]Error getting provider list:[/red] {e}")
        return

    # Sort providers to put current one first, then alphabetically
    provider_items = list(all_providers_info.items())
    provider_items.sort(key=lambda x: (x[0] != current_provider, x[0]))

    for provider_name, provider_info in provider_items:
        # Handle error cases
        if "error" in provider_info:
            tbl.add_row(
                provider_name, 
                "[red]Error[/red]", 
                "-", 
                "-", 
                provider_info["error"][:20] + "..."
            )
            continue
        
        # Mark current provider
        display_name = f"[bold]{provider_name}[/bold]" if provider_name == current_provider else provider_name
        
        # Enhanced status using improved logic
        status_icon, status_text, status_reason = _get_provider_status_enhanced(provider_name, provider_info)
        
        # Color-code the status text
        if status_icon == "✅":
            status_display = f"[green]{status_icon} {status_text}[/green]"
        elif status_icon == "⚠️":
            status_display = f"[yellow]{status_icon} {status_text}[/yellow]"
        else:
            status_display = f"[red]{status_icon} {status_text}[/red]"
        
        # Default model with proper fallback
        default_model = provider_info.get("default_model", "-")
        if not default_model or default_model in ("None", "null"):
            default_model = "-"
        
        # Enhanced model count display
        models_display = _get_model_count_display_enhanced(provider_name, provider_info)
        
        # Enhanced features
        features_display = _get_features_display_enhanced(provider_info)
        
        tbl.add_row(
            display_name,
            status_display,
            default_model,
            models_display,
            features_display
        )

    console.print(tbl)
    console.print("\n[dim]💡 Use 'mcp-cli provider <name>' to switch providers[/dim]")
    
    # Show helpful tips based on current state
    inactive_providers = []
    for name, info in all_providers_info.items():
        if "error" not in info:
            status_icon, _, _ = _get_provider_status_enhanced(name, info)
            if status_icon == "❌":
                inactive_providers.append(name)
    
    if inactive_providers:
        console.print(f"[dim]🔧 Configure providers with: mcp-cli provider set <name> api_key <key>[/dim]")


def _render_diagnostic_optimized(model_manager: ModelManager, target: str | None) -> None:
    """Optimized diagnostic that shows detailed status for providers."""
    if target:
        providers_to_test = [target] if model_manager.validate_provider(target) else []
        if not providers_to_test:
            console.print(f"[red]Unknown provider:[/red] {target}")
            available = ", ".join(model_manager.list_providers())
            console.print(f"[yellow]Available providers:[/yellow] {available}")
            return
    else:
        providers_to_test = model_manager.list_providers()

    tbl = Table(title="Provider Diagnostics")
    tbl.add_column("Provider", style="green")
    tbl.add_column("Status", style="cyan")
    tbl.add_column("Models", style="blue")
    tbl.add_column("Features", style="yellow")
    tbl.add_column("Details", style="magenta")

    try:
        all_providers_data = model_manager.list_available_providers()
    except Exception as e:
        console.print(f"[red]Error getting provider data:[/red] {e}")
        return

    for provider in providers_to_test:
        try:
            provider_info = all_providers_data.get(provider, {})
            
            # Skip if provider has errors
            if "error" in provider_info:
                tbl.add_row(
                    provider, 
                    f"[red]Error[/red]", 
                    "-",
                    "-",
                    provider_info["error"][:30] + "..."
                )
                continue
            
            # Enhanced status
            status_icon, status_text, status_reason = _get_provider_status_enhanced(provider, provider_info)
            
            if status_icon == "✅":
                status_display = f"[green]{status_icon} {status_text}[/green]"
            elif status_icon == "⚠️":
                status_display = f"[yellow]{status_icon} {status_text}[/yellow]"
            else:
                status_display = f"[red]{status_icon} {status_text}[/red]"
            
            # Model count
            models_display = _get_model_count_display_enhanced(provider, provider_info)
            
            # Features
            features_display = _get_features_display_enhanced(provider_info)
            
            # Additional details
            details = []
            if provider_info.get("api_base"):
                details.append(f"API: {provider_info['api_base']}")
            if provider_info.get("discovery_enabled"):
                details.append("Discovery: ✅")
            details_str = " | ".join(details) if details else "-"
            
            tbl.add_row(
                provider, 
                status_display, 
                models_display,
                features_display, 
                details_str
            )
            
        except Exception as exc:
            tbl.add_row(
                provider, 
                f"[red]Error[/red]", 
                "-",
                "-",
                str(exc)[:30] + "..."
            )

    console.print(tbl)


def _switch_provider_enhanced(
    model_manager: ModelManager,
    provider_name: str,
    model_name: str | None,
    context: Dict,
) -> None:
    """Enhanced provider switching with better validation and feedback."""
    
    if not model_manager.validate_provider(provider_name):
        available = ", ".join(model_manager.list_providers())
        console.print(f"[red]Unknown provider:[/red] {provider_name}")
        console.print(f"[yellow]Available providers:[/yellow] {available}")
        return

    # Get provider info for validation
    try:
        all_providers_info = model_manager.list_available_providers()
        provider_info = all_providers_info.get(provider_name, {})
        
        if "error" in provider_info:
            console.print(f"[red]Provider error:[/red] {provider_info['error']}")
            return
        
        # Enhanced status validation
        status_icon, status_text, status_reason = _get_provider_status_enhanced(provider_name, provider_info)
        
        if status_icon == "❌":
            console.print(f"[red]Provider not ready:[/red] {status_reason}")
            
            # Provide specific help
            if provider_name.lower() == "ollama":
                console.print("[yellow]💡 Start Ollama with:[/yellow] ollama serve")
            elif "No API key" in status_reason:
                env_var = f"{provider_name.upper()}_API_KEY"
                console.print(f"[yellow]💡 Set API key with:[/yellow] mcp provider set {provider_name} api_key YOUR_KEY")
                console.print(f"[yellow]💡 Or set environment variable:[/yellow] export {env_var}=YOUR_KEY")
            
            return
        
        elif status_icon == "⚠️":
            console.print(f"[yellow]Warning:[/yellow] {status_reason}")
            console.print("[dim]Continuing anyway...[/dim]")
            
    except Exception as e:
        console.print(f"[yellow]Warning:[/yellow] Could not validate provider: {e}")

    # Determine target model
    if model_name:
        target_model = model_name
    else:
        # Get default model
        try:
            target_model = model_manager.get_default_model(provider_name)
            if not target_model:
                # Fallback to first available model
                available_models = model_manager.get_available_models(provider_name)
                target_model = available_models[0] if available_models else "default"
        except Exception:
            target_model = "default"
    
    console.print(f"[dim]Switching to {provider_name} (model: {target_model})...[/dim]")

    # Perform the switch
    try:
        model_manager.switch_model(provider_name, target_model)
    except Exception as e:
        console.print(f"[red]Failed to switch provider:[/red] {e}")
        return

    # Update context
    try:
        context.update({
            "provider": provider_name,
            "model": target_model,
            "client": model_manager.get_client(),
            "model_manager": model_manager,
        })
    except Exception as e:
        console.print(f"[yellow]Warning:[/yellow] Could not update client context: {e}")
    
    console.print(f"[green]✅ Switched to {provider_name}[/green] (model: {target_model})")


# Update the main action function with enhanced sub-commands
async def provider_action_async(
    args: List[str],
    *,
    context: Dict,
) -> None:
    """Enhanced provider action with all optimizations applied."""
    model_manager: ModelManager = context.get("model_manager") or ModelManager()
    context.setdefault("model_manager", model_manager)

    def _show_status() -> None:
        provider, model = model_manager.get_active_provider_and_model()
        status = model_manager.get_status_summary()
        
        # Get enhanced status for current provider
        try:
            all_providers = model_manager.list_available_providers()
            current_info = all_providers.get(provider, {})
            status_icon, status_text, status_reason = _get_provider_status_enhanced(provider, current_info)
            
            console.print(f"[cyan]Current provider:[/cyan] {provider}")
            console.print(f"[cyan]Current model   :[/cyan] {model}")
            console.print(f"[cyan]Status          :[/cyan] {status_icon} {status_text}")
            console.print(f"[cyan]Features        :[/cyan] {_format_features(status)}")
            
            if status_icon != "✅":
                console.print(f"[yellow]Note:[/yellow] {status_reason}")
                
        except Exception as e:
            console.print(f"[cyan]Current provider:[/cyan] {provider}")
            console.print(f"[cyan]Current model   :[/cyan] {model}")
            console.print(f"[yellow]Status check failed:[/yellow] {e}")

    def _format_features(status: Dict) -> str:
        features = []
        if status.get('supports_streaming'):
            features.append("📡 streaming")
        if status.get('supports_tools'):
            features.append("🔧 tools")
        if status.get('supports_vision'):
            features.append("👁️ vision")
        return " ".join(features) or "📄 text only"

    # Dispatch logic
    if not args:
        _show_status()
        return

    sub, *rest = args
    sub = sub.lower()

    if sub == "list":
        _render_list_optimized(model_manager)
        return

    if sub == "config":
        _render_config(model_manager)
        return

    if sub == "diagnostic":
        target = rest[0] if rest else None
        _render_diagnostic_optimized(model_manager, target)
        return

    if sub == "set" and len(rest) >= 2:
        provider_name, setting = rest[0], rest[1]
        value = rest[2] if len(rest) >= 3 else None
        _mutate(model_manager, provider_name, setting, value)
        return

    # Provider switching
    provider_name = sub
    model_name = rest[0] if rest else None
    _switch_provider_enhanced(model_manager, provider_name, model_name, context)


# Keep existing helper functions but use them in the enhanced versions above
def _render_config(model_manager: ModelManager) -> None:
    """Show detailed configuration - keeping your existing implementation."""
    # ... existing implementation
    pass

def _mutate(model_manager: ModelManager, provider: str, key: str, value: str) -> None:
    """Update provider configuration - keeping your existing implementation.""" 
    # ... existing implementation
    pass

# Sync wrapper
def provider_action(args: List[str], *, context: Dict) -> None:
    """Sync wrapper for provider_action_async."""
    from mcp_cli.utils.async_utils import run_blocking
    run_blocking(provider_action_async(args, context=context))