"""A collection of Errors used by the Naludaq package.

An error in the `naludaq` package should derive from the NaludaqException which is the base
exception. By Having a Common base class it's easier to capture and trace exceptions
generated by the package.

"""


class NaludaqException(Exception):
    """Base class for Naludaq exceptions, for tracability."""


class InvalidParameterFile(NaludaqException):
    """The supplied parameter file is not valid."""


class InvalidFileType(NaludaqException):
    """Rased if the filetype is not recognized."""


class BoardConnectionError(NaludaqException):
    """Raised when the board connection unexpectedly failed."""


class InvalidBoardModelError(NaludaqException):
    """Raised when the boardmodel doesn't exist in the list of models."""


class BadDataError(NaludaqException):
    """Captured data is not a valid event."""


class NotAValidEvent(NaludaqException):
    """Event couldn't be validated.

    This Event is not what the program excepted.
    """


class NoDefaultRegistersError(NaludaqException):
    """This board model have no default registers. User must load a file."""


class ClockFileError(NaludaqException):
    """Board needs a clock file but None is loaded."""


class InvalidClockFileError(NaludaqException):
    """Clock file contains invalid clock data"""


class RegisterError(NaludaqException):
    """Generic register Error"""


class RegisterNameError(RegisterError):
    """Not a valid register name."""


class RegisterFileError(RegisterError):
    """Board needs a valid parameter file."""


class InvalidRegisterError(RegisterError):
    """Not a valid register"""


class IncompatibleParserError(NaludaqException):
    """Parser is not compatible with the selected protocol/board model."""


class PedestalsDataCaptureError(NaludaqException):
    """Couldn't capture pedestals data correctly.

    Either due to bad returning blocks or no data captured at all.
    """


class BoardParameterError(NaludaqException):
    """The board paramters lack enough fields to describe the board.

    This could be due to a faulty or old parameter/register file.
    """


class DaqStateError(NaludaqException):
    """The DAQ is in an invalid state for some operation.

    This could be due to workers already being initialized, started,
    or stopped.
    """


class InvalidTriggerMaskError(NaludaqException):
    """The trigger mask supplied is invalid."""


class NoLinearRegionFoundError(NaludaqException):
    """No linear region was found from DAC sweep data.

    This could be due to a poor dac sweep, or too strict parameters
    set in determining the linear region.
    """


class NoCalibrationDataFoundError(NaludaqException):
    """No calibration data attribute found on the board.

    This could be due to the calibration has not been run or the calibration data is corrupted.
    """


class FTDIError(NaludaqException):
    """A problem occurred while interacting with the FTDI drivers or a device
    connected through FTDI.
    """


# ExtDac related errors ######################################################
class DACError(NaludaqException):
    """There was a problem with setting the DACs."""


class WrongDeviceError(Exception):
    """Raised if the wrong DAC device is called for a specific channel."""


# SPI related errors #########################################################


class NotInitializedError(NaludaqException):
    """The device has not been initialized."""


# i2c related Errors #########################################################


class I2CError(NaludaqException):
    """Communication with device through I2C failed."""


# Autoaction related errors ##################################################


class AutoActionError(NaludaqException):
    """The AutoAction tool encountered an error"""


class AutoTriggerError(AutoActionError):
    """TheAutoTrigger tool encountered an error"""


##############################################################################


class OperationCanceledError(NaludaqException):
    """The operation was canceled."""


class VersionError(NaludaqException):
    """Invalid or unsupported version"""


class ProjectIOError(NaludaqException):
    """Error when reading or writing project files"""


class AcquisitionIOError(NaludaqException):
    """Error when acquiring data"""


class PedestalsIOError(NaludaqException):
    """Error when accessing pedestals files"""


class OptimizationError(NaludaqException):
    """Error while running an optimization script"""


class InvalidOptimizationParameters(NaludaqException):
    """Error with incorrect optimization parameters"""


class IterationError(NaludaqException):
    """Error in the stages of a FancyIterator"""


class DataCaptureError(NaludaqException):
    """Error while capturing data, raised by the data collector when too much data is bad."""
