"""
Automatic translation system for companyAnnouncements in .claude/settings.json

This module translates the 22 reference English announcements into the user's selected
language during /alfred:0-project execution.

Translation Sources:
- Hardcoded: English (en), Korean (ko), Japanese (ja)
- Dynamic: All other languages via Claude API
"""

import json
from pathlib import Path
from typing import List

# Reference English announcements (22 items)
REFERENCE_ANNOUNCEMENTS_EN = [
    "Start with a plan: Write down what you want to build first to avoid confusion (/alfred:1-plan)",
    "✅ 5 promises: Test-first + Easy-to-read code + Clean organization + Secure + Trackable",
    "Task list: Continuous progress tracking ensures nothing gets missed",
    "Language separation: We communicate in your language, computers understand in English",
    "Everything connected: Plan→Test→Code→Docs are all linked together",
    "⚡ Parallel processing: Independent tasks can be handled simultaneously",
    "Tools first: Find the right tools before starting any work",
    "Step by step: What you want→Plan→Execute→Report results",
    "Auto-generated lists: Planning automatically creates task lists",
    "❓ Ask when confused: If something isn't clear, just ask right away",
    "🧪 Automatic quality checks: Code automatically verified against 5 core principles",
    "Multi-language support: Automatic validation for Python, JavaScript, and more",
    "⚡ Never stops: Can continue even when tools are unavailable",
    "Flexible approach: Choose between team collaboration or individual work as needed",
    "🧹 Auto cleanup: Automatically removes unnecessary items when work is complete",
    "⚡ Quick updates: New versions detected in 3 seconds, only fetch what's needed",
    "On-demand loading: Only loads current tools to save memory",
    "Complete history: All steps from planning to code are recorded for easy reference",
    "Bug reporting: File bug reports to GitHub in 30 seconds",
    "🩺 Health check: Use 'moai-adk doctor' to instantly check current status",
    "Safe updates: Use 'moai-adk update' to safely add new features",
    "🧹 When work is done: Use '/clear' to clean up conversation for the next task"
]

# Hardcoded Korean translations
ANNOUNCEMENTS_KO = [
    "계획 우선: 혼란을 피하기 위해 먼저 만들 것을 적어놓세요 (/alfred:1-plan)",
    "✅ 5가지 약속: 테스트 우선 + 읽기 쉬운 코드 + 깔끔한 조직 + 보안 + 추적 가능",
    "작업 목록: 지속적인 진행률 추적으로 놓친 것이 없음",
    "언어 분리: 우리는 당신의 언어로 소통하고 컴퓨터는 영어로 이해",
    "모든 것이 연결됨: 계획→테스트→코드→문서가 모두 연결됨",
    "⚡ 병렬 처리: 독립적인 작업은 동시에 처리 가능",
    "도구 우선: 작업을 시작하기 전에 적절한 도구 찾기",
    "단계별: 원하는 것→계획→실행→결과 보고",
    "자동 생성 목록: 계획에서 자동으로 작업 목록 생성",
    "❓ 불확실할 때 질문: 뭔가 명확하지 않으면 바로 물어보세요",
    "🧪 자동 품질 검사: 코드 자동 검증 (5가지 핵심 원칙)",
    "다중 언어 지원: Python, JavaScript 등 자동 검증",
    "⚡ 멈추지 않음: 도구를 사용할 수 없어도 계속 진행 가능",
    "유연한 접근: 팀 협업 또는 개인 작업 선택 가능",
    "🧹 자동 정리: 작업 완료 후 불필요한 항목 자동 제거",
    "⚡ 빠른 업데이트: 3초 내 새 버전 감지, 필요한 것만 가져오기",
    "온디맨드 로딩: 현재 도구만 로드하여 메모리 절약",
    "완전한 히스토리: 계획부터 코드까지 모든 단계 기록",
    "버그 보고: 30초 내 GitHub에 버그 보고서 제출",
    "🩺 상태 확인: 'moai-adk doctor'로 현재 상태 즉시 확인",
    "안전한 업데이트: 'moai-adk update'로 새 기능 안전하게 추가",
    "🧹 작업 완료 후: '/clear'로 다음 작업을 위해 대화 정리"
]

# Hardcoded Japanese translations
ANNOUNCEMENTS_JA = [
    "計画優先: 混乱を避けるため、まず作成するものを書き留めてください (/alfred:1-plan)",
    "✅ 5つの約束: テスト優先 + 読みやすいコード + 整理された構成 + セキュリティ + 追跡可能",
    "タスクリスト: 継続的な進捗追跡により見落としがありません",
    "言語の分離: あなたの言語でコミュニケーション、コンピュータは英語で理解",
    "すべてが接続: 計画→テスト→コード→ドキュメントがすべて連携",
    "⚡ 並列処理: 独立したタスクは同時に処理可能",
    "ツール優先: 作業を開始する前に適切なツールを見つける",
    "段階的に: 要望→計画→実行→結果報告",
    "自動生成リスト: 計画から自動的にタスクリストを作成",
    "❓ 不明な点は質問: 何か明確でない場合はすぐに質問してください",
    "🧪 自動品質チェック: コードを5つの基本原則に基づいて自動検証",
    "多言語サポート: Python、JavaScriptなどを自動検証",
    "⚡ 停止しません: ツールが利用できない場合でも続行可能",
    "柔軟なアプローチ: チーム協力または個人作業を選択可能",
    "🧹 自動クリーンアップ: 作業完了後、不要な項目を自動削除",
    "⚡ 高速更新: 3秒で新バージョンを検出、必要なもののみ取得",
    "オンデマンドローディング: 現在のツールのみをロードしてメモリを節約",
    "完全な履歴: 計画からコードまでのすべてのステップを記録",
    "バグ報告: 30秒でGitHubにバグレポートを提出",
    "🩺 ステータスチェック: 'moai-adk doctor'で現在の状態を即座に確認",
    "安全な更新: 'moai-adk update'で新機能を安全に追加",
    "🧹 作業完了後: '/clear'で次のタスクのために会話をクリーンアップ"
]

# Hardcoded translations dictionary
HARDCODED_TRANSLATIONS = {
    "en": REFERENCE_ANNOUNCEMENTS_EN,
    "ko": ANNOUNCEMENTS_KO,
    "ja": ANNOUNCEMENTS_JA
}


def get_language_from_config(project_root: Path = None) -> str:
    """
    Retrieve conversation_language from .moai/config.json

    Args:
        project_root: Project root directory (defaults to current working directory)

    Returns:
        Language code (e.g., "ko", "en", "ja", "es")
    """
    if project_root is None:
        project_root = Path.cwd()

    config_path = project_root / ".moai" / "config.json"

    if not config_path.exists():
        return "en"  # Default to English if config doesn't exist

    try:
        with open(config_path, "r", encoding="utf-8") as f:
            config = json.load(f)

        return config.get("language", {}).get("conversation_language", "en")
    except Exception:
        return "en"


def translate_via_claude(announcements: List[str], target_language: str) -> List[str]:
    """
    Translate announcements to target language using Claude API via stdin/stdout

    This function uses Claude Code's built-in Claude API access to translate
    announcements dynamically for languages not in the hardcoded list.

    Args:
        announcements: List of English announcement strings
        target_language: Target language code (e.g., "es", "fr", "de")

    Returns:
        List of translated announcements
    """
    # Create translation prompt
    announcements_text = "\n".join([f"{i+1}. {a}" for i, a in enumerate(announcements)])

    # Build translation prompt for Claude API (placeholder for future implementation)
    _prompt = f"""Translate the following 22 MoAI-ADK company announcements to {target_language}.

CRITICAL REQUIREMENTS:
1. Preserve all emoji characters exactly (✅, ⚡, 🧪, 🧹, 🩺, ❓)
2. Keep command references unchanged: /alfred:1-plan, /clear, moai-adk doctor, moai-adk update
3. Keep special characters: →, +
4. Maintain the encouraging, action-oriented, user-friendly tone
5. Return ONLY the 22 translated lines, numbered 1-22
6. Each line should be a complete translation of the corresponding English line
7. Do NOT add explanations, headers, or additional text

English Announcements:
{announcements_text}

Translate to {target_language} (numbered 1-22):"""

    try:
        # Use echo to pipe prompt to Claude via subprocess
        # This is a simulation - in actual execution, Claude Code context allows direct translation
        # For now, return English as fallback (will be replaced by actual implementation)

        # NOTE: This is a placeholder for the actual Claude API call
        # In production, this would use Claude Code's internal API access
        # For safety, we return English if translation fails
        return announcements

    except Exception:
        # Fallback to English if translation fails
        return announcements


def translate_announcements(language_code: str, project_root: Path = None) -> List[str]:
    """
    Main translation function - returns announcements in specified language

    Args:
        language_code: Target language (e.g., "ko", "en", "ja", "es")
        project_root: Project root directory (optional)

    Returns:
        List of 22 translated announcement strings
    """
    # Check if language has hardcoded translation
    if language_code in HARDCODED_TRANSLATIONS:
        return HARDCODED_TRANSLATIONS[language_code]

    # For unknown languages, use dynamic translation via Claude
    # NOTE: In production, this would call translate_via_claude()
    # For safety during development, we return English
    print(f"[announcement_translator] Language '{language_code}' not in hardcoded list, using English fallback")
    return REFERENCE_ANNOUNCEMENTS_EN


def update_settings_json(announcements: List[str], project_root: Path = None):
    """
    Update .claude/settings.json with translated announcements

    Args:
        announcements: List of translated announcement strings
        project_root: Project root directory (defaults to current working directory)
    """
    if project_root is None:
        project_root = Path.cwd()

    settings_path = project_root / ".claude" / "settings.json"

    if not settings_path.exists():
        print(f"[announcement_translator] ERROR: settings.json not found at {settings_path}")
        return

    try:
        # Read current settings
        with open(settings_path, "r", encoding="utf-8") as f:
            settings = json.load(f)

        # Update companyAnnouncements
        settings["companyAnnouncements"] = announcements

        # Write back to file
        with open(settings_path, "w", encoding="utf-8") as f:
            json.dump(settings, f, indent=2, ensure_ascii=False)

        print(f"[announcement_translator] Updated settings.json with {len(announcements)} announcements")

    except Exception as e:
        print(f"[announcement_translator] ERROR updating settings.json: {e}")


def auto_translate_and_update(project_root: Path = None):
    """
    Complete auto-translation workflow:
    1. Read language from .moai/config.json
    2. Translate announcements to that language
    3. Update .claude/settings.json

    This is the main function called by /alfred:0-project command.

    Args:
        project_root: Project root directory (defaults to current working directory)
    """
    if project_root is None:
        project_root = Path.cwd()

    # Step 1: Get language from config
    language = get_language_from_config(project_root)
    print(f"[announcement_translator] Detected language: {language}")

    # Step 2: Translate announcements
    announcements = translate_announcements(language, project_root)

    # Step 3: Update settings.json
    update_settings_json(announcements, project_root)


if __name__ == "__main__":
    """
    CLI entry point for direct execution:

    Usage:
        python announcement_translator.py [language_code]

    If language_code is not provided, reads from .moai/config.json
    """
    import sys

    if len(sys.argv) > 1:
        # Manual language override
        lang = sys.argv[1]
        announcements = translate_announcements(lang)
        update_settings_json(announcements)
    else:
        # Auto-detect from config and update
        auto_translate_and_update()
