# TRUST Validation Examples

This document provides complete, working examples demonstrating TRUST 5-principles validation across multiple languages and scenarios.

---

## Example 1: Python Authentication Module (Full TRUST Cycle)

### Project Structure
```
auth-service/
   src/
      auth/
          __init__.py
          service.py
          models.py
   tests/
      auth/
          __init__.py
          test_service.py
   .moai/
      specs/
          SPEC-AUTH-001/
              spec.md
   pyproject.toml
   .secrets.baseline
   scripts/
       validate-tags.sh
```

### SPEC Document (`.moai/specs/SPEC-AUTH-001/spec.md`)
```markdown
---
id: SPEC-AUTH-001
version: 0.1.0
status: active
created: 2025-10-22
---

# SPEC: JWT Authentication Service

@SPEC:AUTH-001

## Requirements

WHEN a user provides valid credentials,
THEN the system SHALL generate a JWT token with 1-hour expiration.

WHEN a user provides invalid credentials,
THEN the system SHALL reject the request with 401 Unauthorized.

## HISTORY
- 2025-10-22 v0.1.0: Initial specification
```

### Implementation (`src/auth/service.py`)
```python
# @CODE:AUTH-001 | SPEC: .moai/specs/SPEC-AUTH-001/spec.md | TEST: tests/auth/test_service.py
"""JWT authentication service implementation."""
import jwt
from datetime import datetime, timedelta
from typing import Optional

SECRET_KEY = "your-secret-key"  # In production, use env var


def authenticate(username: str, password: str) -> Optional[str]:
    """
    Authenticate user and generate JWT token.

    Args:
        username: User's username
        password: User's password

    Returns:
        JWT token if credentials are valid, None otherwise
    """
    # @CODE:AUTH-001:DOMAIN - Business logic validation
    if not username or not password:
        return None

    # Mock credential validation (replace with real DB check)
    if username == "admin" and password == "secret123":
        payload = {
            "sub": username,
            "exp": datetime.utcnow() + timedelta(hours=1),
            "iat": datetime.utcnow(),
        }
        return jwt.encode(payload, SECRET_KEY, algorithm="HS256")

    return None


def verify_token(token: str) -> bool:
    """
    Verify JWT token validity.

    Args:
        token: JWT token to verify

    Returns:
        True if token is valid, False otherwise
    """
    try:
        jwt.decode(token, SECRET_KEY, algorithms=["HS256"])
        return True
    except jwt.InvalidTokenError:
        return False
```

### Tests (`tests/auth/test_service.py`)
```python
# @TEST:AUTH-001 | SPEC: .moai/specs/SPEC-AUTH-001/spec.md
"""Tests for JWT authentication service."""
import pytest
from src.auth.service import authenticate, verify_token


class TestAuthentication:
    """Test suite for authentication service."""

    def test_valid_credentials_generate_token(self):
        """WHEN user provides valid credentials, THEN token is generated."""
        # @TEST:AUTH-001 - Valid credential flow
        token = authenticate("admin", "secret123")

        assert token is not None
        assert isinstance(token, str)
        assert len(token) > 0

    def test_invalid_credentials_return_none(self):
        """WHEN user provides invalid credentials, THEN None is returned."""
        # @TEST:AUTH-001 - Invalid credential flow
        token = authenticate("admin", "wrong-password")

        assert token is None

    def test_empty_credentials_return_none(self):
        """WHEN user provides empty credentials, THEN None is returned."""
        # @TEST:AUTH-001 - Edge case validation
        assert authenticate("", "") is None
        assert authenticate("admin", "") is None
        assert authenticate("", "password") is None

    def test_generated_token_is_valid(self):
        """WHEN token is generated, THEN it should be verifiable."""
        # @TEST:AUTH-001 - Token verification
        token = authenticate("admin", "secret123")
        assert token is not None

        is_valid = verify_token(token)
        assert is_valid is True

    def test_invalid_token_verification_fails(self):
        """WHEN invalid token is verified, THEN verification fails."""
        # @TEST:AUTH-001 - Invalid token handling
        is_valid = verify_token("invalid.token.here")
        assert is_valid is False
```

### Configuration (`pyproject.toml`)
```toml
[project]
name = "auth-service"
version = "0.1.0"

[tool.pytest.ini_options]
testpaths = ["tests"]
addopts = [
    "--cov=src",
    "--cov-report=term-missing",
    "--cov-fail-under=85",
]

[tool.coverage.run]
branch = true
source = ["src"]

[tool.ruff]
line-length = 100
target-version = "py311"

[tool.ruff.lint]
select = ["E", "F", "I", "N", "B", "SIM"]

[tool.mypy]
python_version = "3.11"
strict = true
```

### Running TRUST Validation

```bash
# T - Test First (Coverage e85%)
$ pytest --cov=src --cov-report=term-missing --cov-fail-under=85

tests/auth/test_service.py .....                                    [100%]

---------- coverage: platform darwin, python 3.11.8 -----------
Name                  Stmts   Miss  Cover   Missing
---------------------------------------------------
src/auth/service.py      15      0   100%
---------------------------------------------------
TOTAL                    15      0   100%

 Coverage: 100% (e85%)

# R - Readable
$ ruff check .
All checks passed!

# U - Unified
$ mypy src/ --strict
Success: no issues found in 2 source files

# S - Secured
$ detect-secrets scan --baseline .secrets.baseline
No secrets detected!

$ trivy fs --severity HIGH,CRITICAL .
Total: 0 (HIGH: 0, CRITICAL: 0)

# T - Trackable
$ rg '@(SPEC|CODE|TEST):AUTH-001' -n
.moai/specs/SPEC-AUTH-001/spec.md:7:@SPEC:AUTH-001
src/auth/service.py:1:# @CODE:AUTH-001
tests/auth/test_service.py:1:# @TEST:AUTH-001

 TAG chain complete!
```

**Result**: All 5 TRUST principles pass. Ready to merge.

---

## Example 2: TypeScript API Endpoint (Vitest + Biome)

### Project Structure
```
user-api/
   src/
      users/
         service.ts
         types.ts
      index.ts
   tests/
      users/
          service.test.ts
   .moai/
      specs/
          SPEC-USER-001/
              spec.md
   package.json
   tsconfig.json
   vitest.config.ts
   biome.json
```

### SPEC Document (`.moai/specs/SPEC-USER-001/spec.md`)
```markdown
---
id: SPEC-USER-001
version: 0.1.0
status: active
created: 2025-10-22
---

# SPEC: User Creation Service

@SPEC:USER-001

## Requirements

WHEN a client requests user creation with valid data,
THEN the system SHALL create a user and return a unique ID.

WHEN a client requests user creation with invalid email,
THEN the system SHALL reject with validation error.

## HISTORY
- 2025-10-22 v0.1.0: Initial specification
```

### Types (`src/users/types.ts`)
```typescript
// @CODE:USER-001:DATA | SPEC: .moai/specs/SPEC-USER-001/spec.md
export interface User {
  id: string;
  email: string;
  name: string;
  createdAt: Date;
}

export interface CreateUserInput {
  email: string;
  name: string;
}

export class ValidationError extends Error {
  constructor(message: string) {
    super(message);
    this.name = 'ValidationError';
  }
}
```

### Service (`src/users/service.ts`)
```typescript
// @CODE:USER-001 | SPEC: .moai/specs/SPEC-USER-001/spec.md | TEST: tests/users/service.test.ts
import { User, CreateUserInput, ValidationError } from './types';

/**
 * Validates email format.
 */
function isValidEmail(email: string): boolean {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email);
}

/**
 * Creates a new user with validated input.
 *
 * @CODE:USER-001:DOMAIN - Core user creation logic
 */
export async function createUser(input: CreateUserInput): Promise<User> {
  // Validation
  if (!isValidEmail(input.email)) {
    throw new ValidationError('Invalid email format');
  }

  if (!input.name || input.name.trim().length === 0) {
    throw new ValidationError('Name is required');
  }

  // Create user (mock implementation)
  const user: User = {
    id: crypto.randomUUID(),
    email: input.email.toLowerCase(),
    name: input.name.trim(),
    createdAt: new Date(),
  };

  return user;
}

/**
 * Retrieves user by ID.
 */
export async function getUserById(id: string): Promise<User | null> {
  // Mock implementation
  if (!id) {
    return null;
  }

  return null; // Would fetch from database
}
```

### Tests (`tests/users/service.test.ts`)
```typescript
// @TEST:USER-001 | SPEC: .moai/specs/SPEC-USER-001/spec.md
import { describe, it, expect } from 'vitest';
import { createUser } from '../../src/users/service';
import { ValidationError } from '../../src/users/types';

describe('User Service', () => {
  describe('createUser', () => {
    it('should create user with valid input', async () => {
      // @TEST:USER-001 - Valid user creation
      const input = {
        email: 'test@example.com',
        name: 'Test User',
      };

      const user = await createUser(input);

      expect(user).toBeDefined();
      expect(user.id).toBeTruthy();
      expect(user.email).toBe('test@example.com');
      expect(user.name).toBe('Test User');
      expect(user.createdAt).toBeInstanceOf(Date);
    });

    it('should normalize email to lowercase', async () => {
      // @TEST:USER-001 - Email normalization
      const user = await createUser({
        email: 'Test@EXAMPLE.COM',
        name: 'Test User',
      });

      expect(user.email).toBe('test@example.com');
    });

    it('should throw ValidationError for invalid email', async () => {
      // @TEST:USER-001 - Invalid email rejection
      await expect(
        createUser({ email: 'not-an-email', name: 'Test' })
      ).rejects.toThrow(ValidationError);

      await expect(
        createUser({ email: 'missing@domain', name: 'Test' })
      ).rejects.toThrow('Invalid email format');
    });

    it('should throw ValidationError for empty name', async () => {
      // @TEST:USER-001 - Empty name rejection
      await expect(
        createUser({ email: 'test@example.com', name: '' })
      ).rejects.toThrow(ValidationError);

      await expect(
        createUser({ email: 'test@example.com', name: '   ' })
      ).rejects.toThrow('Name is required');
    });

    it('should trim whitespace from name', async () => {
      // @TEST:USER-001 - Name trimming
      const user = await createUser({
        email: 'test@example.com',
        name: '  Test User  ',
      });

      expect(user.name).toBe('Test User');
    });
  });
});
```

### Configuration Files

**`vitest.config.ts`**:
```typescript
import { defineConfig } from 'vitest/config';

export default defineConfig({
  test: {
    globals: true,
    environment: 'node',
    coverage: {
      provider: 'v8',
      reporter: ['text', 'html', 'lcov'],
      exclude: ['node_modules/', 'dist/', '**/*.test.ts'],
      thresholds: {
        lines: 85,
        branches: 80,
        functions: 85,
        statements: 85,
      },
    },
  },
});
```

**`biome.json`**:
```json
{
  "linter": {
    "enabled": true,
    "rules": { "recommended": true }
  },
  "formatter": {
    "enabled": true,
    "indentStyle": "space",
    "indentWidth": 2
  }
}
```

**`tsconfig.json`**:
```json
{
  "compilerOptions": {
    "target": "ES2022",
    "module": "ESNext",
    "strict": true,
    "esModuleInterop": true,
    "skipLibCheck": true
  },
  "include": ["src/**/*"],
  "exclude": ["node_modules", "dist"]
}
```

### Running TRUST Validation

```bash
# T - Test First
$ npm run test:coverage

 tests/users/service.test.ts (5)
   User Service (5)
     createUser (5)
       should create user with valid input
       should normalize email to lowercase
       should throw ValidationError for invalid email
       should throw ValidationError for empty name
       should trim whitespace from name

Test Files  1 passed (1)
     Tests  5 passed (5)

Coverage:
  Lines      : 95.45% (21/22)
  Branches   : 100% (12/12)
  Functions  : 100% (3/3)
  Statements : 95.45% (21/22)

 Coverage e85%

# R - Readable
$ npx biome check .
All checks passed!

# U - Unified
$ npm run type-check
No errors found

# S - Secured
$ npm audit --audit-level=high
found 0 vulnerabilities

$ trivy fs --severity HIGH,CRITICAL .
Total: 0 (HIGH: 0, CRITICAL: 0)

# T - Trackable
$ rg '@(SPEC|CODE|TEST):USER-001' -n
.moai/specs/SPEC-USER-001/spec.md:7:@SPEC:USER-001
src/users/service.ts:1:# @CODE:USER-001
tests/users/service.test.ts:1:# @TEST:USER-001

 TAG chain complete!
```

**Result**: All TRUST principles pass.

---

## Example 3: Go HTTP Handler (Table-Driven Tests)

### Project Structure
```
payment-service/
   internal/
      payment/
          handler.go
          handler_test.go
   .moai/
      specs/
          SPEC-PAY-001/
              spec.md
   go.mod
   go.sum
   Makefile
   scripts/
       validate-tags.sh
```

### SPEC Document (`.moai/specs/SPEC-PAY-001/spec.md`)
```markdown
---
id: SPEC-PAY-001
version: 0.1.0
status: active
created: 2025-10-22
---

# SPEC: Payment Amount Validation

@SPEC:PAY-001

## Requirements

WHEN payment amount is positive,
THEN the system SHALL process the payment.

WHEN payment amount is zero or negative,
THEN the system SHALL reject with validation error.

WHEN payment amount exceeds 10,000,
THEN the system SHALL reject with limit exceeded error.

## HISTORY
- 2025-10-22 v0.1.0: Initial specification
```

### Implementation (`internal/payment/handler.go`)
```go
// @CODE:PAY-001 | SPEC: .moai/specs/SPEC-PAY-001/spec.md | TEST: internal/payment/handler_test.go
package payment

import (
	"errors"
	"fmt"
)

var (
	// ErrInvalidAmount indicates payment amount is invalid
	ErrInvalidAmount = errors.New("payment amount must be positive")
	// ErrLimitExceeded indicates payment exceeds maximum allowed
	ErrLimitExceeded = errors.New("payment amount exceeds limit")
)

const MaxPaymentAmount = 10000.0

// PaymentRequest represents a payment request.
// @CODE:PAY-001:DATA - Payment data structure
type PaymentRequest struct {
	Amount   float64 `json:"amount"`
	Currency string  `json:"currency"`
}

// ValidatePayment validates payment request.
// @CODE:PAY-001:DOMAIN - Payment validation logic
func ValidatePayment(req PaymentRequest) error {
	if req.Amount <= 0 {
		return ErrInvalidAmount
	}

	if req.Amount > MaxPaymentAmount {
		return fmt.Errorf("%w: %.2f", ErrLimitExceeded, req.Amount)
	}

	return nil
}

// ProcessPayment processes a validated payment.
func ProcessPayment(req PaymentRequest) (string, error) {
	if err := ValidatePayment(req); err != nil {
		return "", err
	}

	// Mock payment processing
	txID := fmt.Sprintf("tx_%d", int(req.Amount*100))
	return txID, nil
}
```

### Tests (`internal/payment/handler_test.go`)
```go
// @TEST:PAY-001 | SPEC: .moai/specs/SPEC-PAY-001/spec.md
package payment

import (
	"errors"
	"testing"
)

func TestValidatePayment(t *testing.T) {
	// @TEST:PAY-001 - Table-driven payment validation tests
	tests := []struct {
		name    string
		request PaymentRequest
		wantErr error
	}{
		{
			name:    "valid payment within limits",
			request: PaymentRequest{Amount: 100.00, Currency: "USD"},
			wantErr: nil,
		},
		{
			name:    "valid payment at maximum limit",
			request: PaymentRequest{Amount: 10000.00, Currency: "USD"},
			wantErr: nil,
		},
		{
			name:    "invalid zero amount",
			request: PaymentRequest{Amount: 0, Currency: "USD"},
			wantErr: ErrInvalidAmount,
		},
		{
			name:    "invalid negative amount",
			request: PaymentRequest{Amount: -50.00, Currency: "USD"},
			wantErr: ErrInvalidAmount,
		},
		{
			name:    "exceeds maximum limit",
			request: PaymentRequest{Amount: 10000.01, Currency: "USD"},
			wantErr: ErrLimitExceeded,
		},
		{
			name:    "far exceeds maximum limit",
			request: PaymentRequest{Amount: 50000.00, Currency: "USD"},
			wantErr: ErrLimitExceeded,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			err := ValidatePayment(tt.request)

			if tt.wantErr == nil {
				if err != nil {
					t.Errorf("ValidatePayment() error = %v, wantErr %v", err, tt.wantErr)
				}
			} else {
				if !errors.Is(err, tt.wantErr) {
					t.Errorf("ValidatePayment() error = %v, wantErr %v", err, tt.wantErr)
				}
			}
		})
	}
}

func TestProcessPayment(t *testing.T) {
	// @TEST:PAY-001 - Payment processing workflow
	t.Run("process valid payment", func(t *testing.T) {
		req := PaymentRequest{Amount: 123.45, Currency: "USD"}

		txID, err := ProcessPayment(req)

		if err != nil {
			t.Fatalf("ProcessPayment() unexpected error: %v", err)
		}

		if txID == "" {
			t.Error("ProcessPayment() returned empty transaction ID")
		}
	})

	t.Run("reject invalid payment", func(t *testing.T) {
		req := PaymentRequest{Amount: -10.00, Currency: "USD"}

		_, err := ProcessPayment(req)

		if !errors.Is(err, ErrInvalidAmount) {
			t.Errorf("ProcessPayment() error = %v, want %v", err, ErrInvalidAmount)
		}
	})
}
```

### Makefile
```makefile
.PHONY: test coverage coverage-check lint vet trust

# T - Test First
test:
	go test -v ./...

coverage:
	go test -coverprofile=coverage.out -covermode=atomic ./...
	go tool cover -html=coverage.out -o coverage.html
	@echo "Coverage report: coverage.html"

coverage-check:
	@go test -coverprofile=coverage.out -covermode=atomic ./...
	@go tool cover -func=coverage.out | grep total | awk '{print "Total coverage: " $$3}'
	@go tool cover -func=coverage.out | grep total | awk '{print $$3}' | sed 's/%//' | \
		awk '{if($$1<85){print "FAIL: Coverage below 85%"; exit 1}else{print "PASS: Coverage >= 85%"}}'

# R - Readable
lint:
	golangci-lint run

# U - Unified
vet:
	go vet ./...

# S - Secured
security:
	gosec ./...
	trivy fs --severity HIGH,CRITICAL .

# T - Trackable
tags:
	bash scripts/validate-tags.sh

# Run all TRUST checks
trust: coverage-check lint vet security tags
	@echo " All TRUST principles validated"
```

### Running TRUST Validation

```bash
# T - Test First
$ make coverage-check
go test -coverprofile=coverage.out -covermode=atomic ./...
ok      payment-service/internal/payment    0.234s  coverage: 100.0% of statements
Total coverage: 100.0%
PASS: Coverage >= 85%

# R - Readable
$ make lint
golangci-lint run
 All linters passed

# U - Unified
$ make vet
go vet ./...
 No issues found

# S - Secured
$ make security
gosec ./...
Results: 0 issues
trivy fs --severity HIGH,CRITICAL .
Total: 0 (HIGH: 0, CRITICAL: 0)

# T - Trackable
$ make tags
bash scripts/validate-tags.sh
TAG PAY-001: SPEC=1 CODE=1 TEST=1
TAG validation PASSED

# Run all TRUST checks
$ make trust
 All TRUST principles validated
```

**Result**: Perfect TRUST compliance with 100% test coverage.

---

## Key Takeaways

