"""
Smart Orchestrator for Metis Agent.

This module provides intelligent execution coordination using multiple
strategies for optimal query processing. Works with any LLM provider
through the existing LLM interface.
"""
import json
import time
import os
from datetime import datetime
from typing import Dict, List, Any, Optional, Union

from .models import (
    QueryAnalysis, ExecutionResult, ToolExecutionResult, 
    ExecutionStrategy, ExecutionError
)
from .llm_interface import get_llm
from ..tools.registry import get_tool
from ..blueprints.core.blueprint import Blueprint, BlueprintMetadata, BlueprintInput, BlueprintOutput, BlueprintStep
from ..blueprints.core.engine import BlueprintEngine


class SmartOrchestrator:
    """Orchestrator using existing LLM interface for intelligent execution coordination."""
    
    def __init__(self, tools_registry: Dict[str, Any] = None):
        """
        Initialize the orchestrator.
        
        Args:
            tools_registry: Registry of available tools (optional)
        """
        self.tools_registry = tools_registry or {}
    
    def _get_api_keys(self, config: Any) -> Dict[str, str]:
        """Get all available API keys from config."""
        api_keys = {}
        if config and hasattr(config, 'get_api_key'):
            # Common API key providers
            providers = ['google', 'openai', 'groq', 'anthropic', 'firecrawl', 'GOOGLE_SEARCH_ENGINE']
            for provider in providers:
                try:
                    key = config.get_api_key(provider)
                    if key:
                        api_keys[provider] = key
                        # Also add common variations
                        if provider == 'google':
                            api_keys['google_api_key'] = key
                        elif provider == 'GOOGLE_SEARCH_ENGINE':
                            api_keys['google_cx'] = key
                            api_keys['cx'] = key
                except Exception:
                    continue
        return api_keys
        
    def execute(
        self,
        analysis: QueryAnalysis,
        tools: Dict[str, Any],
        llm: Any,
        memory_context: str = "",
        session_id: Optional[str] = None,
        query: Optional[str] = None,
        system_message: Optional[str] = None,
        config: Optional[Any] = None
    ) -> ExecutionResult:
        """
        Execute query based on analysis results.
        
        Args:
            analysis: Query analysis with strategy and metadata
            tools: Available tools dictionary
            llm: LLM instance for direct responses
            memory_context: Memory context for enhanced responses
            session_id: Session identifier
            query: Original user query (extracted from analysis if not provided)
            
        Returns:
            ExecutionResult with response and metadata
        """
        start_time = time.time()
        
        # Extract query from analysis if not provided
        if query is None:
            query = getattr(analysis, 'query', 'Unknown query')
        
        # Set up context
        context = {
            'tools': tools or {},
            'llm': llm,
            'memory_context': memory_context,
            'session_id': session_id,
            'system_message': system_message,
            'config': config
        }
        
        # Generate planning files for complex queries
        planning_files = None
        
        # Handle both enum and string values
        complexity_value = analysis.complexity
        if hasattr(complexity_value, 'value'):
            complexity_value = complexity_value.value
        
        if hasattr(analysis, 'complexity') and complexity_value in ['complex', 'research']:
            try:
                planning_files = self._generate_planning_files(query, analysis, session_id)
                context['planning_files'] = planning_files
                # Initialize task tracking
                context['task_tracker'] = {
                    'planning_file': planning_files[0] if planning_files else None,
                    'task_file': planning_files[1] if planning_files else None,
                    'completed_tasks': [],
                    'current_step': 0
                }
            except Exception as e:
                # Don't fail execution if planning file generation fails
                context['planning_error'] = str(e)
        
        try:
            if analysis.strategy == ExecutionStrategy.DIRECT_RESPONSE:
                result = self._direct_response(query, context)
            elif analysis.strategy == ExecutionStrategy.SINGLE_TOOL:
                result = self._single_tool_execution(query, analysis, context)
            elif analysis.strategy == ExecutionStrategy.SEQUENTIAL:
                result = self._sequential_execution(query, analysis, context)
            elif analysis.strategy == ExecutionStrategy.PARALLEL:
                result = self._parallel_execution(query, analysis, context)
            elif analysis.strategy == ExecutionStrategy.ITERATIVE:
                result = self._iterative_execution(query, analysis, context)
            else:
                result = self._direct_response(query, context)
            
            execution_time = time.time() - start_time
            result.execution_time = execution_time
            
            # Detect blueprint opportunities for successful executions
            if result.error is None:
                try:
                    opportunity = self.detect_blueprint_opportunity(query, result, context)
                    if opportunity:
                        # Store opportunity in metadata for potential blueprint creation
                        # DO NOT add suggestion to user response - keep it internal only
                        if not hasattr(result, 'metadata') or result.metadata is None:
                            result.metadata = {}
                        result.metadata['blueprint_opportunity'] = opportunity
                        
                        print(f"[BLUEPRINT OPPORTUNITY] Detected {opportunity['blueprint_type']} blueprint opportunity")
                        print(f"[BLUEPRINT OPPORTUNITY] Reusability score: {opportunity['reusability_score']:.1%}")
                        
                except Exception as e:
                    # Don't fail execution if blueprint detection fails
                    print(f"[BLUEPRINT DETECTION ERROR] {e}")
            
            return result
            
        except Exception as e:
            execution_time = time.time() - start_time
            return ExecutionResult(
                response=f"Execution error: {str(e)}",
                strategy_used='error_fallback',
                tools_used=[],
                execution_time=execution_time,
                confidence=0.1,
                metadata={'error': True, 'error_message': str(e)},
                error=str(e)
            )
    
    def _direct_response(self, query: str, context: Dict) -> ExecutionResult:
        """Direct response using existing LLM interface."""
        llm = get_llm()
        
        # Use custom system message from context if available, otherwise fallback to default
        system_prompt = context.get('system_message', "You are a helpful AI assistant. Provide clear, accurate, and concise responses to user queries.")
        
        # Include memory context if available - integrate naturally without mentioning it
        memory_context = context.get('memory_context', '')
        if memory_context.strip():
            system_prompt += f"\n\nBackground information that may be relevant (use naturally without explicitly mentioning context):\n{memory_context}\n\nImportant: Use this background information naturally in your response when relevant, but do not explicitly mention that you have context or previous conversation history unless directly asked about memory or conversation history."
        
        messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": query}
        ]
        
        response = llm.chat(messages)
        
        return ExecutionResult(
            response=response,
            strategy_used='direct_response',
            tools_used=[],
            execution_time=0.0,  # Will be set by caller
            confidence=0.9,
            metadata={'strategy': 'direct_response'}
        )
    
    def _single_tool_execution(
        self, 
        query: str, 
        analysis: QueryAnalysis, 
        context: Dict
    ) -> ExecutionResult:
        """Execute with single tool."""
        if not analysis.required_tools:
            return self._direct_response(query, context)
        
        tool_name = analysis.required_tools[0]
        
        # Get tool from registry
        try:
            if tool_name in self.tools_registry:
                tool = self.tools_registry[tool_name]
            else:
                tool_class = get_tool(tool_name)
                if tool_class:
                    tool = tool_class()  # Instantiate the tool class
                else:
                    tool = None
            
            if tool is None:
                return ExecutionResult(
                    response=f"Tool '{tool_name}' not found",
                    strategy_used='single_tool_error',
                    tools_used=[],
                    execution_time=0.0,
                    confidence=0.1,
                    metadata={'error': True, 'tool_error': 'Tool not found'},
                    error='Tool not found'
                )
        except Exception as e:
            return ExecutionResult(
                response=f"Tool '{tool_name}' not found: {str(e)}",
                strategy_used='single_tool_error',
                tools_used=[],
                execution_time=0.0,
                confidence=0.1,
                metadata={'error': True, 'tool_error': str(e)},
                error=str(e)
            )
        
        # Execute tool with validation
        tool_result, success = self._execute_tool_with_validation(tool_name, tool, query, context)
        
        # If validation failed, return early with error
        if not success:
            return ExecutionResult(
                response=tool_result,  # Contains error message
                strategy_used='single_tool_error',
                tools_used=[],
                execution_time=0.0,
                confidence=0.1,
                metadata={'error': True, 'tool_error': tool_result},
                error=tool_result
            )
        
        # Synthesize with LLM
        llm = get_llm()
        synthesis_prompt = f"""
        User Query: "{query}"
        Tool Used: {tool_name}
        Tool Result: {tool_result}
        
        Synthesize this information into a comprehensive, helpful response for the user.
        Focus on directly answering their question using the tool results.
        """
        
        messages = [
            {"role": "system", "content": "You are a helpful assistant that synthesizes tool results into clear responses."},
            {"role": "user", "content": synthesis_prompt}
        ]
        
        final_response = llm.chat(messages)
        
        return ExecutionResult(
            response=final_response,
            strategy_used='single_tool',
            tools_used=[tool_name],
            execution_time=0.0,
            confidence=analysis.confidence,
            metadata={
                'strategy': 'single_tool',
                'tool_results': {tool_name: tool_result}
            }
        )
    
    def _sequential_execution(
        self, 
        query: str, 
        analysis: QueryAnalysis, 
        context: Dict
    ) -> ExecutionResult:
        """Execute tools in sequence."""
        results = {}
        execution_log = []
        tools_used = []
        
        # Update planning status
        self._update_planning_status(context, "Starting sequential execution", f"Processing {len(analysis.required_tools)} tools")
        
        for i, tool_name in enumerate(analysis.required_tools):
            try:
                # Get tool
                if tool_name in self.tools_registry:
                    tool = self.tools_registry[tool_name]
                else:
                    tool_class = get_tool(tool_name)
                    if tool_class:
                        tool = tool_class()  # Instantiate the tool class
                    else:
                        tool = None
                
                if tool is None:
                    results[tool_name] = f"Error: Tool '{tool_name}' not found"
                    execution_log.append(f"Step {i+1}: {tool_name} not found")
                    self._update_task_progress(context, f"Execute {tool_name}", "failed - tool not found")
                    continue
                
                # Get planning guidance for current step
                planning_guidance = self._reference_planning_file(context, f"Step {i+1}: Execute {tool_name}")
                next_task = self._get_next_task(context)
                
                # Update task status - starting
                self._update_task_progress(context, f"Execute {tool_name}", "started")
                
                # Modify query based on previous results and planning guidance
                if i > 0:
                    context_prompt = f"Previous results: {json.dumps(results, indent=2)}\n\nPlanning guidance: {planning_guidance}\n\nNext task: {next_task}\n\nContinue with original query: {query}"
                else:
                    context_prompt = f"Planning guidance: {planning_guidance}\n\nNext task: {next_task}\n\nOriginal query: {query}"
                
                # Execute tool with validation
                tool_result, success = self._execute_tool_with_validation(tool_name, tool, context_prompt, context)
                
                if success:
                    results[tool_name] = tool_result
                    tools_used.append(tool_name)
                    execution_log.append(f"Step {i+1}: Used {tool_name} successfully")
                    
                    # Update task status - completed
                    self._update_task_progress(context, f"Execute {tool_name}", "completed")
                    self._update_task_progress(context, f"Process {tool_name} results", "completed")
                    
                    # Check if we have sufficient information after this tool
                    if self._has_sufficient_information(query, results):
                        print(f"[EARLY EXIT] Sufficient information gathered after {tool_name}")
                        execution_log.append(f"Early exit: Sufficient information gathered")
                        break
                        
                else:
                    results[tool_name] = tool_result  # Contains error message
                    execution_log.append(f"Step {i+1}: {tool_name} failed - {tool_result}")
                    # Update task status - failed
                    self._update_task_progress(context, f"Execute {tool_name}", f"failed - {tool_result[:50]}")
                    
                    # Continue with next tool instead of stopping completely
                    continue
                
            except Exception as e:
                results[tool_name] = f"Error: {str(e)}"
                execution_log.append(f"Step {i+1}: {tool_name} failed - {str(e)}")
                # Update task status - failed
                self._update_task_progress(context, f"Execute {tool_name}", f"failed - {str(e)}")
        
        # Update final synthesis task
        self._update_task_progress(context, "Integrate multiple tool outputs", "started")
        
        # Final synthesis
        llm = get_llm()
        synthesis_prompt = f"""
        User Query: "{query}"
        Execution Log: {execution_log}
        Tool Results: {json.dumps(results, indent=2)}
        
        Synthesize all these results into a comprehensive final response.
        Address the user's original query using insights from all tool executions.
        """
        
        messages = [
            {"role": "system", "content": "You are a helpful assistant that synthesizes multiple tool results into coherent responses."},
            {"role": "user", "content": synthesis_prompt}
        ]
        
        final_response = llm.chat(messages)
        
        # Update completion tasks
        self._update_task_progress(context, "Integrate multiple tool outputs", "completed")
        self._update_task_progress(context, "Generate final response", "completed")
        self._update_task_progress(context, "Quality check and validation", "completed")
        
        # Update final planning status
        self._update_planning_status(context, "Sequential execution completed", f"Successfully processed {len(tools_used)} tools")
        
        return ExecutionResult(
            response=final_response,
            strategy_used='sequential',
            tools_used=tools_used,
            execution_time=0.0,
            confidence=analysis.confidence,
            metadata={
                'strategy': 'sequential',
                'tool_results': results,
                'execution_log': execution_log
            }
        )
    
    def _parallel_execution(
        self, 
        query: str, 
        analysis: QueryAnalysis, 
        context: Dict
    ) -> ExecutionResult:
        """
        Execute tools in parallel (simulated since current architecture is sync).
        In future, this can be enhanced with actual async execution.
        """
        results = {}
        tools_used = []
        
        # For now, execute sequentially but treat as parallel conceptually
        for tool_name in analysis.required_tools:
            try:
                if tool_name in self.tools_registry:
                    tool = self.tools_registry[tool_name]
                else:
                    tool_class = get_tool(tool_name)
                    if tool_class:
                        tool = tool_class()  # Instantiate the tool class
                    else:
                        tool = None
                
                # Execute tool with validation
                tool_result, success = self._execute_tool_with_validation(tool_name, tool, query, context)
                
                if success:
                    results[tool_name] = tool_result
                    tools_used.append(tool_name)
                else:
                    results[tool_name] = tool_result  # Contains error message
                
            except Exception as e:
                results[tool_name] = f"Error: {str(e)}"
        
        # Synthesize results
        llm = get_llm()
        synthesis_prompt = f"""
        User Query: "{query}"
        Parallel Tool Results: {json.dumps(results, indent=2)}
        
        Synthesize these parallel results into a coherent, comprehensive response.
        Address any conflicts or redundancies in the results.
        Provide a unified answer to the user's query.
        """
        
        messages = [
            {"role": "system", "content": "You are a helpful assistant that synthesizes parallel tool results into unified responses."},
            {"role": "user", "content": synthesis_prompt}
        ]
        
        final_response = llm.chat(messages)
        
        return ExecutionResult(
            response=final_response,
            strategy_used='parallel',
            tools_used=tools_used,
            execution_time=0.0,
            confidence=analysis.confidence,
            metadata={
                'strategy': 'parallel',
                'tool_results': results
            }
        )
    
    def _iterative_execution(
        self, 
        query: str, 
        analysis: QueryAnalysis, 
        context: Dict
    ) -> ExecutionResult:
        """Iterative ReAct-style execution."""
        max_iterations = 3
        conversation_history = []
        tools_used = []
        llm = get_llm()
        
        current_query = query
        
        for iteration in range(max_iterations):
            # Reasoning step
            reasoning_prompt = f"""
            Original Query: {query}
            Current Task: {current_query}
            History: {conversation_history[-2:] if conversation_history else "None"}
            Available Tools: {analysis.required_tools}
            
            Think step by step:
            1. What have I accomplished so far?
            2. What do I still need to do?
            3. Should I use a tool or provide the final answer?
            4. If using a tool, which one and with what query?
            
            Respond in JSON format:
            {{
              "action": "use_tool" or "final_answer",
              "tool_name": "tool name if using tool",
              "tool_query": "query for tool if using tool",
              "reasoning": "your thinking process",
              "ready_for_final": true/false
            }}
            """
            
            messages = [
                {"role": "system", "content": "You are a helpful assistant that reasons step by step and decides on actions."},
                {"role": "user", "content": reasoning_prompt}
            ]
            
            reasoning_response = llm.chat(messages)
            
            # Try to parse JSON response
            try:
                import re
                json_match = re.search(r'\{.*\}', reasoning_response, re.DOTALL)
                if json_match:
                    reasoning_result = json.loads(json_match.group())
                else:
                    reasoning_result = {"action": "final_answer", "reasoning": reasoning_response}
            except:
                reasoning_result = {"action": "final_answer", "reasoning": reasoning_response}
            
            conversation_history.append(f"Iteration {iteration + 1}: {reasoning_result.get('reasoning', 'Thinking...')}")
            
            if reasoning_result.get('action') == 'final_answer' or reasoning_result.get('ready_for_final'):
                # Generate final response
                final_prompt = f"""
                Original Query: "{query}"
                Conversation History: {conversation_history}
                Tools Used: {tools_used}
                
                Provide a comprehensive final answer to the original query.
                """
                
                messages = [
                    {"role": "system", "content": "You are a helpful assistant providing final answers based on reasoning and tool usage."},
                    {"role": "user", "content": final_prompt}
                ]
                
                final_response = llm.chat(messages)
                break
            
            elif reasoning_result.get('action') == 'use_tool':
                # Use the specified tool
                tool_name = reasoning_result.get('tool_name')
                tool_query = reasoning_result.get('tool_query', current_query)
                
                if tool_name in analysis.required_tools:
                    try:
                        if tool_name in self.tools_registry:
                            tool = self.tools_registry[tool_name]
                        else:
                            tool_class = get_tool(tool_name)
                            if tool_class:
                                tool = tool_class()  # Instantiate the tool class
                            else:
                                tool = None
                        
                        # Execute tool with validation
                        tool_result, success = self._execute_tool_with_validation(tool_name, tool, tool_query, context)
                        
                        if not success:
                            conversation_history.append(f"Tool {tool_name} failed: {tool_result}")
                            continue
                        
                        tools_used.append(tool_name)
                        conversation_history.append(f"Tool {tool_name}: {tool_result[:200]}...")
                        current_query = f"Continue with: {query}. Latest info: {tool_result}"
                        
                    except Exception as e:
                        conversation_history.append(f"Tool {tool_name} failed: {str(e)}")
            
            else:
                # Fallback
                messages = [
                    {"role": "system", "content": "You are a helpful assistant."},
                    {"role": "user", "content": current_query}
                ]
                final_response = llm.chat(messages)
                break
        
        else:
            # Max iterations reached
            final_prompt = f"Based on our analysis: {conversation_history}, provide final answer for: {query}"
            messages = [
                {"role": "system", "content": "You are a helpful assistant providing final answers."},
                {"role": "user", "content": final_prompt}
            ]
            final_response = llm.chat(messages)
        
        return ExecutionResult(
            response=final_response,
            strategy_used='iterative',
            tools_used=tools_used,
            execution_time=0.0,
            confidence=analysis.confidence * 0.9,
            metadata={
                'strategy': 'iterative',
                'conversation_history': conversation_history,
                'iterations': iteration + 1
            }
        )
    
    def _generate_planning_files(self, query: str, analysis: QueryAnalysis, session_id: str = None) -> tuple:
        """Generate planning and task MD files for complex queries."""
        # Create planning directory
        planning_dir = "planning_docs"
        os.makedirs(planning_dir, exist_ok=True)
        
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        session_prefix = f"{session_id}_" if session_id else ""
        
        # Generate planning file
        planning_file = f"{planning_dir}/{session_prefix}plan_{timestamp}.md"
        task_file = f"{planning_dir}/{session_prefix}tasks_{timestamp}.md"
        
        # Planning content
        planning_content = f"""# Query Planning Document
Generated: {datetime.now().isoformat()}
Session: {session_id or 'N/A'}

## Original Query
{query}

## Analysis Results
- **Complexity**: {analysis.complexity}
- **Strategy**: {analysis.strategy.value}
- **Confidence**: {analysis.confidence:.2f}
- **Required Tools**: {', '.join(analysis.required_tools)}

## Execution Plan
{self._generate_execution_plan(analysis)}

## Expected Outcomes
{self._generate_expected_outcomes(analysis)}
"""
        
        # Task breakdown
        task_content = f"""# Task Breakdown
Generated: {datetime.now().isoformat()}

## Main Objective
{query}

## Task List
{self._generate_task_list(analysis)}

## Tool Requirements
{self._generate_tool_requirements(analysis)}

## Success Criteria
{self._generate_success_criteria(analysis)}
"""
        
        # Write files
        with open(planning_file, 'w', encoding='utf-8') as f:
            f.write(planning_content)
        
        with open(task_file, 'w', encoding='utf-8') as f:
            f.write(task_content)
        
        return planning_file, task_file
    
    def _generate_execution_plan(self, analysis: QueryAnalysis) -> str:
        """Generate execution plan based on strategy."""
        strategy_plans = {
            ExecutionStrategy.DIRECT_RESPONSE: "Direct LLM response without tool usage",
            ExecutionStrategy.SINGLE_TOOL: f"Execute single tool: {analysis.required_tools[0] if analysis.required_tools else 'Unknown'}",
            ExecutionStrategy.SEQUENTIAL: f"Execute tools sequentially: {' -> '.join(analysis.required_tools)}",
            ExecutionStrategy.PARALLEL: f"Execute tools in parallel: {', '.join(analysis.required_tools)}",
            ExecutionStrategy.ITERATIVE: f"Iterative execution with tools: {', '.join(analysis.required_tools)}"
        }
        
        base_plan = strategy_plans.get(analysis.strategy, "Unknown execution strategy")
        
        return f"""### Strategy: {analysis.strategy.value}
{base_plan}

### Steps:
{self._generate_execution_steps(analysis)}
"""
    
    def _generate_execution_steps(self, analysis: QueryAnalysis) -> str:
        """Generate detailed execution steps."""
        if analysis.strategy == ExecutionStrategy.DIRECT_RESPONSE:
            return "1. Process query with LLM\n2. Return direct response"
        elif analysis.strategy == ExecutionStrategy.SINGLE_TOOL:
            tool_name = analysis.required_tools[0] if analysis.required_tools else "Unknown"
            return f"1. Initialize {tool_name}\n2. Execute tool with query\n3. Process and return results"
        elif analysis.strategy == ExecutionStrategy.SEQUENTIAL:
            steps = []
            for i, tool in enumerate(analysis.required_tools, 1):
                steps.append(f"{i}. Execute {tool}")
            steps.append(f"{len(steps) + 1}. Combine and synthesize results")
            return "\n".join(steps)
        elif analysis.strategy == ExecutionStrategy.PARALLEL:
            return f"1. Initialize all tools: {', '.join(analysis.required_tools)}\n2. Execute tools concurrently\n3. Collect and merge results\n4. Synthesize final response"
        elif analysis.strategy == ExecutionStrategy.ITERATIVE:
            return "1. Analyze query and determine first action\n2. Execute tool if needed\n3. Evaluate results and plan next step\n4. Repeat until completion\n5. Generate final response"
        else:
            return "1. Fallback to direct response"
    
    def _generate_expected_outcomes(self, analysis: QueryAnalysis) -> str:
        """Generate expected outcomes based on complexity and tools."""
        outcomes = []
        
        if analysis.complexity == 'complex':
            outcomes.append("- Comprehensive analysis with multiple data points")
            outcomes.append("- Detailed explanations and reasoning")
            outcomes.append("- Integration of multiple tool outputs")
        elif analysis.complexity == 'research':
            outcomes.append("- In-depth research findings")
            outcomes.append("- Multiple source validation")
            outcomes.append("- Structured research summary")
        else:
            outcomes.append("- Clear and concise response")
            outcomes.append("- Relevant information extraction")
        
        if 'GoogleSearchTool' in analysis.required_tools:
            outcomes.append("- Current web-based information")
        if 'CodeGenerationTool' in analysis.required_tools:
            outcomes.append("- Generated code solutions")
        if 'ContentGenerationTool' in analysis.required_tools:
            outcomes.append("- Generated content/documentation")
        
        return "\n".join(outcomes) if outcomes else "- Standard response output"
    
    def _generate_task_list(self, analysis: QueryAnalysis) -> str:
        """Generate task breakdown list."""
        tasks = []
        
        # Add analysis tasks
        tasks.append("- [ ] Query analysis and complexity assessment")
        tasks.append("- [ ] Strategy selection and tool identification")
        
        # Add tool-specific tasks
        for tool in analysis.required_tools:
            tasks.append(f"- [ ] Execute {tool}")
            tasks.append(f"- [ ] Process {tool} results")
        
        # Add synthesis tasks
        if len(analysis.required_tools) > 1:
            tasks.append("- [ ] Integrate multiple tool outputs")
        
        tasks.append("- [ ] Generate final response")
        tasks.append("- [ ] Quality check and validation")
        
        return "\n".join(tasks)
    
    def _generate_tool_requirements(self, analysis: QueryAnalysis) -> str:
        """Generate tool requirements and dependencies."""
        if not analysis.required_tools:
            return "No external tools required - direct LLM response"
        
        requirements = []
        for tool in analysis.required_tools:
            requirements.append(f"### {tool}")
            
            # Add tool-specific requirements
            if tool == 'GoogleSearchTool':
                requirements.append("- Internet connectivity required")
                requirements.append("- Google Custom Search API access")
            elif tool == 'CodeGenerationTool':
                requirements.append("- LLM access for code generation")
                requirements.append("- Programming language context")
            elif tool == 'ContentGenerationTool':
                requirements.append("- LLM access for content creation")
                requirements.append("- Content type specifications")
            elif tool == 'EnhancedProjectGeneratorTool':
                requirements.append("- File system write permissions")
                requirements.append("- Project template access")
            else:
                requirements.append("- Tool-specific configuration")
            
            requirements.append("")  # Add spacing
        
        return "\n".join(requirements)
    
    def _generate_success_criteria(self, analysis: QueryAnalysis) -> str:
        """Generate success criteria for the task."""
        criteria = [
            "- Query fully addressed and answered",
            "- All required information provided",
            "- Response is clear and well-structured"
        ]
        
        if analysis.complexity in ['complex', 'research']:
            criteria.append("- Multiple perspectives considered")
            criteria.append("- Comprehensive analysis provided")
        
        if len(analysis.required_tools) > 1:
            criteria.append("- All tool outputs successfully integrated")
        
        if 'GoogleSearchTool' in analysis.required_tools:
            criteria.append("- Current and relevant information included")
        
        if any(tool in analysis.required_tools for tool in ['CodeGenerationTool', 'ContentGenerationTool']):
            criteria.append("- Generated content meets quality standards")
        
        return "\n".join(criteria)
    
    def _update_task_progress(self, context: Dict, task_description: str, status: str = "completed"):
        """Update task progress in the task file."""
        task_tracker = context.get('task_tracker')
        if not task_tracker or not task_tracker.get('task_file'):
            return
        
        task_file = task_tracker['task_file']
        if not os.path.exists(task_file):
            return
        
        try:
            # Read current task file
            with open(task_file, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Find and update the specific task
            lines = content.split('\n')
            updated_lines = []
            
            for line in lines:
                if task_description.lower() in line.lower() and '- [ ]' in line:
                    # Mark task as completed
                    updated_line = line.replace('- [ ]', '- [x]')
                    updated_lines.append(updated_line)
                    task_tracker['completed_tasks'].append(task_description)
                else:
                    updated_lines.append(line)
            
            # Add progress timestamp
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            progress_section = f"\n\n## Progress Updates\n- {timestamp}: {task_description} - {status}"
            
            # Check if progress section exists
            if "## Progress Updates" not in content:
                updated_lines.append(progress_section)
            else:
                # Add to existing progress section
                for i, line in enumerate(updated_lines):
                    if "## Progress Updates" in line:
                        updated_lines.insert(i + 1, f"- {timestamp}: {task_description} - {status}")
                        break
            
            # Write updated content
            with open(task_file, 'w', encoding='utf-8') as f:
                f.write('\n'.join(updated_lines))
            
            print(f"[TASK UPDATE] {task_description} - {status}")
            
        except Exception as e:
            print(f"[TASK UPDATE ERROR] Failed to update task progress: {e}")
    
    def _reference_planning_file(self, context: Dict, current_step: str) -> str:
        """Reference the planning file for current step guidance."""
        task_tracker = context.get('task_tracker')
        if not task_tracker or not task_tracker.get('planning_file'):
            return ""
        
        planning_file = task_tracker['planning_file']
        if not os.path.exists(planning_file):
            return ""
        
        try:
            with open(planning_file, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Extract relevant sections for current step
            guidance = f"\n=== PLANNING REFERENCE ===\n"
            guidance += f"Current Step: {current_step}\n"
            
            # Extract execution plan section
            if "## Execution Plan" in content:
                plan_start = content.find("## Execution Plan")
                plan_end = content.find("##", plan_start + 1)
                if plan_end == -1:
                    plan_section = content[plan_start:]
                else:
                    plan_section = content[plan_start:plan_end]
                guidance += f"\nExecution Plan:\n{plan_section}\n"
            
            # Extract expected outcomes
            if "## Expected Outcomes" in content:
                outcomes_start = content.find("## Expected Outcomes")
                outcomes_end = content.find("##", outcomes_start + 1)
                if outcomes_end == -1:
                    outcomes_section = content[outcomes_start:]
                else:
                    outcomes_section = content[outcomes_start:outcomes_end]
                guidance += f"\nExpected Outcomes:\n{outcomes_section}\n"
            
            guidance += "=== END PLANNING REFERENCE ===\n"
            return guidance
            
        except Exception as e:
            print(f"[PLANNING REFERENCE ERROR] Failed to read planning file: {e}")
            return ""
    
    def _get_next_task(self, context: Dict) -> str:
        """Get the next task from the task file."""
        task_tracker = context.get('task_tracker')
        if not task_tracker or not task_tracker.get('task_file'):
            return "Continue with execution"
        
        task_file = task_tracker['task_file']
        if not os.path.exists(task_file):
            return "Continue with execution"
        
        try:
            with open(task_file, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Find the first uncompleted task
            lines = content.split('\n')
            for line in lines:
                if '- [ ]' in line:
                    # Extract task description
                    task = line.replace('- [ ]', '').strip()
                    return task
            
            return "All tasks completed"
            
        except Exception as e:
            print(f"[NEXT TASK ERROR] Failed to get next task: {e}")
            return "Continue with execution"
    
    def _update_planning_status(self, context: Dict, status: str, details: str = ""):
        """Update the planning file with current execution status."""
        task_tracker = context.get('task_tracker')
        if not task_tracker or not task_tracker.get('planning_file'):
            return
        
        planning_file = task_tracker['planning_file']
        if not os.path.exists(planning_file):
            return
        
        try:
            # Read current planning file
            with open(planning_file, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Add execution status section
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            status_update = f"\n\n## Execution Status\n- {timestamp}: {status}"
            if details:
                status_update += f"\n  Details: {details}"
            
            # Check if execution status section exists
            if "## Execution Status" not in content:
                content += status_update
            else:
                # Add to existing status section
                lines = content.split('\n')
                for i, line in enumerate(lines):
                    if "## Execution Status" in line:
                        lines.insert(i + 1, f"- {timestamp}: {status}")
                        if details:
                            lines.insert(i + 2, f"  Details: {details}")
                        break
                content = '\n'.join(lines)
            
            # Write updated content
            with open(planning_file, 'w', encoding='utf-8') as f:
                f.write(content)
            
            print(f"[PLANNING UPDATE] {status}")
            
        except Exception as e:
            print(f"[PLANNING UPDATE ERROR] Failed to update planning status: {e}")
    
    def analyze_task_complexity(self, task: str, execution_result: ExecutionResult = None) -> bool:
        """
        Analyze if a task is complex enough to warrant blueprint creation.
        
        Args:
            task: The original task/query
            execution_result: Result of task execution (optional)
            
        Returns:
            bool: True if task should be considered for blueprint creation
        """
        # Primary complexity indicators
        complexity_indicators = [
            "analyze entire", "complete audit", "process all files",
            "generate comprehensive", "full analysis", "end-to-end",
            "workflow", "pipeline", "automated", "batch process",
            "systematic", "thorough", "detailed analysis",
            "multi-step", "sequential", "orchestrate",
            "integrate multiple", "combine tools", "chain operations"
        ]
        
        # Check task description
        task_lower = task.lower()
        has_complexity_indicator = any(indicator in task_lower for indicator in complexity_indicators)
        
        # Additional checks based on execution result
        execution_complexity = False
        if execution_result:
            # Check if multiple tools were used
            tools_used = getattr(execution_result, 'tools_used', [])
            if len(tools_used) >= 2:
                execution_complexity = True
            
            # Check execution time (longer tasks are more complex)
            execution_time = getattr(execution_result, 'execution_time', 0)
            if execution_time > 10:  # Tasks taking more than 10 seconds
                execution_complexity = True
            
            # Check if iterative execution was used
            strategy = getattr(execution_result, 'strategy', None)
            if strategy in ['iterative', 'sequential', 'parallel']:
                execution_complexity = True
        
        return has_complexity_indicator or execution_complexity
    
    def detect_blueprint_opportunity(self, task: str, execution_result: ExecutionResult, context: Dict) -> Optional[Dict]:
        """
        Detect if a successful task execution could be turned into a blueprint.
        
        Args:
            task: Original task description
            execution_result: Successful execution result
            context: Execution context with tools and metadata
            
        Returns:
            Dict with blueprint opportunity details or None
        """
        # Only consider successful executions
        if execution_result.error:
            return None
        
        # Check if task is complex enough
        if not self.analyze_task_complexity(task, execution_result):
            return None
        
        # Extract blueprint opportunity details
        opportunity = {
            'task': task,
            'execution_time': execution_result.execution_time,
            'tools_used': getattr(execution_result, 'tools_used', []),
            'strategy': getattr(execution_result, 'strategy', 'unknown'),
            'success_indicators': [],
            'blueprint_type': self._classify_blueprint_type(task),
            'reusability_score': self._calculate_reusability_score(task, execution_result)
        }
        
        # Add success indicators
        if execution_result.execution_time > 5:
            opportunity['success_indicators'].append('time_consuming')
        if len(opportunity['tools_used']) >= 2:
            opportunity['success_indicators'].append('multi_tool')
        if 'comprehensive' in task.lower() or 'complete' in task.lower():
            opportunity['success_indicators'].append('comprehensive_task')
        
        # Only suggest blueprint if reusability score is high enough
        if opportunity['reusability_score'] >= 0.5:
            return opportunity
        
        return None
    
    def _classify_blueprint_type(self, task: str) -> str:
        """
        Classify the type of blueprint based on task description.
        
        Args:
            task: Task description
            
        Returns:
            str: Blueprint type category
        """
        task_lower = task.lower()
        
        # Security-related tasks
        if any(keyword in task_lower for keyword in ['security', 'audit', 'vulnerability', 'scan']):
            return 'security_audit'
        
        # Documentation tasks
        if any(keyword in task_lower for keyword in ['document', 'readme', 'docs', 'documentation']):
            return 'documentation_generator'
        
        # Code quality tasks
        if any(keyword in task_lower for keyword in ['quality', 'lint', 'format', 'style', 'review']):
            return 'code_quality_pipeline'
        
        # Project setup tasks
        if any(keyword in task_lower for keyword in ['setup', 'scaffold', 'initialize', 'create project']):
            return 'project_setup'
        
        # Analysis tasks
        if any(keyword in task_lower for keyword in ['analyze', 'analysis', 'examine', 'investigate']):
            return 'analysis_pipeline'
        
        # File processing tasks
        if any(keyword in task_lower for keyword in ['process files', 'batch', 'transform', 'convert']):
            return 'file_processing'
        
        # Default to general workflow
        return 'general_workflow'
    
    def _calculate_reusability_score(self, task: str, execution_result: ExecutionResult) -> float:
        """
        Calculate how reusable this task pattern might be (0.0 to 1.0).
        
        Args:
            task: Task description
            execution_result: Execution result
            
        Returns:
            float: Reusability score
        """
        score = 0.0
        
        # Base score for successful multi-tool execution
        tools_used = getattr(execution_result, 'tools_used', [])
        if len(tools_used) >= 2:
            score += 0.3
        elif len(tools_used) == 1:
            score += 0.1
        
        # Higher score for longer execution times (more complex tasks)
        execution_time = execution_result.execution_time
        if execution_time > 30:
            score += 0.3
        elif execution_time > 10:
            score += 0.2
        elif execution_time > 5:
            score += 0.1
        
        # Score based on task generalizability
        task_lower = task.lower()
        generalizable_patterns = [
            'analyze', 'process', 'generate', 'create', 'build',
            'audit', 'review', 'validate', 'transform', 'optimize'
        ]
        
        for pattern in generalizable_patterns:
            if pattern in task_lower:
                score += 0.1
                break
        
        # Bonus for common workflow patterns
        workflow_patterns = [
            'pipeline', 'workflow', 'automated', 'systematic',
            'comprehensive', 'end-to-end', 'complete'
        ]
        
        for pattern in workflow_patterns:
            if pattern in task_lower:
                score += 0.2
                break
        
        return min(score, 1.0)  # Cap at 1.0
    
    def _validate_tool_requirements(self, tool_name: str, tool_instance, context: Dict) -> tuple:
        """Validate if tool can actually execute successfully"""
        
        # Check API key requirements
        api_keys = self._get_api_keys(context.get('config'))
        
        if tool_name == 'GoogleSearchTool':
            if not api_keys.get('google_api_key') or not api_keys.get('google_cx'):
                return False, "Missing Google Search API credentials (google_api_key and google_cx required)"
        
        elif tool_name == 'FirecrawlTool':
            if not api_keys.get('firecrawl'):
                return False, "Missing Firecrawl API key"
        
        elif tool_name == 'E2BCodeSandboxTool':
            if not api_keys.get('e2b'):
                return False, "Missing E2B API key"
        
        elif tool_name == 'AdvancedMathTool':
            # Check if math operations are actually needed
            pass  # Usually doesn't require external APIs
        
        # Check tool-specific requirements
        if hasattr(tool_instance, 'validate_requirements'):
            try:
                return tool_instance.validate_requirements(context)
            except Exception as e:
                return False, f"Tool validation failed: {e}"
        
        return True, "Tool validated successfully"
    
    def _execute_tool_with_validation(self, tool_name: str, tool, query: str, context: Dict) -> tuple:
        """Execute tool with comprehensive validation and error handling"""
        
        # Pre-validation
        is_valid, validation_msg = self._validate_tool_requirements(tool_name, tool, context)
        if not is_valid:
            return f"[TOOL UNAVAILABLE] {tool_name}: {validation_msg}", False
        
        try:
            # Tool execution
            api_keys = self._get_api_keys(context.get('config'))
            tool_params = self._extract_tool_parameters(tool_name, query, context)
            all_params = {**api_keys, **tool_params}
            
            result = tool.execute(query, **all_params)
            
            # Success validation
            if self._is_tool_result_valid(result):
                return result, True
            else:
                return f"[INCOMPLETE] {tool_name} returned incomplete or empty results", False
                
        except Exception as e:
            error_msg = self._format_tool_error(tool_name, e)
            return error_msg, False
    
    def _is_tool_result_valid(self, result) -> bool:
        """Check if tool result is valid and useful"""
        if not result:
            return False
        
        result_str = str(result).lower()
        
        # Check for common failure indicators
        failure_indicators = [
            'error:', 'failed:', 'not found', 'unable to', 'could not',
            'timeout', 'invalid', 'unauthorized', 'forbidden'
        ]
        
        if any(indicator in result_str for indicator in failure_indicators):
            return False
        
        # Check for minimum content length (reduced threshold)
        if len(str(result).strip()) < 5:
            return False
        
        # For file operations, check for success indicators
        success_indicators = [
            'success', 'created', 'written', 'updated', 'modified', 'completed'
        ]
        if any(indicator in result_str for indicator in success_indicators):
            return True
        
        # If result is substantial (more than 20 chars), consider it valid
        if len(str(result).strip()) > 20:
            return True
        
        return True  # Default to valid for short results that don't contain failures
    
    def _format_tool_error(self, tool_name: str, error: Exception) -> str:
        """Format tool errors for user-friendly display"""
        
        error_str = str(error).lower()
        
        # Common error patterns without emojis
        if "api key" in error_str or "unauthorized" in error_str:
            return f"[API KEY ERROR] {tool_name} requires proper API key configuration"
        
        elif "not found" in error_str or "404" in error_str:
            return f"[NOT FOUND] {tool_name} could not find the requested resource"
        
        elif "timeout" in error_str:
            return f"[TIMEOUT] {tool_name} operation timed out - try again later"
        
        elif "rate limit" in error_str or "too many requests" in error_str:
            return f"[RATE LIMIT] {tool_name} rate limit exceeded - please wait before retrying"
        
        elif "connection" in error_str or "network" in error_str:
            return f"[CONNECTION ERROR] {tool_name} unable to connect to external service"
        
        else:
            # Truncate very long error messages
            error_preview = str(error)[:100] + ("..." if len(str(error)) > 100 else "")
            return f"[ERROR] {tool_name}: {error_preview}"
    
    def _has_sufficient_information(self, query: str, results: Dict) -> bool:
        """Determine if we have enough information to answer the query"""
        
        # For search queries, one good result may be sufficient
        if 'GoogleSearchTool' in results:
            search_result = str(results['GoogleSearchTool'])
            if len(search_result) > 200 and not any(indicator in search_result.lower() for indicator in ['error', 'failed', 'not found']):
                return True
        
        # For file operations, one successful operation is usually enough
        if any(tool in results for tool in ['WriteTool', 'EditTool', 'ReadTool']):
            for tool_name in ['WriteTool', 'EditTool', 'ReadTool']:
                if tool_name in results:
                    result = str(results[tool_name])
                    if len(result) > 50 and 'error' not in result.lower():
                        return True
        
        # For code generation, check if we have substantial code
        if 'CodingTool' in results:
            code_result = str(results['CodingTool'])
            if len(code_result) > 100 and ('def ' in code_result or 'function' in code_result or 'class ' in code_result):
                return True
        
        return False
    
    def suggest_blueprint_creation(self, opportunity: Dict, context: Dict) -> str:
        """
        Generate a suggestion message for blueprint creation.
        
        Args:
            opportunity: Blueprint opportunity details
            context: Execution context
            
        Returns:
            str: Suggestion message
        """
        blueprint_type = opportunity['blueprint_type']
        reusability_score = opportunity['reusability_score']
        tools_used = opportunity['tools_used']
        
        suggestion = f"\n\n=== BLUEPRINT OPPORTUNITY DETECTED ===\n"
        suggestion += f"Task: {opportunity['task']}\n"
        suggestion += f"Type: {blueprint_type.replace('_', ' ').title()}\n"
        suggestion += f"Reusability Score: {reusability_score:.1%}\n"
        suggestion += f"Tools Used: {', '.join(tools_used)}\n"
        suggestion += f"Execution Time: {opportunity['execution_time']:.1f}s\n"
        
        suggestion += "\nThis successful task execution could be turned into a reusable blueprint!\n"
        suggestion += "Benefits:\n"
        suggestion += "- Automate similar tasks in the future\n"
        suggestion += "- Share workflow with team members\n"
        suggestion += "- Ensure consistent execution\n"
        suggestion += "- Save time on repetitive tasks\n"
        
        # Add specific suggestions based on blueprint type
        type_suggestions = {
            'security_audit': 'Perfect for regular security assessments and compliance checks',
            'documentation_generator': 'Ideal for maintaining up-to-date project documentation',
            'code_quality_pipeline': 'Great for enforcing coding standards across projects',
            'project_setup': 'Excellent for standardizing new project initialization',
            'analysis_pipeline': 'Useful for systematic data or code analysis tasks',
            'file_processing': 'Perfect for batch file operations and transformations'
        }
        
        if blueprint_type in type_suggestions:
            suggestion += f"\nUse Case: {type_suggestions[blueprint_type]}\n"
        
        suggestion += "\nTo create a blueprint from this execution, run:\n"
        suggestion += f"python -m metis_agent.blueprints create-from-execution --type {blueprint_type}\n"
        suggestion += "==========================================\n"
        
        return suggestion
    
    def create_blueprint_from_execution(self, opportunity: Dict, context: Dict, save_path: str = None) -> Blueprint:
        """
        Create a blueprint from a successful task execution.
        
        Args:
            opportunity: Blueprint opportunity details
            context: Execution context
            save_path: Optional path to save the blueprint
            
        Returns:
            Blueprint: Created blueprint object
        """
        blueprint_type = opportunity['blueprint_type']
        task = opportunity['task']
        tools_used = opportunity['tools_used']
        
        # Generate blueprint name and metadata
        blueprint_name = self._generate_blueprint_name(blueprint_type, task)
        
        metadata = BlueprintMetadata(
            name=blueprint_name,
            version="1.0.0",
            description=f"Automated workflow for: {task}",
            author="Metis Agent (Auto-generated)",
            tags=[blueprint_type.replace('_', '-'), 'auto-generated', 'workflow'],
            category=blueprint_type.replace('_', '-')
        )
        
        # Generate inputs based on task analysis
        inputs = self._generate_blueprint_inputs(task, tools_used)
        
        # Generate outputs
        outputs = self._generate_blueprint_outputs(task, blueprint_type)
        
        # Generate steps based on tools used
        steps = self._generate_blueprint_steps(tools_used, task)
        
        # Create blueprint data structure
        blueprint_data = {
            'name': metadata.name,
            'version': metadata.version,
            'description': metadata.description,
            'author': metadata.author,
            'tags': metadata.tags,
            'category': metadata.category,
            'inputs': [{
                'name': inp.name,
                'type': inp.type,
                'required': inp.required,
                'description': inp.description,
                'default': getattr(inp, 'default', None),
                'validation': getattr(inp, 'validation', None)
            } for inp in inputs],
            'outputs': [{
                'name': out.name,
                'type': out.type,
                'description': out.description,
                'source': out.source
            } for out in outputs],
            'steps': [{
                'id': step.id,
                'tool': step.tool,
                'action': step.action,
                'inputs': step.inputs,
                'depends_on': step.depends_on
            } for step in steps]
        }
        
        # Create blueprint
        blueprint = Blueprint(blueprint_data)
        
        # Save blueprint if path provided
        if save_path:
            blueprint.save(save_path)
            print(f"[BLUEPRINT CREATED] Saved to: {save_path}")
        
        return blueprint
    
    def _generate_blueprint_name(self, blueprint_type: str, task: str) -> str:
        """
        Generate a descriptive name for the blueprint.
        
        Args:
            blueprint_type: Type of blueprint
            task: Original task description
            
        Returns:
            str: Blueprint name
        """
        # Extract key terms from task
        task_words = task.lower().split()
        key_terms = []
        
        important_words = [
            'analyze', 'process', 'generate', 'create', 'audit',
            'review', 'validate', 'transform', 'optimize', 'scan'
        ]
        
        for word in task_words:
            if word in important_words:
                key_terms.append(word.title())
        
        if key_terms:
            return f"{' '.join(key_terms)} {blueprint_type.replace('_', ' ').title()}"
        else:
            return f"{blueprint_type.replace('_', ' ').title()} Workflow"
    
    def _generate_blueprint_inputs(self, task: str, tools_used: List[str]) -> List[BlueprintInput]:
        """
        Generate blueprint inputs based on task and tools used.
        
        Args:
            task: Original task description
            tools_used: List of tools that were used
            
        Returns:
            List[BlueprintInput]: Generated inputs
        """
        inputs = []
        
        # Common inputs based on task type
        task_lower = task.lower()
        
        # File/directory inputs
        if any(keyword in task_lower for keyword in ['file', 'directory', 'folder', 'path']):
            inputs.append(BlueprintInput(
                name="target_path",
                type="string",
                required=True,
                description="Path to the target file or directory",
                validation="path_exists"
            ))
        
        # Pattern/search inputs
        if any(keyword in task_lower for keyword in ['search', 'pattern', 'find', 'grep']):
            inputs.append(BlueprintInput(
                name="search_pattern",
                type="string",
                required=True,
                description="Pattern or text to search for"
            ))
        
        # Output format inputs
        if any(keyword in task_lower for keyword in ['generate', 'create', 'output', 'report']):
            inputs.append(BlueprintInput(
                name="output_format",
                type="string",
                required=False,
                default="markdown",
                description="Output format (markdown, json, yaml)",
                validation="format_type"
            ))
        
        # Tool-specific inputs
        if 'GoogleSearchTool' in tools_used:
            inputs.append(BlueprintInput(
                name="search_query",
                type="string",
                required=True,
                description="Search query for web search"
            ))
        
        # Default input if none generated
        if not inputs:
            inputs.append(BlueprintInput(
                name="input_data",
                type="string",
                required=True,
                description="Input data for the workflow"
            ))
        
        return inputs
    
    def _generate_blueprint_outputs(self, task: str, blueprint_type: str) -> List[BlueprintOutput]:
        """
        Generate blueprint outputs based on task and type.
        
        Args:
            task: Original task description
            blueprint_type: Type of blueprint
            
        Returns:
            List[BlueprintOutput]: Generated outputs
        """
        outputs = []
        
        # Type-specific outputs
        if blueprint_type == 'security_audit':
            outputs.append(BlueprintOutput(
                name="security_report",
                type="object",
                description="Comprehensive security audit report",
                source="generate_security_report.output"
            ))
        elif blueprint_type == 'documentation_generator':
            outputs.append(BlueprintOutput(
                name="documentation",
                type="string",
                description="Generated documentation content",
                source="generate_docs.output"
            ))
        elif blueprint_type == 'code_quality_pipeline':
            outputs.append(BlueprintOutput(
                name="quality_report",
                type="object",
                description="Code quality assessment report",
                source="quality_check.output"
            ))
        else:
            # Generic output
            outputs.append(BlueprintOutput(
                name="workflow_result",
                type="object",
                description="Result of the workflow execution",
                source="final_step.output"
            ))
        
        return outputs
    
    def _generate_blueprint_steps(self, tools_used: List[str], task: str) -> List[BlueprintStep]:
        """
        Generate blueprint steps based on tools used.
        
        Args:
            tools_used: List of tools that were used
            task: Original task description
            
        Returns:
            List[BlueprintStep]: Generated steps
        """
        steps = []
        
        # Generate steps based on tools used
        for i, tool_name in enumerate(tools_used):
            step_id = f"step_{i+1}_{tool_name.lower().replace('tool', '')}"
            
            # Tool-specific step generation
            if tool_name == 'FileSystemTool':
                steps.append(BlueprintStep(
                    id=step_id,
                    tool=tool_name,
                    action="file_operation",
                    inputs={
                        "operation": "${inputs.operation}",
                        "path": "${inputs.target_path}"
                    }
                ))
            elif tool_name == 'GoogleSearchTool':
                steps.append(BlueprintStep(
                    id=step_id,
                    tool=tool_name,
                    action="web_search",
                    inputs={
                        "query": "${inputs.search_query}"
                    }
                ))
            elif tool_name == 'CodingTool':
                steps.append(BlueprintStep(
                    id=step_id,
                    tool=tool_name,
                    action="code_analysis",
                    inputs={
                        "code": "${inputs.code_input}",
                        "operation": "analyze"
                    }
                ))
            else:
                # Generic step
                steps.append(BlueprintStep(
                    id=step_id,
                    tool=tool_name,
                    action="execute",
                    inputs={
                        "input": "${inputs.input_data}"
                    }
                ))
        
        # Add dependencies for sequential execution
        for i in range(1, len(steps)):
            steps[i].depends_on = [steps[i-1].id]
        
        return steps
    
    def _extract_tool_parameters(self, tool_name: str, query: str, context: Dict) -> Dict[str, Any]:
        """
        Extract tool-specific parameters from natural language query.
        
        Args:
            tool_name: Name of the tool to extract parameters for
            query: Natural language query
            context: Execution context
            
        Returns:
            Dictionary of tool-specific parameters
        """
        import re
        
        params = {}
        query_lower = query.lower()
        
        if tool_name == 'EditTool':
            # Extract file path
            file_patterns = [
                r'edit\s+([\w\.]+)',
                r'modify\s+([\w\.]+)',
                r'update\s+([\w\.]+)',
                r'change\s+([\w\.]+)',
                r'in\s+([\w\.]+)',
                r'file\s+([\w\.]+)',
                r'([\w\.]+\.py)',
                r'([\w\.]+\.js)',
                r'([\w\.]+\.html)',
                r'([\w\.]+\.css)',
                r'([\w\.]+\.md)',
                r'([\w\.]+\.txt)'
            ]
            
            file_path = None
            for pattern in file_patterns:
                match = re.search(pattern, query_lower)
                if match:
                    file_path = match.group(1)
                    break
            
            if file_path:
                params['file_path'] = file_path
            
            # Determine operation type
            if any(word in query_lower for word in ['add', 'append', 'insert']):
                params['operation'] = 'append'
            elif any(word in query_lower for word in ['replace', 'change', 'modify']):
                params['operation'] = 'replace'
            elif any(word in query_lower for word in ['delete', 'remove']):
                params['operation'] = 'delete'
            else:
                params['operation'] = 'append'  # Default
            
            # Generate content using agent for code additions
            if params.get('operation') == 'append':
                # Always generate proper code content using the LLM
                import os
                file_path = params.get('file_path')
                if file_path and os.path.exists(file_path):
                    # Read the current file content
                    try:
                        with open(file_path, 'r', encoding='utf-8') as f:
                            context['file_content'] = f.read()
                    except:
                        context['file_content'] = ''
                
                params['content'] = self._generate_code_content(query, context)
            
        elif tool_name == 'WriteTool':
            # Extract file path for new files
            file_patterns = [
                r'create\s+([\w\.]+)',
                r'new\s+file\s+([\w\.]+)',
                r'file\s+called\s+([\w\.]+)',
                r'writetool\s+to\s+create\s+([\w\.]+)',
                r'use\s+writetool\s+.*?([\w\.]+\.py)',
                r'([\w\.]+\.py)',
                r'([\w\.]+\.js)',
                r'([\w\.]+\.html)',
                r'([\w\.]+\.css)',
                r'([\w\.]+\.md)',
                r'([\w\.]+\.txt)'
            ]
            
            for pattern in file_patterns:
                match = re.search(pattern, query_lower)
                if match:
                    params['file_path'] = match.group(1)
                    break
            
            # Generate content for common requests
            if 'utils' in query_lower and 'helper' in query_lower:
                params['content'] = """# Utility functions\n\ndef format_string(text, capitalize=True):\n    \"\"\"Format a string with optional capitalization.\"\"\"\n    if capitalize:\n        return text.strip().title()\n    return text.strip()\n\ndef safe_divide(a, b):\n    \"\"\"Safely divide two numbers, returning 0 if division by zero.\"\"\"\n    try:\n        return a / b\n    except ZeroDivisionError:\n        return 0\n\ndef list_to_string(items, separator=', '):\n    \"\"\"Convert a list to a formatted string.\"\"\"\n    return separator.join(str(item) for item in items)\n\ndef is_valid_email(email):\n    \"\"\"Basic email validation.\"\"\"\n    import re\n    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,}$'\n    return re.match(pattern, email) is not None"""
            elif 'function' in query_lower and 'circle' in query_lower and 'area' in query_lower:
                params['content'] = """import math\n\ndef calculate_circle_area(radius):\n    \"\"\"Calculate the area of a circle given its radius.\"\"\"\n    if radius < 0:\n        raise ValueError(\"Radius cannot be negative\")\n    return math.pi * radius * radius\n\nif __name__ == \"__main__\":\n    # Test the function\n    radius = 5\n    area = calculate_circle_area(radius)\n    print(f\"Circle with radius {radius} has area {area:.2f}\")"""
        
        elif tool_name == 'FileSystemTool':
            # Extract path and operation
            if 'read' in query_lower or 'view' in query_lower:
                params['operation'] = 'read'
            elif 'list' in query_lower or 'show' in query_lower:
                params['operation'] = 'list'
            elif 'search' in query_lower:
                params['operation'] = 'search'
            
        elif tool_name == 'ProjectManagementTool':
            # Extract project details
            if 'create' in query_lower and 'project' in query_lower:
                params['operation'] = 'create_project'
                # Extract project name if specified
                name_match = re.search(r'project\s+([\w\-]+)', query_lower)
                if name_match:
                    params['project_name'] = name_match.group(1)
        
        return params
    
    def _generate_code_content(self, query: str, context: Dict) -> str:
        """Generate code content using LLM for edit operations."""
        from ..llm.factory import get_llm
        
        # Read current file content if available
        file_content = context.get('file_content', '')
        
        code_generation_prompt = f"""You are a code generator. Generate only the code that should be added to a file based on this request: "{query}"

Current file content:
```
{file_content}
```

Requirements:
1. Generate ONLY the new code to be added, not the entire file
2. Include proper Python function definitions with docstrings
3. Add appropriate error handling where needed
4. Make the code consistent with existing style
5. Do not include explanations, just the code

Return only the code that should be appended to the file:"""

        llm = get_llm()
        messages = [
            {"role": "system", "content": "You are a code generator that produces clean, functional Python code."},
            {"role": "user", "content": code_generation_prompt}
        ]
        
        try:
            generated_code = llm.chat(messages)
            # Clean up the response - remove code block markers if present
            import re
            code_pattern = r'```(?:python)?\n?(.*?)```'
            match = re.search(code_pattern, generated_code, re.DOTALL)
            if match:
                return '\n' + match.group(1).strip()
            return '\n' + generated_code.strip()
        except Exception as e:
            # Fallback for common cases
            if 'multiply' in query.lower() and 'divide' in query.lower():
                return '''
def multiply(a, b):
    """Multiply two numbers"""
    return a * b

def divide(a, b):
    """Divide first number by second"""
    if b == 0:
        raise ValueError("Cannot divide by zero")
    return a / b'''
            return f'\n# TODO: Implement - {query}'
