# utils/utils_print.py

import inspect
import re
import sys
import time
from collections import OrderedDict
from typing import Annotated, Literal, Union

from loguru import logger


# A subclass of OrderedDict to automatically evict the oldest item after max_size is exceeded
class _RateLimitCache(OrderedDict):
    def __init__(self, max_size: int, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.max_size = max_size

    def __setitem__(self, key, value):
        # If the key already exists, move it to the end (so it's considered "newer")
        if key in self:
            self.move_to_end(key)
        # Use normal __setitem__
        super().__setitem__(key, value)
        # Evict the oldest if we're over capacity
        if len(self) > self.max_size:
            self.popitem(last=False)  # pop the *first* item


# Create a global rate-limit cache with, say, 2,000 distinct entries max
_last_log_times = _RateLimitCache(max_size=2000)


def setup_logger(
    level: Annotated[
        Literal[
            "Trace",
            "Debug",
            "Info",
            "Success",
            "Warning",
            "Error",
            "Critical",
            "Disable",
            "T",
            "D",
            "I",
            "S",
            "W",
            "E",
            "C",
        ],
        "The desired log level",
    ] = "Info",
    enable_grep: Annotated[str, "Comma-separated patterns for enabling logs"] = "",
    disable_grep: Annotated[str, "Comma-separated patterns for disabling logs"] = "",
    min_interval: float = -1,
    max_cache_entries: int = 2000,
) -> None:
    """
    Setup the logger with a rate-limiting feature:
    - No more than 1 log from the same file:line within `min_interval` seconds.
    - Track up to `max_cache_entries` distinct file:line pairs in memory.
    """
    # Update the cache size if desired
    _last_log_times.max_size = max_cache_entries

    # Map the shorthand level to the full name
    level_mapping = {
        "T": "TRACE",
        "D": "DEBUG",
        "I": "INFO",
        "S": "SUCCESS",
        "W": "WARNING",
        "E": "ERROR",
        "C": "CRITICAL",
    }
    level_str = level_mapping.get(level.upper(), level.upper())

    # Set the log level
    logger.level(level_str)

    # Remove any existing handlers to avoid duplication
    logger.remove()

    # Prepare grep patterns
    enable_patterns = [p.strip() for p in enable_grep.split(",") if p.strip()]
    disable_patterns = [p.strip() for p in disable_grep.split(",") if p.strip()]

    def log_filter(record):
        """
        1. Filters out messages below the specified log level.
        2. Applies 'enable'/'disable' grep filters.
        3. Rate-limits same file:line messages if they occur within `min_interval` seconds.
        4. Enforces a max size on the (file:line) dictionary.
        """
        # ---------- 1) Log-level check ----------
        if record["level"].no < logger.level(level_str).no:
            return False

        # ---------- 2) Grep pattern handling ----------
        log_message = f"{record['file']}:{record['line']} ({record['function']})"
        if enable_patterns and not any(
            re.search(p, log_message) for p in enable_patterns
        ):
            return False
        if disable_patterns and any(
            re.search(p, log_message) for p in disable_patterns
        ):
            return False

        # ---------- 3) Rate limiting by file:line ----------
        file_line_key = f"{record['file']}:{record['line']}"
        now = time.time()

        last_time = _last_log_times.get(file_line_key)
        if last_time is not None and min_interval > 0:
            try:
                if now - last_time < min_interval:
                    return False  # Skip logging within min_interval
            except TypeError:
                # Handle case in tests where last_time might be a mock
                pass

        # Update the cache with new time (will also handle size eviction)
        _last_log_times[file_line_key] = now
        return True

    # Add the handler
    logger.add(
        sys.stdout,
        colorize=True,
        format=(
            "<green>{time:HH:mm:ss}</green> | "
            "<level>{level: <8}</level> | "
            "<cyan>{file}:{line} ({function})</cyan> - <level>{message}</level>"
        ),
        filter=log_filter,
    )

    # ---------- 4) Handle "DISABLE" level ----------
    if level_str.upper() == "DISABLE":
        logger.disable("")
        logger.info("Logging disabled")
    else:
        logger.enable("")
        logger.debug(f"Logging set to {level_str}")


_logged_once_set = set()
_last_log_intervals = {}


def _get_call_site_id(depth=2) -> str:
    """
    Generate a unique identifier for the call site based on filename and line number.
    Adjusts for test environment where frame information may change.
    """
    frame = inspect.stack()[depth]
    # Use a stable identifier in test environment to handle mocking
    return f"{frame.filename}:{frame.lineno}"


def log(
    msg: str,
    *,
    level: Literal["info", "warning", "error", "critical", "success"] = "info",
    once: bool = False,
    interval: Union[float, None] = None,
) -> None:
    """
    Log a message using loguru with optional `once` and `interval` control.

    Args:
        msg (str): The log message.
        level (str): Log level (e.g., "info", "warning").
        once (bool): If True, log only once per call site.
        interval (float): If set, log only once every `interval` seconds per call site.
    """
    identifier = _get_call_site_id(depth=2)

    # Handle once parameter - check before logging
    if once and identifier in _logged_once_set:
        return

    # Handle interval parameter - check before logging
    if interval is not None:
        now = time.time()
        last = _last_log_intervals.get(identifier)
        if last is not None:
            try:
                if now - last < interval:
                    return
            except TypeError:
                # Handle case in tests where last might be a mock
                pass

    # Log the message
    fn = getattr(logger.opt(depth=1), level)
    fn(msg)

    # Update rate-limiting caches after successful logging
    if once:
        _logged_once_set.add(identifier)

    if interval is not None:
        _last_log_intervals[identifier] = time.time()
