"""
Environment configuration loader for LLM settings.

This module safely loads LLM configuration from environment variables,
with graceful fallbacks if configuration is missing or invalid.
"""

from __future__ import annotations

import logging
import os
from pathlib import Path

from pydocextractor.domain.config import LLMConfig

logger = logging.getLogger(__name__)


def load_llm_config() -> LLMConfig | None:
    """
    Load LLM configuration from environment variables.

    This function attempts to load configuration from:
    1. A .env file in the current directory (if python-dotenv is installed)
    2. System environment variables

    Returns:
        LLMConfig if properly configured and enabled, None otherwise.
        The system will work without LLM features if None is returned.

    Environment Variables:
        LLM_ENABLED: Enable LLM features (default: false)
        LLM_API_URL: API endpoint URL (required if enabled)
        LLM_API_KEY: API key (required if enabled)
        LLM_MODEL_NAME: Model name (default: gpt-4-vision-preview)
        LLM_CONTEXT_LINES: Context lines to include (default: 100)
        LLM_IMAGE_SIZE: Image resize dimension (default: 1024)
        LLM_MAX_IMAGES: Max images per document (default: 5)
        LLM_TIMEOUT: Request timeout in seconds (default: 30)
        LLM_MAX_RETRIES: Max retry attempts (default: 3)
        LLM_PROMPT: Prompt template for image description (default: standard prompt)
    """
    # Try to load environment file if it exists (optional)
    # Check for config.env first, then .env
    try:
        from dotenv import load_dotenv

        # Look for config.env or .env in current directory
        config_env_path = Path.cwd() / "config.env"
        env_path = Path.cwd() / ".env"

        if config_env_path.exists():
            load_dotenv(config_env_path)
            logger.info(f"Loaded config.env from {config_env_path}")
        elif env_path.exists():
            load_dotenv(env_path)
            logger.info(f"Loaded .env from {env_path}")
    except ImportError:
        # python-dotenv not installed, use environment variables directly
        logger.debug("python-dotenv not available, using environment variables")
    except Exception as e:
        logger.debug(f"Could not load environment file: {e}")

    # Check if LLM is enabled (default: false)
    enabled = os.getenv("LLM_ENABLED", "false").lower() in ("true", "1", "yes")

    if not enabled:
        logger.info("LLM image description is disabled (LLM_ENABLED=false or not set)")
        return None

    # Get required fields
    api_url = os.getenv("LLM_API_URL", "").strip()
    api_key = os.getenv("LLM_API_KEY", "").strip()

    # Validate required fields
    if not api_url:
        logger.warning("LLM enabled but LLM_API_URL not set. Disabling LLM features.")
        return None

    if not api_key:
        logger.warning("LLM enabled but LLM_API_KEY not set. Disabling LLM features.")
        return None

    # Get optional fields with defaults
    try:
        # Get prompt template with default
        prompt_template = os.getenv(
            "LLM_PROMPT",
            (
                "Please describe the image in detail, considering the context provided. "
                "Focus on how the image relates to the surrounding content. "
                "Keep the description concise (2-3 sentences)."
            ),
        ).strip()

        config = LLMConfig(
            api_url=api_url,
            api_key=api_key,
            model_name=os.getenv("LLM_MODEL_NAME", "gpt-4-vision-preview"),
            context_lines=int(os.getenv("LLM_CONTEXT_LINES", "100")),
            enabled=True,
            timeout_seconds=int(os.getenv("LLM_TIMEOUT", "30")),
            max_retries=int(os.getenv("LLM_MAX_RETRIES", "3")),
            image_size=int(os.getenv("LLM_IMAGE_SIZE", "1024")),
            max_images_per_document=int(os.getenv("LLM_MAX_IMAGES", "5")),
            prompt_template=prompt_template,
        )

        logger.info(
            f"LLM configuration loaded: model={config.model_name}, "
            f"max_images={config.max_images_per_document}"
        )
        return config

    except ValueError as e:
        logger.error(f"Invalid LLM configuration: {e}. Disabling LLM features.")
        return None
    except Exception as e:
        logger.error(f"Error loading LLM configuration: {e}. Disabling LLM features.")
        return None


def get_llm_status_message(config: LLMConfig | None) -> str:
    """
    Get a human-readable status message for LLM configuration.

    Args:
        config: LLM configuration or None if disabled

    Returns:
        Status message describing LLM state
    """
    if config is None:
        return "LLM image description: Disabled"
    return (
        f"LLM image description: Enabled (max {config.max_images_per_document} images per document)"
    )
