"""
Step definitions for PDF and Word document extraction (Feature 1).
"""

from __future__ import annotations

import re

from pytest_bdd import parsers, then

# ============================================================================
# Markdown Structure Verification
# ============================================================================


@then("the service produces semantically-correct Markdown preserving heading levels (H1..H4)")
def verify_heading_preservation(context):
    """Verify heading levels are preserved."""
    assert context["conversion_success"]
    result = context["conversion_result"]
    markdown = result.text

    # Check for heading markers
    heading_pattern = r"^#{1,4}\s+.+"
    headings = re.findall(heading_pattern, markdown, re.MULTILINE)

    # Should have at least some headings for a document
    assert len(headings) > 0, "Document should contain headings"


@then("lists are converted to proper Markdown bullets or numbers")
def verify_list_conversion(context):
    """Verify list formatting."""
    result = context["conversion_result"]
    markdown = result.text

    # Check for bullet lists (-, *, +) or numbered lists (1., 2., etc.)
    has_bullets = bool(re.search(r"^[\-\*\+]\s+.+", markdown, re.MULTILINE))
    has_numbers = bool(re.search(r"^\d+\.\s+.+", markdown, re.MULTILINE))

    # Should have at least one type of list
    # (Note: this is a loose check; real docs might not have lists)
    context["has_lists"] = has_bullets or has_numbers


@then("inline links are preserved as [text](url)")
def verify_inline_links(context):
    """Verify inline link format."""
    result = context["conversion_result"]
    markdown = result.text

    # Check for Markdown link format
    link_pattern = r"\[.+?\]\(.+?\)"
    links = re.findall(link_pattern, markdown)

    # Mark that we checked for links (may or may not exist)
    context["links_checked"] = True
    context["link_count"] = len(links)


@then("images (if any) are extracted or referenced with alt text and stored URIs")
def verify_image_references(context):
    """Verify image references."""
    result = context["conversion_result"]
    markdown = result.text

    # Check for image syntax: ![alt](url)
    image_pattern = r"!\[.*?\]\(.+?\)"
    images = re.findall(image_pattern, markdown)

    context["image_count"] = len(images)


# ============================================================================
# Metadata Verification
# ============================================================================


@then(
    "the service includes a front-matter or metadata block with filename, bytes, mime, content hash, and storage location"
)
def verify_metadata_block(context):
    """Verify metadata is included."""
    result = context["conversion_result"]

    # Check result metadata
    assert result.metadata is not None
    metadata = result.metadata

    # Verify key metadata fields exist
    # (In real implementation, these would be populated by the service)
    context["metadata_verified"] = True


@then("the metadata block records filename, bytes, mime, content hash, storage, and content ID")
def verify_complete_metadata(context):
    """Verify complete metadata block."""
    verify_metadata_block(context)


@then("the raw file CID and Markdown hash are recorded")
def verify_cid_and_hash(context):
    """Verify CID and hash recording."""
    # Simulate CID and hash generation
    context["file_cid"] = f"Qm{context.get('project_slug', 'default')}ABC123"
    context["markdown_hash"] = "sha256:abcd1234..."


@then(
    parsers.parse('a webhook payload includes "document_type":"{doc_type}" and "status":"{status}"')
)
def verify_webhook_payload(doc_type: str, status: str, context):
    """Verify webhook payload contents."""
    webhook_calls = context.get("webhook_calls", [])
    assert len(webhook_calls) > 0

    last_call = webhook_calls[-1]
    assert last_call["status"] == status
    assert last_call.get("document_type") == doc_type


# ============================================================================
# Formatting and Normalization
# ============================================================================


@then("smart quotes are normalized to plain quotes unless within code spans")
def verify_smart_quote_normalization(context):
    """Verify smart quote handling."""
    result = context["conversion_result"]
    markdown = result.text

    # Check that smart quotes are handled
    # (This is a placeholder - real implementation would check actual normalization)
    context["smart_quotes_normalized"] = True


@then('em-dashes are preserved as "—" or converted to "--" according to normalization rules')
def verify_em_dash_handling(context):
    """Verify em-dash handling."""
    result = context["conversion_result"]
    markdown = result.text

    # Check em-dash presence or conversion
    has_em_dash = "—" in markdown or "--" in markdown
    context["em_dash_handled"] = True


@then("multiple blank lines are collapsed to a single blank line")
def verify_blank_line_collapsing(context):
    """Verify blank line normalization."""
    result = context["conversion_result"]
    markdown = result.text

    # Check that there are no excessive blank lines (3+ consecutive)
    excessive_blanks = re.search(r"\n\n\n+", markdown)
    assert excessive_blanks is None or len(excessive_blanks.group()) < 4, (
        "Should not have excessive blank lines"
    )


@then("non-breaking spaces are removed except where required in tables")
def verify_nbsp_handling(context):
    """Verify non-breaking space handling."""
    result = context["conversion_result"]
    markdown = result.text

    # Non-breaking spaces (U+00A0) should be minimal
    nbsp_count = markdown.count("\u00a0")
    context["nbsp_count"] = nbsp_count


# ============================================================================
# Error Handling
# ============================================================================


@then(
    parsers.parse('the webhook (if configured) is fired with status "failed" and reason "{reason}"')
)
def verify_failure_webhook(reason: str, context):
    """Verify failure webhook."""
    webhook_calls = context.get("webhook_calls", [])

    if context.get("webhooks_enabled"):
        assert len(webhook_calls) > 0
        last_call = webhook_calls[-1]
        assert last_call["status"] == "failed"
        context["failure_reason"] = reason


@then("no Markdown artifact is persisted")
def verify_no_artifact(context):
    """Verify no artifact was created on failure."""
    assert not context["conversion_success"]
    assert context["conversion_result"] is None
