"""
Unit tests for environment configuration loader.

Tests load_llm_config with various environment configurations.
"""

from unittest.mock import patch

from pydocextractor.domain.config import LLMConfig
from pydocextractor.infra.config.env_loader import get_llm_status_message, load_llm_config


class TestLoadLLMConfig:
    """Test load_llm_config function."""

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_disabled_config(self, mock_getenv):
        """Test loading when LLM_ENABLED is false."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "false",
        }.get(key, default)

        config = load_llm_config()

        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_not_enabled(self, mock_getenv):
        """Test loading when LLM_ENABLED not set."""
        mock_getenv.side_effect = lambda key, default="": default

        config = load_llm_config()

        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_enabled_missing_url(self, mock_getenv):
        """Test loading when enabled but URL missing."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_KEY": "test-key",
        }.get(key, default)

        config = load_llm_config()

        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_enabled_missing_key(self, mock_getenv):
        """Test loading when enabled but API key missing."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "https://api.test.com",
        }.get(key, default)

        config = load_llm_config()

        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_valid_minimal_config(self, mock_getenv):
        """Test loading minimal valid configuration."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "https://api.openai.com/v1/chat/completions",
            "LLM_API_KEY": "sk-test123",
        }.get(key, default)

        config = load_llm_config()

        assert config is not None
        assert isinstance(config, LLMConfig)
        assert config.enabled is True
        assert config.api_url == "https://api.openai.com/v1/chat/completions"
        assert config.api_key == "sk-test123"
        assert config.model_name == "gpt-4-vision-preview"  # default

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_full_custom_config(self, mock_getenv):
        """Test loading configuration with all custom values."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "http://localhost:8000",
            "LLM_API_KEY": "local-key",
            "LLM_MODEL_NAME": "llava:13b",
            "LLM_CONTEXT_LINES": "200",
            "LLM_TIMEOUT": "60",
            "LLM_MAX_RETRIES": "5",
            "LLM_IMAGE_SIZE": "512",
            "LLM_MAX_IMAGES": "10",
        }.get(key, default)

        config = load_llm_config()

        assert config is not None
        assert config.api_url == "http://localhost:8000"
        assert config.api_key == "local-key"
        assert config.model_name == "llava:13b"
        assert config.context_lines == 200
        assert config.timeout_seconds == 60
        assert config.max_retries == 5
        assert config.image_size == 512
        assert config.max_images_per_document == 10

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_enabled_variations(self, mock_getenv):
        """Test various ways to enable LLM."""
        test_cases = ["true", "True", "TRUE", "1", "yes", "Yes", "YES"]

        for enabled_value in test_cases:
            mock_getenv.side_effect = lambda key, default="": {
                "LLM_ENABLED": enabled_value,
                "LLM_API_URL": "https://api.test.com",
                "LLM_API_KEY": "test-key",
            }.get(key, default)

            config = load_llm_config()
            assert config is not None, f"Failed for LLM_ENABLED={enabled_value}"

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_invalid_integer_value(self, mock_getenv):
        """Test handling invalid integer values."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "https://api.test.com",
            "LLM_API_KEY": "test-key",
            "LLM_CONTEXT_LINES": "not-a-number",
        }.get(key, default)

        config = load_llm_config()

        # Should return None due to ValueError
        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_whitespace_handling(self, mock_getenv):
        """Test that whitespace is stripped from values."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "  https://api.test.com  ",
            "LLM_API_KEY": "  test-key  ",
        }.get(key, default)

        config = load_llm_config()

        assert config is not None
        assert config.api_url == "https://api.test.com"
        assert config.api_key == "test-key"

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_empty_string_values(self, mock_getenv):
        """Test handling empty string values."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "",
            "LLM_API_KEY": "test-key",
        }.get(key, default)

        config = load_llm_config()

        # Empty URL should return None
        assert config is None

    @patch("pydocextractor.infra.config.env_loader.os.getenv")
    def test_load_with_env_variables(self, mock_getenv):
        """Test loading directly from environment variables (dotenv not needed)."""
        mock_getenv.side_effect = lambda key, default="": {
            "LLM_ENABLED": "true",
            "LLM_API_URL": "https://api.test.com",
            "LLM_API_KEY": "test-key",
        }.get(key, default)

        config = load_llm_config()

        # Should work from environment variables
        assert config is not None


class TestGetLLMStatusMessage:
    """Test get_llm_status_message function."""

    def test_status_message_disabled(self):
        """Test status message when config is None."""
        message = get_llm_status_message(None)

        assert message == "LLM image description: Disabled"

    def test_status_message_enabled(self):
        """Test status message when config is provided."""
        config = LLMConfig(
            api_url="https://api.test.com",
            api_key="test-key",
            max_images_per_document=5,
        )

        message = get_llm_status_message(config)

        assert "Enabled" in message
        assert "5 images" in message

    def test_status_message_custom_max_images(self):
        """Test status message with custom max images."""
        config = LLMConfig(
            api_url="https://api.test.com",
            api_key="test-key",
            max_images_per_document=10,
        )

        message = get_llm_status_message(config)

        assert "10 images" in message
