# Android SMS Gateway Python API Client

This is a Python client library for interfacing with the [Android SMS Gateway](https://sms.capcom.me) API.

## Requirements

- Python >= 3.6
- One of the following packages:
    - [requests](https://pypi.org/project/requests/)
    - [aiohttp](https://pypi.org/project/aiohttp/)
    - [httpx](https://pypi.org/project/httpx/)

## Installation

```bash
pip install android_sms_gateway
```

You can also install with preferred http client:

```bash
pip install android_sms_gateway[requests]
pip install android_sms_gateway[aiohttp]
pip install android_sms_gateway[httpx]
```

## Usage

Here's an example of using the client:

```python
import asyncio
import os

from android_sms_gateway import client, domain

login = os.getenv("ANDROID_SMS_GATEWAY_LOGIN")
password = os.getenv("ANDROID_SMS_GATEWAY_PASSWORD")


message = domain.Message(
    "Your message text here.",
    ["+1234567890"],
)

def sync_client():
    with client.APIClient(login, password) as c:
        state = c.send(message)
        print(state)

        state = c.get_state(state.id)
        print(state)


async def async_client():
    async with client.AsyncAPIClient(login, password) as c:
        state = await c.send(message)
        print(state)

        state = await c.get_state(state.id)
        print(state)

print("Sync client")
sync_client()

print("\nAsync client")
asyncio.run(async_client())
```

## Client

There are two client classes: `APIClient` and `AsyncAPIClient`. The
`APIClient` is synchronous and the `AsyncAPIClient` is asynchronous. Both
implement the same interface and can be used as context managers.

### Methods

There are two methods:

- `send(message: domain.Message) -> domain.MessageState`: Send a new SMS message.
- `get_state(_id: str) -> domain.MessageState`: Retrieve the state of a previously sent message by its ID.

## HTTP Client

The API clients abstract away the HTTP client used to make requests. The library includes support for some popular HTTP clients and trys to discover them automatically:

- [requests](https://pypi.org/project/requests/) - `APIClient` only
- [aiohttp](https://pypi.org/project/aiohttp/) - `AsyncAPIClient` only
- [httpx](https://pypi.org/project/httpx/) - `APIClient` and `AsyncAPIClient`

Also you can implement your own HTTP client that conforms to the `http.HttpClient` or `ahttp.HttpClient` protocol.

# Contributing

Contributions are welcome! Please submit a pull request or create an issue for anything you'd like to add or change.

# License

This library is open-sourced software licensed under the [Apache-2.0 license](LICENSE).