from dataclasses import dataclass, field
from typing import Optional, Dict, Any, Callable, TypeVar, final

# Type alias for callable used as Kafka delivery callback
F = TypeVar("F", bound=Callable[..., Any])


@final
@dataclass(frozen=True, slots=True)
class PortalModel:
    """
    Immutable model describing a Kafka portal entity.

    This dataclass defines the static attributes required for identifying
    and connecting to a Kafka topic and consumer group in a given cluster.

    Attributes
    ----------
    name : str
        Logical name of the portal instance (for internal reference or logging).
    topic : str
        Kafka topic name where messages are published or consumed.
    bootstrap_servers : str
        Comma-separated list of Kafka broker addresses, e.g. "localhost:9092".
    consumer_group : str
        Kafka consumer group ID used to balance message consumption among workers.
    """
    name: str
    topic: str
    bootstrap_servers: str
    consumer_group: str


@final
@dataclass(frozen=True, slots=True)
class PortalConfig:
    """
    Configuration container for the Worklet `Portal` interface.

    This class defines all runtime configuration for Kafka producers and consumers,
    including batching, compression, delivery retries, circuit breaker thresholds,
    and environment-aware grouping. It is immutable and safe for concurrent access.

    Attributes
    ----------
    kafka_consumer_group : Optional[str]
        Kafka consumer group ID for consuming messages.
        If None, a unique group may be auto-generated by the system.

    env : Optional[str]
        Environment name (e.g., "dev", "staging", "prod") used for namespacing topics
        or consumer groups in shared Kafka clusters.

    on_delivery : Optional[Callable]
        Callback invoked when a Kafka message delivery event completes.
        Signature: `on_delivery(err, msg)` where `err` is None if delivery succeeded.

    kafka_producer_config : Dict[str, Any]
        Configuration dictionary passed directly to `confluent_kafka.Producer`.
        Defaults are optimized for **high throughput**, **low latency**, and
        **strong delivery guarantees** using `acks="all"` and LZ4 compression.

    max_queue_size : int
        Maximum number of tasks/messages held in memory before backpressure applies.

    circuit_breaker_failure_threshold : int
        Number of consecutive failures required to open the circuit breaker.

    circuit_breaker_recovery_timeout_seconds : float
        Time (in seconds) to wait before attempting to recover from a breaker trip.

    circuit_breaker_success_threshold : int
        Number of consecutive successful executions required to close the circuit again.
    """

    env: Optional[str] = None
    kafka_consumer_group: Optional[str] = None
    kafka_producer_config: Dict[str, Any] = field(default_factory=lambda: {
        # ------------------------------------------------------------
        # ✅ Core Connection & Reliability Settings
        # ------------------------------------------------------------
        "acks": "all",  # Wait for all replicas to acknowledge (strongest guarantee)
        "retries": 10,  # Retry transient broker/network errors up to 10 times
        "max.in.flight.requests.per.connection": 5,  # Prevent out-of-order errors when idempotent
        "compression.type": "lz4",  # Fastest compression with minimal CPU overhead
        "linger.ms": 5,  # Small batching delay to increase throughput (5ms max)
        "batch.num.messages": 10_000,  # Batch up to 10K messages per request
        "batch.size": 1_048_576,  # 1MB max batch size (Kafka default limit)
        "queue.buffering.max.kbytes": 10_240,  # 10MB memory buffer for queued messages
        "queue.buffering.max.messages": 100_000,  # Prevent unbounded queue growth

        # ------------------------------------------------------------
        # 🕒 Delivery & Timeout Tuning
        # ------------------------------------------------------------
        "request.timeout.ms": 30_000,  # Max time broker waits to respond (30s)
        "delivery.timeout.ms": 120_000,  # Total send time (incl. retries)
        "socket.keepalive.enable": True,  # Maintain TCP keepalive for long-lived connections

        # ------------------------------------------------------------
        # ⚙️ Performance / Threading
        # ------------------------------------------------------------
        "message.send.max.retries": 3,  # Internal librdkafka retry mechanism
        "queue.buffering.backpressure.threshold": 1,  # Apply backpressure early for stability
        "statistics.interval.ms": 0,  # Disable periodic statistics callbacks by default
    })
    on_delivery: Optional[F] = None
    max_queue_size: int = 1000
    circuit_breaker_failure_threshold: int = 10
    circuit_breaker_recovery_timeout_seconds: float = 30.0
    circuit_breaker_success_threshold: int = 3

    def __post_init__(self):
        """
        Perform runtime validation for configuration integrity.

        Ensures numeric parameters are positive and kafka_producer_config is valid.
        This guards against misconfigurations in production environments
        that could cause subtle runtime errors or undefined behavior.
        """
        # --- Validate circuit breaker thresholds ---
        if self.circuit_breaker_failure_threshold <= 0:
            raise ValueError("circuit_breaker_failure_threshold must be > 0")
        if self.circuit_breaker_recovery_timeout_seconds <= 0:
            raise ValueError("circuit_breaker_recovery_timeout_seconds must be > 0")
        if self.circuit_breaker_success_threshold <= 0:
            raise ValueError("circuit_breaker_success_threshold must be > 0")

        # --- Validate producer configuration type ---
        if not isinstance(self.kafka_producer_config, dict):
            raise TypeError("kafka_producer_config must be a dictionary")
