"""
LightAPI User Goal Example

This example demonstrates how to use LightAPI as envisioned by the user:
- Custom validators
- JWT authentication
- Custom middleware
- CORS support
- Multiple endpoints with different configurations
- Request/response handling
- Pagination and caching (commented out due to current limitations)

Run with: LIGHTAPI_JWT_SECRET="test-secret-key-123" uv run python examples/user_goal_example.py
"""

import os

from lightapi import LightApi, Middleware, RestEndpoint
from lightapi.auth import JWTAuthentication
from lightapi.cache import RedisCache
from lightapi.core import Response
from lightapi.filters import ParameterFilter
from lightapi.pagination import Paginator

# Set JWT secret for testing
os.environ["LIGHTAPI_JWT_SECRET"] = "test-secret-key-123"


class CustomEndpointValidator:
    """Custom validator for endpoint data validation"""

    def validate_name(self, value):
        return value

    def validate_email(self, value):
        return value

    def validate_website(self, value):
        return value


class Company(RestEndpoint):
    """Company endpoint - no authentication required"""

    class Configuration:
        http_method_names = ["GET", "POST"]
        validator_class = CustomEndpointValidator
        filter_class = ParameterFilter

    async def post(self, request):
        """Handle POST requests with custom Response object"""
        return Response(
            {"data": "ok", "request_data": await request.get_data()},
            status_code=200,
            content_type="application/json",
        )

    def get(self, request):
        """Handle GET requests with tuple response"""
        return {"data": "ok"}, 200


class CustomPaginator(Paginator):
    """Custom pagination configuration"""

    limit = 100
    sort = True


class CustomEndpoint(RestEndpoint):
    """Custom endpoint with JWT authentication"""

    class Configuration:
        http_method_names = ["GET", "POST"]
        authentication_class = JWTAuthentication
        # Note: Caching and pagination are commented out due to current serialization issues
        # These features work individually but cause conflicts when combined
        # caching_class = RedisCache
        # caching_method_names = ['GET']
        # pagination_class = CustomPaginator

    def post(self, request):
        """Handle authenticated POST requests"""
        return {"data": "ok", "message": "POST successful"}, 200

    def get(self, request):
        """Handle authenticated GET requests"""
        return {"data": "ok", "message": "GET successful"}, 200


class MyCustomMiddleware(Middleware):
    """Custom middleware for additional authentication checks"""

    def process(self, request, response):
        if response is None:  # Pre-processing
            if "Authorization" not in request.headers:
                return Response({"error": "not allowed"}, status_code=403)
            return None
        return response


class CustomCORSMiddleware(Middleware):
    """Custom CORS middleware (renamed to avoid conflicts with Starlette's CORSMiddleware)"""

    def process(self, request, response):
        if response is None:  # Pre-processing
            if request.method == "OPTIONS":
                return Response(
                    {},
                    status_code=200,
                    headers={
                        "Access-Control-Allow-Origin": "*",
                        "Access-Control-Allow-Methods": "GET, POST, PUT, DELETE, OPTIONS",
                        "Access-Control-Allow-Headers": "Authorization, Content-Type",
                    },
                )
            return None

        # Post-processing - add CORS headers
        # Note: Direct header modification can cause serialization issues
        # For production use, consider using LightApi's built-in CORS support
        return response


# Create the API instance
app = LightApi()

# Register endpoints
app.register(CustomEndpoint)
app.register(Company)

# Add middleware
# Note: Custom auth middleware is commented out to avoid blocking all requests
# In production, you would configure this more selectively
# app.add_middleware([MyCustomMiddleware, CustomCORSMiddleware])

if __name__ == "__main__":
    print("🚀 Starting LightAPI User Goal Example")
    print("📋 Available endpoints:")
    print("   • /company - No authentication required")
    print("   • /custom  - JWT authentication required")
    print("🔑 Generate JWT token with:")
    print(
        "   python -c \"import jwt; import datetime; print(jwt.encode({'user_id': 1, 'exp': datetime.datetime.utcnow() + datetime.timedelta(hours=1)}, 'test-secret-key-123', algorithm='HS256'))\""
    )
    print("📚 API Documentation: http://127.0.0.1:8000/api/docs")
    print("=" * 80)

    app.run(host="127.0.0.1", port=8000)
