# -*- coding: utf-8 -*-
"""
Example Network Generator for UESGraphs
=====================================

This module provides functionality to generate example district 
heating networks. It includes predefined network topologies that 
can be used for testing, demonstration, and validation purposes. 

Main Features:
-------------
* Generation of two example networks: 'Pinola' and 'IBPSA'
* Coordinate transformation for Modelica compatibility
* Time series data generation for temperatures and heat inputs
* Network topology creation with detailed pipe specifications

Example Networks:
---------------
1. IBPSA Network:
   - Based on the `PlugFlowAIT` model from the IBPSA Library
   - Contains 1 supply building, 3 demand buildings, and 2 heating network nodes
   - Includes realistic pipe parameters and thermal properties
   - Reference: doi: 10.1016/j.enconman.2017.08.072

2. Pinola Network:
   - Hybrid network combining IBPSA model and Larock 424 example
   - Features 2 supply buildings, 3 demand buildings, and 3 heating network nodes
   - Complete with pipe specifications and thermal parameters

Functions:
---------
make_network_ibpsa(dir_dest):
    Creates the IBPSA example network with predefined topology

make_network_pinola(dir_dest):
    Creates the Pinola example network with predefined topology

convert_positions(ues_network, width_set):
    Transforms network coordinates for Modelica compatibility

get_temperature_ground():
    Returns ground temperature time series data

get_temperature_supply():
    Returns supply temperature time series data

get_input_heat():
    Returns heat input time series data for demand buildings

Notes:
-----
- All networks are exported with complete metadata for simulation purposes
- Time series data uses 900-second time steps
- Coordinates are automatically adjusted for Modelica model generation
- Network parameters are based on realistic district heating system values
"""

import datetime
import os
from shapely.geometry import Point
from uesgraphs.uesgraph import UESGraph
import uesgraphs.utilities as ut

from uesgraphs.examples import e1_readme_example as e1

def main():
    start_time = datetime.datetime.now()

    # Make workspace
    workspace = ut.make_workspace("uesmodel_examples")
    workspace = e1.workspace_example("e10")
    dir_tests = os.path.join(workspace, "tests")
    dir_modelgen = os.path.join(dir_tests, "test_modelgen")
    print(workspace)
    # Make Pinola network
    dir_pinola = os.path.join(dir_modelgen, "Pinola")
    dir_pinola = os.path.abspath(dir_pinola)
    if not os.path.exists(dir_pinola):
        os.makedirs(dir_pinola)

    make_network_pinola(dir_dest=dir_pinola)

    # Make Ibpsa network
    dir_ibpsa = os.path.join(dir_modelgen, "Ibpsa")
    dir_ibpsa = os.path.abspath(dir_ibpsa)
    if not os.path.exists(dir_ibpsa):
        os.makedirs(dir_ibpsa)

    make_network_ibpsa(dir_dest=dir_ibpsa)

    # Print out runtime
    end_time = datetime.datetime.now()
    runtime = end_time - start_time
    print("runtime:", runtime)


def convert_positions(ues_network, width_set=4000):
    """Translate node positions onto a Modelica-ready canvas

    Parameters
    ----------
    ues_network : uesgraphs.uesgraph.UESGraph object
        An uesgraphs representation of the cooling network
    width_set : float
        Width of the Modelica-ready canvas after translation

    Returns
    -------
    ues_network : uesgraphs.uesgraph.UESGraph object
        An uesgraphs representation of the cooling network
    """

    # Width and height of ues_network before translation
    width_before = ues_network.max_position.x - ues_network.min_position.x
    height_before = ues_network.max_position.y - ues_network.min_position.y

    # Width after translation is set by a parameter
    width_after = width_set

    # Height after translation is calculated according to width
    height_after = width_after / width_before * height_before

    # New node positions are calculated
    for node in ues_network.nodes():
        dx = ues_network.node[node]["position"].x - ues_network.min_position.x
        dy = ues_network.node[node]["position"].y - ues_network.min_position.y

        x_new = dx / width_before * width_after
        y_new = dy / height_before * height_after

        ues_network.node[node]["position"] = Point(x_new, y_new)

    # New minimum and maximum position are set
    ues_network.min_position = Point(0, 0)
    ues_network.max_position = Point(width_after, height_after)

    return ues_network


def make_network_ibpsa(dir_dest):
    """Creates a graph corresponding to the `PlugFlowAIT` model in IBPSA

    The Modelica IBPSA Library (and thus AixLib, too) contains a benchmark
    network that was also used in the following paper:
    doi: 10.1016/j.enconman.2017.08.072

    The model is at
    AixLib.Fluid.FixedResistances.Validation.PlugFlowPipes.PlugFlowAIT

    This function returns the UESGraph representing the network

    Parameters
    ----------
    dir_dest : str
        Path where a `nodes.json` export of the network will be stored

    Returns
    -------
    graph_network : uesgraphs.uesgraph.UESGraph
        Graph of the network
    """

    graph_network = UESGraph()

    # Generate a heat supply building
    supply = graph_network.add_building(
        name="supply", position=Point(20 + 115, 38), is_supply_heating=True
    )

    # Generate three demand buildings
    point_2 = graph_network.add_building(name="point_2", position=Point(0, 38 + 76))
    point_3 = graph_network.add_building(name="point_3", position=Point(0, 0))
    point_4 = graph_network.add_building(name="point_4", position=Point(20, 38 + 29))
    # excluded = graph_network.add_building(
    #     name='excluded',
    #     position=Point(20+115, 38+20),
    # )

    # Create two heating network nodes
    hn_2 = graph_network.add_network_node("heating", position=Point(20, 38))
    hn_3 = graph_network.add_network_node("heating", position=Point(0, 38))

    # Connect buildings and network nodes
    graph_network.add_edge(
        supply,
        hn_2,
        diameter=0.0825,
        length=115,
        name="pipe1",
        m_flow_nom=0.3,
        cPip=500,
        rhoPip=8000,
        thickness=0.0032,
        R=4.76363,
    )
    graph_network.add_edge(
        hn_2,
        point_4,
        diameter=0.0337 - 2 * 0.0032,
        length=29,
        name="pipe4",
        m_flow_nom=0.3,
        cPip=500,
        rhoPip=8000,
        thickness=0.0032,
        R=5.06989,
    )
    graph_network.add_edge(
        hn_2,
        hn_3,
        diameter=0.0825,
        length=20,
        name="pipe5",
        m_flow_nom=0.3,
        cPip=500,
        rhoPip=8000,
        thickness=0.0032,
        R=4.76363,
    )
    graph_network.add_edge(
        hn_3,
        point_2,
        diameter=0.0337 - 2 * 0.0032,
        length=76,
        name="pipe2",
        m_flow_nom=0.3,
        cPip=500,
        rhoPip=8000,
        thickness=0.0032,
        R=5.06989,
    )
    graph_network.add_edge(
        hn_3,
        point_3,
        diameter=0.0337 - 2 * 0.0032,
        length=38,
        name="pipe3",
        m_flow_nom=0.3,
        cPip=500,
        rhoPip=8000,
        thickness=0.0032,
        R=5.06989,
    )

    # Make output workspace
    dest = ut.make_workspace(name_workspace="uesmodel_examples")
    dir_exne = os.path.join(dest, "example-networks")
    dir_IBPSA = os.path.join(dir_exne, "IBPSA_AIT")
    dir_ibpsa = os.path.join(dir_IBPSA, "ibpsa")
    dir_ibpsa = os.path.abspath(dir_ibpsa)
    if not os.path.exists(dir_ibpsa):
        os.makedirs(dir_ibpsa)

    # Generate a JSON file of the UESGraph
    graph_network.to_json(
        path=dir_ibpsa,
        name="IBPSA_AIT",
        description="IBPSA_AIT example for uesmodels",
        all_data=True,
    )

    # Add time series data to graph for direct usage
    # Set Pressure of supply building
    graph_network.node[supply]["p_supply"] = [3e5]  # Pa

    # Set supply temperature using get_temperature_supply function
    # time step 900 s
    graph_network.node[supply]["T_supply"] = get_temperature_supply()

    # Heat inputs from PlugFlowAIT simulation with dT_design = 30 and cp = 4184
    # time step 900 s
    map_input_heat = get_input_heat()

    # Connect input heat data to each building
    graph_network.node[point_2]["input_heat"] = map_input_heat["point_2"]
    graph_network.node[point_3]["input_heat"] = map_input_heat["point_3"]
    graph_network.node[point_4]["input_heat"] = map_input_heat["point_4"]

    # Set ground temperature using get_temperature_ground function
    # time step 900 s
    graph_network.graph["T_ground"] = get_temperature_ground()

    # Additional data
    dT_design = 30
    for node in graph_network.nodelist_building:
        if (
            not graph_network.nodes[node]["is_supply_heating"]
            and not graph_network.nodes[node]["is_supply_cooling"]
        ):
            graph_network.nodes[node]["dT_design"] = dT_design

    fac = 1
    dIns = 0.045
    kIns = 0.024
    for edge in graph_network.edges():
        graph_network.edges[edge[0], edge[1]]["fac"] = fac
        graph_network.edges[edge[0], edge[1]]["dIns"] = dIns
        graph_network.edges[edge[0], edge[1]]["kIns"] = kIns

    # Translate node positions onto a Modelica-ready canvas using
    # convert_positions function. Below the parameter 'width_set' is 400.
    convert_positions(graph_network, width_set=400)

    # Export graph to json
    graph_network.to_json(
        path=dir_dest,
        name=None,
        description="Input graph with data of the Ibpsa network",
        all_data=True,
    )

    return graph_network


def make_network_pinola(dir_dest):
    """Creates a graph of the "Pinola" network

    The Modelica IBPSA Library (and thus AixLib, too) contains a benchmark
    network that was also used in the following paper:
    doi: 10.1016/j.enconman.2017.08.072

    The model is at
    AixLib.Fluid.FixedResistances.Validation.PlugFlowPipes.PlugFlowAIT

    This function returns the UESGraph representing a network that is a mix
    between the IBPSA model and the Larock 424 example

    Parameters
    ----------
    dir_dest : str
        Path where a `nodes.json` export of the network will be stored

    Returns
    -------
    graph_network : uesgraphs.uesgraph.UESGraph
        Graph of the network
    """

    graph_network = UESGraph()

    # Generate heat supply buildings
    supply_heating_1 = graph_network.add_building(
        name="S1", position=Point(-250, 100), is_supply_heating=True
    )
    supply_heating_2 = graph_network.add_building(
        name="S2", position=Point(750, -200), is_supply_heating=True
    )

    # Generate three demand buildings
    B1 = graph_network.add_building(name="B1", position=Point(-80, 370))
    B2 = graph_network.add_building(name="B2", position=Point(500, 550))
    B3 = graph_network.add_building(name="B3", position=Point(800, 50))

    # Generate three heating network nodes
    hn_1 = graph_network.add_network_node("heating", position=Point(00, 300))
    hn_2 = graph_network.add_network_node("heating", position=Point(500, 450))
    hn_3 = graph_network.add_network_node("heating", position=Point(700, 0))

    # Connect heat supply buldings, demand buildings and network nodes by edges
    graph_network.add_edge(
        supply_heating_1,
        hn_1,
        pipeID=1,
        diameter=0.0825,
        length=155,
        name="pipe1",
        m_flow_nom=0.3,
    )
    graph_network.add_edge(
        hn_2, hn_1, pipeID=2, diameter=0.0825, length=100, name="pipe2", m_flow_nom=0.3
    )
    graph_network.add_edge(
        hn_2, hn_3, pipeID=3, diameter=0.0825, length=70, name="pipe3", m_flow_nom=0.3
    )
    graph_network.add_edge(
        hn_3, hn_1, pipeID=4, diameter=0.05, length=45, name="pipe4", m_flow_nom=0.3
    )
    graph_network.add_edge(
        hn_3,
        supply_heating_2,
        pipeID=5,
        diameter=0.0825,
        length=175,
        name="pipe5",
        m_flow_nom=0.3,
    )
    graph_network.add_edge(
        hn_1,
        B1,
        pipeID=6,
        diameter=0.0337 - 2 * 0.0032,
        length=15,
        name="pipe6",
        m_flow_nom=0.3,
    )
    graph_network.add_edge(
        hn_2,
        B2,
        pipeID=7,
        diameter=0.0337 - 2 * 0.0032,
        length=15,
        name="pipe7",
        m_flow_nom=0.3,
    )
    graph_network.add_edge(
        hn_3,
        B3,
        pipeID=8,
        diameter=0.0337 - 2 * 0.0032,
        length=15,
        name="pipe8",
        m_flow_nom=0.3,
    )

    # Set names of heating network nodes
    graph_network.node[hn_1]["name"] = "N1"
    graph_network.node[hn_2]["name"] = "N2"
    graph_network.node[hn_3]["name"] = "N3"

    # Export graph without time-series data
    # Make output workspace
    dest = ut.make_workspace(name_workspace="uesmodel_examples")
    dir_exne = os.path.join(dest, "example-networks")
    dir_Pino = os.path.join(dir_exne, "Pino")
    dir_pino = os.path.join(dir_Pino, "pino")
    dir_pino = os.path.abspath(dir_pino)
    if not os.path.exists(dir_pino):
        os.makedirs(dir_pino)

    # Generate a JSON file of the UESGraph
    graph_network.to_json(
        path=dir_pino,
        name="Pino",
        description="Pino example for uesmodels",
        all_data=True,
    )

    # Add time series data to graph for direct usage
    # Add pressure of supply buildings
    graph_network.node[supply_heating_1]["p_supply"] = [3e5]  # Pa
    graph_network.node[supply_heating_2]["p_supply"] = [3e5]  # Pa

    # Generate supply temperature of supply buildings using
    # get_temperature_supply function
    T_supply = get_temperature_supply()
    graph_network.node[supply_heating_1]["T_supply"] = T_supply
    graph_network.node[supply_heating_2]["T_supply"] = T_supply

    # Heat inputs from PlugFlowAIT simulation with dT_design = 30 and cp = 4184
    map_input_heat = get_input_heat()

    graph_network.node[B1]["input_heat"] = map_input_heat["point_2"]
    graph_network.node[B2]["input_heat"] = map_input_heat["point_3"]
    graph_network.node[B3]["input_heat"] = map_input_heat["point_4"]

    # Set ground temperature using get_temperature_ground function
    graph_network.graph["T_ground"] = get_temperature_ground()

    # Additional data
    dT_design = 30
    for node in graph_network.nodelist_building:
        if (
            not graph_network.nodes[node]["is_supply_heating"]
            and not graph_network.nodes[node]["is_supply_cooling"]
        ):
            graph_network.nodes[node]["dT_design"] = dT_design

    fac = 1
    dIns = 0.045
    kIns = 0.024
    for edge in graph_network.edges():
        graph_network.edges[edge[0], edge[1]]["fac"] = fac
        graph_network.edges[edge[0], edge[1]]["dIns"] = dIns
        graph_network.edges[edge[0], edge[1]]["kIns"] = kIns

    # Translate node positions onto a Modelica-ready canvas using
    # convert_positions function. Below the parameter 'width_set' is 400.
    convert_positions(graph_network, width_set=400)

    # Export graph to json
    graph_network.to_json(
        path=dir_dest,
        name=None,
        description="Input graph with data of the Pinola network",
        all_data=True,
    )

    return graph_network


def get_temperature_ground():
    """Returns a time series of ground temperature from IBPSA example
    """
    T_ground = [
        [0.0, 276.79999],
        [900.0, 276.79999],
        [1800.0, 277.0],
        [2700.0, 277.0],
        [3600.0, 277.0],
        [4500.0, 277.10001],
        [5400.0, 277.20001],
        [6300.0, 277.39999],
        [7200.0, 277.20001],
        [8100.0, 277.20001],
        [9000.0, 277.20001],
        [9900.0, 277.20001],
        [10800.0, 277.29999],
        [11700.0, 277.29999],
        [12600.0, 276.89999],
        [13500.0, 276.70001],
        [14400.0, 276.70001],
        [15300.0, 276.79999],
        [16200.0, 276.79999],
        [17100.0, 276.89999],
        [18000.0, 276.89999],
        [18900.0, 276.70001],
        [19800.0, 276.70001],
        [20700.0, 276.70001],
        [21600.0, 276.70001],
        [22500.0, 276.60001],
        [23400.0, 276.60001],
        [24300.0, 276.70001],
        [25200.0, 276.70001],
        [26100.0, 276.70001],
        [27000.0, 276.60001],
        [27900.0, 277.0],
        [28800.0, 276.79999],
        [29700.0, 276.79999],
        [30600.0, 276.70001],
        [31500.0, 276.70001],
        [32400.0, 276.70001],
        [33300.0, 276.79999],
        [34200.0, 276.70001],
        [35100.0, 276.79999],
        [36000.0, 276.79999],
        [36900.0, 276.79999],
        [37800.0, 276.79999],
        [38700.0, 276.89999],
        [39600.0, 277.10001],
        [40500.0, 277.0],
        [41400.0, 277.0],
        [42300.0, 276.70001],
        [43200.0, 276.70001],
        [44100.0, 276.70001],
        [45000.0, 276.89999],
        [45900.0, 276.79999],
        [46800.0, 276.70001],
        [47700.0, 276.70001],
        [48600.0, 276.5],
        [49500.0, 276.5],
        [50400.0, 276.70001],
        [51300.0, 276.79999],
        [52200.0, 276.70001],
        [53100.0, 276.70001],
        [54000.0, 276.89999],
        [54900.0, 276.89999],
        [55800.0, 276.70001],
        [56700.0, 276.60001],
        [57600.0, 276.70001],
        [58500.0, 276.70001],
        [59400.0, 276.70001],
        [60300.0, 276.70001],
        [61200.0, 276.5],
        [62100.0, 276.29999],
        [63000.0, 276.29999],
        [63900.0, 276.39999],
        [64800.0, 276.5],
        [65700.0, 276.60001],
        [66600.0, 276.60001],
        [67500.0, 276.5],
        [68400.0, 276.60001],
        [69300.0, 276.60001],
        [70200.0, 276.5],
        [71100.0, 276.60001],
        [72000.0, 276.60001],
        [72900.0, 276.60001],
        [73800.0, 276.79999],
        [74700.0, 276.89999],
        [75600.0, 277.0],
        [76500.0, 277.20001],
        [77400.0, 277.20001],
        [78300.0, 277.79999],
        [79200.0, 278.5],
        [80100.0, 278.39999],
        [81000.0, 278.29999],
        [81900.0, 278.5],
        [82800.0, 278.5],
        [83700.0, 278.60001],
        [84600.0, 278.70001],
        [85500.0, 278.5],
        [86400.0, 278.89999],
        [87300.0, 279.39999],
        [88200.0, 279.39999],
        [89100.0, 280.10001],
        [90000.0, 280.29999],
        [90900.0, 280.29999],
        [91800.0, 280.0],
        [92700.0, 279.89999],
        [93600.0, 279.79999],
        [94500.0, 279.60001],
        [95400.0, 279.5],
        [96300.0, 279.39999],
        [97200.0, 279.20001],
        [98100.0, 279.20001],
        [99000.0, 279.20001],
        [99900.0, 278.79999],
        [100800.0, 278.79999],
        [101700.0, 278.5],
        [102600.0, 278.10001],
        [103500.0, 277.70001],
        [104400.0, 277.39999],
        [105300.0, 277.10001],
        [106200.0, 277.10001],
        [107100.0, 276.89999],
        [108000.0, 276.70001],
        [108900.0, 276.29999],
        [109800.0, 276.20001],
        [110700.0, 276.0],
        [111600.0, 276.0],
        [112500.0, 275.89999],
        [113400.0, 275.70001],
        [114300.0, 275.60001],
        [115200.0, 275.29999],
        [116100.0, 275.10001],
        [117000.0, 275.10001],
        [117900.0, 275.10001],
        [118800.0, 274.70001],
        [119700.0, 274.5],
        [120600.0, 274.29999],
        [121500.0, 274.20001],
        [122400.0, 274.20001],
        [123300.0, 274.20001],
        [124200.0, 274.10001],
        [125100.0, 273.79999],
        [126000.0, 273.60001],
        [126900.0, 273.5],
        [127800.0, 273.70001],
        [128700.0, 273.70001],
        [129600.0, 273.39999],
        [130500.0, 273.29999],
        [131400.0, 273.20001],
        [132300.0, 273.20001],
        [133200.0, 273.10001],
        [134100.0, 273.10001],
        [135000.0, 272.79999],
        [135900.0, 272.89999],
        [136800.0, 272.79999],
        [137700.0, 272.79999],
        [138600.0, 272.70001],
        [139500.0, 272.70001],
        [140400.0, 272.79999],
        [141300.0, 272.5],
        [142200.0, 272.39999],
        [143100.0, 272.5],
        [144000.0, 272.79999],
        [144900.0, 272.5],
        [145800.0, 272.5],
        [146700.0, 272.29999],
        [147600.0, 272.20001],
        [148500.0, 272.29999],
        [149400.0, 272.29999],
        [150300.0, 272.39999],
        [151200.0, 272.39999],
        [152100.0, 272.5],
        [153000.0, 272.70001],
        [153900.0, 272.5],
        [154800.0, 272.79999],
        [155700.0, 272.60001],
        [156600.0, 272.60001],
        [157500.0, 272.79999],
        [158400.0, 272.89999],
        [159300.0, 273.10001],
        [160200.0, 272.89999],
        [161100.0, 273.0],
        [162000.0, 273.0],
        [162900.0, 273.20001],
        [163800.0, 273.5],
        [164700.0, 273.89999],
        [165600.0, 274.5],
        [166500.0, 275.0],
        [167400.0, 275.0],
        [168300.0, 275.5],
        [169200.0, 275.89999],
        [170100.0, 276.10001],
        [171000.0, 275.89999],
        [171900.0, 275.60001],
        [172800.0, 275.60001],
        [173700.0, 275.79999],
        [174600.0, 275.79999],
        [175500.0, 275.89999],
        [176400.0, 275.89999],
        [177300.0, 276.10001],
        [178200.0, 276.20001],
        [179100.0, 276.20001],
        [180000.0, 276.20001],
        [180900.0, 276.10001],
        [181800.0, 276.29999],
        [182700.0, 276.39999],
        [183600.0, 276.5],
        [184500.0, 276.5],
        [185400.0, 276.5],
        [186300.0, 276.5],
        [187200.0, 276.60001],
        [188100.0, 276.70001],
        [189000.0, 276.70001],
        [189900.0, 276.70001],
        [190800.0, 276.60001],
        [191700.0, 276.60001],
        [192600.0, 276.60001],
        [193500.0, 276.70001],
        [194400.0, 276.79999],
        [195300.0, 276.79999],
        [196200.0, 277.0],
        [197100.0, 276.89999],
        [198000.0, 277.10001],
        [198900.0, 277.10001],
        [199800.0, 277.0],
        [200700.0, 277.0],
        [201600.0, 276.89999],
        [202500.0, 276.89999],
        [203400.0, 276.70001],
        [204300.0, 276.89999],
        [205200.0, 276.79999],
        [206100.0, 276.79999],
        [207000.0, 276.79999],
        [207900.0, 276.89999],
        [208800.0, 277.0],
        [209700.0, 277.0],
        [210600.0, 277.10001],
        [211500.0, 277.10001],
        [212400.0, 277.10001],
        [213300.0, 277.0],
        [214200.0, 277.0],
        [215100.0, 277.0],
        [216000.0, 276.89999],
        [216900.0, 276.89999],
        [217800.0, 277.10001],
        [218700.0, 276.89999],
        [219600.0, 276.89999],
        [220500.0, 276.89999],
        [221400.0, 276.89999],
        [222300.0, 276.89999],
        [223200.0, 276.79999],
        [224100.0, 276.5],
        [225000.0, 276.39999],
        [225900.0, 276.29999],
        [226800.0, 276.20001],
        [227700.0, 276.0],
        [228600.0, 276.0],
        [229500.0, 275.89999],
        [230400.0, 275.5],
        [231300.0, 275.29999],
        [232200.0, 275.10001],
        [233100.0, 275.10001],
        [234000.0, 275.10001],
        [234900.0, 274.89999],
        [235800.0, 274.79999],
        [236700.0, 274.20001],
        [237600.0, 274.29999],
        [238500.0, 273.89999],
        [239400.0, 273.89999],
        [240300.0, 274.0],
        [241200.0, 273.60001],
        [242100.0, 273.5],
        [243000.0, 273.60001],
        [243900.0, 273.5],
        [244800.0, 273.5],
        [245700.0, 273.39999],
        [246600.0, 273.5],
        [247500.0, 273.70001],
        [248400.0, 273.79999],
        [249300.0, 273.5],
        [250200.0, 273.5],
        [251100.0, 274.29999],
        [252000.0, 274.5],
        [252900.0, 275.10001],
        [253800.0, 276.10001],
        [254700.0, 276.70001],
        [255600.0, 276.70001],
        [256500.0, 277.39999],
        [257400.0, 277.70001],
        [258300.0, 277.79999],
        [259200.0, 278.29999],
        [260100.0, 278.5],
        [261000.0, 278.5],
        [261900.0, 278.20001],
        [262800.0, 278.70001],
        [263700.0, 278.5],
        [264600.0, 278.60001],
        [265500.0, 278.70001],
        [266400.0, 278.70001],
        [267300.0, 278.60001],
        [268200.0, 278.39999],
        [269100.0, 278.20001],
        [270000.0, 278.39999],
        [270900.0, 278.20001],
        [271800.0, 278.29999],
        [272700.0, 278.29999],
        [273600.0, 278.39999],
        [274500.0, 278.0],
        [275400.0, 277.60001],
        [276300.0, 277.39999],
        [277200.0, 277.0],
        [278100.0, 277.0],
        [279000.0, 276.5],
        [279900.0, 276.5],
        [280800.0, 276.29999],
        [281700.0, 276.10001],
        [282600.0, 276.0],
        [283500.0, 276.0],
        [284400.0, 275.5],
        [285300.0, 275.39999],
        [286200.0, 275.20001],
        [287100.0, 275.20001],
        [288000.0, 275.20001],
        [288900.0, 275.20001],
        [289800.0, 275.0],
        [290700.0, 275.0],
        [291600.0, 275.0],
        [292500.0, 275.0],
        [293400.0, 275.10001],
        [294300.0, 275.10001],
        [295200.0, 275.10001],
        [296100.0, 275.0],
        [297000.0, 275.29999],
        [297900.0, 275.10001],
        [298800.0, 275.0],
        [299700.0, 275.0],
        [300600.0, 275.10001],
        [301500.0, 275.10001],
        [302400.0, 275.10001],
        [303300.0, 274.89999],
        [304200.0, 274.70001],
        [305100.0, 274.70001],
        [306000.0, 274.79999],
        [306900.0, 274.89999],
        [307800.0, 274.70001],
        [308700.0, 274.39999],
        [309600.0, 274.89999],
        [310500.0, 275.20001],
        [311400.0, 275.20001],
        [312300.0, 275.10001],
        [313200.0, 275.10001],
        [314100.0, 275.20001],
        [315000.0, 275.29999],
        [315900.0, 275.5],
        [316800.0, 275.5],
        [317700.0, 275.29999],
        [318600.0, 275.29999],
        [319500.0, 275.39999],
        [320400.0, 275.5],
        [321300.0, 275.39999],
        [322200.0, 275.39999],
        [323100.0, 275.5],
        [324000.0, 275.5],
        [324900.0, 275.5],
        [325800.0, 275.29999],
        [326700.0, 275.10001],
        [327600.0, 275.10001],
        [328500.0, 274.89999],
        [329400.0, 274.89999],
        [330300.0, 274.79999],
        [331200.0, 274.70001],
        [332100.0, 274.5],
        [333000.0, 274.5],
        [333900.0, 274.60001],
        [334800.0, 274.89999],
        [335700.0, 274.79999],
        [336600.0, 275.10001],
        [337500.0, 275.10001],
        [338400.0, 275.10001],
        [339300.0, 276.0],
        [340200.0, 276.60001],
        [341100.0, 277.10001],
        [342000.0, 277.79999],
        [342900.0, 278.29999],
        [343800.0, 278.29999],
        [344700.0, 278.5],
        [345600.0, 278.70001],
        [346500.0, 278.89999],
        [347400.0, 279.5],
        [348300.0, 279.79999],
        [349200.0, 279.79999],
        [350100.0, 279.70001],
        [351000.0, 279.5],
        [351900.0, 279.20001],
        [352800.0, 279.0],
        [353700.0, 278.39999],
        [354600.0, 278.39999],
        [355500.0, 278.0],
        [356400.0, 277.79999],
        [357300.0, 277.89999],
        [358200.0, 277.89999],
        [359100.0, 277.60001],
        [360000.0, 277.60001],
        [360900.0, 277.60001],
        [361800.0, 277.10001],
        [362700.0, 277.10001],
        [363600.0, 276.70001],
        [364500.0, 276.5],
        [365400.0, 276.10001],
        [366300.0, 276.10001],
        [367200.0, 275.89999],
        [368100.0, 275.70001],
        [369000.0, 275.5],
        [369900.0, 275.20001],
        [370800.0, 275.10001],
        [371700.0, 275.10001],
        [372600.0, 274.89999],
        [373500.0, 274.5],
        [374400.0, 274.39999],
        [375300.0, 274.20001],
        [376200.0, 274.10001],
        [377100.0, 274.10001],
        [378000.0, 273.79999],
        [378900.0, 273.70001],
        [379800.0, 273.60001],
        [380700.0, 273.29999],
        [381600.0, 273.39999],
        [382500.0, 273.39999],
        [383400.0, 273.29999],
        [384300.0, 273.0],
        [385200.0, 272.79999],
        [386100.0, 272.79999],
        [387000.0, 272.60001],
        [387900.0, 272.60001],
        [388800.0, 272.39999],
        [389700.0, 272.29999],
        [390600.0, 272.29999],
        [391500.0, 272.20001],
        [392400.0, 272.10001],
        [393300.0, 271.89999],
        [394200.0, 271.89999],
        [395100.0, 271.5],
        [396000.0, 271.5],
        [396900.0, 271.60001],
        [397800.0, 271.29999],
        [398700.0, 271.29999],
        [399600.0, 271.29999],
        [400500.0, 271.10001],
        [401400.0, 271.10001],
        [402300.0, 271.0],
        [403200.0, 271.10001],
        [404100.0, 271.10001],
        [405000.0, 271.10001],
        [405900.0, 270.70001],
        [406800.0, 270.70001],
        [407700.0, 270.5],
        [408600.0, 270.5],
        [409500.0, 270.5],
        [410400.0, 270.5],
        [411300.0, 270.60001],
        [412200.0, 270.20001],
        [413100.0, 270.10001],
        [414000.0, 270.29999],
        [414900.0, 270.10001],
        [415800.0, 270.10001],
        [416700.0, 269.70001],
        [417600.0, 269.70001],
        [418500.0, 269.70001],
        [419400.0, 269.79999],
        [420300.0, 270.20001],
        [421200.0, 270.5],
        [422100.0, 271.29999],
        [423000.0, 271.70001],
        [423900.0, 271.70001],
        [424800.0, 272.20001],
        [425700.0, 273.0],
        [426600.0, 273.5],
        [427500.0, 274.29999],
        [428400.0, 274.79999],
        [429300.0, 274.79999],
        [430200.0, 275.29999],
        [431100.0, 275.70001],
        [432000.0, 276.10001],
        [432900.0, 276.60001],
        [433800.0, 277.10001],
        [434700.0, 277.10001],
        [435600.0, 277.29999],
        [436500.0, 277.39999],
        [437400.0, 277.39999],
        [438300.0, 277.39999],
        [439200.0, 277.0],
        [440100.0, 277.0],
        [441000.0, 276.79999],
        [441900.0, 276.70001],
        [442800.0, 276.70001],
        [443700.0, 276.70001],
        [444600.0, 276.89999],
        [445500.0, 276.70001],
        [446400.0, 276.70001],
        [447300.0, 276.29999],
        [448200.0, 275.89999],
        [449100.0, 275.60001],
        [450000.0, 275.39999],
        [450900.0, 275.20001],
        [451800.0, 275.20001],
        [452700.0, 274.89999],
        [453600.0, 274.60001],
        [454500.0, 274.29999],
        [455400.0, 274.29999],
        [456300.0, 274.20001],
        [457200.0, 274.20001],
        [458100.0, 274.10001],
        [459000.0, 273.79999],
        [459900.0, 273.60001],
        [460800.0, 273.60001],
        [461700.0, 273.29999],
        [462600.0, 273.29999],
        [463500.0, 273.20001],
        [464400.0, 273.20001],
        [465300.0, 273.20001],
        [466200.0, 272.79999],
        [467100.0, 272.70001],
        [468000.0, 272.70001],
        [468900.0, 272.70001],
        [469800.0, 272.79999],
        [470700.0, 272.29999],
        [471600.0, 272.29999],
        [472500.0, 272.10001],
        [473400.0, 272.10001],
        [474300.0, 272.29999],
        [475200.0, 272.29999],
        [476100.0, 272.29999],
        [477000.0, 272.60001],
        [477900.0, 272.5],
        [478800.0, 272.5],
        [479700.0, 272.5],
        [480600.0, 272.5],
        [481500.0, 272.5],
        [482400.0, 272.39999],
        [483300.0, 272.20001],
        [484200.0, 272.29999],
        [485100.0, 272.29999],
        [486000.0, 272.29999],
        [486900.0, 272.20001],
        [487800.0, 272.29999],
        [488700.0, 272.29999],
        [489600.0, 272.5],
        [490500.0, 272.5],
        [491400.0, 272.5],
        [492300.0, 272.60001],
        [493200.0, 272.79999],
        [494100.0, 272.89999],
        [495000.0, 273.0],
        [495900.0, 273.0],
        [496800.0, 273.0],
        [497700.0, 273.0],
        [498600.0, 273.10001],
        [499500.0, 273.20001],
        [500400.0, 273.20001],
        [501300.0, 273.20001],
        [502200.0, 273.20001],
        [503100.0, 273.29999],
        [504000.0, 273.39999],
        [504900.0, 273.5],
        [505800.0, 273.5],
        [506700.0, 273.60001],
        [507600.0, 273.70001],
        [508500.0, 273.70001],
        [509400.0, 273.89999],
        [510300.0, 273.89999],
        [511200.0, 273.89999],
        [512100.0, 274.10001],
        [513000.0, 274.29999],
        [513900.0, 274.60001],
        [514800.0, 274.79999],
        [515700.0, 274.79999],
        [516600.0, 274.79999],
        [517500.0, 274.70001],
        [518400.0, 274.79999],
        [519300.0, 274.79999],
        [520200.0, 274.70001],
        [521100.0, 274.5],
        [522000.0, 274.5],
        [522900.0, 274.5],
        [523800.0, 274.60001],
        [524700.0, 274.70001],
        [525600.0, 274.89999],
        [526500.0, 274.79999],
        [527400.0, 274.79999],
        [528300.0, 275.0],
        [529200.0, 274.79999],
        [530100.0, 275.10001],
        [531000.0, 274.70001],
        [531900.0, 274.70001],
        [532800.0, 274.70001],
        [533700.0, 274.70001],
        [534600.0, 274.5],
        [535500.0, 274.20001],
        [536400.0, 274.10001],
        [537300.0, 273.89999],
        [538200.0, 273.70001],
        [539100.0, 273.70001],
        [540000.0, 273.89999],
        [540900.0, 273.89999],
        [541800.0, 273.70001],
        [542700.0, 274.0],
        [543600.0, 273.89999],
        [544500.0, 273.89999],
        [545400.0, 274.0],
        [546300.0, 274.0],
        [547200.0, 274.0],
        [548100.0, 273.89999],
        [549000.0, 274.0],
        [549900.0, 274.0],
        [550800.0, 274.0],
        [551700.0, 274.10001],
        [552600.0, 274.0],
        [553500.0, 274.0],
        [554400.0, 273.89999],
        [555300.0, 273.89999],
        [556200.0, 273.89999],
        [557100.0, 274.0],
        [558000.0, 274.0],
        [558900.0, 274.29999],
        [559800.0, 274.10001],
        [560700.0, 274.10001],
        [561600.0, 274.10001],
        [562500.0, 274.0],
        [563400.0, 274.0],
        [564300.0, 274.0],
        [565200.0, 273.89999],
        [566100.0, 273.89999],
        [567000.0, 274.0],
        [567900.0, 273.89999],
        [568800.0, 273.89999],
        [569700.0, 273.79999],
        [570600.0, 273.89999],
        [571500.0, 273.89999],
        [572400.0, 273.70001],
        [573300.0, 273.60001],
        [574200.0, 273.70001],
        [575100.0, 273.70001],
        [576000.0, 273.5],
        [576900.0, 273.5],
        [577800.0, 273.60001],
        [578700.0, 273.60001],
        [579600.0, 273.70001],
        [580500.0, 273.70001],
        [581400.0, 273.79999],
        [582300.0, 273.79999],
        [583200.0, 273.79999],
        [584100.0, 273.79999],
        [585000.0, 273.79999],
        [585900.0, 273.89999],
        [586800.0, 273.79999],
        [587700.0, 273.79999],
        [588600.0, 273.79999],
        [589500.0, 273.79999],
        [590400.0, 273.79999],
        [591300.0, 273.79999],
        [592200.0, 273.89999],
        [593100.0, 273.89999],
        [594000.0, 274.0],
        [594900.0, 274.0],
        [595800.0, 274.0],
        [596700.0, 273.89999],
        [597600.0, 274.0],
        [598500.0, 273.89999],
        [599400.0, 274.10001],
        [600300.0, 274.20001],
        [601200.0, 274.20001],
        [602100.0, 274.20001],
        [603000.0, 274.20001],
        [603900.0, 274.20001],
    ]

    T_ground = [x[1] for x in T_ground]

    return T_ground


def get_temperature_supply():
    """Returns a time series of supply temperature from IBPSA example
    """
    T_supply = [
        [0.0, 371.79901],
        [900.0, 371.79901],
        [1800.0, 371.10043],
        [2700.0, 371.59375],
        [3600.0, 370.70218],
        [4500.0, 371.29498],
        [5400.0, 372.58984],
        [6300.0, 372.39746],
        [7200.0, 372.09778],
        [8100.0, 371.99643],
        [9000.0, 371.49902],
        [9900.0, 371.39691],
        [10800.0, 371.39606],
        [11700.0, 370.89929],
        [12600.0, 372.38809],
        [13500.0, 373.78818],
        [14400.0, 373.29758],
        [15300.0, 370.80957],
        [16200.0, 372.68665],
        [17100.0, 371.00568],
        [18000.0, 370.20114],
        [18900.0, 369.6994],
        [19800.0, 370.29388],
        [20700.0, 368.8053],
        [21600.0, 370.0899],
        [22500.0, 369.30066],
        [23400.0, 370.98712],
        [24300.0, 370.30029],
        [25200.0, 370.69427],
        [26100.0, 370.09958],
        [27000.0, 370.59393],
        [27900.0, 371.29224],
        [28800.0, 370.89975],
        [29700.0, 372.19034],
        [30600.0, 371.20175],
        [31500.0, 371.69415],
        [32400.0, 372.5914],
        [33300.0, 372.59631],
        [34200.0, 369.31415],
        [35100.0, 371.78391],
        [36000.0, 371.49753],
        [36900.0, 370.80014],
        [37800.0, 371.98984],
        [38700.0, 371.20056],
        [39600.0, 372.28986],
        [40500.0, 372.99445],
        [41400.0, 373.89154],
        [42300.0, 372.1059],
        [43200.0, 372.3949],
        [44100.0, 373.69012],
        [45000.0, 370.90878],
        [45900.0, 372.78766],
        [46800.0, 372.69608],
        [47700.0, 372.09949],
        [48600.0, 372.59363],
        [49500.0, 372.79477],
        [50400.0, 372.29922],
        [51300.0, 373.09146],
        [52200.0, 373.59442],
        [53100.0, 371.80603],
        [54000.0, 373.78781],
        [54900.0, 373.19992],
        [55800.0, 373.59451],
        [56700.0, 372.89999],
        [57600.0, 373.79346],
        [58500.0, 373.39926],
        [59400.0, 373.79465],
        [60300.0, 371.00928],
        [61200.0, 370.99844],
        [62100.0, 372.29196],
        [63000.0, 372.39655],
        [63900.0, 372.09912],
        [64800.0, 372.49539],
        [65700.0, 374.19019],
        [66600.0, 373.10178],
        [67500.0, 372.10104],
        [68400.0, 372.79465],
        [69300.0, 373.1955],
        [70200.0, 372.59998],
        [71100.0, 372.69727],
        [72000.0, 373.59412],
        [72900.0, 373.29883],
        [73800.0, 374.09433],
        [74700.0, 373.3006],
        [75600.0, 373.49594],
        [76500.0, 372.70102],
        [77400.0, 374.98944],
        [78300.0, 374.20081],
        [79200.0, 373.89893],
        [80100.0, 375.59241],
        [81000.0, 377.69],
        [81900.0, 376.30276],
        [82800.0, 377.99167],
        [83700.0, 376.70242],
        [84600.0, 376.69754],
        [85500.0, 375.50235],
        [86400.0, 376.2937],
        [87300.0, 375.70029],
        [88200.0, 375.49908],
        [89100.0, 376.59296],
        [90000.0, 375.89972],
        [90900.0, 373.80487],
        [91800.0, 375.09213],
        [92700.0, 375.29474],
        [93600.0, 374.7988],
        [94500.0, 374.19965],
        [95400.0, 375.89066],
        [96300.0, 375.20004],
        [97200.0, 375.19775],
        [98100.0, 375.49628],
        [99000.0, 375.19867],
        [99900.0, 375.29648],
        [100800.0, 375.79486],
        [101700.0, 372.61023],
        [102600.0, 374.38889],
        [103500.0, 375.49246],
        [104400.0, 375.19839],
        [105300.0, 371.6116],
        [106200.0, 374.38461],
        [107100.0, 372.70413],
        [108000.0, 371.50272],
        [108900.0, 370.99948],
        [109800.0, 371.79324],
        [110700.0, 371.00092],
        [111600.0, 371.49512],
        [112500.0, 370.99988],
        [113400.0, 372.38879],
        [114300.0, 369.90698],
        [115200.0, 372.18884],
        [116100.0, 370.90247],
        [117000.0, 371.19574],
        [117900.0, 371.69366],
        [118800.0, 371.89523],
        [119700.0, 370.50369],
        [120600.0, 370.79575],
        [121500.0, 371.39288],
        [122400.0, 371.39752],
        [123300.0, 371.79492],
        [124200.0, 372.5928],
        [125100.0, 372.8945],
        [126000.0, 373.59131],
        [126900.0, 372.20328],
        [127800.0, 373.29147],
        [128700.0, 374.29285],
        [129600.0, 374.59485],
        [130500.0, 372.50662],
        [131400.0, 374.28717],
        [132300.0, 374.9928],
        [133200.0, 374.79709],
        [134100.0, 373.7012],
        [135000.0, 372.4028],
        [135900.0, 371.99979],
        [136800.0, 374.88254],
        [137700.0, 373.50388],
        [138600.0, 374.49133],
        [139500.0, 375.19269],
        [140400.0, 374.89758],
        [141300.0, 374.39871],
        [142200.0, 374.29785],
        [143100.0, 375.19296],
        [144000.0, 373.60406],
        [144900.0, 374.49185],
        [145800.0, 373.8002],
        [146700.0, 373.49789],
        [147600.0, 373.19913],
        [148500.0, 373.69266],
        [149400.0, 372.90024],
        [150300.0, 371.10489],
        [151200.0, 372.49088],
        [152100.0, 373.8891],
        [153000.0, 372.80246],
        [153900.0, 371.90103],
        [154800.0, 371.59833],
        [155700.0, 372.49213],
        [156600.0, 372.89386],
        [157500.0, 372.79684],
        [158400.0, 372.99438],
        [159300.0, 372.40015],
        [160200.0, 372.29782],
        [161100.0, 373.3916],
        [162000.0, 372.50085],
        [162900.0, 374.18887],
        [163800.0, 372.90262],
        [164700.0, 373.79193],
        [165600.0, 373.59833],
        [166500.0, 374.19443],
        [167400.0, 374.99301],
        [168300.0, 376.98593],
        [169200.0, 374.90649],
        [170100.0, 376.19064],
        [171000.0, 375.79901],
        [171900.0, 376.29388],
        [172800.0, 376.19855],
        [173700.0, 377.78864],
        [174600.0, 376.80115],
        [175500.0, 375.50275],
        [176400.0, 374.99893],
        [177300.0, 374.49887],
        [178200.0, 373.99887],
        [179100.0, 374.49438],
        [180000.0, 375.79044],
        [180900.0, 375.6969],
        [181800.0, 374.90021],
        [182700.0, 373.50333],
        [183600.0, 375.28821],
        [184500.0, 375.79425],
        [185400.0, 375.10153],
        [186300.0, 375.7933],
        [187200.0, 375.8956],
        [188100.0, 373.20911],
        [189000.0, 374.78845],
        [189900.0, 375.89127],
        [190800.0, 373.80783],
        [191700.0, 372.10422],
        [192600.0, 373.78995],
        [193500.0, 371.90494],
        [194400.0, 373.68805],
        [195300.0, 372.30331],
        [196200.0, 371.69986],
        [197100.0, 373.09052],
        [198000.0, 372.29959],
        [198900.0, 371.00333],
        [199800.0, 370.99673],
        [200700.0, 371.29541],
        [201600.0, 371.59579],
        [202500.0, 372.69067],
        [203400.0, 372.09866],
        [204300.0, 370.90283],
        [205200.0, 371.59277],
        [206100.0, 371.49741],
        [207000.0, 372.29285],
        [207900.0, 373.39069],
        [208800.0, 372.79947],
        [209700.0, 373.79178],
        [210600.0, 373.89603],
        [211500.0, 373.69769],
        [212400.0, 374.59259],
        [213300.0, 375.98984],
        [214200.0, 373.70859],
        [215100.0, 372.70148],
        [216000.0, 372.89578],
        [216900.0, 373.89102],
        [217800.0, 373.20001],
        [218700.0, 375.78418],
        [219600.0, 375.39828],
        [220500.0, 375.89182],
        [221400.0, 374.30576],
        [222300.0, 375.19107],
        [223200.0, 374.69962],
        [224100.0, 374.79581],
        [225000.0, 375.492],
        [225900.0, 374.60162],
        [226800.0, 374.29767],
        [227700.0, 372.40637],
        [228600.0, 376.47653],
        [229500.0, 375.40149],
        [230400.0, 373.40704],
        [231300.0, 371.60614],
        [232200.0, 374.38031],
        [233100.0, 374.49545],
        [234000.0, 373.69968],
        [234900.0, 372.40341],
        [235800.0, 371.60068],
        [236700.0, 373.18649],
        [237600.0, 373.99133],
        [238500.0, 371.80942],
        [239400.0, 371.79489],
        [240300.0, 371.59695],
        [241200.0, 371.49582],
        [242100.0, 371.89145],
        [243000.0, 372.09354],
        [243900.0, 372.59119],
        [244800.0, 372.49652],
        [245700.0, 373.7876],
        [246600.0, 373.09869],
        [247500.0, 372.10242],
        [248400.0, 371.89725],
        [249300.0, 372.49014],
        [250200.0, 372.29642],
        [251100.0, 373.28796],
        [252000.0, 372.50061],
        [252900.0, 374.1796],
        [253800.0, 375.1861],
        [254700.0, 375.19412],
        [255600.0, 374.0032],
        [256500.0, 375.48334],
        [257400.0, 374.2052],
        [258300.0, 375.2861],
        [259200.0, 373.60733],
        [260100.0, 374.38928],
        [261000.0, 373.40137],
        [261900.0, 372.2038],
        [262800.0, 372.59189],
        [263700.0, 375.17761],
        [264600.0, 374.69742],
        [265500.0, 373.20615],
        [266400.0, 374.38632],
        [267300.0, 373.89865],
        [268200.0, 373.7959],
        [269100.0, 374.19138],
        [270000.0, 371.91226],
        [270900.0, 371.79581],
        [271800.0, 373.38403],
        [272700.0, 371.90601],
        [273600.0, 370.8028],
        [274500.0, 372.58176],
        [275400.0, 372.49542],
        [276300.0, 371.10565],
        [277200.0, 369.90347],
        [278100.0, 371.38425],
        [279000.0, 372.08917],
        [279900.0, 371.59967],
        [280800.0, 371.4957],
        [281700.0, 371.19827],
        [282600.0, 371.69275],
        [283500.0, 370.60278],
        [284400.0, 371.48981],
        [285300.0, 370.40289],
        [286200.0, 370.89142],
        [287100.0, 371.09372],
        [288000.0, 371.19461],
        [288900.0, 370.00372],
        [289800.0, 371.18777],
        [290700.0, 371.79047],
        [291600.0, 371.39822],
        [292500.0, 371.59363],
        [293400.0, 370.7019],
        [294300.0, 372.7829],
        [295200.0, 370.11356],
        [296100.0, 369.30093],
        [297000.0, 369.49402],
        [297900.0, 369.19867],
        [298800.0, 372.37524],
        [299700.0, 372.99136],
        [300600.0, 372.39914],
        [301500.0, 372.99045],
        [302400.0, 372.79648],
        [303300.0, 373.79044],
        [304200.0, 374.49268],
        [305100.0, 373.20425],
        [306000.0, 372.40051],
        [306900.0, 374.18597],
        [307800.0, 373.99707],
        [308700.0, 373.99536],
        [309600.0, 373.89661],
        [310500.0, 374.19516],
        [311400.0, 374.79221],
        [312300.0, 374.29831],
        [313200.0, 373.2019],
        [314100.0, 371.40643],
        [315000.0, 373.18826],
        [315900.0, 372.69897],
        [316800.0, 374.88519],
        [317700.0, 374.19986],
        [318600.0, 370.91467],
        [319500.0, 371.99091],
        [320400.0, 373.19101],
        [321300.0, 371.10657],
        [322200.0, 372.18985],
        [323100.0, 371.40085],
        [324000.0, 370.89911],
        [324900.0, 373.28497],
        [325800.0, 373.39618],
        [326700.0, 371.50488],
        [327600.0, 372.7912],
        [328500.0, 372.79773],
        [329400.0, 373.99213],
        [330300.0, 372.10532],
        [331200.0, 372.39676],
        [332100.0, 373.19226],
        [333000.0, 372.50027],
        [333900.0, 373.29321],
        [334800.0, 372.30219],
        [335700.0, 371.99734],
        [336600.0, 371.29999],
        [337500.0, 371.99573],
        [338400.0, 372.99405],
        [339300.0, 374.29123],
        [340200.0, 373.99796],
        [341100.0, 373.40005],
        [342000.0, 373.69598],
        [342900.0, 372.70114],
        [343800.0, 373.89255],
        [344700.0, 374.39459],
        [345600.0, 373.90067],
        [346500.0, 375.28659],
        [347400.0, 373.60394],
        [348300.0, 371.30637],
        [349200.0, 371.69351],
        [350100.0, 373.8912],
        [351000.0, 374.89404],
        [351900.0, 373.90109],
        [352800.0, 375.39093],
        [353700.0, 373.10583],
        [354600.0, 372.59869],
        [355500.0, 371.50153],
        [356400.0, 374.18741],
        [357300.0, 371.40869],
        [358200.0, 373.78754],
        [359100.0, 373.29813],
        [360000.0, 372.60083],
        [360900.0, 372.89618],
        [361800.0, 374.59027],
        [362700.0, 372.90381],
        [363600.0, 372.29962],
        [364500.0, 371.7005],
        [365400.0, 371.99521],
        [366300.0, 371.39883],
        [367200.0, 370.40134],
        [368100.0, 371.19324],
        [369000.0, 370.69977],
        [369900.0, 371.19443],
        [370800.0, 371.69409],
        [371700.0, 370.30356],
        [372600.0, 370.9939],
        [373500.0, 371.49472],
        [374400.0, 370.99997],
        [375300.0, 370.79709],
        [376200.0, 370.00049],
        [377100.0, 372.88242],
        [378000.0, 371.90039],
        [378900.0, 370.40433],
        [379800.0, 371.88846],
        [380700.0, 371.69751],
        [381600.0, 371.9953],
        [382500.0, 370.30441],
        [383400.0, 371.29166],
        [384300.0, 370.69937],
        [385200.0, 372.48569],
        [386100.0, 371.20334],
        [387000.0, 372.88666],
        [387900.0, 372.59692],
        [388800.0, 371.60141],
        [389700.0, 372.49045],
        [390600.0, 373.19189],
        [391500.0, 371.60449],
        [392400.0, 372.7887],
        [393300.0, 372.69684],
        [394200.0, 373.29236],
        [395100.0, 372.00415],
        [396000.0, 371.59839],
        [396900.0, 371.49741],
        [397800.0, 371.89343],
        [398700.0, 371.89624],
        [399600.0, 374.383],
        [400500.0, 371.11551],
        [401400.0, 370.99625],
        [402300.0, 371.79089],
        [403200.0, 372.1933],
        [404100.0, 370.90402],
        [405000.0, 370.89648],
        [405900.0, 371.19318],
        [406800.0, 371.09619],
        [407700.0, 372.18903],
        [408600.0, 371.20172],
        [409500.0, 372.18985],
        [410400.0, 372.39413],
        [411300.0, 372.19638],
        [412200.0, 371.69879],
        [413100.0, 371.39761],
        [414000.0, 371.49551],
        [414900.0, 372.4888],
        [415800.0, 371.89975],
        [416700.0, 371.896],
        [417600.0, 372.19446],
        [418500.0, 372.98984],
        [419400.0, 373.69122],
        [420300.0, 373.69531],
        [421200.0, 372.30396],
        [422100.0, 373.38895],
        [423000.0, 372.8992],
        [423900.0, 372.49796],
        [424800.0, 372.39636],
        [425700.0, 376.77161],
        [426600.0, 374.50925],
        [427500.0, 373.3028],
        [428400.0, 372.10306],
        [429300.0, 376.56693],
        [430200.0, 374.50848],
        [431100.0, 372.50659],
        [432000.0, 372.69464],
        [432900.0, 375.28012],
        [433800.0, 374.90002],
        [434700.0, 376.28796],
        [435600.0, 375.00327],
        [436500.0, 374.79681],
        [437400.0, 373.10583],
        [438300.0, 374.68683],
        [439200.0, 375.19232],
        [440100.0, 372.70956],
        [441000.0, 374.28836],
        [441900.0, 374.39572],
        [442800.0, 372.90393],
        [443700.0, 371.40393],
        [444600.0, 374.97748],
        [445500.0, 374.29987],
        [446400.0, 372.80417],
        [447300.0, 374.58804],
        [448200.0, 375.39111],
        [449100.0, 374.69983],
        [450000.0, 372.10928],
        [450900.0, 371.49951],
        [451800.0, 373.18863],
        [452700.0, 371.50522],
        [453600.0, 370.79999],
        [454500.0, 371.69147],
        [455400.0, 372.19385],
        [456300.0, 371.99759],
        [457200.0, 370.40436],
        [458100.0, 371.19321],
        [459000.0, 372.29126],
        [459900.0, 371.60095],
        [460800.0, 371.49719],
        [461700.0, 372.39236],
        [462600.0, 371.0032],
        [463500.0, 371.29504],
        [464400.0, 371.69598],
        [465300.0, 372.59253],
        [466200.0, 371.80038],
        [467100.0, 371.89639],
        [468000.0, 370.60349],
        [468900.0, 370.79593],
        [469800.0, 371.09515],
        [470700.0, 372.09143],
        [471600.0, 370.60498],
        [472500.0, 371.29218],
        [473400.0, 371.79443],
        [474300.0, 372.3931],
        [475200.0, 372.59534],
        [476100.0, 373.29358],
        [477000.0, 372.59985],
        [477900.0, 373.59216],
        [478800.0, 374.29282],
        [479700.0, 371.31064],
        [480600.0, 371.19656],
        [481500.0, 370.99786],
        [482400.0, 370.897],
        [483300.0, 370.99655],
        [484200.0, 371.99179],
        [485100.0, 371.99683],
        [486000.0, 372.4939],
        [486900.0, 372.09845],
        [487800.0, 371.69949],
        [488700.0, 372.59158],
        [489600.0, 372.29782],
        [490500.0, 371.30139],
        [491400.0, 370.40088],
        [492300.0, 370.89319],
        [493200.0, 370.00104],
        [494100.0, 370.59332],
        [495000.0, 370.59634],
        [495900.0, 369.4024],
        [496800.0, 370.39142],
        [497700.0, 370.59424],
        [498600.0, 370.99292],
        [499500.0, 370.10123],
        [500400.0, 370.59167],
        [501300.0, 370.09879],
        [502200.0, 370.79172],
        [503100.0, 371.19312],
        [504000.0, 370.89685],
        [504900.0, 370.20059],
        [505800.0, 370.69186],
        [506700.0, 370.39673],
        [507600.0, 369.89896],
        [508500.0, 370.68979],
        [509400.0, 370.59595],
        [510300.0, 369.99973],
        [511200.0, 370.59265],
        [512100.0, 371.59033],
        [513000.0, 373.58154],
        [513900.0, 372.89984],
        [514800.0, 373.19321],
        [515700.0, 374.38705],
        [516600.0, 375.28983],
        [517500.0, 374.99792],
        [518400.0, 375.19385],
        [519300.0, 375.49249],
        [520200.0, 375.89279],
        [521100.0, 372.71573],
        [522000.0, 373.58844],
        [522900.0, 373.09836],
        [523800.0, 374.98041],
        [524700.0, 374.69693],
        [525600.0, 374.89316],
        [526500.0, 374.99408],
        [527400.0, 373.90204],
        [528300.0, 372.30652],
        [529200.0, 373.98297],
        [530100.0, 373.1012],
        [531000.0, 375.67639],
        [531900.0, 375.59579],
        [532800.0, 374.60153],
        [533700.0, 371.41788],
        [534600.0, 372.68619],
        [535500.0, 371.90097],
        [536400.0, 373.68414],
        [537300.0, 371.7084],
        [538200.0, 373.18536],
        [539100.0, 371.50644],
        [540000.0, 371.79276],
        [540900.0, 371.99313],
        [541800.0, 369.71082],
        [542700.0, 370.49033],
        [543600.0, 368.80743],
        [544500.0, 367.50476],
        [545400.0, 367.09796],
        [546300.0, 368.18869],
        [547200.0, 365.21436],
        [548100.0, 366.09085],
        [549000.0, 368.38004],
        [549900.0, 369.48761],
        [550800.0, 367.11133],
        [551700.0, 369.77676],
        [552600.0, 369.29874],
        [553500.0, 371.37924],
        [554400.0, 372.58646],
        [555300.0, 371.80109],
        [556200.0, 371.19928],
        [557100.0, 372.18854],
        [558000.0, 370.40726],
        [558900.0, 369.99789],
        [559800.0, 371.78165],
        [560700.0, 373.18546],
        [561600.0, 371.70535],
        [562500.0, 373.08615],
        [563400.0, 371.60489],
        [564300.0, 371.6955],
        [565200.0, 372.09198],
        [566100.0, 371.89709],
        [567000.0, 372.39249],
        [567900.0, 373.38861],
        [568800.0, 372.8988],
        [569700.0, 372.49799],
        [570600.0, 372.29846],
        [571500.0, 372.89166],
        [572400.0, 372.69708],
        [573300.0, 372.00107],
        [574200.0, 371.69809],
        [575100.0, 372.09341],
        [576000.0, 372.09622],
        [576900.0, 372.29388],
        [577800.0, 371.50043],
        [578700.0, 370.70047],
        [579600.0, 370.59732],
        [580500.0, 369.00452],
        [581400.0, 367.70328],
        [582300.0, 370.48083],
        [583200.0, 369.89816],
        [584100.0, 367.11008],
        [585000.0, 369.68597],
        [585900.0, 370.59268],
        [586800.0, 367.41223],
        [587700.0, 368.79044],
        [588600.0, 368.99512],
        [589500.0, 365.91269],
        [590400.0, 367.4874],
        [591300.0, 367.79477],
        [592200.0, 367.99478],
        [593100.0, 365.21057],
        [594000.0, 366.49036],
        [594900.0, 366.19836],
        [595800.0, 367.39191],
        [596700.0, 364.80905],
        [597600.0, 369.47443],
        [598500.0, 371.78601],
        [599400.0, 371.10016],
        [600300.0, 370.89828],
        [601200.0, 371.69366],
        [602100.0, 371.59747],
        [603000.0, 372.99045],
        [603900.0, 372.9968],
    ]

    T_supply = [x[1] for x in T_supply]

    return T_supply


def get_input_heat():
    """Return the map for heat input
    """
    map_input_heat = {
        "point_2": [
            [0.0, 21087.36],
            [900.0, 12426.47987448],
            [1800.0, 11171.28025104],
            [2700.0, 11924.39987448],
            [3600.0, 19832.1612552],
            [4500.0, 12677.52],
            [5400.0, 12677.52],
            [6300.0, 11045.759999999998],
            [7200.0, 11045.759999999998],
            [8100.0, 12175.44037656],
            [9000.0, 20208.72],
            [9900.0, 11924.39987448],
            [10800.0, 11924.39987448],
            [11700.0, 12175.44037656],
            [12600.0, 11422.319748959999],
            [13500.0, 20083.2],
            [14400.0, 11422.319748959999],
            [15300.0, 11798.87962344],
            [16200.0, 11547.839999999998],
            [17100.0, 11547.839999999998],
            [18000.0, 20710.8012552],
            [18900.0, 11045.759999999998],
            [19800.0, 12300.95962344],
            [20700.0, 11171.28025104],
            [21600.0, 11924.39987448],
            [22500.0, 11924.39987448],
            [23400.0, 20459.760000000002],
            [24300.0, 12426.47987448],
            [25200.0, 11798.87962344],
            [26100.0, 11798.87962344],
            [27000.0, 21087.36],
            [27900.0, 12049.92012552],
            [28800.0, 12300.95962344],
            [29700.0, 11171.28025104],
            [30600.0, 11924.39987448],
            [31500.0, 20961.840000000004],
            [32400.0, 12928.56],
            [33300.0, 12928.56],
            [34200.0, 12049.92012552],
            [35100.0, 11798.87962344],
            [36000.0, 11171.28025104],
            [36900.0, 20334.24],
            [37800.0, 11798.87962344],
            [38700.0, 11798.87962344],
            [39600.0, 12049.92012552],
            [40500.0, 10920.23962344],
            [41400.0, 11798.87962344],
            [42300.0, 10418.159623440002],
            [43200.0, 10418.159623440002],
            [44100.0, 10418.159623440002],
            [45000.0, 11924.39987448],
            [45900.0, 7531.19987448],
            [46800.0, 8535.36050208],
            [47700.0, 11045.759999999998],
            [48600.0, 18451.440000000002],
            [49500.0, 11045.759999999998],
            [50400.0, 11045.759999999998],
            [51300.0, 7782.23987448],
            [52200.0, 7656.72012552],
            [53100.0, 10794.72037656],
            [54000.0, 19204.559999999998],
            [54900.0, 10041.599748960001],
            [55800.0, 10041.599748960001],
            [56700.0, 9790.56025104],
            [57600.0, 10041.599748960001],
            [58500.0, 19957.678744800003],
            [59400.0, 10543.679874480002],
            [60300.0, 9665.04],
            [61200.0, 9665.04],
            [62100.0, 10418.159623440002],
            [63000.0, 11296.80050208],
            [63900.0, 10794.72037656],
            [64800.0, 9916.08050208],
            [65700.0, 9916.08050208],
            [66600.0, 9916.08050208],
            [67500.0, 10418.159623440002],
            [68400.0, 20585.28],
            [69300.0, 10543.679874480002],
            [70200.0, 10418.159623440002],
            [71100.0, 10292.64025104],
            [72000.0, 10292.64025104],
            [72900.0, 19581.12],
            [73800.0, 10794.72037656],
            [74700.0, 10669.200125520001],
            [75600.0, 10418.159623440002],
            [76500.0, 19957.678744800003],
            [77400.0, 19957.678744800003],
            [78300.0, 11045.759999999998],
            [79200.0, 10920.23962344],
            [80100.0, 9916.08050208],
            [81000.0, 10794.72037656],
            [81900.0, 19204.559999999998],
            [82800.0, 11296.80050208],
            [83700.0, 11296.80050208],
            [84600.0, 10292.64025104],
            [85500.0, 10794.72037656],
            [86400.0, 20208.72],
            [87300.0, 11045.759999999998],
            [88200.0, 11045.759999999998],
            [89100.0, 10041.599748960001],
            [90000.0, 9916.08050208],
            [90900.0, 18828.0012552],
            [91800.0, 10041.599748960001],
            [92700.0, 9665.04],
            [93600.0, 9916.08050208],
            [94500.0, 9916.08050208],
            [95400.0, 17949.3612552],
            [96300.0, 11422.319748959999],
            [97200.0, 9665.04],
            [98100.0, 10669.200125520001],
            [99000.0, 19957.678744800003],
            [99900.0, 19957.678744800003],
            [100800.0, 11171.28025104],
            [101700.0, 10041.599748960001],
            [102600.0, 10669.200125520001],
            [103500.0, 19455.600000000002],
            [104400.0, 10543.679874480002],
            [105300.0, 10543.679874480002],
            [106200.0, 9916.08050208],
            [107100.0, 10292.64025104],
            [108000.0, 20208.72],
            [108900.0, 10041.599748960001],
            [109800.0, 9539.519748960001],
            [110700.0, 9539.519748960001],
            [111600.0, 9916.08050208],
            [112500.0, 19330.08],
            [113400.0, 11045.759999999998],
            [114300.0, 10669.200125520001],
            [115200.0, 10794.72037656],
            [116100.0, 10794.72037656],
            [117000.0, 19204.559999999998],
            [117900.0, 10292.64025104],
            [118800.0, 10920.23962344],
            [119700.0, 10418.159623440002],
            [120600.0, 20585.28],
            [121500.0, 20585.28],
            [122400.0, 11171.28025104],
            [123300.0, 10167.12],
            [124200.0, 9916.08050208],
            [125100.0, 10167.12],
            [126000.0, 19455.600000000002],
            [126900.0, 19455.600000000002],
            [127800.0, 11045.759999999998],
            [128700.0, 11798.87962344],
            [129600.0, 6652.55987448],
            [130500.0, 9414.00037656],
            [131400.0, 11171.28025104],
            [132300.0, 11171.28025104],
            [133200.0, 12300.95962344],
            [134100.0, 18325.92],
            [135000.0, 5271.839874480001],
            [135900.0, 11296.80050208],
            [136800.0, 12426.47987448],
            [137700.0, 12552.0],
            [138600.0, 12552.0],
            [139500.0, 6652.55987448],
            [140400.0, 7907.76012552],
            [141300.0, 7029.12025104],
            [142200.0, 11422.319748959999],
            [143100.0, 1129.679949792],
            [144000.0, 1129.679949792],
            [144900.0, 11171.28025104],
            [145800.0, 12300.95962344],
            [146700.0, 19706.6412552],
            [147600.0, 8911.920251040001],
            [148500.0, 10920.23962344],
            [149400.0, 10920.23962344],
            [150300.0, 12300.95962344],
            [151200.0, 7154.639999999999],
            [152100.0, 11296.80050208],
            [153000.0, 10041.599748960001],
            [153900.0, 12426.47987448],
            [154800.0, 12426.47987448],
            [155700.0, 8911.920251040001],
            [156600.0, 19581.12],
            [157500.0, 10794.72037656],
            [158400.0, 10543.679874480002],
            [159300.0, 9916.08050208],
            [160200.0, 9916.08050208],
            [161100.0, 19957.678744800003],
            [162000.0, 11924.39987448],
            [162900.0, 10418.159623440002],
            [163800.0, 10418.159623440002],
            [164700.0, 14434.800000000001],
            [165600.0, 14434.800000000001],
            [166500.0, 13681.68],
            [167400.0, 10794.72037656],
            [168300.0, 10041.599748960001],
            [169200.0, 10543.679874480002],
            [170100.0, 19204.559999999998],
            [171000.0, 19204.559999999998],
            [171900.0, 10543.679874480002],
            [172800.0, 11045.759999999998],
            [173700.0, 10669.200125520001],
            [174600.0, 19581.12],
            [175500.0, 10543.679874480002],
            [176400.0, 10543.679874480002],
            [177300.0, 10292.64025104],
            [178200.0, 10543.679874480002],
            [179100.0, 19204.559999999998],
            [180000.0, 11296.80050208],
            [180900.0, 10418.159623440002],
            [181800.0, 10041.599748960001],
            [182700.0, 10041.599748960001],
            [183600.0, 18451.440000000002],
            [184500.0, 9916.08050208],
            [185400.0, 10167.12],
            [186300.0, 10543.679874480002],
            [187200.0, 18953.5187448],
            [188100.0, 18953.5187448],
            [189000.0, 10669.200125520001],
            [189900.0, 10794.72037656],
            [190800.0, 9790.56025104],
            [191700.0, 19706.6412552],
            [192600.0, 11045.759999999998],
            [193500.0, 11045.759999999998],
            [194400.0, 9539.519748960001],
            [195300.0, 10418.159623440002],
            [196200.0, 19832.1612552],
            [197100.0, 10669.200125520001],
            [198000.0, 10292.64025104],
            [198900.0, 10292.64025104],
            [199800.0, 9916.08050208],
            [200700.0, 19832.1612552],
            [201600.0, 10292.64025104],
            [202500.0, 10418.159623440002],
            [203400.0, 10794.72037656],
            [204300.0, 10794.72037656],
            [205200.0, 19330.08],
            [206100.0, 10418.159623440002],
            [207000.0, 10794.72037656],
            [207900.0, 10669.200125520001],
            [208800.0, 19330.08],
            [209700.0, 19330.08],
            [210600.0, 10292.64025104],
            [211500.0, 10418.159623440002],
            [212400.0, 10543.679874480002],
            [213300.0, 10167.12],
            [214200.0, 12049.92012552],
            [215100.0, 12049.92012552],
            [216000.0, 7907.76012552],
            [216900.0, 7656.72012552],
            [217800.0, 10920.23962344],
            [218700.0, 11296.80050208],
            [219600.0, 2886.9599999999996],
            [220500.0, 2886.9599999999996],
            [221400.0, 10669.200125520001],
            [222300.0, 17196.238744799997],
            [223200.0, 2259.35987448],
            [224100.0, 8409.840251040001],
            [225000.0, 13305.119999999999],
            [225900.0, 3012.48],
            [226800.0, 3012.48],
            [227700.0, 6024.96],
            [228600.0, 11422.319748959999],
            [229500.0, 19706.6412552],
            [230400.0, 5397.36012552],
            [231300.0, 9288.48012552],
            [232200.0, 9288.48012552],
            [233100.0, 11296.80050208],
            [234000.0, 12175.44037656],
            [234900.0, 6652.55987448],
            [235800.0, 12049.92012552],
            [236700.0, 5648.40025104],
            [237600.0, 5648.40025104],
            [238500.0, 10041.599748960001],
            [239400.0, 10920.23962344],
            [240300.0, 17698.32],
            [241200.0, 11171.28025104],
            [242100.0, 9916.08050208],
            [243000.0, 9916.08050208],
            [243900.0, 9790.56025104],
            [244800.0, 10669.200125520001],
            [245700.0, 19204.559999999998],
            [246600.0, 10669.200125520001],
            [247500.0, 9916.08050208],
            [248400.0, 9916.08050208],
            [249300.0, 10794.72037656],
            [250200.0, 19957.678744800003],
            [251100.0, 10920.23962344],
            [252000.0, 10669.200125520001],
            [252900.0, 10669.200125520001],
            [253800.0, 10669.200125520001],
            [254700.0, 19455.600000000002],
            [255600.0, 10418.159623440002],
            [256500.0, 10543.679874480002],
            [257400.0, 11422.319748959999],
            [258300.0, 19706.6412552],
            [259200.0, 19706.6412552],
            [260100.0, 11547.839999999998],
            [261000.0, 10794.72037656],
            [261900.0, 10418.159623440002],
            [262800.0, 18325.92],
            [263700.0, 11798.87962344],
            [264600.0, 10292.64025104],
            [265500.0, 10292.64025104],
            [266400.0, 9916.08050208],
            [267300.0, 19455.600000000002],
            [268200.0, 10920.23962344],
            [269100.0, 10167.12],
            [270000.0, 10794.72037656],
            [270900.0, 10794.72037656],
            [271800.0, 18074.878744799997],
            [272700.0, 10167.12],
            [273600.0, 10669.200125520001],
            [274500.0, 10292.64025104],
            [275400.0, 19581.12],
            [276300.0, 19581.12],
            [277200.0, 13305.119999999999],
            [278100.0, 10418.159623440002],
            [279000.0, 10418.159623440002],
            [279900.0, 19706.6412552],
            [280800.0, 10794.72037656],
            [281700.0, 10794.72037656],
            [282600.0, 10669.200125520001],
            [283500.0, 11045.759999999998],
            [284400.0, 19832.1612552],
            [285300.0, 12049.92012552],
            [286200.0, 10669.200125520001],
            [287100.0, 10543.679874480002],
            [288000.0, 10794.72037656],
            [288900.0, 10794.72037656],
            [289800.0, 11171.28025104],
            [290700.0, 10292.64025104],
            [291600.0, 10543.679874480002],
            [292500.0, 10794.72037656],
            [293400.0, 11045.759999999998],
            [294300.0, 11045.759999999998],
            [295200.0, 11045.759999999998],
            [296100.0, 11045.759999999998],
            [297000.0, 11171.28025104],
            [297900.0, 10669.200125520001],
            [298800.0, 10669.200125520001],
            [299700.0, 11422.319748959999],
            [300600.0, 10543.679874480002],
            [301500.0, 11171.28025104],
            [302400.0, 8786.4],
            [303300.0, 12175.44037656],
            [304200.0, 6778.0801255199995],
            [305100.0, 11171.28025104],
            [306000.0, 10794.72037656],
            [306900.0, 11045.759999999998],
            [307800.0, 1757.28],
            [308700.0, 11547.839999999998],
            [309600.0, 4142.16],
            [310500.0, 1255.199974896],
            [311400.0, 11673.36025104],
            [312300.0, 3765.59987448],
            [313200.0, 12300.95962344],
            [314100.0, 10669.200125520001],
            [315000.0, 10794.72037656],
            [315900.0, 11296.80050208],
            [316800.0, 5773.919999999999],
            [317700.0, 5773.919999999999],
            [318600.0, 9790.56025104],
            [319500.0, 11296.80050208],
            [320400.0, 10041.599748960001],
            [321300.0, 10418.159623440002],
            [322200.0, 10669.200125520001],
            [323100.0, 10669.200125520001],
            [324000.0, 10920.23962344],
            [324900.0, 10167.12],
            [325800.0, 11171.28025104],
            [326700.0, 10794.72037656],
            [327600.0, 10669.200125520001],
            [328500.0, 10669.200125520001],
            [329400.0, 11171.28025104],
            [330300.0, 11045.759999999998],
            [331200.0, 10920.23962344],
            [332100.0, 11296.80050208],
            [333000.0, 19957.678744800003],
            [333900.0, 19957.678744800003],
            [334800.0, 10543.679874480002],
            [335700.0, 10669.200125520001],
            [336600.0, 10794.72037656],
            [337500.0, 12175.44037656],
            [338400.0, 17070.721255200002],
            [339300.0, 17070.721255200002],
            [340200.0, 11422.319748959999],
            [341100.0, 11547.839999999998],
            [342000.0, 11422.319748959999],
            [342900.0, 20585.28],
            [343800.0, 11296.80050208],
            [344700.0, 11296.80050208],
            [345600.0, 11547.839999999998],
            [346500.0, 11422.319748959999],
            [347400.0, 18325.92],
            [348300.0, 12426.47987448],
            [349200.0, 11171.28025104],
            [350100.0, 11171.28025104],
            [351000.0, 11171.28025104],
            [351900.0, 18200.399999999998],
            [352800.0, 11296.80050208],
            [353700.0, 11171.28025104],
            [354600.0, 11422.319748959999],
            [355500.0, 18953.5187448],
            [356400.0, 18953.5187448],
            [357300.0, 14309.279999999999],
            [358200.0, 10920.23962344],
            [359100.0, 11422.319748959999],
            [360000.0, 12049.92012552],
            [360900.0, 19957.678744800003],
            [361800.0, 19957.678744800003],
            [362700.0, 11045.759999999998],
            [363600.0, 11422.319748959999],
            [364500.0, 11171.28025104],
            [365400.0, 19330.08],
            [366300.0, 11296.80050208],
            [367200.0, 11296.80050208],
            [368100.0, 11171.28025104],
            [369000.0, 10920.23962344],
            [369900.0, 19706.6412552],
            [370800.0, 11422.319748959999],
            [371700.0, 10669.200125520001],
            [372600.0, 11045.759999999998],
            [373500.0, 11296.80050208],
            [374400.0, 19204.559999999998],
            [375300.0, 11547.839999999998],
            [376200.0, 11422.319748959999],
            [377100.0, 10920.23962344],
            [378000.0, 11045.759999999998],
            [378900.0, 20208.72],
            [379800.0, 20083.2],
            [380700.0, 11798.87962344],
            [381600.0, 11171.28025104],
            [382500.0, 11171.28025104],
            [383400.0, 11045.759999999998],
            [384300.0, 19957.678744800003],
            [385200.0, 11673.36025104],
            [386100.0, 11171.28025104],
            [387000.0, 11547.839999999998],
            [387900.0, 11547.839999999998],
            [388800.0, 20710.8012552],
            [389700.0, 11924.39987448],
            [390600.0, 11422.319748959999],
            [391500.0, 11422.319748959999],
            [392400.0, 11547.839999999998],
            [393300.0, 11547.839999999998],
            [394200.0, 19957.678744800003],
            [395100.0, 11045.759999999998],
            [396000.0, 11045.759999999998],
            [396900.0, 11171.28025104],
            [397800.0, 19957.678744800003],
            [398700.0, 11798.87962344],
            [399600.0, 11798.87962344],
            [400500.0, 10794.72037656],
            [401400.0, 10543.679874480002],
            [402300.0, 11045.759999999998],
            [403200.0, 15564.48],
            [404100.0, 10920.23962344],
            [405000.0, 10920.23962344],
            [405900.0, 11296.80050208],
            [406800.0, 11171.28025104],
            [407700.0, 20459.760000000002],
            [408600.0, 11296.80050208],
            [409500.0, 11045.759999999998],
            [410400.0, 11045.759999999998],
            [411300.0, 11171.28025104],
            [412200.0, 13681.68],
            [413100.0, 11045.759999999998],
            [414000.0, 11171.28025104],
            [414900.0, 11422.319748959999],
            [415800.0, 11422.319748959999],
            [416700.0, 11296.80050208],
            [417600.0, 19581.12],
            [418500.0, 11171.28025104],
            [419400.0, 11296.80050208],
            [420300.0, 11045.759999999998],
            [421200.0, 11045.759999999998],
            [422100.0, 19957.678744800003],
            [423000.0, 11547.839999999998],
            [423900.0, 11547.839999999998],
            [424800.0, 11422.319748959999],
            [425700.0, 20083.2],
            [426600.0, 20083.2],
            [427500.0, 11296.80050208],
            [428400.0, 11924.39987448],
            [429300.0, 11045.759999999998],
            [430200.0, 19832.1612552],
            [431100.0, 12677.52],
            [432000.0, 12677.52],
            [432900.0, 11422.319748959999],
            [433800.0, 11171.28025104],
            [434700.0, 18576.96],
            [435600.0, 11045.759999999998],
            [436500.0, 11045.759999999998],
            [437400.0, 11045.759999999998],
            [438300.0, 10669.200125520001],
            [439200.0, 12049.92012552],
            [440100.0, 19832.1612552],
            [441000.0, 11422.319748959999],
            [441900.0, 10669.200125520001],
            [442800.0, 10669.200125520001],
            [443700.0, 10794.72037656],
            [444600.0, 19330.08],
            [445500.0, 11547.839999999998],
            [446400.0, 10920.23962344],
            [447300.0, 11045.759999999998],
            [448200.0, 11045.759999999998],
            [449100.0, 19957.678744800003],
            [450000.0, 11422.319748959999],
            [450900.0, 11547.839999999998],
            [451800.0, 11045.759999999998],
            [452700.0, 19581.12],
            [453600.0, 19581.12],
            [454500.0, 12426.47987448],
            [455400.0, 10669.200125520001],
            [456300.0, 10418.159623440002],
            [457200.0, 19455.600000000002],
            [458100.0, 10920.23962344],
            [459000.0, 10920.23962344],
            [459900.0, 11547.839999999998],
            [460800.0, 10794.72037656],
            [461700.0, 10794.72037656],
            [462600.0, 10794.72037656],
            [463500.0, 19079.039999999997],
            [464400.0, 11045.759999999998],
            [465300.0, 11296.80050208],
            [466200.0, 11171.28025104],
            [467100.0, 11171.28025104],
            [468000.0, 18702.48],
            [468900.0, 11673.36025104],
            [469800.0, 10543.679874480002],
            [470700.0, 11171.28025104],
            [471600.0, 11422.319748959999],
            [472500.0, 11422.319748959999],
            [473400.0, 11422.319748959999],
            [474300.0, 20208.72],
            [475200.0, 11422.319748959999],
            [476100.0, 11171.28025104],
            [477000.0, 11171.28025104],
            [477900.0, 11422.319748959999],
            [478800.0, 21087.36],
            [479700.0, 11547.839999999998],
            [480600.0, 10794.72037656],
            [481500.0, 11924.39987448],
            [482400.0, 11171.28025104],
            [483300.0, 11798.87962344],
            [484200.0, 18828.0012552],
            [485100.0, 10920.23962344],
            [486000.0, 10920.23962344],
            [486900.0, 10418.159623440002],
            [487800.0, 10920.23962344],
            [488700.0, 20208.72],
            [489600.0, 10920.23962344],
            [490500.0, 11673.36025104],
            [491400.0, 11045.759999999998],
            [492300.0, 11422.319748959999],
            [493200.0, 12175.44037656],
            [494100.0, 14560.32],
            [495000.0, 11422.319748959999],
            [495900.0, 11045.759999999998],
            [496800.0, 10543.679874480002],
            [497700.0, 10920.23962344],
            [498600.0, 20083.2],
            [499500.0, 11422.319748959999],
            [500400.0, 11171.28025104],
            [501300.0, 10920.23962344],
            [502200.0, 11422.319748959999],
            [503100.0, 20836.3187448],
            [504000.0, 12426.47987448],
            [504900.0, 11673.36025104],
            [505800.0, 11673.36025104],
            [506700.0, 11422.319748959999],
            [507600.0, 21338.4],
            [508500.0, 12928.56],
            [509400.0, 11547.839999999998],
            [510300.0, 11422.319748959999],
            [511200.0, 11547.839999999998],
            [512100.0, 21087.36],
            [513000.0, 11422.319748959999],
            [513900.0, 11171.28025104],
            [514800.0, 11673.36025104],
            [515700.0, 11924.39987448],
            [516600.0, 20836.3187448],
            [517500.0, 12049.92012552],
            [518400.0, 11798.87962344],
            [519300.0, 11924.39987448],
            [520200.0, 11547.839999999998],
            [521100.0, 20710.8012552],
            [522000.0, 11296.80050208],
            [522900.0, 11798.87962344],
            [523800.0, 11422.319748959999],
            [524700.0, 11422.319748959999],
            [525600.0, 20334.24],
            [526500.0, 10543.679874480002],
            [527400.0, 11045.759999999998],
            [528300.0, 10669.200125520001],
            [529200.0, 11045.759999999998],
            [530100.0, 18953.5187448],
            [531000.0, 9790.56025104],
            [531900.0, 10292.64025104],
            [532800.0, 11673.36025104],
            [533700.0, 11422.319748959999],
            [534600.0, 19455.600000000002],
            [535500.0, 10292.64025104],
            [536400.0, 11171.28025104],
            [537300.0, 11296.80050208],
            [538200.0, 10794.72037656],
            [539100.0, 19581.12],
            [540000.0, 12049.92012552],
            [540900.0, 12049.92012552],
            [541800.0, 11924.39987448],
            [542700.0, 10794.72037656],
            [543600.0, 20083.2],
            [544500.0, 12552.0],
            [545400.0, 12175.44037656],
            [546300.0, 11924.39987448],
            [547200.0, 11171.28025104],
            [548100.0, 18953.5187448],
            [549000.0, 10920.23962344],
            [549900.0, 11422.319748959999],
            [550800.0, 11045.759999999998],
            [551700.0, 11296.80050208],
            [552600.0, 12928.56],
            [553500.0, 21840.4787448],
            [554400.0, 11798.87962344],
            [555300.0, 12049.92012552],
            [556200.0, 12049.92012552],
            [557100.0, 12928.56],
            [558000.0, 22719.1187448],
            [558900.0, 15941.04],
            [559800.0, 11422.319748959999],
            [560700.0, 11798.87962344],
            [561600.0, 11798.87962344],
            [562500.0, 11798.87962344],
            [563400.0, 20961.840000000004],
            [564300.0, 12049.92012552],
            [565200.0, 12049.92012552],
            [566100.0, 11171.28025104],
            [567000.0, 11547.839999999998],
            [567900.0, 10920.23962344],
            [568800.0, 11296.80050208],
            [569700.0, 11547.839999999998],
            [570600.0, 10794.72037656],
            [571500.0, 11422.319748959999],
            [572400.0, 10920.23962344],
            [573300.0, 19581.12],
            [574200.0, 10920.23962344],
            [575100.0, 11296.80050208],
            [576000.0, 11296.80050208],
            [576900.0, 11045.759999999998],
            [577800.0, 10920.23962344],
            [578700.0, 15062.399999999998],
            [579600.0, 11045.759999999998],
            [580500.0, 11296.80050208],
            [581400.0, 11171.28025104],
            [582300.0, 11171.28025104],
            [583200.0, 15564.48],
            [584100.0, 12928.56],
            [585000.0, 11045.759999999998],
            [585900.0, 11171.28025104],
            [586800.0, 10669.200125520001],
            [587700.0, 10669.200125520001],
            [588600.0, 20083.2],
            [589500.0, 11547.839999999998],
            [590400.0, 11296.80050208],
            [591300.0, 12049.92012552],
            [592200.0, 20459.760000000002],
            [593100.0, 20459.760000000002],
            [594000.0, 11673.36025104],
            [594900.0, 12300.95962344],
            [595800.0, 12175.44037656],
            [596700.0, 21966.0],
            [597600.0, 13932.720000000001],
            [598500.0, 13932.720000000001],
            [599400.0, 12300.95962344],
            [600300.0, 12049.92012552],
            [601200.0, 15690.0],
            [602100.0, 21589.4412552],
            [603000.0, 11924.39987448],
            [603900.0, 11924.39987448],
        ],
        "point_3": [
            [0.0, 3640.07987448],
            [900.0, 3514.56012552],
            [1800.0, 3891.11987448],
            [2700.0, 3640.07987448],
            [3600.0, 4393.2],
            [4500.0, 4393.2],
            [5400.0, 3640.07987448],
            [6300.0, 4142.16],
            [7200.0, 3891.11987448],
            [8100.0, 4142.16],
            [9000.0, 3891.11987448],
            [9900.0, 3891.11987448],
            [10800.0, 4142.16],
            [11700.0, 4016.6402510400003],
            [12600.0, 3891.11987448],
            [13500.0, 3891.11987448],
            [14400.0, 3640.07987448],
            [15300.0, 4267.68025104],
            [16200.0, 4267.68025104],
            [17100.0, 3891.11987448],
            [18000.0, 4142.16],
            [18900.0, 10418.159623440002],
            [19800.0, 8284.32],
            [20700.0, 8033.28037656],
            [21600.0, 8033.28037656],
            [22500.0, 13305.119999999999],
            [23400.0, 1882.7999999999997],
            [24300.0, 7782.23987448],
            [25200.0, 7405.68],
            [26100.0, 7405.68],
            [27000.0, 7405.68],
            [27900.0, 7029.12025104],
            [28800.0, 7405.68],
            [29700.0, 6150.479874480001],
            [30600.0, 7405.68],
            [31500.0, 5773.919999999999],
            [32400.0, 5773.919999999999],
            [33300.0, 6527.0401255199995],
            [34200.0, 5522.879999999999],
            [35100.0, 5397.36012552],
            [36000.0, 5271.839874480001],
            [36900.0, 4518.71974896],
            [37800.0, 4518.71974896],
            [38700.0, 4393.2],
            [39600.0, 4769.759874480001],
            [40500.0, 5146.32012552],
            [41400.0, 4644.24],
            [42300.0, 4518.71974896],
            [43200.0, 4518.71974896],
            [44100.0, 4644.24],
            [45000.0, 4016.6402510400003],
            [45900.0, 4393.2],
            [46800.0, 4016.6402510400003],
            [47700.0, 4769.759874480001],
            [48600.0, 3891.11987448],
            [49500.0, 3891.11987448],
            [50400.0, 4016.6402510400003],
            [51300.0, 7656.72012552],
            [52200.0, 3640.07987448],
            [53100.0, 4142.16],
            [54000.0, 3640.07987448],
            [54900.0, 3640.07987448],
            [55800.0, 4267.68025104],
            [56700.0, 3640.07987448],
            [57600.0, 3765.59987448],
            [58500.0, 3514.56012552],
            [59400.0, 3640.07987448],
            [60300.0, 3640.07987448],
            [61200.0, 3765.59987448],
            [62100.0, 3765.59987448],
            [63000.0, 3891.11987448],
            [63900.0, 3891.11987448],
            [64800.0, 3891.11987448],
            [65700.0, 3891.11987448],
            [66600.0, 3765.59987448],
            [67500.0, 3891.11987448],
            [68400.0, 4016.6402510400003],
            [69300.0, 4393.2],
            [70200.0, 3891.11987448],
            [71100.0, 3891.11987448],
            [72000.0, 4142.16],
            [72900.0, 125.52],
            [73800.0, 1631.7599999999998],
            [74700.0, 2384.87987448],
            [75600.0, 2008.3201255200001],
            [76500.0, 2008.3201255200001],
            [77400.0, 2510.4],
            [78300.0, 2008.3201255200001],
            [79200.0, 2635.92],
            [80100.0, 2635.92],
            [81000.0, 2635.92],
            [81900.0, 2008.3201255200001],
            [82800.0, 2008.3201255200001],
            [83700.0, 2635.92],
            [84600.0, 2510.4],
            [85500.0, 2635.92],
            [86400.0, 2510.4],
            [87300.0, 2510.4],
            [88200.0, 2510.4],
            [89100.0, 2510.4],
            [90000.0, 2761.4399999999996],
            [90900.0, 2384.87987448],
            [91800.0, 2635.92],
            [92700.0, 2510.4],
            [93600.0, 2510.4],
            [94500.0, 2761.4399999999996],
            [95400.0, 2510.4],
            [96300.0, 2886.9599999999996],
            [97200.0, 2635.92],
            [98100.0, 2886.9599999999996],
            [99000.0, 2886.9599999999996],
            [99900.0, 2761.4399999999996],
            [100800.0, 3138.0],
            [101700.0, 2886.9599999999996],
            [102600.0, 3263.52012552],
            [103500.0, 3138.0],
            [104400.0, 3138.0],
            [105300.0, 3640.07987448],
            [106200.0, 6652.55987448],
            [107100.0, 6903.599999999999],
            [108000.0, 6527.0401255199995],
            [108900.0, 12677.52],
            [109800.0, 12677.52],
            [110700.0, 6652.55987448],
            [111600.0, 7405.68],
            [112500.0, 6652.55987448],
            [113400.0, 7154.639999999999],
            [114300.0, 6903.599999999999],
            [115200.0, 6903.599999999999],
            [116100.0, 7405.68],
            [117000.0, 7907.76012552],
            [117900.0, 7029.12025104],
            [118800.0, 6276.00012552],
            [119700.0, 5773.919999999999],
            [120600.0, 5773.919999999999],
            [121500.0, 4895.28012552],
            [122400.0, 4518.71974896],
            [123300.0, 4142.16],
            [124200.0, 4769.759874480001],
            [125100.0, 5271.839874480001],
            [126000.0, 5271.839874480001],
            [126900.0, 4518.71974896],
            [127800.0, 4895.28012552],
            [128700.0, 4518.71974896],
            [129600.0, 4769.759874480001],
            [130500.0, 4518.71974896],
            [131400.0, 4518.71974896],
            [132300.0, 4518.71974896],
            [133200.0, 4142.16],
            [134100.0, 3891.11987448],
            [135000.0, 4142.16],
            [135900.0, 3891.11987448],
            [136800.0, 4142.16],
            [137700.0, 4142.16],
            [138600.0, 3389.04012552],
            [139500.0, 3765.59987448],
            [140400.0, 3640.07987448],
            [141300.0, 3765.59987448],
            [142200.0, 3765.59987448],
            [143100.0, 3765.59987448],
            [144000.0, 3765.59987448],
            [144900.0, 6150.479874480001],
            [145800.0, 4769.759874480001],
            [146700.0, 4142.16],
            [147600.0, 4267.68025104],
            [148500.0, 4267.68025104],
            [149400.0, 4393.2],
            [150300.0, 5146.32012552],
            [151200.0, 4895.28012552],
            [152100.0, 3389.04012552],
            [153000.0, 4267.68025104],
            [153900.0, 4267.68025104],
            [154800.0, 4267.68025104],
            [155700.0, 4895.28012552],
            [156600.0, 4016.6402510400003],
            [157500.0, 4895.28012552],
            [158400.0, 4267.68025104],
            [159300.0, 4267.68025104],
            [160200.0, 2510.4],
            [161100.0, 2133.84012552],
            [162000.0, 2133.84012552],
            [162900.0, 2259.35987448],
            [163800.0, 2384.87987448],
            [164700.0, 2384.87987448],
            [165600.0, 2259.35987448],
            [166500.0, 2259.35987448],
            [167400.0, 2761.4399999999996],
            [168300.0, 2635.92],
            [169200.0, 2761.4399999999996],
            [170100.0, 2761.4399999999996],
            [171000.0, 2635.92],
            [171900.0, 2635.92],
            [172800.0, 2635.92],
            [173700.0, 2886.9599999999996],
            [174600.0, 2635.92],
            [175500.0, 2635.92],
            [176400.0, 2886.9599999999996],
            [177300.0, 2886.9599999999996],
            [178200.0, 2635.92],
            [179100.0, 2635.92],
            [180000.0, 2635.92],
            [180900.0, 2635.92],
            [181800.0, 2635.92],
            [182700.0, 2761.4399999999996],
            [183600.0, 2635.92],
            [184500.0, 2761.4399999999996],
            [185400.0, 2510.4],
            [186300.0, 2886.9599999999996],
            [187200.0, 2886.9599999999996],
            [188100.0, 2635.92],
            [189000.0, 2886.9599999999996],
            [189900.0, 2635.92],
            [190800.0, 2886.9599999999996],
            [191700.0, 4393.2],
            [192600.0, 4393.2],
            [193500.0, 4644.24],
            [194400.0, 4016.6402510400003],
            [195300.0, 8786.4],
            [196200.0, 5397.36012552],
            [197100.0, 4895.28012552],
            [198000.0, 4895.28012552],
            [198900.0, 4895.28012552],
            [199800.0, 4644.24],
            [200700.0, 5146.32012552],
            [201600.0, 5146.32012552],
            [202500.0, 4895.28012552],
            [203400.0, 4895.28012552],
            [204300.0, 4769.759874480001],
            [205200.0, 4393.2],
            [206100.0, 7656.72012552],
            [207000.0, 5648.40025104],
            [207900.0, 5146.32012552],
            [208800.0, 5146.32012552],
            [209700.0, 5271.839874480001],
            [210600.0, 3891.11987448],
            [211500.0, 3765.59987448],
            [212400.0, 3891.11987448],
            [213300.0, 3891.11987448],
            [214200.0, 3891.11987448],
            [215100.0, 3514.56012552],
            [216000.0, 3514.56012552],
            [216900.0, 3891.11987448],
            [217800.0, 3514.56012552],
            [218700.0, 3640.07987448],
            [219600.0, 3640.07987448],
            [220500.0, 3514.56012552],
            [221400.0, 3514.56012552],
            [222300.0, 4016.6402510400003],
            [223200.0, 3514.56012552],
            [224100.0, 4142.16],
            [225000.0, 3514.56012552],
            [225900.0, 3514.56012552],
            [226800.0, 4393.2],
            [227700.0, 4016.6402510400003],
            [228600.0, 4518.71974896],
            [229500.0, 3891.11987448],
            [230400.0, 4393.2],
            [231300.0, 4393.2],
            [232200.0, 4518.71974896],
            [233100.0, 4267.68025104],
            [234000.0, 7907.76012552],
            [234900.0, 5522.879999999999],
            [235800.0, 5271.839874480001],
            [236700.0, 5271.839874480001],
            [237600.0, 5146.32012552],
            [238500.0, 2259.35987448],
            [239400.0, 4016.6402510400003],
            [240300.0, 4895.28012552],
            [241200.0, 4895.28012552],
            [242100.0, 4895.28012552],
            [243000.0, 5522.879999999999],
            [243900.0, 5146.32012552],
            [244800.0, 5648.40025104],
            [245700.0, 1757.28],
            [246600.0, 2384.87987448],
            [247500.0, 2384.87987448],
            [248400.0, 3263.52012552],
            [249300.0, 2886.9599999999996],
            [250200.0, 2761.4399999999996],
            [251100.0, 3012.48],
            [252000.0, 3138.0],
            [252900.0, 3138.0],
            [253800.0, 3389.04012552],
            [254700.0, 3389.04012552],
            [255600.0, 3514.56012552],
            [256500.0, 3640.07987448],
            [257400.0, 3514.56012552],
            [258300.0, 3514.56012552],
            [259200.0, 4016.6402510400003],
            [260100.0, 4142.16],
            [261000.0, 4016.6402510400003],
            [261900.0, 4142.16],
            [262800.0, 4142.16],
            [263700.0, 4142.16],
            [264600.0, 4142.16],
            [265500.0, 4518.71974896],
            [266400.0, 4393.2],
            [267300.0, 4267.68025104],
            [268200.0, 4267.68025104],
            [269100.0, 4895.28012552],
            [270000.0, 4895.28012552],
            [270900.0, 4644.24],
            [271800.0, 4895.28012552],
            [272700.0, 4393.2],
            [273600.0, 4518.71974896],
            [274500.0, 4267.68025104],
            [275400.0, 4267.68025104],
            [276300.0, 4769.759874480001],
            [277200.0, 4895.28012552],
            [278100.0, 7656.72012552],
            [279000.0, 7782.23987448],
            [279900.0, 7782.23987448],
            [280800.0, 7782.23987448],
            [281700.0, 7405.68],
            [282600.0, 13932.720000000001],
            [283500.0, 11673.36025104],
            [284400.0, 125.52],
            [285300.0, 7656.72012552],
            [286200.0, 7656.72012552],
            [287100.0, 7656.72012552],
            [288000.0, 7154.639999999999],
            [288900.0, 7154.639999999999],
            [289800.0, 7154.639999999999],
            [290700.0, 6903.599999999999],
            [291600.0, 5899.4397489600005],
            [292500.0, 6150.479874480001],
            [293400.0, 5146.32012552],
            [294300.0, 5146.32012552],
            [295200.0, 4142.16],
            [296100.0, 4142.16],
            [297000.0, 4518.71974896],
            [297900.0, 5020.799874480001],
            [298800.0, 5271.839874480001],
            [299700.0, 5271.839874480001],
            [300600.0, 4895.28012552],
            [301500.0, 5020.799874480001],
            [302400.0, 4895.28012552],
            [303300.0, 5522.879999999999],
            [304200.0, 5146.32012552],
            [305100.0, 5146.32012552],
            [306000.0, 4895.28012552],
            [306900.0, 4267.68025104],
            [307800.0, 4267.68025104],
            [308700.0, 4267.68025104],
            [309600.0, 4769.759874480001],
            [310500.0, 4769.759874480001],
            [311400.0, 4142.16],
            [312300.0, 4016.6402510400003],
            [313200.0, 4769.759874480001],
            [314100.0, 4142.16],
            [315000.0, 5020.799874480001],
            [315900.0, 4518.71974896],
            [316800.0, 4518.71974896],
            [317700.0, 4518.71974896],
            [318600.0, 4769.759874480001],
            [319500.0, 4644.24],
            [320400.0, 5397.36012552],
            [321300.0, 5522.879999999999],
            [322200.0, 5522.879999999999],
            [323100.0, 6276.00012552],
            [324000.0, 5522.879999999999],
            [324900.0, 5397.36012552],
            [325800.0, 6401.51987448],
            [326700.0, 5648.40025104],
            [327600.0, 5648.40025104],
            [328500.0, 5773.919999999999],
            [329400.0, 5899.4397489600005],
            [330300.0, 6024.96],
            [331200.0, 6778.0801255199995],
            [332100.0, 376.56],
            [333000.0, 376.56],
            [333900.0, 3138.0],
            [334800.0, 2886.9599999999996],
            [335700.0, 3138.0],
            [336600.0, 3012.48],
            [337500.0, 3012.48],
            [338400.0, 3012.48],
            [339300.0, 3389.04012552],
            [340200.0, 3138.0],
            [341100.0, 3138.0],
            [342000.0, 3012.48],
            [342900.0, 3138.0],
            [343800.0, 3138.0],
            [344700.0, 3640.07987448],
            [345600.0, 3514.56012552],
            [346500.0, 3389.04012552],
            [347400.0, 3012.48],
            [348300.0, 3012.48],
            [349200.0, 3012.48],
            [350100.0, 3138.0],
            [351000.0, 3012.48],
            [351900.0, 3263.52012552],
            [352800.0, 3389.04012552],
            [353700.0, 3514.56012552],
            [354600.0, 3389.04012552],
            [355500.0, 3389.04012552],
            [356400.0, 3640.07987448],
            [357300.0, 3765.59987448],
            [358200.0, 3765.59987448],
            [359100.0, 3640.07987448],
            [360000.0, 3514.56012552],
            [360900.0, 3514.56012552],
            [361800.0, 3389.04012552],
            [362700.0, 3514.56012552],
            [363600.0, 3263.52012552],
            [364500.0, 8033.28037656],
            [365400.0, 7656.72012552],
            [366300.0, 7656.72012552],
            [367200.0, 7280.15974896],
            [368100.0, 6903.599999999999],
            [369000.0, 6652.55987448],
            [369900.0, 6527.0401255199995],
            [370800.0, 6401.51987448],
            [371700.0, 6401.51987448],
            [372600.0, 8033.28037656],
            [373500.0, 6276.00012552],
            [374400.0, 5648.40025104],
            [375300.0, 5146.32012552],
            [376200.0, 5146.32012552],
            [377100.0, 5899.4397489600005],
            [378000.0, 5146.32012552],
            [378900.0, 5648.40025104],
            [379800.0, 5146.32012552],
            [380700.0, 5899.4397489600005],
            [381600.0, 5899.4397489600005],
            [382500.0, 5146.32012552],
            [383400.0, 5146.32012552],
            [384300.0, 4895.28012552],
            [385200.0, 4895.28012552],
            [386100.0, 4393.2],
            [387000.0, 4393.2],
            [387900.0, 5020.799874480001],
            [388800.0, 4518.71974896],
            [389700.0, 4644.24],
            [390600.0, 4644.24],
            [391500.0, 4518.71974896],
            [392400.0, 4518.71974896],
            [393300.0, 4518.71974896],
            [394200.0, 4016.6402510400003],
            [395100.0, 4644.24],
            [396000.0, 4142.16],
            [396900.0, 4393.2],
            [397800.0, 4769.759874480001],
            [398700.0, 4769.759874480001],
            [399600.0, 4267.68025104],
            [400500.0, 4644.24],
            [401400.0, 4518.71974896],
            [402300.0, 4769.759874480001],
            [403200.0, 4644.24],
            [404100.0, 4644.24],
            [405000.0, 4644.24],
            [405900.0, 4769.759874480001],
            [406800.0, 5020.799874480001],
            [407700.0, 5146.32012552],
            [408600.0, 5397.36012552],
            [409500.0, 5397.36012552],
            [410400.0, 5020.799874480001],
            [411300.0, 5271.839874480001],
            [412200.0, 5146.32012552],
            [413100.0, 5773.919999999999],
            [414000.0, 4895.28012552],
            [414900.0, 4895.28012552],
            [415800.0, 5146.32012552],
            [416700.0, 5271.839874480001],
            [417600.0, 5522.879999999999],
            [418500.0, 1506.24],
            [419400.0, 1757.28],
            [420300.0, 1757.28],
            [421200.0, 2133.84012552],
            [422100.0, 2510.4],
            [423000.0, 2259.35987448],
            [423900.0, 2761.4399999999996],
            [424800.0, 2259.35987448],
            [425700.0, 2259.35987448],
            [426600.0, 2761.4399999999996],
            [427500.0, 2635.92],
            [428400.0, 2510.4],
            [429300.0, 2635.92],
            [430200.0, 2635.92],
            [431100.0, 2635.92],
            [432000.0, 2761.4399999999996],
            [432900.0, 2761.4399999999996],
            [433800.0, 2886.9599999999996],
            [434700.0, 2635.92],
            [435600.0, 2886.9599999999996],
            [436500.0, 2886.9599999999996],
            [437400.0, 2635.92],
            [438300.0, 3012.48],
            [439200.0, 2635.92],
            [440100.0, 2886.9599999999996],
            [441000.0, 2635.92],
            [441900.0, 2635.92],
            [442800.0, 3138.0],
            [443700.0, 2635.92],
            [444600.0, 2886.9599999999996],
            [445500.0, 2761.4399999999996],
            [446400.0, 3012.48],
            [447300.0, 3012.48],
            [448200.0, 2886.9599999999996],
            [449100.0, 3012.48],
            [450000.0, 2886.9599999999996],
            [450900.0, 7280.15974896],
            [451800.0, 5899.4397489600005],
            [452700.0, 5899.4397489600005],
            [453600.0, 6652.55987448],
            [454500.0, 13430.64],
            [455400.0, 7782.23987448],
            [456300.0, 6024.96],
            [457200.0, 6024.96],
            [458100.0, 6024.96],
            [459000.0, 5522.879999999999],
            [459900.0, 6401.51987448],
            [460800.0, 5899.4397489600005],
            [461700.0, 5899.4397489600005],
            [462600.0, 5899.4397489600005],
            [463500.0, 6024.96],
            [464400.0, 5397.36012552],
            [465300.0, 5397.36012552],
            [466200.0, 5899.4397489600005],
            [467100.0, 5522.879999999999],
            [468000.0, 5648.40025104],
            [468900.0, 4644.24],
            [469800.0, 5397.36012552],
            [470700.0, 5397.36012552],
            [471600.0, 5146.32012552],
            [472500.0, 5146.32012552],
            [473400.0, 5146.32012552],
            [474300.0, 5146.32012552],
            [475200.0, 5146.32012552],
            [476100.0, 4644.24],
            [477000.0, 4644.24],
            [477900.0, 4644.24],
            [478800.0, 4518.71974896],
            [479700.0, 4518.71974896],
            [480600.0, 4267.68025104],
            [481500.0, 4142.16],
            [482400.0, 4142.16],
            [483300.0, 4142.16],
            [484200.0, 4267.68025104],
            [485100.0, 4518.71974896],
            [486000.0, 4267.68025104],
            [486900.0, 4267.68025104],
            [487800.0, 3891.11987448],
            [488700.0, 4142.16],
            [489600.0, 4142.16],
            [490500.0, 4142.16],
            [491400.0, 3891.11987448],
            [492300.0, 4895.28012552],
            [493200.0, 4142.16],
            [494100.0, 4644.24],
            [495000.0, 4644.24],
            [495900.0, 6652.55987448],
            [496800.0, 4644.24],
            [497700.0, 4644.24],
            [498600.0, 4769.759874480001],
            [499500.0, 4769.759874480001],
            [500400.0, 4769.759874480001],
            [501300.0, 4769.759874480001],
            [502200.0, 5146.32012552],
            [503100.0, 4769.759874480001],
            [504000.0, 5271.839874480001],
            [504900.0, 4895.28012552],
            [505800.0, 4895.28012552],
            [506700.0, 5146.32012552],
            [507600.0, 4895.28012552],
            [508500.0, 125.52],
            [509400.0, 1506.24],
            [510300.0, 2635.92],
            [511200.0, 2635.92],
            [512100.0, 2384.87987448],
            [513000.0, 2635.92],
            [513900.0, 2384.87987448],
            [514800.0, 2635.92],
            [515700.0, 2384.87987448],
            [516600.0, 2384.87987448],
            [517500.0, 2761.4399999999996],
            [518400.0, 2635.92],
            [519300.0, 2886.9599999999996],
            [520200.0, 2761.4399999999996],
            [521100.0, 2886.9599999999996],
            [522000.0, 2886.9599999999996],
            [522900.0, 2761.4399999999996],
            [523800.0, 3012.48],
            [524700.0, 2761.4399999999996],
            [525600.0, 2886.9599999999996],
            [526500.0, 2886.9599999999996],
            [527400.0, 2886.9599999999996],
            [528300.0, 3012.48],
            [529200.0, 2635.92],
            [530100.0, 3012.48],
            [531000.0, 2635.92],
            [531900.0, 3012.48],
            [532800.0, 3012.48],
            [533700.0, 3012.48],
            [534600.0, 2886.9599999999996],
            [535500.0, 3012.48],
            [536400.0, 3012.48],
            [537300.0, 3012.48],
            [538200.0, 3012.48],
            [539100.0, 3012.48],
            [540000.0, 3138.0],
            [540900.0, 10418.159623440002],
            [541800.0, 6652.55987448],
            [542700.0, 5773.919999999999],
            [543600.0, 6150.479874480001],
            [544500.0, 6150.479874480001],
            [545400.0, 125.52],
            [546300.0, 6276.00012552],
            [547200.0, 5397.36012552],
            [548100.0, 6401.51987448],
            [549000.0, 5397.36012552],
            [549900.0, 5397.36012552],
            [550800.0, 6150.479874480001],
            [551700.0, 5899.4397489600005],
            [552600.0, 5773.919999999999],
            [553500.0, 5773.919999999999],
            [554400.0, 5773.919999999999],
            [555300.0, 5397.36012552],
            [556200.0, 6024.96],
            [557100.0, 6024.96],
            [558000.0, 5020.799874480001],
            [558900.0, 5020.799874480001],
            [559800.0, 5020.799874480001],
            [560700.0, 5020.799874480001],
            [561600.0, 4769.759874480001],
            [562500.0, 4769.759874480001],
            [563400.0, 4518.71974896],
            [564300.0, 4518.71974896],
            [565200.0, 4895.28012552],
            [566100.0, 4518.71974896],
            [567000.0, 4895.28012552],
            [567900.0, 4644.24],
            [568800.0, 4769.759874480001],
            [569700.0, 4142.16],
            [570600.0, 4142.16],
            [571500.0, 4393.2],
            [572400.0, 4267.68025104],
            [573300.0, 4016.6402510400003],
            [574200.0, 4267.68025104],
            [575100.0, 4267.68025104],
            [576000.0, 3640.07987448],
            [576900.0, 4393.2],
            [577800.0, 3891.11987448],
            [578700.0, 4518.71974896],
            [579600.0, 4267.68025104],
            [580500.0, 4769.759874480001],
            [581400.0, 4769.759874480001],
            [582300.0, 4016.6402510400003],
            [583200.0, 125.52],
            [584100.0, 4267.68025104],
            [585000.0, 4644.24],
            [585900.0, 4518.71974896],
            [586800.0, 4518.71974896],
            [587700.0, 4895.28012552],
            [588600.0, 4644.24],
            [589500.0, 4895.28012552],
            [590400.0, 4769.759874480001],
            [591300.0, 4895.28012552],
            [592200.0, 4895.28012552],
            [593100.0, 5020.799874480001],
            [594000.0, 4895.28012552],
            [594900.0, 251.040012552],
            [595800.0, 2008.3201255200001],
            [596700.0, 2510.4],
            [597600.0, 2510.4],
            [598500.0, 2510.4],
            [599400.0, 2635.92],
            [600300.0, 2635.92],
            [601200.0, 2761.4399999999996],
            [602100.0, 3263.52012552],
            [603000.0, 3263.52012552],
            [603900.0, 3012.48],
        ],
        "point_4": [
            [0.0, 1631.7599999999998],
            [900.0, 1506.24],
            [1800.0, 1506.24],
            [2700.0, 1380.7199999999998],
            [3600.0, 1380.7199999999998],
            [4500.0, 1506.24],
            [5400.0, 1506.24],
            [6300.0, 1631.7599999999998],
            [7200.0, 1631.7599999999998],
            [8100.0, 1506.24],
            [9000.0, 1506.24],
            [9900.0, 1380.7199999999998],
            [10800.0, 1506.24],
            [11700.0, 1506.24],
            [12600.0, 1506.24],
            [13500.0, 1255.199974896],
            [14400.0, 1380.7199999999998],
            [15300.0, 1380.7199999999998],
            [16200.0, 1380.7199999999998],
            [17100.0, 1380.7199999999998],
            [18000.0, 1380.7199999999998],
            [18900.0, 1506.24],
            [19800.0, 2635.92],
            [20700.0, 2635.92],
            [21600.0, 2761.4399999999996],
            [22500.0, 2510.4],
            [23400.0, 2510.4],
            [24300.0, 4393.2],
            [25200.0, 3514.56012552],
            [26100.0, 3514.56012552],
            [27000.0, 4016.6402510400003],
            [27900.0, 4016.6402510400003],
            [28800.0, 3514.56012552],
            [29700.0, 8535.36050208],
            [30600.0, 0.0],
            [31500.0, 0.0],
            [32400.0, 6778.0801255199995],
            [33300.0, 5522.879999999999],
            [34200.0, 5397.36012552],
            [35100.0, 2635.92],
            [36000.0, 1380.7199999999998],
            [36900.0, 1380.7199999999998],
            [37800.0, 1129.679949792],
            [38700.0, 1757.28],
            [39600.0, 1757.28],
            [40500.0, 1631.7599999999998],
            [41400.0, 3138.0],
            [42300.0, 3138.0],
            [43200.0, 2510.4],
            [44100.0, 2259.35987448],
            [45000.0, 2133.84012552],
            [45900.0, 1631.7599999999998],
            [46800.0, 1882.7999999999997],
            [47700.0, 1882.7999999999997],
            [48600.0, 1882.7999999999997],
            [49500.0, 1757.28],
            [50400.0, 1631.7599999999998],
            [51300.0, 1631.7599999999998],
            [52200.0, 1757.28],
            [53100.0, 1757.28],
            [54000.0, 1757.28],
            [54900.0, 1757.28],
            [55800.0, 1757.28],
            [56700.0, 1631.7599999999998],
            [57600.0, 2008.3201255200001],
            [58500.0, 1882.7999999999997],
            [59400.0, 1882.7999999999997],
            [60300.0, 1631.7599999999998],
            [61200.0, 2133.84012552],
            [62100.0, 1757.28],
            [63000.0, 1757.28],
            [63900.0, 2510.4],
            [64800.0, 2510.4],
            [65700.0, 1380.7199999999998],
            [66600.0, 0.0],
            [67500.0, 0.0],
            [68400.0, 0.0],
            [69300.0, 0.0],
            [70200.0, 0.0],
            [71100.0, 0.0],
            [72000.0, 125.52],
            [72900.0, 0.0],
            [73800.0, 0.0],
            [74700.0, 0.0],
            [75600.0, 0.0],
            [76500.0, 0.0],
            [77400.0, 0.0],
            [78300.0, 0.0],
            [79200.0, 0.0],
            [80100.0, 125.52],
            [81000.0, 0.0],
            [81900.0, 0.0],
            [82800.0, 0.0],
            [83700.0, 0.0],
            [84600.0, 0.0],
            [85500.0, 0.0],
            [86400.0, 0.0],
            [87300.0, 0.0],
            [88200.0, 0.0],
            [89100.0, 125.52],
            [90000.0, 0.0],
            [90900.0, 125.52],
            [91800.0, 0.0],
            [92700.0, 0.0],
            [93600.0, 0.0],
            [94500.0, 125.52],
            [95400.0, 125.52],
            [96300.0, 0.0],
            [97200.0, 125.52],
            [98100.0, 125.52],
            [99000.0, 125.52],
            [99900.0, 125.52],
            [100800.0, 125.52],
            [101700.0, 125.52],
            [102600.0, 125.52],
            [103500.0, 125.52],
            [104400.0, 0.0],
            [105300.0, 2635.92],
            [106200.0, 2635.92],
            [107100.0, 3012.48],
            [108000.0, 3012.48],
            [108900.0, 3012.48],
            [109800.0, 7405.68],
            [110700.0, 4644.24],
            [111600.0, 8409.840251040001],
            [112500.0, 8409.840251040001],
            [113400.0, 4644.24],
            [114300.0, 4644.24],
            [115200.0, 4644.24],
            [116100.0, 8033.28037656],
            [117000.0, 4393.2],
            [117900.0, 5773.919999999999],
            [118800.0, 5773.919999999999],
            [119700.0, 5773.919999999999],
            [120600.0, 4142.16],
            [121500.0, 3263.52012552],
            [122400.0, 2133.84012552],
            [123300.0, 1882.7999999999997],
            [124200.0, 2635.92],
            [125100.0, 2635.92],
            [126000.0, 1631.7599999999998],
            [126900.0, 1631.7599999999998],
            [127800.0, 2008.3201255200001],
            [128700.0, 2886.9599999999996],
            [129600.0, 0.0],
            [130500.0, 0.0],
            [131400.0, 0.0],
            [132300.0, 0.0],
            [133200.0, 125.52],
            [134100.0, 125.52],
            [135000.0, 125.52],
            [135900.0, 0.0],
            [136800.0, 0.0],
            [137700.0, 0.0],
            [138600.0, 0.0],
            [139500.0, 0.0],
            [140400.0, 0.0],
            [141300.0, 0.0],
            [142200.0, 0.0],
            [143100.0, 0.0],
            [144000.0, 0.0],
            [144900.0, 0.0],
            [145800.0, 0.0],
            [146700.0, 0.0],
            [147600.0, 0.0],
            [148500.0, 0.0],
            [149400.0, 0.0],
            [150300.0, 0.0],
            [151200.0, 125.52],
            [152100.0, 125.52],
            [153000.0, 125.52],
            [153900.0, 125.52],
            [154800.0, 0.0],
            [155700.0, 125.52],
            [156600.0, 0.0],
            [157500.0, 0.0],
            [158400.0, 0.0],
            [159300.0, 0.0],
            [160200.0, 0.0],
            [161100.0, 0.0],
            [162000.0, 125.52],
            [162900.0, 0.0],
            [163800.0, 0.0],
            [164700.0, 0.0],
            [165600.0, 125.52],
            [166500.0, 125.52],
            [167400.0, 0.0],
            [168300.0, 0.0],
            [169200.0, 0.0],
            [170100.0, 125.52],
            [171000.0, 0.0],
            [171900.0, 0.0],
            [172800.0, 0.0],
            [173700.0, 0.0],
            [174600.0, 0.0],
            [175500.0, 0.0],
            [176400.0, 125.52],
            [177300.0, 125.52],
            [178200.0, 125.52],
            [179100.0, 125.52],
            [180000.0, 0.0],
            [180900.0, 0.0],
            [181800.0, 125.52],
            [182700.0, 125.52],
            [183600.0, 125.52],
            [184500.0, 0.0],
            [185400.0, 0.0],
            [186300.0, 0.0],
            [187200.0, 125.52],
            [188100.0, 0.0],
            [189000.0, 0.0],
            [189900.0, 0.0],
            [190800.0, 0.0],
            [191700.0, 0.0],
            [192600.0, 0.0],
            [193500.0, 125.52],
            [194400.0, 125.52],
            [195300.0, 10167.12],
            [196200.0, 2510.4],
            [197100.0, 2510.4],
            [198000.0, 2510.4],
            [198900.0, 2886.9599999999996],
            [199800.0, 3012.48],
            [200700.0, 2761.4399999999996],
            [201600.0, 2761.4399999999996],
            [202500.0, 2761.4399999999996],
            [203400.0, 2635.92],
            [204300.0, 2635.92],
            [205200.0, 2635.92],
            [206100.0, 2259.35987448],
            [207000.0, 2384.87987448],
            [207900.0, 2384.87987448],
            [208800.0, 2259.35987448],
            [209700.0, 2259.35987448],
            [210600.0, 1129.679949792],
            [211500.0, 1506.24],
            [212400.0, 1506.24],
            [213300.0, 1506.24],
            [214200.0, 1129.679949792],
            [215100.0, 1380.7199999999998],
            [216000.0, 0.0],
            [216900.0, 125.52],
            [217800.0, 0.0],
            [218700.0, 0.0],
            [219600.0, 0.0],
            [220500.0, 0.0],
            [221400.0, 125.52],
            [222300.0, 125.52],
            [223200.0, 125.52],
            [224100.0, 0.0],
            [225000.0, 0.0],
            [225900.0, 0.0],
            [226800.0, 0.0],
            [227700.0, 0.0],
            [228600.0, 125.52],
            [229500.0, 125.52],
            [230400.0, 125.52],
            [231300.0, 0.0],
            [232200.0, 0.0],
            [233100.0, 125.52],
            [234000.0, 125.52],
            [234900.0, 0.0],
            [235800.0, 0.0],
            [236700.0, 0.0],
            [237600.0, 125.52],
            [238500.0, 125.52],
            [239400.0, 0.0],
            [240300.0, 0.0],
            [241200.0, 0.0],
            [242100.0, 0.0],
            [243000.0, 125.52],
            [243900.0, 19330.08],
            [244800.0, 19330.08],
            [245700.0, 1506.24],
            [246600.0, 1506.24],
            [247500.0, 1506.24],
            [248400.0, 1631.7599999999998],
            [249300.0, 1631.7599999999998],
            [250200.0, 1631.7599999999998],
            [251100.0, 1631.7599999999998],
            [252000.0, 1631.7599999999998],
            [252900.0, 1757.28],
            [253800.0, 1757.28],
            [254700.0, 1631.7599999999998],
            [255600.0, 1631.7599999999998],
            [256500.0, 1631.7599999999998],
            [257400.0, 1631.7599999999998],
            [258300.0, 1631.7599999999998],
            [259200.0, 1631.7599999999998],
            [260100.0, 1757.28],
            [261000.0, 1757.28],
            [261900.0, 1757.28],
            [262800.0, 1757.28],
            [263700.0, 1757.28],
            [264600.0, 1882.7999999999997],
            [265500.0, 1882.7999999999997],
            [266400.0, 1882.7999999999997],
            [267300.0, 1882.7999999999997],
            [268200.0, 1757.28],
            [269100.0, 1757.28],
            [270000.0, 1882.7999999999997],
            [270900.0, 1757.28],
            [271800.0, 1757.28],
            [272700.0, 1882.7999999999997],
            [273600.0, 4267.68025104],
            [274500.0, 4267.68025104],
            [275400.0, 3640.07987448],
            [276300.0, 2761.4399999999996],
            [277200.0, 4142.16],
            [278100.0, 4142.16],
            [279000.0, 3263.52012552],
            [279900.0, 3263.52012552],
            [280800.0, 4016.6402510400003],
            [281700.0, 5522.879999999999],
            [282600.0, 11045.759999999998],
            [283500.0, 0.0],
            [284400.0, 10041.599748960001],
            [285300.0, 10041.599748960001],
            [286200.0, 0.0],
            [287100.0, 10167.12],
            [288000.0, 0.0],
            [288900.0, 0.0],
            [289800.0, 0.0],
            [290700.0, 8158.79974896],
            [291600.0, 8158.79974896],
            [292500.0, 4142.16],
            [293400.0, 4142.16],
            [294300.0, 4142.16],
            [295200.0, 3389.04012552],
            [296100.0, 3138.0],
            [297000.0, 3263.52012552],
            [297900.0, 3514.56012552],
            [298800.0, 3514.56012552],
            [299700.0, 3514.56012552],
            [300600.0, 2886.9599999999996],
            [301500.0, 4895.28012552],
            [302400.0, 4518.71974896],
            [303300.0, 4895.28012552],
            [304200.0, 4895.28012552],
            [305100.0, 3765.59987448],
            [306000.0, 3765.59987448],
            [306900.0, 3012.48],
            [307800.0, 4016.6402510400003],
            [308700.0, 3891.11987448],
            [309600.0, 1631.7599999999998],
            [310500.0, 1631.7599999999998],
            [311400.0, 3765.59987448],
            [312300.0, 1004.160050208],
            [313200.0, 1380.7199999999998],
            [314100.0, 1380.7199999999998],
            [315000.0, 2133.84012552],
            [315900.0, 2133.84012552],
            [316800.0, 1255.199974896],
            [317700.0, 1380.7199999999998],
            [318600.0, 1631.7599999999998],
            [319500.0, 1631.7599999999998],
            [320400.0, 1882.7999999999997],
            [321300.0, 1882.7999999999997],
            [322200.0, 2008.3201255200001],
            [323100.0, 1882.7999999999997],
            [324000.0, 1506.24],
            [324900.0, 0.0],
            [325800.0, 0.0],
            [326700.0, 0.0],
            [327600.0, 0.0],
            [328500.0, 0.0],
            [329400.0, 125.52],
            [330300.0, 0.0],
            [331200.0, 125.52],
            [332100.0, 125.52],
            [333000.0, 1757.28],
            [333900.0, 1631.7599999999998],
            [334800.0, 1631.7599999999998],
            [335700.0, 1506.24],
            [336600.0, 1255.199974896],
            [337500.0, 1255.199974896],
            [338400.0, 1631.7599999999998],
            [339300.0, 1631.7599999999998],
            [340200.0, 1380.7199999999998],
            [341100.0, 1380.7199999999998],
            [342000.0, 1380.7199999999998],
            [342900.0, 1380.7199999999998],
            [343800.0, 1631.7599999999998],
            [344700.0, 1631.7599999999998],
            [345600.0, 1506.24],
            [346500.0, 1380.7199999999998],
            [347400.0, 1380.7199999999998],
            [348300.0, 1380.7199999999998],
            [349200.0, 1255.199974896],
            [350100.0, 1255.199974896],
            [351000.0, 1506.24],
            [351900.0, 1380.7199999999998],
            [352800.0, 1255.199974896],
            [353700.0, 1380.7199999999998],
            [354600.0, 1380.7199999999998],
            [355500.0, 1506.24],
            [356400.0, 1506.24],
            [357300.0, 1380.7199999999998],
            [358200.0, 1380.7199999999998],
            [359100.0, 1506.24],
            [360000.0, 1506.24],
            [360900.0, 1380.7199999999998],
            [361800.0, 2510.4],
            [362700.0, 2510.4],
            [363600.0, 2259.35987448],
            [364500.0, 2008.3201255200001],
            [365400.0, 2008.3201255200001],
            [366300.0, 2510.4],
            [367200.0, 2761.4399999999996],
            [368100.0, 7656.72012552],
            [369000.0, 7656.72012552],
            [369900.0, 4267.68025104],
            [370800.0, 4267.68025104],
            [371700.0, 6903.599999999999],
            [372600.0, 5020.799874480001],
            [373500.0, 6401.51987448],
            [374400.0, 5899.4397489600005],
            [375300.0, 6401.51987448],
            [376200.0, 6401.51987448],
            [377100.0, 6401.51987448],
            [378000.0, 6024.96],
            [378900.0, 5020.799874480001],
            [379800.0, 6401.51987448],
            [380700.0, 6401.51987448],
            [381600.0, 4142.16],
            [382500.0, 4267.68025104],
            [383400.0, 5020.799874480001],
            [384300.0, 5020.799874480001],
            [385200.0, 2510.4],
            [386100.0, 2510.4],
            [387000.0, 4518.71974896],
            [387900.0, 2635.92],
            [388800.0, 3514.56012552],
            [389700.0, 2384.87987448],
            [390600.0, 2384.87987448],
            [391500.0, 2384.87987448],
            [392400.0, 1757.28],
            [393300.0, 4769.759874480001],
            [394200.0, 1506.24],
            [395100.0, 3012.48],
            [396000.0, 1757.28],
            [396900.0, 1757.28],
            [397800.0, 1757.28],
            [398700.0, 1631.7599999999998],
            [399600.0, 5146.32012552],
            [400500.0, 1882.7999999999997],
            [401400.0, 1631.7599999999998],
            [402300.0, 2510.4],
            [403200.0, 2510.4],
            [404100.0, 2510.4],
            [405000.0, 1631.7599999999998],
            [405900.0, 3138.0],
            [406800.0, 1882.7999999999997],
            [407700.0, 1631.7599999999998],
            [408600.0, 1631.7599999999998],
            [409500.0, 3891.11987448],
            [410400.0, 3891.11987448],
            [411300.0, 1882.7999999999997],
            [412200.0, 125.52],
            [413100.0, 0.0],
            [414000.0, 0.0],
            [414900.0, 125.52],
            [415800.0, 125.52],
            [416700.0, 125.52],
            [417600.0, 125.52],
            [418500.0, 0.0],
            [419400.0, 0.0],
            [420300.0, 0.0],
            [421200.0, 125.52],
            [422100.0, 0.0],
            [423000.0, 0.0],
            [423900.0, 0.0],
            [424800.0, 0.0],
            [425700.0, 0.0],
            [426600.0, 0.0],
            [427500.0, 0.0],
            [428400.0, 125.52],
            [429300.0, 125.52],
            [430200.0, 125.52],
            [431100.0, 0.0],
            [432000.0, 125.52],
            [432900.0, 125.52],
            [433800.0, 0.0],
            [434700.0, 125.52],
            [435600.0, 125.52],
            [436500.0, 0.0],
            [437400.0, 0.0],
            [438300.0, 0.0],
            [439200.0, 0.0],
            [440100.0, 125.52],
            [441000.0, 125.52],
            [441900.0, 0.0],
            [442800.0, 0.0],
            [443700.0, 0.0],
            [444600.0, 125.52],
            [445500.0, 125.52],
            [446400.0, 125.52],
            [447300.0, 125.52],
            [448200.0, 125.52],
            [449100.0, 125.52],
            [450000.0, 125.52],
            [450900.0, 753.1200125519999],
            [451800.0, 753.1200125519999],
            [452700.0, 2635.92],
            [453600.0, 2886.9599999999996],
            [454500.0, 6276.00012552],
            [455400.0, 5397.36012552],
            [456300.0, 3640.07987448],
            [457200.0, 3640.07987448],
            [458100.0, 5522.879999999999],
            [459000.0, 4267.68025104],
            [459900.0, 4267.68025104],
            [460800.0, 6024.96],
            [461700.0, 6024.96],
            [462600.0, 6024.96],
            [463500.0, 4142.16],
            [464400.0, 4142.16],
            [465300.0, 5773.919999999999],
            [466200.0, 4895.28012552],
            [467100.0, 4393.2],
            [468000.0, 4393.2],
            [468900.0, 4142.16],
            [469800.0, 4142.16],
            [470700.0, 4016.6402510400003],
            [471600.0, 4267.68025104],
            [472500.0, 4267.68025104],
            [473400.0, 4267.68025104],
            [474300.0, 6150.479874480001],
            [475200.0, 6150.479874480001],
            [476100.0, 3765.59987448],
            [477000.0, 3765.59987448],
            [477900.0, 3765.59987448],
            [478800.0, 3514.56012552],
            [479700.0, 4518.71974896],
            [480600.0, 4142.16],
            [481500.0, 3012.48],
            [482400.0, 3012.48],
            [483300.0, 1631.7599999999998],
            [484200.0, 1631.7599999999998],
            [485100.0, 3640.07987448],
            [486000.0, 1757.28],
            [486900.0, 2886.9599999999996],
            [487800.0, 3640.07987448],
            [488700.0, 3640.07987448],
            [489600.0, 1757.28],
            [490500.0, 4267.68025104],
            [491400.0, 4267.68025104],
            [492300.0, 2008.3201255200001],
            [493200.0, 1380.7199999999998],
            [494100.0, 1380.7199999999998],
            [495000.0, 3891.11987448],
            [495900.0, 2384.87987448],
            [496800.0, 2384.87987448],
            [497700.0, 5397.36012552],
            [498600.0, 2510.4],
            [499500.0, 2510.4],
            [500400.0, 1631.7599999999998],
            [501300.0, 2259.35987448],
            [502200.0, 0.0],
            [503100.0, 0.0],
            [504000.0, 0.0],
            [504900.0, 0.0],
            [505800.0, 125.52],
            [506700.0, 0.0],
            [507600.0, 0.0],
            [508500.0, 0.0],
            [509400.0, 0.0],
            [510300.0, 0.0],
            [511200.0, 0.0],
            [512100.0, 125.52],
            [513000.0, 125.52],
            [513900.0, 0.0],
            [514800.0, 0.0],
            [515700.0, 0.0],
            [516600.0, 0.0],
            [517500.0, 0.0],
            [518400.0, 125.52],
            [519300.0, 0.0],
            [520200.0, 125.52],
            [521100.0, 125.52],
            [522000.0, 0.0],
            [522900.0, 125.52],
            [523800.0, 125.52],
            [524700.0, 0.0],
            [525600.0, 0.0],
            [526500.0, 0.0],
            [527400.0, 0.0],
            [528300.0, 0.0],
            [529200.0, 125.52],
            [530100.0, 125.52],
            [531000.0, 125.52],
            [531900.0, 125.52],
            [532800.0, 125.52],
            [533700.0, 0.0],
            [534600.0, 0.0],
            [535500.0, 125.52],
            [536400.0, 125.52],
            [537300.0, 125.52],
            [538200.0, 125.52],
            [539100.0, 125.52],
            [540000.0, 0.0],
            [540900.0, 8535.36050208],
            [541800.0, 3012.48],
            [542700.0, 3012.48],
            [543600.0, 3012.48],
            [544500.0, 2761.4399999999996],
            [545400.0, 6527.0401255199995],
            [546300.0, 3765.59987448],
            [547200.0, 4142.16],
            [548100.0, 3765.59987448],
            [549000.0, 3765.59987448],
            [549900.0, 5020.799874480001],
            [550800.0, 5020.799874480001],
            [551700.0, 5522.879999999999],
            [552600.0, 5522.879999999999],
            [553500.0, 7907.76012552],
            [554400.0, 7907.76012552],
            [555300.0, 6652.55987448],
            [556200.0, 6652.55987448],
            [557100.0, 5271.839874480001],
            [558000.0, 6150.479874480001],
            [558900.0, 5899.4397489600005],
            [559800.0, 4393.2],
            [560700.0, 4518.71974896],
            [561600.0, 4518.71974896],
            [562500.0, 4518.71974896],
            [563400.0, 4518.71974896],
            [564300.0, 4518.71974896],
            [565200.0, 2008.3201255200001],
            [566100.0, 3514.56012552],
            [567000.0, 1506.24],
            [567900.0, 3389.04012552],
            [568800.0, 3389.04012552],
            [569700.0, 3389.04012552],
            [570600.0, 3389.04012552],
            [571500.0, 2259.35987448],
            [572400.0, 4016.6402510400003],
            [573300.0, 2384.87987448],
            [574200.0, 1757.28],
            [575100.0, 1757.28],
            [576000.0, 1757.28],
            [576900.0, 2008.3201255200001],
            [577800.0, 1380.7199999999998],
            [578700.0, 5020.799874480001],
            [579600.0, 1506.24],
            [580500.0, 1506.24],
            [581400.0, 1882.7999999999997],
            [582300.0, 1882.7999999999997],
            [583200.0, 1882.7999999999997],
            [584100.0, 4769.759874480001],
            [585000.0, 1631.7599999999998],
            [585900.0, 1631.7599999999998],
            [586800.0, 3012.48],
            [587700.0, 3012.48],
            [588600.0, 125.52],
            [589500.0, 0.0],
            [590400.0, 125.52],
            [591300.0, 125.52],
            [592200.0, 125.52],
            [593100.0, 0.0],
            [594000.0, 125.52],
            [594900.0, 125.52],
            [595800.0, 0.0],
            [596700.0, 0.0],
            [597600.0, 0.0],
            [598500.0, 0.0],
            [599400.0, 125.52],
            [600300.0, 125.52],
            [601200.0, 0.0],
            [602100.0, 0.0],
            [603000.0, 0.0],
            [603900.0, 0.0],
        ],
    }

    for key in map_input_heat:
        map_input_heat[key] = [x[1] for x in map_input_heat[key]]

    return map_input_heat


# Main function
if __name__ == "__main__":
    main()
