# -*- coding: utf-8 -*-
"""
Analysis and Visualization of Dymola Simulation Results
=================================================

This module processes and visualizes simulation results from Dymola for UESGraphs-generated 
district heating network models. It demonstrates the analysis workflow from loading 
simulation data to creating network visualizations with key performance indicators.

Prerequisites:
------------
1. Network Definition:
   - Requires a network definition file (nodes.json) generated by example e11
   - Contains network topology and component parameters

2. Simulation Data:
   - Dymola simulation results in either .mat or .parquet.gzip format
   - Example data provided in uesgraphs/data/
   - Results must be from AixLib v2.1.0 models


Features:
--------
- Loads network topology from JSON
- Processes Dymola simulation results
- Assigns time series data to network components
- Generates network visualizations with:
  * Mass flow distribution
  * Pressure distribution
  * Time-averaged performance indicators

Data Processing:
-------------
1. Data Format Handling:
   - Checks for compressed data in .parquet.gzip format
   - If only .mat file exists:
     * Converts data using mat_handler.py
     * Stores intermediate results in compressed .parquet.gzip format
     * Uses compressed format for further processing
   - Subsequent runs use cached .parquet.gzip file for improved performance

2. Data Assignment:
   - Maps time series data to network components
   - Processes both extensive and intensive properties
   - Calculates time averages for visualization

Required Input Files:
------------------
1. nodes.json: Network topology and parameters
2. simulation_results.mat or parquet with .gzip compression: Dymola simulation output
    (If only .mat is available, will be converted to .gzip with mat_handler.py)

Output:
------
1. Network visualizations:
    - Mean Mass flow distribution (m_flow.png)
    - Mean Pressure distribution (press_flow.png)

Notes:
-----
- Simulation period can be specified (default: one week)
- Visualization uses time-averaged values for certain properties
- AixLib version must be specified for correct data mapping
- All paths are relative to the workspace directory
"""

import uesgraphs as ug

import uesgraphs.analyze as analyze
from uesgraphs.examples import e1_readme_example as e1

import os

from datetime import datetime

def main():
    workspace = e1.workspace_example("e13")
    
    dir_ues = os.path.dirname(os.path.dirname(workspace))
    pinola_json = os.path.join(dir_ues, "workspace", "e11", "inputs","test_modelgen", "Pinola", "nodes.json")
    pinola_sim_data = os.path.join(dir_ues,"uesgraphs","data","Pinola_low_temp_network_inputs.gzip")
    pinola_sysm_graph = os.path.join(dir_ues, "workspace","e11","Pinola", "model_sysm_graph_pinola.json")


    if not os.path.exists(pinola_json):
        raise FileNotFoundError(f"File {pinola_json} not found."
                                "Please run example e11 to generate network topology.")
    
    if not os.path.exists(pinola_sysm_graph):
        raise FileNotFoundError(f"File {pinola_sysm_graph} not found."
                                "Please run example e11 to systemmodel graph.")


    graph = ug.UESGraph()
    graph.from_json(path = pinola_json, network_type="heating")
    graph.graph["name"] = "pinola"
    graph.graph["supply_type"] = "supply"
    
    start_date=datetime(2024, 1, 1)
    end_date=datetime(2024, 1, 7)
    time_interval = "15min"

    graph = analyze.assign_data_pipeline(graph,simulation_data_path = pinola_sim_data,
                                            time_interval=time_interval,
                                             start_date=start_date,
                                             end_date=end_date,
                                             aixlib_version="2.1.0", #aixlib version is needed to assign data properly
                                            system_model_path=pinola_sysm_graph)
    
    ### Plotting
    #Visuals cant handle series data, so we just take the mean values, but single time points are possible
    vis = ug.Visuals(graph)
    for edge in graph.edges:
        graph.edges[edge]["m_flow_mean"] = graph.edges[edge]["m_flow"].mean()
    vis.show_network(show_plot=False,
                           scaling_factor=1,
                           scaling_factor_diameter=50,
                           label_size=15,
                           ylabel="Mean mass flow [kg/s]",
                           generic_extensive_size="m_flow_mean",
                           save_as=os.path.join(workspace, "m_flow.png"),
                           timestamp=f"{graph.graph["name"]}: Mean mass flow"
                           )
    
    for node in graph.nodes:
        graph.nodes[node]["press_mean"] = graph.nodes[node]["pressure"].mean()
    vis.show_network(show_plot=False,
                           scaling_factor=1,
                           scaling_factor_diameter=50,
                           ylabel="Mean pressure [Pa]",
                           label_size=15,
                           generic_intensive_size="press_mean",
                           save_as=os.path.join(workspace, "pressure.png"),
                           timestamp=f"{graph.graph["name"]}: Mean pressure"
                           )
    

if __name__ == "__main__":
    main()