import copy
import datetime
from functools import partial
import json
import math
import networkx as nx
import numpy as np
import os
import pandas as pd
import shapely.geometry as sg
from shapely import affinity
from shapely.prepared import prep
from shapely.geometry import Point, MultiLineString

import uesgraphs.utilities as ut
import uuid
import warnings
import xml.etree.ElementTree

import geopandas as gp


from uesgraphs.uesmodels.utilities import utilities as utils


try:
    import pyproj
except:
    msg = (
        "Could not import pyproj package. Thus, from_osm function "
        "is not going to work. If you require it, you have to install "
        "from_osm package."
    )
    warnings.warn(msg)


class UESGraph(nx.Graph):
    """A networkx Graph enhanced for use to describe urban energy systems.

    Attributes
    ----------
    name : str
        Name of the graph
    # TODO: delete input ids
    input_ids : dict
        When input is read from json files with ids in their meta data,
        these ids are stored in this dict
    nodelist_street : list
        List of node ids for all street nodes
    nodelist_building : list
        List of node ids for all building nodes
    nodelists_heating : dict
        Dictionary contains nodelists for all heating networks. Keys are names
        of the networks in str format, values are lists of all node ids that
        belong to the network
    nodelists_cooling : dict
        Dictionary contains nodelists for all cooling networks. Keys are names
        of the networks in str format, values are lists of all node ids that
        belong to the network
    nodelists_electricity : dict
        Dictionary contains nodelists for all electricity networks. Keys are
        names of the networks in str format, values are lists of all node
        ids that belong to the network
    nodelists_gas : dict
        Dictionary contains nodelists for all gas networks. Keys are names
        of the networks in str format, values are lists of all node ids that
        belong to the network
    nodelists_others : dict
        Dictionary contains nodelists for all other networks. Keys are names
        of the networks in str format, values are lists of all node ids that
        belong to the network
    network_types : list
        A list of all supported network types with their names in str format
    nodes_by_name : dict
        A dictionary with building names for keys and node numbers for values.
        Used to retrieve node numbers for a given building name.
    positions : dict
        In general, positions in uesgraphs are defined by
        `shapely.geometry.point` objects. This attribute converts the positions
        into a dict of numpy arrays only for use in uesgraphs.visuals, as the
        networkx drawing functions need this format.
    min_position : shapely.geometry.point object
        Position with smallest x and y values in the graph
    max_position : shapely.geometry.point object
        Position with largest x and y values in the graph
    next_node_number : int
        Node number for the next node to be added
    simplification_level : int
        Higher values indicate more simplification of the graph
        0: no simplification
        1: pipes connected in series are simplified to 1 aggregate pipe
    pipeIDs : list
        List of pipeIDs used in the graph
    """

    def __init__(self):
        """Constructor for UESGraph class."""
        super(UESGraph, self).__init__()
        self.name = ut.name_uesgraph
        # TODO: delete
        self.input_ids = {
            "buildings": None,
            "nodes": None,
            "pipes": None,
            "supplies": None,
        }
        self.nodelist_street = []
        self.nodelist_building = []
        self.nodelists_heating = {"default": []}
        self.nodelists_cooling = {"default": []}
        self.nodelists_electricity = {"default": []}
        self.nodelists_gas = {"default": []}
        self.nodelists_others = {"default": []}

        self.network_types = ["heating", "cooling", "electricity", "gas", "others"]

        self.nodes_by_name = {}
        self.positions = {}
        self.min_position = None
        self.max_position = None

        self.next_node_number = 1001
        self.simplification_level = 0
        self.pipeIDs = []

    @property
    def node(self):
        return self.nodes

    @property
    def positions(self):
        """Set position."""
        for node in self.nodes(data=True):
            if node[0] is not None:
                assert "position" in node[1], "No position for:" + str(node[0])
                if node[1]["position"] is not None:
                    self.__positions[node[0]] = np.array(
                        [node[1]["position"].x, node[1]["position"].y]
                    )
        return self.__positions

    @positions.setter
    def positions(self, value):
        self.__positions = value

    def __repr__(self):
        """Return uesgraphs class description."""
        description = "<uesgraphs.UESGraph object>"
        return description

    def new_node_number(self):
        """Return a new 4 digits node number that is not yet used in graph.

        Returns
        -------
        new_number : int
            4 digit number not yet used for nodes in graph
        """
        new_number = self.next_node_number
        self.next_node_number += 1

        return new_number

    def add_network(self, network_type, network_id):
        """Add a new network of specified type.

        Parameters
        ----------
        network_type : str
            Specifies the type of the new network as {'heating', 'cooling',
            'electricity', 'gas', 'others'}
        network_id : str
            Name of the new network
        """
        assert network_type in self.network_types, "Network type not known"
        assert isinstance(
            network_id, type(str())
        ), "Network name must be a\
            string"

        if network_type == "heating":
            self.nodelists_heating[network_id] = []
        elif network_type == "cooling":
            self.nodelists_cooling[network_id] = []
        elif network_type == "electricity":
            self.nodelists_electricity[network_id] = []
        elif network_type == "gas":
            self.nodelists_gas[network_id] = []
        elif network_type == "others":
            self.nodelists_others[network_id] = []

    def _update_min_max_positions(self, position):
        """Update values for min_positions and max_positions.

        Parameters
        ----------
        position : shapely.geometry.Point
            Definition of the node position with a Point object
        """
        if isinstance(position, type(sg.Point(0, 0))):
            if self.min_position is None:
                self.min_position = position
            else:
                if position.x < self.min_position.x:
                    self.min_position = sg.Point(position.x, self.min_position.y)
                if position.x > self.max_position.x:
                    self.max_position = sg.Point(position.x, self.max_position.y)
            if self.max_position is None:
                self.max_position = position
            else:
                if position.y < self.min_position.y:
                    self.min_position = sg.Point(self.min_position.x, position.y)
                if position.y > self.max_position.y:
                    self.max_position = sg.Point(self.max_position.x, position.y)

    def add_building(
        self,
        name=None,
        position=None,
        is_supply_heating=False,
        is_supply_cooling=False,
        is_supply_electricity=False,
        is_supply_gas=False,
        is_supply_other=False,
        attr_dict=None,
        replaced_node=None,
        **attr
    ):
        """Add a building node to the UESGraph.

        Parameters
        ----------
        name : str, int, or float
            A name for the building represented by this node. If None is given,
            the newly assigned node number will also be used as name.
        position : shapely.geometry.Point object
            New node's position
        is_supply_heating : boolean
            True if the building contains a heat supply unit, False if not
        is_supply_cooling : boolean
            True if the building contains a cooling supply unit, False if not
        is_supply_electricity : boolean
            True if the building contains an electricity supply unit, False if
            not
        is_supply_gas : boolean
            True if the building contains a gas supply unit, False if not
        is_supply_other : boolean
            True if the building contains a supply unit for a network of
            network type other, False if not
        attr_dict : dictionary, optional (default= no attributes)
            Dictionary of building attributes. Key/value pairs set data
            associated with the building
        attr : keyword arguments, optional
            Set attributes of building using key=value

        Returns
        -------

        node_number : int
            Number of the added node in the graph
        """
        node_number = self.new_node_number()
        if name is None:
            name = node_number

        attr_dict_ues = {
            "name": name,
            "node_type": "building",
            "position": position,
            "is_supply_heating": is_supply_heating,
            "is_supply_cooling": is_supply_cooling,
            "is_supply_electricity": is_supply_electricity,
            "is_supply_gas": is_supply_gas,
            "is_supply_other": is_supply_other,
        }

        if attr_dict is not None:
            attr_dict_ues.update(attr_dict)

        attr_dict_ues.update(attr)

        self._update_min_max_positions(position)

        self.add_node(node_for_adding=node_number)
        for key in attr_dict_ues.keys():
            self.nodes[node_number][key] = attr_dict_ues[key]

        self.nodelist_building.append(node_number)

        self.nodes_by_name[name] = node_number

        if replaced_node:
            edges_to_remove = []
            edges = self.edges(replaced_node)
            for edge in edges:
                if edge[0] == edge[1]:
                    pass
                else:
                    if edge[0] == replaced_node:
                        n0 = node_number
                        n1 = edge[1]
                    else:
                        n0 = edge[0]
                        n1 = node_number

                    edge_dict = self.edges[edge[0], edge[1]].get("attr_dict", {})
                    if "diameter" in self.edges[edge[0], edge[1]].keys():
                        diameter = self.edges[edge[0], edge[1]]["diameter"]
                        self.add_edge(n0, n1, attr_dict=edge_dict, diameter=diameter)
                    else:
                        self.add_edge(n0, n1, attr_dict=edge_dict)
                edges_to_remove.append(edge)
                
            for edge in edges_to_remove:
                self.remove_edge(edge[0], edge[1])
            self.remove_network_node(replaced_node)

        return node_number

    def remove_building(self, node_number):
        """Remove the specified building node from the graph.

        Parameters
        ----------
        node_number : int
            Identifier of the node in the graph
        """
        if node_number in self.nodelist_building:
            self.nodelist_building.remove(node_number)
            self.remove_node(node_number)
        else:
            warnings.warn(
                "Node number has not been found in building"
                + "nodelist. Therefore, node has not been removed."
            )

    def add_street_node(
        self, position, resolution=1e-4, check_overlap=True, attr_dict=None, **attr
    ):
        """Add a street node to the UESGraph.

        Parameters
        ----------
        position : shapely.geometry.Point
            Definition of the node position with a shapely Point object
        resolution : float
            Minimum distance between two points in m. If new position is closer
            than resolution to another existing node, the existing node will be
            returned, no new node will be created.
        check_overlap : boolean
            By default, the method checks whether the new position overlaps
            an existing network node. This can be skipped for performance
            reasons by setting check_overlap=False
        attr_dict : dictionary, optional (default= no attributes)
            Dictionary of node attributes. Key/value pairs set data
            associated with the node
        attr : keyword arguments, optional
            Set attributes of node using key=value

        Returns
        -------
        node_number : int
            Number of the added node in the graph
        """
        node_number = self.new_node_number()

        attr_dict_ues = {"node_type": "street", "position": position}

        if attr_dict is not None:
            attr_dict_ues.update(attr_dict)

        attr_dict_ues.update(attr)

        self._update_min_max_positions(position)

        check_node = None
        if check_overlap is True:
            # Check if there is already a node at the given position
            for node in self.nodelist_street:
                if position.distance(self.nodes[node]["position"]) < resolution:
                    check_node = node

        if check_node is not None:
            return check_node
        else:
            self.add_node(node_for_adding=node_number)
            for key in attr_dict_ues.keys():
                self.nodes[node_number][key] = attr_dict_ues[key]
            self.nodelist_street.append(node_number)

            return node_number

    def remove_street_node(self, node_number):
        """Remove the specified street node from the graph.

        Parameters
        ----------
        node_number : int
            Identifier of the node in the graph
        """
        if node_number in self.nodelist_street:
            self.nodelist_street.remove(node_number)
            self.remove_node(node_number)
        else:
            warnings.warn(
                "Node number has not been found in street "
                + "nodelist. Therefore, node has not been removed."
            )

    def add_network_node(
        self,
        network_type,
        network_id="default",
        name=None,
        position=None,
        resolution=1e-4,
        check_overlap=True,
        attr_dict=None,
        **attr
    ):
        """Add a network node to the UESGraph.

        A network node should not be placed at positions where there is already
        a node of the same network or a building node.

        Parameters
        ----------
        network_type : str
            Defines the network type into which to add the node. The string
            must be one of the network_types defined in `self.network_types`.
        network_id : str
            Specifies, to which network of the given type the node belongs.
            If no value is given, the network 'default' will be used. Before
            using a `network_id`, it must be added to the UESGraph with
            `self.add_network()`
        name : str, int, or float
            A name for the network junction represented by this node. If
            None is given, the newly assigned node number will also be used
            as name.
        position : shapely.geometry.Point
            Optional definition of the node position with a shapely Point
            object
        resolution : float
            Minimum distance between two points in m. If new position is
            closer
            than resolution to another existing node, the existing node will
            be
            returned, no new node will be created.
        check_overlap : boolean
            By default, the method checks whether the new position overlaps
            an existing network node. This can be skipped for performance
            reasons by setting check_overlap=False
        attr_dict : dictionary, optional (default= no attributes)
            Dictionary of node attributes. Key/value pairs set data
            associated with the node
        attr : keyword arguments, optional
            Set attributes of node using key=value

        Returns
        -------
        node_number : int
            Number of the added node in the graph
        """
        assert network_type in self.network_types, "Unknown network type"
        node_number = self.new_node_number()
        if name is None:
            name = node_number

        attr_dict_ues = {
            "node_type": "network_" + network_type,
            "network_id": "network_id",
            "position": position,
            "name": name,
        }

        if attr_dict is not None:
            attr_dict_ues.update(attr_dict)

        attr_dict_ues.update(attr)

        self._update_min_max_positions(position)

        if network_type == "heating":
            nodelist = self.nodelists_heating[network_id]
        elif network_type == "cooling":
            nodelist = self.nodelists_cooling[network_id]
        elif network_type == "electricity":
            nodelist = self.nodelists_electricity[network_id]
        elif network_type == "gas":
            nodelist = self.nodelists_gas[network_id]
        elif network_type == "others":
            nodelist = self.nodelists_others[network_id]

        # Check if there is already a node at the given position
        if check_overlap is True:
            check_node = None

            for node in nodelist:
                if position.distance(self.nodes[node]["position"]) < resolution:
                    check_node = node
            if check_node is None:
                for node in self.nodelist_building:
                    if position.distance(self.nodes[node]["position"]) < resolution:
                        check_node = node

            if check_node is not None:
                return check_node
            else:
                self.add_node(node_for_adding=node_number)
                for key in attr_dict_ues.keys():
                    self.nodes[node_number][key] = attr_dict_ues[key]
                if network_type == "heating":
                    nodelist.append(node_number)
                elif network_type == "cooling":
                    nodelist.append(node_number)
                elif network_type == "electricity":
                    nodelist.append(node_number)
                elif network_type == "gas":
                    nodelist.append(node_number)
                elif network_type == "others":
                    nodelist.append(node_number)
        else:
            self.add_node(node_for_adding=node_number)
            for key in attr_dict_ues.keys():
                self.nodes[node_number][key] = attr_dict_ues[key]
            if network_type == "heating":
                nodelist.append(node_number)
            elif network_type == "cooling":
                nodelist.append(node_number)
            elif network_type == "electricity":
                nodelist.append(node_number)
            elif network_type == "gas":
                nodelist.append(node_number)
            elif network_type == "others":
                nodelist.append(node_number)

        self.nodes_by_name[name] = node_number

        return node_number

    def remove_network_node(self, node_number):
        """Remove the specified network node from the graph.

        Parameters
        ----------
        node_number : int
            Identifier of the node in the graph
        """
        #  Search for occurrence of node number within different network dicts
        network_list = [
            self.nodelists_heating,
            self.nodelists_cooling,
            self.nodelists_electricity,
            self.nodelists_gas,
            self.nodelists_others,
        ]

        found_node = False

        for nodelists in network_list:
            for network in nodelists:
                for node_id in nodelists[network]:
                    if node_id == node_number:
                        found_node = True
                        found_nodelists = nodelists
                        found_network = network
                        break

        if found_node:
            found_nodelists[found_network].remove(node_number)
            self.remove_node(node_number)
        else:
            warnings.warn(
                "Chosen node number is not part of any network "
                + "dict. Cannot be removed."
            )

    def get_building_node(self, name):
        """Return the node number for a given building name.

        Parameters
        ----------
        name : str
            Name of the building

        Returns
        -------
        node_number : int
            Number of the corresponding node
        """
        if name in self.nodes_by_name.keys():
            return self.nodes_by_name[name]
        else:
            print(name, "not known")

    def get_node_by_position(self, position, resolution=1e-4):
        """
        Return node name and node_nb for node(s) on input position.

        If no node is placed on position, returns empty dictionary.

        Parameters
        ----------
        position : shapely.geometry.Point
            Queried position
        resolution : float
            Minimum distance between two points in m. If  position is closer
            than resolution to another existing node, the existing node will be
            returned.

        Returns
        -------
        result_dict : dict
            Dictionary of nodes on input position (key: node_id, value: name)
        """
        result_dict = {}

        for node in self:
            if "position" in self.nodes[node]:
                #  If positions are identical, save name and node_id to dict
                if self.nodes[node]["position"].distance(position) < resolution:
                    node_name = self.nodes[node]["name"]
                    result_dict[node] = node_name

        return result_dict

    def create_subgraphs(self, network_type, all_buildings=True, streets=False):
        """Return a list of subgraphs for each network.

        Parameters
        ----------
        network_type : str
            One of the network types defined in `self.network_types`. The
            subgraphs for all networks of the chosen network type will be
            returned
        all_buildings : boolean
            Subgraphs will contain all buildings of uesgraph when
            `all_buildings` is True. If False, only those buildings connected
            to a subgraph's network will be part of the corresponding subgraph
        streets : boolean
            Subgraphs will contain streets if `streets` is True.

        Returns
        -------

        subgraphs : list
            List of uesgraph elements for all networks of chosen `network_type`
        """
        assert (
            network_type in self.network_types
            or network_type is None
            or network_type == "proximity"
        ), "Network type not known"

        if network_type == "heating":
            nodelists = self.nodelists_heating
        elif network_type == "cooling":
            nodelists = self.nodelists_cooling
        elif network_type == "electricity":
            nodelists = self.nodelists_electricity
        elif network_type == "gas":
            nodelists = self.nodelists_gas
        elif network_type == "others":
            nodelists = self.nodelists_others
        elif network_type is None:
            nodelists = {}

        subgraphs = {}

        if network_type is not None and network_type != "proximity":
            if nodelists != {"default": []}:
                for network_id in nodelists.keys():
                    # Largely copied from nx.Graph.subgraphs()
                    bunch = nodelists[network_id] + self.nodelist_building
                    # create new graph and copy subgraph into it
                    h = self.__class__()
                    h.max_position = self.max_position
                    h.min_position = self.min_position
                    # copy node and attribute dictionaries
                    for n in bunch:
                        h.add_node(n)
                        for attr in self.nodes[n]:
                            h.nodes[n][attr] = self.nodes[n][attr]
                    # Add edges
                    for n in h.nodes():
                        for nbr in self.neighbors(n):
                            if nbr in h.nodes():
                                h.add_edge(n, nbr)
                                for attr in self.edges[n, nbr]:
                                    h.edges[n, nbr][attr] = self.edges[n, nbr][attr]

                    h.graph = self.graph
                    for building in self.nodelist_building:
                        h.nodelist_building.append(building)
                    if network_type == "heating":
                        h.nodelists_heating[network_id] = self.nodelists_heating[
                            network_id
                        ]
                    elif network_type == "cooling":
                        h.nodelists_cooling[network_id] = self.nodelists_cooling[
                            network_id
                        ]
                    elif network_type == "electricity":
                        h.nodelists_electricity[
                            network_id
                        ] = self.nodelists_electricity[network_id]
                    elif network_type == "gas":
                        h.nodelists_gas[network_id] = self.nodelists_gas[network_id]
                    elif network_type == "others":
                        h.nodelists_others[network_id] = self.nodelists_others[
                            network_id
                        ]
                    h.nodes_by_name = self.nodes_by_name
                    subgraphs[network_id] = h

            if all_buildings is False:
                for network_id in subgraphs.keys():
                    to_remove = []
                    for building in subgraphs[network_id].nodelist_building:
                        if nx.degree(subgraphs[network_id], building) == 0:
                            to_remove.append(building)
                    for remove_me in to_remove:
                        subgraphs[network_id].remove_building(remove_me)
        elif network_type is None:
            # create new graph and copy subgraph into it
            h = self.__class__()
            for building in self.nodelist_building:
                if all_buildings is True:
                    h.add_node(building)
                    for attr in self.nodes[building]:
                        h.nodes[building][attr] = self.nodes[building][attr]
                    h.nodelist_building.append(building)
                    h._update_min_max_positions(self.nodes[building]["position"])
                else:
                    if (
                        self.nodes[building]["is_supply_heating"] is False
                        and self.nodes[building]["is_supply_cooling"] is False
                        and self.nodes[building]["is_supply_electricity"] is False
                        and self.nodes[building]["is_supply_gas"] is False
                        and self.nodes[building]["is_supply_other"] is False
                    ):
                        # h.node[building] = self.nodes[building]
                        h.add_node(building)
                        for attr in self.nodes[building]:
                            h.nodes[building][attr] = self.nodes[building][attr]
                        h.nodelist_building.append(building)
                        h._update_min_max_positions(self.nodes[building]["position"])

            subgraphs["default"] = h

        if streets is True:
            for network_id in subgraphs.keys():
                # Add edges
                for n in h.nodes():
                    for nbr in self.neighbors(n):
                        if nbr in h.nodes():
                            h.add_edge(n, nbr)
                            for attr in self.edges[n, nbr]:
                                h.edges[n, nbr][attr] = self.edges[n, nbr][attr]

        if network_type == "proximity" and "proximity" in self.graph:
            h = copy.deepcopy(self)
            h.min_position = None
            proximity = self.graph["proximity"]
            for node in self.nodes():
                position = self.nodes[node]["position"]
                if not proximity.contains(position):
                    node_type = self.nodes[node]["node_type"]
                    if "network" in node_type:
                        h.remove_network_node(node)
                    elif "building" in node_type:
                        h.remove_building(node)
                    elif "street" in node_type:
                        h.remove_street_node(node)
            prox_bounds = self.graph["proximity"].bounds
            new_min = sg.Point(prox_bounds[0], prox_bounds[1])
            new_max = sg.Point(prox_bounds[2], prox_bounds[3])
            h.min_position = new_min
            h.max_position = new_max
            return h

        return subgraphs

    def from_json(self, path, network_type, check_overlap=False):
        """Import network from json input.

        Parameters
        ----------
        path : str
            Path, where input files `substations.json`, `nodes.json`,
            `pipes.json` and `supply.json` are located.
        network_type : str
            Specifies the type of the destination network as {'heating',
            'cooling', 'electricity', 'gas', 'others'}
        check_overlap : Boolean
            By default, the method does not check whether network node
            positions overlap existing network nodes. For `True`, this check
            becomes active.

        """
        node_mapping = {}  # input node number => uesgraphs node number

        # Read nodes to dict
        print("read nodes...")
        if path.endswith(".json"):
            input_file = path
        # elif name is not None:
        #    input_file = os.path.join(path, "%s.json" % name)
        else:
            input_file = os.path.join(path, "nodes.json")
        with open(input_file, "r") as input:
            nodes = json.load(input)
        if "input_id" in nodes["meta"]:
            self.input_ids["nodes"] = nodes["meta"]["input_id"]

        node_mapping = {}
        print("******")
        for node in nodes["nodes"]:
            # Create position object
            if "longitude" in node and "latitude" in node:
                this_position = sg.Point(node["longitude"], node["latitude"])
                node["position"] = this_position
            elif "x" in node and "y" in node:
                this_position = sg.Point(node["x"], node["y"])
                node["position"] = this_position
            else:
                warnings.warn("No spatial data input data for " "node {}".format(node))

            # Add buildings
            if "building" in node["node_type"]:
                building_id = node["name"]
                new_node = self.add_building(name=building_id, position=this_position)

            # Add supplies
            # TODO: This currently only supports heating and cooling supplies
            if "supply" in node["node_type"]:
                supply_id = node["name"]
                if "heating" in node["node_type"]:
                    new_node = self.add_building(
                        name=supply_id, position=this_position, is_supply_heating=True
                    )
                if "cooling" in node["node_type"]:
                    new_node = self.add_building(
                        name=supply_id, position=this_position, is_supply_cooling=True
                    )

            # Add network nodes
            if "network" in node["node_type"]:
                new_node = self.add_network_node(
                    network_type=network_type,
                    name=node["name"],
                    position=this_position,
                    resolution=1e-9,
                    check_overlap=check_overlap,
                )

            # Read additional attributes that have not yet been processed
            already_processed = [
                "longitude",
                "latitude",
                "x",
                "y",
                "name",
                "is_supply_heating",
                "is_supply_cooling",
                "node_type",
            ]
            for attrib in node.keys():
                if attrib not in already_processed:
                    self.nodes[new_node][attrib] = node[attrib]

            # Add node to node mapping
            node_mapping[node["name"]] = new_node

        # Add edges
        for pipe in nodes["edges"]:
            if "pipeID" in pipe:
                pipe_id = pipe["pipeID"]

            has_diameter = False
            if "diameter_inner" in pipe:
                diameter = pipe["diameter_inner"]
                has_diameter = True
            elif "diameter" in pipe:
                diameter = pipe["diameter"]
                has_diameter = True

            node_0 = node_mapping[pipe["node_0"]]
            node_1 = node_mapping[pipe["node_1"]]

            self.add_edge(node_0, node_1)
            if has_diameter:
                self.edges[node_0, node_1]["diameter"] = diameter
            if "length" in pipe:
                self.edges[node_0, node_1]["length"] = pipe["length"]
            if "G" in pipe:
                self.edges[node_0, node_1]["G"] = pipe["G"]
            if "pipeID" in pipe:
                self.edges[node_0, node_1]["pipeID"] = pipe["pipeID"]
                self.edges[node_0, node_1]["name"] = pipe["pipeID"]
                self.pipeIDs.append(int(pipe_id))
            if "lambda_insulation" in pipe:
                self.edges[node_0, node_1]["G"] = pipe["lambda_insulation"]
            if "thickness_insulation" in pipe:
                self.edges[node_0, node_1]["G"] = pipe["thickness_insulation"]

            for attrib in pipe.keys():
                if attrib not in self.edges[node_0, node_1]:
                    self.edges[node_0, node_1][attrib] = pipe[attrib]

        print(" input_ids were", self.input_ids)
        print("...finished")

    def to_json(
        self,
        path=None,
        name=None,
        description="json export from uesgraph",
        all_data=False,
        prettyprint=False,
    ):
        """Save UESGraph structure to json files.

        Parameters
        ----------
        path : str
            Path where a directory with output files will be created. If
            `None` is given, the json data will not be written to file, but
            only returned. This only works for `format_old=False`.
        name : str
            The newly created output directory at `path` will be named
            `<name>HeatingUES`
        description : str
            A description string that will be written to all json output
            files' meta data.
        all_data : boolean
            If False, only the main data (x, y, name, node_type) will be
            written to the json output. If True, all node data is exported.
        prettyprint : boolean
            If `True`, the JSON file will use an indent of 4 spaces to pretty-
            print the file. By default, a more efficient output without
            indents will be generated

        Returns
        -------
        output_data : dict
            Contents of the nodes.json file following the new format. For
            `format_old=True` the method returns `None`.

        """
        dir_this = ut.default_json_path()

        if not path:
            dir_wrkspc = os.path.abspath(os.path.join(dir_this, "workspace"))
        else:
            dir_wrkspc = os.path.abspath(os.path.join(path))

        if not os.path.exists(dir_wrkspc):
            os.mkdir(dir_wrkspc)

        path = dir_wrkspc
        path = os.path.abspath(path)

        if os.path.isdir(path):
            os.chdir(path)

        if not name:
            file_path = os.path.join(path, "nodes.json")
            file_path = os.path.abspath(file_path)
        else:
            file_path = os.path.join(path, "%s.json" % name)
            file_path = os.path.abspath(file_path)

        nodes = []
        edges = []

        meta = {
            "description": description,
            "source": "uesgraphs",
            "name": name,
            "created": str(datetime.datetime.now()),
            "simplification_level": self.simplification_level,
            "input_id": str(uuid.uuid4()),
            "units": {"diameter": "m", "length": "m"},
        }

        # Write node data from uesgraph to dict for json output
        for node in self.nodes():
            nodes.append(
                {
                    "x": self.nodes[node]["position"].x,
                    "y": self.nodes[node]["position"].y,
                }
            )
            if "name" in self.nodes[node]:
                nodes[-1]["name"] = self.nodes[node]["name"]
            else:
                nodes[-1]["name"] = str(node)
            if "node_type" in self.nodes[node]:
                node_type = self.nodes[node]["node_type"]
                if "building" in node_type:
                    if "is_supply_heating" in self.nodes[node]:
                        if self.nodes[node]["is_supply_heating"] is True:
                            node_type = "supply_heating"
                    if "is_supply_cooling" in self.nodes[node]:
                        if self.nodes[node]["is_supply_cooling"] is True:
                            node_type = "supply_cooling"
                nodes[-1]["node_type"] = node_type
            if all_data is True:
                for key in self.nodes[node]:
                    if key not in nodes[-1] and key != "position":
                        nodes[-1][key] = self.nodes[node][key]

        # Write pipe data from uesgraph to dict for json output
        for edge in self.edges():
            if "pipeID" in self.edges[edge[0], edge[1]]:
                try:
                    pipe_id = str(int(self.edges[edge[0], edge[1]]["pipeID"]))
                except:
                    pipe_id = self.edges[edge[0], edge[1]]["pipeID"]
            else:
                pipe_id = str(edge[0]) + str(edge[1])

            if "name" in self.nodes[edge[0]]:
                name_0 = self.nodes[edge[0]]["name"]
            else:
                name_0 = str(edge[0])
            if "name" in self.nodes[edge[1]]:
                name_1 = self.nodes[edge[1]]["name"]
            else:
                name_1 = str(edge[1])

            edges.append(
                {
                    "node_0": name_0,
                    "node_1": name_1,
                    "pipeID": str(pipe_id),
                    "name": str(pipe_id),
                }
            )

            if "length" in self.edges[edge[0], edge[1]]:
                length = self.edges[edge[0], edge[1]]["length"]
            else:
                pos_0 = self.nodes[edge[0]]["position"]
                pos_1 = self.nodes[edge[1]]["position"]
                length = pos_0.distance(pos_1)
            edges[-1]["length"] = length

            if "diameter" in self.edges[edge[0], edge[1]]:
                diameter = self.edges[edge[0], edge[1]]["diameter"]
                edges[-1]["diameter"] = diameter

            if "lambda_insulation" in self.edges[edge[0], edge[1]]:
                lambda_insulation = self.edges[edge[0], edge[1]]["lambda_insulation"]
                edges[-1]["lambda_insulation"] = lambda_insulation

            if all_data is True:
                for key in self.edges[edge[0], edge[1]]:
                    if key not in edges[-1]:
                        edges[-1][key] = self.edges[edge[0], edge[1]][key]

        # Write json files
        output_data = {"meta": meta, "nodes": nodes, "edges": edges}

        if path is not None:
            with open(file_path, "w") as outfile:
                if prettyprint:
                    json.dump(output_data, outfile, indent=4)
                else:
                    json.dump(output_data, outfile)
            outfile.close()
        # print(output_data)
        return file_path

    def from_osm(
        self, osm_file, name=None, check_boundary=False, transform_positions=True
    ):
        """Import buildings and streets from OpenStreetMap data in osm_file.

        If available, the following attributes will be written to the imported
        elements:

        For streets:
        - 'street type' (motorway, trunk,primary, secondary, tertiary,
        residential, unclassified, service, living_street, pedestrian)

        For buildings
        - 'position'
        - 'outlines'
        - 'area'
        - 'addr_street'
        - 'addr_housenumber'
        - 'building_levels'
        - 'building_height'
        - 'building_buildyear'
        - 'building_condition'
        - 'building_roof_shape'
        - 'comments'
        - type of usage ('amenity', 'shop', 'leisure')

        Parameters
        ----------
        osm_file : str
            Full path to osm input file
        name : str
            Name of the city for boundary check
        check_boundary : boolean
            If True, the city boundary will be extracted from the osm file and
            only nodes within this boundary will be accepted
        transform_positions : boolean
            By default, positions are transformed to a coordinate system
            that gives distances in Meter setting the origin (0, 0) at the
            minimum position of the graph. If transform_positions is False,
            the positions will remain in longitude and latitude as read from
            the OSM file.

        Returns
        -------
        self : uesgraph object
            UESGraph for the urban energy system read from osm data
        """

        # def latlon2abs(geometry, lat1, lat2):
        #     """Convert a geometry object from lat/lon to absolute coords.

        #     Taken from http://gis.stackexchange.com/questions/127607/

        #     Parameters
        #     ----------
        #     geometry : a shapely geometry object
        #     lat1 : float
        #         First reference latitude
        #     lat 2 : float
        #         Second reference latitude

        #     Returns
        #     -------
        #     converted : a shapely geometry object
        #     """
        #     proj_4326_crs = pyproj.crs.CRS("epsg:4326")
        #     proj_from_latlat_crs = pyproj.crs.CRS(proj="aea", lat_1=lat1, lat_2=lat2)
        #     transformer = pyproj.Transformer.from_crs(
        #         proj_4326_crs,
        #         proj_from_latlat_crs,
        #         always_xy=True).transform

        #     converted = ops.transform(transformer, geometry)

        #     return converted

        print("Starting import from OpenStreetMap file")
        root = xml.etree.ElementTree.parse(osm_file).getroot()

        # Write all node positions to dict
        print("Reading node positions to dict...")
        nodes = {}
        for node in root.findall("node"):
            lon = float(node.get("lon"))
            lat = float(node.get("lat"))
            nodes[node.get("id")] = {"lon": lon, "lat": lat}

        # Define street tags to be used in uesgraph
        street_tags = [
            "motorway",
            "trunk",
            "primary",
            "secondary",
            "tertiary",
            "unclassified",
            "residential",
            "service",
        ]

        streets = []
        # Get city boundaries
        print("Creating boundary polygon...")
        if check_boundary is True:
            city_boundaries_ways = []
            way_counter = 0
            for relation in root.findall("relation"):
                for tag in relation.findall("tag"):
                    if tag.get("k") == "name" and tag.get("v") == name:
                        for member in relation.findall("member"):
                            if member.get("type") == "way":
                                curr_ref = member.get("ref")
                                for way in root.findall("way"):
                                    if way.get("id") == curr_ref:
                                        curr_points = []
                                        for nd in way.findall("nd"):
                                            curr_lon = nodes[nd.get("ref")]["lon"]
                                            curr_lat = nodes[nd.get("ref")]["lat"]
                                            curr_points.append(
                                                sg.Point(curr_lon, curr_lat)
                                            )
                                        curr_way = sg.LineString(curr_points)
                                        city_boundaries_ways.append(curr_way)
                                        way_counter += 1

            # Create one boundary polygon
            end_points = []
            boundary_coords = []
            unused_boundaries = []
            for city_boundaries_way in city_boundaries_ways[1:]:
                unused_boundaries.append(city_boundaries_way)
            for i in range(len(city_boundaries_ways)):
                if i == 0:
                    for coords in city_boundaries_ways[i].coords:
                        boundary_coords.append(coords)
                else:
                    distances = {}
                    curr_end = boundary_coords[-1]
                    curr_end_point = sg.Point(curr_end[0], curr_end[1])
                    end_points.append(curr_end_point)
                    for j in range(len(unused_boundaries)):
                        next_start = unused_boundaries[j].coords[0]
                        next_start_point = sg.Point(next_start[0], next_start[1])
                        distance_0 = curr_end_point.distance(next_start_point)
                        next_end = unused_boundaries[j].coords[-1]
                        next_end_point = sg.Point(next_end[0], next_end[1])
                        distance_1 = curr_end_point.distance(next_end_point)
                        distances[distance_0] = [j, 0]
                        distances[distance_1] = [j, 1]

                    min_distance = min(distances.keys())
                    nearest_boundary = distances[min_distance][0]
                    if distances[min_distance][1] == 0:
                        for coords in unused_boundaries[nearest_boundary].coords:
                            boundary_coords.append(coords)
                    elif distances[min_distance][1] == 1:
                        for coords in unused_boundaries[nearest_boundary].coords[::-1]:
                            boundary_coords.append(coords)
                    del unused_boundaries[distances[min_distance][0]]
            city_boundary = sg.Polygon(boundary_coords)

        # Read buildings and streets
        print("Read building polygons and street ways...")
        all_building_data = {}
        all_street_ways = []
        for way in root.findall("way"):
            curr_positions = []
            curr_dict = {}
            outlines_building = []
            is_building = False
            for nd in way.findall("nd"):
                curr_lat = float(nodes[nd.get("ref")]["lat"])
                curr_lon = float(nodes[nd.get("ref")]["lon"])
                curr_positions.append((curr_lon, curr_lat))
                outlines_building.append([curr_lat, curr_lon])

            for tag in way.findall("tag"):
                if tag.get("k") == "building":
                    if len(curr_positions) > 2:
                        curr_way = sg.Polygon(curr_positions)
                        curr_dict["polygon"] = curr_way
                        curr_dict["outlines"] = outlines_building
                        curr_dict["comment"] = tag.get("v")
                        is_building = True
                if tag.get("k") == "addr:housenumber":
                    curr_dict["addr_housenumber"] = tag.get("v")
                if tag.get("k") == "addr:street":
                    curr_dict["addr_street"] = tag.get("v")
                if tag.get("k") == "building:levels":
                    curr_dict["building_levels"] = tag.get("v")
                if tag.get("k") == "leisure":
                    curr_dict["leisure"] = tag.get("v")
                if tag.get("k") == "name":
                    curr_dict["name"] = tag.get("v")
                if tag.get("k") == "shop":
                    curr_dict["shop"] = tag.get("v")
                if tag.get("k") == "amenity":
                    curr_dict["amenity"] = tag.get("v")
                if tag.get("k") == "building:roof:shape":
                    curr_dict["building_roof_shape"] = tag.get("v")
                if tag.get("k") == "building:buildyear":
                    curr_dict["building_buildyear"] = tag.get("v")
                if tag.get("k") == "building:condition":
                    curr_dict["building_condition"] = tag.get("v")
                if tag.get("k") == "building:height":
                    curr_dict["building_height"] = tag.get("v")

                if tag.get("k") == "highway" and tag.get("v") in street_tags:
                    all_street_ways.append([])
                    for i in range(len(curr_positions)):
                        curr_position = sg.Point(
                            curr_positions[i][0], curr_positions[i][1]
                        )
                        all_street_ways[-1].append(curr_position)
            if is_building is True:
                all_building_data[way.get("id")] = curr_dict

        # Filter buildings and streets for locations within boundary
        street_ways = []
        if check_boundary is True:
            print("Filtering buildings and streets...")
            prepared_boundary = prep(city_boundary)
            building_data = {}
            for id in all_building_data.keys():
                if city_boundary.contains(all_building_data[id]["polygon"]):
                    building_data[id] = all_building_data[id]

            for street_way in all_street_ways:
                street_ways.append(filter(prepared_boundary.contains, street_way))
        else:
            building_data = all_building_data
            street_ways = all_street_ways

        print("Add buildings to graph...")
        counter = 0
        curr_keys = list(building_data.keys())
        ordered_keys = sorted(curr_keys)  # Same node ids for same input
        for id in ordered_keys:
            curr_way = building_data[id]["polygon"]
            curr_position = curr_way.centroid
            geom_aea = utils.latlon2abs(
                curr_way,
                curr_way.bounds[1],
                curr_way.bounds[3])
            counter += 1
            building = self.add_building(position=curr_position)
            self.nodes[building]["area"] = geom_aea.area
            self.nodes[building]["osm_id"] = id
            self.nodes[building]["polygon"] = building_data[id]["polygon"]
            self.nodes[building]["outlines"] = building_data[id]["outlines"]
            self.nodes[building]["comment"] = building_data[id]["comment"]
            if "addr_street" in building_data[id]:
                self.nodes[building]["addr_street"] = building_data[id]["addr_street"]
            if "addr_housenumber" in building_data[id]:
                self.nodes[building]["addr_housenumber"] = building_data[id][
                    "addr_housenumber"
                ]
            if "building_levels" in building_data[id]:
                self.nodes[building]["building_levels"] = building_data[id][
                    "building_levels"
                ]
            if "leisure" in building_data[id]:
                self.nodes[building]["leisure"] = building_data[id]["leisure"]
            if "name" in building_data[id]:
                self.nodes[building]["name"] = building_data[id]["name"]
            if "shop" in building_data[id]:
                self.nodes[building]["shop"] = building_data[id]["shop"]
            if "amenity" in building_data[id]:
                self.nodes[building]["amenity"] = building_data[id]["amenity"]
            if "building_roof_shape" in building_data[id]:
                self.nodes[building]["building_roof_shape"] = building_data[id][
                    "building_roof_shape"
                ]
            if "building_buildyear" in building_data[id]:
                self.nodes[building]["building_buildyear"] = building_data[id][
                    "building_buildyear"
                ]
            if "building_condition" in building_data[id]:
                self.nodes[building]["building_condition"] = building_data[id][
                    "building_condition"
                ]
            if "building_height" in building_data[id]:
                self.nodes[building]["building_height"] = building_data[id][
                    "building_height"
                ]

        print("Add streets to graph...")
        for street_way in street_ways:
            street_nodes = []
            to_line_string = []
            for curr_position in street_way:
                street_nodes.append(self.add_street_node(position=curr_position))
                to_line_string.append(curr_position)
                if len(street_nodes) > 1:
                    if street_nodes[-2] != street_nodes[-1]:
                        self.add_edge(
                            street_nodes[-2], street_nodes[-1], network_type="street"
                        )

            if len(to_line_string) > 1:
                streets.append(sg.LineString(to_line_string))

        # Add boundary to uesgraph
        if check_boundary is True:
            self.graph["boundary"] = city_boundary

        # Transform to new coordinate system
        if transform_positions is True:
            self = utils.transform_to_new_coordination_system(
                self,
                streets=streets)

        self.graph["from_osm"] = True
        print("Finished import from OpenStreetMap data\n")

        return self

    def from_geojson(
            self,
            network_path,
            buildings_path,
            supply_path,
            name,
            save_path=None,
            generate_visualizations=False
        ):
        """
        Import district heating network from GeoJSON files.
        
        Creates a complete UESGraph model from three GeoJSON input files containing
        network topology, supply points, and building locations. All geometries must
        use CRS84 coordinate system and will be automatically transformed to a local
        coordinate system with distances in meters.
        
        The import process follows these steps:
        1. Process network pipes to create nodes and edges
        2. Add supply points as buildings with is_supply_heating=True
        3. Connect buildings to network nodes
        4. Transform to local coordinate system and calculate network length
        
        Parameters
        ----------
        network_path : str
            Path to network GeoJSON file containing LineString or MultiLineString
            geometries representing pipes. Optional DN property specifies nominal
            diameter in mm.
        buildings_path : str
            Path to buildings GeoJSON file containing Point or Polygon geometries.
            Must include 'name' property for each building.
        supply_path : str
            Path to supply points GeoJSON file containing Point geometries.
            Must include 'name' property. Points must coincide with network nodes.
        name : str
            Name identifier for this network model
        save_path : str, optional
            Directory path for saving JSON output and visualizations.
            If None, no files are saved.
        generate_visualizations : bool, default False
            Whether to generate and save network visualization PDFs at each
            processing step. Requires save_path to be specified.
        
        Notes
        -----
        - Network nodes are created at pipe endpoints and junctions
        - Buildings replace network nodes at matching locations
        - Supply points must exactly match existing network node positions
        - Edge lengths are calculated automatically in meters
        - All custom properties from GeoJSON are preserved as edge attributes
        
        Examples
        --------
        >>> graph = UESGraph()
        >>> graph.from_geojson(
        ...     network_path='data/network.geojson',
        ...     buildings_path='data/buildings.geojson',
        ...     supply_path='data/supply.geojson',
        ...     name='district_1',
        ...     save_path='output/',
        ...     generate_visualizations=True
        ... )
        >>> print(f"Total network length: {graph.network_length} m")
        """

        if generate_visualizations and save_path:
            folder_vis = os.path.join(save_path, "visuals_of_uesgraph_creation")
            os.makedirs(folder_vis, exist_ok=True)
        elif generate_visualizations:
            folder_vis=None
            print("Visualizations will not be saved as no save path was provided.")
        else:
            folder_vis = None


        # Build network topology from network geojson
        self._process_network_from_geojson(network_path)

        if folder_vis: # Generate visualization
            self._create_network_visualization(folder_vis, filename="1_basic_uesgraph")
        
        # Process supply points
        self._process_supply_points_from_geojson(supply_path)
    
        if folder_vis: # Generate visualization
            self._create_network_visualization(folder_vis, filename="2_with_supply",labels="heat")

        # Process buildings
        self._process_buildings_from_geojson(buildings_path)
        
        if folder_vis: # Generate visualization
            self._create_network_visualization(folder_vis, filename="3_with_bldg", labels="heat")

        # Transform coordinate system
        utils.transform_to_new_coordination_system(self)

        if folder_vis: # Generate visualization
            self._create_network_visualization(folder_vis, filename="4_coords_transformed")

        #Recalculate Pipe lengths after coordinate transformation and assign it as a main attribute        
        for edge in self.edges():
            pos_0 = self.nodes[edge[0]]["position"]
            pos_1 = self.nodes[edge[1]]["position"]
            self.edges[edge]["length"] = pos_1.distance(pos_0)


        self.network_length = self.calc_network_length("heating")

        print("Total network length (m): ", self.network_length)

        return True
    
    def _process_network_from_geojson(self, network_path):
        """
        Process network pipes from GeoJSON and add nodes/edges to the graph.
        
        Creates network nodes at pipe endpoints and junctions. Handles both
        LineString and MultiLineString geometries. If DN (nominal diameter)
        property exists, calculates inner diameter from ISO standards.
        
        Parameters
        ----------
        network_path : str
            Path to network GeoJSON file with pipe geometries
            
        Notes
        -----
        - Network nodes use 1e-6 resolution to merge nearby endpoints
        - All non-geometry properties are stored as edge attributes
        - MultiLineString geometries are decomposed into individual segments
        """

        network_df = gp.read_file(network_path)
        
        for _, row in network_df.iterrows():
            geometry = row["geometry"]
            attributes = row.to_dict()
            
            # Check if DN exists in attributes
            dn_value = attributes.get("DN", None)
            
            # Process geometry based on its type
            if isinstance(geometry, MultiLineString):
                for linestr in geometry.geoms:
                    self.__process_linestring(linestr, attributes, dn_value)
            else:  # Simple LineString
                self.__process_linestring(geometry, attributes, dn_value)
         
    def __process_linestring(self, linestring, attributes, dn_value=None):
        """
        Process a single LineString geometry into network nodes and edges.
        
        Creates network nodes at each coordinate and edges between consecutive
        points. For LineStrings with more than 2 coordinates, creates multiple
        connected edges.
        
        Parameters
        ----------
        linestring : shapely.geometry.LineString
            Pipe geometry to process
        attributes : dict
            Properties from GeoJSON feature to store as edge attributes.
            'geometry' key is automatically excluded.
        dn_value : int or float, optional
            Nominal diameter in mm. If provided, inner diameter is calculated
            from ISO standards.
        """
        # Remove geometry from attributes
        edge_attributes = {k: v for k, v in attributes.items() if k != "geometry"}
        
        coords = list(linestring.coords)
        
        if len(coords) > 2:
            # Handle multi-segment linestring
            for i in range(1, len(coords)):
                # Add nodes for segment endpoints
                node1 = self.add_network_node(
                    network_type="heating",
                    position=Point(coords[i-1]),
                    resolution=1e-6,
                    check_overlap=True,
                )
                node2 = self.add_network_node(
                    network_type="heating",
                    position=Point(coords[i]),
                    resolution=1e-6,
                    check_overlap=True,
                )
                
                # Add edge with appropriate attributes
                self.__add_edge_with_attributes(node1, node2, edge_attributes, dn_value)
        else:
            # Handle simple linestring (two points)
            node1 = self.add_network_node(
                network_type="heating",
                position=Point(coords[0]),
                resolution=1e-6,
                check_overlap=True,
            )
            node2 = self.add_network_node(
                network_type="heating",
                position=Point(coords[1]),
                resolution=1e-6,
                check_overlap=True,
            )
            
            # Add edge with appropriate attributes
            self.__add_edge_with_attributes(node1, node2, edge_attributes, dn_value)
        
    def __add_edge_with_attributes(self, node1, node2, attributes, dn_value=None):
        """
        Add an edge with DN-based diameter and other attributes.
        
        Parameters
        ----------
        node1 : int
            First node ID
        node2 : int
            Second node ID
        attributes : dict
            Edge attributes from GeoJSON properties
        dn_value : int or float, optional
            Nominal diameter in mm for diameter calculation
            
        Notes
        -----
        Default insulation thickness (dIns) is set to 0.1 m for all edges.
        """

        from uesgraphs.systemmodels.utilities import get_inner_diameter_from_DN

        if dn_value:
            self.add_edge(
                node1,
                node2,
                attr_dict=attributes,
                diameter=get_inner_diameter_from_DN(dn_value),
                dIns=0.1,
            )
        else:
            self.add_edge(
                node1,
                node2,
                attr_dict=attributes,
                dIns=0.1,
            )

    def _process_supply_points_from_geojson(self, supply_path):
        """
        Process supply points and add them as building nodes with heating supply.
        
        Finds network nodes at supply point locations and replaces them with
        building nodes marked as is_supply_heating=True. Supply points that
        don't match any network node are skipped with a warning.
        
        Parameters
        ----------
        supply_path : str
            Path to supply points GeoJSON file (Point geometries)
            
        Notes
        -----
        Supply point coordinates must exactly match an existing network node
        position (within 1e-6 resolution).
        """

        supply_df = gp.read_file(supply_path)
        
        for _, row in supply_df.iterrows():
            name_supply = row["name"]
            pos_supply = row["geometry"]
            
            # Find node at supply position
            repl_node = next(iter(self.get_node_by_position(pos_supply)), None)
            
            if repl_node:
                # Add building marked as supply
                self.add_building(
                    name=name_supply,
                    position=pos_supply,
                    is_supply_heating=True,
                    replaced_node=repl_node,
                )
        
    def _process_buildings_from_geojson(self, buildings_path):
        """
        Process building geometries and connect them to the network.
        
        For Point geometries: building must coincide with a network node.
        For Polygon geometries: any network node within the polygon is used.
        The network node is replaced with a building node, reconnecting all edges.
        
        Parameters
        ----------
        buildings_path : str
            Path to buildings GeoJSON file (Point or Polygon geometries)
            
        Notes
        -----
        Building names are converted to lowercase for consistency.
        Buildings without matching network nodes are skipped.
        """

        bldg_df = gp.read_file(buildings_path)
        
        for _, row in bldg_df.iterrows():
            bldg_name = ut.get_attribute_case_insensitive(row, "name")
            polygon = row["geometry"]

            # Find node within building polygon
            repl_node = self.__find_node_in_polygon(polygon)
            
            if repl_node:
                position = self.nodes[repl_node]["position"]
                
                # Add building
                self.add_building(
                    name=bldg_name.lower(),
                    position=position,
                    replaced_node=repl_node,
                )
    
    def __find_node_in_polygon(self, polygon):
        """
        Find the first network node located within a building polygon.
        
        Parameters
        ----------
        polygon : shapely.geometry.Polygon
            Building footprint to search within
            
        Returns
        -------
        int or None
            Node ID if found within polygon, otherwise None
            
        Notes
        -----
        Returns only the first matching node. If multiple nodes exist
        within the polygon, only one will be used for the building connection.
        """
        for node in self.nodes:
            if polygon.contains(self.nodes[node]["position"]):
                return node
        
        return None
        
    def _create_network_visualization(self, save_path, filename=None, scaling_factor=2, 
                                    show_plot=False, labels="name", simple=False):
        """
        Create and save a PDF visualization of the network.
        
        Parameters
        ----------
        save_path : str
            Directory where PDF file will be saved
        filename : str, optional
            Output filename without extension. Default is 'uesgraph_visualization'
        scaling_factor : int, default 2
            Scaling factor for node and edge sizes in visualization
        show_plot : bool, default False
            Whether to display plot interactively (in addition to saving)
        labels : str, default "name"
            Node label type: 'name', 'heat', or other node attributes
        simple : bool, default False
            If True, uses simplified visualization style
            
        Notes
        -----
        Visualization is saved as PDF
        Useful for tracking network development through processing steps.
        """
        if filename is None:
            filename = "uesgraph_visualization"
        from uesgraphs.visuals import Visuals

        
        save_as = os.path.join(save_path, f"{filename}.pdf")
        vis = Visuals(self)
        vis.show_network(
            save_as=save_as,
            show_plot=show_plot,
            scaling_factor=scaling_factor,
            simple=simple,
            labels=labels,
        )
        print(f"Saved visualization to {save_as}")
        
        return None

    def number_of_nodes(self, node_type):
        """
        Return number of nodes for given `node_type`.

        Parameters
        ----------
        node_type : str
            {'building', 'street', 'heating', 'cooling', 'electricity',
            'gas', 'other'}

        Returns
        -------
        number_of_nodes : int
            The number of nodes for the given `node_type`
        """
        if node_type == "building":
            number_of_nodes = len(self.nodelist_building)
        elif node_type == "street":
            number_of_nodes = len(self.nodelist_street)
        else:
            if node_type == "heating":
                nodelists = list(self.nodelists_heating.values())
            elif node_type == "cooling":
                nodelists = list(self.nodelists_cooling.values())
            elif node_type == "electricity":
                nodelists = list(self.nodelists_electricity.values())
            elif node_type == "gas":
                nodelists = list(self.nodelists_gas.values())
            elif node_type == "other":
                nodelists = list(self.nodelists_others.values())
            number_of_nodes = 0
            for nodelist in nodelists:
                number_of_nodes += len(nodelist)

        return number_of_nodes

    def calc_network_length(self, network_type):
        """
        Calculate the length of all edges for given `network_type`.

        Parameters
        ----------
        network_type : str
            One of the network types defined in `self.network_types`

        Returns
        -------
        total_length : float
            Total length of all edges for given `network_type` in m
        """
        total_length = 0
        for edge in self.edges():
            if (
                network_type in self.nodes[edge[0]]["node_type"]
                or network_type in self.nodes[edge[1]]["node_type"]
            ):
                # Taken from http://gis.stackexchange.com/questions/127607/
                curr_way = sg.LineString(
                    [self.nodes[edge[0]]["position"], self.nodes[edge[1]]["position"]]
                )
                total_length += curr_way.length

        return round(total_length, 2)

    def calc_total_building_ground_area(self):
        """
        Return the sum of all available building ground areas in m**2.

        Returns
        -------
        total_ground_area : float
            Sum of all available building ground areas in m**2
        """
        total_ground_area = 0
        counter = 0
        for building in self.nodelist_building:
            if "area" in self.nodes[building]:
                total_ground_area += self.nodes[building]["area"]
            else:
                counter += 1

        if counter > 0:
            warnings.warn(
                "{} of {} buildings have no area "
                "information".format(counter, self.number_of_nodes("building"))
            )

        return total_ground_area

    def rotate(self, degrees):
        """
        Rotate all nodes of the graph by `degrees`.

        Parameters
        ----------
        degrees : float
            Value of degrees for rotation
        """
        # Find center point of network to plot
        node_points = []
        for node in self.nodes():
            node_points.append(self.nodes[node]["position"])
        center_point = sg.MultiPoint(node_points).envelope.centroid

        self.min_position = None
        self.max_position = None

        for node in self.nodes():
            self.nodes[node]["position"] = affinity.rotate(
                self.nodes[node]["position"], degrees, origin=center_point
            )
            self._update_min_max_positions(self.nodes[node]["position"])

    def network_simplification(self, network_type, network_id="default"):
        """Simplify a pipe network by replacing serial pipes.

        Parameters
        ----------
        network_type : str
            Specifies the type of the network as {'heating', 'cooling',
            'electricity', 'gas', 'others'}
        network_id : str
            Name of the network
        """

        def group_nodes(node, group):
            """Recursive function to find node groups for simplification.

            Parameters
            ----------
            node : int
                Node number
            group : dict
                This dict collects a path of pipes to be replaced and the
                end nodes to be kept

            Returns
            -------
            group : list
            """
            if node not in group["path"]:
                group["path"].append(node)
            neighbors = nx.neighbors(self, node)
            # for neighbor in neighbors:
            #     if "diameter" in self.edges[node, neighbor].keys:
            #         if test_diameter == self.edges[node, neighbor]["diameter"]:

            #         temp_diameter = self.edges[node, neighbor]["diameter"]

            for neighbor in neighbors:
                if nx.degree(self, neighbor) == 2 and neighbor in nodelist:
                    if neighbor not in group["path"]:
                        group["path"].append(neighbor)
                        group = group_nodes(neighbor, group)
                else:
                    if neighbor not in group["ends"]:
                        group["ends"].append(neighbor)
            return group

        # Get nodelist for the chosen network
        if network_type == "heating":
            nodelists = self.nodelists_heating
        elif network_type == "cooling":
            nodelists = self.nodelists_cooling
        elif network_type == "electricity":
            nodelists = self.nodelists_electricity
        elif network_type == "gas":
            nodelists = self.nodelists_gas
        elif network_type == "other":
            nodelists = self.nodelists_others

        assert network_id in nodelists.keys(), "Unknown network_id"

        nodelist = nodelists[network_id]

        # keep all nodes with more then 2 connections
        keep = []
        for node in nodelist:
            if nx.degree(self, node) > 2:
                keep.append(node)

        simplification_finished = False

        # simplify until no more simplification nodes can be found
        while simplification_finished is False:
            group = {"ends": [], "path": []}
            simplification_finished = True
            for found_node in nodelist:
                if nx.degree(self, found_node) == 2 and found_node not in keep:
                    simplification_finished = False
                    break

            if simplification_finished is False:
                group = group_nodes(found_node, group)

                shortest_path = nx.shortest_path(
                    self, group["ends"][0], group["ends"][1]
                )
                dont_remove = False
                # check if the shortest path can be removed or if a diameter
                # change prevents it from simplification
                diameter_check = 0
                for i in range(len(shortest_path) - 1):
                    if "diameter" in self.edges[shortest_path[i], shortest_path[i + 1]]:
                        temp_diameter = self.edges[shortest_path[i], shortest_path[i + 1]][
                            "diameter"
                        ]
                        if diameter_check == 0:
                            diameter_check = temp_diameter
                        else:
                            if diameter_check != temp_diameter:
                                dont_remove = True
                                keep.append(found_node)
                                diameter_check = 0
                                break
                if dont_remove is False:
                    length_total = 0
                    diameter_weights = 0
                    d_ins_weights = 0
                    k_ins_weights = 0
                    fac_weights = 0
                    m_flow_nom_weights = 0
                    for i in range(len(shortest_path) - 1):
                        print(self.edges[shortest_path[i], shortest_path[i + 1]])
                        length = self.edges[shortest_path[i], shortest_path[i + 1]][
                            "length"
                        ]
                        length_total += length

                        if "diameter" in self.edges[shortest_path[i], shortest_path[i + 1]]:
                            diameter = self.edges[shortest_path[i], shortest_path[i + 1]][
                                "diameter"
                            ]
                            diameter_weights += diameter * length
                        if "dIns" in self.edges[shortest_path[i], shortest_path[i + 1]]:
                            d_ins = self.edges[shortest_path[i], shortest_path[i + 1]][
                                "dIns"
                            ]
                            d_ins_weights += d_ins * length
                        if "kIns" in self.edges[shortest_path[i], shortest_path[i + 1]]:
                            k_ins = self.edges[shortest_path[i], shortest_path[i + 1]][
                                "kIns"
                            ]
                            k_ins_weights += k_ins * length
                        if "fac" in self.edges[shortest_path[i], shortest_path[i + 1]]:
                            fac = self.edges[shortest_path[i], shortest_path[i + 1]]["fac"]
                            fac_weights += fac * length
                        if (
                            "m_flow_nominal"
                            in self.edges[shortest_path[i], shortest_path[i + 1]]
                        ):
                            m_flow_nominal = self.edges[
                                shortest_path[i], shortest_path[i + 1]
                            ]["m_flow_nominal"]
                            m_flow_nom_weights += m_flow_nominal * length

                    # if "{}{}".format(group["ends"][0], group["ends"][1]) == "10441046":
                    #     print("test")

                    for node in group["path"]:
                        print("gets removed: %s"%node)
                        self.remove_network_node(node)

                    name_start = self.nodes[group["ends"][0]]["name"]
                    name_end = self.nodes[group["ends"][1]]["name"]

                    self.add_edge(
                        group["ends"][0],
                        group["ends"][1],
                        length=length_total,
                        pipeID="{}{}".format(name_start, name_end),
                        name="{}{}".format(name_start, name_end),
                    )

                    # self.add_edge(
                    #     group["ends"][0],
                    #     group["ends"][1],
                    #     length=length_total,
                    #     pipeID="{}{}".format(group["ends"][0], group["ends"][1]),
                    #     name="{}{}".format(group["ends"][0], group["ends"][1]),
                    # )

                    if diameter_weights != 0:
                        self.edges[group["ends"][0], group["ends"][1]]["diameter"] = round(
                            diameter_weights / length_total, 3
                        )
                    if d_ins_weights != 0:
                        self.edges[group["ends"][0], group["ends"][1]]["dIns"] = (
                            d_ins_weights / length_total
                        )
                    if k_ins_weights != 0:
                        self.edges[group["ends"][0], group["ends"][1]]["kIns"] = (
                            k_ins_weights / length_total
                        )
                    if fac_weights != 0:
                        self.edges[group["ends"][0], group["ends"][1]]["fac"] = (
                            fac_weights / length_total
                        )
                    if m_flow_nom_weights != 0:
                        self.edges[group["ends"][0], group["ends"][1]]["m_flow_nominal"] = (
                            m_flow_nom_weights / length_total
                        )
                else:
                    pass

        self.simplification_level = 1

    def remove_unconnected_nodes(self, network_type, network_id="default", max_iter=10):
        """Remove any edges and network nodes not connected to a supply node.

        Parameters
        ----------
        network_type : str
            Specifies the type of the network as {'heating', 'cooling',
            'electricity', 'gas', 'others'}
        network_id : str
            Name of the network
        max_iter : int
            Maximum number of iterations

        Returns
        -------
        removed : list
            Names of removed network nodes
        """
        removed = []
        if network_type == "heating":
            is_supply = "is_supply_heating"
            nodelist = self.nodelists_heating[network_id]
        elif network_type == "cooling":
            is_supply = "is_supply_cooling"
            nodelist = self.nodelists_cooling[network_id]

        supplies = []
        for node in self.nodelist_building:
            if self.nodes[node][is_supply] is True:
                supplies.append(node)

        accept = False
        counter = 0
        while not accept and counter <= max_iter:
            counter += 1
            curr_removed = []
            for node in nodelist:
                connected = False
                for supply in supplies:
                    if nx.has_path(self, node, supply):
                        connected = True
                if connected is False:
                    if "name" in self.nodes[node]:
                        curr_removed.append(self.nodes[node]["name"])
                    else:
                        curr_removed.append(node)
                    self.remove_network_node(node)

            if len(curr_removed) == 0:
                accept = True
            else:
                for node in curr_removed:
                    if node not in removed:
                        removed.append(node)

        if counter == max_iter:
            warnings.warn("Reached maximum number of iterations")

        return removed

    def remove_self_edges(self, network_type, network_id="default"):
        """Remove edges with length 0 m.

         Parameters
        ----------
        network_type : str
            Specifies the type of the network as {'heating', 'cooling',
            'electricity', 'gas', 'others'}
        network_id : str
            Name of the network

        Returns
        -------
        number_removed_edges : int
            Number of removed network edges
        """
        number_removed_edges = 0
        to_remove = []
        for edge in self.edges():
            if self.edges[edge[0], edge[1]]["length"] <= 1e-9:
                if edge[0] == edge[1]:
                    to_remove.append([edge[0], edge[1]])

        for edge in to_remove:
            self.remove_edge(edge[0], edge[1])
            number_removed_edges += 1

        return number_removed_edges

    def remove_dead_ends(self, network_type, network_id="default"):
        """Remove any nodes and edges that lead to dead ends in the network.

        Parameters
        ----------
        network_type : str
            Specifies the type of the network as {'heating', 'cooling',
            'electricity', 'gas', 'others'}
        network_id : str
            Name of the network

        Returns
        -------
        removed : list
            Names of removed network nodes
        """
        removed = []

        if network_type == "heating":
            nodelist = self.nodelists_heating[network_id]
        elif network_type == "cooling":
            nodelist = self.nodelists_cooling[network_id]

        def remove_end(end):
            """Delete a dead end and follows up on its neighbor for recursion.

            Parameters
            ----------
            end : int
                Node number of the dead end
            """
            neighbor = list(nx.neighbors(self, end))
            if neighbor != []:
                # if nx.degree(self, neighbor[0]) <= 2:
                #     if "name" in self.nodes[end]:
                #         removed.append(self.nodes[end]["name"])
                #     else:
                #         removed.append(end)
                #     self.remove_network_node(end)
                #     remove_end(neighbor[0])
                # else:
                #     if "name" in self.nodes[end]:
                #         removed.append(self.nodes[end]["name"])
                #     else:
                #         removed.append(end)
                #     self.remove_network_node(end)

                try:
                    if (
                        self.nodes[end]["is_supply_heating"] is True
                        or self.nodes[end]["is_supply_cooling"] is True
                        or self.nodes[end]["is_supply_electricity"] is True
                        or self.nodes[end]["is_supply_gas"] is True
                        or self.nodes[end]["is_supply_other"] is True
                    ):
                        print(end)
                        pass
                except:
                    if nx.degree(self, neighbor[0]) <= 2:
                        if "name" in self.nodes[end]:
                            removed.append(self.nodes[end]["name"])
                        else:
                            removed.append(end)
                        self.remove_network_node(end)
                        remove_end(neighbor[0])
                    else:
                        if "name" in self.nodes[end]:
                            removed.append(self.nodes[end]["name"])
                        else:
                            removed.append(end)
                        self.remove_network_node(end)

        dead_ends = []
        for node in nodelist:
            if "network" in self.nodes[node]["node_type"]:
                if nx.degree(self, node) == 1:
                    dead_ends.append(node)

        for end in dead_ends:
            remove_end(end)

        return removed

    def calc_angle(self, a, b, output="rad"):
        """Return the angle of a line from point a to b in rad or degrees.

        Parameters
        ----------
        a : shapely.geometry.point object
        b : shapely.geometry.point object
        output : str
            Selection of output unit between 'rad' and 'degrees'

        Returns
        -------
        angle : float
            Angle of a line from point a to b in rad or degrees
        """
        assert output in ["rad", "degrees"], "Output must be rad or degrees"

        angle = math.atan2(b.y - a.y, b.x - a.x)
        if angle < 0:
            angle = 2 * math.pi + angle
        if output == "degrees":
            angle_degrees = (angle) * 360 / (2 * math.pi)
            return angle_degrees
        else:
            return angle

    def get_min_max(self, key, mode):
        """Get minimum and maximum values for a specific attribute from nodes or edges.
        
        Takes a UESGraph object and returns the minimum and maximum values for a given 
        attribute key, either from all nodes or all edges depending on the specified mode.
        
        Args:
            key (str): Attribute key to look up (e.g. 'capacity', 'flow', etc.)
            mode (str): Either 'node' or 'edge' to specify whether to look at node or edge attributes
            
        Returns:
            tuple: (minimum value, maximum value) for the specified attribute
            
        Raises:
            ValueError: If no values are found or mode is invalid
            KeyError: If the specified attribute key doesn't exist for all nodes/edges
        """
        record_list = {}
        
        try:
            if mode == "node":
                # Access the underlying NetworkX graph and get node attributes
                for node in self.nodes:
                    record_list[node] = self.nodes[node][key]
                    
            elif mode == "edge":  
                # Access the underlying NetworkX graph and get edge attributes
                for edge in self.edges:
                    record_list[edge] = self.edges[edge][key]
                    
            if not record_list:
                raise ValueError(f"No values found for key '{key}' in mode '{mode}'")
                
            return min(record_list.values()), max(record_list.values())
            
        except KeyError:
            raise KeyError(f"Attribute '{key}' not found for all {mode}s")