# 日收益指标计算标准过程

## 概述

日收益指标计算是量化投资中评估策略表现的重要工具。本文档基于 `czsc.daily_performance` 函数，详细说明各项指标的计算方法和实际应用。

## 输入数据要求

- **数据格式**: 日收益率序列，类型为列表或数组
- **数据单位**: 收益率，范围通常在 -1 到 1 之间（如 0.01 表示 1% 的收益）
- **数据样例**: `[0.01, 0.02, -0.01, 0.03, 0.02, -0.02, 0.01, -0.01, 0.02, 0.01]`

## 指标计算公式

### 1. 基础收益指标

#### 绝对收益 (Absolute Return)
```python
绝对收益 = sum(daily_returns)
```
- **含义**: 整个持有期间的累计收益率
- **计算**: 所有日收益率的简单相加

#### 年化收益率 (Annualized Return)
```python
年化收益率 = (sum(daily_returns) / len(daily_returns)) * yearly_days
```
- **含义**: 将日平均收益率推算到全年的收益率
- **参数**: `yearly_days` 默认为 252（中国股市年交易日数）

### 2. 风险调整指标

#### 夏普比率 (Sharpe Ratio)
```python
夏普比率 = mean(daily_returns) / std(daily_returns) * sqrt(yearly_days)
```
- **含义**: 每单位风险获得的超额收益
- **解读**: 
  - 大于1为良好
  - 大于2为优秀
  - 大于3为卓越

#### 年化波动率 (Annualized Volatility)
```python
年化波动率 = std(daily_returns) * sqrt(yearly_days)
```
- **含义**: 收益率的年化标准差，衡量投资风险

#### 下行波动率 (Downside Volatility)
```python
下行波动率 = std(negative_returns) * sqrt(yearly_days)
```
- **含义**: 仅考虑负收益的波动率，更关注下行风险

### 3. 回撤相关指标

#### 最大回撤 (Maximum Drawdown)
```python
cum_returns = cumsum(daily_returns)
drawdown = max(cum_returns) - cum_returns
最大回撤 = max(drawdown)
```
- **含义**: 从最高点到最低点的最大损失
- **计算过程**:
  1. 计算累计收益曲线
  2. 计算每日的回撤值
  3. 取最大回撤值

#### 卡玛比率 (Calmar Ratio)
```python
卡玛比率 = 年化收益率 / 最大回撤 (当最大回撤 ≠ 0)
```
- **含义**: 年化收益与最大回撤的比值
- **解读**: 比值越高，收益回撤比越好

#### 回撤风险 (Drawdown Risk)
```python
回撤风险 = 最大回撤 / 年化波动率
```
- **风险等级**:
  - < 1: 低风险
  - 1-2: 中风险
  - > 2: 高风险

### 4. 胜负相关指标

#### 日胜率 (Win Rate)
```python
日胜率 = count(daily_returns >= 0) / len(daily_returns)
```
- **含义**: 正收益天数占总交易天数的比例

#### 日盈亏比 (Profit Loss Ratio)
```python
平均盈利 = mean(positive_returns)
平均亏损 = abs(mean(negative_returns))
日盈亏比 = 平均盈利 / 平均亏损
```
- **含义**: 平均盈利与平均亏损的比值

#### 日赢面 (Win Expectation)
```python
日赢面 = 日胜率 * 日盈亏比 - (1 - 日胜率)
```
- **含义**: 综合考虑胜率和盈亏比的期望收益指标
- **解读**: > 0 表示正期望，< 0 表示负期望

### 5. 其他指标

#### 非零覆盖 (Non-zero Coverage)
```python
非零覆盖 = count(daily_returns != 0) / len(daily_returns)
```
- **含义**: 有交易（非零收益）的天数比例

#### 盈亏平衡点 (Break Even Point)
```python
盈亏平衡点 = cal_break_even_point(daily_returns)
```
- **含义**: 达到盈亏平衡所需的最小胜率

#### 新高间隔 (New High Interval)
```python
新高间隔 = max(consecutive_days_at_same_high)
```
- **含义**: 连续处于同一高点的最长天数

#### 新高占比 (New High Ratio)
```python
新高占比 = count(days_at_new_high) / total_days
```
- **含义**: 处于历史新高的天数占比

## 实际案例

### 案例1：基础计算示例

```python
import numpy as np
from czsc import daily_performance

# 模拟100天的日收益率数据
np.random.seed(42)
daily_returns = np.random.normal(0.001, 0.02, 100)  # 均值0.1%, 波动率2%

# 计算指标
result = daily_performance(daily_returns.tolist())

print("=== 日收益指标计算结果 ===")
for key, value in result.items():
    print(f"{key}: {value}")
```

### 案例2：不同策略表现对比

```python
import pandas as pd
import matplotlib.pyplot as plt

# 创建三种不同的策略收益
np.random.seed(42)

# 策略A：高收益高波动
strategy_a = np.random.normal(0.002, 0.03, 252)

# 策略B：中等收益中等波动
strategy_b = np.random.normal(0.0015, 0.02, 252)

# 策略C：低收益低波动
strategy_c = np.random.normal(0.001, 0.01, 252)

# 计算各策略指标
from czsc import daily_performance

results = {}
for name, returns in [('策略A', strategy_a), ('策略B', strategy_b), ('策略C', strategy_c)]:
    results[name] = daily_performance(returns.tolist())

# 创建对比表格
df_comparison = pd.DataFrame(results).T
print("=== 策略表现对比 ===")
print(df_comparison[['年化', '夏普', '最大回撤', '卡玛', '日胜率']].round(4))
```

### 案例3：回测结果分析

```python
# 实际策略回测示例
from czsc import daily_performance

def analyze_strategy_performance(returns_list, strategy_name="策略"):
    """
    分析策略表现
    
    :param returns_list: 日收益率列表
    :param strategy_name: 策略名称
    :return: 分析结果
    """
    result = daily_performance(returns_list)
    
    print(f"\n=== {strategy_name} 表现分析 ===")
    print(f"交易周期: {len(returns_list)} 天")
    print(f"绝对收益: {result['绝对收益']:.2%}")
    print(f"年化收益: {result['年化']:.2%}")
    print(f"最大回撤: {result['最大回撤']:.2%}")
    print(f"夏普比率: {result['夏普']:.2f}")
    print(f"卡玛比率: {result['卡玛']:.2f}")
    print(f"日胜率: {result['日胜率']:.2%}")
    print(f"回撤风险: {result['回撤风险']:.2f}")
    
    # 风险评级
    if result['回撤风险'] < 1:
        risk_level = "低风险"
    elif result['回撤风险'] < 2:
        risk_level = "中风险"
    else:
        risk_level = "高风险"
    
    print(f"风险等级: {risk_level}")
    
    return result

# 使用示例
example_returns = [0.01, 0.02, -0.01, 0.03, 0.02, -0.02, 0.01, -0.01, 0.02, 0.01]
analyze_strategy_performance(example_returns, "示例策略")
```

## 指标解读指南

### 收益指标
- **年化收益率**: 
  - > 15%: 优秀
  - 8%-15%: 良好
  - < 8%: 一般

### 风险指标
- **夏普比率**:
  - > 2: 优秀
  - 1-2: 良好
  - 0.5-1: 一般
  - < 0.5: 较差

- **最大回撤**:
  - < 10%: 优秀
  - 10%-20%: 可接受
  - > 20%: 需要关注

### 综合评价
建议综合考虑以下维度：
1. **收益性**: 年化收益率、绝对收益
2. **稳定性**: 夏普比率、卡玛比率
3. **风险性**: 最大回撤、回撤风险
4. **可持续性**: 日胜率、新高占比

## 注意事项

1. **数据质量**: 确保日收益率数据的准确性和完整性
2. **样本大小**: 建议至少有60个交易日的数据才有统计意义
3. **异常值处理**: 注意识别和处理异常的收益率数据
4. **参数设置**: 根据不同市场调整年交易日数参数
5. **指标限制**: 部分指标在极端情况下可能失效，需要结合实际情况分析

## 相关函数

- `czsc.daily_performance()`: 主要计算函数（Rust版本，推荐使用）
- `czsc.utils.stats.rolling_daily_performance()`: 滚动计算版本  
- `czsc.utils.stats.cal_break_even_point()`: 盈亏平衡点计算
- `czsc.utils.stats.daily_performance()`: 已弃用的Python版本
