#!/usr/bin/env python
#
# Copyright (c) 2014 10X Genomics, Inc. All rights reserved.
#

import os

# 10X chemistry types
GEMCODE = "GemCode"
CHROMIUM = "Chromium"

# Mass of 1 bp in nanograms
NG_PER_BP = 1.1454e-12

# Where the code sits
CODE_PATH = os.path.dirname(os.path.abspath(__file__)) + "/"


# Where barcode whitelists live
ALARMS_LOCATION = CODE_PATH + "alarms/"

# Code subfolders
TEST_FILE_IN_DIR = CODE_PATH + "test_files/inputs/"
TEST_FILE_OUT_DIR = CODE_PATH + "test_files/outputs/"

# What is considered a high confidence mapped read pair
HIGH_CONF_MAPQ = 60

# Distance to mate to ensure the
MIN_MATE_OFFSET_DUP_FILTER = 20

# Sequencing settings
ILLUMINA_MAX_QUAL = 50  # higher than possible right now
ILLUMINA_QUAL_OFFSET = 33
DEFAULT_HIGH_MAPQ = 60

# Demultiplex settings
DEMULTIPLEX_DEFAULT_SAMPLE_INDEX_LENGTH = 8
DEMULTIPLEX_BARCODE_LENGTH = 14
DEMULTIPLEX_INVALID_SAMPLE_INDEX = "X"

# Tail trim fraction when computing depth CV
COVERAGE_TRIM_TAIL = 0.01

# Subsampling coverages for dup rate
DUPLICATE_SUBSAMPLE_COVERAGES = [16.0, 8.0, 4.0, 2.0, 1.0, 0.5, 0.25, 0.125]

# Phasing confidence required to call a fragment as phased
FRAGMENT_PHASING_THRESHOLD = 0.995

# TAG names
RAW_BARCODE_TAG = "RX"
PROCESSED_BARCODE_TAG = "BX"
RAW_BARCODE_QUAL_TAG = "QX"
SAMPLE_INDEX_TAG = "BC"
SAMPLE_INDEX_QUAL_TAG = "QT"
PHASE_SET_BAM_TAG = "PS"
HAPLOTYPE_BAM_TAG = "HP"
PHASING_CONF_BAM_TAG = "PC"
TRIM_TAG = "TR"
TRIM_QUAL_TAG = "TQ"

MOLECULE_ID_BAM_TAG = "MI"

# Parallelization settings
PARALLEL_LOCUS_SIZE = int(4e7)
PARALLEL_NUM_READS_SIZE = 1000000

# Single partition fragment calling
FRAGMENT_LINK_DISTANCE = 30000

#
# Settings for metrics computation
#

# Quality cutoffs for bcs
BC_QUAL_CUTOFFS = [15, 20, 25, 30]

# Map quality cutoffs for insert sizes
INSERT_MAPQ_CUTOFFS = [0, 30, 60]

# Map quality cutoffs for target distances
TARGET_MAPQ_CUTOFFS = [0, 30, 60]

# What is considered a high confidence mapped single read
MODERATE_CONF_MAPQ = 29

# Longest insert size to tabulate
MAX_INSERT_SIZE = 10000

# Longest target distance to tabulate
MAX_TARGET_DIST = 10000

# Distance to consider reads far away for far chimeras
READ_MATE_FAR_DIST = 5000

# Distance to exclude directions of reads as untrustworthy for outer and same-dir chimeras
READ_MATE_CHIM_TOO_CLOSE_DIST = 20

# left coverage tail cutoff for customers
CUSTOMER_LEFT_TAIL_COVERAGE = 5

# Bin size for fragment length histogram passed to loupe
FRAG_LEN_HIST_BIN_SIZE = 100

# Bin size for calculating length-weighted mode of fragment size histogram
FRAG_LEN_HIST_BIN_SIZE_FOR_LW_MODE = 2000

# Variant Calling filter string
QUAL_FILTER = "(%QUAL <= 15 || (AF[0] > 0.5 && %QUAL < 50))"
LARIAT_RESCUE_FILTER = "(((RESCUED+NOT_RESCUED) > 0 & RESCUED/(RESCUED+NOT_RESCUED) > 0.1) & (MMD == -1 | MMD >= 3.0)) "
ALLELE_FRACTION_FILTER = "(AO[0] < 2 || AO[0]/(AO[0] + RO) < 0.15)"

VARIANT_CALL_FILTER = {
    "10X_RESCUED_MOLECULE_HIGH_DIVERSITY": LARIAT_RESCUE_FILTER,
    "10X_QUAL_FILTER": QUAL_FILTER,
    "10X_ALLELE_FRACTION_FILTER": ALLELE_FRACTION_FILTER,
}

VCF_WHITE_LIST_INFO_FIELDS = {"AA", "AN", "CIGAR", "END", "DB", "H2", "H3", "1000G", "SOMATIC", "VALIDATED", "TENX"}

# Preflight constants
MIN_PROCESS_NOFILE = 1024
MIN_GLOBAL_NOFILE = 2**15
REQUIRED_MIN_READS = 3
REQUIRED_NUM_READS = 3
GLOBAL_NOFILE_PATH = "/proc/sys/fs/file-max"
BCL_PROCESSOR_FASTQ_MODE = "BCL_PROCESSOR"
ILMN_BCL2FASTQ_FASTQ_MODE = "ILMN_BCL2FASTQ"
PACKAGE_VERSION_CMDS = [
    {
        "name": "mrc",
        "cmd": "mrc --version",
    },
    {
        "name": "mrp",
        "cmd": "mrp --version",
    },
    {
        "name": "Anaconda",
        "cmd": "python --version 2>&1 | cat ",
    },
    {"name": "numpy", "cmd": 'python -c "import numpy; print numpy.__version__"'},
    {"name": "scipy", "cmd": 'python -c "import scipy; print scipy.__version__"'},
    {"name": "pysam", "cmd": 'python -c "import pysam; print pysam.__version__"'},
    {"name": "PyVCF", "cmd": 'python -c "import vcf; print vcf.VERSION"'},
    {"name": "h5py", "cmd": 'python -c "import h5py; print h5py.__version__"'},
    {"name": "pandas", "cmd": 'python -c "import pandas; print pandas.__version__"'},
    {"name": "bwa", "cmd": 'bwa 2>&1 | grep "^ *Version"'},
    {"name": "samtools", "cmd": 'samtools 2>&1 | grep "^ *Version"'},
    {
        "name": "freebayes",
        "cmd": "freebayes -h | grep ^version",
    },
]

# Product microfluidics settings
FLUIDICS_PARAMS = {
    "GemCode": {
        "z2_vol_per_gem": 144e-12,  # 144pL
        "total_z2_vol_input": 65e-6,  # 65uL
    },
    "Chromium": {
        "z2_vol_per_gem": 24.5e-12,  # 24.5pL
        "total_z2_vol_input": 90e-6,  # 90uL
    },
}

# Sample index map

# GemCode Tubes
SI_001 = ["TCGCCATA", "GTATACAC", "AATGGTGG", "CGCATGCT"]
SI_002 = ["TATCCTCG", "GCGAGGTC", "CGCTTCAA", "ATAGAAGT"]
SI_003 = ["TGACGTCG", "CTTGTGTA", "ACGACCGT", "GACTAAAC"]
SI_004 = ["ATCTAGCT", "GAGCGTAC", "TCAGCCTG", "CGTATAGA"]
SI_005 = ["CCGTTCCC", "ATACAGTT", "TGTAGTAA", "GACGCAGG"]
SI_006 = ["TCAATTGG", "AGTTAGAA", "GAGCGCTT", "CTCGCACC"]
SI_007 = ["CTGCCTTG", "ACTAGCCC", "GGCTAGAT", "TAAGTAGA"]
SI_008 = ["GGCAGAAA", "ACGGTTCT", "CATTCGTC", "TTACACGG"]
# GemCode Plate
SI_P01_A1 = ["TTGTAAGA", "GGCGTTTC", "CCTACCAT", "AAACGGCG"]
SI_P01_B1 = ["CTAGCTGT", "GCCAACAA", "AGGCTACC", "TATTGGTG"]
SI_P01_C1 = ["GATGCAGT", "AGACTTTC", "TTCTACAG", "CCGAGGCA"]
SI_P01_D1 = ["AGCTGCGT", "GTGGAGCA", "TCTATTAG", "CAACCATC"]
SI_P01_E1 = ["CGCCCGTA", "GTTTGCCT", "TAAGTTAG", "ACGAAAGC"]
SI_P01_F1 = ["TGACTAGT", "GATAGGTA", "CCGTACAG", "ATCGCTCC"]
SI_P01_G1 = ["CATATGCG", "ATGCGATT", "TCCGCTAC", "GGATACGA"]
SI_P01_H1 = ["TCTTGTCC", "CGGGAGTA", "GTCACAGG", "AAACTCAT"]
SI_P01_A2 = ["AGCCCTTT", "TCTTAGGC", "GTGAGAAG", "CAAGTCCA"]
SI_P01_B2 = ["GGTCGAGC", "TTAGATTG", "CCCACCCA", "AAGTTGAT"]
SI_P01_C2 = ["CCGAGAAC", "TGCTCTGT", "GTAGTGCG", "AATCACTA"]
SI_P01_D2 = ["ACATTCCG", "GACACAAT", "CTGCGGTA", "TGTGATGC"]
SI_P01_E2 = ["TCATCAAG", "GTTGGTCC", "AGGCTGGT", "CACAACTA"]
SI_P01_F2 = ["TGCCCGCT", "GCAAACGC", "CATTGATA", "ATGGTTAG"]
SI_P01_G2 = ["CGGTGAGC", "ATAACCTA", "TCCGAGCG", "GATCTTAT"]
SI_P01_H2 = ["CCGAACTC", "AACGGTCA", "TTATTGGT", "GGTCCAAG"]
SI_P01_A3 = ["AAAGCATA", "GCCTTTAT", "CTGCAGCC", "TGTAGCGG"]
SI_P01_B3 = ["TCATCCTT", "ATTGGACG", "CAGCTTAC", "GGCAAGGA"]
SI_P01_C3 = ["ACGTTACA", "TTACCTAC", "GACGACGG", "CGTAGGTT"]
SI_P01_D3 = ["GAGCACGC", "CGAAGTTG", "TTCGTGAA", "ACTTCACT"]
SI_P01_E3 = ["TCTGCAGG", "CGGCTCCA", "AACAAGTC", "GTATGTAT"]
SI_P01_F3 = ["TTAGGACC", "AGTCTGTA", "GCCTCCGT", "CAGAATAG"]
SI_P01_G3 = ["TACGAGTT", "ATGTCCAG", "GCTATAGC", "CGACGTCA"]
SI_P01_H3 = ["TTGGGCTT", "GACAAACC", "ACACCTAA", "CGTTTGGG"]
SI_P01_A4 = ["CATGGCAG", "AGAACGCC", "GTCTTTGA", "TCGCAATT"]
SI_P01_B4 = ["GACAGGCT", "CCTCTAAC", "AGGGACTG", "TTATCTGA"]
SI_P01_C4 = ["ACATTGGC", "GAGCCCAT", "CTTAGTCA", "TGCGAATG"]
SI_P01_D4 = ["AAATCGTC", "GCGATCGG", "CTTCGAAT", "TGCGATCA"]
SI_P01_E4 = ["GTAAACAT", "TATCCTGA", "AGCTGACG", "CCGGTGTC"]
SI_P01_F4 = ["GCATGATA", "CGTCCTCT", "AACGACAC", "TTGATGGG"]
SI_P01_G4 = ["CCTGCGGT", "GTACAACG", "AGCTTCTC", "TAGAGTAA"]
SI_P01_H4 = ["TTCCATCT", "ACTGGAGC", "CGGTCGTG", "GAAATCAA"]
SI_P01_A5 = ["CAGTCTGG", "TCACACTC", "ATTGGGAA", "GGCATACT"]
SI_P01_B5 = ["TCATGCGA", "ATCGTACT", "CATCAGTG", "GGGACTAC"]
SI_P01_C5 = ["TCAGTCAA", "CACTGACT", "ATGCATTC", "GGTACGGG"]
SI_P01_D5 = ["GTCGACTC", "AGACGGAT", "CCTTTAGA", "TAGACTCG"]
SI_P01_E5 = ["CCGTTGAA", "TATGCTCT", "ATCCAAGG", "GGAAGCTC"]
SI_P01_F5 = ["TCTGACTA", "GTACGGCT", "CGGTTTAG", "AACACAGC"]
SI_P01_G5 = ["ATGAAGTA", "GAAGCTCG", "TCTTTCGT", "CGCCGAAC"]
SI_P01_H5 = ["ATAGTATG", "TATAAGGA", "GGCTCCAC", "CCGCGTCT"]
SI_P01_A6 = ["CTTTCGAC", "ACGGGACT", "TGCATCTG", "GAACATGA"]
SI_P01_B6 = ["GCGCACCT", "AACGCGAA", "CTATTTGG", "TGTAGATC"]
SI_P01_C6 = ["CGCTCAGG", "GAGGTTTA", "ACTCAGAC", "TTAAGCCT"]
SI_P01_D6 = ["GAAGTCTT", "TGCAGGGC", "ATGCCAAA", "CCTTATCG"]
SI_P01_E6 = ["TGATGGCT", "GCCATTTG", "ATTGAAAC", "CAGCCCGA"]
SI_P01_F6 = ["GACTTCCT", "TGAGGAAG", "ATGCCGGC", "CCTAATTA"]
SI_P01_G6 = ["GTGCGACA", "TCAGTGTT", "AGCACTGG", "CATTACAC"]
SI_P01_H6 = ["AAGCATAA", "CCCATCGC", "TTAGCGCT", "GGTTGATG"]
SI_P01_A7 = ["CTCATCAT", "TAACGTCC", "AGGTCATA", "GCTGAGGG"]
SI_P01_B7 = ["TCCACACG", "CTTCTGTT", "GAATGCAC", "AGGGATGA"]
SI_P01_C7 = ["GGCGGAAT", "ACACCGGG", "CATAATCC", "TTGTTCTA"]
SI_P01_D7 = ["CCGGATCC", "GGTCGCAT", "TTAACGTG", "AACTTAGA"]
SI_P01_E7 = ["AAGACGTG", "CCATGTGT", "GTTCACAA", "TGCGTACC"]
SI_P01_F7 = ["GGTTAGAC", "CAAACTTT", "ACCCGAGA", "TTGGTCCG"]
SI_P01_G7 = ["GCCGGTAA", "TGACTGCC", "ATTACCGG", "CAGTAATT"]
SI_P01_H7 = ["TGGCACGA", "AACGGGTG", "CTAATTCT", "GCTTCAAC"]
SI_P01_A8 = ["GACTGTTC", "ATGATACG", "CCACAGAA", "TGTGCCGT"]
SI_P01_B8 = ["ACGTTCAC", "TGCCCAGA", "CAAGGTCT", "GTTAAGTG"]
SI_P01_C8 = ["TTTATCCC", "GCACGTTT", "CAGGAAGA", "AGCTCGAG"]
SI_P01_D8 = ["AATCTTTG", "GGATGAGT", "CTCAAGAC", "TCGGCCCA"]
SI_P01_E8 = ["GCTTACAT", "TAGGGTGC", "AGCCTATG", "CTAACGCA"]
SI_P01_F8 = ["AGTTGGGA", "TACATTCT", "CCAGAAAG", "GTGCCCTC"]
SI_P01_G8 = ["AAGTACTC", "GGAACTCT", "TCCCTGAG", "CTTGGAGA"]
SI_P01_H8 = ["AAGAGCGG", "TCATAGCA", "GGCCCATC", "CTTGTTAT"]
SI_P01_A9 = ["GAGTGCGT", "CTCCAACA", "ACAACTTG", "TGTGTGAC"]
SI_P01_B9 = ["AAGCGTGT", "CTTGACCG", "TGATTAAC", "GCCACGTA"]
SI_P01_C9 = ["AGATCGGT", "CATCGTCG", "GTCATATA", "TCGGACAC"]
SI_P01_D9 = ["CAAGGGAC", "ACCTACTG", "GGGACACA", "TTTCTTGT"]
SI_P01_E9 = ["AGTAAGCA", "TACCGCGG", "CCGGTAAT", "GTATCTTC"]
SI_P01_F9 = ["AGTTAGTT", "GTACTTAA", "CACGCACG", "TCGAGCGC"]
SI_P01_G9 = ["TTGACTTC", "GCCGAAGT", "CAATGGCA", "AGTCTCAG"]
SI_P01_H9 = ["GGAATATG", "ACCTGCCA", "CTTCATAC", "TAGGCGGT"]
SI_P01_A10 = ["ACAGCAAC", "TTTCGCGA", "CGCAATTT", "GAGTTGCG"]
SI_P01_B10 = ["ACCATTAA", "CTGGACGT", "GAACGGTC", "TGTTCACG"]
SI_P01_C10 = ["CGTGCTAA", "TCACTCCT", "ATCTGATC", "GAGAAGGG"]
SI_P01_D10 = ["CTTATTTG", "GCGGGCAT", "AGATAACA", "TACCCGGC"]
SI_P01_E10 = ["GCACCAGT", "CGCAGGAG", "TAGTACCA", "ATTGTTTC"]
SI_P01_F10 = ["TCGACAAT", "GAATACTG", "ATTCGTGC", "CGCGTGCA"]
SI_P01_G10 = ["CGGAGACT", "TCCTATGA", "ATACTGAG", "GATGCCTC"]
SI_P01_H10 = ["GACCGCCA", "TCGAATTG", "ATTTCAGC", "CGAGTGAT"]
SI_P01_A11 = ["CTTTCCTT", "TAGGTAAA", "ACCAGTCC", "GGACAGGG"]
SI_P01_B11 = ["TCCAGATA", "GATTCGCT", "CGACATAG", "ATGGTCGC"]
SI_P01_C11 = ["GTTTGTGG", "ACCGAACA", "TAGACGAC", "CGACTCTT"]
SI_P01_D11 = ["GCTACTTC", "CACCTCAG", "ATATGAGA", "TGGGAGCT"]
SI_P01_E11 = ["ATCGCCAT", "TCACGGTA", "GGGTTTCC", "CATAAAGG"]
SI_P01_F11 = ["GAACCCGG", "AGCAGTTA", "TCGTAGAT", "CTTGTACC"]
SI_P01_G11 = ["AGGGCGTT", "CTATACGC", "TACATAAG", "GCTCGTCA"]
SI_P01_H11 = ["TCTCGACT", "AGGATCGA", "CACGATTC", "GTATCGAG"]
SI_P01_A12 = ["TTATGGAA", "ACTACTGT", "CGGGAACG", "GACCTCTC"]
SI_P01_B12 = ["GAAAGACA", "CGCTACAT", "ACGCTTGG", "TTTGCGTC"]
SI_P01_C12 = ["TAAGCCAC", "CCGTTATG", "GGTAATGT", "ATCCGGCA"]
SI_P01_D12 = ["GCTGTGTA", "AGAAACGT", "CACTCAAC", "TTGCGTCG"]
SI_P01_E12 = ["CGCTATCC", "ACGCGGAA", "TAAATCGT", "GTTGCATG"]
SI_P01_F12 = ["AATTGAAC", "TGGACCCT", "CCAGTGGA", "GTCCATTG"]
SI_P01_G12 = ["CATGCGTA", "ACCCGCAC", "TGATATCG", "GTGATAGT"]
SI_P01_H12 = ["TGTGTATA", "GTCCAGGC", "CAATCCCT", "ACGAGTAG"]

# WGS+ Tubes
SI_T2_1 = ["GGGTGATC", "TTACCGAT", "AATGACGA", "CCCATTCG"]
SI_T2_2 = ["GGGTCGAA", "ATCCGCCC", "TCTATAGT", "CAAGATTG"]
SI_T2_3 = ["GCTGATAT", "TGCCGAGC", "AAATTGCG", "CTGACCTA"]
SI_T2_4 = ["ACTTCTGA", "TTCATCTT", "CGACGACG", "GAGGAGAC"]
SI_T2_5 = ["GAATACAA", "AGCATACC", "TCGGGTTT", "CTTCCGGG"]
SI_T2_6 = ["TATTGAGA", "GTAGTCAG", "CGCCATTC", "ACGACGCT"]
SI_T2_7 = ["AAATCTGT", "GTCCAACC", "TCTGGCTG", "CGGATGAA"]
SI_T2_8 = ["CCTTGAAC", "GAAATCGG", "TGGCCTCT", "ATCGAGTA"]
# WGS+ Plate
SI_P2_A1 = ["GGTTTACT", "CTAAACGG", "TCGGCGTC", "AACCGTAA"]
SI_P2_A2 = ["TTTCATGA", "ACGTCCCT", "CGCATGTG", "GAAGGAAC"]
SI_P2_A3 = ["CAGTACTG", "AGTAGTCT", "GCAGTAGA", "TTCCCGAC"]
SI_P2_A4 = ["TATGATTC", "CCCACAGT", "ATGCTGAA", "GGATGCCG"]
SI_P2_A5 = ["CTAGGTGA", "TCGTTCAG", "AGCCAATT", "GATACGCC"]
SI_P2_A6 = ["CGCTATGT", "GCTGTCCA", "TTGAGATC", "AAACCGAG"]
SI_P2_A7 = ["ACAGAGGT", "TATAGTTG", "CGGTCCCA", "GTCCTAAC"]
SI_P2_A8 = ["GCATCTCC", "TGTAAGGT", "CTGCGATG", "AACGTCAA"]
SI_P2_A9 = ["TCTTAAAG", "CGAGGCTC", "GTCCTTCT", "AAGACGGA"]
SI_P2_A10 = ["GAAACCCT", "TTTCTGTC", "CCGTGTGA", "AGCGAAAG"]
SI_P2_A11 = ["GTCCGGTC", "AAGATCAT", "CCTGAAGG", "TGATCTCA"]
SI_P2_A12 = ["AGTGGAAC", "GTCTCCTT", "TCACATCA", "CAGATGGG"]
SI_P2_B1 = ["GTAATCTT", "TCCGGAAG", "AGTTCGGC", "CAGCATCA"]
SI_P2_B2 = ["TACTCTTC", "CCTGTGCG", "GGACACGT", "ATGAGAAA"]
SI_P2_B3 = ["GTGTATTA", "TGTGCGGG", "ACCATAAC", "CAACGCCT"]
SI_P2_B4 = ["ACTTCATA", "GAGATGAC", "TGCCGTGG", "CTAGACCT"]
SI_P2_B5 = ["AATAATGG", "CCAGGGCA", "TGCCTCAT", "GTGTCATC"]
SI_P2_B6 = ["CGTTAATC", "GCCACGCT", "TTACTCAG", "AAGGGTGA"]
SI_P2_B7 = ["AAACCTCA", "GCCTTGGT", "CTGGACTC", "TGTAGAAG"]
SI_P2_B8 = ["AAAGTGCT", "GCTACCTG", "TGCTGTAA", "CTGCAAGC"]
SI_P2_B9 = ["CTGTAACT", "TCTAGCGA", "AGAGTGTG", "GACCCTAC"]
SI_P2_B10 = ["ACCGTATG", "GATTAGAT", "CTGACTGA", "TGACGCCC"]
SI_P2_B11 = ["GTTCCTCA", "AGGTACGC", "TAAGTATG", "CCCAGGAT"]
SI_P2_B12 = ["TACCACCA", "CTAAGTTT", "GGGTCAAG", "ACTGTGGC"]
SI_P2_C1 = ["CCACTTAT", "AACTGGCG", "TTGGCATA", "GGTAACGC"]
SI_P2_C2 = ["CCTAGACC", "ATCTCTGT", "TAGCTCTA", "GGAGAGAG"]
SI_P2_C3 = ["TCAGCCGT", "CAGAGGCC", "GGTCAATA", "ATCTTTAG"]
SI_P2_C4 = ["ACAATTCA", "TGCGCAGC", "CATCACTT", "GTGTGGAG"]
SI_P2_C5 = ["CGACTTGA", "TACAGACT", "ATTGCGTG", "GCGTACAC"]
SI_P2_C6 = ["ATTACTTC", "TGCGAACT", "GCATTCGG", "CAGCGGAA"]
SI_P2_C7 = ["GTCTCTCG", "AATCTCTC", "CGGAGGGA", "TCAGAAAT"]
SI_P2_C8 = ["GTTGAGAA", "AGATCTGG", "TCGATACT", "CACCGCTC"]
SI_P2_C9 = ["GCGCAGAA", "ATCTTACC", "TATGGTGT", "CGAACCTG"]
SI_P2_C10 = ["TCTCAGTG", "GAGACTAT", "CGCTTAGC", "ATAGGCCA"]
SI_P2_C11 = ["GAGGATCT", "AGACCATA", "TCCTGCGC", "CTTATGAG"]
SI_P2_C12 = ["TCTCGTTT", "GGCTAGCG", "ATGACCGC", "CAAGTAAA"]
SI_P2_D1 = ["CACTCGGA", "GCTGAATT", "TGAAGTAC", "ATGCTCCG"]
SI_P2_D2 = ["TAACAAGG", "GGTTCCTC", "ATCATGCA", "CCGGGTAT"]
SI_P2_D3 = ["ACATTACT", "TTTGGGTA", "CAGCCCAC", "GGCAATGG"]
SI_P2_D4 = ["CCCTAACA", "ATTCCGAT", "TGGATTGC", "GAAGGCTG"]
SI_P2_D5 = ["CTCGTCAC", "GATCAGCA", "ACAACAGG", "TGGTGTTT"]
SI_P2_D6 = ["CATGCGAT", "TGATATTC", "GTGATCGA", "ACCCGACG"]
SI_P2_D7 = ["ATTTGCTA", "TAGACACC", "CCACAGGG", "GGCGTTAT"]
SI_P2_D8 = ["GCAACAAA", "TAGTTGTC", "CGCCATCG", "ATTGGCGT"]
SI_P2_D9 = ["AGGAGATG", "GATGTGGT", "CTACATCC", "TCCTCCAA"]
SI_P2_D10 = ["CAATACCC", "TGTCTATG", "ACCACGAA", "GTGGGTGT"]
SI_P2_D11 = ["CTTTGCGG", "TGCACAAA", "AAGCAGTC", "GCAGTTCT"]
SI_P2_D12 = ["GCACAATG", "CTTGGTAC", "TGCACCGT", "AAGTTGCA"]
SI_P2_E1 = ["TGGTAAAC", "GAAAGGGT", "ACTGCTCG", "CTCCTCTA"]
SI_P2_E2 = ["GTGGTACC", "TACTATAG", "ACAAGGTA", "CGTCCCGT"]
SI_P2_E3 = ["AGGTATTG", "CTCCTAGT", "TCAAGGCC", "GATGCCAA"]
SI_P2_E4 = ["TTCGCCCT", "GGATGGGC", "AATCAATG", "CCGATTAA"]
SI_P2_E5 = ["CATTAGCG", "TTCGCTGA", "ACAAGAAT", "GGGCTCTC"]
SI_P2_E6 = ["CTGCGGCT", "GACTCAAA", "AGAAACTC", "TCTGTTGG"]
SI_P2_E7 = ["CACGCCTT", "GTATATAG", "TCTCGGGC", "AGGATACA"]
SI_P2_E8 = ["ATAGTTAC", "TGCTGAGT", "CCTACGTA", "GAGCACCG"]
SI_P2_E9 = ["TTGTTTCC", "GGAGGAGG", "CCTAACAA", "AACCCGTT"]
SI_P2_E10 = ["AAATGTGC", "GGGCAAAT", "TCTATCCG", "CTCGCGTA"]
SI_P2_E11 = ["AAGCGCTG", "CGTTTGAT", "GTAGCACA", "TCCAATGC"]
SI_P2_E12 = ["ACCGGCTC", "GAGTTAGT", "CGTCCTAG", "TTAAAGCA"]
SI_P2_F1 = ["GTTGCAGC", "TGGAATTA", "CAATGGAG", "ACCCTCCT"]
SI_P2_F2 = ["TTTACATG", "CGCGATAC", "ACGCGGGT", "GAATTCCA"]
SI_P2_F3 = ["TTCAGGTG", "ACGGACAT", "GATCTTGA", "CGATCACC"]
SI_P2_F4 = ["CCCAATAG", "GTGTCGCT", "AGAGTCGC", "TATCGATA"]
SI_P2_F5 = ["GACTACGT", "CTAGCGAG", "TCTATATC", "AGGCGTCA"]
SI_P2_F6 = ["CGGAGCAC", "GACCTATT", "ACTTAGGA", "TTAGCTCG"]
SI_P2_F7 = ["CGTGCAGA", "AACAAGAT", "TCGCTTCG", "GTATGCTC"]
SI_P2_F8 = ["CATGAACA", "TCACTCGC", "AGCTGGAT", "GTGACTTG"]
SI_P2_F9 = ["CAAGCTCC", "GTTCACTG", "TCGTGAAA", "AGCATGGT"]
SI_P2_F10 = ["GCTTGGCT", "AAACAAAC", "CGGGCTTA", "TTCATCGG"]
SI_P2_F11 = ["GCGAGAGT", "TACGTTCA", "AGTCCCAC", "CTATAGTG"]
SI_P2_F12 = ["TGATGCAT", "GCTACTGA", "CACCTGCC", "ATGGAATG"]
SI_P2_G1 = ["ATGAATCT", "GATCTCAG", "CCAGGAGC", "TGCTCGTA"]
SI_P2_G2 = ["TGATTCTA", "ACTAGGAG", "CAGCCACT", "GTCGATGC"]
SI_P2_G3 = ["CCTCATTC", "AGCATCCG", "GTGGCAAT", "TAATGGGA"]
SI_P2_G4 = ["GCGATGTG", "AGATACAA", "TTTCCACT", "CACGGTGC"]
SI_P2_G5 = ["GAGCAAGA", "TCTGTGAT", "CGCAGTTC", "ATATCCCG"]
SI_P2_G6 = ["CTGACGCG", "GGTCGTAC", "TCCTTCTT", "AAAGAAGA"]
SI_P2_G7 = ["GGTATGCA", "CTCGAAAT", "ACACCTTC", "TAGTGCGG"]
SI_P2_G8 = ["TATGAGCT", "CCGATAGC", "ATACCCAA", "GGCTGTTG"]
SI_P2_G9 = ["TAGGACGT", "ATCCCACA", "GGAATGTC", "CCTTGTAG"]
SI_P2_G10 = ["TCGCCAGC", "AATGTTAG", "CGATAGCT", "GTCAGCTA"]
SI_P2_G11 = ["TTATCGTT", "AGCAGAGC", "CATCTCCA", "GCGGATAG"]
SI_P2_G12 = ["ATTCTAAG", "CCCGATTA", "TGGAGGCT", "GAATCCGC"]
SI_P2_H1 = ["GTATGTCA", "TGTCAGAC", "CACGTCGG", "ACGACATT"]
SI_P2_H2 = ["TAATGACC", "ATGCCTTA", "GCCGAGAT", "CGTATCGG"]
SI_P2_H3 = ["CCAAGATG", "AGGCCCGA", "TACGTGAC", "GTTTATCT"]
SI_P2_H4 = ["GCCATTCC", "CAAGAATT", "TTGCCGGA", "AGTTGCAG"]
SI_P2_H5 = ["CCACTACA", "GATTCTGG", "TGCGGCTT", "ATGAAGAC"]
SI_P2_H6 = ["TAGGATAA", "CCTTTGTC", "GTACGCGG", "AGCACACT"]
SI_P2_H7 = ["AGCTATCA", "CATATAAC", "TCAGGGTG", "GTGCCCGT"]
SI_P2_H8 = ["TTGTTGAT", "GCTCAACC", "CAAAGTGG", "AGCGCCTA"]
SI_P2_H9 = ["ACACTGTT", "CAGGATGG", "GGCTGAAC", "TTTACCCA"]
SI_P2_H10 = ["GTAATTGC", "AGTCGCTT", "CACGAGAA", "TCGTCACG"]
SI_P2_H11 = ["GGCGAGTA", "ACTTCTAT", "CAAATACG", "TTGCGCGC"]
SI_P2_H12 = ["GACAGCAT", "TTTGTACA", "AGGCCGTG", "CCATATGC"]

SI_NA_A1 = SI_3A_A1 = SI_P03_A1 = ["AAACGGCG", "CCTACCAT", "GGCGTTTC", "TTGTAAGA"]
SI_NA_B1 = SI_3A_B1 = SI_P03_B1 = ["AGGCTACC", "CTAGCTGT", "GCCAACAA", "TATTGGTG"]
SI_NA_C1 = SI_3A_C1 = SI_P03_C1 = ["AGACTTTC", "CCGAGGCA", "GATGCAGT", "TTCTACAG"]
SI_NA_D1 = SI_3A_D1 = SI_P03_D1 = ["AGCTGCGT", "CAACCATC", "GTGGAGCA", "TCTATTAG"]
SI_NA_E1 = SI_3A_E1 = SI_P03_E1 = ["ACGAAAGC", "CGCCCGTA", "GTTTGCCT", "TAAGTTAG"]
SI_NA_F1 = SI_3A_F1 = SI_P03_F1 = ["ATCGCTCC", "CCGTACAG", "GATAGGTA", "TGACTAGT"]
SI_NA_G1 = SI_3A_G1 = SI_P03_G1 = ["ATGCGATT", "CATATGCG", "GGATACGA", "TCCGCTAC"]
SI_NA_H1 = SI_3A_H1 = SI_P03_H1 = ["AAACTCAT", "CGGGAGTA", "GTCACAGG", "TCTTGTCC"]
SI_NA_A2 = SI_3A_A2 = SI_P03_A2 = ["AGCCCTTT", "CAAGTCCA", "GTGAGAAG", "TCTTAGGC"]
SI_NA_B2 = SI_3A_B2 = SI_P03_B2 = ["AAGTTGAT", "CCCACCCA", "GGTCGAGC", "TTAGATTG"]
SI_NA_C2 = SI_3A_C2 = SI_P03_C2 = ["AATCACTA", "CCGAGAAC", "GTAGTGCG", "TGCTCTGT"]
SI_NA_D2 = SI_3A_D2 = SI_P03_D2 = ["ACATTCCG", "CTGCGGTA", "GACACAAT", "TGTGATGC"]
SI_NA_E2 = SI_3A_E2 = SI_P03_E2 = ["AGGCTGGT", "CACAACTA", "GTTGGTCC", "TCATCAAG"]
SI_NA_F2 = SI_3A_F2 = SI_P03_F2 = ["ATGGTTAG", "CATTGATA", "GCAAACGC", "TGCCCGCT"]
SI_NA_G2 = SI_3A_G2 = SI_P03_G2 = ["ATAACCTA", "CGGTGAGC", "GATCTTAT", "TCCGAGCG"]
SI_NA_H2 = SI_3A_H2 = SI_P03_H2 = ["AACGGTCA", "CCGAACTC", "GGTCCAAG", "TTATTGGT"]
SI_NA_A3 = SI_3A_A3 = SI_P03_A3 = ["AAAGCATA", "CTGCAGCC", "GCCTTTAT", "TGTAGCGG"]
SI_NA_B3 = SI_3A_B3 = SI_P03_B3 = ["ATTGGACG", "CAGCTTAC", "GGCAAGGA", "TCATCCTT"]
SI_NA_C3 = SI_3A_C3 = SI_P03_C3 = ["ACGTTACA", "CGTAGGTT", "GACGACGG", "TTACCTAC"]
SI_NA_D3 = SI_3A_D3 = SI_P03_D3 = ["ACTTCACT", "CGAAGTTG", "GAGCACGC", "TTCGTGAA"]
SI_NA_E3 = SI_3A_E3 = SI_P03_E3 = ["AACAAGTC", "CGGCTCCA", "GTATGTAT", "TCTGCAGG"]
SI_NA_F3 = SI_3A_F3 = SI_P03_F3 = ["AGTCTGTA", "CAGAATAG", "GCCTCCGT", "TTAGGACC"]
SI_NA_G3 = SI_3A_G3 = SI_P03_G3 = ["ATGTCCAG", "CGACGTCA", "GCTATAGC", "TACGAGTT"]
SI_NA_H3 = SI_3A_H3 = SI_P03_H3 = ["ACACCTAA", "CGTTTGGG", "GACAAACC", "TTGGGCTT"]
SI_NA_A4 = SI_3A_A4 = SI_P03_A4 = ["AGAACGCC", "CATGGCAG", "GTCTTTGA", "TCGCAATT"]
SI_NA_B4 = SI_3A_B4 = SI_P03_B4 = ["AGGGACTG", "CCTCTAAC", "GACAGGCT", "TTATCTGA"]
SI_NA_C4 = SI_3A_C4 = SI_P03_C4 = ["ACATTGGC", "CTTAGTCA", "GAGCCCAT", "TGCGAATG"]
SI_NA_D4 = SI_3A_D4 = SI_P03_D4 = ["AAATCGTC", "CTTCGAAT", "GCGATCGG", "TGCGATCA"]
SI_NA_E4 = SI_3A_E4 = SI_P03_E4 = ["AGCTGACG", "CCGGTGTC", "GTAAACAT", "TATCCTGA"]
SI_NA_F4 = SI_3A_F4 = SI_P03_F4 = ["AACGACAC", "CGTCCTCT", "GCATGATA", "TTGATGGG"]
SI_NA_G4 = SI_3A_G4 = SI_P03_G4 = ["AGCTTCTC", "CCTGCGGT", "GTACAACG", "TAGAGTAA"]
SI_NA_H4 = SI_3A_H4 = SI_P03_H4 = ["ACTGGAGC", "CGGTCGTG", "GAAATCAA", "TTCCATCT"]
SI_NA_A5 = SI_3A_A5 = SI_P03_A5 = ["ATTGGGAA", "CAGTCTGG", "GGCATACT", "TCACACTC"]
SI_NA_B5 = SI_3A_B5 = SI_P03_B5 = ["ATCGTACT", "CATCAGTG", "GGGACTAC", "TCATGCGA"]
SI_NA_C5 = SI_3A_C5 = SI_P03_C5 = ["ATGCATTC", "CACTGACT", "GGTACGGG", "TCAGTCAA"]
SI_NA_D5 = SI_3A_D5 = SI_P03_D5 = ["AGACGGAT", "CCTTTAGA", "GTCGACTC", "TAGACTCG"]
SI_NA_E5 = SI_3A_E5 = SI_P03_E5 = ["ATCCAAGG", "CCGTTGAA", "GGAAGCTC", "TATGCTCT"]
SI_NA_F5 = SI_3A_F5 = SI_P03_F5 = ["AACACAGC", "CGGTTTAG", "GTACGGCT", "TCTGACTA"]
SI_NA_G5 = SI_3A_G5 = SI_P03_G5 = ["ATGAAGTA", "CGCCGAAC", "GAAGCTCG", "TCTTTCGT"]
SI_NA_H5 = SI_3A_H5 = SI_P03_H5 = ["ATAGTATG", "CCGCGTCT", "GGCTCCAC", "TATAAGGA"]
SI_NA_A6 = SI_3A_A6 = SI_P03_A6 = ["ACGGGACT", "CTTTCGAC", "GAACATGA", "TGCATCTG"]
SI_NA_B6 = SI_3A_B6 = SI_P03_B6 = ["AACGCGAA", "CTATTTGG", "GCGCACCT", "TGTAGATC"]
SI_NA_C6 = SI_3A_C6 = SI_P03_C6 = ["ACTCAGAC", "CGCTCAGG", "GAGGTTTA", "TTAAGCCT"]
SI_NA_D6 = SI_3A_D6 = SI_P03_D6 = ["ATGCCAAA", "CCTTATCG", "GAAGTCTT", "TGCAGGGC"]
SI_NA_E6 = SI_3A_E6 = SI_P03_E6 = ["ATTGAAAC", "CAGCCCGA", "GCCATTTG", "TGATGGCT"]
SI_NA_F6 = SI_3A_F6 = SI_P03_F6 = ["ATGCCGGC", "CCTAATTA", "GACTTCCT", "TGAGGAAG"]
SI_NA_G6 = SI_3A_G6 = SI_P03_G6 = ["AGCACTGG", "CATTACAC", "GTGCGACA", "TCAGTGTT"]
SI_NA_H6 = SI_3A_H6 = SI_P03_H6 = ["AAGCATAA", "CCCATCGC", "GGTTGATG", "TTAGCGCT"]
SI_NA_A7 = SI_3A_A7 = SI_P03_A7 = ["AGGTCATA", "CTCATCAT", "GCTGAGGG", "TAACGTCC"]
SI_NA_B7 = SI_3A_B7 = SI_P03_B7 = ["AGGGATGA", "CTTCTGTT", "GAATGCAC", "TCCACACG"]
SI_NA_C7 = SI_3A_C7 = SI_P03_C7 = ["ACACCGGG", "CATAATCC", "GGCGGAAT", "TTGTTCTA"]
SI_NA_D7 = SI_3A_D7 = SI_P03_D7 = ["AACTTAGA", "CCGGATCC", "GGTCGCAT", "TTAACGTG"]
SI_NA_E7 = SI_3A_E7 = SI_P03_E7 = ["AAGACGTG", "CCATGTGT", "GTTCACAA", "TGCGTACC"]
SI_NA_F7 = SI_3A_F7 = SI_P03_F7 = ["ACCCGAGA", "CAAACTTT", "GGTTAGAC", "TTGGTCCG"]
SI_NA_G7 = SI_3A_G7 = SI_P03_G7 = ["ATTACCGG", "CAGTAATT", "GCCGGTAA", "TGACTGCC"]
SI_NA_H7 = SI_3A_H7 = SI_P03_H7 = ["AACGGGTG", "CTAATTCT", "GCTTCAAC", "TGGCACGA"]
SI_NA_A8 = SI_3A_A8 = SI_P03_A8 = ["ATGATACG", "CCACAGAA", "GACTGTTC", "TGTGCCGT"]
SI_NA_B8 = SI_3A_B8 = SI_P03_B8 = ["ACGTTCAC", "CAAGGTCT", "GTTAAGTG", "TGCCCAGA"]
SI_NA_C8 = SI_3A_C8 = SI_P03_C8 = ["AGCTCGAG", "CAGGAAGA", "GCACGTTT", "TTTATCCC"]
SI_NA_D8 = SI_3A_D8 = SI_P03_D8 = ["AATCTTTG", "CTCAAGAC", "GGATGAGT", "TCGGCCCA"]
SI_NA_E8 = SI_3A_E8 = SI_P03_E8 = ["AGCCTATG", "CTAACGCA", "GCTTACAT", "TAGGGTGC"]
SI_NA_F8 = SI_3A_F8 = SI_P03_F8 = ["AGTTGGGA", "CCAGAAAG", "GTGCCCTC", "TACATTCT"]
SI_NA_G8 = SI_3A_G8 = SI_P03_G8 = ["AAGTACTC", "CTTGGAGA", "GGAACTCT", "TCCCTGAG"]
SI_NA_H8 = SI_3A_H8 = SI_P03_H8 = ["AAGAGCGG", "CTTGTTAT", "GGCCCATC", "TCATAGCA"]
SI_NA_A9 = SI_3A_A9 = SI_P03_A9 = ["ACAACTTG", "CTCCAACA", "GAGTGCGT", "TGTGTGAC"]
SI_NA_B9 = SI_3A_B9 = SI_P03_B9 = ["AAGCGTGT", "CTTGACCG", "GCCACGTA", "TGATTAAC"]
SI_NA_C9 = SI_3A_C9 = SI_P03_C9 = ["AGATCGGT", "CATCGTCG", "GTCATATA", "TCGGACAC"]
SI_NA_D9 = SI_3A_D9 = SI_P03_D9 = ["ACCTACTG", "CAAGGGAC", "GGGACACA", "TTTCTTGT"]
SI_NA_E9 = SI_3A_E9 = SI_P03_E9 = ["AGTAAGCA", "CCGGTAAT", "GTATCTTC", "TACCGCGG"]
SI_NA_F9 = SI_3A_F9 = SI_P03_F9 = ["AGTTAGTT", "CACGCACG", "GTACTTAA", "TCGAGCGC"]
SI_NA_G9 = SI_3A_G9 = SI_P03_G9 = ["AGTCTCAG", "CAATGGCA", "GCCGAAGT", "TTGACTTC"]
SI_NA_H9 = SI_3A_H9 = SI_P03_H9 = ["ACCTGCCA", "CTTCATAC", "GGAATATG", "TAGGCGGT"]
SI_NA_A10 = SI_3A_A10 = SI_P03_A10 = ["ACAGCAAC", "CGCAATTT", "GAGTTGCG", "TTTCGCGA"]
SI_NA_B10 = SI_3A_B10 = SI_P03_B10 = ["ACCATTAA", "CTGGACGT", "GAACGGTC", "TGTTCACG"]
SI_NA_C10 = SI_3A_C10 = SI_P03_C10 = ["ATCTGATC", "CGTGCTAA", "GAGAAGGG", "TCACTCCT"]
SI_NA_D10 = SI_3A_D10 = SI_P03_D10 = ["AGATAACA", "CTTATTTG", "GCGGGCAT", "TACCCGGC"]
SI_NA_E10 = SI_3A_E10 = SI_P03_E10 = ["ATTGTTTC", "CGCAGGAG", "GCACCAGT", "TAGTACCA"]
SI_NA_F10 = SI_3A_F10 = SI_P03_F10 = ["ATTCGTGC", "CGCGTGCA", "GAATACTG", "TCGACAAT"]
SI_NA_G10 = SI_3A_G10 = SI_P03_G10 = ["ATACTGAG", "CGGAGACT", "GATGCCTC", "TCCTATGA"]
SI_NA_H10 = SI_3A_H10 = SI_P03_H10 = ["ATTTCAGC", "CGAGTGAT", "GACCGCCA", "TCGAATTG"]
SI_NA_A11 = SI_3A_A11 = SI_P03_A11 = ["ACCAGTCC", "CTTTCCTT", "GGACAGGG", "TAGGTAAA"]
SI_NA_B11 = SI_3A_B11 = SI_P03_B11 = ["ATGGTCGC", "CGACATAG", "GATTCGCT", "TCCAGATA"]
SI_NA_C11 = SI_3A_C11 = SI_P03_C11 = ["ACCGAACA", "CGACTCTT", "GTTTGTGG", "TAGACGAC"]
SI_NA_D11 = SI_3A_D11 = SI_P03_D11 = ["ATATGAGA", "CACCTCAG", "GCTACTTC", "TGGGAGCT"]
SI_NA_E11 = SI_3A_E11 = SI_P03_E11 = ["ATCGCCAT", "CATAAAGG", "GGGTTTCC", "TCACGGTA"]
SI_NA_F11 = SI_3A_F11 = SI_P03_F11 = ["AGCAGTTA", "CTTGTACC", "GAACCCGG", "TCGTAGAT"]
SI_NA_G11 = SI_3A_G11 = SI_P03_G11 = ["AGGGCGTT", "CTATACGC", "GCTCGTCA", "TACATAAG"]
SI_NA_H11 = SI_3A_H11 = SI_P03_H11 = ["AGGATCGA", "CACGATTC", "GTATCGAG", "TCTCGACT"]
SI_NA_A12 = SI_3A_A12 = SI_P03_A12 = ["ACTACTGT", "CGGGAACG", "GACCTCTC", "TTATGGAA"]
SI_NA_B12 = SI_3A_B12 = SI_P03_B12 = ["ACGCTTGG", "CGCTACAT", "GAAAGACA", "TTTGCGTC"]
SI_NA_C12 = SI_3A_C12 = SI_P03_C12 = ["ATCCGGCA", "CCGTTATG", "GGTAATGT", "TAAGCCAC"]
SI_NA_D12 = SI_3A_D12 = SI_P03_D12 = ["AGAAACGT", "CACTCAAC", "GCTGTGTA", "TTGCGTCG"]
SI_NA_E12 = SI_3A_E12 = SI_P03_E12 = ["ACGCGGAA", "CGCTATCC", "GTTGCATG", "TAAATCGT"]
SI_NA_F12 = SI_3A_F12 = SI_P03_F12 = ["AATTGAAC", "CCAGTGGA", "GTCCATTG", "TGGACCCT"]
SI_NA_G12 = SI_3A_G12 = SI_P03_G12 = ["ACCCGCAC", "CATGCGTA", "GTGATAGT", "TGATATCG"]
SI_NA_H12 = SI_3A_H12 = SI_P03_H12 = ["ACGAGTAG", "CAATCCCT", "GTCCAGGC", "TGTGTATA"]

# WGS+ Tubes
SI_T2_1 = ["GGGTGATC", "TTACCGAT", "AATGACGA", "CCCATTCG"]
SI_T2_2 = ["GGGTCGAA", "ATCCGCCC", "TCTATAGT", "CAAGATTG"]
SI_T2_3 = ["GCTGATAT", "TGCCGAGC", "AAATTGCG", "CTGACCTA"]
SI_T2_4 = ["ACTTCTGA", "TTCATCTT", "CGACGACG", "GAGGAGAC"]
SI_T2_5 = ["GAATACAA", "AGCATACC", "TCGGGTTT", "CTTCCGGG"]
SI_T2_6 = ["TATTGAGA", "GTAGTCAG", "CGCCATTC", "ACGACGCT"]
SI_T2_7 = ["AAATCTGT", "GTCCAACC", "TCTGGCTG", "CGGATGAA"]
SI_T2_8 = ["CCTTGAAC", "GAAATCGG", "TGGCCTCT", "ATCGAGTA"]
# Chromium WGS Plate
SI_GA_A1 = SI_P2_A1 = ["GGTTTACT", "CTAAACGG", "TCGGCGTC", "AACCGTAA"]
SI_GA_A2 = SI_P2_A2 = ["TTTCATGA", "ACGTCCCT", "CGCATGTG", "GAAGGAAC"]
SI_GA_A3 = SI_P2_A3 = ["CAGTACTG", "AGTAGTCT", "GCAGTAGA", "TTCCCGAC"]
SI_GA_A4 = SI_P2_A4 = ["TATGATTC", "CCCACAGT", "ATGCTGAA", "GGATGCCG"]
SI_GA_A5 = SI_P2_A5 = ["CTAGGTGA", "TCGTTCAG", "AGCCAATT", "GATACGCC"]
SI_GA_A6 = SI_P2_A6 = ["CGCTATGT", "GCTGTCCA", "TTGAGATC", "AAACCGAG"]
SI_GA_A7 = SI_P2_A7 = ["ACAGAGGT", "TATAGTTG", "CGGTCCCA", "GTCCTAAC"]
SI_GA_A8 = SI_P2_A8 = ["GCATCTCC", "TGTAAGGT", "CTGCGATG", "AACGTCAA"]
SI_GA_A9 = SI_P2_A9 = ["TCTTAAAG", "CGAGGCTC", "GTCCTTCT", "AAGACGGA"]
SI_GA_A10 = SI_P2_A10 = ["GAAACCCT", "TTTCTGTC", "CCGTGTGA", "AGCGAAAG"]
SI_GA_A11 = SI_P2_A11 = ["GTCCGGTC", "AAGATCAT", "CCTGAAGG", "TGATCTCA"]
SI_GA_A12 = SI_P2_A12 = ["AGTGGAAC", "GTCTCCTT", "TCACATCA", "CAGATGGG"]
SI_GA_B1 = SI_P2_B1 = ["GTAATCTT", "TCCGGAAG", "AGTTCGGC", "CAGCATCA"]
SI_GA_B2 = SI_P2_B2 = ["TACTCTTC", "CCTGTGCG", "GGACACGT", "ATGAGAAA"]
SI_GA_B3 = SI_P2_B3 = ["GTGTATTA", "TGTGCGGG", "ACCATAAC", "CAACGCCT"]
SI_GA_B4 = SI_P2_B4 = ["ACTTCATA", "GAGATGAC", "TGCCGTGG", "CTAGACCT"]
SI_GA_B5 = SI_P2_B5 = ["AATAATGG", "CCAGGGCA", "TGCCTCAT", "GTGTCATC"]
SI_GA_B6 = SI_P2_B6 = ["CGTTAATC", "GCCACGCT", "TTACTCAG", "AAGGGTGA"]
SI_GA_B7 = SI_P2_B7 = ["AAACCTCA", "GCCTTGGT", "CTGGACTC", "TGTAGAAG"]
SI_GA_B8 = SI_P2_B8 = ["AAAGTGCT", "GCTACCTG", "TGCTGTAA", "CTGCAAGC"]
SI_GA_B9 = SI_P2_B9 = ["CTGTAACT", "TCTAGCGA", "AGAGTGTG", "GACCCTAC"]
SI_GA_B10 = SI_P2_B10 = ["ACCGTATG", "GATTAGAT", "CTGACTGA", "TGACGCCC"]
SI_GA_B11 = SI_P2_B11 = ["GTTCCTCA", "AGGTACGC", "TAAGTATG", "CCCAGGAT"]
SI_GA_B12 = SI_P2_B12 = ["TACCACCA", "CTAAGTTT", "GGGTCAAG", "ACTGTGGC"]
SI_GA_C1 = SI_P2_C1 = ["CCACTTAT", "AACTGGCG", "TTGGCATA", "GGTAACGC"]
SI_GA_C2 = SI_P2_C2 = ["CCTAGACC", "ATCTCTGT", "TAGCTCTA", "GGAGAGAG"]
SI_GA_C3 = SI_P2_C3 = ["TCAGCCGT", "CAGAGGCC", "GGTCAATA", "ATCTTTAG"]
SI_GA_C4 = SI_P2_C4 = ["ACAATTCA", "TGCGCAGC", "CATCACTT", "GTGTGGAG"]
SI_GA_C5 = SI_P2_C5 = ["CGACTTGA", "TACAGACT", "ATTGCGTG", "GCGTACAC"]
SI_GA_C6 = SI_P2_C6 = ["ATTACTTC", "TGCGAACT", "GCATTCGG", "CAGCGGAA"]
SI_GA_C7 = SI_P2_C7 = ["GTCTCTCG", "AATCTCTC", "CGGAGGGA", "TCAGAAAT"]
SI_GA_C8 = SI_P2_C8 = ["GTTGAGAA", "AGATCTGG", "TCGATACT", "CACCGCTC"]
SI_GA_C9 = SI_P2_C9 = ["GCGCAGAA", "ATCTTACC", "TATGGTGT", "CGAACCTG"]
SI_GA_C10 = SI_P2_C10 = ["TCTCAGTG", "GAGACTAT", "CGCTTAGC", "ATAGGCCA"]
SI_GA_C11 = SI_P2_C11 = ["GAGGATCT", "AGACCATA", "TCCTGCGC", "CTTATGAG"]
SI_GA_C12 = SI_P2_C12 = ["TCTCGTTT", "GGCTAGCG", "ATGACCGC", "CAAGTAAA"]
SI_GA_D1 = SI_P2_D1 = ["CACTCGGA", "GCTGAATT", "TGAAGTAC", "ATGCTCCG"]
SI_GA_D2 = SI_P2_D2 = ["TAACAAGG", "GGTTCCTC", "ATCATGCA", "CCGGGTAT"]
SI_GA_D3 = SI_P2_D3 = ["ACATTACT", "TTTGGGTA", "CAGCCCAC", "GGCAATGG"]
SI_GA_D4 = SI_P2_D4 = ["CCCTAACA", "ATTCCGAT", "TGGATTGC", "GAAGGCTG"]
SI_GA_D5 = SI_P2_D5 = ["CTCGTCAC", "GATCAGCA", "ACAACAGG", "TGGTGTTT"]
SI_GA_D6 = SI_P2_D6 = ["CATGCGAT", "TGATATTC", "GTGATCGA", "ACCCGACG"]
SI_GA_D7 = SI_P2_D7 = ["ATTTGCTA", "TAGACACC", "CCACAGGG", "GGCGTTAT"]
SI_GA_D8 = SI_P2_D8 = ["GCAACAAA", "TAGTTGTC", "CGCCATCG", "ATTGGCGT"]
SI_GA_D9 = SI_P2_D9 = ["AGGAGATG", "GATGTGGT", "CTACATCC", "TCCTCCAA"]
SI_GA_D10 = SI_P2_D10 = ["CAATACCC", "TGTCTATG", "ACCACGAA", "GTGGGTGT"]
SI_GA_D11 = SI_P2_D11 = ["CTTTGCGG", "TGCACAAA", "AAGCAGTC", "GCAGTTCT"]
SI_GA_D12 = SI_P2_D12 = ["GCACAATG", "CTTGGTAC", "TGCACCGT", "AAGTTGCA"]
SI_GA_E1 = SI_P2_E1 = ["TGGTAAAC", "GAAAGGGT", "ACTGCTCG", "CTCCTCTA"]
SI_GA_E2 = SI_P2_E2 = ["GTGGTACC", "TACTATAG", "ACAAGGTA", "CGTCCCGT"]
SI_GA_E3 = SI_P2_E3 = ["AGGTATTG", "CTCCTAGT", "TCAAGGCC", "GATGCCAA"]
SI_GA_E4 = SI_P2_E4 = ["TTCGCCCT", "GGATGGGC", "AATCAATG", "CCGATTAA"]
SI_GA_E5 = SI_P2_E5 = ["CATTAGCG", "TTCGCTGA", "ACAAGAAT", "GGGCTCTC"]
SI_GA_E6 = SI_P2_E6 = ["CTGCGGCT", "GACTCAAA", "AGAAACTC", "TCTGTTGG"]
SI_GA_E7 = SI_P2_E7 = ["CACGCCTT", "GTATATAG", "TCTCGGGC", "AGGATACA"]
SI_GA_E8 = SI_P2_E8 = ["ATAGTTAC", "TGCTGAGT", "CCTACGTA", "GAGCACCG"]
SI_GA_E9 = SI_P2_E9 = ["TTGTTTCC", "GGAGGAGG", "CCTAACAA", "AACCCGTT"]
SI_GA_E10 = SI_P2_E10 = ["AAATGTGC", "GGGCAAAT", "TCTATCCG", "CTCGCGTA"]
SI_GA_E11 = SI_P2_E11 = ["AAGCGCTG", "CGTTTGAT", "GTAGCACA", "TCCAATGC"]
SI_GA_E12 = SI_P2_E12 = ["ACCGGCTC", "GAGTTAGT", "CGTCCTAG", "TTAAAGCA"]
SI_GA_F1 = SI_P2_F1 = ["GTTGCAGC", "TGGAATTA", "CAATGGAG", "ACCCTCCT"]
SI_GA_F2 = SI_P2_F2 = ["TTTACATG", "CGCGATAC", "ACGCGGGT", "GAATTCCA"]
SI_GA_F3 = SI_P2_F3 = ["TTCAGGTG", "ACGGACAT", "GATCTTGA", "CGATCACC"]
SI_GA_F4 = SI_P2_F4 = ["CCCAATAG", "GTGTCGCT", "AGAGTCGC", "TATCGATA"]
SI_GA_F5 = SI_P2_F5 = ["GACTACGT", "CTAGCGAG", "TCTATATC", "AGGCGTCA"]
SI_GA_F6 = SI_P2_F6 = ["CGGAGCAC", "GACCTATT", "ACTTAGGA", "TTAGCTCG"]
SI_GA_F7 = SI_P2_F7 = ["CGTGCAGA", "AACAAGAT", "TCGCTTCG", "GTATGCTC"]
SI_GA_F8 = SI_P2_F8 = ["CATGAACA", "TCACTCGC", "AGCTGGAT", "GTGACTTG"]
SI_GA_F9 = SI_P2_F9 = ["CAAGCTCC", "GTTCACTG", "TCGTGAAA", "AGCATGGT"]
SI_GA_F10 = SI_P2_F10 = ["GCTTGGCT", "AAACAAAC", "CGGGCTTA", "TTCATCGG"]
SI_GA_F11 = SI_P2_F11 = ["GCGAGAGT", "TACGTTCA", "AGTCCCAC", "CTATAGTG"]
SI_GA_F12 = SI_P2_F12 = ["TGATGCAT", "GCTACTGA", "CACCTGCC", "ATGGAATG"]
SI_GA_G1 = SI_P2_G1 = ["ATGAATCT", "GATCTCAG", "CCAGGAGC", "TGCTCGTA"]
SI_GA_G2 = SI_P2_G2 = ["TGATTCTA", "ACTAGGAG", "CAGCCACT", "GTCGATGC"]
SI_GA_G3 = SI_P2_G3 = ["CCTCATTC", "AGCATCCG", "GTGGCAAT", "TAATGGGA"]
SI_GA_G4 = SI_P2_G4 = ["GCGATGTG", "AGATACAA", "TTTCCACT", "CACGGTGC"]
SI_GA_G5 = SI_P2_G5 = ["GAGCAAGA", "TCTGTGAT", "CGCAGTTC", "ATATCCCG"]
SI_GA_G6 = SI_P2_G6 = ["CTGACGCG", "GGTCGTAC", "TCCTTCTT", "AAAGAAGA"]
SI_GA_G7 = SI_P2_G7 = ["GGTATGCA", "CTCGAAAT", "ACACCTTC", "TAGTGCGG"]
SI_GA_G8 = SI_P2_G8 = ["TATGAGCT", "CCGATAGC", "ATACCCAA", "GGCTGTTG"]
SI_GA_G9 = SI_P2_G9 = ["TAGGACGT", "ATCCCACA", "GGAATGTC", "CCTTGTAG"]
SI_GA_G10 = SI_P2_G10 = ["TCGCCAGC", "AATGTTAG", "CGATAGCT", "GTCAGCTA"]
SI_GA_G11 = SI_P2_G11 = ["TTATCGTT", "AGCAGAGC", "CATCTCCA", "GCGGATAG"]
SI_GA_G12 = SI_P2_G12 = ["ATTCTAAG", "CCCGATTA", "TGGAGGCT", "GAATCCGC"]
SI_GA_H1 = SI_P2_H1 = ["GTATGTCA", "TGTCAGAC", "CACGTCGG", "ACGACATT"]
SI_GA_H2 = SI_P2_H2 = ["TAATGACC", "ATGCCTTA", "GCCGAGAT", "CGTATCGG"]
SI_GA_H3 = SI_P2_H3 = ["CCAAGATG", "AGGCCCGA", "TACGTGAC", "GTTTATCT"]
SI_GA_H4 = SI_P2_H4 = ["GCCATTCC", "CAAGAATT", "TTGCCGGA", "AGTTGCAG"]
SI_GA_H5 = SI_P2_H5 = ["CCACTACA", "GATTCTGG", "TGCGGCTT", "ATGAAGAC"]
SI_GA_H6 = SI_P2_H6 = ["TAGGATAA", "CCTTTGTC", "GTACGCGG", "AGCACACT"]
SI_GA_H7 = SI_P2_H7 = ["AGCTATCA", "CATATAAC", "TCAGGGTG", "GTGCCCGT"]
SI_GA_H8 = SI_P2_H8 = ["TTGTTGAT", "GCTCAACC", "CAAAGTGG", "AGCGCCTA"]
SI_GA_H9 = SI_P2_H9 = ["ACACTGTT", "CAGGATGG", "GGCTGAAC", "TTTACCCA"]
SI_GA_H10 = SI_P2_H10 = ["GTAATTGC", "AGTCGCTT", "CACGAGAA", "TCGTCACG"]
SI_GA_H11 = SI_P2_H11 = ["GGCGAGTA", "ACTTCTAT", "CAAATACG", "TTGCGCGC"]
SI_GA_H12 = SI_P2_H12 = ["GACAGCAT", "TTTGTACA", "AGGCCGTG", "CCATATGC"]

SAMPLE_INDEX_MAP = {
    # GemCode Tube labels
    "SI-001": SI_001,
    "SI-002": SI_002,
    "SI-003": SI_003,
    "SI-004": SI_004,
    "SI-005": SI_005,
    "SI-006": SI_006,
    "SI-007": SI_007,
    "SI-008": SI_008,
    # GemCode Plate labels
    "SI-P01-A1": SI_P01_A1,
    "SI-P01-B1": SI_P01_B1,
    "SI-P01-C1": SI_P01_C1,
    "SI-P01-D1": SI_P01_D1,
    "SI-P01-E1": SI_P01_E1,
    "SI-P01-F1": SI_P01_F1,
    "SI-P01-G1": SI_P01_G1,
    "SI-P01-H1": SI_P01_H1,
    "SI-P01-A2": SI_P01_A2,
    "SI-P01-B2": SI_P01_B2,
    "SI-P01-C2": SI_P01_C2,
    "SI-P01-D2": SI_P01_D2,
    "SI-P01-E2": SI_P01_E2,
    "SI-P01-F2": SI_P01_F2,
    "SI-P01-G2": SI_P01_G2,
    "SI-P01-H2": SI_P01_H2,
    "SI-P01-A3": SI_P01_A3,
    "SI-P01-B3": SI_P01_B3,
    "SI-P01-C3": SI_P01_C3,
    "SI-P01-D3": SI_P01_D3,
    "SI-P01-E3": SI_P01_E3,
    "SI-P01-F3": SI_P01_F3,
    "SI-P01-G3": SI_P01_G3,
    "SI-P01-H3": SI_P01_H3,
    "SI-P01-A4": SI_P01_A4,
    "SI-P01-B4": SI_P01_B4,
    "SI-P01-C4": SI_P01_C4,
    "SI-P01-D4": SI_P01_D4,
    "SI-P01-E4": SI_P01_E4,
    "SI-P01-F4": SI_P01_F4,
    "SI-P01-G4": SI_P01_G4,
    "SI-P01-H4": SI_P01_H4,
    "SI-P01-A5": SI_P01_A5,
    "SI-P01-B5": SI_P01_B5,
    "SI-P01-C5": SI_P01_C5,
    "SI-P01-D5": SI_P01_D5,
    "SI-P01-E5": SI_P01_E5,
    "SI-P01-F5": SI_P01_F5,
    "SI-P01-G5": SI_P01_G5,
    "SI-P01-H5": SI_P01_H5,
    "SI-P01-A6": SI_P01_A6,
    "SI-P01-B6": SI_P01_B6,
    "SI-P01-C6": SI_P01_C6,
    "SI-P01-D6": SI_P01_D6,
    "SI-P01-E6": SI_P01_E6,
    "SI-P01-F6": SI_P01_F6,
    "SI-P01-G6": SI_P01_G6,
    "SI-P01-H6": SI_P01_H6,
    "SI-P01-A7": SI_P01_A7,
    "SI-P01-B7": SI_P01_B7,
    "SI-P01-C7": SI_P01_C7,
    "SI-P01-D7": SI_P01_D7,
    "SI-P01-E7": SI_P01_E7,
    "SI-P01-F7": SI_P01_F7,
    "SI-P01-G7": SI_P01_G7,
    "SI-P01-H7": SI_P01_H7,
    "SI-P01-A8": SI_P01_A8,
    "SI-P01-B8": SI_P01_B8,
    "SI-P01-C8": SI_P01_C8,
    "SI-P01-D8": SI_P01_D8,
    "SI-P01-E8": SI_P01_E8,
    "SI-P01-F8": SI_P01_F8,
    "SI-P01-G8": SI_P01_G8,
    "SI-P01-H8": SI_P01_H8,
    "SI-P01-A9": SI_P01_A9,
    "SI-P01-B9": SI_P01_B9,
    "SI-P01-C9": SI_P01_C9,
    "SI-P01-D9": SI_P01_D9,
    "SI-P01-E9": SI_P01_E9,
    "SI-P01-F9": SI_P01_F9,
    "SI-P01-G9": SI_P01_G9,
    "SI-P01-H9": SI_P01_H9,
    "SI-P01-A10": SI_P01_A10,
    "SI-P01-B10": SI_P01_B10,
    "SI-P01-C10": SI_P01_C10,
    "SI-P01-D10": SI_P01_D10,
    "SI-P01-E10": SI_P01_E10,
    "SI-P01-F10": SI_P01_F10,
    "SI-P01-G10": SI_P01_G10,
    "SI-P01-H10": SI_P01_H10,
    "SI-P01-A11": SI_P01_A11,
    "SI-P01-B11": SI_P01_B11,
    "SI-P01-C11": SI_P01_C11,
    "SI-P01-D11": SI_P01_D11,
    "SI-P01-E11": SI_P01_E11,
    "SI-P01-F11": SI_P01_F11,
    "SI-P01-G11": SI_P01_G11,
    "SI-P01-H11": SI_P01_H11,
    "SI-P01-A12": SI_P01_A12,
    "SI-P01-B12": SI_P01_B12,
    "SI-P01-C12": SI_P01_C12,
    "SI-P01-D12": SI_P01_D12,
    "SI-P01-E12": SI_P01_E12,
    "SI-P01-F12": SI_P01_F12,
    "SI-P01-G12": SI_P01_G12,
    "SI-P01-H12": SI_P01_H12,
    "SI-P03-A1": SI_P03_A1,
    "SI-P03-B1": SI_P03_B1,
    "SI-P03-C1": SI_P03_C1,
    "SI-P03-D1": SI_P03_D1,
    "SI-P03-E1": SI_P03_E1,
    "SI-P03-F1": SI_P03_F1,
    "SI-P03-G1": SI_P03_G1,
    "SI-P03-H1": SI_P03_H1,
    "SI-P03-A2": SI_P03_A2,
    "SI-P03-B2": SI_P03_B2,
    "SI-P03-C2": SI_P03_C2,
    "SI-P03-D2": SI_P03_D2,
    "SI-P03-E2": SI_P03_E2,
    "SI-P03-F2": SI_P03_F2,
    "SI-P03-G2": SI_P03_G2,
    "SI-P03-H2": SI_P03_H2,
    "SI-P03-A3": SI_P03_A3,
    "SI-P03-B3": SI_P03_B3,
    "SI-P03-C3": SI_P03_C3,
    "SI-P03-D3": SI_P03_D3,
    "SI-P03-E3": SI_P03_E3,
    "SI-P03-F3": SI_P03_F3,
    "SI-P03-G3": SI_P03_G3,
    "SI-P03-H3": SI_P03_H3,
    "SI-P03-A4": SI_P03_A4,
    "SI-P03-B4": SI_P03_B4,
    "SI-P03-C4": SI_P03_C4,
    "SI-P03-D4": SI_P03_D4,
    "SI-P03-E4": SI_P03_E4,
    "SI-P03-F4": SI_P03_F4,
    "SI-P03-G4": SI_P03_G4,
    "SI-P03-H4": SI_P03_H4,
    "SI-P03-A5": SI_P03_A5,
    "SI-P03-B5": SI_P03_B5,
    "SI-P03-C5": SI_P03_C5,
    "SI-P03-D5": SI_P03_D5,
    "SI-P03-E5": SI_P03_E5,
    "SI-P03-F5": SI_P03_F5,
    "SI-P03-G5": SI_P03_G5,
    "SI-P03-H5": SI_P03_H5,
    "SI-P03-A6": SI_P03_A6,
    "SI-P03-B6": SI_P03_B6,
    "SI-P03-C6": SI_P03_C6,
    "SI-P03-D6": SI_P03_D6,
    "SI-P03-E6": SI_P03_E6,
    "SI-P03-F6": SI_P03_F6,
    "SI-P03-G6": SI_P03_G6,
    "SI-P03-H6": SI_P03_H6,
    "SI-P03-A7": SI_P03_A7,
    "SI-P03-B7": SI_P03_B7,
    "SI-P03-C7": SI_P03_C7,
    "SI-P03-D7": SI_P03_D7,
    "SI-P03-E7": SI_P03_E7,
    "SI-P03-F7": SI_P03_F7,
    "SI-P03-G7": SI_P03_G7,
    "SI-P03-H7": SI_P03_H7,
    "SI-P03-A8": SI_P03_A8,
    "SI-P03-B8": SI_P03_B8,
    "SI-P03-C8": SI_P03_C8,
    "SI-P03-D8": SI_P03_D8,
    "SI-P03-E8": SI_P03_E8,
    "SI-P03-F8": SI_P03_F8,
    "SI-P03-G8": SI_P03_G8,
    "SI-P03-H8": SI_P03_H8,
    "SI-P03-A9": SI_P03_A9,
    "SI-P03-B9": SI_P03_B9,
    "SI-P03-C9": SI_P03_C9,
    "SI-P03-D9": SI_P03_D9,
    "SI-P03-E9": SI_P03_E9,
    "SI-P03-F9": SI_P03_F9,
    "SI-P03-G9": SI_P03_G9,
    "SI-P03-H9": SI_P03_H9,
    "SI-P03-A10": SI_P03_A10,
    "SI-P03-B10": SI_P03_B10,
    "SI-P03-C10": SI_P03_C10,
    "SI-P03-D10": SI_P03_D10,
    "SI-P03-E10": SI_P03_E10,
    "SI-P03-F10": SI_P03_F10,
    "SI-P03-G10": SI_P03_G10,
    "SI-P03-H10": SI_P03_H10,
    "SI-P03-A11": SI_P03_A11,
    "SI-P03-B11": SI_P03_B11,
    "SI-P03-C11": SI_P03_C11,
    "SI-P03-D11": SI_P03_D11,
    "SI-P03-E11": SI_P03_E11,
    "SI-P03-F11": SI_P03_F11,
    "SI-P03-G11": SI_P03_G11,
    "SI-P03-H11": SI_P03_H11,
    "SI-P03-A12": SI_P03_A12,
    "SI-P03-B12": SI_P03_B12,
    "SI-P03-C12": SI_P03_C12,
    "SI-P03-D12": SI_P03_D12,
    "SI-P03-E12": SI_P03_E12,
    "SI-P03-F12": SI_P03_F12,
    "SI-P03-G12": SI_P03_G12,
    "SI-P03-H12": SI_P03_H12,
    "SI-3A-A1": SI_3A_A1,
    "SI-3A-B1": SI_3A_B1,
    "SI-3A-C1": SI_3A_C1,
    "SI-3A-D1": SI_3A_D1,
    "SI-3A-E1": SI_3A_E1,
    "SI-3A-F1": SI_3A_F1,
    "SI-3A-G1": SI_3A_G1,
    "SI-3A-H1": SI_3A_H1,
    "SI-3A-A2": SI_3A_A2,
    "SI-3A-B2": SI_3A_B2,
    "SI-3A-C2": SI_3A_C2,
    "SI-3A-D2": SI_3A_D2,
    "SI-3A-E2": SI_3A_E2,
    "SI-3A-F2": SI_3A_F2,
    "SI-3A-G2": SI_3A_G2,
    "SI-3A-H2": SI_3A_H2,
    "SI-3A-A3": SI_3A_A3,
    "SI-3A-B3": SI_3A_B3,
    "SI-3A-C3": SI_3A_C3,
    "SI-3A-D3": SI_3A_D3,
    "SI-3A-E3": SI_3A_E3,
    "SI-3A-F3": SI_3A_F3,
    "SI-3A-G3": SI_3A_G3,
    "SI-3A-H3": SI_3A_H3,
    "SI-3A-A4": SI_3A_A4,
    "SI-3A-B4": SI_3A_B4,
    "SI-3A-C4": SI_3A_C4,
    "SI-3A-D4": SI_3A_D4,
    "SI-3A-E4": SI_3A_E4,
    "SI-3A-F4": SI_3A_F4,
    "SI-3A-G4": SI_3A_G4,
    "SI-3A-H4": SI_3A_H4,
    "SI-3A-A5": SI_3A_A5,
    "SI-3A-B5": SI_3A_B5,
    "SI-3A-C5": SI_3A_C5,
    "SI-3A-D5": SI_3A_D5,
    "SI-3A-E5": SI_3A_E5,
    "SI-3A-F5": SI_3A_F5,
    "SI-3A-G5": SI_3A_G5,
    "SI-3A-H5": SI_3A_H5,
    "SI-3A-A6": SI_3A_A6,
    "SI-3A-B6": SI_3A_B6,
    "SI-3A-C6": SI_3A_C6,
    "SI-3A-D6": SI_3A_D6,
    "SI-3A-E6": SI_3A_E6,
    "SI-3A-F6": SI_3A_F6,
    "SI-3A-G6": SI_3A_G6,
    "SI-3A-H6": SI_3A_H6,
    "SI-3A-A7": SI_3A_A7,
    "SI-3A-B7": SI_3A_B7,
    "SI-3A-C7": SI_3A_C7,
    "SI-3A-D7": SI_3A_D7,
    "SI-3A-E7": SI_3A_E7,
    "SI-3A-F7": SI_3A_F7,
    "SI-3A-G7": SI_3A_G7,
    "SI-3A-H7": SI_3A_H7,
    "SI-3A-A8": SI_3A_A8,
    "SI-3A-B8": SI_3A_B8,
    "SI-3A-C8": SI_3A_C8,
    "SI-3A-D8": SI_3A_D8,
    "SI-3A-E8": SI_3A_E8,
    "SI-3A-F8": SI_3A_F8,
    "SI-3A-G8": SI_3A_G8,
    "SI-3A-H8": SI_3A_H8,
    "SI-3A-A9": SI_3A_A9,
    "SI-3A-B9": SI_3A_B9,
    "SI-3A-C9": SI_3A_C9,
    "SI-3A-D9": SI_3A_D9,
    "SI-3A-E9": SI_3A_E9,
    "SI-3A-F9": SI_3A_F9,
    "SI-3A-G9": SI_3A_G9,
    "SI-3A-H9": SI_3A_H9,
    "SI-3A-A10": SI_3A_A10,
    "SI-3A-B10": SI_3A_B10,
    "SI-3A-C10": SI_3A_C10,
    "SI-3A-D10": SI_3A_D10,
    "SI-3A-E10": SI_3A_E10,
    "SI-3A-F10": SI_3A_F10,
    "SI-3A-G10": SI_3A_G10,
    "SI-3A-H10": SI_3A_H10,
    "SI-3A-A11": SI_3A_A11,
    "SI-3A-B11": SI_3A_B11,
    "SI-3A-C11": SI_3A_C11,
    "SI-3A-D11": SI_3A_D11,
    "SI-3A-E11": SI_3A_E11,
    "SI-3A-F11": SI_3A_F11,
    "SI-3A-G11": SI_3A_G11,
    "SI-3A-H11": SI_3A_H11,
    "SI-3A-A12": SI_3A_A12,
    "SI-3A-B12": SI_3A_B12,
    "SI-3A-C12": SI_3A_C12,
    "SI-3A-D12": SI_3A_D12,
    "SI-3A-E12": SI_3A_E12,
    "SI-3A-F12": SI_3A_F12,
    "SI-3A-G12": SI_3A_G12,
    "SI-3A-H12": SI_3A_H12,
    "SI-GA-A1": SI_GA_A1,
    "SI-GA-B1": SI_GA_B1,
    "SI-GA-C1": SI_GA_C1,
    "SI-GA-D1": SI_GA_D1,
    "SI-GA-E1": SI_GA_E1,
    "SI-GA-F1": SI_GA_F1,
    "SI-GA-G1": SI_GA_G1,
    "SI-GA-H1": SI_GA_H1,
    "SI-GA-A2": SI_GA_A2,
    "SI-GA-B2": SI_GA_B2,
    "SI-GA-C2": SI_GA_C2,
    "SI-GA-D2": SI_GA_D2,
    "SI-GA-E2": SI_GA_E2,
    "SI-GA-F2": SI_GA_F2,
    "SI-GA-G2": SI_GA_G2,
    "SI-GA-H2": SI_GA_H2,
    "SI-GA-A3": SI_GA_A3,
    "SI-GA-B3": SI_GA_B3,
    "SI-GA-C3": SI_GA_C3,
    "SI-GA-D3": SI_GA_D3,
    "SI-GA-E3": SI_GA_E3,
    "SI-GA-F3": SI_GA_F3,
    "SI-GA-G3": SI_GA_G3,
    "SI-GA-H3": SI_GA_H3,
    "SI-GA-A4": SI_GA_A4,
    "SI-GA-B4": SI_GA_B4,
    "SI-GA-C4": SI_GA_C4,
    "SI-GA-D4": SI_GA_D4,
    "SI-GA-E4": SI_GA_E4,
    "SI-GA-F4": SI_GA_F4,
    "SI-GA-G4": SI_GA_G4,
    "SI-GA-H4": SI_GA_H4,
    "SI-GA-A5": SI_GA_A5,
    "SI-GA-B5": SI_GA_B5,
    "SI-GA-C5": SI_GA_C5,
    "SI-GA-D5": SI_GA_D5,
    "SI-GA-E5": SI_GA_E5,
    "SI-GA-F5": SI_GA_F5,
    "SI-GA-G5": SI_GA_G5,
    "SI-GA-H5": SI_GA_H5,
    "SI-GA-A6": SI_GA_A6,
    "SI-GA-B6": SI_GA_B6,
    "SI-GA-C6": SI_GA_C6,
    "SI-GA-D6": SI_GA_D6,
    "SI-GA-E6": SI_GA_E6,
    "SI-GA-F6": SI_GA_F6,
    "SI-GA-G6": SI_GA_G6,
    "SI-GA-H6": SI_GA_H6,
    "SI-GA-A7": SI_GA_A7,
    "SI-GA-B7": SI_GA_B7,
    "SI-GA-C7": SI_GA_C7,
    "SI-GA-D7": SI_GA_D7,
    "SI-GA-E7": SI_GA_E7,
    "SI-GA-F7": SI_GA_F7,
    "SI-GA-G7": SI_GA_G7,
    "SI-GA-H7": SI_GA_H7,
    "SI-GA-A8": SI_GA_A8,
    "SI-GA-B8": SI_GA_B8,
    "SI-GA-C8": SI_GA_C8,
    "SI-GA-D8": SI_GA_D8,
    "SI-GA-E8": SI_GA_E8,
    "SI-GA-F8": SI_GA_F8,
    "SI-GA-G8": SI_GA_G8,
    "SI-GA-H8": SI_GA_H8,
    "SI-GA-A9": SI_GA_A9,
    "SI-GA-B9": SI_GA_B9,
    "SI-GA-C9": SI_GA_C9,
    "SI-GA-D9": SI_GA_D9,
    "SI-GA-E9": SI_GA_E9,
    "SI-GA-F9": SI_GA_F9,
    "SI-GA-G9": SI_GA_G9,
    "SI-GA-H9": SI_GA_H9,
    "SI-GA-A10": SI_GA_A10,
    "SI-GA-B10": SI_GA_B10,
    "SI-GA-C10": SI_GA_C10,
    "SI-GA-D10": SI_GA_D10,
    "SI-GA-E10": SI_GA_E10,
    "SI-GA-F10": SI_GA_F10,
    "SI-GA-G10": SI_GA_G10,
    "SI-GA-H10": SI_GA_H10,
    "SI-GA-A11": SI_GA_A11,
    "SI-GA-B11": SI_GA_B11,
    "SI-GA-C11": SI_GA_C11,
    "SI-GA-D11": SI_GA_D11,
    "SI-GA-E11": SI_GA_E11,
    "SI-GA-F11": SI_GA_F11,
    "SI-GA-G11": SI_GA_G11,
    "SI-GA-H11": SI_GA_H11,
    "SI-GA-A12": SI_GA_A12,
    "SI-GA-B12": SI_GA_B12,
    "SI-GA-C12": SI_GA_C12,
    "SI-GA-D12": SI_GA_D12,
    "SI-GA-E12": SI_GA_E12,
    "SI-GA-F12": SI_GA_F12,
    "SI-GA-G12": SI_GA_G12,
    "SI-GA-H12": SI_GA_H12,
    # Agora/VDJ-FB
    "SI-NA-A1": SI_NA_A1,
    "SI-NA-B1": SI_NA_B1,
    "SI-NA-C1": SI_NA_C1,
    "SI-NA-D1": SI_NA_D1,
    "SI-NA-E1": SI_NA_E1,
    "SI-NA-F1": SI_NA_F1,
    "SI-NA-G1": SI_NA_G1,
    "SI-NA-H1": SI_NA_H1,
    "SI-NA-A2": SI_NA_A2,
    "SI-NA-B2": SI_NA_B2,
    "SI-NA-C2": SI_NA_C2,
    "SI-NA-D2": SI_NA_D2,
    "SI-NA-E2": SI_NA_E2,
    "SI-NA-F2": SI_NA_F2,
    "SI-NA-G2": SI_NA_G2,
    "SI-NA-H2": SI_NA_H2,
    "SI-NA-A3": SI_NA_A3,
    "SI-NA-B3": SI_NA_B3,
    "SI-NA-C3": SI_NA_C3,
    "SI-NA-D3": SI_NA_D3,
    "SI-NA-E3": SI_NA_E3,
    "SI-NA-F3": SI_NA_F3,
    "SI-NA-G3": SI_NA_G3,
    "SI-NA-H3": SI_NA_H3,
    "SI-NA-A4": SI_NA_A4,
    "SI-NA-B4": SI_NA_B4,
    "SI-NA-C4": SI_NA_C4,
    "SI-NA-D4": SI_NA_D4,
    "SI-NA-E4": SI_NA_E4,
    "SI-NA-F4": SI_NA_F4,
    "SI-NA-G4": SI_NA_G4,
    "SI-NA-H4": SI_NA_H4,
    "SI-NA-A5": SI_NA_A5,
    "SI-NA-B5": SI_NA_B5,
    "SI-NA-C5": SI_NA_C5,
    "SI-NA-D5": SI_NA_D5,
    "SI-NA-E5": SI_NA_E5,
    "SI-NA-F5": SI_NA_F5,
    "SI-NA-G5": SI_NA_G5,
    "SI-NA-H5": SI_NA_H5,
    "SI-NA-A6": SI_NA_A6,
    "SI-NA-B6": SI_NA_B6,
    "SI-NA-C6": SI_NA_C6,
    "SI-NA-D6": SI_NA_D6,
    "SI-NA-E6": SI_NA_E6,
    "SI-NA-F6": SI_NA_F6,
    "SI-NA-G6": SI_NA_G6,
    "SI-NA-H6": SI_NA_H6,
    "SI-NA-A7": SI_NA_A7,
    "SI-NA-B7": SI_NA_B7,
    "SI-NA-C7": SI_NA_C7,
    "SI-NA-D7": SI_NA_D7,
    "SI-NA-E7": SI_NA_E7,
    "SI-NA-F7": SI_NA_F7,
    "SI-NA-G7": SI_NA_G7,
    "SI-NA-H7": SI_NA_H7,
    "SI-NA-A8": SI_NA_A8,
    "SI-NA-B8": SI_NA_B8,
    "SI-NA-C8": SI_NA_C8,
    "SI-NA-D8": SI_NA_D8,
    "SI-NA-E8": SI_NA_E8,
    "SI-NA-F8": SI_NA_F8,
    "SI-NA-G8": SI_NA_G8,
    "SI-NA-H8": SI_NA_H8,
    "SI-NA-A9": SI_NA_A9,
    "SI-NA-B9": SI_NA_B9,
    "SI-NA-C9": SI_NA_C9,
    "SI-NA-D9": SI_NA_D9,
    "SI-NA-E9": SI_NA_E9,
    "SI-NA-F9": SI_NA_F9,
    "SI-NA-G9": SI_NA_G9,
    "SI-NA-H9": SI_NA_H9,
    "SI-NA-A10": SI_NA_A10,
    "SI-NA-B10": SI_NA_B10,
    "SI-NA-C10": SI_NA_C10,
    "SI-NA-D10": SI_NA_D10,
    "SI-NA-E10": SI_NA_E10,
    "SI-NA-F10": SI_NA_F10,
    "SI-NA-G10": SI_NA_G10,
    "SI-NA-H10": SI_NA_H10,
    "SI-NA-A11": SI_NA_A11,
    "SI-NA-B11": SI_NA_B11,
    "SI-NA-C11": SI_NA_C11,
    "SI-NA-D11": SI_NA_D11,
    "SI-NA-E11": SI_NA_E11,
    "SI-NA-F11": SI_NA_F11,
    "SI-NA-G11": SI_NA_G11,
    "SI-NA-H11": SI_NA_H11,
    "SI-NA-A12": SI_NA_A12,
    "SI-NA-B12": SI_NA_B12,
    "SI-NA-C12": SI_NA_C12,
    "SI-NA-D12": SI_NA_D12,
    "SI-NA-E12": SI_NA_E12,
    "SI-NA-F12": SI_NA_F12,
    "SI-NA-G12": SI_NA_G12,
    "SI-NA-H12": SI_NA_H12,
    # GemCode Part numbers
    "220027": SI_001,
    "220028": SI_002,
    "220029": SI_003,
    "220030": SI_004,
    "220031": SI_005,
    "220032": SI_006,
    "220033": SI_007,
    "220034": SI_008,
    # WGS+ Tube labels
    "SI-T2-1": SI_T2_1,
    "SI-T2-2": SI_T2_2,
    "SI-T2-3": SI_T2_3,
    "SI-T2-4": SI_T2_4,
    "SI-T2-5": SI_T2_5,
    "SI-T2-6": SI_T2_6,
    "SI-T2-7": SI_T2_7,
    "SI-T2-8": SI_T2_8,
    # WGS+ Plate labels
    "SI-P2-A1": SI_P2_A1,
    "SI-P2-A2": SI_P2_A2,
    "SI-P2-A3": SI_P2_A3,
    "SI-P2-A4": SI_P2_A4,
    "SI-P2-A5": SI_P2_A5,
    "SI-P2-A6": SI_P2_A6,
    "SI-P2-A7": SI_P2_A7,
    "SI-P2-A8": SI_P2_A8,
    "SI-P2-A9": SI_P2_A9,
    "SI-P2-A10": SI_P2_A10,
    "SI-P2-A11": SI_P2_A11,
    "SI-P2-A12": SI_P2_A12,
    "SI-P2-B1": SI_P2_B1,
    "SI-P2-B2": SI_P2_B2,
    "SI-P2-B3": SI_P2_B3,
    "SI-P2-B4": SI_P2_B4,
    "SI-P2-B5": SI_P2_B5,
    "SI-P2-B6": SI_P2_B6,
    "SI-P2-B7": SI_P2_B7,
    "SI-P2-B8": SI_P2_B8,
    "SI-P2-B9": SI_P2_B9,
    "SI-P2-B10": SI_P2_B10,
    "SI-P2-B11": SI_P2_B11,
    "SI-P2-B12": SI_P2_B12,
    "SI-P2-C1": SI_P2_C1,
    "SI-P2-C2": SI_P2_C2,
    "SI-P2-C3": SI_P2_C3,
    "SI-P2-C4": SI_P2_C4,
    "SI-P2-C5": SI_P2_C5,
    "SI-P2-C6": SI_P2_C6,
    "SI-P2-C7": SI_P2_C7,
    "SI-P2-C8": SI_P2_C8,
    "SI-P2-C9": SI_P2_C9,
    "SI-P2-C10": SI_P2_C10,
    "SI-P2-C11": SI_P2_C11,
    "SI-P2-C12": SI_P2_C12,
    "SI-P2-D1": SI_P2_D1,
    "SI-P2-D2": SI_P2_D2,
    "SI-P2-D3": SI_P2_D3,
    "SI-P2-D4": SI_P2_D4,
    "SI-P2-D5": SI_P2_D5,
    "SI-P2-D6": SI_P2_D6,
    "SI-P2-D7": SI_P2_D7,
    "SI-P2-D8": SI_P2_D8,
    "SI-P2-D9": SI_P2_D9,
    "SI-P2-D10": SI_P2_D10,
    "SI-P2-D11": SI_P2_D11,
    "SI-P2-D12": SI_P2_D12,
    "SI-P2-E1": SI_P2_E1,
    "SI-P2-E2": SI_P2_E2,
    "SI-P2-E3": SI_P2_E3,
    "SI-P2-E4": SI_P2_E4,
    "SI-P2-E5": SI_P2_E5,
    "SI-P2-E6": SI_P2_E6,
    "SI-P2-E7": SI_P2_E7,
    "SI-P2-E8": SI_P2_E8,
    "SI-P2-E9": SI_P2_E9,
    "SI-P2-E10": SI_P2_E10,
    "SI-P2-E11": SI_P2_E11,
    "SI-P2-E12": SI_P2_E12,
    "SI-P2-F1": SI_P2_F1,
    "SI-P2-F2": SI_P2_F2,
    "SI-P2-F3": SI_P2_F3,
    "SI-P2-F4": SI_P2_F4,
    "SI-P2-F5": SI_P2_F5,
    "SI-P2-F6": SI_P2_F6,
    "SI-P2-F7": SI_P2_F7,
    "SI-P2-F8": SI_P2_F8,
    "SI-P2-F9": SI_P2_F9,
    "SI-P2-F10": SI_P2_F10,
    "SI-P2-F11": SI_P2_F11,
    "SI-P2-F12": SI_P2_F12,
    "SI-P2-G1": SI_P2_G1,
    "SI-P2-G2": SI_P2_G2,
    "SI-P2-G3": SI_P2_G3,
    "SI-P2-G4": SI_P2_G4,
    "SI-P2-G5": SI_P2_G5,
    "SI-P2-G6": SI_P2_G6,
    "SI-P2-G7": SI_P2_G7,
    "SI-P2-G8": SI_P2_G8,
    "SI-P2-G9": SI_P2_G9,
    "SI-P2-G10": SI_P2_G10,
    "SI-P2-G11": SI_P2_G11,
    "SI-P2-G12": SI_P2_G12,
    "SI-P2-H1": SI_P2_H1,
    "SI-P2-H2": SI_P2_H2,
    "SI-P2-H3": SI_P2_H3,
    "SI-P2-H4": SI_P2_H4,
    "SI-P2-H5": SI_P2_H5,
    "SI-P2-H6": SI_P2_H6,
    "SI-P2-H7": SI_P2_H7,
    "SI-P2-H8": SI_P2_H8,
    "SI-P2-H9": SI_P2_H9,
    "SI-P2-H10": SI_P2_H10,
    "SI-P2-H11": SI_P2_H11,
    "SI-P2-H12": SI_P2_H12,
    # WGS+ Plate Label alternate
    "SI-P02-A1": SI_P2_A1,
    "SI-P02-A2": SI_P2_A2,
    "SI-P02-A3": SI_P2_A3,
    "SI-P02-A4": SI_P2_A4,
    "SI-P02-A5": SI_P2_A5,
    "SI-P02-A6": SI_P2_A6,
    "SI-P02-A7": SI_P2_A7,
    "SI-P02-A8": SI_P2_A8,
    "SI-P02-A9": SI_P2_A9,
    "SI-P02-A10": SI_P2_A10,
    "SI-P02-A11": SI_P2_A11,
    "SI-P02-A12": SI_P2_A12,
    "SI-P02-B1": SI_P2_B1,
    "SI-P02-B2": SI_P2_B2,
    "SI-P02-B3": SI_P2_B3,
    "SI-P02-B4": SI_P2_B4,
    "SI-P02-B5": SI_P2_B5,
    "SI-P02-B6": SI_P2_B6,
    "SI-P02-B7": SI_P2_B7,
    "SI-P02-B8": SI_P2_B8,
    "SI-P02-B9": SI_P2_B9,
    "SI-P02-B10": SI_P2_B10,
    "SI-P02-B11": SI_P2_B11,
    "SI-P02-B12": SI_P2_B12,
    "SI-P02-C1": SI_P2_C1,
    "SI-P02-C2": SI_P2_C2,
    "SI-P02-C3": SI_P2_C3,
    "SI-P02-C4": SI_P2_C4,
    "SI-P02-C5": SI_P2_C5,
    "SI-P02-C6": SI_P2_C6,
    "SI-P02-C7": SI_P2_C7,
    "SI-P02-C8": SI_P2_C8,
    "SI-P02-C9": SI_P2_C9,
    "SI-P02-C10": SI_P2_C10,
    "SI-P02-C11": SI_P2_C11,
    "SI-P02-C12": SI_P2_C12,
    "SI-P02-D1": SI_P2_D1,
    "SI-P02-D2": SI_P2_D2,
    "SI-P02-D3": SI_P2_D3,
    "SI-P02-D4": SI_P2_D4,
    "SI-P02-D5": SI_P2_D5,
    "SI-P02-D6": SI_P2_D6,
    "SI-P02-D7": SI_P2_D7,
    "SI-P02-D8": SI_P2_D8,
    "SI-P02-D9": SI_P2_D9,
    "SI-P02-D10": SI_P2_D10,
    "SI-P02-D11": SI_P2_D11,
    "SI-P02-D12": SI_P2_D12,
    "SI-P02-E1": SI_P2_E1,
    "SI-P02-E2": SI_P2_E2,
    "SI-P02-E3": SI_P2_E3,
    "SI-P02-E4": SI_P2_E4,
    "SI-P02-E5": SI_P2_E5,
    "SI-P02-E6": SI_P2_E6,
    "SI-P02-E7": SI_P2_E7,
    "SI-P02-E8": SI_P2_E8,
    "SI-P02-E9": SI_P2_E9,
    "SI-P02-E10": SI_P2_E10,
    "SI-P02-E11": SI_P2_E11,
    "SI-P02-E12": SI_P2_E12,
    "SI-P02-F1": SI_P2_F1,
    "SI-P02-F2": SI_P2_F2,
    "SI-P02-F3": SI_P2_F3,
    "SI-P02-F4": SI_P2_F4,
    "SI-P02-F5": SI_P2_F5,
    "SI-P02-F6": SI_P2_F6,
    "SI-P02-F7": SI_P2_F7,
    "SI-P02-F8": SI_P2_F8,
    "SI-P02-F9": SI_P2_F9,
    "SI-P02-F10": SI_P2_F10,
    "SI-P02-F11": SI_P2_F11,
    "SI-P02-F12": SI_P2_F12,
    "SI-P02-G1": SI_P2_G1,
    "SI-P02-G2": SI_P2_G2,
    "SI-P02-G3": SI_P2_G3,
    "SI-P02-G4": SI_P2_G4,
    "SI-P02-G5": SI_P2_G5,
    "SI-P02-G6": SI_P2_G6,
    "SI-P02-G7": SI_P2_G7,
    "SI-P02-G8": SI_P2_G8,
    "SI-P02-G9": SI_P2_G9,
    "SI-P02-G10": SI_P2_G10,
    "SI-P02-G11": SI_P2_G11,
    "SI-P02-G12": SI_P2_G12,
    "SI-P02-H1": SI_P2_H1,
    "SI-P02-H2": SI_P2_H2,
    "SI-P02-H3": SI_P2_H3,
    "SI-P02-H4": SI_P2_H4,
    "SI-P02-H5": SI_P2_H5,
    "SI-P02-H6": SI_P2_H6,
    "SI-P02-H7": SI_P2_H7,
    "SI-P02-H8": SI_P2_H8,
    "SI-P02-H9": SI_P2_H9,
    "SI-P02-H10": SI_P2_H10,
    "SI-P02-H11": SI_P2_H11,
    "SI-P02-H12": SI_P2_H12,
}

# Chromium lot-specific oligos
CHROMIUM_LOT1_PART_A = ["AGAGCGA", "CGATTGA", "TAGACCA", "AAATGCC", "CTTTGCG", "TCAGCAA", "CTCCTAG", "ATTATCC"]
CHROMIUM_LOT2_PART_A = ["GACACTA", "CCCTCTC", "ATCGCGG", "CTGGCAG", "CCAGCTT", "CATAGCA", "CGTGTTC", "GCACCAG"]
CHROMIUM_LOT3_PART_A = ["ATGTGAC", "GACGTCG", "ACTGGCG", "TGGCAAT", "GAGGGTA", "GTTTCGC", "CAAGTGT", "TTGAAGC"]
CHROMIUM_LOT4_PART_A = ["CGATCCT", "TGTTGCC", "ACCTATT", "ACAACTG", "CTGTGTC", "CTGGAAT", "CAGAGTT", "GGGCTGT"]
CHROMIUM_LOT5_PART_A = ["TAGCTCC", "CAATTTC", "GCTCGAG", "GAAGGCA", "CGGCATG", "TATTCCA", "TCTCTGG", "AGGTACT"]
CHROMIUM_LOT6_PART_A = ["ACTTGCC", "GTGAGTT", "GTTGTCC", "CATAACG", "TCGTAAG", "TTATCCA", "GTGGAGA", "TCCTGCA"]
CHROMIUM_LOT7_PART_A = ["TAAGCCA", "TCGGTGG", "AAGGTAA", "GGAACAG", "GTGGAAG", "TTAGACG", "ATCCTAT", "TTCCGTG"]
CHROMIUM_LOT8_PART_A = ["GGTTTAG", "CGTATAG", "ATAGGCT", "CTCTCGA", "GTCTTAT", "GATTGCA", "TGAGCTA", "ACGCGTG"]
CHROMIUM_LOT9_PART_A = ["CGACACG", "TCTCGTG", "TGATGAC", "TGCGTAA", "TACCCTG", "AGGTGCC", "CTTGTGC", "GCATGGC"]
CHROMIUM_LOT10_PART_A = ["CAGCACG", "CATGATG", "ATCAACG", "GATAAGA", "CTGGTTC", "CGATTCC", "AGGTGAG", "GGCCTGA"]
CHROMIUM_LOT11_PART_A = ["ACAGTTG", "TAAGCAC", "ATCTTTG", "TCTTGCG", "TACATGG", "CAAGGTT", "AGGCTGC", "GGTCGTG"]
CHROMIUM_LOT12_PART_A = ["CCATTAT", "GTTGCGG", "AGGGTAG", "GCCCAAG", "TGTGCCT", "ATTCTTG", "GGTGCCA", "GTATAGC"]
CHROMIUM_LOT13_PART_A = ["GGCATCG", "GACTGAT", "TGGTGTA", "TCCGTTG", "CCTTCAG", "CAGGCCA", "GCACCGA", "AGATCCA"]

CHROMIUM_LOT_MAP = {
    "Chromium Lot 1": CHROMIUM_LOT1_PART_A,
    "Chromium Lot 2": CHROMIUM_LOT2_PART_A,
    "Chromium Lot 3": CHROMIUM_LOT3_PART_A,
    "Chromium Lot 4": CHROMIUM_LOT4_PART_A,
    "Chromium Lot 5": CHROMIUM_LOT5_PART_A,
    "Chromium Lot 6": CHROMIUM_LOT6_PART_A,
    "Chromium Lot 7": CHROMIUM_LOT7_PART_A,
    "Chromium Lot 8": CHROMIUM_LOT8_PART_A,
    "Chromium Lot 9": CHROMIUM_LOT9_PART_A,
    "Chromium Lot 10": CHROMIUM_LOT10_PART_A,
    "Chromium Lot 11": CHROMIUM_LOT11_PART_A,
    "Chromium Lot 12": CHROMIUM_LOT12_PART_A,
    "Chromium Lot 13": CHROMIUM_LOT13_PART_A,
}

# GemCode lot-specific oligos
# note: lots 1-15 all use the same part As
GEMCODE_LOT1_PART_A = ["GGGTGA", "TTCATC", "CACAAC", "GAAGAT", "CAGCAT", "CGTCAA", "GAAACA", "TGTTTC"]
GEMCODE_LOT16_PART_A = ["CAAGTC", "ACAAAG", "CTGGAT", "TTGTCT", "AGCCTA", "GGGAAC", "TTCCTA", "CCGTAA"]
GEMCODE_LOT17_PART_A = ["AGTCCA", "CAGGAG", "CAATGC", "CAATCG", "AACAGA", "TTACTC", "ACTGAC", "TAAGCC"]
GEMCODE_LOT18_PART_A = ["GCATGT", "CCAACA", "TCGGTA", "ATCGTG", "ATTCTC", "CGTTAG", "TTCACT", "GGTTTG"]
GEMCODE_LOT19_PART_A = ["CTTTCA", "TTGTTC", "TAGCCA", "GCGTAT", "CGTACA", "CCTTCG", "CACACA", "TACTTC"]
GEMCODE_LOT20_PART_A = ["CTTCAT", "ATTCCT", "GTCTCC", "CAGGGA", "ATCCGA", "CGAATC", "AAACCC", "CGCTAA"]
GEMCODE_LOT21_PART_A = ["CAGATC", "AATCCG", "TACGTG", "GAACAA", "AGAGCG", "CCAGAT", "CGCTTC", "TTATCC"]

GEMCODE_LOT_MAP = {
    "GemCode Lots 1-15": GEMCODE_LOT1_PART_A,
    "GemCode Lot 16": GEMCODE_LOT16_PART_A,
    "GemCode Lot 17": GEMCODE_LOT17_PART_A,
    "GemCode Lot 18": GEMCODE_LOT18_PART_A,
    "GemCode Lot 19": GEMCODE_LOT19_PART_A,
    "GemCode Lot 20": GEMCODE_LOT20_PART_A,
    "GemCode Lot 21": GEMCODE_LOT21_PART_A,
}

# if a whitelist isn't in this map, then assume it doesn't contain the alts and don't try to do lot detection
WHITELIST_TO_LOT_MAP = {
    "884K-november-2015": GEMCODE_LOT_MAP,
    "4M-with-alts-february-2016": CHROMIUM_LOT_MAP,
}
