import numpy as np
import pandas as pd
import vcfpy
from figeno.utils import KnownException


def read_SNPs_RNA(ase_file,gene=None,chr=None,start=None,end=None,exons=None,min_depth=0):
    """
    Reads a tsv file generated by fast_ase or ASEReadCounter.
    Return a dataframe whose rows are SNPs, indicating the read counts of each allele in the RNAseq data.
    """
    if gene is None and (chr is None or start is None or end is None):
        raise Exception("Must provide either a gene or coordinates (chr start end)")
    if gene is not None:
        chr = gene.chr
        start = gene.start
        end = gene.end
    try:
        df = pd.read_csv(ase_file,sep="\t",dtype={"contig":str})
    except:
        raise KnownException("Failed to open the ase file: "+str(ase_file))
    if not "contig" in df.columns: raise KnownException("The input file for the ase track needs a contig column.")
    if not "position" in df.columns: raise KnownException("The input file for the ase track needs a position column.")
    if not "refCount" in df.columns: raise KnownException("The input file for the ase track needs a refCount column.")
    if not "altCount" in df.columns: raise KnownException("The input file for the ase track needs a altCount column.")
    if not "variantID" in df.columns: raise KnownException("The input file for the ase track needs a variantID column.")
    df["contig"] = [x.lstrip("chr") for x in df["contig"]]
    df = df.loc[df["contig"]==chr]
    df = df.loc[(df["position"]>=start) & (df["position"]<=end)]
    df = df.loc[(df["refCount"]+df["altCount"]>=min_depth)]
    variables_selected=["contig","position","variantID","refAllele","altAllele","refCount","altCount"]
    if "refCount_DNA" in df.columns: variables_selected+=["refCount_DNA","altCount_DNA"]
    df = df[variables_selected]
    df["VAF"] = [min(df.loc[i,"refCount"],df.loc[i,"altCount"]) / (df.loc[i,"refCount"]+df.loc[i,"altCount"]) for i in df.index]
    df["VAF_DNA"] = [min(df.loc[i,"refCount_DNA"],df.loc[i,"altCount_DNA"]) / (df.loc[i,"refCount_DNA"]+df.loc[i,"altCount_DNA"]) for i in df.index]
    in_exons=[]
    if exons is not None:
        for i in df.index:
            exonic=False
            for exon in exons:
                if exon[0]<=df.loc[i,"position"] and df.loc[i,"position"] <=exon[1]: 
                    exonic=True
            in_exons.append(exonic)
        df["exonic"] = in_exons
    return df

def read_SNPs_DNA(vcf_file,df_ase):
    variants_RNA=[]
    for i in df_ase.index:
        variants_RNA.append(df_ase.loc[i,"contig"]+":"+str(df_ase.loc[i,"position"]))
    positions=[]
    refCount_DNA=[]
    altCount_DNA=[]
    VAF_DNA=[]
    depth_DNA=[]
    reader = vcfpy.Reader.from_path(vcf_file)
    prefix=""
    for record in reader:
        if record.CHROM.startswith("chr"): prefix = "chr"
        break
    reader = vcfpy.Reader.from_path(vcf_file)
    for record in reader.fetch(prefix+df_ase.loc[df_ase.index[0],"contig"], np.min(df_ase["position"])-5, np.max(df_ase["position"])+5):
        if record.CHROM+":"+str(record.POS) in variants_RNA:
            DP = record.calls[0].data["DP"]
            AF = min(record.calls[0].data["AD"][0],record.calls[0].data["AD"][1]) / DP
            refCount_DNA.append(record.calls[0].data["AD"][0])
            altCount_DNA.append(record.calls[0].data["AD"][1])
            VAF_DNA.append(AF)
            depth_DNA.append(DP)
            positions.append(record.POS)
    df_ase = df_ase.copy(deep=True)
    if len(VAF_DNA)==df_ase.shape[0]:
        df_ase["VAF_DNA"] = VAF_DNA
        df_ase["depth_DNA"] = depth_DNA
        df_ase["refCount_DNA"] = refCount_DNA
        df_ase["altCount_DNA"] = altCount_DNA
    else:
        print("Some SNPs in the RNAseq were not found in the DNAseq")
    return df_ase