import numpy as np

from melody_features.import_mid import import_midi
from melody_features.representations import Melody
from melody_features.pitch_spelling import estimate_spelling_from_melody
from melody_features.tonal_tension import estimate_tonaltension

tonal_tension_vals = [( 0.  , 0.15509486, 0.00000000e+00, 0.07465585),
 ( 0.25, 0.15509486, 5.16982861e-02, 0.05403359),
 ( 0.5 , 0.15509486, 2.58491430e-02, 0.05997691),
 ( 1.  , 0.33333334, 7.75474310e-02, 0.11682538),
 ( 1.5 , 0.33333334, 1.66666672e-01, 0.09947067),
 ( 2.  , 0.28215238, 1.66666672e-01, 0.24838723),
 ( 2.25, 0.28215238, 9.40507948e-02, 0.159052  ),
 ( 2.5 , 0.28215238, 4.70253974e-02, 0.11718517),
 ( 3.  , 0.28215238, 9.53057781e-02, 0.04701226),
 ( 3.25, 0.28215238, 1.11988664e-01, 0.12995823),
 ( 3.5 , 0.28215238, 3.17685902e-02, 0.15831354),
 ( 4.  , 0.28215238, 9.53057781e-02, 0.24838723),
 ( 4.25, 0.28215238, 9.40507948e-02, 0.159052  ),
 ( 4.5 , 0.43671626, 4.70253974e-02, 0.11718517),
 ( 5.  , 0.28215238, 2.18358129e-01, 0.10173631),
 ( 5.5 , 0.28215238, 7.75474310e-02, 0.11560608),
 ( 6.  , 0.24899341, 1.41076192e-01, 0.14285868),
 ( 6.25, 0.24899341, 8.29978064e-02, 0.14280711),
 ( 6.5 , 0.33333334, 4.14989032e-02, 0.159853  ),
 ( 7.  , 0.33333334, 1.24496706e-01, 0.09947067),
 ( 7.5 , 0.33333334, 0.00000000e+00, 0.09947067),
 ( 8.  , 0.33333334, 7.75474310e-02, 0.11718517),
 ( 8.25, 0.15509486, 1.54325396e-01, 0.05403359),
 ( 8.5 , 0.15509486, 2.58491430e-02, 0.05997691),
 ( 9.  , 0.33333334, 4.62465808e-02, 0.06094018),
 ( 9.25, 0.33333334, 1.25946969e-01, 0.07612301),
 ( 9.5 , 0.33333334, 4.44269031e-02, 0.11768109),
 (10.  , 0.43671626, 3.87737155e-02, 0.14847033),
 (10.25, 0.43671626, 1.26321182e-01, 0.02954868),
 (10.5 , 0.43671626, 2.53440626e-02, 0.01101149),
 (11.  , 0.28215238, 3.87737155e-02, 0.04701226),
 (11.25, 0.28215238, 1.11988664e-01, 0.12995823),
 (11.5 , 0.28215238, 3.17685902e-02, 0.15831354),
 (12.  , 0.28215238, 9.53057781e-02, 0.24838723),
 (12.25, 0.28215238, 9.40507948e-02, 0.159052  ),
 (12.5 , 0.43671626, 4.70253974e-02, 0.11718517),
 (13.  , 0.28215238, 2.18358129e-01, 0.10173631),
 (13.5 , 0.28215238, 7.75474310e-02, 0.11560608),
 (14.  , 0.24899341, 1.41076192e-01, 0.14285868),
 (14.25, 0.24899341, 8.29978064e-02, 0.14280711),
 (14.5 , 0.24899341, 4.14989032e-02, 0.159853  ),
 (16.  , 0.24899341, 2.66561419e-01, 0.11682538),
 (16.5 , 0.24899341, 1.24496706e-01, 0.03910186),
 (17.  , 0.24899341, 1.24496706e-01, 0.14285868),
 (17.25, 0.43671626, 8.29978064e-02, 0.06616535),
 (17.5 , 0.43671626, 8.01832825e-02, 0.10625248),
 (17.75, 0.43671626, 3.87737155e-02, 0.11974406),
 (18.  , 0.43671626, 3.87737155e-02, 0.09306473),
 (18.25, 0.33333334, 6.22483529e-02, 0.14411044),
 (18.5 , 0.24899341, 7.05380961e-02, 0.1066012 ),
 (19.  , 0.24899341, 4.62465808e-02, 0.14285868),
 (19.5 , 0.43671626, 7.60321841e-02, 0.10625248),
 (19.75, 0.43671626, 6.82011023e-02, 0.12995823),
 (20.  , 0.43671626, 4.24232110e-02, 0.09306473),
 (20.25, 0.33333334, 6.22483529e-02, 0.14411044),
 (20.5 , 0.24899341, 7.05380961e-02, 0.1066012 ),
 (21.  , 0.15509486, 4.62465808e-02, 0.14285868),
 (21.5 , 0.15509486, 7.75474310e-02, 0.0835311 ),
 (23.  , 0.28215238, 1.66666672e-01, 0.21187344),
 (23.25, 0.43671626, 4.70253974e-02, 0.17974147),
 (23.5 , 0.43671626, 7.38302097e-02, 0.10625248),
 (23.75, 0.43671626, 6.22483529e-02, 0.08682629),
 (24.  , 0.43671626, 7.05380961e-02, 0.11974406),
 (24.25, 0.43671626, 3.87737155e-02, 0.09306473),
 (24.5 , 0.33333334, 3.87737155e-02, 0.11925672),
 (25.  , 0.        , 1.33280709e-01, 0.07465585),
 (25.5 , 0.        , 0.00000000e+00, 0.07465585),
 (26.5 , 0.        , 1.55094862e-01, 0.11682538),
 (27.  , 0.33333334, 0.00000000e+00, 0.11682538),
 (27.5 , 0.33333334, 1.33280709e-01, 0.04285823),
 (27.75, 0.43671626, 1.53553531e-01, 0.18481453),
 (28.  , 0.43671626, 9.35102850e-02, 0.10448093),
 (28.25, 0.43671626, 7.05380961e-02, 0.04823514),
 (28.5 , 0.43671626, 3.87737155e-02, 0.01101149),
 (29.  , 0.28215238, 3.87737155e-02, 0.04701226),
 (29.25, 0.28215238, 1.11988664e-01, 0.12995823),
 (29.5 , 0.43671626, 3.17685902e-02, 0.15831354),
 (30.  , 0.43671626, 3.87737155e-02, 0.14847033),
 (30.25, 0.43671626, 1.26321182e-01, 0.02954868),
 (30.5 , 0.43671626, 2.53440626e-02, 0.01101149),
 (31.  , 0.28215238, 1.09179065e-01, 0.10173631),
 (31.5 , 0.28215238, 7.75474310e-02, 0.11560608),
 (32.  , 0.24899341, 1.41076192e-01, 0.14285868),
 (32.25, 0.24899341, 8.29978064e-02, 0.14280711),
 (32.5 , 0.33333334, 4.14989032e-02, 0.159853  ),
 (33.  , 0.33333334, 1.24496706e-01, 0.09947067),
 (33.5 , 0.33333334, 0.00000000e+00, 0.09947067),
 (34.  , 0.28215238, 1.66666672e-01, 0.24838723),
 (34.25, 0.28215238, 9.40507948e-02, 0.159052  ),
 (34.5 , 0.43671626, 4.70253974e-02, 0.11718517),
 (35.  , 0.28215238, 2.18358129e-01, 0.10173631),
 (35.5 , 0.28215238, 7.75474310e-02, 0.11560608),
 (36.  , 0.24899341, 1.41076192e-01, 0.14285868),
 (36.25, 0.24899341, 8.29978064e-02, 0.14280711),
 (36.5 , 0.24899341, 4.14989032e-02, 0.159853  ),
 (38.  , 0.43671626, 7.75474310e-02, 0.11718517),
 (38.25, 0.43671626, 1.08759113e-01, 0.02954868),
 (38.5 , 0.43671626, 5.43795563e-02, 0.05757529),
 (39.  , 0.28215238, 1.05933644e-01, 0.15738398),
 (39.25, 0.28215238, 1.17563494e-01, 0.10625411),
 (39.5 , 0.28215238, 2.35126987e-02, 0.1091378 ),
 (39.75, 0.33333334, 7.05380961e-02, 0.11560608),
 (40.  , 0.33333334, 8.33333358e-02, 0.10718157),
 (40.25, 0.33333334, 3.87737155e-02, 0.07074618),
 (40.5 , 0.43671626, 3.87737155e-02, 0.03910186),
 (41.  , 0.33333334, 1.41076192e-01, 0.11718517),
 (41.25, 0.33333334, 6.68610036e-02, 0.14487527),
 (41.5 , 0.33333334, 5.55555560e-02, 0.09947067),
 (42.  , 0.43671626, 7.05380961e-02, 0.04207373),
 (42.25, 0.43671626, 4.54769433e-02, 0.02954868),
 (42.5 , 0.43671626, 5.43795563e-02, 0.05757529),
 (43.  , 0.28215238, 1.05933644e-01, 0.15738398),
 (43.25, 0.28215238, 1.17563494e-01, 0.10625411),
 (43.5 , 0.28215238, 2.35126987e-02, 0.1091378 ),
 (43.75, 0.33333334, 7.05380961e-02, 0.11560608),
 (44.  , 0.33333334, 8.33333358e-02, 0.10718157),
 (44.25, 0.33333334, 3.87737155e-02, 0.07074618),
 (44.5 , 0.43671626, 3.87737155e-02, 0.03910186),
 (45.  , 0.33333334, 1.41076192e-01, 0.11718517),
 (45.25, 0.33333334, 6.68610036e-02, 0.14487527),
 (45.5 , 0.33333334, 5.55555560e-02, 0.09947067),
 (46.  , 0.33333334, 7.05380961e-02, 0.04207373),
 (46.25, 0.33333334, 3.63930203e-02, 0.05575557),
 (46.5 , 0.33333334, 3.63930203e-02, 0.04207373),
 (47.75, 0.28215238, 1.09179065e-01, 0.11718517),
 (48.  , 0.33333334, 4.70253974e-02, 0.08117861),
 (48.25, 0.33333334, 4.70253974e-02, 0.04147713),
 (48.5 , 0.33333334, 3.87737155e-02, 0.04207373),
 (49.75, 0.28215238, 1.09179065e-01, 0.11718517),
 (50.  , 0.33333334, 4.70253974e-02, 0.08117861),
 (50.25, 0.33333334, 4.70253974e-02, 0.04147713),
 (50.5 , 0.33333334, 1.02688038e-17, 0.04147713),
 (50.75, 0.33333334, 1.14808715e-17, 0.04147713),
 (51.  , 0.33333334, 1.14808715e-17, 0.04147713),
 (51.25, 0.33333334, 1.02688038e-17, 0.04147713),
 (51.5 , 0.33333334, 1.02688038e-17, 0.04147713),
 (51.75, 0.33333334, 7.05380961e-02, 0.05551542),
 (52.  , 0.33333334, 7.05380961e-02, 0.04147713),
 (52.25, 0.33333334, 1.02688038e-17, 0.04147713),
 (52.5 , 0.33333334, 3.87737155e-02, 0.04207373)]

pitch_spelling_vals = [('D', 0, 4), ('D', 0, 4), ('G', 0, 4), ('G', 0, 4), ('B', 0, 4), ('B', 0, 4),
 ('B', 0, 4), ('D', 0, 5), ('E', 0, 5), ('D', 0, 5), ('B', 0, 4), ('B', 0, 4),
 ('B', 0, 4), ('D', 0, 5), ('C', 0, 5), ('A', 0, 4), ('A', 0, 4), ('A', 0, 4),
 ('B', 0, 4), ('G', 0, 5), ('B', 0, 5), ('D', 0, 4), ('D', 0, 4), ('G', 0, 4),
 ('G', 0, 4), ('A', 0, 4), ('B', 0, 4), ('B', 0, 4), ('C', 0, 5), ('D', 0, 5),
 ('E', 0, 5), ('D', 0, 5), ('B', 0, 4), ('B', 0, 4), ('B', 0, 4), ('D', 0, 5),
 ('C', 0, 5), ('A', 0, 4), ('A', 0, 4), ('A', 0, 4), ('B', 0, 4), ('G', 0, 4),
 ('A', 0, 4), ('A', 0, 4), ('A', 0, 4), ('G', 0, 4), ('F', 1, 4), ('E', 0, 4),
 ('D', 0, 4), ('A', 0, 4), ('A', 0, 4), ('G', 0, 4), ('F', 1, 4), ('E', 0, 4),
 ('D', 0, 4), ('A', 0, 4), ('A', 0, 4), ('D', 0, 5), ('F', 1, 4), ('A', 0, 4),
 ('A', 0, 4), ('G', 0, 4), ('E', 0, 4), ('F', 1, 4), ('D', 0, 4), ('D', 0, 4),
 ('D', 0, 4), ('G', 0, 4), ('G', 0, 4), ('B', 0, 4), ('A', 0, 4), ('B', 0, 4),
 ('C', 0, 5), ('D', 0, 5), ('E', 0, 5), ('D', 0, 5), ('B', 0, 4), ('C', 0, 5),
 ('B', 0, 4), ('D', 0, 5), ('C', 0, 5), ('A', 0, 4), ('A', 0, 4), ('A', 0, 4),
 ('B', 0, 4), ('G', 0, 5), ('B', 0, 5), ('B', 0, 4), ('B', 0, 4), ('D', 0, 5),
 ('C', 0, 5), ('A', 0, 4), ('A', 0, 4), ('A', 0, 4), ('B', 0, 4), ('B', 0, 4),
 ('D', 0, 5), ('C', 0, 5), ('A', 0, 4), ('C', 0, 5), ('A', 0, 5), ('A', 0, 4),
 ('C', 0, 5), ('E', 0, 5), ('D', 0, 5), ('B', 0, 4), ('B', 0, 5), ('G', 0, 5),
 ('B', 0, 4), ('D', 0, 5), ('C', 0, 5), ('A', 0, 4), ('C', 0, 5), ('A', 0, 5),
 ('A', 0, 4), ('C', 0, 5), ('E', 0, 5), ('D', 0, 5), ('B', 0, 4), ('B', 0, 5),
 ('G', 0, 5), ('D', 0, 5), ('B', 0, 5), ('G', 0, 5), ('D', 0, 5), ('B', 0, 5),
 ('D', 0, 5), ('G', 0, 5), ('D', 0, 5), ('B', 0, 5), ('D', 0, 5), ('G', 0, 5),
 ('D', 0, 5), ('B', 0, 5), ('D', 0, 5), ('G', 0, 5), ('D', 0, 5), ('D', 0, 6),
 ('B', 0, 5), ('G', 0, 5)]


def _load_appenzel_melody() -> Melody:
    midi_path = "src/melody_features/corpora/Essen_Corpus/appenzel.mid"
    midi_data = import_midi(midi_path)
    assert midi_data is not None, f"Failed to import MIDI at {midi_path}"
    return Melody(midi_data)


def test_pitch_spelling_appenzel_matches_expected():
    melody = _load_appenzel_melody()
    spelling = estimate_spelling_from_melody(melody)
    produced = [(str(step), int(alter), int(octave)) for step, alter, octave in spelling]
    assert produced == pitch_spelling_vals


def test_tonal_tension_appenzel_matches_expected():
    melody = _load_appenzel_melody()
    tt = estimate_tonaltension(melody, ws=1.0, ss="onset")

    onset_field = list(tt.keys())[0]
    produced = list(zip(
        tt[onset_field],
        tt["cloud_diameter"],
        tt["cloud_momentum"],
        tt["tensile_strain"],
    ))

    # Length must match
    assert len(produced) == len(tonal_tension_vals)

    # Compare numerically with tolerance
    prod_arr = np.array(produced, dtype=float)
    exp_arr = np.array(tonal_tension_vals, dtype=float)
    np.testing.assert_allclose(prod_arr, exp_arr, rtol=1e-5, atol=1e-5)