""" Plotting utility imitating the uq_display functionality of UQLab """
from uqpylab import sessions
import numpy as np
import scipy as sp 
from scipy.stats import norm
import pandas as pd
import plotly.graph_objects as go
import plotly.express as px
import plotly.figure_factory as ff
from plotly.subplots import make_subplots
import plotly.io as pio
import matplotlib as mpl
import matplotlib.pyplot as plt
import matplotlib.cbook
import warnings
from IPython import get_ipython

FIGURE_WIDTH_CM = 15
FIGURE_HEIGHT_CM = 10
FIGURE_DPI = 200
## Figure template
pio.templates.default = 'simple_white'

## Template modifications
# Size
pio.templates[pio.templates.default].layout.height = round(FIGURE_HEIGHT_CM/2.54*FIGURE_DPI)
pio.templates[pio.templates.default].layout.width = round(FIGURE_WIDTH_CM/2.54*FIGURE_DPI)
# Grid
pio.templates[pio.templates.default].layout.xaxis.showgrid = True
pio.templates[pio.templates.default].layout.xaxis.linewidth = 2
pio.templates[pio.templates.default].layout.xaxis.mirror = True
pio.templates[pio.templates.default].layout.yaxis.showgrid = True
pio.templates[pio.templates.default].layout.yaxis.mirror = True
pio.templates[pio.templates.default].layout.yaxis.linewidth = 2
pio.templates[pio.templates.default].layout.autosize = False

# Font
pio.templates[pio.templates.default].layout.font.size = 16

# Markers
pio.templates[pio.templates.default].data.histogram[0].marker.line.width = 0


def _hex_to_rgba(hex, opacity=1):
    '''Auxiliary function to transform HEX color in rgba, optionally with transparency'''
    hex = hex.split('#')[-1]
    rgba = list(int(hex[i:i+2], 16) for i in (0, 2, 4))
    rgba.append(opacity)
    rgba=f'rgba({rgba[0]},{rgba[1]},{rgba[2]},{rgba[3]})'

    return rgba

def is_notebook() -> bool:
    '''
    Helper function that determines whether the code is currently 
    running in a Jupyter notebook or not. 
    Thanks to Gustavo Bezerra at Stack Overflow for this solution.
    '''
    try:
        shell = get_ipython().__class__.__name__
        if shell == 'ZMQInteractiveShell':
            return True   # Jupyter notebook or qtconsole
        elif shell == 'TerminalInteractiveShell':
            return False  # Terminal running IPython
        else:
            return False  # Other type (?)
    except NameError:
        return False      # Probably standard Python interpreter


def get_uq_color_order(*args):
    ''' Returns a list of lists, each containing an RGB triplet for a color that is 
    compatible with the colors used in the original UQLab examples    
    '''
    color_order = np.array([
        [0,      76,     153],
        [240,    135,    0],
        [34,     139,    34],
        [237,    177,    34],
        [186,    85,     211],
        [0,      204,    204],
        [204,    0,     0]])/255
    
    default_length = color_order.shape[0]
    
    if len(args) == 0:
        N = default_length
    else:
        N = args[0]
    
    if N <= default_length:
        colors = color_order[0:N,:]
    else:
        colors = np.zeros([N,3])
        colors[:,0] = np.interp(np.linspace(0,1,N), np.linspace(0, 1, color_order.shape[0]), color_order[:,0])
        colors[:,1] = np.interp(np.linspace(0,1,N), np.linspace(0, 1, color_order.shape[0]), color_order[:,1])
        colors[:,2] = np.interp(np.linspace(0,1,N), np.linspace(0, 1, color_order.shape[0]), color_order[:,2])
        
    return colors.tolist()

def load_plt_defaults():
    ''' Helper function for setting some figure styling defaults for matplotlib'''
    # By default matplotlib uses inches for figure size
    mpl.rcParams["figure.figsize"] = (round(FIGURE_WIDTH_CM/2.54), round(FIGURE_HEIGHT_CM/2.54))
    mpl.rcParams["figure.dpi"] = FIGURE_DPI
    mpl.rcParams["lines.linewidth"] = 0.8
    mpl.rcParams["axes.linewidth"] = 0.8
    mpl.rcParams["axes.grid"] = True
    mpl.rcParams['font.size']= 8
    mpl.rcParams['axes.prop_cycle'] = plt.cycler("color", get_uq_color_order())
    if is_notebook():
        warnings.filterwarnings("ignore",category=matplotlib.cbook.mplDeprecation)
        rc = plt.rcParams.copy()
        ipython = get_ipython()
        ipython.magic("matplotlib inline")
        plt.rcParams.update(rc)

def Input(Module, theInterface, idx = None, plot_density=False, show_vine=False, **kwargs):
    '''Input display is a utility function for visualising a random
    vector specified in an Input object of generated by the uq_default_input
    module. By default, it generates a scatter plot matrix of the elements of
    the random vector. There is also the option to plot the PDF and CDF of
    the marginal distributions of the vector components.

    Args:
        Module (dict, optional): Input parameters of multivariate random vectors. Defaults to None.
        idx (list, optional): Set of indexes that define the variables to be displayed . Defaults to None.
        plot_density (bool, optional): Flag to plot the marginal distributions. Defaults to False.
        show_vine (bool, optional): Flag to plot a graphical representation of a given vine copula. Defaults to False.
    '''

    # Vines visualization
    if show_vine:
        # Define structure
        n_trees = len(Module['Copula']['Structure'])-1
        n_nodes = len(Module['Copula']['Structure'])
        x = [np.cos(np.pi/2 - 2*np.pi*var/n_nodes) for var in range(n_nodes)]
        y = [np.sin(np.pi/2 - 2*np.pi*var/n_nodes) for var in range(n_nodes)]

        # Define subplot grid
        n_cols = n_trees if n_trees<=4 else 4
        n_rows = n_trees//n_cols + (n_trees % n_cols > 0)

        # Create figure
        fig = make_subplots(rows=n_rows, cols=n_cols,  shared_xaxes=True, shared_yaxes=True, subplot_titles=[f'<b>T<sub>{tree}' for tree in range(1, n_trees+1)])

        # Create subplot for each tree
        for tree in range(1, n_trees+1):
            row = tree//4 + (tree % 4 > 0)
            col = tree - (row*4 - 4)

            # Pairs
            filter = [x==tree for x in Module['Copula']['Trees']]
            pairs = [Module['Copula']['Pairs'][i] for i in range(len( Module['Copula']['Pairs'])) if filter[i]]
            i = 0
            for pair in pairs:
                trace = go.Scatter(
                    x = [x[Module['Copula']['Structure'].index(pair[0])], x[Module['Copula']['Structure'].index(pair[1])]],
                    y = [y[Module['Copula']['Structure'].index(pair[0])], y[Module['Copula']['Structure'].index(pair[1])]],
                    mode = 'lines',
                    line = dict(
                        dash = 'dot' if Module['Copula']['Families'][i]=='Independent' else 'solid',
                        width = 8,
                        color = _hex_to_rgba(pio.templates[pio.templates.default].layout.colorway[tree-1], 0.5)
                    ),
                )
                fig.add_trace(trace, row = row, col = col)
                i += 1

            # Nodes
            color = ['black' if i in [node for pair in pairs for node in pair] else 'grey' for i in Module['Copula']['Structure']]
            color[tree-1] = pio.templates[pio.templates.default].layout.colorway[tree-1]
            trace = go.Scatter(
                x = x,
                y = y,
                text = [f'X<sub>{var}' for var in Module['Copula']['Structure']],
                mode = 'markers+text',
                marker = dict(size = 50, color=color),
                textfont = dict(color = 'white'),
                name=f'T<sub>{tree}',
                customdata=Module['Copula']['Families'],
                hovertemplate='<b>%{text}</b><br>Families: %{customdata}'
            )
            fig.add_trace(trace, row = row, col = col)

            # Subplot layout
            fig.update_xaxes(showline=False, showgrid=False, zeroline=False, showticklabels=False, ticks='', row=row, col=col)
            fig.update_yaxes(showline=False, showgrid=False, zeroline=False, showticklabels=False, ticks='', row=row, col=col)

        # General layout
        fig.update_layout(showlegend=False, hovermode='closest', width=800, height=400)

    # Random vectors visualization
    else:
        # Parameters
        M_max = 10 # Input dimension limit
        N = 1000 # Sample size
        n_std = 5 # Number of std's for the x_range in case of input_array scalar (displaying of a single marginal)
        nbins = 15 # Number of bins of marginal histograms (displaying of two marginals)
        
        # Define idx
        if isinstance(Module['Marginals'], dict):
            Module['Marginals'] = [Module['Marginals']]
        M = len(Module['Marginals'])
        if idx is None:
            idx = range(1, M+1)
        idx_names = [Module["Marginals"][i-1]["Name"] for i in idx]
        idx = [f'X{var}' for var in idx]
        n_var = len(idx)

        # Consistency checks
        if M > M_max:
            return print('Requested input range is too large, please select a smaller range.')

        # Get sample for plotting
        X = theInterface.getSample(Module,N,'LHS')
        df = pd.DataFrame(X)
        df.columns = [f'X{var}' for var in range(1,X.shape[1]+1)]
        df = df[idx]

        ## Produce the plot
        # Scatter plot for two inputs
        if (n_var==2) & (not plot_density):
            # Figure
            fig = go.FigureWidget()
            fig.add_scatter(x=X[:,0], y=X[:,1], 
                            mode='markers', 
                            marker = {'opacity':.4}
                            )
            fig.add_histogram(x=X[:,0], 
                            #   nbinsx=nbins, # Specifies the maximum number of desired bins. This value will be used in an algorithm that will decide the optimal bin size such that the histogram best visualizes the distribution of the data. Ignored if `xbins.size` is provided.
                              name='', 
                              yaxis='y2'
                              )  
            fig.add_histogram(y=X[:,1], 
                            #   nbinsy=nbins, 
                              name='', 
                              xaxis='x2'
                              )
            # Aesthetics
            fig.layout = {
                            'xaxis': {
                                'domain': [0, 0.75], 
                                'showgrid': True,
                                'mirror': True,
                                },
                            'yaxis': {
                                'domain': [0, 0.75], 
                                'showgrid': True, 
                                'mirror': True,
                                },
                            'showlegend': False,
                            'margin': {'t': 50},
                            'xaxis2': {
                                'domain': [0.75, 1],  
                                'zeroline': False, 
                                'visible': False, 
                                'showticklabels': False,
                                },
                            'yaxis2': {
                                'domain': [0.75, 1], 
                                'zeroline': False, 
                                'visible': False, 
                                'showticklabels': False,
                                },
                        }
            fig._data_objs[1].marker.color = fig._data_objs[2].marker.color = pio.templates[pio.templates.default].layout.colorway[1]
            fig.data[1]['opacity'] = fig.data[2]['opacity'] = 1
            fig.data[1].xbins.size = (max(X[:,0])-min(X[:,0]))/nbins 
            fig.data[2].ybins.size = (max(X[:,1])-min(X[:,1]))/nbins
            idx_names = [idx_name.replace("_","<sub>") for idx_name in idx_names]
            fig.update_layout(xaxis_title=idx_names[0], yaxis_title=idx_names[1])

        # Scatter matrix for >2 inputs
        elif (n_var>2) & (not plot_density):
            width = 250*n_var if n_var<=5 else 250*5
            # Figure
            fig = ff.create_scatterplotmatrix(
                df,
                diag='histogram',
                marker_opacity=0.4,
                width=width,
                height=width
            )
            # Aesthetics
            fig.update_layout(
                title='',
                hovermode = False,
                colorway = [pio.templates[pio.templates.default].layout.colorway[0]] # Scatter color
            )
            fig.update_xaxes(mirror=True)
            fig.update_yaxes(mirror=True)
            for data in fig.data:
                if type(data)==type(go.Histogram()):
                    data.xbins.size = (max(data.x)-min(data.x))/nbins # Bin width
                    data.autobinx=False
                    data.marker.color = pio.templates[pio.templates.default].layout.colorway[1] # Histogram color

            idx_names = [idx_name.replace("_","<sub>") for idx_name in idx_names]
            for i in range(len(idx)):
                for j in range(len(idx)):
                    fig.update_xaxes(title_text=idx_names[j], row=len(idx), col=j+1)
                    fig.update_yaxes(title_text=idx_names[i], row=i+1, col=1)

        # Line plots for density functions
        else:
            # Figure
            fig = make_subplots(
                rows=n_var, cols=2,
                vertical_spacing=0.12,
                subplot_titles=('PDF', 'CDF')
            )
            row = 1
            for var in idx_names:
                # Prepare data
                marginal = [x for x in Module['Marginals'] if x['Name']==var][0]
                x_mean, x_std = marginal['Moments']
                x = np.linspace(x_mean-n_std*x_std, x_mean+n_std*x_std, N)

                # Calculate PDF and CDF
                pdf=theInterface.all_pdf(x, marginal)
                cdf = theInterface.all_cdf(x, marginal)

                # Add data to figure
                fig.add_trace(
                    go.Scatter(
                            x=x, y=pdf,
                            line=dict(color=pio.templates[pio.templates.default].layout.colorway[row-1]),
                            name=f'{var} PDF'
                        ),
                        row=row, col=1
                )

                fig.add_trace(
                    go.Scatter(
                            x=x, y=cdf,
                            line=dict(color=pio.templates[pio.templates.default].layout.colorway[row-1]),
                            name=f'{var} CDF'
                        ),
                        row=row, col=2
                )
                var = var.replace("_","<sub>")
                fig.update_xaxes(title_text=var, row=row, col=1)
                fig.update_xaxes(title_text=var, row=row, col=2)
                fig.update_yaxes(title_text=f'f<sub>{var}', row=row, col=1)
                fig.update_yaxes(title_text=f'F<sub>{var}', row=row, col=2)
                row += 1

            # Aesthetics
            fig.update_layout(showlegend=False, height=500*n_var, width=1000)

    return fig


def PCE(Model, **kwargs):

    # Prepare data
    Coefficients = np.array(Model['PCE']['Coefficients'])
    nnz_idx = np.abs(Coefficients) > 0
    alpha = np.array(np.nonzero(Coefficients)).squeeze()
    Coefficients = Coefficients[nnz_idx]
    Indices = np.array(Model['PCE']['Basis']['Indices'])
    try:
        Indices = Indices[nnz_idx]
    except IndexError as e:
        # This error will be raised if Indices is a sparse matrix 
        Indices = Indices[()].toarray()
        Indices = Indices[nnz_idx]

    Indices_row_num = np.arange(1,Indices.shape[0]+1)
    Degree = Indices.sum(axis=1)
    legend_txt = ['Mean','<i>p=1','<i>p=2','<i>p=3','<i>p>3']
    logCoeffs = np.log10(np.abs(Coefficients))
    logCoeffRange = np.max([1, np.max(logCoeffs) - np.min(logCoeffs)])

    # Create figure traces
    traces = []
    for k in range(5):
        if k < 4:
            idx = Degree == k
        else:
            idx = Degree >= k
        traces.append(
            go.Scatter(
                x=alpha[idx],
                y=np.log10(np.abs(Coefficients[idx])),
                mode='markers',
                marker=dict(size=15-2*k, opacity=0.5),
                name=legend_txt[k]
            )
        )

    # Define figure layout
    layout = go.Layout(
        title=f'NNZ Coeffs: {Coefficients.shape[0]}',
        xaxis_title='𝛼',
        yaxis_title='log<sub>10</sub>(|𝑦<sub>𝛼</sub>|)'
    )
    # Create figure
    fig = go.Figure(data=traces, layout=layout)

    return fig


def Kriging(Model, theInterface, **kwargs):
    """Display Kriging dictionary with 1-dimensional output"""

    N1d = 500  # Number of points to plot in one-dimension
    N2d = 80   # Number of points to plot in two-dimension

    # One-dimensional case

    # Compute points to plot
    X = np.array(Model['ExpDesign']['X'])
    Y = np.array(Model['ExpDesign']['Y'])
    Xmin = np.min(X)
    Xmax = np.max(X)
    Xval = np.linspace(Xmin, Xmax, N1d).transpose()
    Xval = np.sort(np.concatenate((Xval, X)))  # Exp. design points belong to evaluation
    [Ymu_krg,Ysigma_krg] = theInterface.evalModel(Model, Xval.reshape([Xval.shape[0],1]), nargout=2)

    # Compute upper and lower bounds of the confidence interval
    conf_level = 0.95;  # 95% confidendence level
    conf_interval = norm.ppf(1-0.5*(1-conf_level), 0, 1) * np.sqrt(Ysigma_krg)

    ## Figure
    # Traces
    traces = [
        go.Scatter(
            x=Xval.squeeze(),
            y=Ymu_krg.squeeze(),
            name='Kriging approximation',
            fill=None,
            mode='lines',
            legendgroup=0
        ),
        go.Scatter(
            x=Xval.squeeze(),
            y=(Ymu_krg - conf_interval).squeeze(),
            name='95% confidence interval',
            fill=None,
            legendgroup=1,
            mode='lines',
            marker=dict(color='lightgrey'),
            showlegend=False
        ),
        go.Scatter(
            x=Xval.squeeze(),
            y=(Ymu_krg + conf_interval).squeeze(),
            fill='tonexty',
            name='95% confidence interval',
            legendgroup=1,
            mode='lines',
            marker=dict(color='lightgrey')
        ),
        go.Scatter(
            x=X.squeeze(),
            y=Y.squeeze(),
            name='Observations',
            fill=None,
            legendgroup=2,
            mode='markers',
            marker=dict(color='black')
        )
    ]

    # Layout
    layout = go.Layout(
        title=None,
        xaxis_title_text='X',
        xaxis_range=[Xmin, Xmax],
        yaxis_title_text='Y',
        hovermode='closest',
        legend=dict(
            yanchor="top",
            y=0.99,
            xanchor="right",
            x=0.99,
            bgcolor='rgba(255,255,255,0.5)'

        )
    )

    # Create figure
    fig = go.Figure(data=traces, layout=layout)
    return fig


def colorOrder(N=None):
    # Define the color order(s)
    alt2 = np.array(
        [[0,      76,     153], 
        [240,    135,       0],
        [34,     139,      34],
        [186,     23,     117],
        [237,    177,      34],
        [186,     85,     211],
        [0,      204,     204],
        [204,      0,     	0]]
    )   
    
    # Assign the color order
    colorOrder = alt2
    defaultlength = colorOrder.shape[0]

    # Give all Chair colors if no number is specified
    if N is None:
        N=defaultlength
    if N <= defaultlength:
        colors = colorOrder[0:N,:]
    else:
        colors = np.zeros([N,3])
        colors[:,0] = np.interp(np.linspace(0,1,N), np.linspace(0,1,defaultlength), colorOrder[:,0])
        colors[:,1] = np.interp(np.linspace(0,1,N), np.linspace(0,1,defaultlength), colorOrder[:,1])
        colors[:,2] = np.interp(np.linspace(0,1,N), np.linspace(0,1,defaultlength), colorOrder[:,2])

    colors = np.around(colors)
    colors = colors.astype(int)
    colors_rgb = []
    for ii in range(N):
        colors_rgb.append( f'rgb({colors[ii,0]}, {colors[ii,1]}, {colors[ii,2]})')
    
    return colors_rgb

