<!--
Copyright (C) 2024 Roberto Rossini <roberros@uio.no>

SPDX-License-Identifier: MIT
-->

# StripePy

[![License](https://img.shields.io/badge/license-MIT-green)](https://github.com/paulsengroup/StripePy/blob/main/LICENCE)
[![CI](https://github.com/paulsengroup/StripePy/actions/workflows/ci.yml/badge.svg)](https://github.com/paulsengroup/StripePy/actions/workflows/ci.yml)
[![Build Dockerfile](https://github.com/paulsengroup/StripePy/actions/workflows/build-dockerfile.yml/badge.svg)](https://github.com/paulsengroup/StripePy/actions/workflows/build-dockerfile.yml)
[![Download from Bioconda](https://img.shields.io/conda/vn/bioconda/stripepy-hic?label=bioconda&logo=Anaconda)](https://anaconda.org/bioconda/stripepy-hic)
[![Zenodo DOI](https://zenodo.org/badge/DOI/10.5281/zenodo.14394042.svg)](https://doi.org/10.5281/zenodo.14394041)

---

<!--
> [!IMPORTANT]
> You are reading the README for the latest version of StripePy.
>
> Some of the functionality and features described in this version of the README may not apply to previous versions of StripePy.
>
> If you did not install StripePy from source, please refer to the README for the latest stable release - [link](https://github.com/paulsengroup/StripePy/tree/v1.0.0).
>
> <details>
>  <summary>Older releases</summary>
>
> - [v0.0.2](https://github.com/paulsengroup/StripePy/tree/v0.0.2)
> - [v0.0.1](https://github.com/paulsengroup/StripePy/tree/v0.0.1)
>
> </details>
-->

StripePy is a CLI application written in Python that recognizes architectural stripes found in the interaction matrix files
generated by Chromosome Conformation Capture experiments, such as Hi-C and Micro-C.
Matrix files in `.cool`, `.mcool`, and `.hic` (including `.hic` v9 files) are supported.

StripePy is developed on Linux and macOS and is also tested on Windows.

## Installing StripePy

### Installing with pip

```bash
pip install 'stripepy-hic[all]'
```

### Installing with conda

```bash
conda create -n stripepy -c conda-forge -c bioconda stripepy-hic
```

### Installing from source

Instructions for Linux and macOS:

```bash
# create and activate a venv (optional)
python3 -m venv venv
. venv/bin/activate

# get StripePy source code
git clone https://github.com/paulsengroup/StripePy.git

# optional, checkout a specific version
# git checkout v0.0.2

# install StripePy
cd StripePy
pip install '.[all]'

# ensure StripePy is in your PATH
stripepy --help
```

<details>
<summary>Instructions for Windows</summary>

```bash
# create and activate a venv (optional)
python3 -m venv venv
venv\Scripts\activate

# get StripePy source code
git clone https://github.com/paulsengroup/StripePy.git

# optional, checkout a specific version
# git checkout v0.0.2

# install StripePy
cd StripePy
pip install .

# ensure StripePy is in your PATH
stripepy --help
```

</details>

## Running StripePy

StripePy is organized into a few subcommands:

- `stripepy call`: run the stripe detection algorithm and store the identified stripes in a `.hdf5` file.
- `stripepy view`: take the `result.hdf5` file generated by `stripepy call` and extract stripes in BEDPE format.
- `stripepy plot`: generate various kinds of plots to inspect the stripes identified by `stripepy call`.
- `stripepy download`: download a minified sample dataset suitable to quickly test StripePy.

### Walkthrough

The following is an example of a typical run of StripePy.
The steps outlined in this section assume that StripePy is running on a UNIX system.
Some commands may need a bit of tweaking to run on Windows.

#### 1) Download a sample dataset (optional)

Feel free to use your own interaction matrix instead of `4DNFI9GMP2J8.mcool`.

Please make sure the matrix is in `.cool`, `.mcool`, or `.hic` format.

```console
# This may take a while on slow internet connections
user@dev:/tmp$ stripepy download --name 4DNFI9GMP2J8

2025-01-14 12:46:01.304277 [info     ] downloading dataset "4DNFI9GMP2J8" (assembly=hg38)...
2025-01-14 12:46:23.900411 [info     ] DONE! Downloading dataset "4DNFI9GMP2J8" took 22.596s.
2025-01-14 12:46:23.901141 [info     ] computing MD5 digest for file "/tmp/4DNFI9GMP2J8.dvizz7v1"...
2025-01-14 12:46:24.050566 [info     ] MD5 checksum match!
2025-01-14 12:46:24.050695 [info     ] successfully downloaded dataset "https://zenodo.org/records/14643417/files/4DNFI9GMP2J8.stripepy.mcool?download=1" to file "4DNFI9GMP2J8.mcool"
2025-01-14 12:46:24.050752 [info     ] file size: 106.84MB. Elapsed time: 22.979s
```

#### 2) Detect architectural stripes

This is the core of the analysis and may take several minutes when processing large files.

```console
user@dev:/tmp$ stripepy call 4DNFI9GMP2J8.mcool 10000

2025-04-15 08:13:24.639742 [info     ] running StripePy v1.0.0
2025-04-15 08:13:24.637358 [info     ] [main      ] CONFIG:
{
  "constrain_heights": false,
  "contact_map": "4DNFI9GMP2J8.mcool",
  "force": false,
  "genomic_belt": 5000000,
  "glob_pers_min": 0.04,
  "loc_pers_min": 0.33,
  "loc_trend_min": 0.25,
  "log_file": null,
  "max_width": 100000,
  "min_chrom_size": 2000000,
  "normalization": null,
  "nproc": 1,
  "output_file": "/tmp/4DNFI9GMP2J8.10000.hdf5",
  "plot_dir": null,
  "resolution": 10000,
  "roi": null,
  "verbosity": "info"
}
2025-04-15 08:13:24.637440 [info     ] [main      ] validating file "4DNFI9GMP2J8.mcool" (10000bp)
2025-04-15 08:13:24.650236 [info     ] [main      ] file "4DNFI9GMP2J8.mcool" successfully validated
2025-04-15 08:13:24.650445 [info     ] [IO        ] initializing result file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:24.672613 [info     ] [chr1 ] [main      ] begin processing
2025-04-15 08:13:24.672729 [info     ] [chr1 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:25.483686 [info     ] [chr1 ] [IO        ] fetched 6823257 pixels in 810.948ms
2025-04-15 08:13:25.483913 [info     ] [chr1 ] [step 1    ] data pre-processing
2025-04-15 08:13:25.483995 [info     ] [chr1 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:25.535171 [info     ] [chr1 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/6823257)
2025-04-15 08:13:25.535378 [info     ] [chr1 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:25.549232 [info     ] [chr1 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:25.553946 [info     ] [chr1 ] [step 1    ] preprocessing took 69.937ms
2025-04-15 08:13:25.558918 [info     ] [chr1 ] [step 2    ] topological data analysis
2025-04-15 08:13:25.559059 [info     ] [chr1 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:25.583652 [info     ] [chr1 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:25.583770 [info     ] [chr1 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:25.625730 [info     ] [chr1 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:25.626417 [info     ] [chr1 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:25.686625 [info     ] [chr1 ] [step 2.2.3] [LT] number of seed sites reduced from 1807 to 1748
2025-04-15 08:13:25.686795 [info     ] [chr1 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:25.687662 [info     ] [chr1 ] [step 2.3.1] [LT] identified 1748 candidate stripes
2025-04-15 08:13:25.687864 [info     ] [chr1 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:25.713048 [info     ] [chr1 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:25.713154 [info     ] [chr1 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:25.753436 [info     ] [chr1 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:25.753932 [info     ] [chr1 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:25.813509 [info     ] [chr1 ] [step 2.2.3] [UT] number of seed sites reduced from 1698 to 1647
2025-04-15 08:13:25.813687 [info     ] [chr1 ] [step 2.3.1] [UT] generating the list of candidate stripes
...
2025-04-15 08:14:59.123408 [info     ] [IO        ] finalizing file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:59.127303 [info     ] [main      ] DONE!
2025-04-15 08:14:59.127399 [info     ] [main      ] processed 24 chromosomes in 1m:34.490s
```

<details>
<summary>Complete log</summary>

```txt
2025-04-15 08:13:24.639742 [info     ] running StripePy v1.0.0
2025-04-15 08:13:24.637358 [info     ] [main      ] CONFIG:
{
  "constrain_heights": false,
  "contact_map": "4DNFI9GMP2J8.mcool",
  "force": false,
  "genomic_belt": 5000000,
  "glob_pers_min": 0.04,
  "loc_pers_min": 0.33,
  "loc_trend_min": 0.25,
  "log_file": null,
  "max_width": 100000,
  "min_chrom_size": 2000000,
  "normalization": null,
  "nproc": 1,
  "output_file": "/tmp/4DNFI9GMP2J8.10000.hdf5",
  "plot_dir": null,
  "resolution": 10000,
  "roi": null,
  "verbosity": "info"
}
2025-04-15 08:13:24.637440 [info     ] [main      ] validating file "4DNFI9GMP2J8.mcool" (10000bp)
2025-04-15 08:13:24.650236 [info     ] [main      ] file "4DNFI9GMP2J8.mcool" successfully validated
2025-04-15 08:13:24.650445 [info     ] [IO        ] initializing result file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:24.672613 [info     ] [chr1 ] [main      ] begin processing
2025-04-15 08:13:24.672729 [info     ] [chr1 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:25.483686 [info     ] [chr1 ] [IO        ] fetched 6823257 pixels in 810.948ms
2025-04-15 08:13:25.483913 [info     ] [chr1 ] [step 1    ] data pre-processing
2025-04-15 08:13:25.483995 [info     ] [chr1 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:25.535171 [info     ] [chr1 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/6823257)
2025-04-15 08:13:25.535378 [info     ] [chr1 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:25.549232 [info     ] [chr1 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:25.553946 [info     ] [chr1 ] [step 1    ] preprocessing took 69.937ms
2025-04-15 08:13:25.558918 [info     ] [chr1 ] [step 2    ] topological data analysis
2025-04-15 08:13:25.559059 [info     ] [chr1 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:25.583652 [info     ] [chr1 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:25.583770 [info     ] [chr1 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:25.625730 [info     ] [chr1 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:25.626417 [info     ] [chr1 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:25.686625 [info     ] [chr1 ] [step 2.2.3] [LT] number of seed sites reduced from 1807 to 1748
2025-04-15 08:13:25.686795 [info     ] [chr1 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:25.687662 [info     ] [chr1 ] [step 2.3.1] [LT] identified 1748 candidate stripes
2025-04-15 08:13:25.687864 [info     ] [chr1 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:25.713048 [info     ] [chr1 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:25.713154 [info     ] [chr1 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:25.753436 [info     ] [chr1 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:25.753932 [info     ] [chr1 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:25.813509 [info     ] [chr1 ] [step 2.2.3] [UT] number of seed sites reduced from 1698 to 1647
2025-04-15 08:13:25.813687 [info     ] [chr1 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:25.814397 [info     ] [chr1 ] [step 2.3.1] [UT] identified 1647 candidate stripes
2025-04-15 08:13:25.814632 [info     ] [chr1 ] [step 2    ] topological data analysis took 255.640ms
2025-04-15 08:13:25.814740 [info     ] [chr1 ] [step 3    ] shape analysis
2025-04-15 08:13:25.815046 [info     ] [chr1 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:25.826410 [info     ] [chr1 ] [step 3.1  ] [LT] width estimation of 1748 stripes took 11.247ms (mean=57 kbp; std=19 kbp)
2025-04-15 08:13:25.826482 [info     ] [chr1 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:25.831772 [info     ] [chr1 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:27.817125 [info     ] [chr1 ] [step 3.3  ] [LT] height estimation of 1748 stripes tool 1.985s (mean=874 kbp; std=364 kbp)
2025-04-15 08:13:27.817211 [info     ] [chr1 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:27.822574 [info     ] [chr1 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:27.832784 [info     ] [chr1 ] [step 3.1  ] [UT] width estimation of 1647 stripes took 10.146ms (mean=57 kbp; std=19 kbp)
2025-04-15 08:13:27.832882 [info     ] [chr1 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:27.837603 [info     ] [chr1 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:13:30.239395 [info     ] [chr1 ] [step 3.3  ] [UT] height estimation of 1647 stripes tool 2.402s (mean=879 kbp; std=376 kbp)
2025-04-15 08:13:30.239518 [info     ] [chr1 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:13:30.244612 [info     ] [chr1 ] [step 3    ] shape analysis took 4.430s
2025-04-15 08:13:30.244799 [info     ] [chr1 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:13:30.244897 [info     ] [chr1 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:13:30.642057 [info     ] [chr1 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:13:31.133886 [info     ] [chr1 ] [step 4    ] statistical analysis and post-processing took 889.078ms
2025-04-15 08:13:31.134173 [info     ] [chr1 ] [main      ] processing took 6.462s
2025-04-15 08:13:31.134287 [info     ] [chr1 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:31.207334 [info     ] [chr1 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 72.988ms
2025-04-15 08:13:31.207606 [info     ] [chr2 ] [main      ] begin processing
2025-04-15 08:13:31.207785 [info     ] [chr2 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:31.963097 [info     ] [chr2 ] [IO        ] fetched 7182898 pixels in 755.306ms
2025-04-15 08:13:31.963348 [info     ] [chr2 ] [step 1    ] data pre-processing
2025-04-15 08:13:31.963413 [info     ] [chr2 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:32.018461 [info     ] [chr2 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/7182898)
2025-04-15 08:13:32.018687 [info     ] [chr2 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:32.032683 [info     ] [chr2 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:32.037780 [info     ] [chr2 ] [step 1    ] preprocessing took 74.350ms
2025-04-15 08:13:32.043148 [info     ] [chr2 ] [step 2    ] topological data analysis
2025-04-15 08:13:32.043252 [info     ] [chr2 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:32.067427 [info     ] [chr2 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:32.067555 [info     ] [chr2 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:32.109899 [info     ] [chr2 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:32.110423 [info     ] [chr2 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:32.172155 [info     ] [chr2 ] [step 2.2.3] [LT] number of seed sites reduced from 2052 to 2011
2025-04-15 08:13:32.172338 [info     ] [chr2 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:32.173237 [info     ] [chr2 ] [step 2.3.1] [LT] identified 2011 candidate stripes
2025-04-15 08:13:32.173454 [info     ] [chr2 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:32.199762 [info     ] [chr2 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:32.199906 [info     ] [chr2 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:32.241973 [info     ] [chr2 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:32.242495 [info     ] [chr2 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:32.303911 [info     ] [chr2 ] [step 2.2.3] [UT] number of seed sites reduced from 2028 to 1987
2025-04-15 08:13:32.304067 [info     ] [chr2 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:32.304928 [info     ] [chr2 ] [step 2.3.1] [UT] identified 1987 candidate stripes
2025-04-15 08:13:32.305160 [info     ] [chr2 ] [step 2    ] topological data analysis took 261.924ms
2025-04-15 08:13:32.306037 [info     ] [chr2 ] [step 3    ] shape analysis
2025-04-15 08:13:32.306145 [info     ] [chr2 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:32.318279 [info     ] [chr2 ] [step 3.1  ] [LT] width estimation of 2011 stripes took 12.064ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:13:32.318379 [info     ] [chr2 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:32.324132 [info     ] [chr2 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:34.587703 [info     ] [chr2 ] [step 3.3  ] [LT] height estimation of 2011 stripes tool 2.264s (mean=916 kbp; std=383 kbp)
2025-04-15 08:13:34.587811 [info     ] [chr2 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:34.593684 [info     ] [chr2 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:34.605716 [info     ] [chr2 ] [step 3.1  ] [UT] width estimation of 1987 stripes took 11.978ms (mean=55 kbp; std=18 kbp)
2025-04-15 08:13:34.605813 [info     ] [chr2 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:34.611573 [info     ] [chr2 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:13:37.457280 [info     ] [chr2 ] [step 3.3  ] [UT] height estimation of 1987 stripes tool 2.846s (mean=915 kbp; std=386 kbp)
2025-04-15 08:13:37.457388 [info     ] [chr2 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:13:37.463513 [info     ] [chr2 ] [step 3    ] shape analysis took 5.157s
2025-04-15 08:13:37.463693 [info     ] [chr2 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:13:37.463822 [info     ] [chr2 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:13:37.922763 [info     ] [chr2 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:13:38.501737 [info     ] [chr2 ] [step 4    ] statistical analysis and post-processing took 1.038s
2025-04-15 08:13:38.502010 [info     ] [chr2 ] [main      ] processing took 7.294s
2025-04-15 08:13:38.502111 [info     ] [chr2 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:38.587222 [info     ] [chr2 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 85.050ms
2025-04-15 08:13:38.587466 [info     ] [chr3 ] [main      ] begin processing
2025-04-15 08:13:38.587709 [info     ] [chr3 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:39.235967 [info     ] [chr3 ] [IO        ] fetched 5966083 pixels in 648.256ms
2025-04-15 08:13:39.236286 [info     ] [chr3 ] [step 1    ] data pre-processing
2025-04-15 08:13:39.236392 [info     ] [chr3 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:39.311664 [info     ] [chr3 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/5966083)
2025-04-15 08:13:39.311899 [info     ] [chr3 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:39.328207 [info     ] [chr3 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:39.335635 [info     ] [chr3 ] [step 1    ] preprocessing took 99.222ms
2025-04-15 08:13:39.336072 [info     ] [chr3 ] [step 2    ] topological data analysis
2025-04-15 08:13:39.336223 [info     ] [chr3 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:39.365779 [info     ] [chr3 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:39.366051 [info     ] [chr3 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:39.426420 [info     ] [chr3 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:39.427652 [info     ] [chr3 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:39.511423 [info     ] [chr3 ] [step 2.2.3] [LT] number of seed sites reduced from 1637 to 1630
2025-04-15 08:13:39.512254 [info     ] [chr3 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:39.513641 [info     ] [chr3 ] [step 2.3.1] [LT] identified 1630 candidate stripes
2025-04-15 08:13:39.514090 [info     ] [chr3 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:39.549185 [info     ] [chr3 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:39.549878 [info     ] [chr3 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:39.597171 [info     ] [chr3 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:39.597979 [info     ] [chr3 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:39.673279 [info     ] [chr3 ] [step 2.2.3] [UT] number of seed sites reduced from 1824 to 1814
2025-04-15 08:13:39.673543 [info     ] [chr3 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:39.674443 [info     ] [chr3 ] [step 2.3.1] [UT] identified 1814 candidate stripes
2025-04-15 08:13:39.674692 [info     ] [chr3 ] [step 2    ] topological data analysis took 338.496ms
2025-04-15 08:13:39.675910 [info     ] [chr3 ] [step 3    ] shape analysis
2025-04-15 08:13:39.676010 [info     ] [chr3 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:39.686625 [info     ] [chr3 ] [step 3.1  ] [LT] width estimation of 1630 stripes took 10.512ms (mean=55 kbp; std=18 kbp)
2025-04-15 08:13:39.686701 [info     ] [chr3 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:39.691815 [info     ] [chr3 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:41.614175 [info     ] [chr3 ] [step 3.3  ] [LT] height estimation of 1630 stripes tool 1.922s (mean=901 kbp; std=337 kbp)
2025-04-15 08:13:41.614232 [info     ] [chr3 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:41.619093 [info     ] [chr3 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:41.629700 [info     ] [chr3 ] [step 3.1  ] [UT] width estimation of 1814 stripes took 10.541ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:13:41.629759 [info     ] [chr3 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:41.635097 [info     ] [chr3 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:13:44.258769 [info     ] [chr3 ] [step 3.3  ] [UT] height estimation of 1814 stripes tool 2.624s (mean=898 kbp; std=373 kbp)
2025-04-15 08:13:44.258877 [info     ] [chr3 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:13:44.264314 [info     ] [chr3 ] [step 3    ] shape analysis took 4.588s
2025-04-15 08:13:44.264547 [info     ] [chr3 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:13:44.264661 [info     ] [chr3 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:13:44.626636 [info     ] [chr3 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:13:45.147885 [info     ] [chr3 ] [step 4    ] statistical analysis and post-processing took 883.334ms
2025-04-15 08:13:45.148170 [info     ] [chr3 ] [main      ] processing took 6.561s
2025-04-15 08:13:45.148263 [info     ] [chr3 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:45.225337 [info     ] [chr3 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 77.018ms
2025-04-15 08:13:45.225581 [info     ] [chr4 ] [main      ] begin processing
2025-04-15 08:13:45.225744 [info     ] [chr4 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:45.775500 [info     ] [chr4 ] [IO        ] fetched 5491958 pixels in 549.751ms
2025-04-15 08:13:45.775752 [info     ] [chr4 ] [step 1    ] data pre-processing
2025-04-15 08:13:45.775808 [info     ] [chr4 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:45.815438 [info     ] [chr4 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/5491958)
2025-04-15 08:13:45.815625 [info     ] [chr4 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:45.826592 [info     ] [chr4 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:45.830532 [info     ] [chr4 ] [step 1    ] preprocessing took 54.708ms
2025-04-15 08:13:45.830878 [info     ] [chr4 ] [step 2    ] topological data analysis
2025-04-15 08:13:45.830938 [info     ] [chr4 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:45.850608 [info     ] [chr4 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:45.850691 [info     ] [chr4 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:45.884007 [info     ] [chr4 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:45.884469 [info     ] [chr4 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:45.931659 [info     ] [chr4 ] [step 2.2.3] [LT] number of seed sites reduced from 1606 to 1596
2025-04-15 08:13:45.931808 [info     ] [chr4 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:45.932440 [info     ] [chr4 ] [step 2.3.1] [LT] identified 1596 candidate stripes
2025-04-15 08:13:45.932741 [info     ] [chr4 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:45.953722 [info     ] [chr4 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:45.953834 [info     ] [chr4 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:45.987554 [info     ] [chr4 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:45.988028 [info     ] [chr4 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:46.036525 [info     ] [chr4 ] [step 2.2.3] [UT] number of seed sites reduced from 1426 to 1414
2025-04-15 08:13:46.036672 [info     ] [chr4 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:46.037403 [info     ] [chr4 ] [step 2.3.1] [UT] identified 1414 candidate stripes
2025-04-15 08:13:46.037602 [info     ] [chr4 ] [step 2    ] topological data analysis took 206.682ms
2025-04-15 08:13:46.038474 [info     ] [chr4 ] [step 3    ] shape analysis
2025-04-15 08:13:46.038574 [info     ] [chr4 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:46.048635 [info     ] [chr4 ] [step 3.1  ] [LT] width estimation of 1596 stripes took 9.961ms (mean=55 kbp; std=18 kbp)
2025-04-15 08:13:46.048746 [info     ] [chr4 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:46.053522 [info     ] [chr4 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:47.929414 [info     ] [chr4 ] [step 3.3  ] [LT] height estimation of 1596 stripes tool 1.876s (mean=869 kbp; std=326 kbp)
2025-04-15 08:13:47.929536 [info     ] [chr4 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:47.934772 [info     ] [chr4 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:47.943414 [info     ] [chr4 ] [step 3.1  ] [UT] width estimation of 1414 stripes took 8.596ms (mean=55 kbp; std=17 kbp)
2025-04-15 08:13:47.943519 [info     ] [chr4 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:47.947634 [info     ] [chr4 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:13:49.971745 [info     ] [chr4 ] [step 3.3  ] [UT] height estimation of 1414 stripes tool 2.024s (mean=881 kbp; std=342 kbp)
2025-04-15 08:13:49.971826 [info     ] [chr4 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:13:49.976082 [info     ] [chr4 ] [step 3    ] shape analysis took 3.938s
2025-04-15 08:13:49.976272 [info     ] [chr4 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:13:49.976373 [info     ] [chr4 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:13:50.324434 [info     ] [chr4 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:13:50.738461 [info     ] [chr4 ] [step 4    ] statistical analysis and post-processing took 762.181ms
2025-04-15 08:13:50.738707 [info     ] [chr4 ] [main      ] processing took 5.513s
2025-04-15 08:13:50.738801 [info     ] [chr4 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:50.805180 [info     ] [chr4 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 66.326ms
2025-04-15 08:13:50.805407 [info     ] [chr5 ] [main      ] begin processing
2025-04-15 08:13:50.805531 [info     ] [chr5 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:51.337168 [info     ] [chr5 ] [IO        ] fetched 5299698 pixels in 531.631ms
2025-04-15 08:13:51.337439 [info     ] [chr5 ] [step 1    ] data pre-processing
2025-04-15 08:13:51.337500 [info     ] [chr5 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:51.372763 [info     ] [chr5 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/5299698)
2025-04-15 08:13:51.372966 [info     ] [chr5 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:51.383520 [info     ] [chr5 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:51.387284 [info     ] [chr5 ] [step 1    ] preprocessing took 49.768ms
2025-04-15 08:13:51.387563 [info     ] [chr5 ] [step 2    ] topological data analysis
2025-04-15 08:13:51.387633 [info     ] [chr5 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:51.405264 [info     ] [chr5 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:51.405362 [info     ] [chr5 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:51.435763 [info     ] [chr5 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:51.436248 [info     ] [chr5 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:51.479349 [info     ] [chr5 ] [step 2.2.3] [LT] number of seed sites reduced from 1374 to 1360
2025-04-15 08:13:51.479498 [info     ] [chr5 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:51.480093 [info     ] [chr5 ] [step 2.3.1] [LT] identified 1360 candidate stripes
2025-04-15 08:13:51.480236 [info     ] [chr5 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:51.498393 [info     ] [chr5 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:51.498483 [info     ] [chr5 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:51.529065 [info     ] [chr5 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:51.529570 [info     ] [chr5 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:51.573954 [info     ] [chr5 ] [step 2.2.3] [UT] number of seed sites reduced from 1786 to 1770
2025-04-15 08:13:51.574092 [info     ] [chr5 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:51.575050 [info     ] [chr5 ] [step 2.3.1] [UT] identified 1770 candidate stripes
2025-04-15 08:13:51.575274 [info     ] [chr5 ] [step 2    ] topological data analysis took 187.658ms
2025-04-15 08:13:51.576084 [info     ] [chr5 ] [step 3    ] shape analysis
2025-04-15 08:13:51.576168 [info     ] [chr5 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:51.584220 [info     ] [chr5 ] [step 3.1  ] [LT] width estimation of 1360 stripes took 7.988ms (mean=56 kbp; std=18 kbp)
2025-04-15 08:13:51.584315 [info     ] [chr5 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:51.588070 [info     ] [chr5 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:53.110461 [info     ] [chr5 ] [step 3.3  ] [LT] height estimation of 1360 stripes tool 1.522s (mean=912 kbp; std=351 kbp)
2025-04-15 08:13:53.110566 [info     ] [chr5 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:53.114645 [info     ] [chr5 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:53.124833 [info     ] [chr5 ] [step 3.1  ] [UT] width estimation of 1770 stripes took 10.134ms (mean=52 kbp; std=19 kbp)
2025-04-15 08:13:53.124893 [info     ] [chr5 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:53.129658 [info     ] [chr5 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:13:55.746637 [info     ] [chr5 ] [step 3.3  ] [UT] height estimation of 1770 stripes tool 2.617s (mean=904 kbp; std=391 kbp)
2025-04-15 08:13:55.746746 [info     ] [chr5 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:13:55.752435 [info     ] [chr5 ] [step 3    ] shape analysis took 4.176s
2025-04-15 08:13:55.752600 [info     ] [chr5 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:13:55.752687 [info     ] [chr5 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:13:56.090018 [info     ] [chr5 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:13:56.679411 [info     ] [chr5 ] [step 4    ] statistical analysis and post-processing took 926.801ms
2025-04-15 08:13:56.679647 [info     ] [chr5 ] [main      ] processing took 5.874s
2025-04-15 08:13:56.679729 [info     ] [chr5 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:13:56.753872 [info     ] [chr5 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 74.102ms
2025-04-15 08:13:56.754115 [info     ] [chr6 ] [main      ] begin processing
2025-04-15 08:13:56.754253 [info     ] [chr6 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:13:57.250895 [info     ] [chr6 ] [IO        ] fetched 5079546 pixels in 496.635ms
2025-04-15 08:13:57.251136 [info     ] [chr6 ] [step 1    ] data pre-processing
2025-04-15 08:13:57.251197 [info     ] [chr6 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:13:57.284898 [info     ] [chr6 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/5079546)
2025-04-15 08:13:57.285100 [info     ] [chr6 ] [step 1.2  ] applying log-transformation
2025-04-15 08:13:57.294070 [info     ] [chr6 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:13:57.297369 [info     ] [chr6 ] [step 1    ] preprocessing took 46.164ms
2025-04-15 08:13:57.297600 [info     ] [chr6 ] [step 2    ] topological data analysis
2025-04-15 08:13:57.297672 [info     ] [chr6 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:13:57.314597 [info     ] [chr6 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:57.314722 [info     ] [chr6 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:13:57.343916 [info     ] [chr6 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:13:57.344406 [info     ] [chr6 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:13:57.385163 [info     ] [chr6 ] [step 2.2.3] [LT] number of seed sites reduced from 1497 to 1481
2025-04-15 08:13:57.385301 [info     ] [chr6 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:13:57.385882 [info     ] [chr6 ] [step 2.3.1] [LT] identified 1481 candidate stripes
2025-04-15 08:13:57.386060 [info     ] [chr6 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:13:57.403563 [info     ] [chr6 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:13:57.403664 [info     ] [chr6 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:13:57.432346 [info     ] [chr6 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:13:57.432778 [info     ] [chr6 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:13:57.474774 [info     ] [chr6 ] [step 2.2.3] [UT] number of seed sites reduced from 1386 to 1371
2025-04-15 08:13:57.474911 [info     ] [chr6 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:13:57.475710 [info     ] [chr6 ] [step 2.3.1] [UT] identified 1371 candidate stripes
2025-04-15 08:13:57.475891 [info     ] [chr6 ] [step 2    ] topological data analysis took 178.236ms
2025-04-15 08:13:57.477075 [info     ] [chr6 ] [step 3    ] shape analysis
2025-04-15 08:13:57.477191 [info     ] [chr6 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:13:57.488798 [info     ] [chr6 ] [step 3.1  ] [LT] width estimation of 1481 stripes took 11.556ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:13:57.488905 [info     ] [chr6 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:13:57.493810 [info     ] [chr6 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:13:59.334769 [info     ] [chr6 ] [step 3.3  ] [LT] height estimation of 1481 stripes tool 1.841s (mean=874 kbp; std=335 kbp)
2025-04-15 08:13:59.334877 [info     ] [chr6 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:13:59.339610 [info     ] [chr6 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:13:59.348234 [info     ] [chr6 ] [step 3.1  ] [UT] width estimation of 1371 stripes took 8.565ms (mean=56 kbp; std=18 kbp)
2025-04-15 08:13:59.348341 [info     ] [chr6 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:13:59.352468 [info     ] [chr6 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:01.592980 [info     ] [chr6 ] [step 3.3  ] [UT] height estimation of 1371 stripes tool 2.240s (mean=881 kbp; std=318 kbp)
2025-04-15 08:14:01.593090 [info     ] [chr6 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:01.597565 [info     ] [chr6 ] [step 3    ] shape analysis took 4.120s
2025-04-15 08:14:01.597766 [info     ] [chr6 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:01.597883 [info     ] [chr6 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:01.960262 [info     ] [chr6 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:02.412706 [info     ] [chr6 ] [step 4    ] statistical analysis and post-processing took 814.931ms
2025-04-15 08:14:02.412967 [info     ] [chr6 ] [main      ] processing took 5.659s
2025-04-15 08:14:02.413049 [info     ] [chr6 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:02.482371 [info     ] [chr6 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 69.278ms
2025-04-15 08:14:02.482623 [info     ] [chr7 ] [main      ] begin processing
2025-04-15 08:14:02.482758 [info     ] [chr7 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:02.922178 [info     ] [chr7 ] [IO        ] fetched 4526584 pixels in 439.413ms
2025-04-15 08:14:02.922411 [info     ] [chr7 ] [step 1    ] data pre-processing
2025-04-15 08:14:02.922462 [info     ] [chr7 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:02.948198 [info     ] [chr7 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/4526584)
2025-04-15 08:14:02.948403 [info     ] [chr7 ] [step 1.2  ] applying log-transformation
2025-04-15 08:14:02.956162 [info     ] [chr7 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:02.959087 [info     ] [chr7 ] [step 1    ] preprocessing took 36.609ms
2025-04-15 08:14:02.960305 [info     ] [chr7 ] [step 2    ] topological data analysis
2025-04-15 08:14:02.960370 [info     ] [chr7 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:02.976325 [info     ] [chr7 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:02.976417 [info     ] [chr7 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:03.002727 [info     ] [chr7 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:03.003149 [info     ] [chr7 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:03.041152 [info     ] [chr7 ] [step 2.2.3] [LT] number of seed sites reduced from 1075 to 1032
2025-04-15 08:14:03.041298 [info     ] [chr7 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:03.041712 [info     ] [chr7 ] [step 2.3.1] [LT] identified 1032 candidate stripes
2025-04-15 08:14:03.041882 [info     ] [chr7 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:03.058150 [info     ] [chr7 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:03.058262 [info     ] [chr7 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:03.084526 [info     ] [chr7 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:03.084977 [info     ] [chr7 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:03.123636 [info     ] [chr7 ] [step 2.2.3] [UT] number of seed sites reduced from 1507 to 1468
2025-04-15 08:14:03.123773 [info     ] [chr7 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:03.124607 [info     ] [chr7 ] [step 2.3.1] [UT] identified 1468 candidate stripes
2025-04-15 08:14:03.124790 [info     ] [chr7 ] [step 2    ] topological data analysis took 164.437ms
2025-04-15 08:14:03.125852 [info     ] [chr7 ] [step 3    ] shape analysis
2025-04-15 08:14:03.125954 [info     ] [chr7 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:03.134825 [info     ] [chr7 ] [step 3.1  ] [LT] width estimation of 1032 stripes took 8.815ms (mean=58 kbp; std=19 kbp)
2025-04-15 08:14:03.134931 [info     ] [chr7 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:03.138372 [info     ] [chr7 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:04.416911 [info     ] [chr7 ] [step 3.3  ] [LT] height estimation of 1032 stripes tool 1.278s (mean=915 kbp; std=434 kbp)
2025-04-15 08:14:04.417047 [info     ] [chr7 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:04.420417 [info     ] [chr7 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:04.429718 [info     ] [chr7 ] [step 3.1  ] [UT] width estimation of 1468 stripes took 9.244ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:04.429826 [info     ] [chr7 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:04.434377 [info     ] [chr7 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:06.810630 [info     ] [chr7 ] [step 3.3  ] [UT] height estimation of 1468 stripes tool 2.376s (mean=898 kbp; std=427 kbp)
2025-04-15 08:14:06.810738 [info     ] [chr7 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:06.815541 [info     ] [chr7 ] [step 3    ] shape analysis took 3.690s
2025-04-15 08:14:06.815739 [info     ] [chr7 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:06.815845 [info     ] [chr7 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:07.070715 [info     ] [chr7 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:07.552479 [info     ] [chr7 ] [step 4    ] statistical analysis and post-processing took 736.731ms
2025-04-15 08:14:07.552703 [info     ] [chr7 ] [main      ] processing took 5.070s
2025-04-15 08:14:07.552794 [info     ] [chr7 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:07.616970 [info     ] [chr7 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 64.115ms
2025-04-15 08:14:07.617198 [info     ] [chr8 ] [main      ] begin processing
2025-04-15 08:14:07.617321 [info     ] [chr8 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:08.047024 [info     ] [chr8 ] [IO        ] fetched 4287169 pixels in 429.696ms
2025-04-15 08:14:08.047276 [info     ] [chr8 ] [step 1    ] data pre-processing
2025-04-15 08:14:08.047331 [info     ] [chr8 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:08.074943 [info     ] [chr8 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/4287169)
2025-04-15 08:14:08.075153 [info     ] [chr8 ] [step 1.2  ] applying log-transformation
2025-04-15 08:14:08.082918 [info     ] [chr8 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:08.085832 [info     ] [chr8 ] [step 1    ] preprocessing took 38.490ms
2025-04-15 08:14:08.086070 [info     ] [chr8 ] [step 2    ] topological data analysis
2025-04-15 08:14:08.086134 [info     ] [chr8 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:08.100171 [info     ] [chr8 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:08.100273 [info     ] [chr8 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:08.124769 [info     ] [chr8 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:08.125221 [info     ] [chr8 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:08.159531 [info     ] [chr8 ] [step 2.2.3] [LT] number of seed sites reduced from 1319 to 1294
2025-04-15 08:14:08.159665 [info     ] [chr8 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:08.160171 [info     ] [chr8 ] [step 2.3.1] [LT] identified 1294 candidate stripes
2025-04-15 08:14:08.160357 [info     ] [chr8 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:08.175263 [info     ] [chr8 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:08.175359 [info     ] [chr8 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:08.199388 [info     ] [chr8 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:08.199796 [info     ] [chr8 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:08.235141 [info     ] [chr8 ] [step 2.2.3] [UT] number of seed sites reduced from 1128 to 1106
2025-04-15 08:14:08.235287 [info     ] [chr8 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:08.235929 [info     ] [chr8 ] [step 2.3.1] [UT] identified 1106 candidate stripes
2025-04-15 08:14:08.236099 [info     ] [chr8 ] [step 2    ] topological data analysis took 149.982ms
2025-04-15 08:14:08.237111 [info     ] [chr8 ] [step 3    ] shape analysis
2025-04-15 08:14:08.237207 [info     ] [chr8 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:08.247555 [info     ] [chr8 ] [step 3.1  ] [LT] width estimation of 1294 stripes took 10.283ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:08.247666 [info     ] [chr8 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:08.251919 [info     ] [chr8 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:09.845600 [info     ] [chr8 ] [step 3.3  ] [LT] height estimation of 1294 stripes tool 1.594s (mean=923 kbp; std=399 kbp)
2025-04-15 08:14:09.845705 [info     ] [chr8 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:09.849901 [info     ] [chr8 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:09.857032 [info     ] [chr8 ] [step 3.1  ] [UT] width estimation of 1106 stripes took 7.082ms (mean=55 kbp; std=18 kbp)
2025-04-15 08:14:09.857135 [info     ] [chr8 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:09.860607 [info     ] [chr8 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:11.664919 [info     ] [chr8 ] [step 3.3  ] [UT] height estimation of 1106 stripes tool 1.804s (mean=931 kbp; std=392 kbp)
2025-04-15 08:14:11.665012 [info     ] [chr8 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:11.668710 [info     ] [chr8 ] [step 3    ] shape analysis took 3.432s
2025-04-15 08:14:11.668891 [info     ] [chr8 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:11.668992 [info     ] [chr8 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:11.984183 [info     ] [chr8 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:12.352778 [info     ] [chr8 ] [step 4    ] statistical analysis and post-processing took 683.878ms
2025-04-15 08:14:12.352974 [info     ] [chr8 ] [main      ] processing took 4.736s
2025-04-15 08:14:12.353053 [info     ] [chr8 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:12.413212 [info     ] [chr8 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 60.115ms
2025-04-15 08:14:12.413449 [info     ] [chr9 ] [main      ] begin processing
2025-04-15 08:14:12.413588 [info     ] [chr9 ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:12.760690 [info     ] [chr9 ] [IO        ] fetched 3411928 pixels in 347.100ms
2025-04-15 08:14:12.760914 [info     ] [chr9 ] [step 1    ] data pre-processing
2025-04-15 08:14:12.760962 [info     ] [chr9 ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:12.779567 [info     ] [chr9 ] [step 1.1  ] removed 0.00% of the non-zero entries (0/3411928)
2025-04-15 08:14:12.779769 [info     ] [chr9 ] [step 1.2  ] applying log-transformation
2025-04-15 08:14:12.785843 [info     ] [chr9 ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:12.788323 [info     ] [chr9 ] [step 1    ] preprocessing took 27.351ms
2025-04-15 08:14:12.788583 [info     ] [chr9 ] [step 2    ] topological data analysis
2025-04-15 08:14:12.788648 [info     ] [chr9 ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:12.801395 [info     ] [chr9 ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:12.801501 [info     ] [chr9 ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:12.823060 [info     ] [chr9 ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:12.823474 [info     ] [chr9 ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:12.855739 [info     ] [chr9 ] [step 2.2.3] [LT] number of seed sites reduced from 836 to 816
2025-04-15 08:14:12.855873 [info     ] [chr9 ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:12.856224 [info     ] [chr9 ] [step 2.3.1] [LT] identified 816 candidate stripes
2025-04-15 08:14:12.856376 [info     ] [chr9 ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:12.869599 [info     ] [chr9 ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:12.869691 [info     ] [chr9 ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:12.891218 [info     ] [chr9 ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:12.891613 [info     ] [chr9 ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:12.924411 [info     ] [chr9 ] [step 2.2.3] [UT] number of seed sites reduced from 936 to 911
2025-04-15 08:14:12.924563 [info     ] [chr9 ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:12.924947 [info     ] [chr9 ] [step 2.3.1] [UT] identified 911 candidate stripes
2025-04-15 08:14:12.925114 [info     ] [chr9 ] [step 2    ] topological data analysis took 136.483ms
2025-04-15 08:14:12.926538 [info     ] [chr9 ] [step 3    ] shape analysis
2025-04-15 08:14:12.926651 [info     ] [chr9 ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:12.933756 [info     ] [chr9 ] [step 3.1  ] [LT] width estimation of 816 stripes took 7.056ms (mean=55 kbp; std=19 kbp)
2025-04-15 08:14:12.933853 [info     ] [chr9 ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:12.936509 [info     ] [chr9 ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:13.938302 [info     ] [chr9 ] [step 3.3  ] [LT] height estimation of 816 stripes tool 1.002s (mean=914 kbp; std=359 kbp)
2025-04-15 08:14:13.938408 [info     ] [chr9 ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:13.941152 [info     ] [chr9 ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:13.947065 [info     ] [chr9 ] [step 3.1  ] [UT] width estimation of 911 stripes took 5.862ms (mean=55 kbp; std=19 kbp)
2025-04-15 08:14:13.947162 [info     ] [chr9 ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:13.949961 [info     ] [chr9 ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:15.448193 [info     ] [chr9 ] [step 3.3  ] [UT] height estimation of 911 stripes tool 1.498s (mean=937 kbp; std=383 kbp)
2025-04-15 08:14:15.448300 [info     ] [chr9 ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:15.451461 [info     ] [chr9 ] [step 3    ] shape analysis took 2.525s
2025-04-15 08:14:15.451697 [info     ] [chr9 ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:15.451845 [info     ] [chr9 ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:15.651585 [info     ] [chr9 ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:15.959832 [info     ] [chr9 ] [step 4    ] statistical analysis and post-processing took 508.132ms
2025-04-15 08:14:15.960016 [info     ] [chr9 ] [main      ] processing took 3.547s
2025-04-15 08:14:15.960104 [info     ] [chr9 ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:16.012391 [info     ] [chr9 ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 52.219ms
2025-04-15 08:14:16.012632 [info     ] [chr10] [main      ] begin processing
2025-04-15 08:14:16.012765 [info     ] [chr10] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:16.400625 [info     ] [chr10] [IO        ] fetched 4037234 pixels in 387.854ms
2025-04-15 08:14:16.400886 [info     ] [chr10] [step 1    ] data pre-processing
2025-04-15 08:14:16.400941 [info     ] [chr10] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:16.427162 [info     ] [chr10] [step 1.1  ] removed 0.00% of the non-zero entries (0/4037234)
2025-04-15 08:14:16.427339 [info     ] [chr10] [step 1.2  ] applying log-transformation
2025-04-15 08:14:16.434054 [info     ] [chr10] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:16.436819 [info     ] [chr10] [step 1    ] preprocessing took 35.864ms
2025-04-15 08:14:16.440215 [info     ] [chr10] [step 2    ] topological data analysis
2025-04-15 08:14:16.440298 [info     ] [chr10] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:16.453620 [info     ] [chr10] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:16.453716 [info     ] [chr10] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:16.475650 [info     ] [chr10] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:16.476086 [info     ] [chr10] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:16.507658 [info     ] [chr10] [step 2.2.3] [LT] number of seed sites reduced from 1140 to 1107
2025-04-15 08:14:16.507795 [info     ] [chr10] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:16.508231 [info     ] [chr10] [step 2.3.1] [LT] identified 1107 candidate stripes
2025-04-15 08:14:16.508386 [info     ] [chr10] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:16.522052 [info     ] [chr10] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:16.522160 [info     ] [chr10] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:16.544341 [info     ] [chr10] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:16.544767 [info     ] [chr10] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:16.577537 [info     ] [chr10] [step 2.2.3] [UT] number of seed sites reduced from 1187 to 1160
2025-04-15 08:14:16.577696 [info     ] [chr10] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:16.578637 [info     ] [chr10] [step 2.3.1] [UT] identified 1160 candidate stripes
2025-04-15 08:14:16.578806 [info     ] [chr10] [step 2    ] topological data analysis took 138.525ms
2025-04-15 08:14:16.579467 [info     ] [chr10] [step 3    ] shape analysis
2025-04-15 08:14:16.579575 [info     ] [chr10] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:16.588196 [info     ] [chr10] [step 3.1  ] [LT] width estimation of 1107 stripes took 8.578ms (mean=56 kbp; std=20 kbp)
2025-04-15 08:14:16.588305 [info     ] [chr10] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:16.591979 [info     ] [chr10] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:17.956108 [info     ] [chr10] [step 3.3  ] [LT] height estimation of 1107 stripes tool 1.364s (mean=934 kbp; std=411 kbp)
2025-04-15 08:14:17.956226 [info     ] [chr10] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:17.959808 [info     ] [chr10] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:17.967256 [info     ] [chr10] [step 3.1  ] [UT] width estimation of 1160 stripes took 7.392ms (mean=54 kbp; std=19 kbp)
2025-04-15 08:14:17.967358 [info     ] [chr10] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:17.970915 [info     ] [chr10] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:19.869646 [info     ] [chr10] [step 3.3  ] [UT] height estimation of 1160 stripes tool 1.899s (mean=949 kbp; std=434 kbp)
2025-04-15 08:14:19.869726 [info     ] [chr10] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:19.873584 [info     ] [chr10] [step 3    ] shape analysis took 3.294s
2025-04-15 08:14:19.873822 [info     ] [chr10] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:19.873941 [info     ] [chr10] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:20.146719 [info     ] [chr10] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:20.541675 [info     ] [chr10] [step 4    ] statistical analysis and post-processing took 667.837ms
2025-04-15 08:14:20.541978 [info     ] [chr10] [main      ] processing took 4.529s
2025-04-15 08:14:20.542081 [info     ] [chr10] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:20.603732 [info     ] [chr10] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 61.596ms
2025-04-15 08:14:20.604034 [info     ] [chr11] [main      ] begin processing
2025-04-15 08:14:20.604218 [info     ] [chr11] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:21.043404 [info     ] [chr11] [IO        ] fetched 4030066 pixels in 439.185ms
2025-04-15 08:14:21.043684 [info     ] [chr11] [step 1    ] data pre-processing
2025-04-15 08:14:21.043806 [info     ] [chr11] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:21.072920 [info     ] [chr11] [step 1.1  ] removed 0.00% of the non-zero entries (0/4030066)
2025-04-15 08:14:21.073127 [info     ] [chr11] [step 1.2  ] applying log-transformation
2025-04-15 08:14:21.080115 [info     ] [chr11] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:21.082817 [info     ] [chr11] [step 1    ] preprocessing took 38.999ms
2025-04-15 08:14:21.083040 [info     ] [chr11] [step 2    ] topological data analysis
2025-04-15 08:14:21.083094 [info     ] [chr11] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:21.096293 [info     ] [chr11] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:21.096395 [info     ] [chr11] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:21.118433 [info     ] [chr11] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:21.119177 [info     ] [chr11] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:21.151370 [info     ] [chr11] [step 2.2.3] [LT] number of seed sites reduced from 1230 to 1217
2025-04-15 08:14:21.151611 [info     ] [chr11] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:21.152153 [info     ] [chr11] [step 2.3.1] [LT] identified 1217 candidate stripes
2025-04-15 08:14:21.152335 [info     ] [chr11] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:21.166021 [info     ] [chr11] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:21.166133 [info     ] [chr11] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:21.188361 [info     ] [chr11] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:21.188833 [info     ] [chr11] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:21.221443 [info     ] [chr11] [step 2.2.3] [UT] number of seed sites reduced from 1360 to 1348
2025-04-15 08:14:21.221589 [info     ] [chr11] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:21.257633 [info     ] [chr11] [step 2.3.1] [UT] identified 1348 candidate stripes
2025-04-15 08:14:21.257957 [info     ] [chr11] [step 2    ] topological data analysis took 174.878ms
2025-04-15 08:14:21.258884 [info     ] [chr11] [step 3    ] shape analysis
2025-04-15 08:14:21.259016 [info     ] [chr11] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:21.269060 [info     ] [chr11] [step 3.1  ] [LT] width estimation of 1217 stripes took 9.964ms (mean=55 kbp; std=19 kbp)
2025-04-15 08:14:21.269178 [info     ] [chr11] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:21.273348 [info     ] [chr11] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:22.768703 [info     ] [chr11] [step 3.3  ] [LT] height estimation of 1217 stripes tool 1.495s (mean=849 kbp; std=320 kbp)
2025-04-15 08:14:22.768777 [info     ] [chr11] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:22.772718 [info     ] [chr11] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:22.781230 [info     ] [chr11] [step 3.1  ] [UT] width estimation of 1348 stripes took 8.465ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:22.781305 [info     ] [chr11] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:22.785413 [info     ] [chr11] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:24.734771 [info     ] [chr11] [step 3.3  ] [UT] height estimation of 1348 stripes tool 1.949s (mean=838 kbp; std=341 kbp)
2025-04-15 08:14:24.734884 [info     ] [chr11] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:24.738956 [info     ] [chr11] [step 3    ] shape analysis took 3.480s
2025-04-15 08:14:24.739108 [info     ] [chr11] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:24.739210 [info     ] [chr11] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:25.010017 [info     ] [chr11] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:25.398165 [info     ] [chr11] [step 4    ] statistical analysis and post-processing took 659.044ms
2025-04-15 08:14:25.398445 [info     ] [chr11] [main      ] processing took 4.794s
2025-04-15 08:14:25.398566 [info     ] [chr11] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:25.459578 [info     ] [chr11] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 60.951ms
2025-04-15 08:14:25.459814 [info     ] [chr12] [main      ] begin processing
2025-04-15 08:14:25.459952 [info     ] [chr12] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:25.842265 [info     ] [chr12] [IO        ] fetched 3936601 pixels in 382.306ms
2025-04-15 08:14:25.842512 [info     ] [chr12] [step 1    ] data pre-processing
2025-04-15 08:14:25.842623 [info     ] [chr12] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:25.865683 [info     ] [chr12] [step 1.1  ] removed 0.00% of the non-zero entries (0/3936601)
2025-04-15 08:14:25.865885 [info     ] [chr12] [step 1.2  ] applying log-transformation
2025-04-15 08:14:25.872754 [info     ] [chr12] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:25.875339 [info     ] [chr12] [step 1    ] preprocessing took 32.709ms
2025-04-15 08:14:25.875554 [info     ] [chr12] [step 2    ] topological data analysis
2025-04-15 08:14:25.875616 [info     ] [chr12] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:25.889506 [info     ] [chr12] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:25.889621 [info     ] [chr12] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:25.913388 [info     ] [chr12] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:25.913846 [info     ] [chr12] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:25.948492 [info     ] [chr12] [step 2.2.3] [LT] number of seed sites reduced from 919 to 913
2025-04-15 08:14:25.948728 [info     ] [chr12] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:25.949167 [info     ] [chr12] [step 2.3.1] [LT] identified 913 candidate stripes
2025-04-15 08:14:25.949286 [info     ] [chr12] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:25.963913 [info     ] [chr12] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:25.964025 [info     ] [chr12] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:25.987387 [info     ] [chr12] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:25.987838 [info     ] [chr12] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:26.022231 [info     ] [chr12] [step 2.2.3] [UT] number of seed sites reduced from 1207 to 1200
2025-04-15 08:14:26.022359 [info     ] [chr12] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:26.022984 [info     ] [chr12] [step 2.3.1] [UT] identified 1200 candidate stripes
2025-04-15 08:14:26.023137 [info     ] [chr12] [step 2    ] topological data analysis took 147.540ms
2025-04-15 08:14:26.023915 [info     ] [chr12] [step 3    ] shape analysis
2025-04-15 08:14:26.023997 [info     ] [chr12] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:26.029768 [info     ] [chr12] [step 3.1  ] [LT] width estimation of 913 stripes took 5.703ms (mean=58 kbp; std=19 kbp)
2025-04-15 08:14:26.029830 [info     ] [chr12] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:26.032500 [info     ] [chr12] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:27.066994 [info     ] [chr12] [step 3.3  ] [LT] height estimation of 913 stripes tool 1.034s (mean=877 kbp; std=330 kbp)
2025-04-15 08:14:27.067085 [info     ] [chr12] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:27.069943 [info     ] [chr12] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:27.077272 [info     ] [chr12] [step 3.1  ] [UT] width estimation of 1200 stripes took 7.276ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:27.077332 [info     ] [chr12] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:27.080924 [info     ] [chr12] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:28.853861 [info     ] [chr12] [step 3.3  ] [UT] height estimation of 1200 stripes tool 1.773s (mean=862 kbp; std=344 kbp)
2025-04-15 08:14:28.853963 [info     ] [chr12] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:28.857620 [info     ] [chr12] [step 3    ] shape analysis took 2.834s
2025-04-15 08:14:28.857768 [info     ] [chr12] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:28.857862 [info     ] [chr12] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:29.066203 [info     ] [chr12] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:29.451967 [info     ] [chr12] [step 4    ] statistical analysis and post-processing took 594.190ms
2025-04-15 08:14:29.452241 [info     ] [chr12] [main      ] processing took 3.992s
2025-04-15 08:14:29.452329 [info     ] [chr12] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:29.507510 [info     ] [chr12] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 55.111ms
2025-04-15 08:14:29.507794 [info     ] [chr13] [main      ] begin processing
2025-04-15 08:14:29.507965 [info     ] [chr13] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:29.792637 [info     ] [chr13] [IO        ] fetched 2844903 pixels in 284.665ms
2025-04-15 08:14:29.792921 [info     ] [chr13] [step 1    ] data pre-processing
2025-04-15 08:14:29.793044 [info     ] [chr13] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:29.808997 [info     ] [chr13] [step 1.1  ] removed 0.00% of the non-zero entries (0/2844903)
2025-04-15 08:14:29.809211 [info     ] [chr13] [step 1.2  ] applying log-transformation
2025-04-15 08:14:29.814346 [info     ] [chr13] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:29.816397 [info     ] [chr13] [step 1    ] preprocessing took 23.343ms
2025-04-15 08:14:29.817916 [info     ] [chr13] [step 2    ] topological data analysis
2025-04-15 08:14:29.817975 [info     ] [chr13] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:29.828204 [info     ] [chr13] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:29.828301 [info     ] [chr13] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:29.846542 [info     ] [chr13] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:29.846973 [info     ] [chr13] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:29.873977 [info     ] [chr13] [step 2.2.3] [LT] number of seed sites reduced from 928 to 916
2025-04-15 08:14:29.874134 [info     ] [chr13] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:29.874573 [info     ] [chr13] [step 2.3.1] [LT] identified 916 candidate stripes
2025-04-15 08:14:29.874765 [info     ] [chr13] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:29.885527 [info     ] [chr13] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:29.885626 [info     ] [chr13] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:29.903632 [info     ] [chr13] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:29.904002 [info     ] [chr13] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:29.931089 [info     ] [chr13] [step 2.2.3] [UT] number of seed sites reduced from 903 to 891
2025-04-15 08:14:29.931222 [info     ] [chr13] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:29.931616 [info     ] [chr13] [step 2.3.1] [UT] identified 891 candidate stripes
2025-04-15 08:14:29.931767 [info     ] [chr13] [step 2    ] topological data analysis took 113.808ms
2025-04-15 08:14:29.932624 [info     ] [chr13] [step 3    ] shape analysis
2025-04-15 08:14:29.932739 [info     ] [chr13] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:29.940294 [info     ] [chr13] [step 3.1  ] [LT] width estimation of 916 stripes took 7.505ms (mean=53 kbp; std=18 kbp)
2025-04-15 08:14:29.940402 [info     ] [chr13] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:29.943387 [info     ] [chr13] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:31.067895 [info     ] [chr13] [step 3.3  ] [LT] height estimation of 916 stripes tool 1.124s (mean=881 kbp; std=319 kbp)
2025-04-15 08:14:31.067965 [info     ] [chr13] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:31.071631 [info     ] [chr13] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:31.077368 [info     ] [chr13] [step 3.1  ] [UT] width estimation of 891 stripes took 5.691ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:31.077434 [info     ] [chr13] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:31.080241 [info     ] [chr13] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:32.528653 [info     ] [chr13] [step 3.3  ] [UT] height estimation of 891 stripes tool 1.448s (mean=897 kbp; std=338 kbp)
2025-04-15 08:14:32.528766 [info     ] [chr13] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:32.531831 [info     ] [chr13] [step 3    ] shape analysis took 2.599s
2025-04-15 08:14:32.532019 [info     ] [chr13] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:32.532110 [info     ] [chr13] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:32.753448 [info     ] [chr13] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:33.046281 [info     ] [chr13] [step 4    ] statistical analysis and post-processing took 514.258ms
2025-04-15 08:14:33.046475 [info     ] [chr13] [main      ] processing took 3.539s
2025-04-15 08:14:33.046620 [info     ] [chr13] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:33.099098 [info     ] [chr13] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 52.417ms
2025-04-15 08:14:33.099370 [info     ] [chr14] [main      ] begin processing
2025-04-15 08:14:33.099518 [info     ] [chr14] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:33.377764 [info     ] [chr14] [IO        ] fetched 2649697 pixels in 278.248ms
2025-04-15 08:14:33.377949 [info     ] [chr14] [step 1    ] data pre-processing
2025-04-15 08:14:33.378029 [info     ] [chr14] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:33.392369 [info     ] [chr14] [step 1.1  ] removed 0.00% of the non-zero entries (0/2649697)
2025-04-15 08:14:33.392554 [info     ] [chr14] [step 1.2  ] applying log-transformation
2025-04-15 08:14:33.398030 [info     ] [chr14] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:33.399452 [info     ] [chr14] [step 1    ] preprocessing took 21.420ms
2025-04-15 08:14:33.399627 [info     ] [chr14] [step 2    ] topological data analysis
2025-04-15 08:14:33.399685 [info     ] [chr14] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:33.410235 [info     ] [chr14] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:33.410337 [info     ] [chr14] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:33.428773 [info     ] [chr14] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:33.429200 [info     ] [chr14] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:33.455889 [info     ] [chr14] [step 2.2.3] [LT] number of seed sites reduced from 767 to 758
2025-04-15 08:14:33.456032 [info     ] [chr14] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:33.456409 [info     ] [chr14] [step 2.3.1] [LT] identified 758 candidate stripes
2025-04-15 08:14:33.456567 [info     ] [chr14] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:33.467636 [info     ] [chr14] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:33.467737 [info     ] [chr14] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:33.486144 [info     ] [chr14] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:33.486529 [info     ] [chr14] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:33.514336 [info     ] [chr14] [step 2.2.3] [UT] number of seed sites reduced from 711 to 708
2025-04-15 08:14:33.514470 [info     ] [chr14] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:33.514951 [info     ] [chr14] [step 2.3.1] [UT] identified 708 candidate stripes
2025-04-15 08:14:33.515141 [info     ] [chr14] [step 2    ] topological data analysis took 115.472ms
2025-04-15 08:14:33.515872 [info     ] [chr14] [step 3    ] shape analysis
2025-04-15 08:14:33.515973 [info     ] [chr14] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:33.521148 [info     ] [chr14] [step 3.1  ] [LT] width estimation of 758 stripes took 5.109ms (mean=56 kbp; std=18 kbp)
2025-04-15 08:14:33.521249 [info     ] [chr14] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:33.523655 [info     ] [chr14] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:34.448006 [info     ] [chr14] [step 3.3  ] [LT] height estimation of 758 stripes tool 924.290ms (mean=875 kbp; std=311 kbp)
2025-04-15 08:14:34.448107 [info     ] [chr14] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:34.450639 [info     ] [chr14] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:34.455366 [info     ] [chr14] [step 3.1  ] [UT] width estimation of 708 stripes took 4.681ms (mean=57 kbp; std=18 kbp)
2025-04-15 08:14:34.455471 [info     ] [chr14] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:34.457710 [info     ] [chr14] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:35.616464 [info     ] [chr14] [step 3.3  ] [UT] height estimation of 708 stripes tool 1.159s (mean=897 kbp; std=332 kbp)
2025-04-15 08:14:35.616613 [info     ] [chr14] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:35.619063 [info     ] [chr14] [step 3    ] shape analysis took 2.103s
2025-04-15 08:14:35.619215 [info     ] [chr14] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:35.619296 [info     ] [chr14] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:35.801858 [info     ] [chr14] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:36.033543 [info     ] [chr14] [step 4    ] statistical analysis and post-processing took 414.323ms
2025-04-15 08:14:36.033743 [info     ] [chr14] [main      ] processing took 2.934s
2025-04-15 08:14:36.033817 [info     ] [chr14] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:36.080432 [info     ] [chr14] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 46.563ms
2025-04-15 08:14:36.080657 [info     ] [chr15] [main      ] begin processing
2025-04-15 08:14:36.080784 [info     ] [chr15] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:36.332330 [info     ] [chr15] [IO        ] fetched 2384015 pixels in 251.547ms
2025-04-15 08:14:36.332486 [info     ] [chr15] [step 1    ] data pre-processing
2025-04-15 08:14:36.332569 [info     ] [chr15] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:36.345146 [info     ] [chr15] [step 1.1  ] removed 0.00% of the non-zero entries (0/2384015)
2025-04-15 08:14:36.345275 [info     ] [chr15] [step 1.2  ] applying log-transformation
2025-04-15 08:14:36.350001 [info     ] [chr15] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:36.351348 [info     ] [chr15] [step 1    ] preprocessing took 18.777ms
2025-04-15 08:14:36.353884 [info     ] [chr15] [step 2    ] topological data analysis
2025-04-15 08:14:36.353967 [info     ] [chr15] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:36.363727 [info     ] [chr15] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:36.363803 [info     ] [chr15] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:36.381397 [info     ] [chr15] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:36.381810 [info     ] [chr15] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:36.407167 [info     ] [chr15] [step 2.2.3] [LT] number of seed sites reduced from 829 to 791
2025-04-15 08:14:36.407286 [info     ] [chr15] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:36.407680 [info     ] [chr15] [step 2.3.1] [LT] identified 791 candidate stripes
2025-04-15 08:14:36.407810 [info     ] [chr15] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:36.418193 [info     ] [chr15] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:36.418281 [info     ] [chr15] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:36.435720 [info     ] [chr15] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:36.436073 [info     ] [chr15] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:36.462273 [info     ] [chr15] [step 2.2.3] [UT] number of seed sites reduced from 819 to 787
2025-04-15 08:14:36.462387 [info     ] [chr15] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:36.462758 [info     ] [chr15] [step 2.3.1] [UT] identified 787 candidate stripes
2025-04-15 08:14:36.462891 [info     ] [chr15] [step 2    ] topological data analysis took 108.940ms
2025-04-15 08:14:36.463474 [info     ] [chr15] [step 3    ] shape analysis
2025-04-15 08:14:36.463584 [info     ] [chr15] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:36.468831 [info     ] [chr15] [step 3.1  ] [LT] width estimation of 791 stripes took 5.181ms (mean=54 kbp; std=19 kbp)
2025-04-15 08:14:36.468897 [info     ] [chr15] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:36.471362 [info     ] [chr15] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:37.433214 [info     ] [chr15] [step 3.3  ] [LT] height estimation of 791 stripes tool 961.793ms (mean=879 kbp; std=458 kbp)
2025-04-15 08:14:37.433311 [info     ] [chr15] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:37.435974 [info     ] [chr15] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:37.441122 [info     ] [chr15] [step 3.1  ] [UT] width estimation of 787 stripes took 5.102ms (mean=54 kbp; std=19 kbp)
2025-04-15 08:14:37.441217 [info     ] [chr15] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:37.443649 [info     ] [chr15] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:38.738170 [info     ] [chr15] [step 3.3  ] [UT] height estimation of 787 stripes tool 1.294s (mean=870 kbp; std=425 kbp)
2025-04-15 08:14:38.738243 [info     ] [chr15] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:38.741009 [info     ] [chr15] [step 3    ] shape analysis took 2.277s
2025-04-15 08:14:38.741168 [info     ] [chr15] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:38.741257 [info     ] [chr15] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:38.933375 [info     ] [chr15] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:39.188200 [info     ] [chr15] [step 4    ] statistical analysis and post-processing took 447.030ms
2025-04-15 08:14:39.188356 [info     ] [chr15] [main      ] processing took 3.108s
2025-04-15 08:14:39.188430 [info     ] [chr15] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:39.237366 [info     ] [chr15] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 48.894ms
2025-04-15 08:14:39.237561 [info     ] [chr16] [main      ] begin processing
2025-04-15 08:14:39.237684 [info     ] [chr16] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:39.514294 [info     ] [chr16] [IO        ] fetched 2285929 pixels in 276.604ms
2025-04-15 08:14:39.514474 [info     ] [chr16] [step 1    ] data pre-processing
2025-04-15 08:14:39.514571 [info     ] [chr16] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:39.532216 [info     ] [chr16] [step 1.1  ] removed 0.00% of the non-zero entries (0/2285929)
2025-04-15 08:14:39.532340 [info     ] [chr16] [step 1.2  ] applying log-transformation
2025-04-15 08:14:39.536816 [info     ] [chr16] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:39.538047 [info     ] [chr16] [step 1    ] preprocessing took 23.479ms
2025-04-15 08:14:39.538224 [info     ] [chr16] [step 2    ] topological data analysis
2025-04-15 08:14:39.538280 [info     ] [chr16] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:39.547386 [info     ] [chr16] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:39.547485 [info     ] [chr16] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:39.563990 [info     ] [chr16] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:39.564404 [info     ] [chr16] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:39.587838 [info     ] [chr16] [step 2.2.3] [LT] number of seed sites reduced from 472 to 437
2025-04-15 08:14:39.587985 [info     ] [chr16] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:39.588237 [info     ] [chr16] [step 2.3.1] [LT] identified 437 candidate stripes
2025-04-15 08:14:39.588377 [info     ] [chr16] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:39.598077 [info     ] [chr16] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:39.598176 [info     ] [chr16] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:39.614513 [info     ] [chr16] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:39.614878 [info     ] [chr16] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:39.638913 [info     ] [chr16] [step 2.2.3] [UT] number of seed sites reduced from 679 to 631
2025-04-15 08:14:39.639049 [info     ] [chr16] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:39.639380 [info     ] [chr16] [step 2.3.1] [UT] identified 631 candidate stripes
2025-04-15 08:14:39.639552 [info     ] [chr16] [step 2    ] topological data analysis took 101.286ms
2025-04-15 08:14:39.640163 [info     ] [chr16] [step 3    ] shape analysis
2025-04-15 08:14:39.640260 [info     ] [chr16] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:39.643539 [info     ] [chr16] [step 3.1  ] [LT] width estimation of 437 stripes took 3.211ms (mean=60 kbp; std=19 kbp)
2025-04-15 08:14:39.643630 [info     ] [chr16] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:39.645091 [info     ] [chr16] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:40.179417 [info     ] [chr16] [step 3.3  ] [LT] height estimation of 437 stripes tool 534.271ms (mean=950 kbp; std=458 kbp)
2025-04-15 08:14:40.179571 [info     ] [chr16] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:40.181611 [info     ] [chr16] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:40.185913 [info     ] [chr16] [step 3.1  ] [UT] width estimation of 631 stripes took 4.255ms (mean=57 kbp; std=20 kbp)
2025-04-15 08:14:40.186012 [info     ] [chr16] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:40.187997 [info     ] [chr16] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:41.194228 [info     ] [chr16] [step 3.3  ] [UT] height estimation of 631 stripes tool 1.006s (mean=1028 kbp; std=623 kbp)
2025-04-15 08:14:41.194332 [info     ] [chr16] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:41.196621 [info     ] [chr16] [step 3    ] shape analysis took 1.556s
2025-04-15 08:14:41.196772 [info     ] [chr16] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:41.196857 [info     ] [chr16] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:41.303697 [info     ] [chr16] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:41.519550 [info     ] [chr16] [step 4    ] statistical analysis and post-processing took 322.777ms
2025-04-15 08:14:41.519767 [info     ] [chr16] [main      ] processing took 2.282s
2025-04-15 08:14:41.519830 [info     ] [chr16] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:41.559336 [info     ] [chr16] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 39.467ms
2025-04-15 08:14:41.559525 [info     ] [chr17] [main      ] begin processing
2025-04-15 08:14:41.559650 [info     ] [chr17] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:41.810147 [info     ] [chr17] [IO        ] fetched 2377674 pixels in 250.497ms
2025-04-15 08:14:41.810297 [info     ] [chr17] [step 1    ] data pre-processing
2025-04-15 08:14:41.810362 [info     ] [chr17] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:41.823271 [info     ] [chr17] [step 1.1  ] removed 0.00% of the non-zero entries (0/2377674)
2025-04-15 08:14:41.823412 [info     ] [chr17] [step 1.2  ] applying log-transformation
2025-04-15 08:14:41.828009 [info     ] [chr17] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:41.829263 [info     ] [chr17] [step 1    ] preprocessing took 18.899ms
2025-04-15 08:14:41.829443 [info     ] [chr17] [step 2    ] topological data analysis
2025-04-15 08:14:41.829506 [info     ] [chr17] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:41.838224 [info     ] [chr17] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:41.838331 [info     ] [chr17] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:41.853795 [info     ] [chr17] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:41.854207 [info     ] [chr17] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:41.875869 [info     ] [chr17] [step 2.2.3] [LT] number of seed sites reduced from 736 to 707
2025-04-15 08:14:41.876013 [info     ] [chr17] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:41.876366 [info     ] [chr17] [step 2.3.1] [LT] identified 707 candidate stripes
2025-04-15 08:14:41.876501 [info     ] [chr17] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:41.886073 [info     ] [chr17] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:41.886180 [info     ] [chr17] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:41.901281 [info     ] [chr17] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:41.901663 [info     ] [chr17] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:41.924013 [info     ] [chr17] [step 2.2.3] [UT] number of seed sites reduced from 728 to 694
2025-04-15 08:14:41.924154 [info     ] [chr17] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:41.924504 [info     ] [chr17] [step 2.3.1] [UT] identified 694 candidate stripes
2025-04-15 08:14:41.924640 [info     ] [chr17] [step 2    ] topological data analysis took 95.156ms
2025-04-15 08:14:41.925117 [info     ] [chr17] [step 3    ] shape analysis
2025-04-15 08:14:41.925226 [info     ] [chr17] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:41.930059 [info     ] [chr17] [step 3.1  ] [LT] width estimation of 707 stripes took 4.774ms (mean=55 kbp; std=20 kbp)
2025-04-15 08:14:41.930129 [info     ] [chr17] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:41.932364 [info     ] [chr17] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:42.791802 [info     ] [chr17] [step 3.3  ] [LT] height estimation of 707 stripes tool 859.377ms (mean=925 kbp; std=556 kbp)
2025-04-15 08:14:42.791910 [info     ] [chr17] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:42.794327 [info     ] [chr17] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:42.798934 [info     ] [chr17] [step 3.1  ] [UT] width estimation of 694 stripes took 4.563ms (mean=57 kbp; std=20 kbp)
2025-04-15 08:14:42.799039 [info     ] [chr17] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:42.801198 [info     ] [chr17] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:43.925410 [info     ] [chr17] [step 3.3  ] [UT] height estimation of 694 stripes tool 1.124s (mean=912 kbp; std=508 kbp)
2025-04-15 08:14:43.925577 [info     ] [chr17] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:43.928292 [info     ] [chr17] [step 3    ] shape analysis took 2.003s
2025-04-15 08:14:43.928453 [info     ] [chr17] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:43.928545 [info     ] [chr17] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:44.103235 [info     ] [chr17] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:44.329905 [info     ] [chr17] [step 4    ] statistical analysis and post-processing took 401.450ms
2025-04-15 08:14:44.330094 [info     ] [chr17] [main      ] processing took 2.771s
2025-04-15 08:14:44.330168 [info     ] [chr17] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:44.376945 [info     ] [chr17] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 46.731ms
2025-04-15 08:14:44.377137 [info     ] [chr18] [main      ] begin processing
2025-04-15 08:14:44.377282 [info     ] [chr18] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:44.615211 [info     ] [chr18] [IO        ] fetched 2243369 pixels in 237.925ms
2025-04-15 08:14:44.615351 [info     ] [chr18] [step 1    ] data pre-processing
2025-04-15 08:14:44.615442 [info     ] [chr18] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:44.626272 [info     ] [chr18] [step 1.1  ] removed 0.00% of the non-zero entries (0/2243369)
2025-04-15 08:14:44.626411 [info     ] [chr18] [step 1.2  ] applying log-transformation
2025-04-15 08:14:44.630942 [info     ] [chr18] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:44.632155 [info     ] [chr18] [step 1    ] preprocessing took 16.711ms
2025-04-15 08:14:44.632334 [info     ] [chr18] [step 2    ] topological data analysis
2025-04-15 08:14:44.632388 [info     ] [chr18] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:44.640845 [info     ] [chr18] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:44.640945 [info     ] [chr18] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:44.656499 [info     ] [chr18] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:44.656929 [info     ] [chr18] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:44.677910 [info     ] [chr18] [step 2.2.3] [LT] number of seed sites reduced from 682 to 673
2025-04-15 08:14:44.678064 [info     ] [chr18] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:44.678434 [info     ] [chr18] [step 2.3.1] [LT] identified 673 candidate stripes
2025-04-15 08:14:44.678610 [info     ] [chr18] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:44.687673 [info     ] [chr18] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:44.687778 [info     ] [chr18] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:44.702985 [info     ] [chr18] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:44.703370 [info     ] [chr18] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:44.724901 [info     ] [chr18] [step 2.2.3] [UT] number of seed sites reduced from 726 to 719
2025-04-15 08:14:44.725047 [info     ] [chr18] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:44.725408 [info     ] [chr18] [step 2.3.1] [UT] identified 719 candidate stripes
2025-04-15 08:14:44.725601 [info     ] [chr18] [step 2    ] topological data analysis took 93.226ms
2025-04-15 08:14:44.726252 [info     ] [chr18] [step 3    ] shape analysis
2025-04-15 08:14:44.726349 [info     ] [chr18] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:44.730879 [info     ] [chr18] [step 3.1  ] [LT] width estimation of 673 stripes took 4.465ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:44.730951 [info     ] [chr18] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:44.733056 [info     ] [chr18] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:45.547706 [info     ] [chr18] [step 3.3  ] [LT] height estimation of 673 stripes tool 814.592ms (mean=922 kbp; std=320 kbp)
2025-04-15 08:14:45.547811 [info     ] [chr18] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:45.550117 [info     ] [chr18] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:45.554870 [info     ] [chr18] [step 3.1  ] [UT] width estimation of 719 stripes took 4.708ms (mean=52 kbp; std=17 kbp)
2025-04-15 08:14:45.554977 [info     ] [chr18] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:45.557229 [info     ] [chr18] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:46.678052 [info     ] [chr18] [step 3.3  ] [UT] height estimation of 719 stripes tool 1.121s (mean=918 kbp; std=350 kbp)
2025-04-15 08:14:46.678159 [info     ] [chr18] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:46.680695 [info     ] [chr18] [step 3    ] shape analysis took 1.954s
2025-04-15 08:14:46.680869 [info     ] [chr18] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:46.680947 [info     ] [chr18] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:46.840340 [info     ] [chr18] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:47.057074 [info     ] [chr18] [step 4    ] statistical analysis and post-processing took 376.197ms
2025-04-15 08:14:47.057312 [info     ] [chr18] [main      ] processing took 2.680s
2025-04-15 08:14:47.057418 [info     ] [chr18] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:47.104248 [info     ] [chr18] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 46.767ms
2025-04-15 08:14:47.104430 [info     ] [chr19] [main      ] begin processing
2025-04-15 08:14:47.104605 [info     ] [chr19] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:47.290807 [info     ] [chr19] [IO        ] fetched 1766605 pixels in 186.204ms
2025-04-15 08:14:47.290918 [info     ] [chr19] [step 1    ] data pre-processing
2025-04-15 08:14:47.290976 [info     ] [chr19] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:47.299586 [info     ] [chr19] [step 1.1  ] removed 0.00% of the non-zero entries (0/1766605)
2025-04-15 08:14:47.299675 [info     ] [chr19] [step 1.2  ] applying log-transformation
2025-04-15 08:14:47.303300 [info     ] [chr19] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:47.304324 [info     ] [chr19] [step 1    ] preprocessing took 13.347ms
2025-04-15 08:14:47.306516 [info     ] [chr19] [step 2    ] topological data analysis
2025-04-15 08:14:47.306589 [info     ] [chr19] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:47.312669 [info     ] [chr19] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:47.312745 [info     ] [chr19] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:47.324084 [info     ] [chr19] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:47.324482 [info     ] [chr19] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:47.339728 [info     ] [chr19] [step 2.2.3] [LT] number of seed sites reduced from 686 to 682
2025-04-15 08:14:47.339866 [info     ] [chr19] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:47.340245 [info     ] [chr19] [step 2.3.1] [LT] identified 682 candidate stripes
2025-04-15 08:14:47.340379 [info     ] [chr19] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:47.347280 [info     ] [chr19] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:47.347356 [info     ] [chr19] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:47.358697 [info     ] [chr19] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:47.359044 [info     ] [chr19] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:47.374983 [info     ] [chr19] [step 2.2.3] [UT] number of seed sites reduced from 672 to 667
2025-04-15 08:14:47.375099 [info     ] [chr19] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:47.375821 [info     ] [chr19] [step 2.3.1] [UT] identified 667 candidate stripes
2025-04-15 08:14:47.375970 [info     ] [chr19] [step 2    ] topological data analysis took 69.396ms
2025-04-15 08:14:47.376460 [info     ] [chr19] [step 3    ] shape analysis
2025-04-15 08:14:47.376578 [info     ] [chr19] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:47.381246 [info     ] [chr19] [step 3.1  ] [LT] width estimation of 682 stripes took 4.596ms (mean=52 kbp; std=19 kbp)
2025-04-15 08:14:47.381299 [info     ] [chr19] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:47.383611 [info     ] [chr19] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:48.140984 [info     ] [chr19] [step 3.3  ] [LT] height estimation of 682 stripes tool 757.307ms (mean=823 kbp; std=289 kbp)
2025-04-15 08:14:48.141084 [info     ] [chr19] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:48.143340 [info     ] [chr19] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:48.147528 [info     ] [chr19] [step 3.1  ] [UT] width estimation of 667 stripes took 4.118ms (mean=53 kbp; std=19 kbp)
2025-04-15 08:14:48.147590 [info     ] [chr19] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:48.149928 [info     ] [chr19] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:49.103570 [info     ] [chr19] [step 3.3  ] [UT] height estimation of 667 stripes tool 953.576ms (mean=859 kbp; std=371 kbp)
2025-04-15 08:14:49.103652 [info     ] [chr19] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:49.105896 [info     ] [chr19] [step 3    ] shape analysis took 1.729s
2025-04-15 08:14:49.106238 [info     ] [chr19] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:49.106436 [info     ] [chr19] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:49.258725 [info     ] [chr19] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:49.450086 [info     ] [chr19] [step 4    ] statistical analysis and post-processing took 343.849ms
2025-04-15 08:14:49.450359 [info     ] [chr19] [main      ] processing took 2.346s
2025-04-15 08:14:49.450481 [info     ] [chr19] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:49.493730 [info     ] [chr19] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 43.156ms
2025-04-15 08:14:49.493952 [info     ] [chr20] [main      ] begin processing
2025-04-15 08:14:49.494063 [info     ] [chr20] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:49.709790 [info     ] [chr20] [IO        ] fetched 1949555 pixels in 215.723ms
2025-04-15 08:14:49.709958 [info     ] [chr20] [step 1    ] data pre-processing
2025-04-15 08:14:49.710019 [info     ] [chr20] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:49.722376 [info     ] [chr20] [step 1.1  ] removed 0.00% of the non-zero entries (0/1949555)
2025-04-15 08:14:49.722505 [info     ] [chr20] [step 1.2  ] applying log-transformation
2025-04-15 08:14:49.726447 [info     ] [chr20] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:49.727690 [info     ] [chr20] [step 1    ] preprocessing took 17.667ms
2025-04-15 08:14:49.727884 [info     ] [chr20] [step 2    ] topological data analysis
2025-04-15 08:14:49.727942 [info     ] [chr20] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:49.734796 [info     ] [chr20] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:49.734867 [info     ] [chr20] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:49.747997 [info     ] [chr20] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:49.748404 [info     ] [chr20] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:49.764965 [info     ] [chr20] [step 2.2.3] [LT] number of seed sites reduced from 629 to 606
2025-04-15 08:14:49.765088 [info     ] [chr20] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:49.765413 [info     ] [chr20] [step 2.3.1] [LT] identified 606 candidate stripes
2025-04-15 08:14:49.765538 [info     ] [chr20] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:49.773144 [info     ] [chr20] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:49.773221 [info     ] [chr20] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:49.786009 [info     ] [chr20] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:49.786353 [info     ] [chr20] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:49.803929 [info     ] [chr20] [step 2.2.3] [UT] number of seed sites reduced from 565 to 541
2025-04-15 08:14:49.804045 [info     ] [chr20] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:49.804347 [info     ] [chr20] [step 2.3.1] [UT] identified 541 candidate stripes
2025-04-15 08:14:49.804482 [info     ] [chr20] [step 2    ] topological data analysis took 76.556ms
2025-04-15 08:14:49.805132 [info     ] [chr20] [step 3    ] shape analysis
2025-04-15 08:14:49.805243 [info     ] [chr20] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:49.809396 [info     ] [chr20] [step 3.1  ] [LT] width estimation of 606 stripes took 4.090ms (mean=54 kbp; std=19 kbp)
2025-04-15 08:14:49.809463 [info     ] [chr20] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:49.811530 [info     ] [chr20] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:50.477126 [info     ] [chr20] [step 3.3  ] [LT] height estimation of 606 stripes tool 665.532ms (mean=926 kbp; std=295 kbp)
2025-04-15 08:14:50.477237 [info     ] [chr20] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:50.479250 [info     ] [chr20] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:50.482553 [info     ] [chr20] [step 3.1  ] [UT] width estimation of 541 stripes took 3.241ms (mean=54 kbp; std=19 kbp)
2025-04-15 08:14:50.482624 [info     ] [chr20] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:50.484338 [info     ] [chr20] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:51.263471 [info     ] [chr20] [step 3.3  ] [UT] height estimation of 541 stripes tool 779.066ms (mean=942 kbp; std=311 kbp)
2025-04-15 08:14:51.263561 [info     ] [chr20] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:51.265622 [info     ] [chr20] [step 3    ] shape analysis took 1.460s
2025-04-15 08:14:51.265806 [info     ] [chr20] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:51.265898 [info     ] [chr20] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:51.403545 [info     ] [chr20] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:51.560581 [info     ] [chr20] [step 4    ] statistical analysis and post-processing took 294.767ms
2025-04-15 08:14:51.560893 [info     ] [chr20] [main      ] processing took 2.067s
2025-04-15 08:14:51.561062 [info     ] [chr20] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:51.605068 [info     ] [chr20] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 43.863ms
2025-04-15 08:14:51.605269 [info     ] [chr21] [main      ] begin processing
2025-04-15 08:14:51.605378 [info     ] [chr21] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:51.713899 [info     ] [chr21] [IO        ] fetched 979432 pixels in 108.524ms
2025-04-15 08:14:51.713997 [info     ] [chr21] [step 1    ] data pre-processing
2025-04-15 08:14:51.714063 [info     ] [chr21] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:51.718482 [info     ] [chr21] [step 1.1  ] removed 0.00% of the non-zero entries (0/979432)
2025-04-15 08:14:51.718563 [info     ] [chr21] [step 1.2  ] applying log-transformation
2025-04-15 08:14:51.720546 [info     ] [chr21] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:51.721128 [info     ] [chr21] [step 1    ] preprocessing took 7.064ms
2025-04-15 08:14:51.723872 [info     ] [chr21] [step 2    ] topological data analysis
2025-04-15 08:14:51.723941 [info     ] [chr21] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:51.728447 [info     ] [chr21] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:51.728526 [info     ] [chr21] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:51.737047 [info     ] [chr21] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:51.737425 [info     ] [chr21] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:51.749171 [info     ] [chr21] [step 2.2.3] [LT] number of seed sites reduced from 314 to 287
2025-04-15 08:14:51.749286 [info     ] [chr21] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:51.749484 [info     ] [chr21] [step 2.3.1] [LT] identified 287 candidate stripes
2025-04-15 08:14:51.749633 [info     ] [chr21] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:51.754586 [info     ] [chr21] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:51.754661 [info     ] [chr21] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:51.763243 [info     ] [chr21] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:51.763573 [info     ] [chr21] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:51.775716 [info     ] [chr21] [step 2.2.3] [UT] number of seed sites reduced from 299 to 280
2025-04-15 08:14:51.775830 [info     ] [chr21] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:51.776012 [info     ] [chr21] [step 2.3.1] [UT] identified 280 candidate stripes
2025-04-15 08:14:51.776103 [info     ] [chr21] [step 2    ] topological data analysis took 52.174ms
2025-04-15 08:14:51.776661 [info     ] [chr21] [step 3    ] shape analysis
2025-04-15 08:14:51.776767 [info     ] [chr21] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:51.779120 [info     ] [chr21] [step 3.1  ] [LT] width estimation of 287 stripes took 2.295ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:51.779177 [info     ] [chr21] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:51.780279 [info     ] [chr21] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:52.096309 [info     ] [chr21] [step 3.3  ] [LT] height estimation of 287 stripes tool 315.970ms (mean=928 kbp; std=347 kbp)
2025-04-15 08:14:52.096380 [info     ] [chr21] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:52.097469 [info     ] [chr21] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:52.099673 [info     ] [chr21] [step 3.1  ] [UT] width estimation of 280 stripes took 2.161ms (mean=52 kbp; std=19 kbp)
2025-04-15 08:14:52.099722 [info     ] [chr21] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:52.100675 [info     ] [chr21] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:52.493269 [info     ] [chr21] [step 3.3  ] [UT] height estimation of 280 stripes tool 392.534ms (mean=941 kbp; std=388 kbp)
2025-04-15 08:14:52.493341 [info     ] [chr21] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:52.494575 [info     ] [chr21] [step 3    ] shape analysis took 717.849ms
2025-04-15 08:14:52.494900 [info     ] [chr21] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:52.495124 [info     ] [chr21] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:52.555938 [info     ] [chr21] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:52.634208 [info     ] [chr21] [step 4    ] statistical analysis and post-processing took 139.306ms
2025-04-15 08:14:52.634557 [info     ] [chr21] [main      ] processing took 1.029s
2025-04-15 08:14:52.634731 [info     ] [chr21] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:52.667330 [info     ] [chr21] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 32.449ms
2025-04-15 08:14:52.667510 [info     ] [chr22] [main      ] begin processing
2025-04-15 08:14:52.667623 [info     ] [chr22] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:52.801972 [info     ] [chr22] [IO        ] fetched 1126270 pixels in 134.345ms
2025-04-15 08:14:52.802115 [info     ] [chr22] [step 1    ] data pre-processing
2025-04-15 08:14:52.802175 [info     ] [chr22] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:52.809085 [info     ] [chr22] [step 1.1  ] removed 0.00% of the non-zero entries (0/1126270)
2025-04-15 08:14:52.809170 [info     ] [chr22] [step 1.2  ] applying log-transformation
2025-04-15 08:14:52.811527 [info     ] [chr22] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:52.812205 [info     ] [chr22] [step 1    ] preprocessing took 10.029ms
2025-04-15 08:14:52.812361 [info     ] [chr22] [step 2    ] topological data analysis
2025-04-15 08:14:52.812420 [info     ] [chr22] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:52.817218 [info     ] [chr22] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:52.817294 [info     ] [chr22] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:52.826333 [info     ] [chr22] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:52.826728 [info     ] [chr22] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:52.839371 [info     ] [chr22] [step 2.2.3] [LT] number of seed sites reduced from 387 to 356
2025-04-15 08:14:52.839497 [info     ] [chr22] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:52.839734 [info     ] [chr22] [step 2.3.1] [LT] identified 356 candidate stripes
2025-04-15 08:14:52.839845 [info     ] [chr22] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:52.845054 [info     ] [chr22] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:52.845135 [info     ] [chr22] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:52.853985 [info     ] [chr22] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:52.854315 [info     ] [chr22] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:52.867442 [info     ] [chr22] [step 2.2.3] [UT] number of seed sites reduced from 349 to 332
2025-04-15 08:14:52.867569 [info     ] [chr22] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:52.867789 [info     ] [chr22] [step 2.3.1] [UT] identified 332 candidate stripes
2025-04-15 08:14:52.867936 [info     ] [chr22] [step 2    ] topological data analysis took 55.528ms
2025-04-15 08:14:52.868308 [info     ] [chr22] [step 3    ] shape analysis
2025-04-15 08:14:52.868416 [info     ] [chr22] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:52.871122 [info     ] [chr22] [step 3.1  ] [LT] width estimation of 356 stripes took 2.649ms (mean=52 kbp; std=20 kbp)
2025-04-15 08:14:52.871214 [info     ] [chr22] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:52.872517 [info     ] [chr22] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:53.261178 [info     ] [chr22] [step 3.3  ] [LT] height estimation of 356 stripes tool 388.596ms (mean=927 kbp; std=386 kbp)
2025-04-15 08:14:53.261278 [info     ] [chr22] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:53.262576 [info     ] [chr22] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:53.264759 [info     ] [chr22] [step 3.1  ] [UT] width estimation of 332 stripes took 2.138ms (mean=55 kbp; std=20 kbp)
2025-04-15 08:14:53.264808 [info     ] [chr22] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:53.265903 [info     ] [chr22] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:53.744115 [info     ] [chr22] [step 3.3  ] [UT] height estimation of 332 stripes tool 478.152ms (mean=975 kbp; std=406 kbp)
2025-04-15 08:14:53.744221 [info     ] [chr22] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:53.745591 [info     ] [chr22] [step 3    ] shape analysis took 877.227ms
2025-04-15 08:14:53.745964 [info     ] [chr22] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:53.746212 [info     ] [chr22] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:53.822032 [info     ] [chr22] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:53.920128 [info     ] [chr22] [step 4    ] statistical analysis and post-processing took 174.165ms
2025-04-15 08:14:53.920585 [info     ] [chr22] [main      ] processing took 1.253s
2025-04-15 08:14:53.920796 [info     ] [chr22] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:53.957097 [info     ] [chr22] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 36.194ms
2025-04-15 08:14:53.957276 [info     ] [chrX ] [main      ] begin processing
2025-04-15 08:14:53.957389 [info     ] [chrX ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:54.306338 [info     ] [chrX ] [IO        ] fetched 3121669 pixels in 348.943ms
2025-04-15 08:14:54.306507 [info     ] [chrX ] [step 1    ] data pre-processing
2025-04-15 08:14:54.306576 [info     ] [chrX ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:54.329521 [info     ] [chrX ] [step 1.1  ] removed 0.00% of the non-zero entries (0/3121669)
2025-04-15 08:14:54.329666 [info     ] [chrX ] [step 1.2  ] applying log-transformation
2025-04-15 08:14:54.336307 [info     ] [chrX ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:54.338278 [info     ] [chrX ] [step 1    ] preprocessing took 31.700ms
2025-04-15 08:14:54.338440 [info     ] [chrX ] [step 2    ] topological data analysis
2025-04-15 08:14:54.338514 [info     ] [chrX ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:54.355477 [info     ] [chrX ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:54.355556 [info     ] [chrX ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:54.386638 [info     ] [chrX ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:54.387102 [info     ] [chrX ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:54.427387 [info     ] [chrX ] [step 2.2.3] [LT] number of seed sites reduced from 1546 to 1455
2025-04-15 08:14:54.427520 [info     ] [chrX ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:54.428193 [info     ] [chrX ] [step 2.3.1] [LT] identified 1455 candidate stripes
2025-04-15 08:14:54.428387 [info     ] [chrX ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:54.446238 [info     ] [chrX ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:54.446313 [info     ] [chrX ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:54.475758 [info     ] [chrX ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:54.476185 [info     ] [chrX ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:54.517789 [info     ] [chrX ] [step 2.2.3] [UT] number of seed sites reduced from 1518 to 1422
2025-04-15 08:14:54.518033 [info     ] [chrX ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:54.519324 [info     ] [chrX ] [step 2.3.1] [UT] identified 1422 candidate stripes
2025-04-15 08:14:54.519548 [info     ] [chrX ] [step 2    ] topological data analysis took 181.062ms
2025-04-15 08:14:54.519847 [info     ] [chrX ] [step 3    ] shape analysis
2025-04-15 08:14:54.519931 [info     ] [chrX ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:54.528867 [info     ] [chrX ] [step 3.1  ] [LT] width estimation of 1455 stripes took 8.865ms (mean=53 kbp; std=19 kbp)
2025-04-15 08:14:54.528923 [info     ] [chrX ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:54.533208 [info     ] [chrX ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:56.104509 [info     ] [chrX ] [step 3.3  ] [LT] height estimation of 1455 stripes tool 1.571s (mean=657 kbp; std=333 kbp)
2025-04-15 08:14:56.104606 [info     ] [chrX ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:56.109093 [info     ] [chrX ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:56.117668 [info     ] [chrX ] [step 3.1  ] [UT] width estimation of 1422 stripes took 8.521ms (mean=54 kbp; std=18 kbp)
2025-04-15 08:14:56.117739 [info     ] [chrX ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:56.121765 [info     ] [chrX ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:57.960788 [info     ] [chrX ] [step 3.3  ] [UT] height estimation of 1422 stripes tool 1.839s (mean=664 kbp; std=351 kbp)
2025-04-15 08:14:57.960867 [info     ] [chrX ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:57.965120 [info     ] [chrX ] [step 3    ] shape analysis took 3.445s
2025-04-15 08:14:57.965265 [info     ] [chrX ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:57.965344 [info     ] [chrX ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:58.266929 [info     ] [chrX ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:58.610040 [info     ] [chrX ] [step 4    ] statistical analysis and post-processing took 644.765ms
2025-04-15 08:14:58.610305 [info     ] [chrX ] [main      ] processing took 4.653s
2025-04-15 08:14:58.610399 [info     ] [chrX ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:58.673854 [info     ] [chrX ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 63.399ms
2025-04-15 08:14:58.674153 [info     ] [chrY ] [main      ] begin processing
2025-04-15 08:14:58.674393 [info     ] [chrY ] [IO        ] fetching interactions using normalization=NONE
2025-04-15 08:14:58.703469 [info     ] [chrY ] [IO        ] fetched 227360 pixels in 29.084ms
2025-04-15 08:14:58.703645 [info     ] [chrY ] [step 1    ] data pre-processing
2025-04-15 08:14:58.703753 [info     ] [chrY ] [step 1.1  ] focusing on a neighborhood of the main diagonal
2025-04-15 08:14:58.705457 [info     ] [chrY ] [step 1.1  ] removed 0.00% of the non-zero entries (0/227360)
2025-04-15 08:14:58.705520 [info     ] [chrY ] [step 1.2  ] applying log-transformation
2025-04-15 08:14:58.705942 [info     ] [chrY ] [step 1.3  ] projecting interactions onto [1, 0]
2025-04-15 08:14:58.706109 [info     ] [chrY ] [step 1    ] preprocessing took 2.358ms
2025-04-15 08:14:58.707407 [info     ] [chrY ] [step 2    ] topological data analysis
2025-04-15 08:14:58.707457 [info     ] [chrY ] [step 2.1.0] [LT] computing global 1D pseudo-distribution
2025-04-15 08:14:58.711190 [info     ] [chrY ] [step 2.2.0] [LT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:58.711242 [info     ] [chrY ] [step 2.2.1] [LT] computing persistence
2025-04-15 08:14:58.718449 [info     ] [chrY ] [step 2.2.2] [LT] filtering low persistence values
2025-04-15 08:14:58.718806 [info     ] [chrY ] [step 2.2.3] [LT] removing seeds overlapping sparse regions
2025-04-15 08:14:58.730525 [info     ] [chrY ] [step 2.2.3] [LT] number of seed sites reduced from 175 to 136
2025-04-15 08:14:58.730632 [info     ] [chrY ] [step 2.3.1] [LT] generating the list of candidate stripes
2025-04-15 08:14:58.730737 [info     ] [chrY ] [step 2.3.1] [LT] identified 136 candidate stripes
2025-04-15 08:14:58.730833 [info     ] [chrY ] [step 2.1.0] [UT] computing global 1D pseudo-distribution
2025-04-15 08:14:58.734585 [info     ] [chrY ] [step 2.2.0] [UT] detection of persistent maxima and corresponding minima
2025-04-15 08:14:58.734677 [info     ] [chrY ] [step 2.2.1] [UT] computing persistence
2025-04-15 08:14:58.741616 [info     ] [chrY ] [step 2.2.2] [UT] filtering low persistence values
2025-04-15 08:14:58.741932 [info     ] [chrY ] [step 2.2.3] [UT] removing seeds overlapping sparse regions
2025-04-15 08:14:58.753763 [info     ] [chrY ] [step 2.2.3] [UT] number of seed sites reduced from 177 to 134
2025-04-15 08:14:58.753893 [info     ] [chrY ] [step 2.3.1] [UT] generating the list of candidate stripes
2025-04-15 08:14:58.754009 [info     ] [chrY ] [step 2.3.1] [UT] identified 134 candidate stripes
2025-04-15 08:14:58.754121 [info     ] [chrY ] [step 2    ] topological data analysis took 46.678ms
2025-04-15 08:14:58.755002 [info     ] [chrY ] [step 3    ] shape analysis
2025-04-15 08:14:58.755079 [info     ] [chrY ] [step 3.1  ] [LT] estimating candidate stripe widths
2025-04-15 08:14:58.756319 [info     ] [chrY ] [step 3.1  ] [LT] width estimation of 136 stripes took 1.179ms (mean=51 kbp; std=16 kbp)
2025-04-15 08:14:58.756372 [info     ] [chrY ] [step 3.2  ] [LT] updating candidate stripes with width information
2025-04-15 08:14:58.756882 [info     ] [chrY ] [step 3.3  ] [LT] estimating candidate stripe heights
2025-04-15 08:14:58.891185 [info     ] [chrY ] [step 3.3  ] [LT] height estimation of 136 stripes tool 134.248ms (mean=800 kbp; std=931 kbp)
2025-04-15 08:14:58.891284 [info     ] [chrY ] [step 3.4  ] [LT] updating candidate stripes with height information
2025-04-15 08:14:58.891956 [info     ] [chrY ] [step 3.1  ] [UT] estimating candidate stripe widths
2025-04-15 08:14:58.893029 [info     ] [chrY ] [step 3.1  ] [UT] width estimation of 134 stripes took 1.035ms (mean=51 kbp; std=19 kbp)
2025-04-15 08:14:58.893084 [info     ] [chrY ] [step 3.2  ] [UT] updating candidate stripes with width information
2025-04-15 08:14:58.893576 [info     ] [chrY ] [step 3.3  ] [UT] estimating candidate stripe heights
2025-04-15 08:14:59.040807 [info     ] [chrY ] [step 3.3  ] [UT] height estimation of 134 stripes tool 147.167ms (mean=609 kbp; std=267 kbp)
2025-04-15 08:14:59.040905 [info     ] [chrY ] [step 3.4  ] [UT] updating candidate stripes with height information
2025-04-15 08:14:59.041616 [info     ] [chrY ] [step 3    ] shape analysis took 286.578ms
2025-04-15 08:14:59.041776 [info     ] [chrY ] [step 4    ] statistical analysis and post-processing
2025-04-15 08:14:59.041855 [info     ] [chrY ] [step 4.1  ] [LT] computing stripe biological descriptors
2025-04-15 08:14:59.069704 [info     ] [chrY ] [step 4.1  ] [UT] computing stripe biological descriptors
2025-04-15 08:14:59.099486 [info     ] [chrY ] [step 4    ] statistical analysis and post-processing took 57.706ms
2025-04-15 08:14:59.099709 [info     ] [chrY ] [main      ] processing took 425.558ms
2025-04-15 08:14:59.099784 [info     ] [chrY ] [IO        ] writing results to file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:59.123165 [info     ] [chrY ] [IO        ] successfully written results to "/tmp/4DNFI9GMP2J8.10000.hdf5" in 23.338ms
2025-04-15 08:14:59.123408 [info     ] [IO        ] finalizing file "/tmp/4DNFI9GMP2J8.10000.hdf5"
2025-04-15 08:14:59.127303 [info     ] [main      ] DONE!
2025-04-15 08:14:59.127399 [info     ] [main      ] processed 24 chromosomes in 1m:34.490s
```

</details>

Running the above command produces a single output file named `4DNFI9GMP2J8.10000.hdf5`.

The path to the output file can be changed by using the `--output-file` CLI option.

Furthermore, it is possible to save the complete log of a run to a file by specifying the path where to store the log file through the `--log-file` CLI option.

Finally, `stripepy call` can generate several diagnostic plots that can be of help to gain more insights into the decisions made by the tool.
To generate the diagnostic plots, pass `--roi=middle` e specify the path to a folder where to store the plots using `--plot-dir`.

When processing larger Hi-C matrix, StripePy can take advantage of multicore processors.

The maximum number of CPU cores use by StripePy can be changed through option `--nproc` (set to 1 core by default).

Whenever possible, we recommend using 4-8 CPU cores.
Using more than 8 CPU cores is unlikely to result in significantly better computational performance (that is unless your Hi-C dataset is particularly dense).

#### 4) Fetch stripes in BEDPE format

The `.hdf5` file produced by `stripepy call` contains various kinds of information, including stripe coordinates, various descriptive statistics, persistence vectors, and more.

While having access to all this information can be useful, usually we are mostly interested in the stripe coordinates, which can be fetched using `stripepy view`.

```console
# Fetch the first 10 stripes in BEDPE format
user@dev:/tmp$ stripepy view 4DNFI9GMP2J8.10000.hdf5 | head

chr1	910000	960000	chr1	930000	3590000
chr1	1060000	1110000	chr1	1080000	3540000
chr1	1400000	1490000	chr1	1430000	3540000
chr1	1600000	1670000	chr1	880000	1620000
chr1	1670000	1700000	chr1	1680000	2610000
chr1	1730000	1780000	chr1	1750000	2570000
chr1	1890000	1940000	chr1	1920000	3540000
chr1	2020000	2060000	chr1	2020000	3550000
chr1	2070000	2120000	chr1	2090000	3540000
chr1	2170000	2230000	chr1	2190000	3500000

# Redirect stdout to a file
user@dev:/tmp$ stripepy view 4DNFI9GMP2J8.10000.hdf5 > stripes.bedpe

# Compress stripes on the fly before writing to a file
user@dev:/tmp$ stripepy view 4DNFI9GMP2J8.10000.hdf5 | gzip -9 > stripes.bedpe.gz
```

If you are interested in the biodescriptors associated with each individual stripe, you can pass `--with-header` and `--with-biodescriptors` when calling `stripepy view`.

<details>
<summary>Example output</summary>

This is the output generated by running `stripepy view` on the `.hdf5` generated using `stripepy call` v1.0.0.
Files generated by older versions of StripePy may have different columns.

```console
user@dev:/tmp$ stripepy view 4DNFI9GMP2J8.10000.hdf5 --with-biodescriptors --with-header | head

chrom1	start1	end1	chrom2	start2	end2	top_persistence	inner_mean	inner_std	outer_lsum	outer_lsize	outer_rsum	outer_rsize	min	q1	q2	q3	max	outer_lmean	outer_rmean	outer_mean	rel_change
chr1	910000	960000	chr1	930000	3590000	0.3984904019	0.2506571890861574	0.14123131812515843	144.79589039186396	801	192.25135582429806	801	0.0	0.17139833204774585	0.22938081658911763	0.28656944403925566	0.9741568863537948	0.18076890186250183	0.24001417705904876	0.2103915394607753	19.138435760573497
chr1	1060000	1110000	chr1	1080000	3540000	0.0826359687	0.23019685453871336	0.14481608064533394	186.18030631678906	741	179.64345985134207	741	0.0	0.1539575922232785	0.21018481227951455	0.2710230083036015	0.9903418421799679	0.2512554741117261	0.24243381896267485	0.24684464653720048	6.744238626207448
chr1	1570000	1620000	chr1	1600000	2590000	0.04103011280000002	0.33195798369580404	0.10697974882795283	99.02697827900961	300	85.58022773213244	300	0.10509240613975727	0.2710230083036015	0.3152772184192718	0.3662448898065007	0.9887477925105556	0.3300899275966987	0.2852674257737748	0.3076786766852368	7.891124361343245
chr1	1600000	1670000	chr1	880000	1620000	0.10798038449999997	0.34673478460468343	0.12547401272240433	79.95811315769556	225	63.18147668278408	225	0.0	0.25904999836303577	0.33447322272887486	0.4155250840484962	0.9887477925105556	0.3553693918119803	0.2808065630345959	0.3180879774232881	9.0059383612837
chr1	1670000	1700000	chr1	1680000	2610000	0.08521339110000004	0.30510000180174507	0.11602295320194354	84.13794539599031	282	71.90225464650885	282	0.0	0.22938081658911763	0.304010183863723	0.37277167877770423	0.8753282776351561	0.29836150849641957	0.2549725342074782	0.2766670213519489	10.276967710447305
chr1	1730000	1780000	chr1	1750000	2570000	0.09549401749999997	0.34157106048803376	0.12939228310023276	66.96694495052422	249	77.44100032822071	249	0.06630592590798857	0.25245019336736707	0.32535592427102433	0.41427461878487365	0.9374989352738993	0.26894355401816955	0.3110080334466695	0.28997579373241955	17.792956471126924
chr1	1780000	1840000	chr1	1780000	2580000	0.14961356020000005	0.31446872398046843	0.14174768874612398	89.65252960337472	243	73.53776985594494	243	0.0	0.2202635181312671	0.28656944403925566	0.3761154144433587	0.9150948504497306	0.3689404510426943	0.3026245673084154	0.33578250917555486	6.347497148501883
chr1	1890000	1940000	chr1	1920000	3540000	0.13643510830000005	0.27087952940479454	0.15589512088714813	98.34422915113818	489	137.9512119037385	489	0.0	0.17139833204774585	0.2453610817780414	0.3592307814635864	0.989227567682685	0.20111294304936234	0.2821088177990563	0.24161088042420928	12.113961477726793
chr1	1940000	2020000	chr1	1960000	3590000	0.05824488140000006	0.267059000791004	0.1518633129658817	138.54936114286124	492	138.81994263073136	492	0.0	0.17139833204774585	0.2453610817780414	0.34858989163711346	0.9751278353396942	0.28160439256679115	0.2821543549405109	0.281879373753651	5.257700400455457
```

</details>

#### 5) Quickly visualize architectural stripes

It is often a good idea to visually inspect at least some of the stripes to make sure that the used parameters are suitable for the dataset that was given to `stripepy call`.

We provide a Jupyter notebook ([visualize_stripes_with_highlass.ipynb](utils/visualize_stripes_with_highlass.ipynb)) to facilitate this visual inspection.
The notebook expects the input file to be in `.mcool` format.

If your matrix is in `.hic` format you can easily convert it to `.mcool` format using hictk by running `hictk convert matrix.hic matrix.mcool`.
HiGlass cannot visualize single-resolution Cooler files. If you are working with `.cool` files you can use hictk to generate `.mcool` files by running `hictk zoomify matrix.cool matrix.mcool`.

For more details, please refer to hictk's documentation: [hictk.readthedocs.io](https://hictk.readthedocs.io/en/stable/quickstart_cli.html).

We recommend running the notebook using [JupyterLab](https://jupyter.org/install).

Furthermore, the notebook depends on a few Python packages that can be installed with `pip`.
Please make sure that the following packages are installed in a virtual environment that is accessible from Jupyter.
Refer to [IPython](https://ipython.readthedocs.io/en/stable/install/kernel_install.html) documentation for instructions on how to add a virtual environment to Jupyter.

```bash
pip install 'clodius>=0.20,<1' 'hictkpy>=1,<2' 'higlass-python>=1.2,<2'
```

Next, launch JupyterLab and open notebook [visualize_stripes_with_highlass.ipynb](utils/visualize_stripes_with_highlass.ipynb).

```bash
jupyter lab
```

Before running the notebook, scroll down to the following cell

```jupyter
mcool = ensure_file_exists("CHANGEME.mcool")
bedpe = ensure_file_exists("CHANGEME.bedpe")
```

and set the `mcool` and `bedpe` variables to the path to the `.mcool` file used to call stripes and the path to the stripe coordinates extracted with `stripepy view`, respectively.

```jupyter
mcool = ensure_file_exists("4DNFI9GMP2J8.mcool")
bedpe = ensure_file_exists("stripes.bedpe")
```

Now you are ready to run all cells.

Running the last cell will display a HiGlass window embedded in the Jupyter notebook (note that the interface may take a while to load).

![HiGlass window](https://github.com/paulsengroup/StripePy/blob/main/docs/assets/4DNFI9GMP2J8_chr2_156mbp_higlass_view.png?raw=true)

## Generating plots

StripePy comes with a `plot` subcommand that can be used to generate various kinds of plots.

`stripepy plot` supports the following subcommands:

- `contact-map` (`cm`): plot stripes and other features over the Hi-C matrix
- `pseudodistribution` (`pd`): plot the pseudo-distribution over the given region of interest
- `stripe-hist` (`hist`): generate and plot the histograms showing the distribution of the stripe heights and widths

`stripepy cm` takes as input a Hi-C matrix in `.cool`, `.mcool`, or `.hic` format, and optionally the `.hdf5` file generated by `stripepy call`
(this parameter is mandatory when highlighting stripes or stripe seeds).

`stripepy pd` and `stripepy hist` do not require the Hi-C matrix file, and require the `.hdf5` file generated by `stripepy call` instead.

All three subcommands support specifying a region of interest through the `--region` option.
When the commands are run without specifying the region of interest, `stripepy cm` and `stripepy pd` will generate plots for a random 2.5 Mbp region,
while `stripepy hist` will generate histograms using data from the entire genome.

Example usage:

```bash
# Plot the pseudo-distribution over a region of interest
stripepy plot pd 4DNFI9GMP2J8.10000.hdf5 /tmp/pseudodistribution.png --region chr2:120100000-122100000

# Plot the histograms using genome-wide data
stripepy plot hist 4DNFI9GMP2J8.10000.hdf5 /tmp/stripe_hist_gw.png

# Plot the Hi-C matrix
stripepy plot cm 4DNFI9GMP2J8.mcool 10000 /tmp/matrix.png

# Plot the Hi-C matrix highlighting the stripe seeds
stripepy plot cm 4DNFI9GMP2J8.mcool 10000 /tmp/matrix_with_seeds.png --stripepy-hdf5 4DNFI9GMP2J8.10000.hdf5 --highlight-seeds

# Plot the Hi-C matrix highlighting the architectural stripes
stripepy plot cm 4DNFI9GMP2J8.mcool 10000 /tmp/matrix_with_stripes.png --stripepy-hdf5 4DNFI9GMP2J8.10000.hdf5 --highlight-stripes
```

Some example plots generated with `stripepy plot` can be found in file `stripepy-plot-test-images.tar.xz` from [doi.org/10.5281/zenodo.14283921](https://doi.org/10.5281/zenodo.14283921)

## Getting help

For any issues regarding StripePy installation, walkthrough, and output interpretation please open a [discussion](https://github.com/paulsengroup/StripePy/discussions) on GitHub.

If you've found a bug or would like to suggest a new feature, please open a new [issue](https://github.com/paulsengroup/StripePy/issues) instead.

## Citing

If you use StripePy in your research, please cite the following publication:

Andrea Raffo, Roberto Rossini, Jonas Paulsen, StripePy: fast and robust characterization of architectural stripes
_bioRxiv_ 2024.12.20.629789; doi: [https://doi.org/10.1101/2024.12.20.629789](https://doi.org/10.1101/2024.12.20.629789)

<details>
<summary>BibTex</summary>

```bibtex
@article {stripepy,
  author = {Raffo, Andrea and Rossini, Roberto and Paulsen, Jonas},
  title = {StripePy: fast and robust characterization of architectural stripes},
  elocation-id = {2024.12.20.629789},
  year = {2024},
  doi = {10.1101/2024.12.20.629789},
  publisher = {Cold Spring Harbor Laboratory},
  abstract = {
    Architectural stripes in Hi-C and related data are crucial for gene regulation, development, and DNA repair.
    Despite their importance, few tools exist for automatic stripe detection.
    We introduce StripePy, which leverages computational geometry methods to identify and analyze architectural stripes in contact maps from Chromosome Conformation Capture experiments like Hi-C and Micro-C.
    StripePy outperforms existing tools, as shown through tests on various datasets and a newly developed simulated benchmark, StripeBench, providing a valuable resource for the community.Competing Interest StatementThe authors have declared no competing interest.@
  },
  URL = {https://www.biorxiv.org/content/early/2024/12/22/2024.12.20.629789},
  eprint = {https://www.biorxiv.org/content/early/2024/12/22/2024.12.20.629789.full.pdf},
  journal = {bioRxiv}
}
```

</details>
