import copy
from typing import Any, ClassVar, Dict

import pytest
from eth_pydantic_types import HashBytes32, HexBytes
from eth_typing import HexAddress, HexStr
from ethpm_types.abi import ABIType, EventABI, MethodABI

from ape.api import PluginConfig
from ape.api.networks import LOCAL_NETWORK_NAME, NetworkAPI
from ape.exceptions import DecodingError, NetworkError, NetworkNotFoundError
from ape.types import AddressType
from ape.utils import DEFAULT_LOCAL_TRANSACTION_ACCEPTANCE_TIMEOUT
from ape_ethereum.ecosystem import (
    BLUEPRINT_HEADER,
    BaseEthereumConfig,
    Block,
    Ethereum,
    EthereumConfig,
)
from ape_ethereum.transactions import (
    DynamicFeeTransaction,
    Receipt,
    SharedBlobReceipt,
    SharedBlobTransaction,
    StaticFeeTransaction,
    TransactionType,
)

LOG = {
    "removed": False,
    "logIndex": "0x0",
    "transactionIndex": "0x0",
    "transactionHash": "0x74dd040dfa06f0af9af8ca95d7aae409978400151c746f55ecce19e7356cfc5a",
    "blockHash": "0x2c99950b07accf3e442512a3352a11e6fed37b2331de5f71b7743b357d96e4e8",
    "blockNumber": "0xa946ac",
    "address": "0x274b028b03a250ca03644e6c578d81f019ee1323",
    "data": "0xabffd4675206dab5d04a6b0d62c975049665d1f512f29f303908abdd20bc08a100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000060000000000000000000000000000000000000000000000000000000000000000744796e616d696300000000000000000000000000000000000000000000000000",  # noqa: E501
    "topics": [
        "0xa84473122c11e32cd505595f246a28418b8ecd6cf819f4e3915363fad1b8f968",
        "0x0000000000000000000000000000000000000000000000000000000000000006",
        "0x9f3d45ac20ccf04b45028b8080bb191eab93e29f7898ed43acf480dd80bba94d",
    ],
}
CUSTOM_ECOSYSTEM_NAME = "custom-ecosystem"


@pytest.fixture
def event_abi(vyper_contract_instance):
    return vyper_contract_instance.NumberChange.abi


@pytest.fixture
def custom_ecosystem_config(
    custom_networks_config_dict, temp_config, networks, custom_network_name_0
):
    data = copy.deepcopy(custom_networks_config_dict)
    data["networks"]["custom"][0]["ecosystem"] = CUSTOM_ECOSYSTEM_NAME

    # Also ensure we can configure the custom ecosystem as if
    # it were from a plugin.
    data[CUSTOM_ECOSYSTEM_NAME] = {"default_network": custom_network_name_0}

    with temp_config(data):
        yield


def test_name(ethereum):
    assert ethereum.name == "ethereum"


def test_name_when_custom(custom_ecosystem_config, networks):
    ecosystem = networks.get_ecosystem(CUSTOM_ECOSYSTEM_NAME)
    actual = ecosystem.name
    expected = CUSTOM_ECOSYSTEM_NAME
    assert actual == expected


@pytest.mark.parametrize(
    "address",
    (
        "0x63953eB1B3D8DB28334E7C1C69456C851F934199".lower(),
        0x63953EB1B3D8DB28334E7C1C69456C851F934199,
    ),
)
def test_decode_address(ethereum, address):
    expected = "0x63953eB1B3D8DB28334E7C1C69456C851F934199"
    actual = ethereum.decode_address(address)
    assert actual == expected


def test_encode_address(ethereum):
    raw_address = "0x63953eB1B3D8DB28334E7C1C69456C851F934199"
    address = AddressType(HexAddress(HexStr(raw_address)))
    actual = ethereum.encode_address(address)
    assert actual == raw_address


def test_encode_calldata(ethereum, address):
    abi = MethodABI(
        type="function",
        name="callMe",
        inputs=[
            ABIType(name="a", type="bytes4"),
            ABIType(name="b", type="address"),
            ABIType(name="c", type="uint256"),
            ABIType(name="d", type="bytes4[]"),
        ],
    )
    byte_array = ["0x456", "0x678"]
    values = ("0x123", address, HexBytes(55), byte_array)

    actual = ethereum.encode_calldata(abi, *values)
    expected = HexBytes(
        # 0x123
        "0123000000000000000000000000000000000000000000000000000000000000"
        # address
        "000000000000000000000000d8da6bf26964af9d7eed9e03e53415d37aa96045"
        # HexBytes(55)
        "0000000000000000000000000000000000000000000000000000000000000037"
        # byte_array
        "0000000000000000000000000000000000000000000000000000000000000080"
        "0000000000000000000000000000000000000000000000000000000000000002"
        "0456000000000000000000000000000000000000000000000000000000000000"
        "0678000000000000000000000000000000000000000000000000000000000000"
    )
    assert actual == expected


@pytest.mark.parametrize(
    "sequence_type,item_type",
    [
        (list, str),
        (tuple, str),
        (list, HexBytes),
        (tuple, HexBytes),
    ],
)
def test_encode_calldata_byte_array(ethereum, sequence_type, item_type):
    """
    Tests against a bug where we could not pass a tuple of HexStr
    for a byte-array.
    """
    raw_abi = {
        "type": "function",
        "name": "mint",
        "stateMutability": "nonpayable",
        "inputs": [{"name": "leaf", "type": "bytes32"}, {"name": "proof", "type": "bytes32[]"}],
        "outputs": [],
    }
    abi = MethodABI.model_validate(raw_abi)
    hexstr_array = sequence_type(
        (
            item_type("0xfadbd3"),
            item_type("0x9c6b2c"),
            item_type("0xc5d246"),
        )
    )
    actual = ethereum.encode_calldata(abi, "0x0123", hexstr_array)
    assert isinstance(actual, bytes)


def test_encode_calldata_nested_structs(ethereum):
    abi = MethodABI(
        type="function",
        name="check",
        stateMutability="view",
        inputs=[
            ABIType(
                name="data",
                type="tuple",
                components=[
                    ABIType(
                        name="tokenIn", type="address", components=None, internal_type="address"
                    ),
                    ABIType(
                        name="amountIn", type="uint256", components=None, internal_type="uint256"
                    ),
                    ABIType(
                        name="minAmountOut",
                        type="uint256",
                        components=None,
                        internal_type="uint256",
                    ),
                    ABIType(
                        name="path",
                        type="tuple",
                        components=[
                            ABIType(
                                name="pairBinSteps",
                                type="uint256[]",
                                components=None,
                                internal_type="uint256[]",
                            ),
                            ABIType(
                                name="versions",
                                type="uint8[]",
                                components=None,
                                internal_type="enum SwapWrapper.Version[]",
                            ),
                            ABIType(
                                name="tokenPath",
                                type="address[]",
                                components=None,
                                internal_type="address[]",
                            ),
                        ],
                        internal_type="struct SwapWrapper.Path",
                    ),
                ],
                internal_type="struct SwapWrapper.SwapData",
            )
        ],
        outputs=[ABIType(name="", type="bool", components=None, internal_type="bool")],
    )
    calldata = {
        "tokenIn": "0xB31f66AA3C1e785363F0875A1B74E27b85FD66c7",
        "amountIn": 1000000000000000000,
        "minAmountOut": 10000000,
        "path": {
            "pairBinSteps": [0],
            "versions": [0],
            "tokenPath": [
                "0xB31f66AA3C1e785363F0875A1B74E27b85FD66c7",
                "0xB97EF9Ef8734C71904D8002F8b6Bc66Dd9c48a6E",
            ],
        },
    }
    actual = ethereum.encode_calldata(abi, calldata)
    expected = HexBytes(
        "0x000000000000000000000000000000000000000000000000000000000000002"
        "0000000000000000000000000b31f66aa3c1e785363f0875a1b74e27b85fd66c7"
        "0000000000000000000000000000000000000000000000000de0b6b3a76400000"
        "00000000000000000000000000000000000000000000000000000000098968000"
        "00000000000000000000000000000000000000000000000000000000000080000"
        "00000000000000000000000000000000000000000000000000000000000600000"
        "0000000000000000000000000000000000000000000000000000000000a000000"
        "000000000000000000000000000000000000000000000000000000000e0000000"
        "00000000000000000000000000000000000000000000000000000000010000000"
        "00000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000001000000000"
        "00000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000200000000000"
        "0000000000000b31f66aa3c1e785363f0875a1b74e27b85fd66c7000000000000"
        "000000000000b97ef9ef8734c71904d8002f8b6bc66dd9c48a6e"
    )
    assert actual == expected


def test_block_handles_snake_case_parent_hash(eth_tester_provider, sender, receiver):
    # Transaction to change parent hash of next block
    sender.transfer(receiver, "1 gwei")

    # Replace 'parentHash' key with 'parent_hash'
    latest_block = eth_tester_provider.get_block("latest")
    latest_block_dict = eth_tester_provider.get_block("latest").model_dump()
    latest_block_dict["parent_hash"] = latest_block_dict.pop("parentHash")

    redefined_block = Block.model_validate(latest_block_dict)
    assert redefined_block.parent_hash == latest_block.parent_hash


def test_transaction_acceptance_timeout(temp_config, config, networks):
    assert (
        networks.provider.network.transaction_acceptance_timeout
        == DEFAULT_LOCAL_TRANSACTION_ACCEPTANCE_TIMEOUT
    )
    new_value = DEFAULT_LOCAL_TRANSACTION_ACCEPTANCE_TIMEOUT + 1
    timeout_config = {"ethereum": {"local": {"transaction_acceptance_timeout": new_value}}}
    with temp_config(timeout_config):
        assert networks.provider.network.transaction_acceptance_timeout == new_value


def test_decode_logs(ethereum, vyper_contract_instance):
    abi = vyper_contract_instance.NumberChange.abi
    result = [x for x in ethereum.decode_logs([LOG], abi)]
    assert len(result) == 1
    assert result[0].model_dump() == {
        "event_name": "NumberChange",
        "contract_address": "0x274b028b03A250cA03644E6c578D81f019eE1323",
        "event_arguments": {
            "newNum": 6,
            "dynIndexed": HexBytes(
                "0x9f3d45ac20ccf04b45028b8080bb191eab93e29f7898ed43acf480dd80bba94d"
            ),
            "b": HexBytes("0xabffd4675206dab5d04a6b0d62c975049665d1f512f29f303908abdd20bc08a1"),
            "prevNum": 0,
            "dynData": "Dynamic",
        },
        "transaction_hash": "0x74dd040dfa06f0af9af8ca95d7aae409978400151c746f55ecce19e7356cfc5a",
        "block_number": 11093676,
        "block_hash": "0x2c99950b07accf3e442512a3352a11e6fed37b2331de5f71b7743b357d96e4e8",
        "log_index": 0,
        "transaction_index": 0,
    }


def test_decode_logs_empty_list(ethereum, event_abi):
    actual = [x for x in ethereum.decode_logs([], event_abi)]
    assert actual == []


def test_decode_logs_with_struct_from_interface(ethereum):
    abi = EventABI.model_validate(
        {
            "type": "event",
            "name": "ConditionalOrderCreated",
            "inputs": [
                {"name": "owner", "type": "address", "internalType": "address", "indexed": True},
                {
                    "name": "params",
                    "type": "tuple",
                    "components": [
                        {
                            "name": "handler",
                            "type": "address",
                            "internalType": "contract IConditionalOrder",
                        },
                        {"name": "salt", "type": "bytes32", "internalType": "bytes32"},
                        {"name": "staticInput", "type": "bytes", "internalType": "bytes"},
                    ],
                    "internalType": "struct IConditionalOrder.ConditionalOrderParams",
                    "indexed": False,
                },
            ],
            "anonymous": False,
        }
    )
    logs = [
        {
            "address": "0xfdafc9d1902f4e0b84f65f49f244b32b31013b74",
            "blockHash": "0xec1b8e18412dd91114c0b3ea03c41d02243d60cd05661861df5765fb5ec462c6",
            "blockNumber": "0x11101cc",
            "data": "0x00000000000000000000000000000000000000000000000000000000000000200000000000000000000000006cf1e9ca41f7611def408122793c358a3d11e5a5000000000000000000000000000000000000000000000000000000189e491e96000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000001400000000000000000000000006b175474e89094c44da98b954eedeac495271d0f000000000000000000000000c02aaa39b223fe8d0a0e5c4f27ead9083c756cc2000000000000000000000000e7602ca44f83a5e9ba8bd14125ddcb295f3d63bd00000000000000000000000000000000000000000000018eb03406b6af33c8c3000000000000000000000000000000000000000000000000345d7669a1e8815700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000002000000000000000000000000000000000000000000000000000000000000012c00000000000000000000000000000000000000000000000000000000000000007b9bc44550e2cc369010139a1d3a5a6ec6d65910379e0ee35a7c3cdf24cb12a7",  # noqa: E501
            "logIndex": "0x97",
            "removed": False,
            "topics": [
                "0x2cceac5555b0ca45a3744ced542f54b56ad2eb45e521962372eef212a2cbf361",
                "0x000000000000000000000000e7602ca44f83a5e9ba8bd14125ddcb295f3d63bd",
            ],
            "transactionHash": "0x55859e74e591100e0f37f146abc0dbebe2252eaf5f533319ca77427745b85ceb",
            "transactionIndex": "0x45",
        }
    ]
    actual = list(ethereum.decode_logs(logs, abi))
    assert len(actual) == 1
    assert actual[0].owner == "0xE7602Ca44f83a5E9Ba8BD14125dDcb295f3D63BD"
    assert actual[0].params == [
        "0x6cf1e9ca41f7611def408122793c358a3d11e5a5",
        HexBytes("0x000000000000000000000000000000000000000000000000000000189e491e96"),
        HexBytes(
            "0x0000000000000000000000006b175474e89094c44da98b954eedeac495271d0f0000000"
            "00000000000000000c02aaa39b223fe8d0a0e5c4f27ead9083c756cc20000000000000000"
            "00000000e7602ca44f83a5e9ba8bd14125ddcb295f3d63bd0000000000000000000000000"
            "0000000000000000000018eb03406b6af33c8c30000000000000000000000000000000000"
            "00000000000000345d7669a1e881570000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000000000000000000"
            "0000000000020000000000000000000000000000000000000000000000000000000000000"
            "12c00000000000000000000000000000000000000000000000000000000000000007b9bc4"
            "4550e2cc369010139a1d3a5a6ec6d65910379e0ee35a7c3cdf24cb12a7"
        ),
    ]


def test_decode_block_when_hash_is_none(ethereum):
    # When using some providers, such as hardhat, the hash of the pending block is None
    block_data_with_none_hash: Dict[str, Any] = {
        "number": None,
        "hash": None,
        "parentHash": HexBytes(
            "0xcb94e150c06faee9ab2bf12a40b0937ac9eab1879c733ebe3249aafbba2f80b1"
        ),
        "nonce": None,
        "mixHash": None,
        "sha3Uncles": HexBytes(
            "0x1dcc4de8dec75d7aab85b567b6ccd41ad312451b948a7413f0a142fd40d49347"
        ),
        "logsBloom": None,
        "transactionsRoot": HexBytes(
            "0x56e81f171bcc55a6ff8345e692c0f86e5b48e01b996cadc001622fb5e363b421"
        ),
        "stateRoot": HexBytes("0x8728474146565003152f9cee496de043fd68566dabdb06116a0d5bfc63e1a5a9"),
        "receiptsRoot": HexBytes(
            "0x56e81f171bcc55a6ff8345e692c0f86e5b48e01b996cadc001622fb5e363b421"
        ),
        "miner": "0xC014BA5EC014ba5ec014Ba5EC014ba5Ec014bA5E",
        "difficulty": 131072,
        "totalDifficulty": 131073,
        "extraData": HexBytes("0x"),
        "size": 513,
        "gasLimit": 0,
        "gasUsed": 0,
        "timestamp": 1660932629,
        "transactions": [],
        "uncles": [],
        "baseFeePerGas": 0,
    }
    actual = ethereum.decode_block(block_data_with_none_hash)
    assert actual.hash is None
    assert actual.gas_limit == 0


def test_decode_block_with_hex_values(ethereum):
    """
    When using WS providers directly, the values are all hex strings (more raw).
    This test ensures we can still decode blocks that are in this form.
    """
    raw_block = {
        "baseFeePerGas": "0x62d76fae2",
        "difficulty": "0x0",
        "extraData": "0x506f776572656420627920626c6f58726f757465",
        "gasLimit": "0x1c9c380",
        "gasUsed": "0x91b5e5",
        "hash": "0x925cac44d8bac5df3e3eeed6379b3924d6769a054b3c4079899c1d9b442a4041",
        "logsBloom": "0x7823d304ab8c72f91270a023c11482f341a08a00c91b00f8859906c8e9f22e04b19d311e422cfa79d8107e2b901fc7d9c62fcd2d8a673802f500f4a023693525a2ce66481cc51d1a4e97422b8424912b8c719c6821589a240d5015c5cbe4de46b3b60ad43621cd2f540119b3c938d88f011a49d755a31cc415f34896596912d40b0397d0c544571704ea001c56303f02086e16a3a3b7168f443d224034102ca4c6b21456932dae401c8da7c51d36ae3804c8a084811802c708ae94b7010302100700a28f156d2ca2d61420320e82e8b9927b626bc0ec003dfa09009e3d1668798fb0f8c81a205eb100ee8ea2e20c6041b2d8b2b01bd4d14e8acfaa317049d654",  # noqa: E501
        "miner": "0x6b3a8798e5fb9fc5603f3ab5ea2e8136694e55d0",
        "mixHash": "0x0493af8ac5b7de66c7660c22a703a54069cbd561a2767537eaef994bd77e084b",
        "nonce": "0x0000000000000000",
        "number": "0x11234df",
        "parentHash": "0x47651c13bda353d1bbc610bbdb8a9aa1436629f05f8a32a27f143200da698df8",
        "receiptsRoot": "0xa5abe7eeac74aa03ea82421fcef9b74d1822cfe0ee4b29a8dc01eef31e7d0019",
        "sha3Uncles": "0x1dcc4de8dec75d7aab85b567b6ccd41ad312451b948a7413f0a142fd40d49347",
        "size": "0x1cf02",
        "stateRoot": "0x59eb1dfb098361d1f74dffc8436da680715f671dc3f02be28f4db3f44b1a09bf",
        "timestamp": "0x64e4b0cb",
        "transactionsRoot": "0xf2e3c3e4fbe06ff882f3deb8a7ec8cbbe09e55b2c995d93dd0a36d5e843f3efc",
    }
    actual = ethereum.decode_block(raw_block)
    assert actual.difficulty == 0


def test_decode_logs_topics_not_first(ethereum):
    """
    Tests against a condition where we could not decode logs if the
    topics were not defined first.
    """
    abi_dict = {
        "anonymous": False,
        "inputs": [
            {"indexed": False, "internalType": "address", "name": "sender", "type": "address"},
            {"indexed": True, "internalType": "address", "name": "owner", "type": "address"},
            {"indexed": True, "internalType": "int24", "name": "tickLower", "type": "int24"},
            {"indexed": True, "internalType": "int24", "name": "tickUpper", "type": "int24"},
            {"indexed": False, "internalType": "uint128", "name": "amount", "type": "uint128"},
            {"indexed": False, "internalType": "uint256", "name": "amount0", "type": "uint256"},
            {"indexed": False, "internalType": "uint256", "name": "amount1", "type": "uint256"},
        ],
        "name": "Mint",
        "type": "event",
    }
    abi = EventABI.model_validate(abi_dict)
    logs = [
        {
            "address": "0x3416cf6c708da44db2624d63ea0aaef7113527c6",
            "blockHash": "0x488f23ba55f64bf1aac02ee7278b70c3f4bb2fb57b8aaa6ab3b481f1809f18ea",
            "blockNumber": "0xcfa869",
            "data": "0x000000000000000000000000c36442b4a4522e871399cd717abdd847ab11fe8800000000000000000000000000000000000000000000000000821d6b102660fb000000000000000000000000000000000000000000000000000009fde8545338000000000000000000000000000000000000000000000000000003548cf35031",  # noqa: E501
            "logIndex": "0x7a",
            "removed": False,
            "topics": [
                "0x7a53080ba414158be7ec69b987b5fb7d07dee101fe85488f0853ae16239d0bde",
                "0x000000000000000000000000c36442b4a4522e871399cd717abdd847ab11fe88",
                "0xfffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffc",
                "0x0000000000000000000000000000000000000000000000000000000000000004",
            ],
            "transactionHash": "0xf093a630478562d03e3b2476a5b0551609722747d442a462579fa02e1332a941",
            "transactionIndex": "0x41",
        }
    ]
    actual = list(ethereum.decode_logs(logs, abi))
    assert actual


def test_decode_receipt(eth_tester_provider, ethereum):
    receipt_data = {
        "provider": eth_tester_provider,
        "required_confirmations": 0,
        "blockHash": HexBytes("0x8988adc8a0b346526f1d769841b7464e38a282b25ed346f1f810be8cb7393bc2"),
        "blockNumber": 2,
        "from": "0x318b469BBa396AEc2C60342F9441be36A1945174",
        "gas": 1764431,
        "hash": HexBytes("0x58d086e6b7ecf55c92a5fe420d870ff2eebe99824a711a3cdbe29f497a0e534c"),
        "input": "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",  # noqa: E501
        "nonce": 1,
        "to": None,
        "transactionIndex": 0,
        "value": 0,
        "v": 1,
        "r": HexBytes("0xfa532141efbfa5bbeaa542cf68bfe5df97dd18b696b8a183f74714758a646238"),
        "s": HexBytes("0x1b80edf76bb6f0295ff847c1b99eb2928ea3533fe41125cfe0fa17fdc634b938"),
        "type": 2,
        "accessList": [],
        "chainId": 31337,
        "gasPrice": 0,
        "maxFeePerGas": 0,
        "maxPriorityFeePerGas": 0,
        "transactionHash": HexBytes(
            "0x58d086e6b7ecf55c92a5fe420d870ff2eebe99824a711a3cdbe29f497a0e534c"
        ),
        "cumulativeGasUsed": 1764431,
        "gasUsed": 1764431,
        "contractAddress": "0xC072c85922b7233998B2Ab990fEFdE80218Ca63F",
        "logs": [],
        "logsBloom": HexBytes(
            "0x00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"  # noqa: E501
        ),
        "status": 1,
        "effectiveGasPrice": 0,
    }
    receipt = ethereum.decode_receipt(receipt_data)
    assert type(receipt) is Receipt  # NOTE: Purposely not using isinstance()

    # Tests against bug where input data would come back improperly
    assert receipt.data == HexBytes(
        "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"  # noqa: E501
    )


def test_decode_receipt_from_etherscan(eth_tester_provider, ethereum):
    receipt = ethereum.decode_receipt(
        {
            "blockNumber": "11291970",
            "timeStamp": "1661846925",
            "hash": "0x5780b43d819035ed1fa079171bdce7f0bbeaa6b01f201f8985d279a66cfc6844",
            "nonce": "0",
            "blockHash": "0x3175f953c1da4bf3d15b853dae4a150ae44e2e71380936463e89142c12961968",
            "transactionIndex": "31",
            "from": "0xf39Fd6e51aad88F6F4ce6aB8827279cffFb92266",
            "to": "0xe7f1725E7734CE288F8367e1Bb143E90bb3F0512",
            "value": "0",
            "gas": "4712388",
            "gasPrice": "1499999989",
            "isError": "0",
            "input": "0x608060405234801561000f575f80fd5b506101248061001d5f395ff3fe6080604052348015600e575f80fd5b50600436106026575f3560e01c8063f207564e14602a575b5f80fd5b6039603536600460b4565b603b565b005b6040516390b5561d60e01b81526004810182905273274b028b03a250ca03644e6c578d81f019ee1323906390b5561d90602401602060405180830381865af41580156088573d5f803e3d5ffd5b505050506040513d601f19601f8201168201806040525081019060aa919060ca565b60b1575f80fd5b50565b5f6020828403121560c3575f80fd5b5035919050565b5f6020828403121560d9575f80fd5b8151801515811460e7575f80fd5b939250505056fea2646970667358221220050eefcb4ae3417fbe131ccad8d577d9488919884d4c4a4d0bf8b9c46ce640f664736f6c63430008150033",  # noqa: E501
            "contractAddress": "",
            "cumulativeGasUsed": "8978461",
            "gasUsed": "257131",
            "methodId": "0xf926657e",
            "functionName": "setup(address _reputationOracle, address _recordingOracle, uint256 _reputationOracleStake, uint256 _recordingOracleStake, string _url, string _hash)",  # noqa: E501
            "required_confirmations": "13703",
            "status": "1",
            "chainId": 1,
        }
    )
    assert type(receipt) is Receipt  # NOTE: Purposely not using isinstance()
    assert receipt.type == 0
    assert receipt.max_fee == 0
    assert receipt.gas_price == 1499999989


@pytest.mark.parametrize("blob_gas_used", ("0x20000", 131072, 0, None))
def test_decode_receipt_shared_blob(ethereum, blob_gas_used):
    blob_gas_price = "0x4d137e31b"

    data = {
        "required_confirmations": 0,
        "blockHash": HexBytes("0x051fb508617fcbe0034538b7ee54c1dedbbbaa6f8d0aeb776edf81bafbc883bd"),
        "blockNumber": 10526428,
        "from": "0x194E22F49BC3f58903866d55488E1e9e8d69b517",
        "gas": 5500000,
        "gasPrice": 1000000008,
        "maxFeePerGas": 150000000000,
        "maxPriorityFeePerGas": 1000000000,
        "maxFeePerBlobGas": "0x22ecb25c00",
        "hash": HexBytes("0xd2882bae0d79a6c8e0fbf0089bbcb4b2eef3a1365471ad9f779b06a41ba47d3c"),
        "input": HexBytes("0xb72d42a1000000000000000000000000000000000000000000"),  # Note: abridged
        "nonce": 329925,
        "to": "0xd5c325D183C592C94998000C5e0EED9e6655c020",
        "transactionIndex": 48,
        "value": 0,
        "type": 3,
        "accessList": [],
        "chainId": 5,
        "blobVersionedHashes": [
            "0x015405d502a27897ad38ffcb80566d1559fa53764befc6d90731082837c2d19e"
        ],
        "v": 0,
        "r": HexBytes("0x4e5bdcbba31548cb8f9806491c5ced7b0f1eac78c7dc0677616c23ee0e469f74"),
        "s": HexBytes("0x31c98ea044c97225d83b9a3f0fa719e2299b9fbc6436e4b3eb096ea528de03ff"),
        "yParity": 0,
        "blobGasPrice": blob_gas_price,
        "blobGasUsed": blob_gas_used,
        "contractAddress": None,
        "cumulativeGasUsed": 23085827,
        "effectiveGasPrice": 1000000008,
        "gasUsed": 219756,
        "logs": [],
        "logsBloom": HexBytes("0x00000000010002"),
        "status": 1,
        "transactionHash": HexBytes(
            "0xd2882bae0d79a6c8e0fbf0089bbcb4b2eef3a1365471ad9f779b06a41ba47d3c"
        ),
    }
    actual = ethereum.decode_receipt(data)
    assert isinstance(actual, SharedBlobReceipt)
    assert actual.blob_gas_price == int(blob_gas_price, 16)

    if blob_gas_used:
        # all test-values are this when they exist.
        assert actual.blob_gas_used == 131072
    else:
        # when None, should also default to 0.
        assert actual.blob_gas_used == 0


def test_default_transaction_type_not_connected_used_default_network(
    temp_config, ethereum, networks
):
    value = TransactionType.STATIC.value
    config_dict = {"ethereum": {"mainnet_fork": {"default_transaction_type": value}}}
    assert ethereum.default_transaction_type == TransactionType.DYNAMIC

    with temp_config(config_dict):
        ethereum._default_network = "mainnet-fork"
        provider = networks.active_provider

        # Disconnect so it uses default.
        networks.active_provider = None
        try:
            assert ethereum.default_network_name == "mainnet-fork"
            assert ethereum.default_transaction_type == TransactionType.STATIC
            ethereum._default_network = LOCAL_NETWORK_NAME
        finally:
            networks.active_provider = provider


def test_default_transaction_type_configured_from_local_network(
    eth_tester_provider, ethereum, temp_config
):
    _ = eth_tester_provider  # Connection required so 'ethereum' knows the network.
    value = TransactionType.STATIC.value
    config = {"ethereum": {LOCAL_NETWORK_NAME: {"default_transaction_type": value}}}
    with temp_config(config):
        assert ethereum.default_transaction_type == TransactionType.STATIC


def test_default_transaction_type_changed_at_class_level(ethereum):
    """
    Simulates an L2 plugin changing the default at the definition-level.
    """

    class Subconfig(BaseEthereumConfig):
        DEFAULT_TRANSACTION_TYPE: ClassVar[int] = TransactionType.STATIC.value

    config = Subconfig()
    assert config.local.default_transaction_type.value == 0
    assert config.mainnet.default_transaction_type.value == 0
    assert config.mainnet_fork.default_transaction_type.value == 0


def test_default_transaction_type_configured_from_custom_network():
    pass


@pytest.mark.parametrize("network_name", (LOCAL_NETWORK_NAME, "mainnet-fork", "mainnet_fork"))
def test_gas_limit_local_networks(ethereum, network_name):
    network = ethereum.get_network(network_name)
    assert network.gas_limit == "max"


def test_gas_limit_live_networks(ethereum):
    network = ethereum.get_network("goerli")
    assert network.gas_limit == "auto"


def test_encode_blueprint_contract(ethereum, vyper_contract_type):
    actual = ethereum.encode_contract_blueprint(vyper_contract_type)
    ct_bytes = vyper_contract_type.deployment_bytecode.to_bytes()

    # EIP-5202
    expected = (
        HexBytes("0x61")
        + (len(ct_bytes) + 3).to_bytes(2, "big")  # preface and length
        + HexBytes("0x3d81600a3d39f3")  # return stuff
        + BLUEPRINT_HEADER
        + HexBytes(0)
        + ct_bytes
    )

    assert actual.data == HexBytes(expected)


def test_decode_returndata(ethereum):
    abi = MethodABI.model_validate(
        {
            "type": "function",
            "name": "doThing",
            "stateMutability": "nonpayable",
            "inputs": [],
            "outputs": [{"name": "", "type": "bool"}],
        }
    )
    data = HashBytes32.__eth_pydantic_validate__(0)
    actual = ethereum.decode_returndata(abi, data)
    assert actual == (False,)


def test_decode_returndata_non_empty_padding_bytes(ethereum):
    raw_data = HexBytes(
        "0x08c379a000000000000000000000000000000000000000000000000000000000000000200"
        "000000000000000000000000000000000000000000000000000000000000012696e73756666"
        "696369656e742066756e64730000000000000000000000000000"
    )
    abi = MethodABI.model_validate(
        {
            "type": "function",
            "name": "transfer",
            "stateMutability": "nonpayable",
            "inputs": [
                {"name": "receiver", "type": "address"},
                {"name": "amount", "type": "uint256"},
            ],
            "outputs": [{"name": "", "type": "bool"}],
        }
    )
    with pytest.raises(DecodingError):
        ethereum.decode_returndata(abi, raw_data)


def test_decode_returndata_no_bytes_returns_zero(ethereum):
    abi = MethodABI.model_validate(
        {
            "type": "function",
            "name": "doThing",
            "stateMutability": "nonpayable",
            "inputs": [],
            "outputs": [{"name": "", "type": "bool"}],
        }
    )
    actual = ethereum.decode_returndata(abi, b"")
    assert actual == (0,)


def test_decode_returndata_list_with_1_struct(ethereum):
    """
    Tests a condition where an array of a list with 1 struct
    would be turned into a raw tuple instead of the Struct class.
    """
    abi = MethodABI(
        type="function",
        name="getArrayOfStructs",
        stateMutability="view",
        inputs=[],
        outputs=[
            ABIType(
                name="",
                type="tuple[]",
                components=[
                    ABIType(name="a", type="address", components=None, internal_type=None),
                    ABIType(name="b", type="bytes32", components=None, internal_type=None),
                ],
                internal_type=None,
            )
        ],
    )
    raw_data = HexBytes(
        "0x000000000000000000000000000000000000000000000000000000000000002"
        "00000000000000000000000000000000000000000000000000000000000000001"
        "000000000000000000000000f39fd6e51aad88f6f4ce6ab8827279cfffb92266f"
        "d91dc47b758f65fd0f6fe511566770c0ae3f94ff9999ceb23bfec3ac9fdc168"
    )
    actual = ethereum.decode_returndata(abi, raw_data)

    assert len(actual) == 1
    # The result should still be a list.
    # There was also a bug where it was mistakenly a tuple!
    assert isinstance(actual[0], list)
    assert len(actual[0]) == 1
    assert actual[0][0].a == "0xf39Fd6e51aad88F6F4ce6aB8827279cffFb92266"
    assert actual[0][0].b == HexBytes(
        "0xfd91dc47b758f65fd0f6fe511566770c0ae3f94ff9999ceb23bfec3ac9fdc168"
    )


@pytest.mark.parametrize("tx_type", TransactionType)
def test_create_transaction_uses_network_gas_limit(tx_type, ethereum, eth_tester_provider, owner):
    tx = ethereum.create_transaction(type=tx_type.value, sender=owner.address)
    assert tx.type == tx_type.value
    assert tx.gas_limit == eth_tester_provider.max_gas


def test_create_transaction_with_none_values(ethereum, eth_tester_provider):
    """
    Tests against None being in place of values in kwargs,
    causing their actual defaults not to get used and ValidationErrors
    to occur.
    """
    static = ethereum.create_transaction(
        value=None, data=None, chain_id=None, gas_price=None, nonce=None, receiver=None
    )
    dynamic = ethereum.create_transaction(
        value=None,
        data=None,
        chain_id=None,
        max_fee=None,
        max_prioriy_fee=None,
        nonce=None,
        receiver=None,
    )
    assert isinstance(static, StaticFeeTransaction)  # Because used gas_price
    assert isinstance(dynamic, DynamicFeeTransaction)  # Because used max_fee

    for tx in (static, dynamic):
        assert tx.data == b""  # None is not allowed.
        assert tx.value == 0  # None is same as 0.
        assert tx.chain_id == eth_tester_provider.chain_id
        assert tx.nonce is None

    assert static.gas_price is None
    assert dynamic.max_fee is None
    assert dynamic.max_priority_fee is None


@pytest.mark.parametrize("kwarg_name", ("max_fee", "max_fee_per_gas", "maxFee", "maxFeePerGas"))
def test_create_transaction_max_fee_per_gas(kwarg_name, ethereum):
    fee = 1_000_000_000
    tx = ethereum.create_transaction(**{kwarg_name: fee})
    assert tx.max_fee == fee


@pytest.mark.parametrize("kwarg_name", ("gas_price", "gasPrice"))
def test_create_transaction_with_gas_price(kwarg_name, ethereum):
    price = 123
    tx = ethereum.create_transaction(**{kwarg_name: price})
    assert tx.gas_price == price


@pytest.mark.parametrize(
    "tx_kwargs",
    [{"type": 3}, {"max_fee_per_blob_gas": 123}, {"blob_versioned_hashes": [HexBytes(123)]}],
)
def test_create_transaction_shared_blob(tx_kwargs, ethereum):
    tx = ethereum.create_transaction(**tx_kwargs)
    assert isinstance(tx, SharedBlobTransaction)
    assert tx.type == TransactionType.SHARED_BLOB.value


@pytest.mark.parametrize(
    "kwarg_name,value", [("max_fee_per_blob_gas", 345), ("maxFeePerBlobGas", "0x2343")]
)
def test_create_transaction_max_fee_per_blob_gas(kwarg_name, value, ethereum):
    tx = ethereum.create_transaction(**{kwarg_name: value})
    assert isinstance(tx, SharedBlobTransaction)
    expected = value if isinstance(value, int) else int(HexBytes(value).hex(), 16)
    assert tx.max_fee_per_blob_gas == expected


@pytest.mark.parametrize(
    "kwarg_name,value",
    [("blob_versioned_hashes", "0x123"), ("blobVersionedHashes", HexBytes("0x123"))],
)
def test_create_transaction_blob_versioned_hashed(kwarg_name, value, ethereum):
    tx = ethereum.create_transaction(**{kwarg_name: [value]})
    assert isinstance(tx, SharedBlobTransaction)
    assert tx.blob_versioned_hashes == [HexBytes(value)]


@pytest.mark.parametrize("tx_type", TransactionType)
def test_encode_transaction(tx_type, ethereum, vyper_contract_instance, owner, eth_tester_provider):
    abi = vyper_contract_instance.contract_type.methods[0]
    actual = ethereum.encode_transaction(
        vyper_contract_instance.address, abi, sender=owner.address, type=tx_type.value
    )
    assert actual.gas_limit == eth_tester_provider.max_gas


def test_set_default_network_not_exists(temp_config, ethereum):
    bad_network = "NOT_EXISTS"
    expected = f"No network in 'ethereum' named '{bad_network}'. Options:.*"
    with pytest.raises(NetworkNotFoundError, match=expected):
        ethereum.set_default_network(bad_network)


def test_networks(ethereum):
    actual = ethereum.networks
    for net in ("goerli", "sepolia", "mainnet", LOCAL_NETWORK_NAME):
        assert net in actual
        assert isinstance(actual[net], NetworkAPI)


def test_networks_includes_custom_networks(
    ethereum, custom_networks_config, custom_network_name_0, custom_network_name_1
):
    actual = ethereum.networks
    for net in (
        "goerli",
        "sepolia",
        "mainnet",
        LOCAL_NETWORK_NAME,
        custom_network_name_0,
        custom_network_name_1,
    ):
        assert net in actual
        assert isinstance(actual[net], NetworkAPI)


def test_networks_when_custom_ecosystem(custom_ecosystem_config, networks, custom_network_name_0):
    obj = networks.custom_ecosystem
    actual = obj.networks
    assert obj.name == CUSTOM_ECOSYSTEM_NAME
    assert custom_network_name_0 in actual
    assert "mainnet" not in actual


def test_networks_multiple_networks_with_same_name(
    temp_config, custom_networks_config_dict, ethereum
):
    data = copy.deepcopy(custom_networks_config_dict)
    data["networks"]["custom"][0]["name"] = "mainnet"  # There already is a mainnet in "ethereum".
    expected = ".*More than one network named 'mainnet' in ecosystem 'ethereum'.*"
    with temp_config(data):
        with pytest.raises(NetworkError, match=expected):
            _ = ethereum.networks


def test_getattr(ethereum):
    assert ethereum.mainnet.name == "mainnet"
    assert isinstance(ethereum.mainnet, NetworkAPI)


def test_getattr_custom_networks(ethereum, custom_networks_config, custom_network_name_0):
    actual = getattr(ethereum, custom_network_name_0)
    assert actual.name == custom_network_name_0
    assert isinstance(actual, NetworkAPI)


def test_default_network(ethereum):
    assert ethereum.default_network_name == LOCAL_NETWORK_NAME


def test_default_network_when_custom_and_set_in_config(
    custom_ecosystem_config, networks, custom_network_name_0
):
    ecosystem = networks.get_ecosystem(CUSTOM_ECOSYSTEM_NAME)
    # Force it to use config value (in case was set from previous test)
    ecosystem._default_network = None
    assert ecosystem.default_network_name == custom_network_name_0


def test_default_network_name_set_programmatically(ethereum):
    ethereum._default_network = "testnet"
    assert ethereum.default_network_name == "testnet"
    ethereum._default_network = None


def test_default_network_name_from_config(config, ethereum):
    orig = config._plugin_configs["ethereum"]
    data = {"default_network": "testnet"}
    config._plugin_configs["ethereum"] = EthereumConfig.model_validate(data)
    ethereum._default_network = None
    assert ethereum.default_network_name == "testnet"
    config._plugin_configs["ethereum"] = orig


def test_default_network_name_when_not_set_uses_local(config, ethereum):
    orig = config._plugin_configs["ethereum"]
    data = orig if isinstance(orig, dict) else orig.model_dump()
    data = {k: v for k, v in data.items() if k not in ("default_network",)}
    data["default_network"] = None
    config._plugin_configs["ethereum"] = data
    ethereum._default_network = None
    assert ethereum.default_network_name == LOCAL_NETWORK_NAME
    config._plugin_configs["ethereum"] = orig


def test_default_network_name_when_not_set_and_no_local_uses_only(mocker, config, ethereum):
    # Delete cache.
    if "_networks_from_plugins" in ethereum.__dict__:
        del ethereum.__dict__["_networks_from_plugins"]

    orig_eth = config._plugin_configs["ethereum"]
    orig_pm = ethereum.plugin_manager

    net_name = "onlynet"
    config._plugin_configs["ethereum"] = PluginConfig()
    mock_pm = mocker.MagicMock()
    mock_net = mocker.MagicMock()
    mock_net.name = net_name
    mock_pm.networks = ((None, ("ethereum", net_name, lambda *args, **kwargs: mock_net)),)
    Ethereum.plugin_manager = mock_pm
    ethereum._default_network = None

    try:
        assert ethereum.default_network_name == net_name
    finally:
        config._plugin_configs["ethereum"] = orig_eth
        Ethereum.plugin_manager = orig_pm
        if "_networks_from_plugins" in ethereum.__dict__:
            del ethereum.__dict__["_networks_from_plugins"]
