"""Unit tests for Loppers."""

from __future__ import annotations

import tempfile
import unittest
from pathlib import Path

from binaryornot.check import is_binary

from loppers import concatenate_files, extract_skeleton, find_files, get_skeleton
from loppers.loppers import SkeletonExtractor


class TestSkeletonExtractor(unittest.TestCase):
    """Test skeleton extraction."""

    def test_python_extraction(self) -> None:
        """Test Python skeleton extraction."""
        code: str = '''
def hello(name):
    """Greet someone."""
    print(f"Hello {name}")
    return True
'''
        skeleton: str = extract_skeleton(code, "python")
        self.assertIn("def hello", skeleton)
        self.assertIn("Greet someone", skeleton)
        self.assertNotIn("print", skeleton)

    def test_python_multiline_function(self) -> None:
        """Test multi-line Python function."""
        code: str = '''
def process(
    items: list,
    verbose: bool = False
) -> dict:
    result = {}
    for item in items:
        result[item] = process_item(item)
    return result
'''
        skeleton: str = extract_skeleton(code, "python")
        self.assertIn("def process", skeleton)
        self.assertNotIn("result = {}", skeleton)

    def test_language_not_supported(self) -> None:
        """Test unsupported language error."""
        with self.assertRaises(ValueError):
            SkeletonExtractor("cobol")

    def test_supported_languages(self) -> None:
        """Test all supported languages can initialize."""
        languages: list[str] = [
            "python",
            "javascript",
            "typescript",
            "java",
            "go",
            "rust",
            "cpp",
            "c",
            "csharp",
            "ruby",
            "php",
            "kotlin",
            "swift",
            "lua",
            "scala",
            "groovy",
            "objc",
        ]
        for lang in languages:
            try:
                extractor: SkeletonExtractor = SkeletonExtractor(lang)
                self.assertIsNotNone(extractor)
            except ImportError:
                # Skip if language not installed
                pass

    def test_javascript_arrow_functions(self) -> None:
        """Test JavaScript arrow function extraction."""
        code: str = '''
const add = (a, b) => {
    const result = a + b;
    return result;
};

const greet = (name) => {
    console.log(`Hello ${name}`);
    return true;
};

const concise = (x) => x * 2;
'''
        skeleton: str = extract_skeleton(code, "javascript")
        self.assertIn("const add = (a, b) =>", skeleton)
        self.assertIn("const greet = (name) =>", skeleton)
        self.assertNotIn("const result", skeleton)
        self.assertNotIn("console.log", skeleton)

    def test_java_constructor(self) -> None:
        """Test Java constructor extraction."""
        code: str = '''
public class User {
    private String name;
    
    public User(String name) {
        this.name = name;
        this.validate();
    }
    
    private void validate() {
        if (name == null) throw new Exception("Invalid");
    }
}
'''
        skeleton: str = extract_skeleton(code, "java")
        self.assertIn("public User(String name)", skeleton)
        self.assertIn("private void validate()", skeleton)
        self.assertNotIn("this.name = name", skeleton)
        self.assertNotIn("throw new Exception", skeleton)

    def test_python_with_underscore_methods(self) -> None:
        """Test Python __init__ and other dunder methods."""
        code: str = '''
class MyClass:
    def __init__(self, value):
        """Initialize."""
        self.value = value
        self._setup()
    
    def __str__(self):
        """String representation."""
        return f"MyClass({self.value})"
    
    def _setup(self):
        """Private setup method."""
        self.ready = True
'''
        skeleton: str = extract_skeleton(code, "python")
        self.assertIn("def __init__", skeleton)
        self.assertIn('"""Initialize."""', skeleton)
        self.assertIn("def __str__", skeleton)
        self.assertIn('"""String representation."""', skeleton)
        self.assertIn("def _setup", skeleton)
        self.assertNotIn("self.value = value", skeleton)
        self.assertNotIn("self.ready = True", skeleton)

    def test_java_lambda(self) -> None:
        """Test Java lambda expressions."""
        code: str = '''
public class Example {
    public void test() {
        Function<Integer, Integer> doubler = x -> x * 2;
        Runnable r = () -> System.out.println("Hello");
    }
}
'''
        skeleton: str = extract_skeleton(code, "java")
        self.assertIn("public void test()", skeleton)
        self.assertNotIn("System.out.println", skeleton)

    def test_csharp_properties_and_lambdas(self) -> None:
        """Test C# properties and lambda expressions."""
        code: str = '''
public class User {
    public string Name { get; set; }
    
    public int Age { 
        get { return _age; }
        set { _age = value; }
    }
    
    public void Process() {
        var result = items.Where(x => x.Value > 10);
        var anon = delegate(int x) { return x * 2; };
    }
}
'''
        skeleton: str = extract_skeleton(code, "csharp")
        self.assertIn("public void Process()", skeleton)
        self.assertNotIn("return x * 2", skeleton)

    def test_rust_closures(self) -> None:
        """Test Rust closures."""
        code: str = '''
fn main() {
    let add = |a, b| a + b;
    let result = add(5, 3);
    
    let expensive_closure = |num| {
        let expensive_result = num * num;
        expensive_result + 1
    };
}
'''
        skeleton: str = extract_skeleton(code, "rust")
        self.assertIn("fn main()", skeleton)
        # Closures should have bodies removed

    def test_cpp_lambdas(self) -> None:
        """Test C++ lambda expressions."""
        code: str = '''
void process() {
    auto add = [](int a, int b) { return a + b; };
    
    std::vector<int> v = {1, 2, 3};
    std::sort(v.begin(), v.end(), 
        [](int a, int b) { return a > b; });
}
'''
        skeleton: str = extract_skeleton(code, "cpp")
        self.assertIn("void process()", skeleton)

    def test_ruby_methods_and_blocks(self) -> None:
        """Test Ruby methods and blocks."""
        code: str = '''
class Calculator
  def add(a, b)
    result = a + b
    result
  end
  
  def self.multiply(a, b)
    a * b
  end
  
  def process
    [1, 2, 3].each do |x|
      puts x * 2
    end
  end
end
'''
        skeleton: str = extract_skeleton(code, "ruby")
        self.assertIn("def add", skeleton)
        self.assertIn("def self.multiply", skeleton)
        self.assertNotIn("result = a + b", skeleton)

    def test_php_anonymous_functions(self) -> None:
        """Test PHP anonymous and arrow functions."""
        code: str = '''
<?php
class Service {
    public function process($data) {
        $callback = function($item) {
            return $item * 2;
        };

        $arrow = fn($x) => $x + 1;

        return array_map($callback, $data);
    }
}
'''
        skeleton: str = extract_skeleton(code, "php")
        self.assertIn("public function process", skeleton)
        self.assertNotIn("return $item * 2", skeleton)

    def test_swift_functions_and_methods(self) -> None:
        """Test Swift function and method extraction."""
        code: str = '''
func greet(name: String) -> String {
    return "Hello, \\(name)!"
}

class Greeter {
    func sayHello() {
        print("Hello!")
    }

    func sayGoodbye(name: String) {
        print("Goodbye, \\(name)!")
    }
}
'''
        skeleton: str = extract_skeleton(code, "swift")
        self.assertIn("func greet(name: String) -> String", skeleton)
        self.assertIn("func sayHello()", skeleton)
        self.assertIn("func sayGoodbye(name: String)", skeleton)
        self.assertNotIn("print", skeleton)
        self.assertNotIn("return", skeleton)

    def test_lua_functions(self) -> None:
        """Test Lua function extraction."""
        code: str = '''
function greet(name)
    local greeting = "Hello " .. name
    print(greeting)
    return greeting
end

function sayGoodbye()
    print("Goodbye")
end
'''
        skeleton: str = extract_skeleton(code, "lua")
        self.assertIn("function greet(name)", skeleton)
        self.assertIn("function sayGoodbye()", skeleton)
        self.assertNotIn("local greeting", skeleton)
        self.assertNotIn("print", skeleton)

    def test_scala_functions_and_methods(self) -> None:
        """Test Scala function and method extraction."""
        code: str = '''
def greet(name: String): String = {
    val greeting = "Hello " + name
    println(greeting)
    greeting
}

class Greeter {
    def sayHello(): Unit = {
        println("Hello!")
    }
}
'''
        skeleton: str = extract_skeleton(code, "scala")
        self.assertIn("def greet(name: String): String =", skeleton)
        self.assertIn("def sayHello(): Unit =", skeleton)
        self.assertNotIn("val greeting", skeleton)
        self.assertNotIn("println", skeleton)

    def test_groovy_functions(self) -> None:
        """Test Groovy function extraction."""
        code: str = '''
def greet(name) {
    def greeting = "Hello " + name
    println(greeting)
    return greeting
}

class Greeter {
    def sayHello() {
        println("Hello!")
    }
}
'''
        skeleton: str = extract_skeleton(code, "groovy")
        self.assertIn("def greet(name)", skeleton)
        self.assertIn("def sayHello()", skeleton)
        self.assertNotIn("def greeting", skeleton)
        self.assertNotIn("println", skeleton)

    def test_objective_c_methods(self) -> None:
        """Test Objective-C method extraction."""
        code: str = '''
- (NSString *)greet:(NSString *)name {
    NSString *greeting = @"Hello";
    NSLog(@"%@", greeting);
    return greeting;
}

- (void)printMessage {
    NSLog(@"Message");
}
'''
        skeleton: str = extract_skeleton(code, "objc")
        self.assertIn("- (NSString *)greet:(NSString *)name", skeleton)
        self.assertIn("- (void)printMessage", skeleton)
        self.assertNotIn("NSString *greeting", skeleton)
        self.assertNotIn("NSLog", skeleton)

    def test_kotlin_functions_and_properties(self) -> None:
        """Test Kotlin function and property extraction."""
        code: str = '''
fun greet(name: String): String {
    return "Hello, $name"
}

class User {
    private var _age: Int = 0

    var age: Int
        get() {
            return _age
        }
        set(value) {
            _age = value
        }

    fun validate() {
        if (_age < 0) throw IllegalArgumentException("Invalid age")
    }

    fun getName(): String {
        return "User"
    }
}
'''
        skeleton: str = extract_skeleton(code, "kotlin")
        self.assertIn("fun greet(name: String): String", skeleton)
        self.assertIn("fun validate()", skeleton)
        self.assertIn("fun getName(): String", skeleton)
        self.assertIn("var age: Int", skeleton)
        self.assertNotIn('return "Hello, $name"', skeleton)
        self.assertNotIn("throw IllegalArgumentException", skeleton)


class TestBinaryFileDetection(unittest.TestCase):
    """Test binary file detection."""

    def test_text_file_not_binary(self) -> None:
        """Test that text files are not detected as binary."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".txt", delete=False) as f:
            f.write("Hello, this is a text file\n")
            f.flush()
            path = Path(f.name)
            try:
                self.assertFalse(is_binary(str(path)))
            finally:
                path.unlink()

    def test_binary_extension_detected(self) -> None:
        """Test that files with known binary extensions are detected."""
        # binaryornot detects based on known extensions and null bytes
        # Test with a known binary extension + null byte content
        with tempfile.NamedTemporaryFile(suffix=".pyc", delete=False, mode="wb") as f:
            f.write(b"Python compiled\x00binary\x00content")
            f.flush()
            path = Path(f.name)
            try:
                self.assertTrue(is_binary(str(path)))
            finally:
                path.unlink()

    def test_null_byte_detection(self) -> None:
        """Test that files with null bytes are detected as binary."""
        with tempfile.NamedTemporaryFile(mode="wb", suffix=".txt", delete=False) as f:
            # Files with null bytes fail UTF-8 decoding
            f.write(b"Some text\x00with null bytes")
            f.flush()
            path = Path(f.name)
            try:
                self.assertTrue(is_binary(str(path)))
            finally:
                path.unlink()

    def test_python_file_not_binary(self) -> None:
        """Test that Python files are not detected as binary."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".py", delete=False) as f:
            f.write("def hello():\n    print('hello')\n")
            f.flush()
            path = Path(f.name)
            try:
                self.assertFalse(is_binary(str(path)))
            finally:
                path.unlink()

    def test_empty_file_not_binary(self) -> None:
        """Test that empty files are not detected as binary."""
        with tempfile.NamedTemporaryFile(delete=False) as f:
            f.flush()
            path = Path(f.name)
            try:
                self.assertFalse(is_binary(str(path)))
            finally:
                path.unlink()

    def test_image_extension_detected(self) -> None:
        """Test that binary image files with null bytes are detected."""
        # binaryornot detects binary content via null bytes
        test_cases = [
            (".jpg", b"\xFF\xD8\xFF\xE0\x00image"),  # JPEG with null byte
            (".png", b"\x89PNG\r\n\x1a\n\x00data"),  # PNG with null byte
            (".gif", b"GIF89a\x00binary"),  # GIF with null byte
        ]
        for ext, content in test_cases:
            with tempfile.NamedTemporaryFile(suffix=ext, delete=False, mode="wb") as f:
                f.write(content)
                f.flush()
                path = Path(f.name)
                try:
                    self.assertTrue(is_binary(str(path)), f"{ext} should be detected as binary")
                finally:
                    path.unlink()


class TestFindFilesWithMixedTypes(unittest.TestCase):
    """Test find_files returns all non-binary text files."""

    def test_find_files_includes_non_code_files(self) -> None:
        """Test that find_files includes markdown, json, yaml, etc."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create various file types
            (tmppath / "readme.md").write_text("# Test\n")
            (tmppath / "config.json").write_text('{"key": "value"}\n')
            (tmppath / "data.yaml").write_text("key: value\n")
            (tmppath / "script.py").write_text("def hello():\n    pass\n")

            # Find files
            files = find_files(tmppath, recursive=False)

            # All text files should be included
            self.assertIn("readme.md", files)
            self.assertIn("config.json", files)
            self.assertIn("data.yaml", files)
            self.assertIn("script.py", files)

    def test_find_files_excludes_binary_files(self) -> None:
        """Test that find_files excludes binary files."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create text and binary files
            (tmppath / "text.txt").write_text("text content\n")
            (tmppath / "binary.bin").write_bytes(b"\x00\x01\x02\x03")

            files = find_files(tmppath, recursive=False)

            self.assertIn("text.txt", files)
            self.assertNotIn("binary.bin", files)


class TestMixedFileTypes(unittest.TestCase):
    """Test mixed file type handling."""

    def test_get_skeleton_on_unsupported_file_as_is(self) -> None:
        """Test that unsupported files can't be extracted without language."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".json", delete=False) as f:
            f.write('{"key": "value"}\n')
            f.flush()
            path = Path(f.name)

            try:
                # Should raise ValueError for unsupported file type
                with self.assertRaises(ValueError):
                    get_skeleton(path)
            finally:
                path.unlink()


class TestExtractWithUnsupportedFiles(unittest.TestCase):
    """Test extract command behavior with unsupported files."""

    def test_extract_requires_language_for_unsupported(self) -> None:
        """Test that extract requires -l for unsupported file types."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".md", delete=False) as f:
            f.write("# Test\n")
            f.flush()
            path = Path(f.name)

            try:
                # Should raise ValueError for unsupported file type
                with self.assertRaises(ValueError):
                    get_skeleton(path)
            finally:
                path.unlink()

    def test_extract_with_explicit_language(self) -> None:
        """Test that extract works with explicit language for any file."""
        with tempfile.NamedTemporaryFile(mode="w", suffix=".md", delete=False) as f:
            f.write("def hello():\n    print('hi')\n")
            f.flush()
            path = Path(f.name)

            try:
                # With explicit language, should work
                skeleton = extract_skeleton(path.read_text(), "python")
                self.assertIn("def hello()", skeleton)
                self.assertNotIn("print('hi')", skeleton)
            finally:
                path.unlink()


class TestIgnorePatternsWithMixedFiles(unittest.TestCase):
    """Test ignore patterns work correctly with all file types."""

    def test_default_ignore_patterns_apply(self) -> None:
        """Test that default ignore patterns apply to all file types."""
        with tempfile.TemporaryDirectory() as tmpdir:
            tmppath = Path(tmpdir)

            # Create node_modules with various files
            node_modules = tmppath / "node_modules"
            node_modules.mkdir()
            (node_modules / "package.json").write_text("{}\n")
            (node_modules / "module.js").write_text("module.exports = {};\n")

            # Create regular files
            (tmppath / "main.js").write_text("console.log('hi');\n")

            files = find_files(
                tmppath,
                recursive=True,
                use_default_ignore=True,
            )

            # Files in node_modules should be excluded
            self.assertIn("main.js", files)
            self.assertNotIn("node_modules/package.json", files)
            self.assertNotIn("node_modules/module.js", files)


class TestConcatenateFiles(unittest.TestCase):
    """Test file concatenation."""

    def test_concatenate_files_with_extraction(self) -> None:
        """Test concatenating files with skeleton extraction."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)

            # Create test files
            py_file = root / "script.py"
            py_file.write_text(
                "def hello():\n"
                '    """Say hello."""\n'
                "    print('Hello')\n"
            )

            js_file = root / "app.js"
            js_file.write_text(
                "function greet() {\n"
                "    console.log('Hi');\n"
                "}\n"
            )

            # Concatenate with extraction
            result = concatenate_files(root, ["script.py", "app.js"], extract=True)

            # Check headers
            self.assertIn("--- script.py", result)
            self.assertIn("--- app.js", result)

            # Check skeleton extraction (no body)
            self.assertIn("def hello", result)
            self.assertNotIn("print", result)
            self.assertIn("function greet", result)
            self.assertNotIn("console.log", result)

    def test_concatenate_files_without_extraction(self) -> None:
        """Test concatenating files without skeleton extraction."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)

            py_file = root / "script.py"
            py_file.write_text(
                "def hello():\n"
                "    print('Hello')\n"
            )

            # Concatenate without extraction
            result = concatenate_files(root, ["script.py"], extract=False)

            # Check full content is preserved
            self.assertIn("print", result)
            self.assertIn("def hello", result)

    def test_concatenate_files_not_found_strict(self) -> None:
        """Test error when file not found with ignore_not_found=False."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)
            with self.assertRaises(FileNotFoundError):
                concatenate_files(root, ["missing.py"], ignore_not_found=False)

    def test_concatenate_files_not_found_ignore(self) -> None:
        """Test ignoring missing files with ignore_not_found=True."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)

            # Create one valid file
            py_file = root / "script.py"
            py_file.write_text("def hello():\n    pass\n")

            # Concatenate with missing file, ignoring not found
            result = concatenate_files(
                root, ["script.py", "missing.py"], ignore_not_found=True
            )

            # Should only include the found file
            self.assertIn("--- script.py", result)
            self.assertNotIn("missing.py", result)

    def test_concatenate_empty_paths(self) -> None:
        """Test error when no paths provided."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)
            with self.assertRaises(ValueError):
                concatenate_files(root, [])

    def test_concatenate_invalid_root(self) -> None:
        """Test error when root does not exist."""
        with self.assertRaises(FileNotFoundError):
            concatenate_files("/nonexistent/path", ["file.py"])

    def test_concatenate_root_not_directory(self) -> None:
        """Test error when root is not a directory."""
        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)
            file_path = root / "file.txt"
            file_path.write_text("content")

            with self.assertRaises(NotADirectoryError):
                concatenate_files(file_path, ["file.py"])


class TestFileSizeFormatting(unittest.TestCase):
    """Test file size formatting utility."""

    def test_format_bytes(self):
        """Test formatting of byte sizes."""
        from loppers.source_utils import format_file_size

        self.assertEqual(format_file_size(0), "0B")
        self.assertEqual(format_file_size(100), "100B")
        self.assertEqual(format_file_size(1023), "1023B")

    def test_format_kilobytes(self):
        """Test formatting of kilobyte sizes."""
        from loppers.source_utils import format_file_size

        self.assertEqual(format_file_size(1024), "1.0KB")
        self.assertEqual(format_file_size(2048), "2.0KB")
        self.assertEqual(format_file_size(1536), "1.5KB")  # 1.5KB

    def test_format_megabytes(self):
        """Test formatting of megabyte sizes."""
        from loppers.source_utils import format_file_size

        kb = 1024
        mb = kb * 1024
        self.assertEqual(format_file_size(mb), "1.0MB")
        self.assertEqual(format_file_size(mb * 2), "2.0MB")
        self.assertEqual(format_file_size(mb * 5 + kb * 300), "5.3MB")

    def test_format_gigabytes(self):
        """Test formatting of gigabyte sizes."""
        from loppers.source_utils import format_file_size

        kb = 1024
        mb = kb * 1024
        gb = mb * 1024
        self.assertEqual(format_file_size(gb), "1.0GB")
        self.assertEqual(format_file_size(gb * 2), "2.0GB")
        self.assertEqual(format_file_size(gb + mb * 256), "1.2GB")


class TestTreeWithSizes(unittest.TestCase):
    """Test tree display with file sizes."""

    def test_tree_as_str_with_sizes(self):
        """Test tree_as_str with file sizes."""
        from loppers.source_utils import tree_as_str

        paths = ["file1.py", "dir1/file2.py", "dir1/file3.txt"]
        file_sizes = {
            "file1.py": 1024,
            "dir1/file2.py": 2048,
            "dir1/file3.txt": 512,
        }
        tree = tree_as_str(paths, file_sizes=file_sizes)
        self.assertIn("file1.py  (1.0KB)", tree)
        self.assertIn("file2.py  (2.0KB)", tree)
        self.assertIn("file3.txt  (512B)", tree)

    def test_tree_as_str_without_sizes(self):
        """Test tree_as_str without file sizes."""
        from loppers.source_utils import tree_as_str

        paths = ["file1.py", "dir1/file2.py"]
        tree = tree_as_str(paths, file_sizes=None)
        self.assertIn("file1.py", tree)
        self.assertIn("file2.py", tree)
        # Should not have size info
        self.assertNotIn("(", tree) or "." not in tree

    def test_tree_as_str_with_collapse_and_sizes(self):
        """Test tree_as_str with both collapse and sizes."""
        from loppers.source_utils import tree_as_str

        paths = ["main/java/com/example/Source.java"]
        file_sizes = {"main/java/com/example/Source.java": 5120}
        tree = tree_as_str(paths, collapse_single_dirs=True, file_sizes=file_sizes)
        # Should collapse the nested path
        self.assertIn("main/java/com/example", tree)
        # Should show size
        self.assertIn("(5.0KB)", tree)

    def test_get_tree_with_show_sizes(self):
        """Test get_tree API with show_sizes parameter."""
        from loppers import get_tree

        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)
            file1 = root / "file1.py"
            file1.write_text("x" * 100)  # 100 bytes
            file2 = root / "file2.txt"
            file2.write_text("y" * 2048)  # 2048 bytes

            # With sizes
            tree_with_sizes = get_tree(root, show_sizes=True)
            self.assertIn("file1.py", tree_with_sizes)
            self.assertIn("file2.txt", tree_with_sizes)
            self.assertIn("(100B)", tree_with_sizes)
            self.assertIn("(2.0KB)", tree_with_sizes)

            # Without sizes (default)
            tree_no_sizes = get_tree(root, show_sizes=False)
            self.assertIn("file1.py", tree_no_sizes)
            self.assertNotIn("(100B)", tree_no_sizes)

    def test_get_tree_with_sizes_and_collapse(self):
        """Test get_tree with both show_sizes and collapse_single_dirs."""
        from loppers import get_tree

        with tempfile.TemporaryDirectory() as tmpdir:
            root = Path(tmpdir)
            deep_dir = root / "a" / "b" / "c"
            deep_dir.mkdir(parents=True)
            file = deep_dir / "file.py"
            file.write_text("x" * 1024)

            tree = get_tree(
                root,
                show_sizes=True,
                collapse_single_dirs=True,
            )
            # Should collapse the path
            self.assertIn("a/b/c", tree)
            # Should show size
            self.assertIn("(1.0KB)", tree)


if __name__ == "__main__":
    unittest.main()
