"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DailyCloudWatchLogsArchiveStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_secure_bucket_1 = require("@gammarers/aws-secure-bucket");
const aws_secure_log_bucket_1 = require("@gammarers/aws-secure-log-bucket");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const scheduler = require("aws-cdk-lib/aws-scheduler");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const log_archiver_function_1 = require("./funcs/log-archiver-function");
class DailyCloudWatchLogsArchiveStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        // 👇 Get current account & region
        //const account = cdk.Stack.of(this).account;
        //const stackName: string = cdk.Stack.of(this).stackName;
        //const region = cdk.Stack.of(this).region;
        //const account = this.account;
        const region = this.region;
        const randomNameKey = crypto.createHash('shake256', { outputLength: 4 })
            .update(cdk.Names.uniqueId(this))
            .digest('hex');
        // 👇 Create Backup S3 Bucket
        const logArchiveBucket = new aws_secure_log_bucket_1.SecureLogBucket(this, 'LogArchiveBucket', {
            bucketName: `log-archive-${randomNameKey}`,
            encryption: aws_secure_bucket_1.SecureBucketEncryption.S3_MANAGED,
        });
        logArchiveBucket.addToResourcePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal(`logs.${region}.amazonaws.com`),
            ],
            actions: [
                's3:GetBucketAcl',
            ],
            resources: [
                logArchiveBucket.bucketArn,
            ],
        }));
        logArchiveBucket.addToResourcePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal(`logs.${region}.amazonaws.com`),
            ],
            actions: [
                's3:PutObject',
            ],
            resources: [
                `${logArchiveBucket.bucketArn}/*`,
            ],
            conditions: {
                StringEquals: {
                    's3:x-amz-acl': 'bucket-owner-full-control',
                },
            },
        }));
        // 👇 Create Lambda Execution role.
        const lambdaExecutionRole = new iam.Role(this, 'LambdaExecutionRole', {
            roleName: `daily-cw-logs-archive-lambda-exec-${randomNameKey}-role`,
            description: 'daily CloudWatch Logs archive machine exec role.',
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                ['log-export-policy']: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                'logs:CreateExportTask',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
                ['put-bucket-policy']: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                's3:GetBucketAcl',
                                's3:PutObject',
                            ],
                            resources: [
                                logArchiveBucket.bucketArn,
                            ],
                        }),
                    ],
                }),
            },
        });
        // 👇 Create Lambda Function
        const lambdaFunction = new log_archiver_function_1.LogArchiverFunction(this, 'LogArchiveFunction', {
            functionName: `daily-cw-logs-archive-${randomNameKey}-func`,
            description: 'A function to archive logs s3 bucket from CloudWatch Logs.',
            environment: {
                BUCKET_NAME: logArchiveBucket.bucketName,
            },
            role: lambdaExecutionRole,
        });
        // 👇 Create Lambda Function Log Group
        new logs.LogGroup(this, 'LambdaFunctionLogGroup', {
            // logGroupName: lambdaFunction.logGroup.logGroupName, // <- If you specify this line to Custom:LogRotation resource created.
            logGroupName: `/aws/lambda/${lambdaFunction.functionName}`,
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const succeed = new sfn.Succeed(this, 'Succeed');
        // 👇 Get CloudWatch Logs Resources
        const getLogGroupResources = new tasks.CallAwsService(this, 'GetResources', {
            iamResources: ['*'],
            iamAction: 'tag:GetResources',
            service: 'resourcegroupstaggingapi',
            action: 'getResources',
            parameters: {
                ResourceTypeFilters: [
                    'logs:log-group',
                ],
                TagFilters: [
                    {
                        'Key.$': '$.tagKey',
                        'Values.$': '$.tagValues',
                    },
                ],
            },
            resultPath: '$.Result',
            resultSelector: {
                'TargetResources.$': '$..ResourceTagMappingList[*].ResourceARN',
            },
        });
        // Log Group Export Map
        const logGroupExportMap = new sfn.Map(this, 'LogGroupExportMap', {
            itemsPath: sfn.JsonPath.stringAt('$.Result.TargetResources'),
            maxConcurrency: 1,
        });
        // 👇 Get Log Group Name
        const getLogGroupName = new sfn.Pass(this, 'GetLogGroupName', {
            parameters: {
                'TargetLogGroupName.$': "States.ArrayGetItem(States.StringSplit($, ':'), 6)",
            },
        });
        logGroupExportMap.iterator(getLogGroupName);
        // 👇 Invoke Lambda Function
        const invokeLambdaFunction = new tasks.LambdaInvoke(this, 'InvokeLambdaFunction', {
            lambdaFunction: lambdaFunction,
            outputPath: '$.Payload',
            payload: sfn.TaskInput.fromJsonPathAt('$'),
            retryOnServiceExceptions: true,
        });
        getLogGroupName.next(invokeLambdaFunction);
        // 👇 Describe Export Tasks
        const describeExportTasks = new tasks.CallAwsService(this, 'DescribeExportTasks', {
            iamResources: ['*'],
            iamAction: 'logs:DescribeExportTasks',
            service: 'cloudwatchlogs',
            action: 'describeExportTasks',
            parameters: {
                'TaskId.$': '$.TaskId',
            },
            resultPath: '$.Result',
            resultSelector: {
                'DescribeExportTasksStatus.$': '$.ExportTasks[0].Status.Code',
            },
        });
        invokeLambdaFunction.next(describeExportTasks);
        const exportRunningWait = new sfn.Wait(this, 'ExportRunningWait', {
            time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(10)),
        });
        const exportPendingWait = new sfn.Wait(this, 'ExportPendingWait', {
            time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(3)),
        });
        // 👇 Export Status Check
        const exportTaskStatusCheck = new sfn.Choice(this, 'ExportTaskStatusCheck')
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'FAILED'), getLogGroupName)
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'RUNNING'), exportRunningWait
            .next(describeExportTasks))
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'PENDING'), exportPendingWait
            .next(describeExportTasks))
            .otherwise(succeed);
        describeExportTasks.next(exportTaskStatusCheck);
        getLogGroupResources.next(logGroupExportMap);
        //
        const machine = new sfn.StateMachine(this, 'StateMachine', {
            stateMachineName: `daily-cw-logs-archive-${randomNameKey}-machine`,
            definition: getLogGroupResources,
        });
        // 👇 auto generated role name & description renaming.
        const role = machine.node.findChild('Role');
        const cfnRole = role.node.defaultChild;
        cfnRole.addPropertyOverride('RoleName', `daily-cw-logs-archive-machine-${randomNameKey}-role`);
        cfnRole.addPropertyOverride('Description', 'daily CloudWatch Logs archive machine role.');
        const policy = role.node.findChild('DefaultPolicy');
        const cfnPolicy = policy.node.defaultChild;
        cfnPolicy.addPropertyOverride('PolicyName', `daily-cw-logs-archive-machine-${randomNameKey}-default-policy`);
        // 👇 EventBridge Scheduler IAM Role (StateMachine Start Execution)
        const schedulerExecutionRole = new iam.Role(this, 'SchedulerExecutionRole', {
            roleName: `daily-cw-logs-archive-${randomNameKey}-schedule-role`,
            description: 'daily CloudWatch Log archive schedule',
            assumedBy: new iam.ServicePrincipal('scheduler.amazonaws.com'),
            inlinePolicies: {
                'state-machine-exec-policy': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                'states:StartExecution',
                            ],
                            resources: [
                                machine.stateMachineArn,
                            ],
                        }),
                    ],
                }),
            },
        });
        // 👇 Schedule
        new scheduler.CfnSchedule(this, 'Schedule', {
            name: `daily-cw-logs-archive-${randomNameKey}-schedule`,
            description: 'daily CloudWatch Logs archive schedule',
            state: 'ENABLED',
            flexibleTimeWindow: {
                mode: 'OFF',
            },
            scheduleExpressionTimezone: 'UTC',
            scheduleExpression: 'cron(1 13 * * ? *)',
            target: {
                arn: machine.stateMachineArn,
                roleArn: schedulerExecutionRole.roleArn,
                input: JSON.stringify({
                    tagKey: props.targetResourceTag.key,
                    tagValues: props.targetResourceTag.values,
                }),
                retryPolicy: {
                    maximumEventAgeInSeconds: 60,
                    maximumRetryAttempts: 0,
                },
            },
        });
    }
}
exports.DailyCloudWatchLogsArchiveStack = DailyCloudWatchLogsArchiveStack;
_a = JSII_RTTI_SYMBOL_1;
DailyCloudWatchLogsArchiveStack[_a] = { fqn: "@gammarers/aws-daily-cloud-watch-logs-archive-stack.DailyCloudWatchLogsArchiveStack", version: "2.7.9" };
//# sourceMappingURL=data:application/json;base64,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