"""
Base Processor Classes for Edge v2

Provides the foundation for all data processors in the edge system.
Combines the v1 processor functionality with v2 enhancements.
"""
from __future__ import annotations

import asyncio
import logging
import time
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from typing import Any, Dict, List, Optional, Callable
from enum import Enum

logger = logging.getLogger(__name__)

class ProcessorPriority(Enum):
    """Processor priority levels."""
    CRITICAL = 10      # Safety, emergency systems
    HIGH = 50          # Real-time control, teleoperation  
    NORMAL = 100       # Computer vision, analytics
    LOW = 200          # Background processing, logging

@dataclass
class ProcessorConfig:
    """Configuration for a processor."""
    name: str
    enabled: bool = True
    priority: int = ProcessorPriority.NORMAL.value
    config: Dict[str, Any] = field(default_factory=dict)
    max_memory_mb: Optional[int] = None
    max_cpu_percent: Optional[float] = None
    timeout_seconds: Optional[float] = None

@dataclass
class EdgeEvent:
    """Event generated by edge processors."""
    timestamp: float
    event_type: str
    source: str
    confidence: float
    data: Dict[str, Any]
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert to dictionary for serialization."""
        return {
            "timestamp": self.timestamp,
            "event_type": self.event_type,
            "source": self.source,
            "confidence": self.confidence,
            "data": self.data
        }

class BaseProcessor(ABC):
    """
    Enhanced base class for all edge processors.
    
    Combines v1 processor functionality with v2 architecture enhancements:
    - Resource management and limits
    - Priority-based scheduling
    - Health monitoring and diagnostics
    - Async/await support throughout
    - Event-driven architecture
    """
    
    def __init__(self, name: str, config: Dict[str, Any]):
        self.name = name
        self.config = config
        self.enabled = config.get("enabled", True)
        self.priority = config.get("priority", ProcessorPriority.NORMAL.value)
        
        # Resource limits
        self.max_memory_mb = config.get("max_memory_mb")
        self.max_cpu_percent = config.get("max_cpu_percent") 
        self.timeout_seconds = config.get("timeout_seconds", 30.0)
        
        # State tracking
        self.initialized = False
        self.running = False
        self.error_count = 0
        self.last_error: Optional[str] = None
        self.process_count = 0
        self.last_process_time = 0.0
        
        # Performance metrics
        self.total_processing_time = 0.0
        self.average_processing_time = 0.0
        self.max_processing_time = 0.0
        
        # Event callbacks
        self.event_callbacks: List[Callable[[EdgeEvent], None]] = []
        
        # Health status
        self.health_status = "unknown"
        self.last_health_check = 0.0
        
        logger.info(f"Processor '{name}' created with priority {self.priority}")

    async def initialize(self) -> None:
        """Initialize the processor (optional override)."""
        self.initialized = True
        self.health_status = "healthy"
        logger.info(f"Processor '{self.name}' initialized")

    @abstractmethod
    async def process(self, data: Any) -> List[EdgeEvent]:
        """
        Process input data and return events.
        
        Args:
            data: Input data to process
            
        Returns:
            List of events generated from processing
        """
        pass

    async def process_with_monitoring(self, data: Any) -> List[EdgeEvent]:
        """
        Process data with performance monitoring and error handling.
        
        This wrapper provides:
        - Performance timing
        - Error handling and recovery
        - Resource monitoring
        - Health status updates
        """
        if not self.enabled or not self.initialized:
            return []
        
        start_time = time.time()
        events = []
        
        try:
            # Check timeout
            if self.timeout_seconds:
                events = await asyncio.wait_for(
                    self.process(data),
                    timeout=self.timeout_seconds
                )
            else:
                events = await self.process(data)
            
            # Update success metrics
            processing_time = time.time() - start_time
            self._update_performance_metrics(processing_time)
            
            # Emit events to callbacks
            for event in events:
                await self._emit_event(event)
            
            self.health_status = "healthy"
            
        except asyncio.TimeoutError:
            self.error_count += 1
            self.last_error = f"Processing timeout after {self.timeout_seconds}s"
            self.health_status = "timeout"
            logger.error(f"Processor '{self.name}' timed out")
            
        except Exception as e:
            self.error_count += 1
            self.last_error = str(e)
            self.health_status = "error"
            logger.error(f"Processor '{self.name}' error: {e}")
        
        finally:
            self.process_count += 1
            self.last_process_time = time.time()
        
        return events

    def _update_performance_metrics(self, processing_time: float) -> None:
        """Update performance metrics."""
        self.total_processing_time += processing_time
        self.average_processing_time = self.total_processing_time / self.process_count
        self.max_processing_time = max(self.max_processing_time, processing_time)

    async def _emit_event(self, event: EdgeEvent) -> None:
        """Emit event to registered callbacks."""
        for callback in self.event_callbacks:
            try:
                if asyncio.iscoroutinefunction(callback):
                    await callback(event)
                else:
                    callback(event)
            except Exception as e:
                logger.error(f"Event callback error in '{self.name}': {e}")

    def add_event_callback(self, callback: Callable[[EdgeEvent], None]) -> None:
        """Add event callback."""
        self.event_callbacks.append(callback)

    def remove_event_callback(self, callback: Callable[[EdgeEvent], None]) -> None:
        """Remove event callback."""
        if callback in self.event_callbacks:
            self.event_callbacks.remove(callback)

    async def health_check(self) -> Dict[str, Any]:
        """Perform health check and return status."""
        self.last_health_check = time.time()
        
        # Check resource usage if limits are set
        resource_status = await self._check_resources()
        
        health_data = {
            "name": self.name,
            "enabled": self.enabled,
            "initialized": self.initialized,
            "running": self.running,
            "health_status": self.health_status,
            "error_count": self.error_count,
            "last_error": self.last_error,
            "process_count": self.process_count,
            "last_process_time": self.last_process_time,
            "performance": {
                "average_processing_time": self.average_processing_time,
                "max_processing_time": self.max_processing_time,
                "total_processing_time": self.total_processing_time
            },
            "resources": resource_status,
            "priority": self.priority
        }
        
        return health_data

    async def _check_resources(self) -> Dict[str, Any]:
        """Check resource usage against limits."""
        resource_status = {
            "memory_ok": True,
            "cpu_ok": True,
            "memory_usage_mb": 0,
            "cpu_usage_percent": 0
        }
        
        try:
            import psutil
            import os
            
            process = psutil.Process(os.getpid())
            
            # Memory check
            memory_info = process.memory_info()
            memory_mb = memory_info.rss / (1024 * 1024)
            resource_status["memory_usage_mb"] = memory_mb
            
            if self.max_memory_mb and memory_mb > self.max_memory_mb:
                resource_status["memory_ok"] = False
                self.health_status = "resource_limit"
            
            # CPU check (average over short interval)
            cpu_percent = process.cpu_percent(interval=0.1)
            resource_status["cpu_usage_percent"] = cpu_percent
            
            if self.max_cpu_percent and cpu_percent > self.max_cpu_percent:
                resource_status["cpu_ok"] = False
                self.health_status = "resource_limit"
                
        except ImportError:
            # psutil not available, skip resource checking
            pass
        except Exception as e:
            logger.debug(f"Resource check failed for '{self.name}': {e}")
        
        return resource_status

    async def start(self) -> None:
        """Start the processor."""
        if not self.initialized:
            await self.initialize()
        
        self.running = True
        self.health_status = "running"
        logger.info(f"Processor '{self.name}' started")

    async def stop(self) -> None:
        """Stop the processor."""
        self.running = False
        self.health_status = "stopped"
        logger.info(f"Processor '{self.name}' stopped")

    async def cleanup(self) -> None:
        """Cleanup resources (optional override)."""
        self.running = False
        self.initialized = False
        self.health_status = "cleanup"
        logger.info(f"Processor '{self.name}' cleaned up")

    def enable(self) -> None:
        """Enable the processor."""
        self.enabled = True
        logger.info(f"Processor '{self.name}' enabled")

    def disable(self) -> None:
        """Disable the processor."""
        self.enabled = False
        logger.info(f"Processor '{self.name}' disabled")

    def get_status(self) -> Dict[str, Any]:
        """Get processor status (synchronous version)."""
        return {
            "name": self.name,
            "enabled": self.enabled,
            "initialized": self.initialized,
            "running": self.running,
            "health_status": self.health_status,
            "error_count": self.error_count,
            "process_count": self.process_count,
            "priority": self.priority
        }

    def __repr__(self) -> str:
        return f"<{self.__class__.__name__}(name='{self.name}', enabled={self.enabled}, priority={self.priority})>"

class ProcessorGroup:
    """Group of processors that can be managed together."""
    
    def __init__(self, name: str, processors: List[BaseProcessor]):
        self.name = name
        self.processors = processors
        self.enabled = True
    
    async def process_all(self, data: Any) -> List[EdgeEvent]:
        """Process data through all processors in the group."""
        if not self.enabled:
            return []
        
        all_events = []
        
        # Process in priority order
        sorted_processors = sorted(self.processors, key=lambda p: p.priority)
        
        for processor in sorted_processors:
            if processor.enabled:
                events = await processor.process_with_monitoring(data)
                all_events.extend(events)
        
        return all_events
    
    async def start_all(self) -> None:
        """Start all processors in the group."""
        for processor in self.processors:
            await processor.start()
    
    async def stop_all(self) -> None:
        """Stop all processors in the group."""
        for processor in self.processors:
            await processor.stop()
    
    def get_group_status(self) -> Dict[str, Any]:
        """Get status of all processors in the group."""
        return {
            "name": self.name,
            "enabled": self.enabled,
            "processor_count": len(self.processors),
            "processors": [p.get_status() for p in self.processors]
        }
