/*
 * fr Bytecode Virtual Machine Runtime with GMP
 * Executes bytecode generated by compiler.py
 * Uses GMP for arbitrary precision integers (matching Python)
 * Optimized version with aggressive compiler hints
 */
#define _POSIX_C_SOURCE 200809L
#define _USE_MATH_DEFINES  // For M_PI on some platforms
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stdbool.h>
#include <stdint.h>
#include <math.h>

// Ensure M_PI is defined
#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

#include <ctype.h>
#include <gmp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/stat.h>
#include <dirent.h>
#include <libgen.h>
#include <limits.h>
#include <sys/wait.h>

// Python C API for embedding Python interpreter
#define PY_SSIZE_T_CLEAN
#include <Python.h>

// Helper function to unescape string literals from bytecode
char* unescape_string(const char* str) {
    if (!str) return strdup("");

    size_t len = strlen(str);
    char* result = malloc(len + 1); // Worst case: same length
    if (!result) return strdup("");

    size_t j = 0;
    for (size_t i = 0; i < len; i++) {
        if (str[i] == '\\' && i + 1 < len) {
            switch (str[i + 1]) {
                case 'n':  result[j++] = '\n'; i++; break;
                case 'r':  result[j++] = '\r'; i++; break;
                case 't':  result[j++] = '\t'; i++; break;
                case '0':  result[j++] = '\0'; i++; break;
                case '\\': result[j++] = '\\'; i++; break;
                case '"':  result[j++] = '"';  i++; break;
                default:   result[j++] = str[i]; break; // Keep backslash if unknown
            }
        } else {
            result[j++] = str[i];
        }
    }
    result[j] = '\0';
    return result;
}

// Branch prediction hints
#define likely(x) __builtin_expect(!!(x), 1)
#define unlikely(x) __builtin_expect(!!(x), 0)
#define MAX_STACK 1024
#define MAX_VARS 256
#define MAX_FUNCTIONS 128
#define MAX_CODE 65536
#define MAX_LABELS 512
#define MAX_CALL_STACK 256
#define MAX_STRING_LEN 65536  // Increased to support large list literals in bytecode

// Forward declarations
typedef struct Value Value;
typedef struct List List;
typedef struct Set Set;
typedef struct Struct Struct;
typedef struct VM VM;  // Forward declare VM

// Global VM pointer for error reporting (set during vm_run)
static VM *g_current_vm = NULL;

// Safe multi-value structures (avoid type-punning)
typedef struct {
    int count;
    int64_t values[];  // Flexible array member
} MultiInt64;

typedef struct {
    int count;
    double values[];  // Flexible array member
} MultiF64;

typedef struct {
    int count;
    char *values[];  // Flexible array member
} MultiStr;

typedef struct {
    int count;
    int values[];  // Flexible array member
} MultiInt;

// Structures for STORE_CONST instructions (slot/value pairs)
typedef struct {
    int count;  // Number of pairs
    int64_t pairs[];  // Interleaved: slot1, val1, slot2, val2, ...
} StoreConstI64;

typedef struct {
    int count;  // Number of pairs
    int64_t pairs[];  // Interleaved: slot1 (int), val1 (double bits), slot2, val2, ...
} StoreConstF64;

typedef struct {
    int count;  // Number of pairs
    int64_t pairs[];  // Interleaved: slot1, val1 (0 or 1), slot2, val2, ...
} StoreConstBool;

typedef struct {
    int count;  // Number of pairs
    char *pairs[];  // Interleaved: slot1 (cast to char*), str1, slot2, str2, ...
} StoreConstStr;

// Value types
typedef enum
{
    VAL_INT,    // Fast native 64-bit integer (default)
    VAL_BIGINT, // Arbitrary precision integer (mpz_t) - only when explicitly requested
    VAL_F64,
    VAL_STR,
    VAL_BOOL,
    VAL_VOID,
    VAL_LIST,   // List/Array type
    VAL_SET,    // Set type (unordered collection of unique elements)
    VAL_STRUCT, // Struct/Object type
    VAL_PYOBJECT // Python object type (PyObject*)
} ValueType;

// List structure
typedef struct List
{
    Value *items; // Dynamic array of values
    int length;   // Current number of items
    int capacity; // Allocated capacity
} List;

// Struct structure
typedef struct Struct
{
    int struct_id;    // ID identifying which struct type this is
    Value *fields;    // Dynamic array of field values
    int field_count;  // Number of fields
} Struct;

// Runtime value
typedef struct Value
{
    ValueType type;
    union
    {
        int64_t int64; // Fast native integer (default)
        mpz_t *bigint; // GMP integer (only for gmpint type)
        double f64;
        char *str;
        bool boolean;
        List *list;     // List pointer
        Set *set;       // Set pointer
        Struct *struct_val; // Struct pointer
        PyObject *pyobj;    // Python object pointer
    } as;
} Value;

// Hash table entry for Set (must be after Value definition)
typedef struct SetEntry
{
    Value key;
    bool occupied;
    bool deleted;  // For tombstone marking
} SetEntry;

// Set structure (hash table for O(1) operations)
typedef struct Set
{
    SetEntry *entries;  // Hash table entries
    int length;         // Number of elements in set
    int capacity;       // Allocated capacity
    int tombstones;     // Count of deleted entries
} Set;

// Variable storage
typedef struct
{
    Value vars[MAX_VARS];
    int var_count;
} VarTable;

// Instruction
typedef enum
{
    OP_CONST_I64, // Keep this name for compatibility with compiler
    OP_CONST_F64,
    OP_CONST_STR,
    OP_CONST_BOOL,
    OP_CONST_I64_MULTI,  // Push multiple int64 constants
    OP_CONST_F64_MULTI,  // Push multiple float64 constants
    OP_CONST_STR_MULTI,  // Push multiple string constants
    OP_CONST_BOOL_MULTI, // Push multiple bool constants
    OP_LOAD,
    OP_STORE,
    OP_STORE_CONST_I64,  // Store int constants directly to slots without using stack (STORE_CONST_I64 slot1 val1 slot2 val2 ...)
    OP_STORE_CONST_F64,  // Store float constants directly to slots without using stack (STORE_CONST_F64 slot1 val1 slot2 val2 ...)
    OP_STORE_CONST_BOOL, // Store bool constants directly to slots without using stack (STORE_CONST_BOOL slot1 val1 slot2 val2 ...)
    OP_STORE_CONST_STR,  // Store string constants directly to slots without using stack (STORE_CONST_STR slot1 val1 slot2 val2 ...)
    OP_ADD_I64,
    OP_SUB_I64,
    OP_MUL_I64,
    OP_DIV_I64,
    OP_MOD_I64,
    OP_ADD_F64,
    OP_SUB_F64,
    OP_MUL_F64,
    OP_DIV_F64,
    OP_ADD_STR,
    OP_CMP_EQ,
    OP_CMP_NE,
    OP_CMP_LT,
    OP_CMP_GT,
    OP_CMP_LE,
    OP_CMP_GE,
    OP_AND,
    OP_OR,
    OP_NOT,
    OP_JUMP,
    OP_JUMP_IF_FALSE,
    OP_JUMP_IF_TRUE,
    OP_GOTO_CALL,
    OP_CALL,
    OP_RETURN,
    OP_RETURN_VOID,
    OP_BUILTIN_PRINT,
    OP_BUILTIN_PRINTLN,
    OP_BUILTIN_STR,
    OP_BUILTIN_LEN,
    OP_BUILTIN_SQRT,
    OP_BUILTIN_ROUND,
    OP_BUILTIN_FLOOR,
    OP_BUILTIN_CEIL,
    OP_BUILTIN_PI,
    OP_POP,
    OP_DUP,
    OP_LABEL,
    OP_HALT,
    // Optimized instructions for better performance
    OP_INC_LOCAL,     // Increment local variable in-place
    OP_DEC_LOCAL,     // Decrement local variable in-place
    OP_ADD_CONST_I64, // Add constant to top of stack
    OP_SUB_CONST_I64, // Subtract constant from top of stack
    OP_MUL_CONST_I64, // Multiply top of stack by constant
    OP_DIV_CONST_I64, // Divide top of stack by constant
    OP_MOD_CONST_I64, // Modulo top of stack by constant
    OP_ADD_CONST_I64_MULTI, // Add multiple constants to top of stack
    OP_ADD_CONST_F64, // Add float constant to top of stack
    OP_SUB_CONST_F64, // Subtract float constant from top of stack
    OP_MUL_CONST_F64, // Multiply top of stack by float constant
    OP_DIV_CONST_F64, // Divide top of stack by float constant
    OP_AND_CONST,     // Logical AND with constant bool
    OP_OR_CONST,      // Logical OR with constant bool
    OP_AND_CONST_I64, // Bitwise AND with constant
    OP_OR_CONST_I64,  // Bitwise OR with constant
    OP_XOR_CONST_I64, // Bitwise XOR with constant
    // Stack manipulation (reduce memory traffic)
    OP_SWAP, // Swap top two stack values
    OP_ROT,  // Rotate top three stack values (a b c -> b c a)
    OP_OVER, // Copy second item to top (a b -> a b a)
    OP_DUP2, // Duplicate top two items (a b -> a b a b)
    // Bitwise operations (general purpose, not app-specific)
    OP_AND_I64, // Bitwise AND
    OP_OR_I64,  // Bitwise OR
    OP_XOR_I64, // Bitwise XOR
    OP_SHL_I64, // Shift left (multiply by power of 2)
    OP_SHR_I64, // Shift right (divide by power of 2)
    // Variable copy optimization
    OP_COPY_LOCAL,     // Copy one local to another without stack (COPY src dst) - deep copy
    OP_COPY_LOCAL_REF, // Alias one local to another (COPY_LOCAL_REF src dst) - pointer alias
    OP_STORE_REF,      // Store reference/pointer instead of copying value (for aliasing)
    // Multi-load/store optimizations
    OP_LOAD_MULTI,       // Load multiple variables at once (LOAD_MULTI n v1 v2 ... vn)
    OP_FUSED_LOAD_STORE, // Interleaved load/store (FUSED_LOAD_STORE n src1 dst1 src2 dst2 ...)
    OP_FUSED_STORE_LOAD, // Interleaved store/load (FUSED_STORE_LOAD n dst1 src1 dst2 src2 ...)
    // Fused load + arithmetic operations (reduce dispatch overhead in hot loops)
    OP_LOAD2_ADD_I64, // LOAD x y, ADD_I64 -> single instruction
    OP_LOAD2_SUB_I64, // LOAD x y, SUB_I64 -> single instruction
    OP_LOAD2_MUL_I64, // LOAD x y, MUL_I64 -> single instruction
    OP_LOAD2_MOD_I64, // LOAD x y, MOD_I64 -> single instruction
    OP_LOAD2_ADD_F64, // LOAD x y, ADD_F64 -> single instruction (float)
    OP_LOAD2_SUB_F64, // LOAD x y, SUB_F64 -> single instruction (float)
    OP_LOAD2_MUL_F64, // LOAD x y, MUL_F64 -> single instruction (float)
    OP_LOAD2_DIV_F64, // LOAD x y, DIV_F64 -> single instruction (float)
    OP_LOAD2_CMP_LT,  // LOAD x y, CMP_LT -> single instruction
    // Switch optimization
    OP_SWITCH_JUMP_TABLE, // Jump table for dense integer switches (min max label1 label2 ... default)
    OP_LOAD2_CMP_GT,  // LOAD x y, CMP_GT -> single instruction
    OP_LOAD2_CMP_LE,  // LOAD x y, CMP_LE -> single instruction
    OP_LOAD2_CMP_GE,  // LOAD x y, CMP_GE -> single instruction
    OP_LOAD2_CMP_EQ,  // LOAD x y, CMP_EQ -> single instruction
    OP_LOAD2_CMP_NE,  // LOAD x y, CMP_NE -> single instruction
    // Conditional operations
    OP_SELECT, // Select between two values based on condition (cond a b -> result)
    // Faster than if-else for simple selections

    // List/Array operations
    OP_LIST_NEW,    // Create new empty list
    OP_LIST_APPEND, // Append value to list (list, value -> list)
    OP_LIST_GET,    // Get element at index (list, index -> value)
    OP_LIST_SET,    // Set element at index (list, index, value -> list)
    OP_LIST_LEN,    // Get list length (list -> int)
    OP_LIST_POP,    // Pop last element (list -> value)
    OP_LIST_NEW_I64,    // Create list with int64 values (count val1 val2 ... -> list)
    OP_LIST_NEW_F64,    // Create list with float64 values (count val1 val2 ... -> list)
    OP_LIST_NEW_STR,    // Create list with string values (count val1 val2 ... -> list)
    OP_LIST_NEW_BOOL,   // Create list with boolean values (count val1 val2 ... -> list)

    // Set operations (hash table for O(1) operations)
    OP_SET_NEW,      // Create new empty set (-> set)
    OP_SET_ADD,      // Add value to set (set, value -> set)
    OP_SET_REMOVE,   // Remove value from set (set, value -> set)
    OP_SET_CONTAINS, // Check if set contains value (set, value -> bool)
    OP_SET_LEN,      // Get set length (set -> int)

    // Container membership check
    OP_CONTAINS,     // Generic membership check (container, value -> bool)

    // Break/Continue with levels
    OP_BREAK,    // Break from loop with level
    OP_CONTINUE, // Continue loop with level

    // Type conversions
    OP_TO_INT,   // Convert to int
    OP_TO_FLOAT, // Convert to float
    OP_TO_BOOL,  // Convert to bool
    OP_TO_STR,   // Convert to string (alias for BUILTIN_STR)

    // String methods
    OP_STR_UPPER,   // Convert string to uppercase
    OP_STR_LOWER,   // Convert string to lowercase
    OP_STR_STRIP,   // Strip whitespace
    OP_STR_SPLIT,   // Split string (str, sep -> list)
    OP_STR_JOIN,    // Join list with separator (sep, list -> str)
    OP_STR_REPLACE, // Replace substring (str, old, new -> str)
    OP_ENCODE,      // Encode string to bytes (str, encoding -> str)
    OP_DECODE,      // Decode bytes to string (str, encoding -> str)

    // Math functions
    OP_ABS,   // Absolute value
    OP_POW,   // Power (base, exp -> result)
    OP_MIN,   // Minimum of two values
    OP_MAX,   // Maximum of two values
    OP_NEG,   // Unary negation (negate top of stack)
    OP_FLOOR, // Floor function
    OP_CEIL,  // Ceil function
    OP_SIN,   // Sine
    OP_COS,   // Cosine
    OP_TAN,   // Tangent

    // Assert
    OP_ASSERT, // Assert: pops condition and optional message, errors if condition is false

    // Struct operations
    OP_STRUCT_NEW,    // Create new struct instance (struct_id -> struct)
    OP_STRUCT_GET,    // Get field from struct (struct, field_idx -> value)
    OP_STRUCT_SET,    // Set field in struct (struct, field_idx, value -> struct)

    // File I/O operations
    OP_FILE_OPEN,     // Open file (path, mode -> fd)
    OP_FILE_READ,     // Read from file (fd, size -> string)
    OP_FILE_WRITE,    // Write to file (fd, data -> bytes_written)
    OP_FILE_CLOSE,    // Close file (fd -> void)
    OP_FILE_EXISTS,   // Check if file/directory exists (path -> bool)
    OP_FILE_ISFILE,   // Check if path is a file (path -> bool)
    OP_FILE_ISDIR,    // Check if path is a directory (path -> bool)
    OP_FILE_LISTDIR,  // List directory contents (path -> list)
    OP_FILE_MKDIR,    // Create directory (path -> void)
    OP_FILE_MAKEDIRS, // Create directory and parents (path -> void)
    OP_FILE_REMOVE,   // Remove file (path -> void)
    OP_FILE_RMDIR,    // Remove directory (path -> void)
    OP_FILE_RENAME,   // Rename/move file (old_path, new_path -> void)
    OP_FILE_GETSIZE,  // Get file size (path -> int)
    OP_FILE_GETCWD,   // Get current working directory (-> string)
    OP_FILE_CHDIR,    // Change working directory (path -> void)
    OP_FILE_ABSPATH,  // Get absolute path (path -> string)
    OP_FILE_BASENAME, // Get basename (path -> string)
    OP_FILE_DIRNAME,  // Get directory name (path -> string)
    OP_FILE_JOIN,     // Join paths (list -> string)

    // Socket I/O operations
    OP_SOCKET_CREATE,     // Create socket (family, type -> sock_id)
    OP_SOCKET_CONNECT,    // Connect socket (sock_id, host, port -> void)
    OP_SOCKET_BIND,       // Bind socket (sock_id, host, port -> void)
    OP_SOCKET_LISTEN,     // Listen on socket (sock_id, backlog -> void)
    OP_SOCKET_ACCEPT,     // Accept connection (sock_id -> client_sock_id)
    OP_SOCKET_SEND,       // Send data (sock_id, data -> bytes_sent)
    OP_SOCKET_RECV,       // Receive data (sock_id, size -> string)
    OP_SOCKET_CLOSE,      // Close socket (sock_id -> void)
    OP_SOCKET_SETSOCKOPT, // Set socket option (sock_id, level, option, value -> void)

    // Process management
    OP_FORK,              // Fork process (-> pid)
    OP_WAIT,              // Wait for child process (-> status)

    // Python library integration
    OP_PY_IMPORT,         // Import a Python module (module_name -> module_object)
    OP_PY_CALL,           // Call a Python function (module_name, func_name, arg1, ..., argN, num_args -> result)
    OP_PY_GETATTR,        // Get attribute from Python object (obj, attr_name -> value)
    OP_PY_SETATTR,        // Set attribute on Python object (obj, attr_name, value -> none)
    OP_PY_CALL_METHOD,    // Call method on Python object (obj, method_name, arg1, ..., argN, num_args -> result)

    // Exception handling
    OP_TRY_BEGIN,         // Begin exception handler (exc_type, label -> void)
    OP_TRY_END,           // End exception handler (-> void)
    OP_RAISE,             // Raise an exception (exc_type, message -> void)

    // Comparison with constant optimizations
    OP_CMP_LT_CONST,      // Compare top of stack < constant (TOS, const -> bool)
    OP_CMP_GT_CONST,      // Compare top of stack > constant (TOS, const -> bool)
    OP_CMP_LE_CONST,      // Compare top of stack <= constant (TOS, const -> bool)
    OP_CMP_GE_CONST,      // Compare top of stack >= constant (TOS, const -> bool)
    OP_CMP_EQ_CONST,      // Compare top of stack == constant (TOS, const -> bool)
    OP_CMP_NE_CONST,      // Compare top of stack != constant (TOS, const -> bool)
    OP_CMP_LT_CONST_F64,  // Compare top of stack < constant (float)
    OP_CMP_GT_CONST_F64,  // Compare top of stack > constant (float)
    OP_CMP_LE_CONST_F64,  // Compare top of stack <= constant (float)
    OP_CMP_GE_CONST_F64,  // Compare top of stack >= constant (float)
    OP_CMP_EQ_CONST_F64,  // Compare top of stack == constant (float)
    OP_CMP_NE_CONST_F64,  // Compare top of stack != constant (float)
} OpCode;

// Instruction structure
typedef struct
{
    OpCode op;
    union
    {
        char *str_val; // For string/label literals
        int64_t int64; // For pre-parsed int64 constants and jump targets
        double f64;
        int index;
        void *ptr; // For fused instruction argument arrays
        struct
        { // For instructions with two operands (COPY_LOCAL, etc.)
            int src;
            int dst;
        } indices;
        struct
        { // For SWITCH_JUMP_TABLE instruction
            int min_val;
            int max_val;
            int num_labels;
            char **labels; // Array of label strings (to be resolved to PCs)
            int *pcs;      // Array of resolved PC indices (filled during label resolution)
        } switch_table;
    } operand;
} Instruction;

// Function info
typedef struct
{
    char *name;
    int arg_count;
    int local_count;
    int start_pc;
    int end_pc;
    ValueType return_type;
} Function;

// Label info
typedef struct
{
    char *name;
    int pc;
    int func_index;  // Which function this label belongs to
} Label;

// Call frame
typedef struct
{
    Function *func;
    VarTable vars;
    int return_pc;
} CallFrame;

// Struct definition metadata
#define MAX_STRUCTS 256
#define MAX_STRUCT_FIELDS 32
#define MAX_EXCEPTION_HANDLERS 64

typedef struct
{
    char *name;           // Struct type name
    int field_count;      // Number of fields
    char *field_names[MAX_STRUCT_FIELDS]; // Field names for debugging
} StructDef;

// Exception handler
typedef struct
{
    char *exc_type;       // Exception type to catch (e.g., "ZeroDivisionError")
    int handler_pc;       // PC to jump to when exception is caught
    int stack_top;        // Stack top at the time of TRY_BEGIN
    int call_stack_top;   // Call stack top at the time of TRY_BEGIN
} ExceptionHandler;

// VM state
typedef struct VM
{
    Value stack[MAX_STACK];
    int stack_top;
    Instruction code[MAX_CODE];
    int code_count;
    int pc;

    Function functions[MAX_FUNCTIONS];
    int func_count;
    Label labels[MAX_LABELS];
    int label_count;

    CallFrame call_stack[MAX_CALL_STACK];
    int call_stack_top;
    char *entry_point;
    bool running;

    int exit_code;  // Exit code for the program (0 = success, 1 = error)
    StructDef structs[MAX_STRUCTS];  // Struct definitions
    int struct_count;                // Number of registered structs
    int prog_argc;                   // Program argument count
    char **prog_argv;                // Program arguments

    // Exception handling
    ExceptionHandler exception_handlers[MAX_EXCEPTION_HANDLERS];
    int exception_handler_count;     // Number of active exception handlers

    // Debug information
    char *debug_source_file;         // Source file path for error reporting
    char **debug_source_lines;       // Array of source code lines
    int debug_source_line_count;     // Number of source lines
    char *debug_current_func;        // Current function name for error reporting
    int *debug_line_map;             // Maps PC to source line number
    int debug_line_map_count;        // Number of entries in line map
    
    // Performance optimizations
    struct Arena *arena;             // Arena allocator for temporary allocations
    struct StringIntern *string_intern; // String interning table for constants
} VM;

// ============================================================================
// ARENA ALLOCATOR - Batch allocate/free temporary values
// ============================================================================
#define ARENA_BLOCK_SIZE (64 * 1024)  // 64KB blocks

typedef struct ArenaBlock {
    char *memory;
    size_t used;
    size_t capacity;
    struct ArenaBlock *next;
} ArenaBlock;

typedef struct Arena {
    ArenaBlock *current;
    ArenaBlock *first;
    size_t total_allocated;
} Arena;

// Create new arena
Arena* arena_create() {
    Arena *arena = malloc(sizeof(Arena));
    arena->first = malloc(sizeof(ArenaBlock));
    arena->first->memory = malloc(ARENA_BLOCK_SIZE);
    arena->first->used = 0;
    arena->first->capacity = ARENA_BLOCK_SIZE;
    arena->first->next = NULL;
    arena->current = arena->first;
    arena->total_allocated = ARENA_BLOCK_SIZE;
    return arena;
}

// Allocate from arena
void* arena_alloc(Arena *arena, size_t size) {
    // Align to 8 bytes
    size = (size + 7) & ~7;
    
    // If current block doesn't have space, create new block
    if (arena->current->used + size > arena->current->capacity) {
        size_t block_size = size > ARENA_BLOCK_SIZE ? size : ARENA_BLOCK_SIZE;
        ArenaBlock *new_block = malloc(sizeof(ArenaBlock));
        new_block->memory = malloc(block_size);
        new_block->used = 0;
        new_block->capacity = block_size;
        new_block->next = NULL;
        arena->current->next = new_block;
        arena->current = new_block;
        arena->total_allocated += block_size;
    }
    
    void *ptr = arena->current->memory + arena->current->used;
    arena->current->used += size;
    return ptr;
}

// Reset arena (keep memory, reset used counters)
void arena_reset(Arena *arena) {
    ArenaBlock *block = arena->first;
    while (block) {
        block->used = 0;
        block = block->next;
    }
    arena->current = arena->first;
}

// Destroy arena
void arena_destroy(Arena *arena) {
    ArenaBlock *block = arena->first;
    while (block) {
        ArenaBlock *next = block->next;
        free(block->memory);
        free(block);
        block = next;
    }
    free(arena);
}

// ============================================================================
// STRING INTERNING - Deduplicate constant strings
// ============================================================================
#define STRING_INTERN_SIZE 1024

typedef struct InternedString {
    char *str;
    size_t hash;
    struct InternedString *next;
} InternedString;

typedef struct StringIntern {
    InternedString *buckets[STRING_INTERN_SIZE];
    int count;
} StringIntern;

// Simple hash function
static inline size_t hash_string(const char *str) {
    size_t hash = 5381;
    int c;
    while ((c = *str++))
        hash = ((hash << 5) + hash) + c; // hash * 33 + c
    return hash;
}

// Create string intern table
StringIntern* string_intern_create() {
    StringIntern *intern = malloc(sizeof(StringIntern));
    memset(intern->buckets, 0, sizeof(intern->buckets));
    intern->count = 0;
    return intern;
}

// Intern a string (return canonical pointer)
const char* string_intern_add(StringIntern *intern, const char *str) {
    size_t hash = hash_string(str);
    int bucket = hash % STRING_INTERN_SIZE;
    
    // Check if already interned
    InternedString *entry = intern->buckets[bucket];
    while (entry) {
        if (strcmp(entry->str, str) == 0) {
            return entry->str; // Return existing interned string
        }
        entry = entry->next;
    }
    
    // Not found, add new entry
    InternedString *new_entry = malloc(sizeof(InternedString));
    new_entry->str = strdup(str);
    new_entry->hash = hash;
    new_entry->next = intern->buckets[bucket];
    intern->buckets[bucket] = new_entry;
    intern->count++;
    return new_entry->str;
}

// Destroy string intern table
void string_intern_destroy(StringIntern *intern) {
    for (int i = 0; i < STRING_INTERN_SIZE; i++) {
        InternedString *entry = intern->buckets[i];
        while (entry) {
            InternedString *next = entry->next;
            free(entry->str);
            free(entry);
            entry = next;
        }
    }
    free(intern);
}

// Forward declarations for helper functions
void value_free(Value v);
Value value_copy(Value v);
void list_free(List *list);
void vm_runtime_error(VM *vm, const char *message, int char_pos);

// Safe parsing helpers (use strtol family instead of unsafe ato* functions)
static inline int64_t safe_atoll(const char *str)
{
    char *endptr;
    int64_t result = strtoll(str, &endptr, 10);
    return result;  // Return result; caller handles validation if needed
}

static inline int safe_atoi(const char *str)
{
    char *endptr;
    long result = strtol(str, &endptr, 10);
    return (int)result;  // Return result; caller handles validation if needed
}

static inline double safe_atof(const char *str)
{
    char *endptr;
    double result = strtod(str, &endptr);
    return result;  // Return result; caller handles validation if needed
}

// Helper functions for Value
Value value_make_int(const char *str)
{
    Value v;
    v.type = VAL_INT;
    v.as.int64 = safe_atoll(str); // Fast native int
    return v;
}
Value value_make_int_si(long val)
{
    Value v;
    v.type = VAL_INT;
    v.as.int64 = val; // Fast native int
    return v;
}
Value value_make_bigint(const char *str)
{
    Value v;
    v.type = VAL_BIGINT;
    v.as.bigint = malloc(sizeof(mpz_t));
    mpz_init(*v.as.bigint);
    mpz_set_str(*v.as.bigint, str, 10);
    return v;
}
Value value_make_bigint_si(long val)
{
    Value v;
    v.type = VAL_BIGINT;
    v.as.bigint = malloc(sizeof(mpz_t));
    mpz_init_set_si(*v.as.bigint, val);
    return v;
}
Value value_make_f64(double val)
{
    Value v;
    v.type = VAL_F64;
    v.as.f64 = val;
    return v;
}
Value value_make_str(const char *str)
{
    Value v;
    v.type = VAL_STR;
    v.as.str = strdup(str);
    return v;
}
Value value_make_bool(bool val)
{
    Value v;
    v.type = VAL_BOOL;
    v.as.boolean = val;
    return v;
}
Value value_make_void()
{
    Value v;
    v.type = VAL_VOID;
    return v;
}

// List helper functions
List *list_new() {
    List *list = malloc(sizeof(List));
    if (!list)
    {
        fprintf(stderr, "Error: Failed to allocate list\n");
        exit(1);
    }
    list->capacity = 8; // Start with capacity of 8
    list->length = 0;
    list->items = malloc(sizeof(Value) * list->capacity);
    if (!list->items)
    {
        fprintf(stderr, "Error: Failed to allocate list items\n");
        exit(1);
    }
    return list;
}

void list_append(List *list, Value value) {
    if (list->length >= list->capacity)
    {
        // Double capacity
        list->capacity *= 2;
        list->items = realloc(list->items, sizeof(Value) * list->capacity);
        if (!list->items)
        {
            fprintf(stderr, "Error: Failed to reallocate list items\n");
            exit(1);
        }
    }
    list->items[list->length++] = value_copy(value);
}

Value list_get(List *list, int index) {
    // Handle negative indices (Python-style)
    int original_index = index;  // Save for error message
    if (index < 0)
    {
        index = list->length + index;
    }

    if (index < 0 || index >= list->length)
    {
        char error_msg[256];
        snprintf(error_msg, sizeof(error_msg), "Index error: list index out of range: %d (length: %d)", original_index, list->length);

        // Try to find the exact index value in the source line for better error reporting
        int char_pos = 0;
        if (g_current_vm && g_current_vm->debug_source_lines) {
            int line_idx = g_current_vm->pc >= 2 && g_current_vm->debug_line_map_count > 0
                ? g_current_vm->debug_line_map[(g_current_vm->pc - 2) % g_current_vm->debug_line_map_count]
                : 0;

            if (line_idx > 0 && line_idx <= g_current_vm->debug_source_line_count) {
                const char *source_line = g_current_vm->debug_source_lines[line_idx - 1];

                // Try to find the exact index pattern [index]
                char search_pattern[32];
                int found = 0;

                // Try searching for [original_index] first
                snprintf(search_pattern, sizeof(search_pattern), "[%d]", original_index);
                const char *pattern_pos = strstr(source_line, search_pattern);
                if (pattern_pos) {
                    // Point to the last digit before ']'
                    char_pos = pattern_pos - source_line + strlen(search_pattern) - 2;
                    found = 1;
                }

                // Fallback: find any bracket pair if exact match not found
                if (!found) {
                    const char *bracket = strchr(source_line, '[');
                    if (bracket) {
                        const char *close_bracket = strchr(bracket, ']');
                        if (close_bracket && close_bracket > bracket) {
                            char_pos = close_bracket - source_line - 1;
                        } else {
                            char_pos = bracket - source_line;
                        }
                    }
                }
            }
        }

        if (g_current_vm) {
            vm_runtime_error(g_current_vm, error_msg, char_pos);
        } else {
            fprintf(stderr, "Error: %s\n", error_msg);
        }
        exit(1);
    }
    return value_copy(list->items[index]);
}

void list_set(List *list, int index, Value value) {
    // Handle negative indices (Python-style)
    int original_index = index;  // Save for error message
    if (index < 0)
    {
        index = list->length + index;
    }

    if (index < 0 || index >= list->length)
    {
        char error_msg[256];
        snprintf(error_msg, sizeof(error_msg), "Index error: list index out of range: %d (length: %d)", original_index, list->length);

        // Try to find the exact index value in the source line for better error reporting
        int char_pos = 0;
        if (g_current_vm && g_current_vm->debug_source_lines) {
            int line_idx = g_current_vm->pc >= 2 && g_current_vm->debug_line_map_count > 0
                ? g_current_vm->debug_line_map[(g_current_vm->pc - 2) % g_current_vm->debug_line_map_count]
                : 0;

            if (line_idx > 0 && line_idx <= g_current_vm->debug_source_line_count) {
                const char *source_line = g_current_vm->debug_source_lines[line_idx - 1];

                // Try to find the exact index pattern [index]
                char search_pattern[32];
                int found = 0;

                // Try searching for [original_index] first
                snprintf(search_pattern, sizeof(search_pattern), "[%d]", original_index);
                const char *pattern_pos = strstr(source_line, search_pattern);
                if (pattern_pos) {
                    // Point to the last digit before ']'
                    char_pos = pattern_pos - source_line + strlen(search_pattern) - 2;
                    found = 1;
                }

                // Fallback: find any bracket pair if exact match not found
                if (!found) {
                    const char *bracket = strchr(source_line, '[');
                    if (bracket) {
                        const char *close_bracket = strchr(bracket, ']');
                        if (close_bracket && close_bracket > bracket) {
                            char_pos = close_bracket - source_line - 1;
                        } else {
                            char_pos = bracket - source_line;
                        }
                    }
                }
            }
        }

        if (g_current_vm) {
            vm_runtime_error(g_current_vm, error_msg, char_pos);
        } else {
            fprintf(stderr, "Error: %s\n", error_msg);
        }
        exit(1);
    }
    value_free(list->items[index]);
    list->items[index] = value_copy(value);
}

Value list_pop(List *list) {
    if (list->length == 0)
    {
        fprintf(stderr, "Error: Cannot pop from empty list\n");
        exit(1);
    }
    return list->items[--list->length];
}

void list_free(List *list) {
    if (!list)
        return;
    for (int i = 0; i < list->length; i++)
    {
        value_free(list->items[i]);
    }
    free(list->items);
    free(list);
}

// ============================================================================
// Set operations (hash table implementation for O(1) operations)
// ============================================================================

// Hash function for values
static uint64_t value_hash(Value v) {
    switch (v.type) {
        case VAL_INT:
            // Simple integer hash
            return (uint64_t)v.as.int64;
        case VAL_F64: {
            // Hash the bits of the double
            uint64_t bits;
            memcpy(&bits, &v.as.f64, sizeof(double));
            return bits;
        }
        case VAL_STR: {
            // FNV-1a hash for strings
            uint64_t hash = 14695981039346656037ULL;
            const char *str = v.as.str;
            while (*str) {
                hash ^= (uint64_t)*str++;
                hash *= 1099511628211ULL;
            }
            return hash;
        }
        case VAL_BOOL:
            return v.as.boolean ? 1 : 0;
        default:
            // For other types, use pointer address
            return (uint64_t)(uintptr_t)&v;
    }
}

// Compare two values for equality
static bool value_equals(Value a, Value b) {
    if (a.type != b.type)
        return false;
    
    switch (a.type) {
        case VAL_INT:
            return a.as.int64 == b.as.int64;
        case VAL_F64:
            return a.as.f64 == b.as.f64;
        case VAL_STR:
            return strcmp(a.as.str, b.as.str) == 0;
        case VAL_BOOL:
            return a.as.boolean == b.as.boolean;
        case VAL_VOID:
            return true;
        default:
            // For complex types, compare pointers
            return a.as.list == b.as.list;
    }
}

Set *set_new() {
    Set *set = malloc(sizeof(Set));
    if (!set) {
        fprintf(stderr, "Error: Failed to allocate set\n");
        exit(1);
    }
    set->capacity = 16;  // Start with capacity of 16
    set->length = 0;
    set->tombstones = 0;
    set->entries = calloc(set->capacity, sizeof(SetEntry));
    if (!set->entries) {
        fprintf(stderr, "Error: Failed to allocate set entries\n");
        exit(1);
    }
    return set;
}

// Find entry in hash table
static SetEntry* set_find_entry(SetEntry *entries, int capacity, Value key) {
    uint64_t hash = value_hash(key);
    uint32_t index = (uint32_t)(hash % capacity);
    SetEntry *tombstone = NULL;
    
    for (;;) {
        SetEntry *entry = &entries[index];
        
        if (!entry->occupied) {
            if (!entry->deleted) {
                // Empty entry found
                return tombstone != NULL ? tombstone : entry;
            } else {
                // Tombstone found
                if (tombstone == NULL) tombstone = entry;
            }
        } else if (value_equals(entry->key, key)) {
            // Key found
            return entry;
        }
        
        // Linear probing
        index = (index + 1) % capacity;
    }
}

// Resize the hash table
static void set_resize(Set *set, int new_capacity) {
    SetEntry *new_entries = calloc(new_capacity, sizeof(SetEntry));
    if (!new_entries) {
        fprintf(stderr, "Error: Failed to allocate new set entries\n");
        exit(1);
    }
    
    // Rehash all entries
    set->length = 0;
    set->tombstones = 0;
    for (int i = 0; i < set->capacity; i++) {
        SetEntry *entry = &set->entries[i];
        if (!entry->occupied) continue;
        
        SetEntry *dest = set_find_entry(new_entries, new_capacity, entry->key);
        dest->key = entry->key;
        dest->occupied = true;
        dest->deleted = false;
        set->length++;
    }
    
    free(set->entries);
    set->entries = new_entries;
    set->capacity = new_capacity;
}

bool set_add(Set *set, Value value) {
    // Resize if load factor exceeds 0.75
    if (set->length + set->tombstones + 1 > set->capacity * 0.75) {
        set_resize(set, set->capacity * 2);
    }
    
    SetEntry *entry = set_find_entry(set->entries, set->capacity, value);
    bool is_new = !entry->occupied;
    
    if (is_new) {
        if (entry->deleted) set->tombstones--;
        entry->key = value_copy(value);
        entry->occupied = true;
        entry->deleted = false;
        set->length++;
    }
    
    return is_new;
}

bool set_remove(Set *set, Value value) {
    if (set->length == 0) return false;
    
    SetEntry *entry = set_find_entry(set->entries, set->capacity, value);
    if (!entry->occupied) return false;
    
    // Mark as tombstone
    value_free(entry->key);
    entry->occupied = false;
    entry->deleted = true;
    set->tombstones++;
    set->length--;
    return true;
}

bool set_contains(Set *set, Value value) {
    if (set->length == 0) return false;
    
    SetEntry *entry = set_find_entry(set->entries, set->capacity, value);
    return entry->occupied;
}

void set_free(Set *set) {
    if (!set) return;
    
    for (int i = 0; i < set->capacity; i++) {
        if (set->entries[i].occupied) {
            value_free(set->entries[i].key);
        }
    }
    free(set->entries);
    free(set);
}

Value value_make_list() {
    Value v;
    v.type = VAL_LIST;
    v.as.list = list_new();
    return v;
}

Value value_wrap_list(List *list) {
    Value v;
    v.type = VAL_LIST;
    v.as.list = list;
    return v;
}

Value value_make_set() {
    Value v;
    v.type = VAL_SET;
    v.as.set = set_new();
    return v;
}

Value value_wrap_set(Set *set) {
    Value v;
    v.type = VAL_SET;
    v.as.set = set;
    return v;
}

void value_free(Value v) {
    if (v.type == VAL_BIGINT && v.as.bigint)
    {
        mpz_clear(*v.as.bigint);
        free(v.as.bigint);
    }
    else if (v.type == VAL_STR && v.as.str)
    {
        free(v.as.str);
    }
    else if (v.type == VAL_LIST && v.as.list)
    {
        list_free(v.as.list);
    }
    else if (v.type == VAL_SET && v.as.set)
    {
        set_free(v.as.set);
    }
    else if (v.type == VAL_STRUCT && v.as.struct_val)
    {
        // Free all field values
        for (int i = 0; i < v.as.struct_val->field_count; i++)
        {
            value_free(v.as.struct_val->fields[i]);
        }
        free(v.as.struct_val->fields);
        free(v.as.struct_val);
    }
    else if (v.type == VAL_PYOBJECT && v.as.pyobj)
    {
        // Decrement Python object reference count
        Py_XDECREF(v.as.pyobj);
    }
}

Value value_copy(Value v)
{
    if (v.type == VAL_INT)
    {
        return v; // Fast copy - just copy the int64
    }
    else if (v.type == VAL_BIGINT)
    {
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init_set(*result.as.bigint, *v.as.bigint);
        return result;
    }
    else if (v.type == VAL_STR)
    {
        return value_make_str(v.as.str);
    }
    else if (v.type == VAL_LIST)
    {
        // Deep copy list
        Value result;
        result.type = VAL_LIST;
        result.as.list = list_new();
        for (int i = 0; i < v.as.list->length; i++)
        {
            list_append(result.as.list, v.as.list->items[i]);
        }
        return result;
    }
    else if (v.type == VAL_SET)
    {
        // Deep copy set
        Value result;
        result.type = VAL_SET;
        result.as.set = set_new();
        for (int i = 0; i < v.as.set->capacity; i++)
        {
            if (v.as.set->entries[i].occupied)
            {
                set_add(result.as.set, v.as.set->entries[i].key);
            }
        }
        return result;
    }
    else if (v.type == VAL_STRUCT)
    {
        // Deep copy struct
        Value result;
        result.type = VAL_STRUCT;
        result.as.struct_val = malloc(sizeof(Struct));
        result.as.struct_val->struct_id = v.as.struct_val->struct_id;
        result.as.struct_val->field_count = v.as.struct_val->field_count;
        result.as.struct_val->fields = malloc(sizeof(Value) * result.as.struct_val->field_count);
        for (int i = 0; i < result.as.struct_val->field_count; i++)
        {
            result.as.struct_val->fields[i] = value_copy(v.as.struct_val->fields[i]);
        }
        return result;
    }
    else if (v.type == VAL_PYOBJECT)
    {
        // Increment reference count for Python object
        Value result;
        result.type = VAL_PYOBJECT;
        result.as.pyobj = v.as.pyobj;
        Py_XINCREF(result.as.pyobj);
        return result;
    }
    return v; // Other types are simple values
}

// Create an alias to another value - for now just shallow copy (good enough for int64)
static inline void value_copy_ref(Value *src, Value *dst)
{
    // For int64, just copy the value (they're immutable anyway)
    // For other types, do deep copy to be safe
    if (src->type == VAL_INT || src->type == VAL_F64 || src->type == VAL_BOOL || src->type == VAL_VOID)
    {
        *dst = *src; // Shallow copy is safe for immutable types
    }
    else
    {
        *dst = value_copy(*src); // Deep copy for mutable types
    }
}

// Function prototypes
void vm_init(VM *vm);
void vm_free(VM *vm);

static inline Value vm_pop(VM *vm) __attribute__((always_inline));
static inline void vm_push(VM *vm, Value value) __attribute__((always_inline));

void vm_run(VM *vm) __attribute__((hot));
bool vm_load_bytecode(VM *vm,const char *filename);

Function *vm_find_function(VM *vm,const char *name);

int vm_find_label(VM *vm,const char *name, int func_index);
void value_print(Value val);
Value value_to_string(Value val);

static inline Value value_add(Value a, Value b);
static inline Value value_sub(Value a, Value b);
static inline Value value_mul(Value a, Value b);
static inline Value value_div(Value a, Value b);
static inline Value value_mod(Value a, Value b);
static inline Value value_compare(Value a, Value b, OpCode op);

// Initialize VM
void vm_init(VM *vm)
{
    vm->stack_top = 0;
    vm->code_count = 0;
    vm->pc = 0;
    vm->func_count = 0;
    vm->label_count = 0;
    vm->call_stack_top = 0;
    vm->entry_point = NULL;
    vm->running = false;
    vm->exit_code = 0;
    vm->struct_count = 0;
    vm->prog_argc = 0;
    vm->prog_argv = NULL;
    vm->exception_handler_count = 0;
    vm->debug_source_file = NULL;
    vm->debug_source_lines = NULL;
    vm->debug_source_line_count = 0;
    vm->debug_current_func = strdup("<module>");
    vm->debug_line_map = NULL;
    vm->debug_line_map_count = 0;
    
    // Initialize performance optimizations
    vm->arena = arena_create();
    vm->string_intern = string_intern_create();

    // Initialize Python interpreter
    if (!Py_IsInitialized()) {
        Py_Initialize();
        // Add current directory to Python path
        PyRun_SimpleString("import sys");
        PyRun_SimpleString("sys.path.insert(0, '.')");
    }
}

// Free VM resources
void vm_free(VM *vm)
{
    // Free stack values
    for (int i = 0; i < vm->stack_top; i++)
    {
        value_free(vm->stack[i]);
    }
    // Free exception handlers
    for (int i = 0; i < vm->exception_handler_count; i++)
    {
        if (vm->exception_handlers[i].exc_type)
            free(vm->exception_handlers[i].exc_type);
    }
    // Free function names
    for (int i = 0; i < vm->func_count; i++)
    {
        if (vm->functions[i].name)
            free(vm->functions[i].name);
    }
    // Free label names
    for (int i = 0; i < vm->label_count; i++)
    {
        if (vm->labels[i].name)
            free(vm->labels[i].name);
    }
    // Free entry point
    if (vm->entry_point)
        free(vm->entry_point);
    // Free string values in code
    for (int i = 0; i < vm->code_count; i++)
    {
        // Note: OP_CONST_STR operands are interned strings - don't free them
        // They will be freed when string_intern is destroyed
        if (vm->code[i].op == OP_CONST_STR && vm->code[i].operand.str_val)
        {
            // Interned strings - don't free here
        }
        // Free TRY_BEGIN and RAISE operands
        else if ((vm->code[i].op == OP_TRY_BEGIN || vm->code[i].op == OP_RAISE) && vm->code[i].operand.str_val)
        {
            free(vm->code[i].operand.str_val);
        }
        // Free multi-arg constant instructions
        else if (vm->code[i].op == OP_CONST_I64_MULTI ||
                 vm->code[i].op == OP_CONST_F64_MULTI ||
                 vm->code[i].op == OP_CONST_BOOL_MULTI ||
                 vm->code[i].op == OP_LIST_NEW_I64 ||
                 vm->code[i].op == OP_LIST_NEW_F64 ||
                 vm->code[i].op == OP_LIST_NEW_BOOL)
        {
            if (vm->code[i].operand.ptr)
                free(vm->code[i].operand.ptr);
        }
        else if (vm->code[i].op == OP_CONST_STR_MULTI ||
                 vm->code[i].op == OP_LIST_NEW_STR)
        {
            // Free the MultiStr structure but not the interned strings
            if (vm->code[i].operand.ptr)
            {
                MultiStr *multi = (MultiStr*)vm->code[i].operand.ptr;
                // Strings are interned - don't free them
                free(multi); // Just free the struct itself
            }
        }
    }

    // Free debug information
    if (vm->debug_source_file)
        free(vm->debug_source_file);
    if (vm->debug_current_func)
        free(vm->debug_current_func);
    if (vm->debug_source_lines) {
        for (int i = 0; i < vm->debug_source_line_count; i++) {
            if (vm->debug_source_lines[i])
                free(vm->debug_source_lines[i]);
        }
        free(vm->debug_source_lines);
    }
    if (vm->debug_line_map)
        free(vm->debug_line_map);
    
    // Free performance optimization structures
    if (vm->arena)
        arena_destroy(vm->arena);
    if (vm->string_intern)
        string_intern_destroy(vm->string_intern);

    // Finalize Python interpreter
    if (Py_IsInitialized()) {
        Py_Finalize();
    }
}

// Helper function to format runtime errors with debug information
void vm_runtime_error(VM *vm, const char *message, int char_pos) {
    // Check if there's an active exception handler
    if (vm->exception_handler_count > 0) {
        // Find the most recent handler
        ExceptionHandler *handler = &vm->exception_handlers[vm->exception_handler_count - 1];

        // Extract exception type from message
        // Format: "[ExceptionType] actual message" or just "actual message"
        char extracted_type[256] = "RuntimeError";  // Default, on stack but persistent during function
        const char *exc_type = extracted_type;
        const char *display_message = message;  // Message to display

        // Check for [ExceptionType] marker at start
        if (message[0] == '[') {
            const char *end_bracket = strchr(message, ']');
            if (end_bracket) {
                // Extract exception type
                size_t type_len = end_bracket - message - 1;
                if (type_len < sizeof(extracted_type)) {
                    strncpy(extracted_type, message + 1, type_len);
                    extracted_type[type_len] = '\0';
                    exc_type = extracted_type;

                    // Skip past the marker and space to get display message
                    display_message = end_bracket + 1;
                    while (*display_message == ' ') display_message++;
                }
            }
        }
        // Fallback: try to detect from message content
        else if (strstr(message, "division by zero") || strstr(message, "Division by zero") ||
            strstr(message, "float division by zero")) {
            strcpy(extracted_type, "ZeroDivisionError");
            exc_type = extracted_type;
        } else if (strstr(message, "Index error") || strstr(message, "index out of range")) {
            strcpy(extracted_type, "IndexError");
            exc_type = extracted_type;
        } else if (strstr(message, "Type error")) {
            strcpy(extracted_type, "TypeError");
            exc_type = extracted_type;
        } else if (strstr(message, "Value error")) {
            strcpy(extracted_type, "ValueError");
            exc_type = extracted_type;
        }

        // Check if the handler matches this exception type
        if (strcmp(handler->exc_type, exc_type) == 0 || strcmp(handler->exc_type, "") == 0) {
            // Restore stack state
            vm->stack_top = handler->stack_top;
            vm->call_stack_top = handler->call_stack_top;

            // Jump to exception handler
            vm->pc = handler->handler_pc;

            // Pop the exception handler
            free(handler->exc_type);
            handler->exc_type = NULL;
            vm->exception_handler_count--;

            return;  // Continue execution at handler
        }

        // Handler doesn't match - pop it and continue searching or error
        free(handler->exc_type);
        handler->exc_type = NULL;
        vm->exception_handler_count--;

        // Recursively check for other handlers
        // Use display_message to avoid showing the [Type] marker
        if (vm->exception_handler_count > 0) {
            vm_runtime_error(vm, display_message, char_pos);
            return;
        }

        // No more handlers - will fall through to error printing
        // Use display_message for cleaner output
        message = display_message;
    }

    // No handler found or handler doesn't match - print error and exit
    fprintf(stderr, "Exception: Runtime Error\n");

    if (vm->debug_source_file && vm->debug_source_lines && vm->debug_source_line_count > 0) {
        // Get line number from line map based on PC
        // Note: PC has already been incremented by DISPATCH(), so we need PC-2 to get the current instruction
        int line = 1; // Default to line 1
        int pc_index = vm->pc - 2;
        if (vm->debug_line_map && pc_index >= 0 && pc_index < vm->debug_line_map_count) {
            line = vm->debug_line_map[pc_index];
        }

        // Try to find current function name from call stack
        const char *func_name = vm->debug_current_func ? vm->debug_current_func : "<module>";
        if (vm->call_stack_top > 0) {
            func_name = vm->call_stack[vm->call_stack_top - 1].func->name;
        }

        fprintf(stderr, "  File \"%s\" line %d in %s\n", vm->debug_source_file, line, func_name);

        if (line > 0 && line <= vm->debug_source_line_count) {
            const char *source_line = vm->debug_source_lines[line - 1];
            fprintf(stderr, "      %s\n", source_line);

            // Print pointer at character position
            fprintf(stderr, "      ");
            for (int i = 0; i < char_pos; i++) {
                fprintf(stderr, " ");
            }
            fprintf(stderr, "^\n");

            fprintf(stderr, "    %s:%d:%d: %s\n", vm->debug_source_file, line, char_pos, message);
        } else {
            fprintf(stderr, "    %s:%d:%d: %s\n", vm->debug_source_file, line, char_pos, message);
        }
    } else {
        fprintf(stderr, "Error: %s\n", message);
    }

    exit(1);
}

// Stack operations - inline for performance with always_inline hint
static inline void vm_push(VM *vm, Value value) __attribute__((always_inline));
static inline Value vm_pop(VM *vm) __attribute__((always_inline));

static inline void vm_push(VM *vm, Value value)
{
    if (unlikely(vm->stack_top >= MAX_STACK))
    {
        fprintf(stderr, "Stack overflow\n");
        exit(1);
    }
    vm->stack[vm->stack_top++] = value;
}
static inline Value vm_pop(VM *vm)
{
    if (unlikely(vm->stack_top <= 0))
    {
        fprintf(stderr, "Stack underflow at PC=%d\n", vm->pc - 1);
        exit(1);
    }
    return vm->stack[--vm->stack_top];
}

// Print value
void value_print(Value val)
{
    switch (val.type)
    {
    case VAL_INT:
        printf("%ld", val.as.int64);
        break;
    case VAL_BIGINT:
        gmp_printf("%Zd", *val.as.bigint);
        break;
    case VAL_F64:
        // Always show at least one decimal place for floats
        if (val.as.f64 == (long)val.as.f64)
            printf("%.1f", val.as.f64);
        else
            printf("%g", val.as.f64);
        break;
    case VAL_STR:
        printf("%s", val.as.str);
        break;
    case VAL_BOOL:
        printf("%s", val.as.boolean ? "true" : "false");
        break;
    case VAL_VOID:
        break;
    case VAL_LIST:
        printf("[");
        for (int i = 0; i < val.as.list->length; i++)
        {
            if (i > 0)
                printf(", ");
            value_print(val.as.list->items[i]);
        }
        printf("]");
        break;
    case VAL_SET:
        printf("{");
        int count = 0;
        for (int i = 0; i < val.as.set->capacity; i++)
        {
            if (val.as.set->entries[i].occupied)
            {
                if (count > 0)
                    printf(", ");
                value_print(val.as.set->entries[i].key);
                count++;
            }
        }
        printf("}");
        break;
    case VAL_STRUCT:
        // Print struct in a simple format (can be improved)
        printf("<struct>");
        break;
    case VAL_PYOBJECT:
        // Print Python object using str()
        if (val.as.pyobj) {
            PyObject *str_obj = PyObject_Str(val.as.pyobj);
            if (str_obj) {
                const char *str_val = PyUnicode_AsUTF8(str_obj);
                if (str_val) {
                    printf("%s", str_val);
                }
                Py_DECREF(str_obj);
            }
        }
        break;
    }
}

// Convert value to string
Value value_to_string(Value val)
{
    Value result;
    result.type = VAL_STR;
    switch (val.type)
    {
    case VAL_INT:
        result.as.str = malloc(32);
        sprintf(result.as.str, "%ld", val.as.int64);
        break;
    case VAL_BIGINT:
        result.as.str = mpz_get_str(NULL, 10, *val.as.bigint);
        break;
    case VAL_F64:
        result.as.str = malloc(64);
        // Always show at least one decimal place for floats
        if (val.as.f64 == (long)val.as.f64)
            sprintf(result.as.str, "%.1f", val.as.f64);
        else
            sprintf(result.as.str, "%g", val.as.f64);
        break;
    case VAL_STR:
        result.as.str = strdup(val.as.str);
        break;
    case VAL_BOOL:
        result.as.str = strdup(val.as.boolean ? "true" : "false");
        break;
    case VAL_VOID:
        result.as.str = strdup("");
        break;
    case VAL_LIST:
    {
        // Build string representation of list
        char *buffer = malloc(4096);
        int offset = 0;
        offset += sprintf(buffer + offset, "[");
        for (int i = 0; i < val.as.list->length; i++)
        {
            if (i > 0)
                offset += sprintf(buffer + offset, ", ");
            Value str_val = value_to_string(val.as.list->items[i]);
            offset += sprintf(buffer + offset, "%s", str_val.as.str);
            value_free(str_val);
        }
        offset += sprintf(buffer + offset, "]");
        result.as.str = strdup(buffer);
        free(buffer);
        break;
    }
    case VAL_SET:
    {
        // Build string representation of set
        char *buffer = malloc(4096);
        int offset = 0;
        offset += sprintf(buffer + offset, "{");
        int count = 0;
        for (int i = 0; i < val.as.set->capacity; i++)
        {
            if (val.as.set->entries[i].occupied)
            {
                if (count > 0)
                    offset += sprintf(buffer + offset, ", ");
                Value str_val = value_to_string(val.as.set->entries[i].key);
                offset += sprintf(buffer + offset, "%s", str_val.as.str);
                value_free(str_val);
                count++;
            }
        }
        offset += sprintf(buffer + offset, "}");
        result.as.str = strdup(buffer);
        free(buffer);
        break;
    }
    case VAL_STRUCT:
        result.as.str = strdup("<struct>");
        break;
    case VAL_PYOBJECT:
        // Convert Python object to string using str()
        if (val.as.pyobj) {
            PyObject *str_obj = PyObject_Str(val.as.pyobj);
            if (str_obj) {
                const char *str_val = PyUnicode_AsUTF8(str_obj);
                if (str_val) {
                    result.as.str = strdup(str_val);
                } else {
                    result.as.str = strdup("<pyobject>");
                }
                Py_DECREF(str_obj);
            } else {
                result.as.str = strdup("<pyobject>");
            }
        } else {
            result.as.str = strdup("<pyobject>");
        }
        break;
    }
    return result;
}

// Helper: Check if int64 addition would overflow
static inline bool would_add_overflow(int64_t a, int64_t b)
{
    if (b > 0 && a > INT64_MAX - b)
        return true;
    if (b < 0 && a < INT64_MIN - b)
        return true;
    return false;
}

// Helper: Check if int64 multiplication would overflow
static inline bool would_mul_overflow(int64_t a, int64_t b)
{
    if (a == 0 || b == 0)
        return false;
    if (a == INT64_MIN || b == INT64_MIN)
        return true;
    if (a > 0 && b > 0 && a > INT64_MAX / b)
        return true;
    if (a < 0 && b < 0 && a < INT64_MAX / b)
        return true;
    if (a > 0 && b < 0 && b < INT64_MIN / a)
        return true;
    if (a < 0 && b > 0 && a < INT64_MIN / b)
        return true;
    return false;
}

// Helper: Promote int64 to bigint
static inline Value promote_to_bigint(int64_t val)
{
    Value result;
    result.type = VAL_BIGINT;
    result.as.bigint = malloc(sizeof(mpz_t));
    mpz_init_set_si(*result.as.bigint, val);
    return result;
}

// Arithmetic operations - inline for performance
static inline Value value_add(Value a, Value b)
{
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 addition with overflow detection
        if (would_add_overflow(a.as.int64, b.as.int64))
        {
            // Overflow detected - promote to bigint
            Value a_big = promote_to_bigint(a.as.int64);
            Value b_big = promote_to_bigint(b.as.int64);
            Value result;
            result.type = VAL_BIGINT;
            result.as.bigint = malloc(sizeof(mpz_t));
            mpz_init(*result.as.bigint);
            mpz_add(*result.as.bigint, *a_big.as.bigint, *b_big.as.bigint);
            value_free(a_big);
            value_free(b_big);
            return result;
        }
        Value result;
        result.type = VAL_INT;
        result.as.int64 = a.as.int64 + b.as.int64;
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path: arbitrary precision
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_add(*result.as.bigint, *a.as.bigint, *b.as.bigint);
        return result;
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int + bigint -> bigint
        Value a_big = promote_to_bigint(a.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_add(*result.as.bigint, *a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint + int -> bigint
        Value b_big = promote_to_bigint(b.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_add(*result.as.bigint, *a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
        return result;
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        return value_make_f64(a.as.f64 + b.as.f64);
    }
    else if (a.type == VAL_STR && b.type == VAL_STR)
    {
        Value result;
        result.type = VAL_STR;
        result.as.str = malloc(strlen(a.as.str) + strlen(b.as.str) + 1);
        strcpy(result.as.str, a.as.str);
        strcat(result.as.str, b.as.str);
        return result;
    }
    fprintf(stderr, "Type error in addition\n");
    exit(1);
}

Value value_sub(Value a, Value b)
{
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 subtraction
        Value result;
        result.type = VAL_INT;
        result.as.int64 = a.as.int64 - b.as.int64;
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_sub(*result.as.bigint, *a.as.bigint, *b.as.bigint);
        return result;
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int - bigint -> bigint
        Value a_big = promote_to_bigint(a.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_sub(*result.as.bigint, *a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint - int -> bigint
        Value b_big = promote_to_bigint(b.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_sub(*result.as.bigint, *a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
        return result;
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        return value_make_f64(a.as.f64 - b.as.f64);
    }
    fprintf(stderr, "Type error in subtraction\n");
    exit(1);
}

Value value_mul(Value a, Value b)
{
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 multiplication with overflow detection
        if (would_mul_overflow(a.as.int64, b.as.int64))
        {
            // Overflow detected - promote to bigint
            Value a_big = promote_to_bigint(a.as.int64);
            Value b_big = promote_to_bigint(b.as.int64);
            Value result;
            result.type = VAL_BIGINT;
            result.as.bigint = malloc(sizeof(mpz_t));
            mpz_init(*result.as.bigint);
            mpz_mul(*result.as.bigint, *a_big.as.bigint, *b_big.as.bigint);
            value_free(a_big);
            value_free(b_big);
            return result;
        }
        Value result;
        result.type = VAL_INT;
        result.as.int64 = a.as.int64 * b.as.int64;
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mul(*result.as.bigint, *a.as.bigint, *b.as.bigint);
        return result;
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int * bigint -> bigint
        Value a_big = promote_to_bigint(a.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mul(*result.as.bigint, *a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint * int -> bigint
        Value b_big = promote_to_bigint(b.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mul(*result.as.bigint, *a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
        return result;
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        return value_make_f64(a.as.f64 * b.as.f64);
    }
    fprintf(stderr, "Type error in multiplication\n");
    exit(1);
}

Value value_div(Value a, Value b)
{
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 division
        Value result;
        result.type = VAL_INT;
        result.as.int64 = a.as.int64 / b.as.int64;
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_fdiv_q(*result.as.bigint, *a.as.bigint, *b.as.bigint); // Floor division
        return result;
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int / bigint -> bigint
        Value a_big = promote_to_bigint(a.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_fdiv_q(*result.as.bigint, *a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint / int -> bigint
        Value b_big = promote_to_bigint(b.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_fdiv_q(*result.as.bigint, *a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
        return result;
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        return value_make_f64(a.as.f64 / b.as.f64);
    }
    fprintf(stderr, "Type error in division\n");
    exit(1);
}

Value value_mod(Value a, Value b)
{
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 modulo
        Value result;
        result.type = VAL_INT;
        result.as.int64 = a.as.int64 % b.as.int64;
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mod(*result.as.bigint, *a.as.bigint, *b.as.bigint);
        return result;
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int % bigint -> bigint
        Value a_big = promote_to_bigint(a.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mod(*result.as.bigint, *a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
        return result;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint % int -> bigint
        Value b_big = promote_to_bigint(b.as.int64);
        Value result;
        result.type = VAL_BIGINT;
        result.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result.as.bigint);
        mpz_mod(*result.as.bigint, *a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
        return result;
    }
    fprintf(stderr, "Type error in modulo\n");
    exit(1);
}
// Comparison operations - inline for performance
static inline Value value_compare(Value a, Value b, OpCode op)
{
    int cmp = 0;
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        // Fast path: native int64 comparison
        if (a.as.int64 < b.as.int64)
            cmp = -1;
        else if (a.as.int64 > b.as.int64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
    {
        // GMP path
        cmp = mpz_cmp(*a.as.bigint, *b.as.bigint);
    }
    else if (a.type == VAL_INT && b.type == VAL_BIGINT)
    {
        // Mixed: int vs bigint
        Value a_big = promote_to_bigint(a.as.int64);
        cmp = mpz_cmp(*a_big.as.bigint, *b.as.bigint);
        value_free(a_big);
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_INT)
    {
        // Mixed: bigint vs int
        Value b_big = promote_to_bigint(b.as.int64);
        cmp = mpz_cmp(*a.as.bigint, *b_big.as.bigint);
        value_free(b_big);
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        if (a.as.f64 < b.as.f64)
            cmp = -1;
        else if (a.as.f64 > b.as.f64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_INT && b.type == VAL_F64)
    {
        // Mixed: int vs float - convert int to float for comparison
        double a_f64 = (double)a.as.int64;
        if (a_f64 < b.as.f64)
            cmp = -1;
        else if (a_f64 > b.as.f64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_F64 && b.type == VAL_INT)
    {
        // Mixed: float vs int - convert int to float for comparison
        double b_f64 = (double)b.as.int64;
        if (a.as.f64 < b_f64)
            cmp = -1;
        else if (a.as.f64 > b_f64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_BIGINT && b.type == VAL_F64)
    {
        // Mixed: bigint vs float - convert bigint to float for comparison
        double a_f64 = mpz_get_d(*a.as.bigint);
        if (a_f64 < b.as.f64)
            cmp = -1;
        else if (a_f64 > b.as.f64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_F64 && b.type == VAL_BIGINT)
    {
        // Mixed: float vs bigint - convert bigint to float for comparison
        double b_f64 = mpz_get_d(*b.as.bigint);
        if (a.as.f64 < b_f64)
            cmp = -1;
        else if (a.as.f64 > b_f64)
            cmp = 1;
        else
            cmp = 0;
    }
    else if (a.type == VAL_STR && b.type == VAL_STR)
    {
        cmp = strcmp(a.as.str, b.as.str);
    }
    else if (a.type == VAL_BOOL && b.type == VAL_BOOL)
    {
        cmp = a.as.boolean == b.as.boolean ? 0 : (a.as.boolean ? 1 : -1);
    }
    else
    {
        fprintf(stderr, "Type error in comparison\n");
        exit(1);
    }
    bool result = false;
    switch (op)
    {
    case OP_CMP_EQ:
        result = (cmp == 0);
        break;
    case OP_CMP_NE:
        result = (cmp != 0);
        break;
    case OP_CMP_LT:
        result = (cmp < 0);
        break;
    case OP_CMP_GT:
        result = (cmp > 0);
        break;
    case OP_CMP_LE:
        result = (cmp <= 0);
        break;
    case OP_CMP_GE:
        result = (cmp >= 0);
        break;
    default:
        fprintf(stderr, "Invalid comparison op\n");
        exit(1);
    }
    return value_make_bool(result);
}

// Python <-> fr Value conversion functions
PyObject* fr_value_to_python(Value val) {
    switch (val.type) {
        case VAL_INT:
            return PyLong_FromLong(val.as.int64);
        case VAL_BIGINT:
            // Convert GMP bigint to Python int
            {
                char *str = mpz_get_str(NULL, 10, *val.as.bigint);
                PyObject *result = PyLong_FromString(str, NULL, 10);
                free(str);
                return result;
            }
        case VAL_F64:
            return PyFloat_FromDouble(val.as.f64);
        case VAL_STR:
            return PyUnicode_FromString(val.as.str);
        case VAL_BOOL:
            return PyBool_FromLong(val.as.boolean ? 1 : 0);
        case VAL_LIST: {
            PyObject *list = PyList_New(val.as.list->length);
            for (int i = 0; i < val.as.list->length; i++) {
                PyList_SetItem(list, i, fr_value_to_python(val.as.list->items[i]));
            }
            return list;
        }
        case VAL_PYOBJECT:
            // Just increment reference and return
            Py_XINCREF(val.as.pyobj);
            return val.as.pyobj;
        case VAL_VOID:
        default:
            Py_RETURN_NONE;
    }
}

Value python_to_fr_value(PyObject *obj) {
    if (obj == NULL || obj == Py_None) {
        return value_make_void();
    } else if (PyLong_Check(obj)) {
        long val = PyLong_AsLong(obj);
        if (val == -1 && PyErr_Occurred()) {
            // Number too large for long, convert to bigint
            PyErr_Clear();
            PyObject *str_obj = PyObject_Str(obj);
            const char *str = PyUnicode_AsUTF8(str_obj);
            Value result = value_make_bigint(str);
            Py_DECREF(str_obj);
            return result;
        }
        return value_make_int_si(val);
    } else if (PyFloat_Check(obj)) {
        return value_make_f64(PyFloat_AsDouble(obj));
    } else if (PyUnicode_Check(obj)) {
        const char *str = PyUnicode_AsUTF8(obj);
        return value_make_str(str ? str : "");
    } else if (PyBool_Check(obj)) {
        return value_make_bool(obj == Py_True);
    } else if (PyList_Check(obj)) {
        Value list_val = value_make_list();
        Py_ssize_t size = PyList_Size(obj);
        for (Py_ssize_t i = 0; i < size; i++) {
            PyObject *item = PyList_GetItem(obj, i);
            Value fr_item = python_to_fr_value(item);
            list_append(list_val.as.list, fr_item);
            value_free(fr_item);
        }
        return list_val;
    } else {
        // Store as Python object
        Value result;
        result.type = VAL_PYOBJECT;
        result.as.pyobj = obj;
        Py_XINCREF(obj);
        return result;
    }
}

// Find function by name
Function *vm_find_function(VM *vm, const char *name) {
    for (int i = 0; i < vm->func_count; i++)
    {
        if (strcmp(vm->functions[i].name, name) == 0)
        {
            return &vm->functions[i];
        }
    }
    return NULL;
}

// Find which function a PC belongs to
int vm_find_function_for_pc(VM *vm, int pc) {
    for (int i = 0; i < vm->func_count; i++) {
        if (pc >= vm->functions[i].start_pc && pc <= vm->functions[i].end_pc) {
            return i;
        }
    }
    return -1;  // Not in any function (global code)
}

// Find label by name within a specific function
int vm_find_label(VM *vm, const char *name, int func_index) {
    for (int i = 0; i < vm->label_count; i++)
    {
        if (vm->labels[i].func_index == func_index && strcmp(vm->labels[i].name, name) == 0)
        {
            return vm->labels[i].pc;
        }
    }
    return -1;
}
// Load bytecode from file
bool vm_load_bytecode(VM *vm, const char *filename) {
    FILE *file = fopen(filename, "r");
    if (!file)
    {
        fprintf(stderr, "Failed to open bytecode file: %s\n", filename);
        return false;
    }
    char line[MAX_STRING_LEN];
    Function *current_func = NULL;
    int arg_count = 0;
    int local_count = 0;
    while (fgets(line, sizeof(line), file))
    {
        // Remove newline
        line[strcspn(line, "\n")] = 0;
        // Skip empty lines and comments
        if (line[0] == '\0' || line[0] == '#')
            continue;
        // Trim leading whitespace
        char *trimmed = line;
        while (*trimmed == ' ' || *trimmed == '\t')
            trimmed++;
        // Parse instruction
        char *token = strtok(trimmed, " ");
        if (!token)
            continue;
        // Handle directives
        if (strcmp(token, ".version") == 0)
        {
            // Skip version line
            continue;
        }
        else if (strcmp(token, ".struct") == 0)
        {
            // .struct <id> <field_count> <field_names...>
            char *struct_id_str = strtok(NULL, " ");
            char *field_count_str = strtok(NULL, " ");

            if (!struct_id_str || !field_count_str)
            {
                fprintf(stderr, "Invalid .struct directive\n");
                return false;
            }

            int struct_id = safe_atoi(struct_id_str);
            int field_count = safe_atoi(field_count_str);

            if (struct_id < 0 || struct_id >= MAX_STRUCTS)
            {
                fprintf(stderr, "Struct ID out of range: %d\n", struct_id);
                return false;
            }

            vm->structs[struct_id].field_count = field_count;

            // Parse field names
            for (int i = 0; i < field_count && i < MAX_STRUCT_FIELDS; i++)
            {
                char *field_name = strtok(NULL, " ");
                if (field_name)
                {
                    vm->structs[struct_id].field_names[i] = strdup(field_name);
                }
            }

            vm->struct_count = (vm->struct_count > struct_id + 1) ? vm->struct_count : (struct_id + 1);
            continue;
        }
        else if (strcmp(token, ".func") == 0)
        {
            char *name = strtok(NULL, " ");
            strtok(NULL, " "); // return_type (unused for now)
            char *total_vars = strtok(NULL, " ");
            current_func = &vm->functions[vm->func_count++];
            current_func->name = strdup(name);
            current_func->start_pc = vm->code_count;
            current_func->return_type = VAL_INT; // Default
            arg_count = 0;
            local_count = safe_atoi(total_vars ? total_vars : "0");
        }
        else if (strcmp(token, ".arg") == 0)
        {
            // .arg name type
            arg_count++;
            if (current_func)
            {
                current_func->arg_count = arg_count;
            }
            continue;
        }
        else if (strcmp(token, ".local") == 0)
        {
            // .local name type - just count
            continue;
        }
        else if (strcmp(token, ".end") == 0)
        {
            if (current_func)
            {
                current_func->end_pc = vm->code_count;
                current_func->local_count = local_count;
                current_func = NULL;
            }
            continue;
        }
        else if (strcmp(token, ".entry") == 0)
        {
            char *name = strtok(NULL, " ");
            vm->entry_point = strdup(name);
            continue;
        }
        else if (strcmp(token, "LABEL") == 0)
        {
            char *name = strtok(NULL, " ");
            vm->labels[vm->label_count].name = strdup(name);
            vm->labels[vm->label_count].pc = vm->code_count;
            vm->labels[vm->label_count].func_index = current_func ? (current_func - vm->functions) : -1;
            vm->label_count++;
            // Add LABEL instruction
            vm->code[vm->code_count].op = OP_LABEL;
            vm->code[vm->code_count].operand.index = vm->label_count - 1;
            vm->code_count++;
        }
        else
        {
            // Parse opcode
            Instruction inst;
            if (strcmp(token, "CONST_I64") == 0)
            {
                // Support CONST_I64 with multiple arguments: CONST_I64 n1 n2 n3...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: CONST_I64 requires at least one argument\n");
                    exit(1);
                }

                // Parse all values
                int64_t values[256];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");
                while (val_str != NULL && count < 256)
                {
                    values[count++] = safe_atoll(val_str);
                    val_str = strtok(NULL, " ");
                }

                if (count == 1)
                {
                    // Single CONST_I64 - use original instruction
                    inst.op = OP_CONST_I64;
                    inst.operand.int64 = values[0];
                }
                else
                {
                    // Multiple CONST_I64 - use CONST_I64_MULTI with safe struct
                    inst.op = OP_CONST_I64_MULTI;
                    MultiInt64 *multi = malloc(sizeof(MultiInt64) + count * sizeof(int64_t));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        multi->values[j] = values[j];
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "CONST_F64") == 0)
            {
                // Support CONST_F64 with multiple arguments
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: CONST_F64 requires at least one argument\n");
                    exit(1);
                }

                // Parse all values
                double values[256];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");
                while (val_str != NULL && count < 256)
                {
                    values[count++] = safe_atof(val_str);
                    val_str = strtok(NULL, " ");
                }

                if (count == 1)
                {
                    // Single CONST_F64 - use original instruction
                    inst.op = OP_CONST_F64;
                    inst.operand.f64 = values[0];
                }
                else
                {
                    // Multiple CONST_F64 - use CONST_F64_MULTI with safe struct
                    inst.op = OP_CONST_F64_MULTI;
                    MultiF64 *multi = malloc(sizeof(MultiF64) + count * sizeof(double));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        multi->values[j] = values[j];
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "CONST_STR") == 0)
            {
                // Support CONST_STR with multiple arguments
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: CONST_STR requires at least one argument\n");
                    exit(1);
                }

                // Parse quoted strings
                char *strings[256];
                int count = 0;

                // Manually parse quoted strings from rest_of_line
                char *p = rest_of_line;
                while (*p && count < 256)
                {
                    // Skip whitespace
                    while (*p == ' ' || *p == '\t') p++;
                    if (*p == '\0') break;

                    // Expect a quote
                    if (*p != '"')
                    {
                        fprintf(stderr, "Error: CONST_STR expects quoted strings\n");
                        exit(1);
                    }
                    p++; // Skip opening quote

                    // Find closing quote (considering escapes)
                    char *start = p;
                    while (*p && (*p != '"' || (p > start && *(p-1) == '\\')))
                    {
                        p++;
                    }

                    if (*p != '"')
                    {
                        fprintf(stderr, "Error: CONST_STR unterminated string\n");
                        exit(1);
                    }

                    // Extract and unescape the string
                    size_t len = p - start;
                    char *temp = malloc(len + 1);
                    strncpy(temp, start, len);
                    temp[len] = '\0';
                    strings[count++] = unescape_string(temp);
                    free(temp);

                    p++; // Skip closing quote
                }

                if (count == 1)
                {
                    // Single CONST_STR - use original instruction
                    inst.op = OP_CONST_STR;
                    // Intern the string constant
                    const char *interned = string_intern_add(vm->string_intern, strings[0]);
                    free(strings[0]); // Free the duplicate
                    inst.operand.str_val = (char*)interned; // Store interned pointer
                }
                else
                {
                    // Multiple CONST_STR - use CONST_STR_MULTI with safe struct
                    inst.op = OP_CONST_STR_MULTI;
                    MultiStr *multi = malloc(sizeof(MultiStr) + count * sizeof(char*));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        // Intern each string
                        const char *interned = string_intern_add(vm->string_intern, strings[j]);
                        free(strings[j]); // Free the duplicate
                        multi->values[j] = (char*)interned; // Store interned pointer
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "CONST_BOOL") == 0)
            {
                // Support CONST_BOOL with multiple arguments
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: CONST_BOOL requires at least one argument\n");
                    exit(1);
                }

                // Parse all values
                int values[256];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");
                while (val_str != NULL && count < 256)
                {
                    if (strcmp(val_str, "true") == 0 || strcmp(val_str, "1") == 0)
                        values[count++] = 1;
                    else
                        values[count++] = 0;
                    val_str = strtok(NULL, " ");
                }

                if (count == 1)
                {
                    // Single CONST_BOOL - use original instruction
                    inst.op = OP_CONST_BOOL;
                    inst.operand.index = values[0];
                }
                else
                {
                    // Multiple CONST_BOOL - use CONST_BOOL_MULTI with safe struct
                    inst.op = OP_CONST_BOOL_MULTI;
                    MultiInt *multi = malloc(sizeof(MultiInt) + count * sizeof(int));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        multi->values[j] = values[j];
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "LOAD") == 0)
            {
                // Support LOAD with multiple arguments: LOAD n1 n2 n3...
                // Count how many arguments we have
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: LOAD requires at least one argument\n");
                    exit(1);
                }
                // Parse all indices
                int indices[256];
                int count = 0;
                char *idx_str = strtok(rest_of_line, " ");
                while (idx_str != NULL && count < 256)
                {
                    indices[count++] = safe_atoi(idx_str);
                    idx_str = strtok(NULL, " ");
                }
                if (count == 1)
                {
                    // Single LOAD - use original instruction
                    inst.op = OP_LOAD;
                    inst.operand.index = indices[0];
                }
                else
                {
                    // Multiple LOAD - use LOAD_MULTI with safe struct
                    inst.op = OP_LOAD_MULTI;
                    MultiInt *multi = malloc(sizeof(MultiInt) + count * sizeof(int));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        multi->values[j] = indices[j];
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "STORE") == 0)
            {
                char *idx = strtok(NULL, " ");
                inst.op = OP_STORE;
                inst.operand.index = safe_atoi(idx);
            }
            else if (strcmp(token, "STORE_CONST_I64") == 0)
            {
                // STORE_CONST_I64 slot1 val1 slot2 val2 ...
                // Parse pairs of (slot, value) - count pairs, allocate, then parse again
                int64_t temp_pairs[128];  // Temporary storage
                int pair_count = 0;

                char *arg;
                while ((arg = strtok(NULL, " ")) != NULL && pair_count < 128)
                {
                    temp_pairs[pair_count++] = safe_atoll(arg);
                }

                if (pair_count == 0 || pair_count % 2 != 0)
                {
                    fprintf(stderr, "Error: STORE_CONST_I64 requires even number of arguments (slot/value pairs)\n");
                    exit(1);
                }

                StoreConstI64 *multi = malloc(sizeof(StoreConstI64) + pair_count * sizeof(int64_t));
                multi->count = pair_count / 2;
                memcpy(multi->pairs, temp_pairs, pair_count * sizeof(int64_t));

                inst.op = OP_STORE_CONST_I64;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "STORE_CONST_F64") == 0)
            {
                // STORE_CONST_F64 slot1 val1 slot2 val2 ...
                int64_t temp_pairs[128];
                int pair_count = 0;

                char *arg;
                int idx = 0;
                while ((arg = strtok(NULL, " ")) != NULL && idx < 128)
                {
                    if (idx % 2 == 0) {
                        // Slot index
                        temp_pairs[idx] = safe_atoi(arg);
                    } else {
                        // Float value - store as bits
                        double val = atof(arg);
                        memcpy(&temp_pairs[idx], &val, sizeof(double));
                    }
                    idx++;
                }
                pair_count = idx;

                if (pair_count == 0 || pair_count % 2 != 0)
                {
                    fprintf(stderr, "Error: STORE_CONST_F64 requires even number of arguments (slot/value pairs)\n");
                    exit(1);
                }

                StoreConstF64 *multi = malloc(sizeof(StoreConstF64) + pair_count * sizeof(int64_t));
                multi->count = pair_count / 2;
                memcpy(multi->pairs, temp_pairs, pair_count * sizeof(int64_t));

                inst.op = OP_STORE_CONST_F64;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "STORE_CONST_BOOL") == 0)
            {
                // STORE_CONST_BOOL slot1 val1 slot2 val2 ...
                int64_t temp_pairs[128];
                int pair_count = 0;

                char *arg;
                int idx = 0;
                while ((arg = strtok(NULL, " ")) != NULL && idx < 128)
                {
                    if (idx % 2 == 0) {
                        // Slot
                        temp_pairs[idx] = safe_atoi(arg);
                    } else {
                        // Bool value
                        if (strcmp(arg, "true") == 0) {
                            temp_pairs[idx] = 1;
                        } else if (strcmp(arg, "false") == 0) {
                            temp_pairs[idx] = 0;
                        } else {
                            temp_pairs[idx] = safe_atoi(arg);
                        }
                    }
                    idx++;
                }
                pair_count = idx;

                if (pair_count == 0 || pair_count % 2 != 0)
                {
                    fprintf(stderr, "Error: STORE_CONST_BOOL requires even number of arguments (slot/value pairs)\n");
                    exit(1);
                }

                StoreConstBool *multi = malloc(sizeof(StoreConstBool) + pair_count * sizeof(int64_t));
                multi->count = pair_count / 2;
                memcpy(multi->pairs, temp_pairs, pair_count * sizeof(int64_t));

                inst.op = OP_STORE_CONST_BOOL;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "STORE_CONST_STR") == 0)
            {
                // STORE_CONST_STR slot1 "val1" slot2 "val2" ...
                // For now, skip this optimization - strings are complex
                // Just treat it as unsupported and don't merge
                fprintf(stderr, "Error: STORE_CONST_STR not yet implemented\n");
                exit(1);
            }
            else if (strcmp(token, "STORE_REF") == 0)
            {
                char *idx = strtok(NULL, " ");
                inst.op = OP_STORE_REF;
                inst.operand.index = safe_atoi(idx);
            }
            else if (strcmp(token, "ADD_I64") == 0)
                inst.op = OP_ADD_I64;
            else if (strcmp(token, "SUB_I64") == 0)
                inst.op = OP_SUB_I64;
            else if (strcmp(token, "MUL_I64") == 0)
                inst.op = OP_MUL_I64;
            else if (strcmp(token, "DIV_I64") == 0)
                inst.op = OP_DIV_I64;
            else if (strcmp(token, "MOD_I64") == 0)
                inst.op = OP_MOD_I64;
            else if (strcmp(token, "ADD_F64") == 0)
                inst.op = OP_ADD_F64;
            else if (strcmp(token, "SUB_F64") == 0)
                inst.op = OP_SUB_F64;
            else if (strcmp(token, "MUL_F64") == 0)
                inst.op = OP_MUL_F64;
            else if (strcmp(token, "DIV_F64") == 0)
                inst.op = OP_DIV_F64;
            else if (strcmp(token, "ADD_STR") == 0)
                inst.op = OP_ADD_STR;
            else if (strcmp(token, "CMP_EQ") == 0)
                inst.op = OP_CMP_EQ;
            else if (strcmp(token, "CMP_NE") == 0)
                inst.op = OP_CMP_NE;
            else if (strcmp(token, "CMP_LT") == 0)
                inst.op = OP_CMP_LT;
            else if (strcmp(token, "CMP_GT") == 0)
                inst.op = OP_CMP_GT;
            else if (strcmp(token, "CMP_LE") == 0)
                inst.op = OP_CMP_LE;
            else if (strcmp(token, "CMP_GE") == 0)
                inst.op = OP_CMP_GE;
            else if (strcmp(token, "AND") == 0)
                inst.op = OP_AND;
            else if (strcmp(token, "OR") == 0)
                inst.op = OP_OR;
            else if (strcmp(token, "NOT") == 0)
                inst.op = OP_NOT;
            else if (strcmp(token, "JUMP") == 0)
            {
                char *label = strtok(NULL, " ");
                inst.op = OP_JUMP;
                inst.operand.str_val = strdup(label);
            }
            else if (strcmp(token, "JUMP_IF_FALSE") == 0)
            {
                char *label = strtok(NULL, " ");
                inst.op = OP_JUMP_IF_FALSE;
                inst.operand.str_val = strdup(label);
            }
            else if (strcmp(token, "JUMP_IF_TRUE") == 0)
            {
                char *label = strtok(NULL, " ");
                inst.op = OP_JUMP_IF_TRUE;
                inst.operand.str_val = strdup(label);
            }
            else if (strcmp(token, "GOTO_CALL") == 0)
            {
                char *label = strtok(NULL, " ");
                inst.op = OP_GOTO_CALL;
                inst.operand.str_val = strdup(label);
            }
            else if (strcmp(token, "CALL") == 0)
            {
                char *name = strtok(NULL, " ");
                inst.op = OP_CALL;
                inst.operand.str_val = strdup(name);
            }
            else if (strcmp(token, "RETURN") == 0)
                inst.op = OP_RETURN;
            else if (strcmp(token, "RETURN_VOID") == 0)
                inst.op = OP_RETURN_VOID;
            else if (strcmp(token, "BUILTIN_PRINT") == 0)
                inst.op = OP_BUILTIN_PRINT;
            else if (strcmp(token, "BUILTIN_PRINTLN") == 0)
                inst.op = OP_BUILTIN_PRINTLN;
            else if (strcmp(token, "BUILTIN_STR") == 0)
                inst.op = OP_BUILTIN_STR;
            else if (strcmp(token, "BUILTIN_LEN") == 0)
                inst.op = OP_BUILTIN_LEN;
            else if (strcmp(token, "BUILTIN_SQRT") == 0)
                inst.op = OP_BUILTIN_SQRT;
            else if (strcmp(token, "BUILTIN_ROUND") == 0)
                inst.op = OP_BUILTIN_ROUND;
            else if (strcmp(token, "BUILTIN_FLOOR") == 0)
                inst.op = OP_BUILTIN_FLOOR;
            else if (strcmp(token, "BUILTIN_CEIL") == 0)
                inst.op = OP_BUILTIN_CEIL;
            else if (strcmp(token, "BUILTIN_PI") == 0)
                inst.op = OP_BUILTIN_PI;
            else if (strcmp(token, "LIST_NEW") == 0)
                inst.op = OP_LIST_NEW;
            else if (strcmp(token, "LIST_APPEND") == 0)
                inst.op = OP_LIST_APPEND;
            else if (strcmp(token, "LIST_GET") == 0)
                inst.op = OP_LIST_GET;
            else if (strcmp(token, "LIST_SET") == 0)
                inst.op = OP_LIST_SET;
            else if (strcmp(token, "LIST_LEN") == 0)
                inst.op = OP_LIST_LEN;
            else if (strcmp(token, "LIST_POP") == 0)
                inst.op = OP_LIST_POP;
            else if (strcmp(token, "SET_NEW") == 0)
                inst.op = OP_SET_NEW;
            else if (strcmp(token, "SET_ADD") == 0)
                inst.op = OP_SET_ADD;
            else if (strcmp(token, "SET_REMOVE") == 0)
                inst.op = OP_SET_REMOVE;
            else if (strcmp(token, "SET_CONTAINS") == 0)
                inst.op = OP_SET_CONTAINS;
            else if (strcmp(token, "SET_LEN") == 0)
                inst.op = OP_SET_LEN;
            else if (strcmp(token, "CONTAINS") == 0)
                inst.op = OP_CONTAINS;
            else if (strcmp(token, "LIST_NEW_I64") == 0)
            {
                // LIST_NEW_I64 count val1 val2 val3 ...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: LIST_NEW_I64 requires count and values\n");
                    exit(1);
                }

                // Parse count and values
                int64_t values[10000];  // Support up to 10000 elements
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");

                if (val_str == NULL) {
                    fprintf(stderr, "Error: LIST_NEW_I64 missing count\n");
                    exit(1);
                }

                count = safe_atoi(val_str);

                // Now read the values
                for (int j = 0; j < count && j < 10000; j++) {
                    val_str = strtok(NULL, " ");
                    if (val_str == NULL) {
                        fprintf(stderr, "Error: LIST_NEW_I64 missing value %d\n", j);
                        exit(1);
                    }
                    values[j] = safe_atoll(val_str);
                }

                // Create the multi-value struct
                MultiInt64 *multi = malloc(sizeof(MultiInt64) + count * sizeof(int64_t));
                multi->count = count;
                for (int j = 0; j < count; j++)
                {
                    multi->values[j] = values[j];
                }

                inst.op = OP_LIST_NEW_I64;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "LIST_NEW_F64") == 0)
            {
                // LIST_NEW_F64 count val1 val2 val3 ...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: LIST_NEW_F64 requires count and values\n");
                    exit(1);
                }

                double values[10000];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");

                if (val_str == NULL) {
                    fprintf(stderr, "Error: LIST_NEW_F64 missing count\n");
                    exit(1);
                }

                count = safe_atoi(val_str);

                for (int j = 0; j < count && j < 10000; j++) {
                    val_str = strtok(NULL, " ");
                    if (val_str == NULL) {
                        fprintf(stderr, "Error: LIST_NEW_F64 missing value %d\n", j);
                        exit(1);
                    }
                    values[j] = safe_atof(val_str);
                }

                MultiF64 *multi = malloc(sizeof(MultiF64) + count * sizeof(double));
                multi->count = count;
                for (int j = 0; j < count; j++)
                {
                    multi->values[j] = values[j];
                }

                inst.op = OP_LIST_NEW_F64;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "LIST_NEW_STR") == 0)
            {
                // LIST_NEW_STR count "str1" "str2" "str3" ...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: LIST_NEW_STR requires count and values\n");
                    exit(1);
                }

                char *strings[10000];
                int count = 0;

                // Parse count
                char *p = rest_of_line;
                while (*p == ' ' || *p == '\t') p++;
                char *count_start = p;
                while (*p && *p != ' ' && *p != '\t') p++;
                if (*p) *p++ = '\0';
                count = safe_atoi(count_start);

                // Parse quoted strings
                for (int j = 0; j < count && j < 10000; j++) {
                    while (*p == ' ' || *p == '\t') p++;
                    if (*p != '"') {
                        fprintf(stderr, "Error: LIST_NEW_STR expects quoted strings\n");
                        exit(1);
                    }
                    p++; // Skip opening quote

                    char *start = p;
                    while (*p && (*p != '"' || (p > start && *(p-1) == '\\'))) {
                        p++;
                    }

                    if (*p != '"') {
                        fprintf(stderr, "Error: LIST_NEW_STR unterminated string\n");
                        exit(1);
                    }

                    size_t len = p - start;
                    char *temp = malloc(len + 1);
                    strncpy(temp, start, len);
                    temp[len] = '\0';
                    strings[j] = unescape_string(temp);
                    free(temp);

                    p++; // Skip closing quote
                }

                MultiStr *multi = malloc(sizeof(MultiStr) + count * sizeof(char*));
                multi->count = count;
                for (int j = 0; j < count; j++)
                {
                    multi->values[j] = strings[j];
                }

                inst.op = OP_LIST_NEW_STR;
                inst.operand.ptr = multi;
            }
            else if (strcmp(token, "LIST_NEW_BOOL") == 0)
            {
                // LIST_NEW_BOOL count val1 val2 val3 ...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: LIST_NEW_BOOL requires count and values\n");
                    exit(1);
                }

                int values[10000];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");

                if (val_str == NULL) {
                    fprintf(stderr, "Error: LIST_NEW_BOOL missing count\n");
                    exit(1);
                }

                count = safe_atoi(val_str);

                for (int j = 0; j < count && j < 10000; j++) {
                    val_str = strtok(NULL, " ");
                    if (val_str == NULL) {
                        fprintf(stderr, "Error: LIST_NEW_BOOL missing value %d\n", j);
                        exit(1);
                    }
                    if (strcmp(val_str, "true") == 0 || strcmp(val_str, "1") == 0)
                        values[j] = 1;
                    else
                        values[j] = 0;
                }

                MultiInt *multi = malloc(sizeof(MultiInt) + count * sizeof(int));
                multi->count = count;
                for (int j = 0; j < count; j++)
                {
                    multi->values[j] = values[j];
                }

                inst.op = OP_LIST_NEW_BOOL;
                inst.operand.ptr = multi;
            }
            // Struct operations
            else if (strcmp(token, "STRUCT_NEW") == 0)
            {
                char *struct_id_str = strtok(NULL, " ");
                inst.op = OP_STRUCT_NEW;
                inst.operand.index = safe_atoi(struct_id_str);
            }
            else if (strcmp(token, "STRUCT_GET") == 0)
            {
                char *field_idx_str = strtok(NULL, " ");
                inst.op = OP_STRUCT_GET;
                inst.operand.index = safe_atoi(field_idx_str);
            }
            else if (strcmp(token, "STRUCT_SET") == 0)
            {
                char *field_idx_str = strtok(NULL, " ");
                inst.op = OP_STRUCT_SET;
                inst.operand.index = safe_atoi(field_idx_str);
            }
            // Type conversions
            else if (strcmp(token, "TO_INT") == 0)
                inst.op = OP_TO_INT;
            else if (strcmp(token, "TO_FLOAT") == 0)
                inst.op = OP_TO_FLOAT;
            else if (strcmp(token, "TO_BOOL") == 0)
                inst.op = OP_TO_BOOL;
            else if (strcmp(token, "STR_UPPER") == 0)
                inst.op = OP_STR_UPPER;
            else if (strcmp(token, "STR_LOWER") == 0)
                inst.op = OP_STR_LOWER;
            else if (strcmp(token, "STR_STRIP") == 0)
                inst.op = OP_STR_STRIP;
            else if (strcmp(token, "ENCODE") == 0)
                inst.op = OP_ENCODE;
            else if (strcmp(token, "DECODE") == 0)
                inst.op = OP_DECODE;
            else if (strcmp(token, "STR_SPLIT") == 0)
                inst.op = OP_STR_SPLIT;
            else if (strcmp(token, "ABS") == 0)
                inst.op = OP_ABS;
            else if (strcmp(token, "POW") == 0)
                inst.op = OP_POW;
            else if (strcmp(token, "MIN") == 0)
                inst.op = OP_MIN;
            else if (strcmp(token, "MAX") == 0)
                inst.op = OP_MAX;
            else if (strcmp(token, "NEG") == 0)
                inst.op = OP_NEG;
            else if (strcmp(token, "ASSERT") == 0)
                inst.op = OP_ASSERT;
            // File I/O operations
            else if (strcmp(token, "FILE_OPEN") == 0)
                inst.op = OP_FILE_OPEN;
            else if (strcmp(token, "FILE_READ") == 0)
                inst.op = OP_FILE_READ;
            else if (strcmp(token, "FILE_WRITE") == 0)
                inst.op = OP_FILE_WRITE;
            else if (strcmp(token, "FILE_CLOSE") == 0)
                inst.op = OP_FILE_CLOSE;
            else if (strcmp(token, "FILE_EXISTS") == 0)
                inst.op = OP_FILE_EXISTS;
            else if (strcmp(token, "FILE_ISFILE") == 0)
                inst.op = OP_FILE_ISFILE;
            else if (strcmp(token, "FILE_ISDIR") == 0)
                inst.op = OP_FILE_ISDIR;
            else if (strcmp(token, "FILE_LISTDIR") == 0)
                inst.op = OP_FILE_LISTDIR;
            else if (strcmp(token, "FILE_MKDIR") == 0)
                inst.op = OP_FILE_MKDIR;
            else if (strcmp(token, "FILE_MAKEDIRS") == 0)
                inst.op = OP_FILE_MAKEDIRS;
            else if (strcmp(token, "FILE_REMOVE") == 0)
                inst.op = OP_FILE_REMOVE;
            else if (strcmp(token, "FILE_RMDIR") == 0)
                inst.op = OP_FILE_RMDIR;
            else if (strcmp(token, "FILE_RENAME") == 0)
                inst.op = OP_FILE_RENAME;
            else if (strcmp(token, "FILE_GETSIZE") == 0)
                inst.op = OP_FILE_GETSIZE;
            else if (strcmp(token, "FILE_GETCWD") == 0)
                inst.op = OP_FILE_GETCWD;
            else if (strcmp(token, "FILE_CHDIR") == 0)
                inst.op = OP_FILE_CHDIR;
            else if (strcmp(token, "FILE_ABSPATH") == 0)
                inst.op = OP_FILE_ABSPATH;
            else if (strcmp(token, "FILE_BASENAME") == 0)
                inst.op = OP_FILE_BASENAME;
            else if (strcmp(token, "FILE_DIRNAME") == 0)
                inst.op = OP_FILE_DIRNAME;
            else if (strcmp(token, "FILE_JOIN") == 0)
                inst.op = OP_FILE_JOIN;
            // Socket I/O operations
            else if (strcmp(token, "SOCKET_CREATE") == 0)
                inst.op = OP_SOCKET_CREATE;
            else if (strcmp(token, "SOCKET_CONNECT") == 0)
                inst.op = OP_SOCKET_CONNECT;
            else if (strcmp(token, "SOCKET_BIND") == 0)
                inst.op = OP_SOCKET_BIND;
            else if (strcmp(token, "SOCKET_LISTEN") == 0)
                inst.op = OP_SOCKET_LISTEN;
            else if (strcmp(token, "SOCKET_ACCEPT") == 0)
                inst.op = OP_SOCKET_ACCEPT;
            else if (strcmp(token, "SOCKET_SEND") == 0)
                inst.op = OP_SOCKET_SEND;
            else if (strcmp(token, "SOCKET_RECV") == 0)
                inst.op = OP_SOCKET_RECV;
            else if (strcmp(token, "SOCKET_CLOSE") == 0)
                inst.op = OP_SOCKET_CLOSE;
            else if (strcmp(token, "SOCKET_SETSOCKOPT") == 0)
                inst.op = OP_SOCKET_SETSOCKOPT;
            else if (strcmp(token, "FORK") == 0)
                inst.op = OP_FORK;
            else if (strcmp(token, "WAIT") == 0)
                inst.op = OP_WAIT;
            else if (strcmp(token, "STR_JOIN") == 0)
                inst.op = OP_STR_JOIN;
            else if (strcmp(token, "STR_REPLACE") == 0)
                inst.op = OP_STR_REPLACE;
            // Python library integration
            else if (strcmp(token, "PY_IMPORT") == 0)
                inst.op = OP_PY_IMPORT;
            else if (strcmp(token, "PY_CALL") == 0)
                inst.op = OP_PY_CALL;
            else if (strcmp(token, "PY_GETATTR") == 0)
                inst.op = OP_PY_GETATTR;
            else if (strcmp(token, "PY_SETATTR") == 0)
                inst.op = OP_PY_SETATTR;
            else if (strcmp(token, "PY_CALL_METHOD") == 0)
                inst.op = OP_PY_CALL_METHOD;
            // Exception handling
            else if (strcmp(token, "TRY_BEGIN") == 0)
            {
                // TRY_BEGIN "exc_type" label
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: TRY_BEGIN requires exc_type and label\n");
                    exit(1);
                }

                // Parse quoted exception type
                char *p = rest_of_line;
                while (*p == ' ' || *p == '\t') p++; // Skip whitespace

                if (*p != '"')
                {
                    fprintf(stderr, "Error: TRY_BEGIN expects quoted exception type\n");
                    exit(1);
                }
                p++; // Skip opening quote

                char *exc_type_start = p;
                while (*p && *p != '"') p++;

                if (*p != '"')
                {
                    fprintf(stderr, "Error: TRY_BEGIN unterminated string\n");
                    exit(1);
                }

                size_t exc_type_len = p - exc_type_start;
                char *exc_type = malloc(exc_type_len + 1);
                strncpy(exc_type, exc_type_start, exc_type_len);
                exc_type[exc_type_len] = '\0';

                p++; // Skip closing quote

                // Skip whitespace to get label
                while (*p == ' ' || *p == '\t') p++;

                char *label = strdup(p);

                inst.op = OP_TRY_BEGIN;
                // Store both exc_type and label as a combined string (we'll parse it in execution)
                // Format: "exc_type|label"
                size_t combined_len = strlen(exc_type) + strlen(label) + 2;
                char *combined = malloc(combined_len);
                snprintf(combined, combined_len, "%s|%s", exc_type, label);
                inst.operand.str_val = combined;

                free(exc_type);
                free(label);
            }
            else if (strcmp(token, "TRY_END") == 0)
            {
                inst.op = OP_TRY_END;
            }
            else if (strcmp(token, "RAISE") == 0)
            {
                // RAISE "exc_type" "message"
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: RAISE requires exc_type and message\n");
                    exit(1);
                }

                inst.op = OP_RAISE;
                inst.operand.str_val = strdup(rest_of_line);
            }
            else if (strcmp(token, "POP") == 0)
                inst.op = OP_POP;
            else if (strcmp(token, "DUP") == 0)
                inst.op = OP_DUP;
            else if (strcmp(token, "HALT") == 0)
                inst.op = OP_HALT;
            // Optimized instructions
            else if (strcmp(token, "INC_LOCAL") == 0)
            {
                char *idx = strtok(NULL, " ");
                inst.op = OP_INC_LOCAL;
                inst.operand.index = safe_atoi(idx);
            }
            else if (strcmp(token, "DEC_LOCAL") == 0)
            {
                char *idx = strtok(NULL, " ");
                inst.op = OP_DEC_LOCAL;
                inst.operand.index = safe_atoi(idx);
            }
            else if (strcmp(token, "ADD_CONST_I64") == 0)
            {
                // Support ADD_CONST_I64 with multiple arguments: ADD_CONST_I64 n1 n2 n3...
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: ADD_CONST_I64 requires at least one argument\n");
                    exit(1);
                }

                // Parse all values
                int64_t values[256];
                int count = 0;
                char *val_str = strtok(rest_of_line, " ");
                while (val_str != NULL && count < 256)
                {
                    values[count++] = safe_atoll(val_str);
                    val_str = strtok(NULL, " ");
                }

                if (count == 1)
                {
                    // Single value - use simple instruction
                    inst.op = OP_ADD_CONST_I64;
                    inst.operand.int64 = values[0];
                }
                else
                {
                    // Multiple values - use multi instruction with safe struct
                    inst.op = OP_ADD_CONST_I64_MULTI;
                    MultiInt64 *multi = malloc(sizeof(MultiInt64) + count * sizeof(int64_t));
                    multi->count = count;
                    for (int j = 0; j < count; j++)
                    {
                        multi->values[j] = values[j];
                    }
                    inst.operand.ptr = multi;
                }
            }
            else if (strcmp(token, "SUB_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_SUB_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "MUL_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_MUL_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "MOD_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_MOD_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "DIV_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_DIV_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "ADD_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_ADD_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "SUB_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_SUB_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "MUL_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_MUL_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "DIV_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_DIV_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "AND_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_AND_CONST;
                inst.operand.int64 = safe_atoi(val);
            }
            else if (strcmp(token, "OR_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_OR_CONST;
                inst.operand.int64 = safe_atoi(val);
            }
            else if (strcmp(token, "AND_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_AND_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "OR_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_OR_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "XOR_CONST_I64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_XOR_CONST_I64;
                inst.operand.int64 = safe_atoll(val);
            }
            // Comparison with constant
            else if (strcmp(token, "CMP_LT_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_LT_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_GT_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_GT_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_LE_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_LE_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_GE_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_GE_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_EQ_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_EQ_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_NE_CONST") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_NE_CONST;
                inst.operand.int64 = safe_atoll(val);
            }
            else if (strcmp(token, "CMP_LT_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_LT_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "CMP_GT_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_GT_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "CMP_LE_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_LE_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "CMP_GE_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_GE_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "CMP_EQ_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_EQ_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            else if (strcmp(token, "CMP_NE_CONST_F64") == 0)
            {
                char *val = strtok(NULL, " ");
                inst.op = OP_CMP_NE_CONST_F64;
                inst.operand.f64 = atof(val);
            }
            // Stack manipulation
            else if (strcmp(token, "SWAP") == 0)
                inst.op = OP_SWAP;
            else if (strcmp(token, "ROT") == 0)
                inst.op = OP_ROT;
            else if (strcmp(token, "OVER") == 0)
                inst.op = OP_OVER;
            else if (strcmp(token, "DUP2") == 0)
                inst.op = OP_DUP2;
            // Bitwise operations
            else if (strcmp(token, "AND_I64") == 0)
                inst.op = OP_AND_I64;
            else if (strcmp(token, "OR_I64") == 0)
                inst.op = OP_OR_I64;
            else if (strcmp(token, "XOR_I64") == 0)
                inst.op = OP_XOR_I64;
            else if (strcmp(token, "SHL_I64") == 0)
                inst.op = OP_SHL_I64;
            else if (strcmp(token, "SHR_I64") == 0)
                inst.op = OP_SHR_I64;
            // Variable copy
            else if (strcmp(token, "COPY_LOCAL") == 0)
            {
                char *src = strtok(NULL, " ");
                char *dst = strtok(NULL, " ");
                inst.op = OP_COPY_LOCAL;
                inst.operand.indices.src = safe_atoi(src);
                inst.operand.indices.dst = safe_atoi(dst);
            }
            else if (strcmp(token, "COPY_LOCAL_REF") == 0)
            {
                char *src = strtok(NULL, " ");
                char *dst = strtok(NULL, " ");
                inst.op = OP_COPY_LOCAL_REF;
                inst.operand.indices.src = safe_atoi(src);
                inst.operand.indices.dst = safe_atoi(dst);
            }
            // Fused load/store
            else if (strcmp(token, "FUSED_LOAD_STORE") == 0)
            {
                // FUSED_LOAD_STORE src1 dst1 src2 dst2 ... (alternating)
                inst.op = OP_FUSED_LOAD_STORE;
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: FUSED_LOAD_STORE requires arguments\n");
                    exit(1);
                }

                // Parse arguments
                int temp_args[256];
                int arg_count = 0;
                char *arg_str = strtok(rest_of_line, " ");
                while (arg_str != NULL && arg_count < 256)
                {
                    temp_args[arg_count++] = safe_atoi(arg_str);
                    arg_str = strtok(NULL, " ");
                }

                // Create safe struct
                MultiInt *multi = malloc(sizeof(MultiInt) + arg_count * sizeof(int));
                multi->count = arg_count; // Total number of arguments
                for (int j = 0; j < arg_count; j++)
                {
                    multi->values[j] = temp_args[j];
                }
                inst.operand.ptr = multi;
            }
            // Fused store/load (reverse order)
            else if (strcmp(token, "FUSED_STORE_LOAD") == 0)
            {
                // FUSED_STORE_LOAD dst1 src1 dst2 src2 ... [dstN]
                // Can have pairs plus optional trailing store
                inst.op = OP_FUSED_STORE_LOAD;
                char *rest_of_line = strtok(NULL, "\n");
                if (rest_of_line == NULL)
                {
                    fprintf(stderr, "Error: FUSED_STORE_LOAD requires arguments\n");
                    exit(1);
                }

                // Parse arguments
                int temp_args[256];
                int arg_count = 0;
                char *arg_str = strtok(rest_of_line, " ");
                while (arg_str != NULL && arg_count < 256)
                {
                    temp_args[arg_count++] = safe_atoi(arg_str);
                    arg_str = strtok(NULL, " ");
                }

                // Create safe struct
                MultiInt *multi = malloc(sizeof(MultiInt) + arg_count * sizeof(int));
                multi->count = arg_count; // Total number of arguments (not pairs)
                for (int j = 0; j < arg_count; j++)
                {
                    multi->values[j] = temp_args[j];
                }
                inst.operand.ptr = multi;
            }
            // Fused LOAD2 + arithmetic instructions
            else if (strcmp(token, "LOAD2_ADD_I64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_ADD_I64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_SUB_I64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_SUB_I64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_MUL_I64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_MUL_I64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_MOD_I64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_MOD_I64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_ADD_F64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_ADD_F64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_SUB_F64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_SUB_F64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_MUL_F64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_MUL_F64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_DIV_F64") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_DIV_F64;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            // Fused LOAD2 + comparison instructions
            else if (strcmp(token, "LOAD2_CMP_LT") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_LT;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_CMP_GT") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_GT;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_CMP_LE") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_LE;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_CMP_GE") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_GE;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_CMP_EQ") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_EQ;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "LOAD2_CMP_NE") == 0)
            {
                char *var1 = strtok(NULL, " ");
                char *var2 = strtok(NULL, " ");
                inst.op = OP_LOAD2_CMP_NE;
                inst.operand.indices.src = safe_atoi(var1);
                inst.operand.indices.dst = safe_atoi(var2);
            }
            else if (strcmp(token, "SWITCH_JUMP_TABLE") == 0)
            {
                // Format: SWITCH_JUMP_TABLE min max label1 label2 ... labelN default
                char *min_str = strtok(NULL, " ");
                char *max_str = strtok(NULL, " ");
                int min_val = safe_atoi(min_str);
                int max_val = safe_atoi(max_str);
                int num_cases = max_val - min_val + 1;

                inst.op = OP_SWITCH_JUMP_TABLE;
                inst.operand.switch_table.min_val = min_val;
                inst.operand.switch_table.max_val = max_val;
                inst.operand.switch_table.num_labels = num_cases + 1; // +1 for default

                // Allocate labels and pcs arrays
                inst.operand.switch_table.labels = malloc((num_cases + 1) * sizeof(char*));
                inst.operand.switch_table.pcs = malloc((num_cases + 1) * sizeof(int));

                // Read case labels
                for (int j = 0; j < num_cases; j++) {
                    char *label = strtok(NULL, " ");
                    if (!label) {
                        fprintf(stderr, "SWITCH_JUMP_TABLE: Missing label for case %d\n", j);
                        exit(1);
                    }
                    inst.operand.switch_table.labels[j] = strdup(label);
                }

                // Read default label
                char *default_label = strtok(NULL, " ");
                if (!default_label) {
                    fprintf(stderr, "SWITCH_JUMP_TABLE: Missing default label\n");
                    exit(1);
                }
                inst.operand.switch_table.labels[num_cases] = strdup(default_label);
            }
            else if (strcmp(token, "SELECT") == 0)
                inst.op = OP_SELECT;
            else
            {
                fprintf(stderr, "Unknown instruction: %s\n", token);
                continue;
            }
            vm->code[vm->code_count++] = inst;
        }
    }
    fclose(file);
    // Second pass: resolve all jump labels to PC indices for performance
    for (int i = 0; i < vm->code_count; i++)
    {
        Instruction *inst = &vm->code[i];
        if (inst->op == OP_JUMP || inst->op == OP_JUMP_IF_FALSE || inst->op == OP_JUMP_IF_TRUE || inst->op == OP_GOTO_CALL)
        {
            int func_index = vm_find_function_for_pc(vm, i);
            int target = vm_find_label(vm, inst->operand.str_val, func_index);
            if (target == -1)
            {
                fprintf(stderr, "Label not found during resolution: %s\n", inst->operand.str_val);
                exit(1);
            }
            // Free the string and replace with PC index
            free(inst->operand.str_val);
            inst->operand.int64 = target;
        }
        else if (inst->op == OP_SWITCH_JUMP_TABLE)
        {
            int func_index = vm_find_function_for_pc(vm, i);
            // Resolve all labels in the jump table
            for (int j = 0; j < inst->operand.switch_table.num_labels; j++)
            {
                int target = vm_find_label(vm, inst->operand.switch_table.labels[j], func_index);
                if (target == -1)
                {
                    fprintf(stderr, "Label not found in SWITCH_JUMP_TABLE: %s\n", inst->operand.switch_table.labels[j]);
                    exit(1);
                }
                inst->operand.switch_table.pcs[j] = target;
                free(inst->operand.switch_table.labels[j]);
            }
            // Free the labels array (we now use pcs array)
            free(inst->operand.switch_table.labels);
            inst->operand.switch_table.labels = NULL;
        }
    }
    return true;
}
// Execute VM
__attribute__((hot)) void vm_run(VM *vm) {
    // Set global VM for error reporting
    g_current_vm = vm;

    // Find entry point function
    Function *entry = vm_find_function(vm, vm->entry_point);
    if (!entry)
    {
        fprintf(stderr, "Entry point function not found: %s\n", vm->entry_point);
        return;
    }
    // Set up initial call frame
    vm->call_stack[0].func = entry;
    vm->call_stack[0].vars.var_count = entry->local_count;
    for (int i = 0; i < entry->local_count; i++)
    {
        vm->call_stack[0].vars.vars[i] = value_make_int_si(0);
    }

    // If main() expects an argument and we have program arguments, pass argv as a list
    if (entry->arg_count > 0)
    {
        // Create a list of argument strings
        List *argv_list = malloc(sizeof(List));
        argv_list->length = vm->prog_argc;
        argv_list->capacity = (vm->prog_argc > 0) ? vm->prog_argc : 1;  // At least capacity of 1
        argv_list->items = malloc(sizeof(Value) * argv_list->capacity);

        // Convert each C string argument to a Value string
        for (int i = 0; i < vm->prog_argc; i++)
        {
            argv_list->items[i] = value_make_str(vm->prog_argv[i]);
        }

        // Pass the list as the first argument to main
        value_free(vm->call_stack[0].vars.vars[0]);
        vm->call_stack[0].vars.vars[0] = (Value){.type = VAL_LIST, .as.list = argv_list};
    }

    vm->call_stack[0].return_pc = -1;
    vm->call_stack_top = 1;
    vm->pc = entry->start_pc;
    vm->running = true;

    // Cache current call frame to reduce pointer arithmetic overhead
    register CallFrame *current_frame = &vm->call_stack[0];

    // Main interpreter loop - optimized with aligned dispatch table for cache efficiency
    static const void *dispatch_table[] __attribute__((aligned(64))) = {
        [OP_CONST_I64] = &&L_CONST_I64,
        [OP_CONST_F64] = &&L_CONST_F64,
        [OP_CONST_STR] = &&L_CONST_STR,
        [OP_CONST_BOOL] = &&L_CONST_BOOL,
        [OP_CONST_I64_MULTI] = &&L_CONST_I64_MULTI,
        [OP_CONST_F64_MULTI] = &&L_CONST_F64_MULTI,
        [OP_CONST_STR_MULTI] = &&L_CONST_STR_MULTI,
        [OP_CONST_BOOL_MULTI] = &&L_CONST_BOOL_MULTI,
        [OP_LOAD] = &&L_LOAD,
        [OP_STORE] = &&L_STORE,
        [OP_STORE_CONST_I64] = &&L_STORE_CONST_I64,
        [OP_STORE_CONST_F64] = &&L_STORE_CONST_F64,
        [OP_STORE_CONST_BOOL] = &&L_STORE_CONST_BOOL,
        [OP_STORE_CONST_STR] = &&L_STORE_CONST_STR,
        [OP_ADD_I64] = &&L_ADD_I64,
        [OP_SUB_I64] = &&L_SUB_I64,
        [OP_MUL_I64] = &&L_MUL_I64,
        [OP_DIV_I64] = &&L_DIV_I64,
        [OP_MOD_I64] = &&L_MOD_I64,
        [OP_ADD_F64] = &&L_ADD_F64,
        [OP_SUB_F64] = &&L_SUB_F64,
        [OP_MUL_F64] = &&L_MUL_F64,
        [OP_DIV_F64] = &&L_DIV_F64,
        [OP_ADD_STR] = &&L_ADD_STR,
        [OP_CMP_EQ] = &&L_CMP_EQ,
        [OP_CMP_NE] = &&L_CMP_NE,
        [OP_CMP_LT] = &&L_CMP_LT,
        [OP_CMP_GT] = &&L_CMP_GT,
        [OP_CMP_LE] = &&L_CMP_LE,
        [OP_CMP_GE] = &&L_CMP_GE,
        [OP_AND] = &&L_AND,
        [OP_OR] = &&L_OR,
        [OP_NOT] = &&L_NOT,
        [OP_JUMP] = &&L_JUMP,
        [OP_JUMP_IF_FALSE] = &&L_JUMP_IF_FALSE,
        [OP_JUMP_IF_TRUE] = &&L_JUMP_IF_TRUE,
        [OP_GOTO_CALL] = &&L_GOTO_CALL,
        [OP_CALL] = &&L_CALL,
        [OP_RETURN] = &&L_RETURN,
        [OP_RETURN_VOID] = &&L_RETURN_VOID,
        [OP_BUILTIN_PRINT] = &&L_BUILTIN_PRINT,
        [OP_BUILTIN_PRINTLN] = &&L_BUILTIN_PRINTLN,
        [OP_BUILTIN_STR] = &&L_BUILTIN_STR,
        [OP_BUILTIN_LEN] = &&L_BUILTIN_LEN,
        [OP_BUILTIN_SQRT] = &&L_BUILTIN_SQRT,
        [OP_BUILTIN_ROUND] = &&L_BUILTIN_ROUND,
        [OP_BUILTIN_FLOOR] = &&L_BUILTIN_FLOOR,
        [OP_BUILTIN_CEIL] = &&L_BUILTIN_CEIL,
        [OP_BUILTIN_PI] = &&L_BUILTIN_PI,
        [OP_POP] = &&L_POP,
        [OP_DUP] = &&L_DUP,
        [OP_LABEL] = &&L_LABEL,
        [OP_HALT] = &&L_HALT,
        [OP_INC_LOCAL] = &&L_INC_LOCAL,
        [OP_DEC_LOCAL] = &&L_DEC_LOCAL,
        [OP_ADD_CONST_I64] = &&L_ADD_CONST_I64,
        [OP_SUB_CONST_I64] = &&L_SUB_CONST_I64,
        [OP_MUL_CONST_I64] = &&L_MUL_CONST_I64,
        [OP_MOD_CONST_I64] = &&L_MOD_CONST_I64,
        [OP_DIV_CONST_I64] = &&L_DIV_CONST_I64,
        [OP_ADD_CONST_I64_MULTI] = &&L_ADD_CONST_I64_MULTI,
        [OP_ADD_CONST_F64] = &&L_ADD_CONST_F64,
        [OP_SUB_CONST_F64] = &&L_SUB_CONST_F64,
        [OP_MUL_CONST_F64] = &&L_MUL_CONST_F64,
        [OP_DIV_CONST_F64] = &&L_DIV_CONST_F64,
        [OP_AND_CONST] = &&L_AND_CONST,
        [OP_OR_CONST] = &&L_OR_CONST,
        [OP_AND_CONST_I64] = &&L_AND_CONST_I64,
        [OP_OR_CONST_I64] = &&L_OR_CONST_I64,
        [OP_XOR_CONST_I64] = &&L_XOR_CONST_I64,
        [OP_SWAP] = &&L_SWAP,
        [OP_ROT] = &&L_ROT,
        [OP_OVER] = &&L_OVER,
        [OP_DUP2] = &&L_DUP2,
        [OP_AND_I64] = &&L_AND_I64,
        [OP_OR_I64] = &&L_OR_I64,
        [OP_XOR_I64] = &&L_XOR_I64,
        [OP_SHL_I64] = &&L_SHL_I64,
        [OP_SHR_I64] = &&L_SHR_I64,
        [OP_COPY_LOCAL] = &&L_COPY_LOCAL,
        [OP_COPY_LOCAL_REF] = &&L_COPY_LOCAL_REF,
        [OP_STORE_REF] = &&L_STORE_REF,
        [OP_LOAD_MULTI] = &&L_LOAD_MULTI,
        [OP_FUSED_LOAD_STORE] = &&L_FUSED_LOAD_STORE,
        [OP_FUSED_STORE_LOAD] = &&L_FUSED_STORE_LOAD,
        [OP_LOAD2_ADD_I64] = &&L_LOAD2_ADD_I64,
        [OP_LOAD2_SUB_I64] = &&L_LOAD2_SUB_I64,
        [OP_LOAD2_MUL_I64] = &&L_LOAD2_MUL_I64,
        [OP_LOAD2_MOD_I64] = &&L_LOAD2_MOD_I64,
        [OP_LOAD2_ADD_F64] = &&L_LOAD2_ADD_F64,
        [OP_LOAD2_SUB_F64] = &&L_LOAD2_SUB_F64,
        [OP_LOAD2_MUL_F64] = &&L_LOAD2_MUL_F64,
        [OP_LOAD2_DIV_F64] = &&L_LOAD2_DIV_F64,
        [OP_LOAD2_CMP_LT] = &&L_LOAD2_CMP_LT,
        [OP_LOAD2_CMP_GT] = &&L_LOAD2_CMP_GT,
        [OP_LOAD2_CMP_LE] = &&L_LOAD2_CMP_LE,
        [OP_LOAD2_CMP_GE] = &&L_LOAD2_CMP_GE,
        [OP_LOAD2_CMP_EQ] = &&L_LOAD2_CMP_EQ,
        [OP_LOAD2_CMP_NE] = &&L_LOAD2_CMP_NE,
        [OP_SELECT] = &&L_SELECT,
        [OP_LIST_NEW] = &&L_LIST_NEW,
        [OP_LIST_APPEND] = &&L_LIST_APPEND,
        [OP_LIST_GET] = &&L_LIST_GET,
        [OP_LIST_SET] = &&L_LIST_SET,
        [OP_LIST_LEN] = &&L_LIST_LEN,
        [OP_LIST_POP] = &&L_LIST_POP,
        [OP_LIST_NEW_I64] = &&L_LIST_NEW_I64,
        [OP_LIST_NEW_F64] = &&L_LIST_NEW_F64,
        [OP_LIST_NEW_STR] = &&L_LIST_NEW_STR,
        [OP_LIST_NEW_BOOL] = &&L_LIST_NEW_BOOL,
        [OP_SET_NEW] = &&L_SET_NEW,
        [OP_SET_ADD] = &&L_SET_ADD,
        [OP_SET_REMOVE] = &&L_SET_REMOVE,
        [OP_SET_CONTAINS] = &&L_SET_CONTAINS,
        [OP_SET_LEN] = &&L_SET_LEN,
        [OP_CONTAINS] = &&L_CONTAINS,
        [OP_STRUCT_NEW] = &&L_STRUCT_NEW,
        [OP_STRUCT_GET] = &&L_STRUCT_GET,
        [OP_STRUCT_SET] = &&L_STRUCT_SET,
        [OP_TO_INT] = &&L_TO_INT,
        [OP_TO_FLOAT] = &&L_TO_FLOAT,
        [OP_TO_BOOL] = &&L_TO_BOOL,
        [OP_ABS] = &&L_ABS,
        [OP_POW] = &&L_POW,
        [OP_MIN] = &&L_MIN,
        [OP_MAX] = &&L_MAX,
        [OP_NEG] = &&L_NEG,
        [OP_ASSERT] = &&L_ASSERT,
        [OP_STR_UPPER] = &&L_STR_UPPER,
        [OP_STR_LOWER] = &&L_STR_LOWER,
        [OP_STR_STRIP] = &&L_STR_STRIP,
        [OP_STR_SPLIT] = &&L_STR_SPLIT,
        [OP_STR_JOIN] = &&L_STR_JOIN,
        [OP_STR_REPLACE] = &&L_STR_REPLACE,
        [OP_ENCODE] = &&L_ENCODE,
        [OP_DECODE] = &&L_DECODE,
        [OP_FILE_OPEN] = &&L_FILE_OPEN,
        [OP_FILE_READ] = &&L_FILE_READ,
        [OP_FILE_WRITE] = &&L_FILE_WRITE,
        [OP_FILE_CLOSE] = &&L_FILE_CLOSE,
        [OP_FILE_EXISTS] = &&L_FILE_EXISTS,
        [OP_FILE_ISFILE] = &&L_FILE_ISFILE,
        [OP_FILE_ISDIR] = &&L_FILE_ISDIR,
        [OP_FILE_LISTDIR] = &&L_FILE_LISTDIR,
        [OP_FILE_MKDIR] = &&L_FILE_MKDIR,
        [OP_FILE_MAKEDIRS] = &&L_FILE_MAKEDIRS,
        [OP_FILE_REMOVE] = &&L_FILE_REMOVE,
        [OP_FILE_RMDIR] = &&L_FILE_RMDIR,
        [OP_FILE_RENAME] = &&L_FILE_RENAME,
        [OP_FILE_GETSIZE] = &&L_FILE_GETSIZE,
        [OP_FILE_GETCWD] = &&L_FILE_GETCWD,
        [OP_FILE_CHDIR] = &&L_FILE_CHDIR,
        [OP_FILE_ABSPATH] = &&L_FILE_ABSPATH,
        [OP_FILE_BASENAME] = &&L_FILE_BASENAME,
        [OP_FILE_DIRNAME] = &&L_FILE_DIRNAME,
        [OP_FILE_JOIN] = &&L_FILE_JOIN,
        [OP_SOCKET_CREATE] = &&L_SOCKET_CREATE,
        [OP_SOCKET_CONNECT] = &&L_SOCKET_CONNECT,
        [OP_SOCKET_BIND] = &&L_SOCKET_BIND,
        [OP_SOCKET_LISTEN] = &&L_SOCKET_LISTEN,
        [OP_SOCKET_ACCEPT] = &&L_SOCKET_ACCEPT,
        [OP_SOCKET_SEND] = &&L_SOCKET_SEND,
        [OP_SOCKET_RECV] = &&L_SOCKET_RECV,
        [OP_SOCKET_CLOSE] = &&L_SOCKET_CLOSE,
        [OP_SOCKET_SETSOCKOPT] = &&L_SOCKET_SETSOCKOPT,
        [OP_FORK] = &&L_FORK,
        [OP_WAIT] = &&L_WAIT,
        [OP_PY_IMPORT] = &&L_PY_IMPORT,
        [OP_PY_CALL] = &&L_PY_CALL,
        [OP_PY_GETATTR] = &&L_PY_GETATTR,
        [OP_PY_SETATTR] = &&L_PY_SETATTR,
        [OP_PY_CALL_METHOD] = &&L_PY_CALL_METHOD,
        [OP_TRY_BEGIN] = &&L_TRY_BEGIN,
        [OP_TRY_END] = &&L_TRY_END,
        [OP_RAISE] = &&L_RAISE,
        [OP_CMP_LT_CONST] = &&L_CMP_LT_CONST,
        [OP_CMP_GT_CONST] = &&L_CMP_GT_CONST,
        [OP_CMP_LE_CONST] = &&L_CMP_LE_CONST,
        [OP_CMP_GE_CONST] = &&L_CMP_GE_CONST,
        [OP_CMP_EQ_CONST] = &&L_CMP_EQ_CONST,
        [OP_CMP_NE_CONST] = &&L_CMP_NE_CONST,
        [OP_CMP_LT_CONST_F64] = &&L_CMP_LT_CONST_F64,
        [OP_CMP_GT_CONST_F64] = &&L_CMP_GT_CONST_F64,
        [OP_CMP_LE_CONST_F64] = &&L_CMP_LE_CONST_F64,
        [OP_CMP_GE_CONST_F64] = &&L_CMP_GE_CONST_F64,
        [OP_CMP_EQ_CONST_F64] = &&L_CMP_EQ_CONST_F64,
        [OP_CMP_NE_CONST_F64] = &&L_CMP_NE_CONST_F64,
        [OP_SWITCH_JUMP_TABLE] = &&L_SWITCH_JUMP_TABLE,
    };
#define DISPATCH() goto *dispatch_table[vm->code[vm->pc++].op]
    DISPATCH();
L_CONST_I64: // OP_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    vm_push(vm, value_make_int_si(inst.operand.int64));
    DISPATCH();
}
L_CONST_F64: // OP_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    vm_push(vm, value_make_f64(inst.operand.f64));
    DISPATCH();
}
L_CONST_STR: // OP_CONST_STR
{
    Instruction inst = vm->code[vm->pc - 1];
    vm_push(vm, value_make_str(inst.operand.str_val));
    DISPATCH();
}
L_CONST_BOOL: // OP_CONST_BOOL
{
    Instruction inst = vm->code[vm->pc - 1];
    vm_push(vm, value_make_bool(inst.operand.index != 0));
    DISPATCH();
}
L_CONST_I64_MULTI: // OP_CONST_I64_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt64 *multi = (MultiInt64*)inst.operand.ptr;
    for (int i = 0; i < multi->count; i++)
    {
        vm_push(vm, value_make_int_si(multi->values[i]));
    }
    DISPATCH();
}
L_CONST_F64_MULTI: // OP_CONST_F64_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiF64 *multi = (MultiF64*)inst.operand.ptr;
    for (int i = 0; i < multi->count; i++)
    {
        vm_push(vm, value_make_f64(multi->values[i]));
    }
    DISPATCH();
}
L_CONST_STR_MULTI: // OP_CONST_STR_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiStr *multi = (MultiStr*)inst.operand.ptr;
    for (int i = 0; i < multi->count; i++)
    {
        vm_push(vm, value_make_str(multi->values[i]));
    }
    DISPATCH();
}
L_CONST_BOOL_MULTI: // OP_CONST_BOOL_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt *multi = (MultiInt*)inst.operand.ptr;
    for (int i = 0; i < multi->count; i++)
    {
        vm_push(vm, value_make_bool(multi->values[i] != 0));
    }
    DISPATCH();
}
L_LOAD: // OP_LOAD
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value v = current_frame->vars.vars[inst.operand.index];
        // Fast path for immutable types - avoid expensive copy
        if (likely(v.type == VAL_INT || v.type == VAL_F64 ||
                   v.type == VAL_BOOL || v.type == VAL_VOID)) {
            vm_push(vm, v);
        } else {
            // Slow path for mutable types - deep copy required
            vm_push(vm, value_copy(v));
        }
        DISPATCH();
    }
}
L_STORE: // OP_STORE
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value v = vm_pop(vm);
        value_free(current_frame->vars.vars[inst.operand.index]);
        current_frame->vars.vars[inst.operand.index] = v;
        DISPATCH();
    }
}
L_STORE_REF: // OP_STORE_REF
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        // Store a reference/pointer to stack top value instead of copying
        Value v = vm_pop(vm);
        value_free(current_frame->vars.vars[inst.operand.index]);
        // Create an alias that points to the source variable
        // We need to find which variable holds this value
        // For now, just do a shallow copy - optimizer will use this for safe cases
        current_frame->vars.vars[inst.operand.index] = v;
        DISPATCH();
    }
}
L_STORE_CONST_I64: // OP_STORE_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    StoreConstI64 *multi = (StoreConstI64 *)inst.operand.ptr;

    // Process each slot/value pair
    for (int i = 0; i < multi->count; i++)
    {
        int slot = (int)multi->pairs[i * 2];
        int64_t val = multi->pairs[i * 2 + 1];

        value_free(current_frame->vars.vars[slot]);
        current_frame->vars.vars[slot] = value_make_int_si(val);
    }
    DISPATCH();
}
L_STORE_CONST_F64: // OP_STORE_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    StoreConstF64 *multi = (StoreConstF64 *)inst.operand.ptr;

    // Process each slot/value pair
    for (int i = 0; i < multi->count; i++)
    {
        int slot = (int)multi->pairs[i * 2];
        double val;
        memcpy(&val, &multi->pairs[i * 2 + 1], sizeof(double));

        value_free(current_frame->vars.vars[slot]);
        current_frame->vars.vars[slot] = value_make_f64(val);
    }
    DISPATCH();
}
L_STORE_CONST_BOOL: // OP_STORE_CONST_BOOL
{
    Instruction inst = vm->code[vm->pc - 1];
    StoreConstBool *multi = (StoreConstBool *)inst.operand.ptr;

    // Process each slot/value pair
    for (int i = 0; i < multi->count; i++)
    {
        int slot = (int)multi->pairs[i * 2];
        int val = (int)multi->pairs[i * 2 + 1];

        value_free(current_frame->vars.vars[slot]);
        current_frame->vars.vars[slot] = value_make_bool(val);
    }
    DISPATCH();
}
L_STORE_CONST_STR: // OP_STORE_CONST_STR
{
    Instruction inst = vm->code[vm->pc - 1];
    StoreConstStr *multi = (StoreConstStr *)inst.operand.ptr;

    // Process each slot/value pair
    for (int i = 0; i < multi->count; i++)
    {
        int slot = (int)(intptr_t)multi->pairs[i * 2];
        char *str = multi->pairs[i * 2 + 1];

        value_free(current_frame->vars.vars[slot]);
        current_frame->vars.vars[slot] = value_make_str(str);
    }
    DISPATCH();
}
L_ADD_I64: // OP_ADD_I64
{
L_ADD_F64: // OP_ADD_F64
L_ADD_STR: // OP_ADD_STR
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);
    Value result = value_add(a, b);
    value_free(a);
    value_free(b);
    vm_push(vm, result);
    DISPATCH();
}
}
L_SUB_I64: // OP_SUB_I64
{
L_SUB_F64: // OP_SUB_F64
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);
    // Always use value_sub to handle overflow detection
    Value result = value_sub(a, b);
    value_free(a);
    value_free(b);
    vm_push(vm, result);
    DISPATCH();
}
}
L_MUL_I64: // OP_MUL_I64
{
L_MUL_F64: // OP_MUL_F64
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);
    // Always use value_mul to handle overflow detection
    Value result = value_mul(a, b);
    value_free(a);
    value_free(b);
    vm_push(vm, result);
    DISPATCH();
}
}
L_DIV_I64: // OP_DIV_I64
{
L_DIV_F64: // OP_DIV_F64
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);
    // For DIV_F64, always return float result
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        if (b.as.int64 == 0) {
            vm_runtime_error(vm, "float division by zero", 0);
            exit(1);
        }
        double result = (double)a.as.int64 / (double)b.as.int64;
        vm_push(vm, value_make_f64(result));
    }
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        if (b.as.f64 == 0.0) {
            vm_runtime_error(vm, "float division by zero", 0);
            exit(1);
        }
        vm_push(vm, value_make_f64(a.as.f64 / b.as.f64));
    }
    else if (a.type == VAL_INT && b.type == VAL_F64)
    {
        if (b.as.f64 == 0.0) {
            vm_runtime_error(vm, "float division by zero", 0);
            exit(1);
        }
        vm_push(vm, value_make_f64((double)a.as.int64 / b.as.f64));
    }
    else if (a.type == VAL_F64 && b.type == VAL_INT)
    {
        if (b.as.int64 == 0) {
            vm_runtime_error(vm, "float division by zero", 0);
            exit(1);
        }
        vm_push(vm, value_make_f64(a.as.f64 / (double)b.as.int64));
    }
    else
    {
        Value result = value_div(a, b);
        value_free(a);
        value_free(b);
        vm_push(vm, result);
    }
    DISPATCH();
}
}
L_MOD_I64: // OP_MOD_I64
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        // Fast path for int64 % int64
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            int64_t result = a.as.int64 % b.as.int64;
            vm_push(vm, value_make_int_si(result));
        }
        else
        {
            Value result = value_mod(a, b);
            value_free(a);
            value_free(b);
            vm_push(vm, result);
        }
        DISPATCH();
    }
}
L_CMP_EQ: // OP_CMP_EQ
L_CMP_NE: // OP_CMP_NE
L_CMP_LT: // OP_CMP_LT
L_CMP_GT: // OP_CMP_GT
L_CMP_LE: // OP_CMP_LE
L_CMP_GE: // OP_CMP_GE
{
    Instruction inst = vm->code[vm->pc - 1];
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);
    // Fast path for int64 comparisons (very common in loops)
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        bool result;
        switch (inst.op)
        {
        case OP_CMP_EQ:
            result = (a.as.int64 == b.as.int64);
            break;
        case OP_CMP_NE:
            result = (a.as.int64 != b.as.int64);
            break;
        case OP_CMP_LT:
            result = (a.as.int64 < b.as.int64);
            break;
        case OP_CMP_GT:
            result = (a.as.int64 > b.as.int64);
            break;
        case OP_CMP_LE:
            result = (a.as.int64 <= b.as.int64);
            break;
        case OP_CMP_GE:
            result = (a.as.int64 >= b.as.int64);
            break;
        default:
            result = false;
        }
        vm_push(vm, value_make_bool(result));
    }
    else
    {
        Value result = value_compare(a, b, inst.op);
        value_free(a);
        value_free(b);
        vm_push(vm, result);
    }
    DISPATCH();
}
L_AND: // OP_AND
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        vm_push(vm, value_make_bool(a.as.boolean && b.as.boolean));
        DISPATCH();
    }
}
L_OR: // OP_OR
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        vm_push(vm, value_make_bool(a.as.boolean || b.as.boolean));
        DISPATCH();
    }
}
L_NOT: // OP_NOT
{
    {
        Value a = vm_pop(vm);
        bool result;
        if (a.type == VAL_BOOL)
            result = !a.as.boolean;
        else if (a.type == VAL_INT)
            result = !(a.as.int64);
        else if (a.type == VAL_F64)
            result = !(a.as.f64);
        else
            result = false; // Default for other types

        value_free(a);
        vm_push(vm, value_make_bool(result));
        DISPATCH();
    }
}
L_JUMP: // OP_JUMP
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        // Label already resolved to PC index during loading
        vm->pc = inst.operand.int64;
        DISPATCH();
    }
}
L_JUMP_IF_FALSE: // OP_JUMP_IF_FALSE
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value cond = vm_pop(vm);
        if (!cond.as.boolean)
        {
            // Label already resolved to PC index during loading
            vm->pc = inst.operand.int64;
        }
        value_free(cond);
        DISPATCH();
    }
}
L_JUMP_IF_TRUE: // OP_JUMP_IF_TRUE
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value cond = vm_pop(vm);
        if (cond.as.boolean)
        {
            // Label already resolved to PC index during loading
            vm->pc = inst.operand.int64;
        }
        value_free(cond);
        DISPATCH();
    }
}
L_GOTO_CALL: // OP_GOTO_CALL
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        // Save current position as return address but don't create a full call frame
        // We'll use a lightweight call frame that just tracks the return PC
        CallFrame *frame = &vm->call_stack[vm->call_stack_top++];
        frame->func = current_frame->func; // Same function context
        frame->return_pc = vm->pc; // Save return address
        frame->vars.var_count = 0; // No local variables for goto
        
        // Jump to the label
        vm->pc = inst.operand.int64;
        
        // Don't update current_frame since we're staying in the same function context
        DISPATCH();
    }
}
L_CALL: // OP_CALL
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Function *func = vm_find_function(vm, inst.operand.str_val);
        if (unlikely(!func))
        {
            fprintf(stderr, "Function not found: %s\n", inst.operand.str_val);
            exit(1);
        }

        // Set up new call frame
        CallFrame *frame = &vm->call_stack[vm->call_stack_top++];
        frame->func = func;
        frame->return_pc = vm->pc;
        frame->vars.var_count = func->local_count;
        // Initialize locals to 0
        for (int i = 0; i < func->local_count; i++)
        {
            frame->vars.vars[i] = value_make_int_si(0);
        }
        // Pop arguments and store in reverse order
        for (int i = func->arg_count - 1; i >= 0; i--)
        {
            Value arg = vm_pop(vm);
            value_free(frame->vars.vars[i]);
            frame->vars.vars[i] = arg;
        }
        vm->pc = func->start_pc;
        // Update cached frame pointer
        current_frame = frame;
        DISPATCH();
    }
}
L_RETURN: // OP_RETURN
{
    Value ret_val = vm_pop(vm);

    if (vm->call_stack_top <= 1)
    {
        vm->running = false;
        return; // Returning from main - exit interpreter
    }
    else
    {
        // Clean up current frame
        CallFrame *frame = &vm->call_stack[--vm->call_stack_top];
        for (int i = 0; i < frame->vars.var_count; i++)
        {
            value_free(frame->vars.vars[i]);
        }
        vm->pc = frame->return_pc;
        // Update cached frame pointer to caller's frame
        current_frame = &vm->call_stack[vm->call_stack_top - 1];
        vm_push(vm, ret_val);
    }

    DISPATCH();
}

L_RETURN_VOID: // OP_RETURN_VOID
{

    if (vm->call_stack_top <= 1)
    {
        vm->running = false;
        return; // Returning from main - exit interpreter
    }
    else
    {
        // Clean up current frame
        CallFrame *frame = &vm->call_stack[--vm->call_stack_top];
        for (int i = 0; i < frame->vars.var_count; i++)
        {
            value_free(frame->vars.vars[i]);
        }
        vm->pc = frame->return_pc;
        // Update cached frame pointer to caller's frame
        current_frame = &vm->call_stack[vm->call_stack_top - 1];
    }

    DISPATCH();
}
L_BUILTIN_PRINT: // OP_BUILTIN_PRINT
{
    {
        Value v = vm_pop(vm);
        value_print(v);
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_PRINTLN: // OP_BUILTIN_PRINTLN
{
    {
        Value v = vm_pop(vm);
        value_print(v);
        printf("\n");
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_STR: // OP_BUILTIN_STR
{
    {
        Value v = vm_pop(vm);
        Value str = value_to_string(v);
        value_free(v);
        vm_push(vm, str);
        DISPATCH();
    }
}
L_BUILTIN_LEN: // OP_BUILTIN_LEN
{
    {
        Value v = vm_pop(vm);
        if (v.type == VAL_STR)
        {
            vm_push(vm, value_make_int_si(strlen(v.as.str)));
        }
        else if (v.type == VAL_LIST)
        {
            vm_push(vm, value_make_int_si(v.as.list->length));
        }
        else if (v.type == VAL_SET)
        {
            vm_push(vm, value_make_int_si(v.as.set->length));
        }
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_SQRT: // OP_BUILTIN_SQRT
{
    {
        Value v = vm_pop(vm);
        if (v.type == VAL_F64)
        {
            vm_push(vm, value_make_f64(sqrt(v.as.f64)));
        }
        else if (v.type == VAL_INT)
        {
            double d = mpz_get_d(*v.as.bigint);
            vm_push(vm, value_make_f64(sqrt(d)));
        }
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_ROUND: // OP_BUILTIN_ROUND
{
    {
        Value v = vm_pop(vm);
        if (v.type == VAL_F64)
        {
            vm_push(vm, value_make_int_si((long)round(v.as.f64)));
        }
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_FLOOR: // OP_BUILTIN_FLOOR
{
    {
        Value v = vm_pop(vm);
        if (v.type == VAL_F64)
        {
            vm_push(vm, value_make_int_si((long)floor(v.as.f64)));
        }
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_CEIL: // OP_BUILTIN_CEIL
{
    {
        Value v = vm_pop(vm);
        if (v.type == VAL_F64)
        {
            vm_push(vm, value_make_int_si((long)ceil(v.as.f64)));
        }
        value_free(v);
        DISPATCH();
    }
}
L_BUILTIN_PI: // OP_BUILTIN_PI
{
    {
        // Push the value of PI (3.141592653589793)
        vm_push(vm, value_make_f64(M_PI));
        DISPATCH();
    }
}
L_POP: // OP_POP
{
    {
        Value v = vm_pop(vm);
        value_free(v);
        DISPATCH();
    }
}
L_DUP: // OP_DUP
{
    {
        Value v = vm->stack[vm->stack_top - 1];
        vm_push(vm, value_copy(v));
        DISPATCH();
    }
}
L_LABEL: // OP_LABEL
{
    // No-op at runtime
    DISPATCH();
}
L_INC_LOCAL: // OP_INC_LOCAL
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        int var_id = inst.operand.index;
        Value *var = &current_frame->vars.vars[var_id];
        if (likely(var->type == VAL_INT))
        {
            var->as.int64++;
        }
        else if (var->type == VAL_BIGINT)
        {
            mpz_add_ui(*var->as.bigint, *var->as.bigint, 1);
        }
        DISPATCH();
    }
}
L_DEC_LOCAL: // OP_DEC_LOCAL
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        int var_id = inst.operand.index;
        Value *var = &current_frame->vars.vars[var_id];
        if (likely(var->type == VAL_INT))
        {
            var->as.int64--;
        }
        else if (var->type == VAL_BIGINT)
        {
            mpz_sub_ui(*var->as.bigint, *var->as.bigint, 1);
        }
        DISPATCH();
    }
}
L_ADD_CONST_I64: // OP_ADD_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            // Check for overflow
            if (would_add_overflow(a.as.int64, const_val))
            {
                // Promote to bigint
                Value a_big = promote_to_bigint(a.as.int64);
                if (const_val >= 0)
                {
                    mpz_add_ui(*a_big.as.bigint, *a_big.as.bigint, (unsigned long)const_val);
                }
                else
                {
                    mpz_sub_ui(*a_big.as.bigint, *a_big.as.bigint, (unsigned long)(-const_val));
                }
                vm_push(vm, a_big);
            }
            else
            {
                a.as.int64 += const_val;
                vm_push(vm, a);
            }
        }
        else if (a.type == VAL_BIGINT)
        {
            if (const_val >= 0)
            {
                mpz_add_ui(*a.as.bigint, *a.as.bigint, (unsigned long)const_val);
            }
            else
            {
                mpz_sub_ui(*a.as.bigint, *a.as.bigint, (unsigned long)(-const_val));
            }
            vm_push(vm, a);
        }
        else if (a.type == VAL_F64)
        {
            a.as.f64 += (double)const_val;
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_SUB_CONST_I64: // OP_SUB_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 -= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            if (const_val >= 0)
            {
                mpz_sub_ui(*a.as.bigint, *a.as.bigint, (unsigned long)const_val);
            }
            else
            {
                mpz_add_ui(*a.as.bigint, *a.as.bigint, (unsigned long)(-const_val));
            }
            vm_push(vm, a);
        }
        else if (a.type == VAL_F64)
        {
            a.as.f64 -= (double)const_val;
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_MUL_CONST_I64: // OP_MUL_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 *= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_mul_si(*a.as.bigint, *a.as.bigint, const_val);
            vm_push(vm, a);
        }
        else if (a.type == VAL_F64)
        {
            a.as.f64 *= (double)const_val;
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_MOD_CONST_I64: // OP_MOD_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (const_val == 0)
        {
            if (g_current_vm) {
                vm_runtime_error(g_current_vm, "Modulo by zero", 0);
            } else {
                fprintf(stderr, "Modulo by zero\n");
            }
            exit(1);
        }
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 %= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_mod_ui(*a.as.bigint, *a.as.bigint, const_val < 0 ? -const_val : const_val);
            if (const_val < 0) {
                mpz_neg(*a.as.bigint, *a.as.bigint);
            }
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_DIV_CONST_I64: // OP_DIV_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (const_val == 0)
        {
            if (g_current_vm) {
                vm_runtime_error(g_current_vm, "Division by zero", 0);
            } else {
                fprintf(stderr, "Division by zero\n");
            }
            exit(1);
        }
        if (likely(a.type == VAL_INT))
        {
            // Python-style true division: always return float
            double result = (double)a.as.int64 / (double)const_val;
            value_free(a);
            vm_push(vm, value_make_f64(result));
        }
        else if (a.type == VAL_BIGINT)
        {
            // Convert bigint to double for true division
            double a_val = mpz_get_d(*a.as.bigint);
            double result = a_val / (double)const_val;
            value_free(a);
            vm_push(vm, value_make_f64(result));
        }
        else if (a.type == VAL_F64)
        {
            a.as.f64 /= (double)const_val;
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_ADD_CONST_I64_MULTI: // OP_ADD_CONST_I64_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        MultiInt64 *multi = (MultiInt64*)inst.operand.ptr;

        for (int i = 0; i < multi->count; i++)
        {
            int64_t const_val = multi->values[i];
            if (likely(a.type == VAL_INT))
            {
                a.as.int64 += const_val;
            }
            else if (a.type == VAL_BIGINT)
            {
                if (const_val >= 0)
                {
                    mpz_add_ui(*a.as.bigint, *a.as.bigint, (unsigned long)const_val);
                }
                else
                {
                    mpz_sub_ui(*a.as.bigint, *a.as.bigint, (unsigned long)(-const_val));
                }
            }
            else if (a.type == VAL_F64)
            {
                a.as.f64 += (double)const_val;
            }
            else
            {
                value_free(a);
                DISPATCH();
            }
        }
        vm_push(vm, a);
        DISPATCH();
    }
}
L_ADD_CONST_F64: // OP_ADD_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;

    if (likely(a.type == VAL_F64))
    {
        a.as.f64 += const_val;
    }
    else if (a.type == VAL_INT)
    {
        a.as.f64 = (double)a.as.int64 + const_val;
        a.type = VAL_F64;
    }
    else if (a.type == VAL_BIGINT)
    {
        a.as.f64 = mpz_get_d(*a.as.bigint) + const_val;
        mpz_clear(*a.as.bigint);
        free(a.as.bigint);
        a.type = VAL_F64;
    }
    vm_push(vm, a);
    DISPATCH();
}
L_SUB_CONST_F64: // OP_SUB_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;

    if (likely(a.type == VAL_F64))
    {
        a.as.f64 -= const_val;
    }
    else if (a.type == VAL_INT)
    {
        a.as.f64 = (double)a.as.int64 - const_val;
        a.type = VAL_F64;
    }
    else if (a.type == VAL_BIGINT)
    {
        a.as.f64 = mpz_get_d(*a.as.bigint) - const_val;
        mpz_clear(*a.as.bigint);
        free(a.as.bigint);
        a.type = VAL_F64;
    }
    vm_push(vm, a);
    DISPATCH();
}
L_MUL_CONST_F64: // OP_MUL_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;

    if (likely(a.type == VAL_F64))
    {
        a.as.f64 *= const_val;
    }
    else if (a.type == VAL_INT)
    {
        a.as.f64 = (double)a.as.int64 * const_val;
        a.type = VAL_F64;
    }
    else if (a.type == VAL_BIGINT)
    {
        a.as.f64 = mpz_get_d(*a.as.bigint) * const_val;
        mpz_clear(*a.as.bigint);
        free(a.as.bigint);
        a.type = VAL_F64;
    }
    vm_push(vm, a);
    DISPATCH();
}
L_DIV_CONST_F64: // OP_DIV_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;

    if (const_val == 0.0)
    {
        if (g_current_vm) {
            vm_runtime_error(g_current_vm, "float division by zero", 0);
        } else {
            fprintf(stderr, "float division by zero\n");
        }
        exit(1);
    }

    if (likely(a.type == VAL_F64))
    {
        a.as.f64 /= const_val;
    }
    else if (a.type == VAL_INT)
    {
        a.as.f64 = (double)a.as.int64 / const_val;
        a.type = VAL_F64;
    }
    else if (a.type == VAL_BIGINT)
    {
        a.as.f64 = mpz_get_d(*a.as.bigint) / const_val;
        mpz_clear(*a.as.bigint);
        free(a.as.bigint);
        a.type = VAL_F64;
    }
    vm_push(vm, a);
    DISPATCH();
}
L_AND_CONST: // OP_AND_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int const_val = inst.operand.int64;
        bool a_truthy = false;

        if (a.type == VAL_BOOL)
            a_truthy = a.as.boolean;
        else if (a.type == VAL_INT)
            a_truthy = (a.as.int64 != 0);
        else if (a.type == VAL_F64)
            a_truthy = (a.as.f64 != 0.0);
        else if (a.type == VAL_STR)
            a_truthy = (strlen(a.as.str) > 0);
        else if (a.type == VAL_LIST)
            a_truthy = (a.as.list->length > 0);

        bool result = a_truthy && (const_val != 0);
        value_free(a);
        vm_push(vm, value_make_bool(result));
        DISPATCH();
    }
}
L_OR_CONST: // OP_OR_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int const_val = inst.operand.int64;
        bool a_truthy = false;

        if (a.type == VAL_BOOL)
            a_truthy = a.as.boolean;
        else if (a.type == VAL_INT)
            a_truthy = (a.as.int64 != 0);
        else if (a.type == VAL_F64)
            a_truthy = (a.as.f64 != 0.0);
        else if (a.type == VAL_STR)
            a_truthy = (strlen(a.as.str) > 0);
        else if (a.type == VAL_LIST)
            a_truthy = (a.as.list->length > 0);

        bool result = a_truthy || (const_val != 0);
        value_free(a);
        vm_push(vm, value_make_bool(result));
        DISPATCH();
    }
}
L_AND_CONST_I64: // OP_AND_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 &= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_t temp;
            mpz_init_set_si(temp, const_val);
            mpz_and(*a.as.bigint, *a.as.bigint, temp);
            mpz_clear(temp);
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_OR_CONST_I64: // OP_OR_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 |= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_t temp;
            mpz_init_set_si(temp, const_val);
            mpz_ior(*a.as.bigint, *a.as.bigint, temp);
            mpz_clear(temp);
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
L_XOR_CONST_I64: // OP_XOR_CONST_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        Value a = vm_pop(vm);
        int64_t const_val = inst.operand.int64;
        if (likely(a.type == VAL_INT))
        {
            a.as.int64 ^= const_val;
            vm_push(vm, a);
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_t temp;
            mpz_init_set_si(temp, const_val);
            mpz_xor(*a.as.bigint, *a.as.bigint, temp);
            mpz_clear(temp);
            vm_push(vm, a);
        }
        else
        {
            value_free(a);
        }
        DISPATCH();
    }
}
// Comparison with constant instructions
L_CMP_LT_CONST: // OP_CMP_LT_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 < const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) < 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 < (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_GT_CONST: // OP_CMP_GT_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 > const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) > 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 > (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_LE_CONST: // OP_CMP_LE_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 <= const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) <= 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 <= (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_GE_CONST: // OP_CMP_GE_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 >= const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) >= 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 >= (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_EQ_CONST: // OP_CMP_EQ_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 == const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) == 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 == (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_NE_CONST: // OP_CMP_NE_CONST
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    int64_t const_val = inst.operand.int64;
    bool result = false;

    if (likely(a.type == VAL_INT))
    {
        result = (a.as.int64 != const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_cmp_si(*a.as.bigint, const_val) != 0);
    }
    else if (a.type == VAL_F64)
    {
        result = (a.as.f64 != (double)const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_LT_CONST_F64: // OP_CMP_LT_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 < const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 < const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) < const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_GT_CONST_F64: // OP_CMP_GT_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 > const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 > const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) > const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_LE_CONST_F64: // OP_CMP_LE_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 <= const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 <= const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) <= const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_GE_CONST_F64: // OP_CMP_GE_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 >= const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 >= const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) >= const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_EQ_CONST_F64: // OP_CMP_EQ_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 == const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 == const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) == const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_CMP_NE_CONST_F64: // OP_CMP_NE_CONST_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    Value a = vm_pop(vm);
    double const_val = inst.operand.f64;
    bool result = false;

    if (a.type == VAL_F64)
    {
        result = (a.as.f64 != const_val);
    }
    else if (a.type == VAL_INT)
    {
        result = ((double)a.as.int64 != const_val);
    }
    else if (a.type == VAL_BIGINT)
    {
        result = (mpz_get_d(*a.as.bigint) != const_val);
    }
    value_free(a);
    vm_push(vm, value_make_bool(result));
    DISPATCH();
}
L_SWITCH_JUMP_TABLE: // OP_SWITCH_JUMP_TABLE
{
    Instruction inst = vm->code[vm->pc - 1];
    Value switch_val = vm_pop(vm);

    // Convert switch value to int64
    int64_t val;
    if (switch_val.type == VAL_INT)
    {
        val = switch_val.as.int64;
    }
    else if (switch_val.type == VAL_BIGINT)
    {
        if (!mpz_fits_slong_p(*switch_val.as.bigint))
        {
            // Value too large, jump to default
            value_free(switch_val);
            vm->pc = inst.operand.switch_table.pcs[inst.operand.switch_table.num_labels - 1];
            DISPATCH();
        }
        val = mpz_get_si(*switch_val.as.bigint);
    }
    else
    {
        fprintf(stderr, "SWITCH_JUMP_TABLE: invalid switch value type\n");
        value_free(switch_val);
        exit(1);
    }

    value_free(switch_val);

    // Check if value is in range [min_val, max_val]
    if (val >= inst.operand.switch_table.min_val && val <= inst.operand.switch_table.max_val)
    {
        // Jump to corresponding case
        int case_index = val - inst.operand.switch_table.min_val;
        vm->pc = inst.operand.switch_table.pcs[case_index];
    }
    else
    {
        // Jump to default (last PC in array)
        vm->pc = inst.operand.switch_table.pcs[inst.operand.switch_table.num_labels - 1];
    }
    DISPATCH();
}
// Stack manipulation instructions
L_SWAP: // OP_SWAP
{
    {
        // Swap top two stack values (a b -> b a)
        if (vm->stack_top < 2)
        {
            fprintf(stderr, "Stack underflow in SWAP\n");
            exit(1);
        }
        Value b = vm->stack[vm->stack_top - 1];
        Value a = vm->stack[vm->stack_top - 2];
        vm->stack[vm->stack_top - 1] = a;
        vm->stack[vm->stack_top - 2] = b;
        DISPATCH();
    }
}
L_ROT: // OP_ROT
{
    {
        // Rotate top three (a b c -> b c a)
        if (vm->stack_top < 3)
        {
            fprintf(stderr, "Stack underflow in ROT\n");
            exit(1);
        }
        Value c = vm->stack[vm->stack_top - 1];
        Value b = vm->stack[vm->stack_top - 2];
        Value a = vm->stack[vm->stack_top - 3];
        vm->stack[vm->stack_top - 1] = a;
        vm->stack[vm->stack_top - 2] = c;
        vm->stack[vm->stack_top - 3] = b;
        DISPATCH();
    }
}
L_OVER: // OP_OVER
{
    {
        // Copy second item to top (a b -> a b a)
        if (vm->stack_top < 2)
        {
            fprintf(stderr, "Stack underflow in OVER\n");
            exit(1);
        }
        Value a = vm->stack[vm->stack_top - 2];
        vm_push(vm, value_copy(a));
        DISPATCH();
    }
}
L_DUP2: // OP_DUP2
{
    {
        // Duplicate top two (a b -> a b a b)
        if (vm->stack_top < 2)
        {
            fprintf(stderr, "Stack underflow in DUP2\n");
            exit(1);
        }
        Value b = vm->stack[vm->stack_top - 1];
        Value a = vm->stack[vm->stack_top - 2];
        vm_push(vm, value_copy(a));
        vm_push(vm, value_copy(b));
        DISPATCH();
    }
}
// Bitwise operations
L_AND_I64: // OP_AND_I64
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            vm_push(vm, value_make_int_si(a.as.int64 & b.as.int64));
            value_free(a);
            value_free(b);
        }
        else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
        {
            // Create result bigint
            Value result;
            result.type = VAL_BIGINT;
            result.as.bigint = malloc(sizeof(mpz_t));
            mpz_init(*result.as.bigint);
            mpz_and(*result.as.bigint, *a.as.bigint, *b.as.bigint);
            value_free(a);
            value_free(b);
            vm_push(vm, result);
        }
        else
        {
            fprintf(stderr, "Type error in AND_I64\n");
            exit(1);
        }
        DISPATCH();
    }
}
L_OR_I64: // OP_OR_I64
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            vm_push(vm, value_make_int_si(a.as.int64 | b.as.int64));
            value_free(a);
            value_free(b);
        }
        else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
        {
            Value result;
            result.type = VAL_BIGINT;
            result.as.bigint = malloc(sizeof(mpz_t));
            mpz_init(*result.as.bigint);
            mpz_ior(*result.as.bigint, *a.as.bigint, *b.as.bigint);
            value_free(a);
            value_free(b);
            vm_push(vm, result);
        }
        else
        {
            fprintf(stderr, "Type error in OR_I64\n");
            exit(1);
        }
        DISPATCH();
    }
}
L_XOR_I64: // OP_XOR_I64
{
    {
        Value b = vm_pop(vm);
        Value a = vm_pop(vm);
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            vm_push(vm, value_make_int_si(a.as.int64 ^ b.as.int64));
            value_free(a);
            value_free(b);
        }
        else if (a.type == VAL_BIGINT && b.type == VAL_BIGINT)
        {
            Value result;
            result.type = VAL_BIGINT;
            result.as.bigint = malloc(sizeof(mpz_t));
            mpz_init(*result.as.bigint);
            mpz_xor(*result.as.bigint, *a.as.bigint, *b.as.bigint);
            value_free(a);
            value_free(b);
            vm_push(vm, result);
        }
        else
        {
            fprintf(stderr, "Type error in XOR_I64\n");
            exit(1);
        }
        DISPATCH();
    }
}
L_SHL_I64: // OP_SHL_I64
{
    {
        Value b = vm_pop(vm); // shift amount
        Value a = vm_pop(vm); // value
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            if (b.as.int64 < 0 || b.as.int64 >= 64)
            {
                fprintf(stderr, "Invalid shift amount: %ld\n", b.as.int64);
                exit(1);
            }
            vm_push(vm, value_make_int_si(a.as.int64 << b.as.int64));
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_mul_2exp(*a.as.bigint, *a.as.bigint, b.as.int64);
            vm_push(vm, a);
        }
        else
        {
            fprintf(stderr, "Type error in SHL_I64\n");
            exit(1);
        }
        if (a.type != VAL_BIGINT)
            value_free(a);
        value_free(b);
        DISPATCH();
    }
}
L_SHR_I64: // OP_SHR_I64
{
    {
        Value b = vm_pop(vm); // shift amount
        Value a = vm_pop(vm); // value
        if (likely(a.type == VAL_INT && b.type == VAL_INT))
        {
            if (b.as.int64 < 0 || b.as.int64 >= 64)
            {
                fprintf(stderr, "Invalid shift amount: %ld\n", b.as.int64);
                exit(1);
            }
            vm_push(vm, value_make_int_si(a.as.int64 >> b.as.int64));
        }
        else if (a.type == VAL_BIGINT)
        {
            mpz_tdiv_q_2exp(*a.as.bigint, *a.as.bigint, b.as.int64);
            vm_push(vm, a);
        }
        else
        {
            fprintf(stderr, "Type error in SHR_I64\n");
            exit(1);
        }
        if (a.type != VAL_BIGINT)
            value_free(a);
        value_free(b);
        DISPATCH();
    }
}
L_COPY_LOCAL: // OP_COPY_LOCAL
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        // Copy local variable without using stack - deep copy
        int src_idx = inst.operand.indices.src;
        int dst_idx = inst.operand.indices.dst;
        value_free(current_frame->vars.vars[dst_idx]);
        current_frame->vars.vars[dst_idx] = value_copy(current_frame->vars.vars[src_idx]);
        DISPATCH();
    }
}
L_COPY_LOCAL_REF: // OP_COPY_LOCAL_REF
{
    Instruction inst = vm->code[vm->pc - 1];
    {
        // Alias local variable - pointer copy (no deep copy)
        // This is safe when dst won't be modified before src is reassigned
        int src_idx = inst.operand.indices.src;
        int dst_idx = inst.operand.indices.dst;
        value_free(current_frame->vars.vars[dst_idx]);
        value_copy_ref(&current_frame->vars.vars[src_idx], &current_frame->vars.vars[dst_idx]);
        DISPATCH();
    }
}
L_SELECT: // OP_SELECT
{
    {
        // Stack: cond val_true val_false -> result
        // If cond is true, push val_true, else push val_false
        Value val_false = vm_pop(vm);
        Value val_true = vm_pop(vm);
        Value cond = vm_pop(vm);
        bool is_true = false;
        if (cond.type == VAL_BOOL)
        {
            is_true = cond.as.boolean;
        }
        else if (cond.type == VAL_INT)
        {
            is_true = (cond.as.int64 != 0);
        }
        else
        {
            fprintf(stderr, "Type error in SELECT: condition not boolean or int\n");
            exit(1);
        }
        if (is_true)
        {
            vm_push(vm, val_true);
            value_free(val_false);
        }
        else
        {
            vm_push(vm, val_false);
            value_free(val_true);
        }
        value_free(cond);
        DISPATCH();
    }
}

// List operations
L_LIST_NEW: // OP_LIST_NEW
{
    vm_push(vm, value_make_list());
    DISPATCH();
}

L_LIST_APPEND: // OP_LIST_APPEND
{
    Value value = vm_pop(vm);
    Value list_val = vm_pop(vm);

    if (list_val.type != VAL_LIST)
    {
        fprintf(stderr, "Error: Cannot append to non-list type\n");
        exit(1);
    }

    list_append(list_val.as.list, value);
    value_free(value);
    vm_push(vm, list_val);
    DISPATCH();
}

L_LIST_GET: // OP_LIST_GET - Also works for strings
{
    Value index_val = vm_pop(vm);
    Value container_val = vm_pop(vm);

    int index = (int)index_val.as.int64;
    value_free(index_val);

    if (container_val.type == VAL_LIST)
    {
        Value result = list_get(container_val.as.list, index);
        value_free(container_val);
        vm_push(vm, result);
    }
    else if (container_val.type == VAL_STR)
    {
        // Get character at index as a string
        int original_index = index;
        int str_len = (int)strlen(container_val.as.str);

        // Handle negative indices
        if (index < 0)
        {
            index = str_len + index;
        }

        if (index < 0 || index >= str_len)
        {
            char error_msg[256];
            snprintf(error_msg, sizeof(error_msg), "Index error: string index out of range: %d (length: %d)", original_index, str_len);

            // Try to find the exact index value in the source line
            int char_pos = 0;
            if (vm->debug_source_lines) {
                int line_idx = vm->pc >= 2 && vm->debug_line_map_count > 0
                    ? vm->debug_line_map[(vm->pc - 2) % vm->debug_line_map_count]
                    : 0;

                if (line_idx > 0 && line_idx <= vm->debug_source_line_count) {
                    const char *source_line = vm->debug_source_lines[line_idx - 1];

                    // Try to find the exact index pattern [index]
                    char search_pattern[32];
                    snprintf(search_pattern, sizeof(search_pattern), "[%d]", original_index);
                    const char *pattern_pos = strstr(source_line, search_pattern);
                    if (pattern_pos) {
                        // Point to the last digit before ']'
                        char_pos = pattern_pos - source_line + strlen(search_pattern) - 2;
                    } else {
                        // Fallback: find any bracket pair
                        const char *bracket = strchr(source_line, '[');
                        if (bracket) {
                            const char *close_bracket = strchr(bracket, ']');
                            if (close_bracket && close_bracket > bracket) {
                                char_pos = close_bracket - source_line - 1;
                            }
                        }
                    }
                }
            }

            vm_runtime_error(vm, error_msg, char_pos);
            fflush(stderr);
            value_free(container_val);
            exit(1);
        }
        char *char_str = malloc(2);
        char_str[0] = container_val.as.str[index];
        char_str[1] = '\0';
        value_free(container_val);
        vm_push(vm, value_make_str(char_str));
    }
    else
    {
        fprintf(stderr, "Error: Cannot index into non-list/non-string type\n");
        value_free(container_val);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    DISPATCH();
}

L_LIST_SET: // OP_LIST_SET
{
    Value new_value = vm_pop(vm);
    Value index_val = vm_pop(vm);
    Value list_val = vm_pop(vm);

    if (list_val.type != VAL_LIST)
    {
        fprintf(stderr, "Error: Cannot set index on non-list type\n");
        exit(1);
    }

    int index = (int)index_val.as.int64;
    list_set(list_val.as.list, index, new_value);
    value_free(new_value);
    value_free(index_val);
    vm_push(vm, list_val);
    DISPATCH();
}

L_LIST_LEN: // OP_LIST_LEN
{
    Value list_val = vm_pop(vm);

    if (list_val.type != VAL_LIST)
    {
        fprintf(stderr, "Error: Cannot get length of non-list type\n");
        exit(1);
    }

    Value result = value_make_int_si(list_val.as.list->length);
    value_free(list_val);
    vm_push(vm, result);
    DISPATCH();
}

L_LIST_POP: // OP_LIST_POP
{
    Value list_val = vm_pop(vm);

    if (list_val.type != VAL_LIST)
    {
        fprintf(stderr, "Error: Cannot pop from non-list type\n");
        exit(1);
    }

    Value popped_value = list_pop(list_val.as.list);
    // Push modified list first (bottom of stack)
    vm_push(vm, list_val);
    // Then push popped value (top of stack, will be stored first)
    vm_push(vm, popped_value);
    DISPATCH();
}

// Set operations
L_SET_NEW: // OP_SET_NEW
{
    vm_push(vm, value_make_set());
    DISPATCH();
}

L_SET_ADD: // OP_SET_ADD
{
    Value value = vm_pop(vm);
    Value set_val = vm_pop(vm);

    if (set_val.type != VAL_SET)
    {
        fprintf(stderr, "Error: Cannot add to non-set type\n");
        exit(1);
    }

    set_add(set_val.as.set, value);
    value_free(value);
    vm_push(vm, set_val);
    DISPATCH();
}

L_SET_REMOVE: // OP_SET_REMOVE
{
    Value value = vm_pop(vm);
    Value set_val = vm_pop(vm);

    if (set_val.type != VAL_SET)
    {
        fprintf(stderr, "Error: Cannot remove from non-set type\n");
        exit(1);
    }

    set_remove(set_val.as.set, value);
    value_free(value);
    vm_push(vm, set_val);
    DISPATCH();
}

L_SET_CONTAINS: // OP_SET_CONTAINS
{
    Value value = vm_pop(vm);
    Value set_val = vm_pop(vm);

    if (set_val.type != VAL_SET)
    {
        fprintf(stderr, "Error: Cannot check containment in non-set type\n");
        exit(1);
    }

    bool contains = set_contains(set_val.as.set, value);
    value_free(value);
    value_free(set_val);
    vm_push(vm, value_make_bool(contains));
    DISPATCH();
}

L_SET_LEN: // OP_SET_LEN
{
    Value set_val = vm_pop(vm);

    if (set_val.type != VAL_SET)
    {
        fprintf(stderr, "Error: Cannot get length of non-set type\n");
        exit(1);
    }

    Value result = value_make_int_si(set_val.as.set->length);
    value_free(set_val);
    vm_push(vm, result);
    DISPATCH();
}

L_CONTAINS: // OP_CONTAINS - Generic membership check
{
    Value value = vm_pop(vm);
    Value container = vm_pop(vm);
    bool contains = false;

    if (container.type == VAL_LIST)
    {
        // Check if value is in list
        for (int i = 0; i < container.as.list->length; i++)
        {
            if (value_equals(container.as.list->items[i], value))
            {
                contains = true;
                break;
            }
        }
    }
    else if (container.type == VAL_SET)
    {
        contains = set_contains(container.as.set, value);
    }
    else if (container.type == VAL_STR && value.type == VAL_STR)
    {
        // Check if substring exists
        contains = (strstr(container.as.str, value.as.str) != NULL);
    }
    else
    {
        fprintf(stderr, "Error: Cannot check membership in type\n");
        exit(1);
    }

    value_free(value);
    value_free(container);
    vm_push(vm, value_make_bool(contains));
    DISPATCH();
}

L_LIST_NEW_I64: // OP_LIST_NEW_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt64 *multi = (MultiInt64*)inst.operand.ptr;

    List *list = list_new();
    for (int i = 0; i < multi->count; i++)
    {
        Value val = value_make_int_si(multi->values[i]);
        list_append(list, val);
        value_free(val);
    }

    vm_push(vm, value_wrap_list(list));
    DISPATCH();
}

L_LIST_NEW_F64: // OP_LIST_NEW_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiF64 *multi = (MultiF64*)inst.operand.ptr;

    List *list = list_new();
    for (int i = 0; i < multi->count; i++)
    {
        Value val = value_make_f64(multi->values[i]);
        list_append(list, val);
        value_free(val);
    }

    vm_push(vm, value_wrap_list(list));
    DISPATCH();
}

L_LIST_NEW_STR: // OP_LIST_NEW_STR
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiStr *multi = (MultiStr*)inst.operand.ptr;

    List *list = list_new();
    for (int i = 0; i < multi->count; i++)
    {
        Value val = value_make_str(multi->values[i]);
        list_append(list, val);
        value_free(val);
    }

    vm_push(vm, value_wrap_list(list));
    DISPATCH();
}

L_LIST_NEW_BOOL: // OP_LIST_NEW_BOOL
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt *multi = (MultiInt*)inst.operand.ptr;

    List *list = list_new();
    for (int i = 0; i < multi->count; i++)
    {
        Value val = value_make_bool(multi->values[i] != 0);
        list_append(list, val);
        value_free(val);
    }

    vm_push(vm, value_wrap_list(list));
    DISPATCH();
}

L_STRUCT_NEW: // OP_STRUCT_NEW
{
    Instruction inst = vm->code[vm->pc - 1];
    int struct_id = inst.operand.index;

    // Validate struct ID
    if (struct_id < 0 || struct_id >= vm->struct_count)
    {
        fprintf(stderr, "Error: Invalid struct ID: %d\n", struct_id);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    StructDef *def = &vm->structs[struct_id];

    // Create new struct instance
    Struct *s = malloc(sizeof(Struct));
    s->struct_id = struct_id;
    s->field_count = def->field_count;
    s->fields = malloc(sizeof(Value) * s->field_count);

    // Pop field values from stack in reverse order (last field pushed first)
    for (int i = s->field_count - 1; i >= 0; i--)
    {
        s->fields[i] = vm_pop(vm);
    }

    // Push struct value
    Value struct_val;
    struct_val.type = VAL_STRUCT;
    struct_val.as.struct_val = s;
    vm_push(vm, struct_val);
    DISPATCH();
}

L_STRUCT_GET: // OP_STRUCT_GET
{
    Instruction inst = vm->code[vm->pc - 1];
    int field_idx = inst.operand.index;

    Value struct_val = vm_pop(vm);

    if (struct_val.type != VAL_STRUCT)
    {
        fprintf(stderr, "Error: Cannot get field from non-struct type\n");
        value_free(struct_val);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    if (field_idx < 0 || field_idx >= struct_val.as.struct_val->field_count)
    {
        fprintf(stderr, "Error: Field index %d out of range\n", field_idx);
        value_free(struct_val);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    Value field_val = value_copy(struct_val.as.struct_val->fields[field_idx]);
    value_free(struct_val);
    vm_push(vm, field_val);
    DISPATCH();
}

L_STRUCT_SET: // OP_STRUCT_SET
{
    Instruction inst = vm->code[vm->pc - 1];
    int field_idx = inst.operand.index;

    Value new_value = vm_pop(vm);
    Value struct_val = vm_pop(vm);

    if (struct_val.type != VAL_STRUCT)
    {
        fprintf(stderr, "Error: Cannot set field on non-struct type\n");
        value_free(struct_val);
        value_free(new_value);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    if (field_idx < 0 || field_idx >= struct_val.as.struct_val->field_count)
    {
        fprintf(stderr, "Error: Field index %d out of range\n", field_idx);
        value_free(struct_val);
        value_free(new_value);
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    // Free old value and set new one
    value_free(struct_val.as.struct_val->fields[field_idx]);
    struct_val.as.struct_val->fields[field_idx] = new_value;

    vm_push(vm, struct_val);
    DISPATCH();
}

L_TO_INT: // OP_TO_INT - Convert value to integer
{
    Value v = vm_pop(vm);
    Value result;

    if (v.type == VAL_INT)
    {
        result = v; // Already an int
    }
    else if (v.type == VAL_F64)
    {
        result = value_make_int_si((int64_t)v.as.f64);
    }
    else if (v.type == VAL_STR)
    {
        result = value_make_int_si(safe_atoll(v.as.str));
    }
    else if (v.type == VAL_BOOL)
    {
        result = value_make_int_si(v.as.boolean ? 1 : 0);
    }
    else
    {
        result = value_make_int_si(0);
    }

    if (v.type != VAL_INT)
        value_free(v);
    vm_push(vm, result);
    DISPATCH();
}

L_TO_FLOAT: // OP_TO_FLOAT - Convert value to float
{
    Value v = vm_pop(vm);
    Value result;

    if (v.type == VAL_F64)
    {
        result = v; // Already a float
    }
    else if (v.type == VAL_INT)
    {
        result = value_make_f64((double)v.as.int64);
    }
    else if (v.type == VAL_STR)
    {
        result = value_make_f64(safe_atof(v.as.str));
    }
    else if (v.type == VAL_BOOL)
    {
        result = value_make_f64(v.as.boolean ? 1.0 : 0.0);
    }
    else
    {
        result = value_make_f64(0.0);
    }

    if (v.type != VAL_F64)
        value_free(v);
    vm_push(vm, result);
    DISPATCH();
}

L_TO_BOOL: // OP_TO_BOOL - Convert value to boolean
{
    Value v = vm_pop(vm);
    Value result;

    if (v.type == VAL_BOOL)
    {
        result = v; // Already a bool
    }
    else if (v.type == VAL_INT)
    {
        result = value_make_bool(v.as.int64 != 0);
    }
    else if (v.type == VAL_F64)
    {
        result = value_make_bool(v.as.f64 != 0.0);
    }
    else if (v.type == VAL_STR)
    {
        result = value_make_bool(strlen(v.as.str) > 0);
    }
    else if (v.type == VAL_LIST)
    {
        result = value_make_bool(v.as.list->length > 0);
    }
    else
    {
        result = value_make_bool(false);
    }

    if (v.type != VAL_BOOL)
        value_free(v);
    vm_push(vm, result);
    DISPATCH();
}

L_STR_UPPER: // OP_STR_UPPER - Convert string to uppercase
{
    Value v = vm_pop(vm);
    if (v.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: upper() expects a string\n");
        value_free(v);
        exit(1);
    }

    char *result = strdup(v.as.str);
    for (int i = 0; result[i]; i++)
    {
        result[i] = toupper(result[i]);
    }

    value_free(v);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_STR_LOWER: // OP_STR_LOWER - Convert string to lowercase
{
    Value v = vm_pop(vm);
    if (v.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: lower() expects a string\n");
        value_free(v);
        exit(1);
    }

    char *result = strdup(v.as.str);
    for (int i = 0; result[i]; i++)
    {
        result[i] = tolower(result[i]);
    }

    value_free(v);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_STR_STRIP: // OP_STR_STRIP - Strip leading/trailing whitespace
{
    Value v = vm_pop(vm);
    if (v.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: strip() expects a string\n");
        value_free(v);
        exit(1);
    }

    const char *str = v.as.str;
    const char *start = str;
    const char *end = str + strlen(str) - 1;

    // Skip leading whitespace
    while (*start && isspace(*start))
        start++;

    // Skip trailing whitespace
    while (end > start && isspace(*end))
        end--;

    // Create result
    int len = end - start + 1;
    char *result = malloc(len + 1);
    memcpy(result, start, len);
    result[len] = '\0';

    value_free(v);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_STR_SPLIT: // OP_STR_SPLIT - Split string by delimiter (str, sep -> list)
{
    Value sep = vm_pop(vm);
    Value str = vm_pop(vm);

    if (str.type != VAL_STR || sep.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: split() expects two strings\n");
        value_free(str);
        value_free(sep);
        exit(1);
    }

    List *result_list = list_new();

    // Handle empty string - should return list with one empty string
    if (strlen(str.as.str) == 0) {
        list_append(result_list, value_make_str(strdup("")));
        value_free(str);
        value_free(sep);
        vm_push(vm, value_wrap_list(result_list));
        DISPATCH();
    }

    // Handle empty separator - split into individual characters
    if (strlen(sep.as.str) == 0) {
        for (size_t i = 0; str.as.str[i] != '\0'; i++) {
            char char_str[2] = {str.as.str[i], '\0'};
            list_append(result_list, value_make_str(strdup(char_str)));
        }
        value_free(str);
        value_free(sep);
        vm_push(vm, value_wrap_list(result_list));
        DISPATCH();
    }

    // Proper string splitting (not character-set splitting like strtok)
    char *remaining = str.as.str;
    size_t sep_len = strlen(sep.as.str);

    while (*remaining) {
        char *found = strstr(remaining, sep.as.str);
        if (found) {
            // Found separator - extract substring before it
            size_t len = found - remaining;
            char *part = malloc(len + 1);
            strncpy(part, remaining, len);
            part[len] = '\0';
            list_append(result_list, value_make_str(part));
            remaining = found + sep_len;
        } else {
            // No more separators - add rest of string
            list_append(result_list, value_make_str(strdup(remaining)));
            break;
        }
    }

    // If string ends with separator, add empty string
    if (remaining == str.as.str + strlen(str.as.str) && strlen(str.as.str) > 0) {
        list_append(result_list, value_make_str(strdup("")));
    }

    value_free(str);
    value_free(sep);
    vm_push(vm, value_wrap_list(result_list));
    DISPATCH();
}

L_STR_JOIN: // OP_STR_JOIN - Join list with separator (sep, list -> str)
{
    Value list = vm_pop(vm);
    Value sep = vm_pop(vm);

    if (sep.type != VAL_STR || list.type != VAL_LIST)
    {
        fprintf(stderr, "Runtime error: join() expects separator string and list\n");
        value_free(sep);
        value_free(list);
        exit(1);
    }

    // Calculate total length needed
    int total_len = 0;
    for (int i = 0; i < list.as.list->length; i++)
    {
        Value elem_str = value_to_string(list.as.list->items[i]);
        total_len += strlen(elem_str.as.str);
        value_free(elem_str);
        if (i > 0)
            total_len += strlen(sep.as.str);
    }

    // Build result string
    char *result = malloc(total_len + 1);
    result[0] = '\0';

    for (int i = 0; i < list.as.list->length; i++)
    {
        if (i > 0)
            strcat(result, sep.as.str);
        Value elem_str = value_to_string(list.as.list->items[i]);
        strcat(result, elem_str.as.str);
        value_free(elem_str);
    }

    value_free(sep);
    value_free(list);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_STR_REPLACE: // OP_STR_REPLACE - Replace substring (str, old, new -> str)
{
    Value new_str = vm_pop(vm);
    Value old_str = vm_pop(vm);
    Value str = vm_pop(vm);

    if (str.type != VAL_STR || old_str.type != VAL_STR || new_str.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: replace() expects three strings\n");
        value_free(str);
        value_free(old_str);
        value_free(new_str);
        exit(1);
    }

    const char *s = str.as.str;
    const char *old = old_str.as.str;
    const char *new = new_str.as.str;
    int old_len = strlen(old);
    int new_len = strlen(new);

    // Count occurrences
    int count = 0;
    const char *p = s;
    while ((p = strstr(p, old)) != NULL)
    {
        count++;
        p += old_len;
    }

    // Allocate result
    int result_len = strlen(s) + count * (new_len - old_len);
    char *result = malloc(result_len + 1);
    char *dest = result;

    // Replace occurrences
    p = s;
    while (*p)
    {
        const char *match = strstr(p, old);
        if (match == p)
        {
            strcpy(dest, new);
            dest += new_len;
            p += old_len;
        }
        else
        {
            *dest++ = *p++;
        }
    }
    *dest = '\0';

    value_free(str);
    value_free(old_str);
    value_free(new_str);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_ENCODE: // OP_ENCODE - Encode string to bytes (str, encoding -> str)
{
    Value encoding = vm_pop(vm);
    Value str = vm_pop(vm);

    if (str.type != VAL_STR || encoding.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: encode() expects (string, string)\n");
        value_free(str);
        value_free(encoding);
        exit(1);
    }

    // In C runtime, strings are already UTF-8 bytes, so this is a no-op
    // Just free the encoding parameter and keep the string
    value_free(encoding);
    vm_push(vm, str);
    DISPATCH();
}

L_DECODE: // OP_DECODE - Decode bytes to string (str, encoding -> str)
{
    Value encoding = vm_pop(vm);
    Value str = vm_pop(vm);

    if (str.type != VAL_STR || encoding.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: decode() expects (string, string)\n");
        value_free(str);
        value_free(encoding);
        exit(1);
    }

    // In C runtime, strings are already UTF-8, so this is a no-op
    // Just free the encoding parameter and keep the string
    value_free(encoding);
    vm_push(vm, str);
    DISPATCH();
}

L_FILE_OPEN: // OP_FILE_OPEN - Open file (path, mode -> fd)
{
    Value mode_val = vm_pop(vm);
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR || mode_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: fopen() expects (string, string)\n");
        value_free(path_val);
        value_free(mode_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    const char *mode = mode_val.as.str;

    int flags = O_RDONLY;
    if (strcmp(mode, "w") == 0)
        flags = O_WRONLY | O_CREAT | O_TRUNC;
    else if (strcmp(mode, "a") == 0)
        flags = O_WRONLY | O_CREAT | O_APPEND;
    else if (strcmp(mode, "r+") == 0)
        flags = O_RDWR;
    else if (strcmp(mode, "w+") == 0)
        flags = O_RDWR | O_CREAT | O_TRUNC;

    int fd = open(path, flags, 0666);

    value_free(path_val);
    value_free(mode_val);
    vm_push(vm, value_make_int_si(fd));
    DISPATCH();
}

L_FILE_READ: // OP_FILE_READ - Read from file (fd, size -> string)
{
    Value size_val = vm_pop(vm);
    Value fd_val = vm_pop(vm);

    if (fd_val.type != VAL_INT || size_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: fread() expects (int, int)\n");
        value_free(fd_val);
        value_free(size_val);
        exit(1);
    }

    int fd = (int)fd_val.as.int64;
    int size = (int)size_val.as.int64;

    char *buffer;
    int total_read = 0;

    if (size < 0)
    {
        // Read all
        int capacity = 4096;
        buffer = malloc(capacity);
        int n;
        while ((n = read(fd, buffer + total_read, 4096)) > 0)
        {
            total_read += n;
            if (total_read + 4096 > capacity)
            {
                capacity *= 2;
                buffer = realloc(buffer, capacity);
            }
        }
        buffer[total_read] = '\0';
    }
    else
    {
        buffer = malloc(size + 1);
        total_read = read(fd, buffer, size);
        if (total_read < 0) total_read = 0;
        buffer[total_read] = '\0';
    }

    value_free(fd_val);
    value_free(size_val);
    vm_push(vm, value_make_str(buffer));
    DISPATCH();
}

L_FILE_WRITE: // OP_FILE_WRITE - Write to file (fd, data -> bytes_written)
{
    Value data_val = vm_pop(vm);
    Value fd_val = vm_pop(vm);

    if (fd_val.type != VAL_INT || data_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: fwrite() expects (int, string)\n");
        value_free(fd_val);
        value_free(data_val);
        exit(1);
    }

    int fd = (int)fd_val.as.int64;
    const char *data = data_val.as.str;
    int bytes_written = write(fd, data, strlen(data));

    value_free(fd_val);
    value_free(data_val);
    vm_push(vm, value_make_int_si(bytes_written));
    DISPATCH();
}

L_FILE_CLOSE: // OP_FILE_CLOSE - Close file (fd -> void)
{
    Value fd_val = vm_pop(vm);

    if (fd_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: fclose() expects int\n");
        value_free(fd_val);
        exit(1);
    }

    int fd = (int)fd_val.as.int64;
    close(fd);

    value_free(fd_val);
    DISPATCH();
}

L_FILE_EXISTS: // OP_FILE_EXISTS - Check if file/directory exists (path -> bool)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: exists() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    struct stat st;
    bool exists = (stat(path, &st) == 0);

    value_free(path_val);
    vm_push(vm, value_make_bool(exists));
    DISPATCH();
}

L_FILE_ISFILE: // OP_FILE_ISFILE - Check if path is a file (path -> bool)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: isfile() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    struct stat st;
    bool is_file = (stat(path, &st) == 0 && S_ISREG(st.st_mode));

    value_free(path_val);
    vm_push(vm, value_make_bool(is_file));
    DISPATCH();
}

L_FILE_ISDIR: // OP_FILE_ISDIR - Check if path is a directory (path -> bool)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: isdir() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    struct stat st;
    bool is_dir = (stat(path, &st) == 0 && S_ISDIR(st.st_mode));

    value_free(path_val);
    vm_push(vm, value_make_bool(is_dir));
    DISPATCH();
}

L_FILE_LISTDIR: // OP_FILE_LISTDIR - List directory contents (path -> list)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: listdir() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    DIR *dir = opendir(path);

    if (!dir)
    {
        value_free(path_val);
        vm_push(vm, value_make_list()); // Return empty list on error
        DISPATCH();
    }

    List *list = malloc(sizeof(List));
    list->capacity = 16;
    list->length = 0;
    list->items = malloc(sizeof(Value) * list->capacity);

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL)
    {
        // Skip . and ..
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        if (list->length >= list->capacity)
        {
            list->capacity *= 2;
            list->items = realloc(list->items, sizeof(Value) * list->capacity);
        }

        list->items[list->length++] = value_make_str(strdup(entry->d_name));
    }

    closedir(dir);
    value_free(path_val);

    Value result;
    result.type = VAL_LIST;
    result.as.list = list;
    vm_push(vm, result);
    DISPATCH();
}

L_FILE_MKDIR: // OP_FILE_MKDIR - Create directory (path -> void)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: mkdir() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    mkdir(path, 0755);

    value_free(path_val);
    DISPATCH();
}

L_FILE_MAKEDIRS: // OP_FILE_MAKEDIRS - Create directory and parents (path -> void)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: makedirs() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    char *path_copy = strdup(path);
    char *p = path_copy;

    // Skip leading /
    if (*p == '/') p++;

    while (*p)
    {
        if (*p == '/')
        {
            *p = '\0';
            mkdir(path_copy, 0755);
            *p = '/';
        }
        p++;
    }
    mkdir(path_copy, 0755);

    free(path_copy);
    value_free(path_val);
    DISPATCH();
}

L_FILE_REMOVE: // OP_FILE_REMOVE - Remove file (path -> void)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: remove() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    unlink(path);

    value_free(path_val);
    DISPATCH();
}

L_FILE_RMDIR: // OP_FILE_RMDIR - Remove directory (path -> void)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: rmdir() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    rmdir(path);

    value_free(path_val);
    DISPATCH();
}

L_FILE_RENAME: // OP_FILE_RENAME - Rename/move file (old_path, new_path -> void)
{
    Value new_path_val = vm_pop(vm);
    Value old_path_val = vm_pop(vm);

    if (old_path_val.type != VAL_STR || new_path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: rename() expects (string, string)\n");
        value_free(old_path_val);
        value_free(new_path_val);
        exit(1);
    }

    const char *old_path = old_path_val.as.str;
    const char *new_path = new_path_val.as.str;
    rename(old_path, new_path);

    value_free(old_path_val);
    value_free(new_path_val);
    DISPATCH();
}

L_FILE_GETSIZE: // OP_FILE_GETSIZE - Get file size (path -> int)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: getsize() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    struct stat st;
    int64_t size = 0;

    if (stat(path, &st) == 0)
    {
        size = st.st_size;
    }

    value_free(path_val);
    vm_push(vm, value_make_int_si(size));
    DISPATCH();
}

L_FILE_GETCWD: // OP_FILE_GETCWD - Get current working directory (-> string)
{
    char cwd[PATH_MAX];
    if (getcwd(cwd, sizeof(cwd)) != NULL)
    {
        vm_push(vm, value_make_str(strdup(cwd)));
    }
    else
    {
        vm_push(vm, value_make_str(strdup("")));
    }
    DISPATCH();
}

L_FILE_CHDIR: // OP_FILE_CHDIR - Change working directory (path -> void)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: chdir() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    if (chdir(path) != 0) {
        fprintf(stderr, "Runtime error: chdir() failed for path: %s\n", path);
        value_free(path_val);
        exit(1);
    }

    value_free(path_val);
    DISPATCH();
}

L_FILE_ABSPATH: // OP_FILE_ABSPATH - Get absolute path (path -> string)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: abspath() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    char resolved[PATH_MAX];

    if (realpath(path, resolved) != NULL)
    {
        value_free(path_val);
        vm_push(vm, value_make_str(strdup(resolved)));
    }
    else
    {
        // If realpath fails, return the original path
        vm_push(vm, path_val);
    }
    DISPATCH();
}

L_FILE_BASENAME: // OP_FILE_BASENAME - Get basename (path -> string)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: basename() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    char *path_copy = strdup(path);
    char *base = basename(path_copy);
    char *result = strdup(base);

    free(path_copy);
    value_free(path_val);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_FILE_DIRNAME: // OP_FILE_DIRNAME - Get directory name (path -> string)
{
    Value path_val = vm_pop(vm);

    if (path_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: dirname() expects string\n");
        value_free(path_val);
        exit(1);
    }

    const char *path = path_val.as.str;
    char *path_copy = strdup(path);
    char *dir = dirname(path_copy);
    char *result = strdup(dir);

    free(path_copy);
    value_free(path_val);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_FILE_JOIN: // OP_FILE_JOIN - Join paths (list -> string)
{
    Value list_val = vm_pop(vm);

    if (list_val.type != VAL_LIST)
    {
        fprintf(stderr, "Runtime error: pathjoin() expects list\n");
        value_free(list_val);
        exit(1);
    }

    List *list = list_val.as.list;
    if (list->length == 0)
    {
        value_free(list_val);
        vm_push(vm, value_make_str(strdup("")));
        DISPATCH();
    }

    // Calculate total length needed
    size_t total_len = 0;
    for (int i = 0; i < list->length; i++)
    {
        if (list->items[i].type == VAL_STR)
        {
            total_len += strlen(list->items[i].as.str);
        }
    }
    total_len += list->length; // For path separators and null terminator

    char *result = malloc(total_len);
    result[0] = '\0';

    for (int i = 0; i < list->length; i++)
    {
        if (list->items[i].type == VAL_STR)
        {
            if (i > 0 && result[strlen(result) - 1] != '/')
            {
                strcat(result, "/");
            }
            strcat(result, list->items[i].as.str);
        }
    }

    value_free(list_val);
    vm_push(vm, value_make_str(result));
    DISPATCH();
}

L_SOCKET_CREATE: // OP_SOCKET_CREATE - Create socket (family, type -> sock_id)
{
    Value type_val = vm_pop(vm);
    Value family_val = vm_pop(vm);

    if (family_val.type != VAL_STR || type_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: socket() expects (string, string)\n");
        value_free(family_val);
        value_free(type_val);
        exit(1);
    }

    const char *family_str = family_val.as.str;
    const char *type_str = type_val.as.str;

    int family = AF_INET;
    if (strcasecmp(family_str, "inet6") == 0)
        family = AF_INET6;
    else if (strcasecmp(family_str, "unix") == 0)
        family = AF_UNIX;

    int type = SOCK_STREAM;
    if (strcasecmp(type_str, "dgram") == 0)
        type = SOCK_DGRAM;
    else if (strcasecmp(type_str, "raw") == 0)
        type = SOCK_RAW;

    int sock = socket(family, type, 0);

    value_free(family_val);
    value_free(type_val);
    vm_push(vm, value_make_int_si(sock));
    DISPATCH();
}

L_SOCKET_CONNECT: // OP_SOCKET_CONNECT - Connect socket (sock_id, host, port -> void)
{
    Value port_val = vm_pop(vm);
    Value host_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || host_val.type != VAL_STR || port_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: connect() expects (int, string, int)\n");
        value_free(sock_val);
        value_free(host_val);
        value_free(port_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    const char *host = host_val.as.str;
    int port = (int)port_val.as.int64;

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    inet_pton(AF_INET, host, &addr.sin_addr);

    connect(sock, (struct sockaddr*)&addr, sizeof(addr));

    value_free(sock_val);
    value_free(host_val);
    value_free(port_val);
    DISPATCH();
}

L_SOCKET_BIND: // OP_SOCKET_BIND - Bind socket (sock_id, host, port -> void)
{
    Value port_val = vm_pop(vm);
    Value host_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || host_val.type != VAL_STR || port_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: bind() expects (int, string, int)\n");
        value_free(sock_val);
        value_free(host_val);
        value_free(port_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    const char *host = host_val.as.str;
    int port = (int)port_val.as.int64;

    struct sockaddr_in addr;
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    inet_pton(AF_INET, host, &addr.sin_addr);

    bind(sock, (struct sockaddr*)&addr, sizeof(addr));

    value_free(sock_val);
    value_free(host_val);
    value_free(port_val);
    DISPATCH();
}

L_SOCKET_LISTEN: // OP_SOCKET_LISTEN - Listen on socket (sock_id, backlog -> void)
{
    Value backlog_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || backlog_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: listen() expects (int, int)\n");
        value_free(sock_val);
        value_free(backlog_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    int backlog = (int)backlog_val.as.int64;

    listen(sock, backlog);

    value_free(sock_val);
    value_free(backlog_val);
    DISPATCH();
}

L_SOCKET_ACCEPT: // OP_SOCKET_ACCEPT - Accept connection (sock_id -> client_sock_id)
{
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: accept() expects int\n");
        value_free(sock_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    int client = accept(sock, NULL, NULL);

    value_free(sock_val);
    vm_push(vm, value_make_int_si(client));
    DISPATCH();
}

L_SOCKET_SEND: // OP_SOCKET_SEND - Send data (sock_id, data -> bytes_sent)
{
    Value data_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || data_val.type != VAL_STR)
    {
        fprintf(stderr, "Runtime error: send() expects (int, string)\n");
        value_free(sock_val);
        value_free(data_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    const char *data = data_val.as.str;
    int sent = send(sock, data, strlen(data), 0);

    value_free(sock_val);
    value_free(data_val);
    vm_push(vm, value_make_int_si(sent));
    DISPATCH();
}

L_SOCKET_RECV: // OP_SOCKET_RECV - Receive data (sock_id, size -> string)
{
    Value size_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || size_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: recv() expects (int, int)\n");
        value_free(sock_val);
        value_free(size_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    int size = (int)size_val.as.int64;

    char *buffer = malloc(size + 1);
    int received = recv(sock, buffer, size, 0);
    if (received < 0) received = 0;
    buffer[received] = '\0';

    value_free(sock_val);
    value_free(size_val);
    vm_push(vm, value_make_str(buffer));
    DISPATCH();
}

L_SOCKET_CLOSE: // OP_SOCKET_CLOSE - Close socket (sock_id -> void)
{
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: sclose() expects int\n");
        value_free(sock_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    close(sock);

    value_free(sock_val);
    DISPATCH();
}

L_SOCKET_SETSOCKOPT: // OP_SOCKET_SETSOCKOPT - Set socket option (sock_id, level, option, value -> void)
{
    Value value_val = vm_pop(vm);
    Value option_val = vm_pop(vm);
    Value level_val = vm_pop(vm);
    Value sock_val = vm_pop(vm);

    if (sock_val.type != VAL_INT || level_val.type != VAL_STR ||
        option_val.type != VAL_STR || value_val.type != VAL_INT)
    {
        fprintf(stderr, "Runtime error: setsockopt() expects (int, string, string, int)\n");
        value_free(sock_val);
        value_free(level_val);
        value_free(option_val);
        value_free(value_val);
        exit(1);
    }

    int sock = (int)sock_val.as.int64;
    const char *level_str = level_val.as.str;
    const char *option_str = option_val.as.str;
    int value = (int)value_val.as.int64;

    int level = SOL_SOCKET;
    if (strcasecmp(level_str, "IPPROTO_TCP") == 0)
        level = IPPROTO_TCP;
    else if (strcasecmp(level_str, "IPPROTO_IP") == 0)
        level = IPPROTO_IP;

    int option = SO_REUSEADDR;
    if (strcasecmp(option_str, "SO_KEEPALIVE") == 0)
        option = SO_KEEPALIVE;
    else if (strcasecmp(option_str, "SO_BROADCAST") == 0)
        option = SO_BROADCAST;
    else if (strcasecmp(option_str, "SO_RCVBUF") == 0)
        option = SO_RCVBUF;
    else if (strcasecmp(option_str, "SO_SNDBUF") == 0)
        option = SO_SNDBUF;

    setsockopt(sock, level, option, &value, sizeof(value));

    value_free(sock_val);
    value_free(level_val);
    value_free(option_val);
    value_free(value_val);
    DISPATCH();
}

L_FORK: // OP_FORK - Fork process (-> pid)
{
    pid_t pid = fork();
    vm_push(vm, value_make_int_si((int64_t)pid));
    DISPATCH();
}

L_WAIT: // OP_WAIT - Wait for child process (-> status)
{
    int status = 0;
    wait(&status);
    // Return the exit status
    vm_push(vm, value_make_int_si((int64_t)WEXITSTATUS(status)));
    DISPATCH();
}

L_ABS: // OP_ABS - Absolute value
{
    Value v = vm_pop(vm);
    Value result;

    if (v.type == VAL_INT)
    {
        result = value_make_int_si(llabs(v.as.int64));
    }
    else if (v.type == VAL_F64)
    {
        result = value_make_f64(fabs(v.as.f64));
    }
    else if (v.type == VAL_BIGINT)
    {
        // For bigint, create new bigint with absolute value
        Value result_val;
        result_val.type = VAL_BIGINT;
        result_val.as.bigint = malloc(sizeof(mpz_t));
        mpz_init(*result_val.as.bigint);
        mpz_abs(*result_val.as.bigint, *v.as.bigint);
        result = result_val;
    }
    else
    {
        fprintf(stderr, "Runtime error: abs() expects a number\n");
        value_free(v);
        exit(1);
    }

    value_free(v);
    vm_push(vm, result);
    DISPATCH();
}

L_POW: // OP_POW - Power (base, exponent)
{
    Value exp = vm_pop(vm);
    Value base = vm_pop(vm);

    // Convert to doubles for pow()
    double base_d, exp_d;

    if (base.type == VAL_INT)
    {
        base_d = (double)base.as.int64;
    }
    else if (base.type == VAL_F64)
    {
        base_d = base.as.f64;
    }
    else if (base.type == VAL_BIGINT)
    {
        base_d = mpz_get_d(*base.as.bigint);
    }
    else
    {
        fprintf(stderr, "Runtime error: pow() base must be a number\n");
        value_free(base);
        value_free(exp);
        exit(1);
    }

    if (exp.type == VAL_INT)
    {
        exp_d = (double)exp.as.int64;
    }
    else if (exp.type == VAL_F64)
    {
        exp_d = exp.as.f64;
    }
    else if (exp.type == VAL_BIGINT)
    {
        exp_d = mpz_get_d(*exp.as.bigint);
    }
    else
    {
        fprintf(stderr, "Runtime error: pow() exponent must be a number\n");
        value_free(base);
        value_free(exp);
        exit(1);
    }

    double result_d = pow(base_d, exp_d);

    value_free(base);
    value_free(exp);

    // Return as int if result is whole number and fits
    if (floor(result_d) == result_d && result_d >= LLONG_MIN && result_d <= LLONG_MAX)
    {
        vm_push(vm, value_make_int_si((int64_t)result_d));
    }
    else
    {
        vm_push(vm, value_make_f64(result_d));
    }
    DISPATCH();
}

L_MIN: // OP_MIN - Minimum of two values
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);

    Value result;

    // Both integers
    if (a.type == VAL_INT && b.type == VAL_INT)
    {
        result = value_make_int_si(a.as.int64 < b.as.int64 ? a.as.int64 : b.as.int64);
        value_free(a);
        value_free(b);
    }
    // Both floats
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        result = value_make_f64(a.as.f64 < b.as.f64 ? a.as.f64 : b.as.f64);
        value_free(a);
        value_free(b);
    }
    // Mixed - convert to double
    else
    {
        double a_d = (a.type == VAL_INT) ? (double)a.as.int64 : (a.type == VAL_F64)  ? a.as.f64
                                                            : (a.type == VAL_BIGINT) ? mpz_get_d(*a.as.bigint)
                                                                                     : 0;
        double b_d = (b.type == VAL_INT) ? (double)b.as.int64 : (b.type == VAL_F64)  ? b.as.f64
                                                            : (b.type == VAL_BIGINT) ? mpz_get_d(*b.as.bigint)
                                                                                     : 0;
        result = value_make_f64(a_d < b_d ? a_d : b_d);
        value_free(a);
        value_free(b);
    }

    vm_push(vm, result);
    DISPATCH();
}

L_MAX: // OP_MAX - Maximum of two values
{
    Value b = vm_pop(vm);
    Value a = vm_pop(vm);

    Value result;

    // Both integers
    if (a.type == VAL_INT && b.type == VAL_INT)
    {
        result = value_make_int_si(a.as.int64 > b.as.int64 ? a.as.int64 : b.as.int64);
        value_free(a);
        value_free(b);
    }
    // Both floats
    else if (a.type == VAL_F64 && b.type == VAL_F64)
    {
        result = value_make_f64(a.as.f64 > b.as.f64 ? a.as.f64 : b.as.f64);
        value_free(a);
        value_free(b);
    }
    // Mixed - convert to double
    else
    {
        double a_d = (a.type == VAL_INT) ? (double)a.as.int64 : (a.type == VAL_F64)  ? a.as.f64
                                                            : (a.type == VAL_BIGINT) ? mpz_get_d(*a.as.bigint)
                                                                                     : 0;
        double b_d = (b.type == VAL_INT) ? (double)b.as.int64 : (b.type == VAL_F64)  ? b.as.f64
                                                            : (b.type == VAL_BIGINT) ? mpz_get_d(*b.as.bigint)
                                                                                     : 0;
        result = value_make_f64(a_d > b_d ? a_d : b_d);
        value_free(a);
        value_free(b);
    }

    vm_push(vm, result);
    DISPATCH();
}

L_NEG: // OP_NEG - Unary negation
{
    Value a = vm_pop(vm);

    if (a.type == VAL_INT)
    {
        // Negate small integer
        a.as.int64 = -a.as.int64;
        vm_push(vm, a);
    }
    else if (a.type == VAL_BIGINT)
    {
        // Negate bigint in place
        mpz_neg(*a.as.bigint, *a.as.bigint);
        vm_push(vm, a);
    }
    else if (a.type == VAL_F64)
    {
        // Negate float
        a.as.f64 = -a.as.f64;
        vm_push(vm, a);
    }
    else
    {
        // Type error
        fprintf(stderr, "Runtime Error: Cannot negate type %d\n", a.type);
        value_free(a);
        vm->running = false;
        return;
    }

    DISPATCH();
}

L_ASSERT: // OP_ASSERT - Assert with optional message
{
    // Pop condition from stack
    Value condition = vm_pop(vm);

    bool is_true = false;
    if (condition.type == VAL_BOOL)
        is_true = condition.as.boolean;
    else if (condition.type == VAL_INT)
        is_true = condition.as.int64 != 0;
    else if (condition.type == VAL_F64)
        is_true = condition.as.f64 != 0.0;

    value_free(condition);

    if (!is_true)
    {
        // Check if the next value on stack is a message (string)
        if (vm->stack_top > 0)
        {
            Value msg = vm_pop(vm);
            if (msg.type == VAL_STR)
            {
                fprintf(stderr, "%s\n", msg.as.str);
                value_free(msg);
            }
            else
            {
                // Not a message, push it back
                vm_push(vm, msg);
                fprintf(stderr, "Assertion failed\n");
            }
        }
        else
        {
            fprintf(stderr, "Assertion failed\n");
        }
        vm->running = false;
        vm->exit_code = 1;
        return;
    }

    DISPATCH();
}

L_HALT: // OP_HALT
{
    vm->running = false;
    g_current_vm = NULL; // Clear global VM pointer
    return; // Exit the interpreter loop
}

// Fused opcodes for reducing dispatch overhead
L_LOAD_MULTI: // OP_LOAD_MULTI
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt *multi = (MultiInt *)inst.operand.ptr;

    // Load each variable and push onto stack
    for (int i = 0; i < multi->count; i++)
    {
        int index = multi->values[i];
        Value v = current_frame->vars.vars[index];
        // Fast path for immutable types
        if (likely(v.type == VAL_INT || v.type == VAL_F64 ||
                   v.type == VAL_BOOL || v.type == VAL_VOID)) {
            vm_push(vm, v);
        } else {
            vm_push(vm, value_copy(v));
        }
    }

    DISPATCH();
}

L_FUSED_LOAD_STORE: // OP_FUSED_LOAD_STORE
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt *multi = (MultiInt *)inst.operand.ptr;

    // Process each operation sequentially
    // Even indices are LOAD operations, odd indices are STORE operations
    for (int i = 0; i < multi->count; i++)
    {
        int var_id = multi->values[i];
        
        if (i % 2 == 0)
        {
            // LOAD: load from var_id and push
            Value v = current_frame->vars.vars[var_id];
            if (likely(v.type == VAL_INT || v.type == VAL_F64 ||
                       v.type == VAL_BOOL || v.type == VAL_VOID)) {
                vm_push(vm, v);
            } else {
                vm_push(vm, value_copy(v));
            }
        }
        else
        {
            // STORE: pop value and store to var_id
            Value val = vm_pop(vm);
            value_free(current_frame->vars.vars[var_id]);
            current_frame->vars.vars[var_id] = val;
        }
    }

    DISPATCH();
}

L_FUSED_STORE_LOAD: // OP_FUSED_STORE_LOAD
{
    Instruction inst = vm->code[vm->pc - 1];
    MultiInt *multi = (MultiInt *)inst.operand.ptr;

    CallFrame *frame = &vm->call_stack[vm->call_stack_top - 1];

    // Process each operation sequentially
    // Even indices are STORE operations, odd indices are LOAD operations
    for (int i = 0; i < multi->count; i++)
    {
        int var_id = multi->values[i];
        
        if (i % 2 == 0)
        {
            // STORE: pop value and store to var_id
            Value val = vm_pop(vm);
            value_free(current_frame->vars.vars[var_id]);
            current_frame->vars.vars[var_id] = val;
        }
        else
        {
            // LOAD: load from var_id and push
            vm_push(vm, value_copy(frame->vars.vars[var_id]));
        }
    }

    DISPATCH();
}

// Fused LOAD2 + arithmetic operations (hot loop optimizations)
L_LOAD2_ADD_I64: // OP_LOAD2_ADD_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src; // First index
    int idx2 = inst.operand.indices.dst; // Second index

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    // Always use value_add to handle overflow detection
    Value result = value_add(a, b);
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_SUB_I64: // OP_LOAD2_SUB_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    // Always use value_sub to handle overflow detection
    Value result = value_sub(a, b);
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_MUL_I64: // OP_LOAD2_MUL_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    // Always use value_mul to handle overflow detection
    Value result = value_mul(a, b);
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_MOD_I64: // OP_LOAD2_MOD_I64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (a.type != VAL_INT || b.type != VAL_INT)
    {
        vm_runtime_error(vm, "Modulo requires integer operands", 0);
        return;
    }

    if (b.as.int64 == 0)
    {
        vm_runtime_error(vm, "[ZeroDivisionError] Division by zero in modulo operation", 0);
        return;
    }

    Value result;
    result.type = VAL_INT;
    result.as.int64 = a.as.int64 % b.as.int64;
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_ADD_F64: // OP_LOAD2_ADD_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (a.type != VAL_F64 || b.type != VAL_F64)
    {
        vm_runtime_error(vm, "LOAD2_ADD_F64 requires float operands", 0);
        return;
    }

    Value result;
    result.type = VAL_F64;
    result.as.f64 = a.as.f64 + b.as.f64;
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_SUB_F64: // OP_LOAD2_SUB_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (a.type != VAL_F64 || b.type != VAL_F64)
    {
        vm_runtime_error(vm, "LOAD2_SUB_F64 requires float operands", 0);
        return;
    }

    Value result;
    result.type = VAL_F64;
    result.as.f64 = a.as.f64 - b.as.f64;
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_MUL_F64: // OP_LOAD2_MUL_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (a.type != VAL_F64 || b.type != VAL_F64)
    {
        vm_runtime_error(vm, "LOAD2_MUL_F64 requires float operands", 0);
        return;
    }

    Value result;
    result.type = VAL_F64;
    result.as.f64 = a.as.f64 * b.as.f64;
    vm_push(vm, result);

    DISPATCH();
}

L_LOAD2_DIV_F64: // OP_LOAD2_DIV_F64
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (a.type != VAL_F64 || b.type != VAL_F64)
    {
        vm_runtime_error(vm, "LOAD2_DIV_F64 requires float operands", 0);
        return;
    }

    if (b.as.f64 == 0.0)
    {
        vm_runtime_error(vm, "[ZeroDivisionError] Float division by zero", 0);
        return;
    }

    Value result;
    result.type = VAL_F64;
    result.as.f64 = a.as.f64 / b.as.f64;
    vm_push(vm, result);

    DISPATCH();
}

// Fused LOAD2 + comparison operations
L_LOAD2_CMP_LT: // OP_LOAD2_CMP_LT
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    // Fast path for int64 comparison
    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 < b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_LT);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_LOAD2_CMP_GT: // OP_LOAD2_CMP_GT
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 > b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_GT);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_LOAD2_CMP_LE: // OP_LOAD2_CMP_LE
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 <= b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_LE);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_LOAD2_CMP_GE: // OP_LOAD2_CMP_GE
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 >= b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_GE);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_LOAD2_CMP_EQ: // OP_LOAD2_CMP_EQ
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 == b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_EQ);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_LOAD2_CMP_NE: // OP_LOAD2_CMP_NE
{
    Instruction inst = vm->code[vm->pc - 1];
    int idx1 = inst.operand.indices.src;
    int idx2 = inst.operand.indices.dst;

    Value a = current_frame->vars.vars[idx1];
    Value b = current_frame->vars.vars[idx2];

    if (likely(a.type == VAL_INT && b.type == VAL_INT))
    {
        vm_push(vm, value_make_bool(a.as.int64 != b.as.int64));
    }
    else
    {
        Value result = value_compare(a, b, OP_CMP_NE);
        vm_push(vm, result);
    }

    DISPATCH();
}

L_PY_IMPORT: // OP_PY_IMPORT - Import a Python module (module_name -> module_object)
{
    Value module_name_val = vm_pop(vm);

    if (module_name_val.type != VAL_STR) {
        fprintf(stderr, "Runtime error: py_import() expects a string module name\n");
        value_free(module_name_val);
        exit(1);
    }

    const char *module_name = module_name_val.as.str;

    // Import the module
    PyObject *pName = PyUnicode_DecodeFSDefault(module_name);
    PyObject *pModule = PyImport_Import(pName);
    Py_DECREF(pName);

    if (pModule == NULL) {
        PyErr_Print();
        fprintf(stderr, "Runtime error: Failed to import Python module: %s\n", module_name);
        value_free(module_name_val);
        exit(1);
    }

    // Return module as PyObject value
    Value result;
    result.type = VAL_PYOBJECT;
    result.as.pyobj = pModule;

    value_free(module_name_val);
    vm_push(vm, result);
    DISPATCH();
}

L_PY_CALL: // OP_PY_CALL - Call a Python function (module_name, func_name, arg1, ..., argN, num_args -> result)
{
    Value num_args_val = vm_pop(vm);

    if (num_args_val.type != VAL_INT) {
        fprintf(stderr, "Runtime error: py_call() num_args must be an integer\n");
        value_free(num_args_val);
        exit(1);
    }

    int num_args = (int)num_args_val.as.int64;

    // Collect arguments
    Value args[num_args];
    for (int i = num_args - 1; i >= 0; i--) {
        args[i] = vm_pop(vm);
    }

    Value func_name_val = vm_pop(vm);
    Value module_name_val = vm_pop(vm);

    if (module_name_val.type != VAL_STR || func_name_val.type != VAL_STR) {
        fprintf(stderr, "Runtime error: py_call() expects string module and function names\n");
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(func_name_val);
        value_free(module_name_val);
        value_free(num_args_val);
        exit(1);
    }

    const char *module_name = module_name_val.as.str;
    const char *func_name = func_name_val.as.str;

    // Import module
    PyObject *pName = PyUnicode_DecodeFSDefault(module_name);
    PyObject *pModule = PyImport_Import(pName);
    Py_DECREF(pName);

    if (pModule == NULL) {
        PyErr_Print();
        fprintf(stderr, "Runtime error: Failed to import Python module: %s\n", module_name);
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(func_name_val);
        value_free(module_name_val);
        value_free(num_args_val);
        exit(1);
    }

    // Get function
    PyObject *pFunc = PyObject_GetAttrString(pModule, func_name);

    if (pFunc == NULL || !PyCallable_Check(pFunc)) {
        if (PyErr_Occurred()) PyErr_Print();
        fprintf(stderr, "Runtime error: Cannot find or call function: %s.%s\n", module_name, func_name);
        Py_XDECREF(pFunc);
        Py_DECREF(pModule);
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(func_name_val);
        value_free(module_name_val);
        value_free(num_args_val);
        exit(1);
    }

    // Convert arguments to Python objects
    PyObject *pArgs = PyTuple_New(num_args);
    for (int i = 0; i < num_args; i++) {
        PyObject *pValue = fr_value_to_python(args[i]);
        PyTuple_SetItem(pArgs, i, pValue);
    }

    // Call the function
    PyObject *pResult = PyObject_CallObject(pFunc, pArgs);
    Py_DECREF(pArgs);
    Py_DECREF(pFunc);
    Py_DECREF(pModule);

    if (pResult == NULL) {
        PyErr_Print();
        fprintf(stderr, "Runtime error: Python function call failed: %s.%s\n", module_name, func_name);
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(func_name_val);
        value_free(module_name_val);
        value_free(num_args_val);
        exit(1);
    }

    // Convert result back to fr Value
    Value result = python_to_fr_value(pResult);
    Py_DECREF(pResult);

    // Cleanup
    for (int i = 0; i < num_args; i++) {
        value_free(args[i]);
    }
    value_free(func_name_val);
    value_free(module_name_val);
    value_free(num_args_val);

    vm_push(vm, result);
    DISPATCH();
}

L_PY_GETATTR: // OP_PY_GETATTR - Get attribute from Python object (obj, attr_name -> value)
{
    Value attr_name_val = vm_pop(vm);
    Value obj_val = vm_pop(vm);

    if (attr_name_val.type != VAL_STR) {
        fprintf(stderr, "Runtime error: py_getattr() attribute name must be a string\n");
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    if (obj_val.type != VAL_PYOBJECT) {
        fprintf(stderr, "Runtime error: py_getattr() expects a Python object\n");
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    const char *attr_name = attr_name_val.as.str;
    PyObject *pObj = obj_val.as.pyobj;

    // Get attribute
    PyObject *pAttr = PyObject_GetAttrString(pObj, attr_name);

    if (pAttr == NULL) {
        if (PyErr_Occurred()) PyErr_Print();
        fprintf(stderr, "Runtime error: Attribute not found: %s\n", attr_name);
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    // Convert to fr value
    Value result = python_to_fr_value(pAttr);
    Py_DECREF(pAttr);

    value_free(attr_name_val);
    value_free(obj_val);

    vm_push(vm, result);
    DISPATCH();
}

L_PY_SETATTR: // OP_PY_SETATTR - Set attribute on Python object (obj, attr_name, value -> none)
{
    Value value_val = vm_pop(vm);
    Value attr_name_val = vm_pop(vm);
    Value obj_val = vm_pop(vm);

    if (attr_name_val.type != VAL_STR) {
        fprintf(stderr, "Runtime error: py_setattr() attribute name must be a string\n");
        value_free(value_val);
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    if (obj_val.type != VAL_PYOBJECT) {
        fprintf(stderr, "Runtime error: py_setattr() expects a Python object\n");
        value_free(value_val);
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    const char *attr_name = attr_name_val.as.str;
    PyObject *pObj = obj_val.as.pyobj;

    // Convert fr value to Python object
    PyObject *pValue = fr_value_to_python(value_val);
    if (pValue == NULL) {
        fprintf(stderr, "Runtime error: Failed to convert value to Python object\n");
        value_free(value_val);
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    // Set attribute
    int result = PyObject_SetAttrString(pObj, attr_name, pValue);
    Py_DECREF(pValue);

    if (result == -1) {
        if (PyErr_Occurred()) PyErr_Print();
        fprintf(stderr, "Runtime error: Failed to set attribute: %s\n", attr_name);
        value_free(value_val);
        value_free(attr_name_val);
        value_free(obj_val);
        exit(1);
    }

    value_free(value_val);
    value_free(attr_name_val);
    value_free(obj_val);

    // Push void value (since setattr returns None)
    Value void_val = {.type = VAL_VOID};
    vm_push(vm, void_val);
    DISPATCH();
}L_PY_CALL_METHOD: // OP_PY_CALL_METHOD - Call method on Python object (obj, method_name, arg1, ..., argN, num_args -> result)
{
    Value num_args_val = vm_pop(vm);

    if (num_args_val.type != VAL_INT) {
        fprintf(stderr, "Runtime error: py_call_method() num_args must be an integer\n");
        value_free(num_args_val);
        exit(1);
    }

    int num_args = (int)num_args_val.as.int64;

    // Collect arguments
    Value args[num_args];
    for (int i = num_args - 1; i >= 0; i--) {
        args[i] = vm_pop(vm);
    }

    Value method_name_val = vm_pop(vm);
    Value obj_val = vm_pop(vm);

    if (method_name_val.type != VAL_STR) {
        fprintf(stderr, "Runtime error: py_call_method() method name must be a string\n");
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(method_name_val);
        value_free(obj_val);
        value_free(num_args_val);
        exit(1);
    }

    if (obj_val.type != VAL_PYOBJECT) {
        fprintf(stderr, "Runtime error: py_call_method() expects a Python object\n");
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(method_name_val);
        value_free(obj_val);
        value_free(num_args_val);
        exit(1);
    }

    const char *method_name = method_name_val.as.str;
    PyObject *pObj = obj_val.as.pyobj;

    // Get method
    PyObject *pMethod = PyObject_GetAttrString(pObj, method_name);

    if (pMethod == NULL || !PyCallable_Check(pMethod)) {
        if (PyErr_Occurred()) PyErr_Print();
        fprintf(stderr, "Runtime error: Method not found or not callable: %s\n", method_name);
        Py_XDECREF(pMethod);
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(method_name_val);
        value_free(obj_val);
        value_free(num_args_val);
        exit(1);
    }

    // Convert arguments to Python objects
    PyObject *pArgs = PyTuple_New(num_args);
    for (int i = 0; i < num_args; i++) {
        PyObject *pValue = fr_value_to_python(args[i]);
        PyTuple_SetItem(pArgs, i, pValue);
    }

    // Call the method
    PyObject *pResult = PyObject_CallObject(pMethod, pArgs);
    Py_DECREF(pArgs);
    Py_DECREF(pMethod);

    if (pResult == NULL) {
        if (PyErr_Occurred()) PyErr_Print();
        fprintf(stderr, "Runtime error: Python method call failed: %s\n", method_name);
        for (int i = 0; i < num_args; i++) value_free(args[i]);
        value_free(method_name_val);
        value_free(obj_val);
        value_free(num_args_val);
        exit(1);
    }

    // Convert result back to fr Value
    Value result_val = python_to_fr_value(pResult);
    Py_DECREF(pResult);

    // Cleanup
    for (int i = 0; i < num_args; i++) {
        value_free(args[i]);
    }
    value_free(method_name_val);
    value_free(obj_val);
    value_free(num_args_val);

    vm_push(vm, result_val);
    DISPATCH();
}

L_TRY_BEGIN: // OP_TRY_BEGIN - Begin exception handler
{
    Instruction inst = vm->code[vm->pc - 1];

    // Parse combined string "exc_type|label"
    char *combined = strdup(inst.operand.str_val);
    char *exc_type = strtok(combined, "|");
    char *label_name = strtok(NULL, "|");

    if (!label_name) {
        fprintf(stderr, "Error: TRY_BEGIN malformed operand\n");
        free(combined);
        exit(1);
    }

    // Find label PC within current function
    int func_index = vm_find_function_for_pc(vm, vm->pc - 1);
    int handler_pc = vm_find_label(vm, label_name, func_index);
    if (handler_pc == -1) {
        fprintf(stderr, "Error: Exception handler label not found: %s\n", label_name);
        free(combined);
        exit(1);
    }

    // Push exception handler onto stack
    if (vm->exception_handler_count >= MAX_EXCEPTION_HANDLERS) {
        fprintf(stderr, "Error: Maximum exception handler depth exceeded\n");
        free(combined);
        exit(1);
    }

    ExceptionHandler *handler = &vm->exception_handlers[vm->exception_handler_count++];
    handler->exc_type = strdup(exc_type);
    handler->handler_pc = handler_pc;
    handler->stack_top = vm->stack_top;
    handler->call_stack_top = vm->call_stack_top;

    free(combined);
    DISPATCH();
}

L_TRY_END: // OP_TRY_END - End exception handler (pop from handler stack)
{
    if (vm->exception_handler_count > 0) {
        vm->exception_handler_count--;
        ExceptionHandler *handler = &vm->exception_handlers[vm->exception_handler_count];
        free(handler->exc_type);
        handler->exc_type = NULL;
    }
    DISPATCH();
}

L_RAISE: // OP_RAISE - Raise an exception
{
    Instruction inst = vm->code[vm->pc - 1];

    // Parse exception type and message from quoted strings
    // Format: "ExceptionType" "message" or "ExceptionType" "" or "" ""
    char *msg = inst.operand.str_val;

    // Make a working copy to avoid modifying the original
    char *working = strdup(msg);
    char *p = working;
    while (*p == ' ' || *p == '\t') p++;

    char exc_type[256] = "";
    char message[1024] = "";

    if (*p == '"') {
        // Parse exception type
        p++; // Skip opening quote
        char *exc_start = p;
        while (*p && *p != '"') p++;

        if (*p == '"') {
            size_t exc_len = p - exc_start;
            if (exc_len < sizeof(exc_type)) {
                strncpy(exc_type, exc_start, exc_len);
                exc_type[exc_len] = '\0';
            }
            p++; // Skip closing quote

            // Skip whitespace to message
            while (*p == ' ' || *p == '\t') p++;

            if (*p == '"') {
                p++; // Skip opening quote of message
                char *msg_start = p;
                while (*p && *p != '"') p++;
                if (*p == '"') {
                    size_t msg_len = p - msg_start;
                    if (msg_len < sizeof(message)) {
                        strncpy(message, msg_start, msg_len);
                        message[msg_len] = '\0';
                    }
                }
            }
        }
    }

    free(working);

    // Build the error message
    // For detection of exception type, we'll use the exc_type directly in vm_runtime_error
    // But the message shown to user should just be the message itself
    char full_message[2048];
    if (message[0] != '\0') {
        snprintf(full_message, sizeof(full_message), "%s", message);
    } else if (exc_type[0] != '\0') {
        snprintf(full_message, sizeof(full_message), "%s", exc_type);
    } else {
        // Bare raise (re-raise current exception) - not supported in C VM yet
        fprintf(stderr, "Error: Bare raise not supported in C VM\n");
        exit(1);
    }

    // Trigger the error
    // Pass the exc_type info by including it as a detectable pattern
    // We'll modify vm_runtime_error to detect the exception type from exc_type variable
    // For now, prepend it in a way vm_runtime_error can detect
    char error_with_type[4096];  // Increased buffer size to prevent truncation
    if (exc_type[0] != '\0') {
        // Add a marker that vm_runtime_error can detect and remove
        snprintf(error_with_type, sizeof(error_with_type), "[%s] %s", exc_type, full_message);
    } else {
        snprintf(error_with_type, sizeof(error_with_type), "%s", full_message);
    }

    vm_runtime_error(vm, error_with_type, 0);
    // If we get here, a handler was found and vm_runtime_error returned
    // The PC has been updated to jump to the handler, so just dispatch
    DISPATCH();
}

    exit(1);
}

int main(int argc, char **argv) {
    if (argc < 2)
    {
        fprintf(stderr, "Usage: %s [--debug-info] <bytecode_file> [args...]\n", argv[0]);
        return 1;
    }

    VM vm;
    vm_init(&vm);

    // Check for --debug-info flag
    int arg_offset = 1;
    bool has_debug_info = false;
    if (strcmp(argv[1], "--debug-info") == 0) {
        has_debug_info = true;
        arg_offset = 2;

        if (argc < 3) {
            fprintf(stderr, "Error: --debug-info flag requires bytecode file argument\n");
            return 1;
        }
    }

    if (has_debug_info) {
        // Read JSON debug info from stdin
        char buffer[1024 * 1024]; // 1MB buffer
        size_t total_read = 0;
        size_t bytes_read;

        while ((bytes_read = fread(buffer + total_read, 1, sizeof(buffer) - total_read - 1, stdin)) > 0) {
            total_read += bytes_read;
            if (total_read >= sizeof(buffer) - 1) break;
        }
        buffer[total_read] = '\0';

        // Simple JSON parsing (looking for "file" and "source" fields)
        char *file_start = strstr(buffer, "\"file\":");
        char *source_start = strstr(buffer, "\"source\":");

        if (file_start) {
            // Skip past "file": and find the opening quote of the value
            file_start = strchr(file_start + 6, '"'); // Skip past "file"
            if (file_start) {
                file_start++; // Skip the opening quote
                char *file_end = strchr(file_start, '"');
                if (file_end) {
                    vm.debug_source_file = strndup(file_start, file_end - file_start);
                }
            }
        }

        if (source_start) {
            // Skip past "source": and find the opening quote of the value
            source_start = strchr(source_start + 8, '"'); // Skip past "source"
            if (source_start) {
                source_start++; // Skip the opening quote

                // Parse the JSON string, handling escape sequences
                char *source_buffer = malloc(strlen(source_start) + 1);
                int src_idx = 0;
                int dst_idx = 0;
            bool in_escape = false;

            while (source_start[src_idx] && source_start[src_idx] != '"') {
                if (in_escape) {
                    switch (source_start[src_idx]) {
                        case 'n': source_buffer[dst_idx++] = '\n'; break;
                        case 't': source_buffer[dst_idx++] = '\t'; break;
                        case 'r': source_buffer[dst_idx++] = '\r'; break;
                        case '\\': source_buffer[dst_idx++] = '\\'; break;
                        case '"': source_buffer[dst_idx++] = '"'; break;
                        default: source_buffer[dst_idx++] = source_start[src_idx]; break;
                    }
                    in_escape = false;
                } else if (source_start[src_idx] == '\\') {
                    in_escape = true;
                } else {
                    source_buffer[dst_idx++] = source_start[src_idx];
                }
                src_idx++;
            }
            source_buffer[dst_idx] = '\0';

            // Split source into lines
            int line_count = 1;
            for (int i = 0; i < dst_idx; i++) {
                if (source_buffer[i] == '\n') line_count++;
            }

            vm.debug_source_lines = malloc(sizeof(char*) * line_count);
            vm.debug_source_line_count = line_count;

            char *line_start = source_buffer;
            int line_idx = 0;
            for (int i = 0; i <= dst_idx; i++) {
                if (source_buffer[i] == '\n' || source_buffer[i] == '\0') {
                    int line_len = source_buffer + i - line_start;
                    vm.debug_source_lines[line_idx] = strndup(line_start, line_len);
                    line_idx++;
                    line_start = source_buffer + i + 1;
                }
            }

            free(source_buffer);
            }
        }

        // Parse line_map array
        char *line_map_start = strstr(buffer, "\"line_map\":");
        if (line_map_start) {
            line_map_start = strchr(line_map_start, '[');
            if (line_map_start) {
                line_map_start++; // Skip [

                // Count entries
                int count = 0;
                char *p = line_map_start;
                while (*p && *p != ']') {
                    if (*p == ',' || (*p >= '0' && *p <= '9')) {
                        if (*p >= '0' && *p <= '9') {
                            while (*p >= '0' && *p <= '9') p++;
                            count++;
                            while (*p == ' ' || *p == ',') p++;
                        } else {
                            p++;
                        }
                    } else {
                        p++;
                    }
                }

                if (count > 0) {
                    vm.debug_line_map = malloc(sizeof(int) * count);
                    vm.debug_line_map_count = count;

                    p = line_map_start;
                    int idx = 0;
                    while (*p && *p != ']' && idx < count) {
                        while (*p == ' ' || *p == ',') p++;
                        if (*p >= '0' && *p <= '9') {
                            vm.debug_line_map[idx++] = atoi(p);
                            while (*p >= '0' && *p <= '9') p++;
                        } else {
                            p++;
                        }
                    }
                }
            }
        }
    }

    // Store program arguments (skip program name, optional --debug-info flag, and bytecode file)
    vm.prog_argc = argc - arg_offset - 1;
    vm.prog_argv = (argc > arg_offset + 1) ? &argv[arg_offset + 1] : NULL;

    if (!vm_load_bytecode(&vm, argv[arg_offset]))
    {
        return 1;
    }
    vm_run(&vm);
    int exit_code = vm.exit_code;
    vm_free(&vm);
    return exit_code;
}


