"""
Bytecode Optimizer for fr
Performs peephole optimizations on generated bytecode
"""

from typing import List, Dict, Tuple

class BytecodeOptimizer:
    """Optimizes bytecode instruction sequences"""

    def __init__(self):
        self.labels: Dict[str, int] = {}  # Label -> line number
        self.label_counter: int = 0  # Counter for generating unique labels

    def optimize(self, bytecode: str) -> str:
        """Apply all optimization passes to bytecode"""
        lines = bytecode.strip().split('\n')

        # Pass 1: Combine common patterns (does INC_LOCAL conversion and fused operations)
        lines = self.combine_patterns(lines)

        # Pass 1.5: Merge CONST + STORE initialization patterns
        lines = self.merge_const_store_init(lines)

        # Pass 2: Merge consecutive constant instructions
        lines = self.merge_constants(lines)

        # Pass 3: Optimize variable copies (LOAD+STORE -> COPY_LOCAL_REF)
        lines = self.optimize_variable_copies(lines)

        # Pass 4: Optimize list creation patterns
        lines = self.optimize_list_creation(lines)

        # Pass 5: Remove redundant LOAD/STORE pairs
        lines = self.remove_redundant_loads(lines)

        # Pass 6: Optimize negation patterns
        lines = self.optimize_negation(lines)

        # Pass 7: Optimize jump chains and redundant jumps
        lines = self.optimize_jumps(lines)

        # Pass 8: Remove dead stores
        lines = self.remove_dead_stores(lines)

        # Pass 9: Algebraic simplifications (identity operations)
        lines = self.algebraic_simplifications(lines)

        # Pass 10: Optimize comparison inversions (NOT after comparison)
        lines = self.optimize_comparison_inversions(lines)

        # Pass 11: Constant folding
        lines = self.constant_folding(lines)

        # Pass 12: Fuse repeated arithmetic+store blocks with differing operands
        lines = self.fuse_repeated_arith_store(lines)

        # Pass 13: Optimize switch-case patterns
        lines = self.optimize_switch_cases(lines)

        # Pass 14: Fuse similar labeled blocks (switch case bodies with same pattern)
        lines = self.fuse_similar_case_bodies(lines)

        # Pass 15: Cache and reuse loaded values with DUP
        lines = self.cache_loaded_values(lines)

        # Pass 16: Eliminate redundant loop continuation labels
        lines = self.eliminate_redundant_continue_labels(lines)

        # Pass 17: Optimize if-else to SELECT instruction
        lines = self.optimize_if_else_select(lines)

        return '\n'.join(lines)

    def fuse_repeated_arith_store(self, lines: List[str]) -> List[str]:
        """Detect repeated blocks of the form:
        LOAD <src>
        CONST_I64|CONST_F64 <val>
        <OP>
        STORE <dst>

        And fuse consecutive repetitions into a single FUSED_ARITH_CONST
        instruction that carries the op, type and a flat list of (src dst val)
        triples. This reduces instruction overhead when the same operation is
        applied multiple times with different operands.
        """
        result: List[str] = []
        i = 0

        while i < len(lines):
            # Try to match a block at i
            if i + 3 < len(lines):
                a = lines[i].strip()
                b = lines[i+1].strip()
                c = lines[i+2].strip()
                d = lines[i+3].strip()

                # Match LOAD, CONST, OP, STORE
                if a.startswith('LOAD ') and (b.startswith('CONST_I64 ') or b.startswith('CONST_F64 ')) and d.startswith('STORE '):
                    # Determine op and type
                    if c in ('ADD_I64', 'SUB_I64', 'MUL_I64', 'DIV_I64', 'MOD_I64') and b.startswith('CONST_I64 '):
                        typ = 'I64'
                        op = c.replace('_I64', '')
                    elif c in ('ADD_F64', 'SUB_F64', 'MUL_F64', 'DIV_F64') and b.startswith('CONST_F64 '):
                        typ = 'F64'
                        op = c.replace('_F64', '')
                    else:
                        typ = None
                        op = None

                    if typ is not None:
                        # Collect consecutive matching blocks
                        blocks: List[Tuple[str, str, str]] = []
                        j = i
                        while j + 3 < len(lines):
                            a2 = lines[j].strip()
                            b2 = lines[j+1].strip()
                            c2 = lines[j+2].strip()
                            d2 = lines[j+3].strip()

                            if not (a2.startswith('LOAD ') and d2.startswith('STORE ')):
                                break

                            # same type and op
                            if typ == 'I64' and not (b2.startswith('CONST_I64 ') and c2 == f'{op}_I64'):
                                break
                            if typ == 'F64' and not (b2.startswith('CONST_F64 ') and c2 == f'{op}_F64'):
                                break

                            src = a2.split()[1]
                            const_val = b2.split(' ', 1)[1]
                            dst = d2.split()[1]
                            blocks.append((src, dst, const_val))
                            j += 4

                        if len(blocks) >= 2:
                            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '
                            # Build fused instr: FUSED_ARITH_CONST <OP> <TYPE> <count> <src dst val ...>
                            args = ' '.join(f"{s} {t} {v}" for s, t, v in blocks)
                            result.append(f"{indent}FUSED_ARITH_CONST {op} {typ} {len(blocks)} {args}")
                            i = j
                            continue

            # default: copy
            result.append(lines[i])
            i += 1

        return result

    def merge_const_store_init(self, lines: List[str]) -> List[str]:
        """
        Optimize CONST + STORE patterns into STORE_CONST_*.
        E.g., CONST_I64 0, STORE 1 -> STORE_CONST_I64 1 0
        Multiple pairs: STORE_CONST_I64 slot1 val1 slot2 val2 ...
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '

            # Pattern: CONST_I64 X followed by STORE Y
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('STORE ')):

                pairs = []
                j = i

                # Collect consecutive CONST_I64 + STORE pairs
                while (j + 1 < len(lines) and
                       lines[j].strip().startswith('CONST_I64 ') and
                       lines[j+1].strip().startswith('STORE ')):
                    const_val = lines[j].strip().split('CONST_I64 ', 1)[1]
                    store_slot = lines[j+1].strip().split('STORE ', 1)[1]
                    pairs.append((store_slot, const_val))
                    j += 2

                if pairs:
                    args = []
                    for slot, val in pairs:
                        args.extend([slot, val])
                    result.append(f"{indent}STORE_CONST_I64 {' '.join(args)}")
                    i = j
                    continue

            # Pattern: CONST_F64 X followed by STORE Y
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip().startswith('STORE ')):

                pairs = []
                j = i

                while (j + 1 < len(lines) and
                       lines[j].strip().startswith('CONST_F64 ') and
                       lines[j+1].strip().startswith('STORE ')):
                    const_val = lines[j].strip().split('CONST_F64 ', 1)[1]
                    store_slot = lines[j+1].strip().split('STORE ', 1)[1]
                    pairs.append((store_slot, const_val))
                    j += 2

                if pairs:
                    args = []
                    for slot, val in pairs:
                        args.extend([slot, val])
                    result.append(f"{indent}STORE_CONST_F64 {' '.join(args)}")
                    i = j
                    continue

            # Pattern: CONST_BOOL X followed by STORE Y
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip().startswith('STORE ')):

                pairs = []
                j = i

                while (j + 1 < len(lines) and
                       lines[j].strip().startswith('CONST_BOOL ') and
                       lines[j+1].strip().startswith('STORE ')):
                    const_val = lines[j].strip().split('CONST_BOOL ', 1)[1]
                    store_slot = lines[j+1].strip().split('STORE ', 1)[1]
                    pairs.append((store_slot, const_val))
                    j += 2

                if pairs:
                    args = []
                    for slot, val in pairs:
                        args.extend([slot, val])
                    result.append(f"{indent}STORE_CONST_BOOL {' '.join(args)}")
                    i = j
                    continue

            # Pattern: CONST_STR X followed by STORE Y
            # Strings are complex with quoting, skip for now

            result.append(lines[i])
            i += 1

        return result

    def merge_constants(self, lines: List[str]) -> List[str]:
        """
        Merge consecutive constant instructions into multi-arg versions.
        E.g., CONST_I64 1, CONST_I64 2 -> CONST_I64 1 2
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '

            # Check for consecutive CONST_I64 instructions
            if line.startswith('CONST_I64 '):
                constants = [line.split('CONST_I64 ', 1)[1]]
                j = i + 1

                # Collect consecutive CONST_I64 instructions
                while j < len(lines) and lines[j].strip().startswith('CONST_I64 '):
                    constants.append(lines[j].strip().split('CONST_I64 ', 1)[1])
                    j += 1

                # Only merge if we have 2 or more
                if len(constants) >= 2:
                    result.append(f"{indent}CONST_I64 {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_F64 instructions
            if line.startswith('CONST_F64 '):
                constants = [line.split('CONST_F64 ', 1)[1]]
                j = i + 1

                while j < len(lines) and lines[j].strip().startswith('CONST_F64 '):
                    constants.append(lines[j].strip().split('CONST_F64 ', 1)[1])
                    j += 1

                if len(constants) >= 2:
                    result.append(f"{indent}CONST_F64 {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_BOOL instructions
            if line.startswith('CONST_BOOL '):
                constants = [line.split('CONST_BOOL ', 1)[1]]
                j = i + 1

                while j < len(lines) and lines[j].strip().startswith('CONST_BOOL '):
                    constants.append(lines[j].strip().split('CONST_BOOL ', 1)[1])
                    j += 1

                if len(constants) >= 2:
                    result.append(f"{indent}CONST_BOOL {' '.join(constants)}")
                    i = j
                    continue

            # Check for consecutive CONST_STR instructions
            # Note: These need special handling due to quoted strings
            # BUT: Don't merge if they're arguments to PY_CALL (module_name, func_name)
            if line.startswith('CONST_STR "'):
                constants = []
                j = i

                while j < len(lines) and lines[j].strip().startswith('CONST_STR "'):
                    # Extract the quoted string (everything after 'CONST_STR ')
                    const_str = lines[j].strip()[10:]  # len('CONST_STR ') = 10
                    constants.append(const_str)
                    j += 1

                # Check if the next non-whitespace line after the CONST_STR sequence is PY_CALL
                # If so, don't merge - these strings are separate arguments
                next_line = lines[j].strip() if j < len(lines) else ""
                is_py_call_args = next_line.startswith('CONST_I64 ') and j + 1 < len(lines) and lines[j + 1].strip() == 'PY_CALL'
                
                if len(constants) >= 2 and not is_py_call_args:
                    result.append(f"{indent}CONST_STR {' '.join(constants)}")
                    i = j
                    continue

            result.append(lines[i])
            i += 1

        return result

    def optimize_variable_copies(self, lines: List[str]) -> List[str]:
        """
        Convert LOAD x; STORE y patterns to COPY_LOCAL_REF x y when safe.
        Safe when: y is not modified before next use (simple heuristic: y is reassigned before next load)
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: LOAD x, STORE y
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('STORE ')):

                src_var = line.split()[1]
                dst_var = lines[i+1].strip().split()[1]

                # Don't optimize if it's the same variable (handled by other pass)
                if src_var != dst_var:
                    if is_safe := self._is_copy_ref_safe(
                        lines, i + 2, src_var, dst_var
                    ):
                        result.append(f"{indent}COPY_LOCAL_REF {src_var} {dst_var}")
                    else:
                        result.append(f"{indent}COPY_LOCAL {src_var} {dst_var}")
                    i += 2
                    continue

            result.append(lines[i])
            i += 1

        return result

    def _is_copy_ref_safe(self, lines: List[str], start_idx: int, src_var: str, dst_var: str, lookahead: int = 10) -> bool:
        """
        Determine if using COPY_LOCAL_REF is safe.
        Safe if dst is not modified (via ADD/SUB/etc) before being reassigned.
        """
        for i in range(start_idx, min(start_idx + lookahead, len(lines))):
            line = lines[i].strip()

            # If we hit a label or control flow, be conservative
            if line.startswith('LABEL ') or line.startswith('JUMP'):
                return False

            # If dst is stored to again, it's safe (old alias is replaced)
            if (
                line.startswith(f'STORE {dst_var}')
                or line.startswith('COPY_LOCAL_REF ')
                and dst_var in line
            ):
                return True

            # If dst is loaded for an operation, we need to copy (not safe for ref)
            if line.startswith(f'LOAD {dst_var}'):
                # Check if next instruction modifies it
                if i + 1 < len(lines):
                    next_line = lines[i+1].strip()
                    # If it's used in arithmetic, we need deep copy
                    if any(next_line.startswith(op) for op in ['ADD', 'SUB', 'MUL', 'DIV', 'MOD']):
                        return False

            # If src is modified, the alias becomes invalid
            if line.startswith(f'STORE {src_var}'):
                return False

        # Default: be conservative
        return False

    def remove_redundant_loads(self, lines: List[str]) -> List[str]:
        """Remove STORE followed by LOAD of same variable when safe"""
        result = []
        i = 0
        while i < len(lines):
            line = lines[i].strip()

            # Check for STORE N followed by LOAD N
            if i + 1 < len(lines) and line.startswith('STORE '):
                next_line = lines[i + 1].strip()
                if next_line.startswith('LOAD '):
                    var1 = line.split()[1]
                    var2 = next_line.split()[1]
                    if var1 == var2:
                        # Only optimize if the LOAD is immediately consumed by an operation
                        # Check what follows the LOAD
                        if i + 2 < len(lines):
                            following_line = lines[i + 2].strip()
                            # Safe to optimize if followed by an arithmetic/comparison operation
                            safe_ops = ['ADD_', 'SUB_', 'MUL_', 'DIV_', 'MOD_', 'CMP_', 'AND', 'OR', 'NOT']
                            is_safe = any(following_line.startswith(op) for op in safe_ops)

                            if is_safe:
                                # Keep STORE, add DUP before it, skip LOAD
                                result.append(lines[i].replace('STORE', 'DUP\n  STORE'))
                                i += 2
                                continue
                        # If not safe or no following line, keep both STORE and LOAD

            result.append(lines[i])
            i += 1

        return result

    def combine_patterns(self, lines: List[str]) -> List[str]:
        """Combine common instruction patterns into optimized forms"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: LOAD N, CONST_I64 1, ADD_I64, STORE N -> INC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'ADD_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    # Replace with optimized INC instruction
                    result.append(f"{indent}INC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: LOAD N, CONST_I64 1, SUB_I64, STORE N -> DEC_LOCAL N
            if (i + 3 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip() == 'CONST_I64 1' and
                lines[i+2].strip() == 'SUB_I64' and
                lines[i+3].strip().startswith('STORE ')):

                load_var = line.split()[1]
                store_var = lines[i+3].strip().split()[1]

                if load_var == store_var:
                    result.append(f"{indent}DEC_LOCAL {load_var}")
                    i += 4
                    continue

            # Pattern: Multiple ADD_CONST_I64 -> ADD_CONST_I64 with multiple values
            if line.startswith('ADD_CONST_I64 '):
                constants = [line.split('ADD_CONST_I64 ', 1)[1]]
                j = i + 1

                while j < len(lines) and lines[j].strip().startswith('ADD_CONST_I64 '):
                    constants.append(lines[j].strip().split('ADD_CONST_I64 ', 1)[1])
                    j += 1

                if len(constants) >= 2:
                    result.append(f"{indent}ADD_CONST_I64 {' '.join(constants)}")
                    i = j
                    continue

            # Pattern: CONST_I64 N, ADD_I64 -> ADD_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'ADD_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}ADD_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, SUB_I64 -> SUB_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'SUB_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}SUB_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, MUL_I64 -> MUL_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'MUL_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}MUL_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, DIV_I64 -> DIV_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'DIV_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}DIV_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, MOD_I64 -> MOD_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'MOD_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}MOD_CONST_I64 {const_val}")
                i += 2
                continue

            # Float constant operations
            # Pattern: CONST_F64 N, ADD_F64 -> ADD_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'ADD_F64'):

                const_val = line.split()[1]
                result.append(f"{indent}ADD_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, SUB_F64 -> SUB_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'SUB_F64'):

                const_val = line.split()[1]
                result.append(f"{indent}SUB_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, MUL_F64 -> MUL_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'MUL_F64'):

                const_val = line.split()[1]
                result.append(f"{indent}MUL_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, DIV_F64 -> DIV_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'DIV_F64'):

                const_val = line.split()[1]
                result.append(f"{indent}DIV_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_BOOL N, AND -> AND_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip() == 'AND'):

                const_val = line.split()[1]
                result.append(f"{indent}AND_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_BOOL N, OR -> OR_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip() == 'OR'):

                const_val = line.split()[1]
                result.append(f"{indent}OR_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, AND_I64 -> AND_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'AND_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}AND_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, OR_I64 -> OR_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'OR_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}OR_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, XOR_I64 -> XOR_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'XOR_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}XOR_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, SHL_I64 -> SHL_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'SHL_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}SHL_CONST_I64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, SHR_I64 -> SHR_CONST_I64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'SHR_I64'):

                const_val = line.split()[1]
                result.append(f"{indent}SHR_CONST_I64 {const_val}")
                i += 2
                continue

            # Comparison constant optimizations
            # Pattern: CONST_I64 N, CMP_LT -> CMP_LT_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_LT'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_LT_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, CMP_GT -> CMP_GT_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_GT'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_GT_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, CMP_LE -> CMP_LE_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_LE'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_LE_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, CMP_GE -> CMP_GE_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_GE'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_GE_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, CMP_EQ -> CMP_EQ_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_EQ'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_EQ_CONST {const_val}")
                i += 2
                continue

            # Pattern: CONST_I64 N, CMP_NE -> CMP_NE_CONST N
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip() == 'CMP_NE'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_NE_CONST {const_val}")
                i += 2
                continue

            # Float comparison constant optimizations
            # Pattern: CONST_F64 N, CMP_LT -> CMP_LT_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'CMP_LT'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_LT_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, CMP_GT -> CMP_GT_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'CMP_GT'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_GT_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, CMP_LE -> CMP_LE_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'CMP_LE'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_LE_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: CONST_F64 N, CMP_GE -> CMP_GE_CONST_F64 N
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip() == 'CMP_GE'):

                const_val = line.split()[1]
                result.append(f"{indent}CMP_GE_CONST_F64 {const_val}")
                i += 2
                continue

            # Pattern: LOAD N, LOAD N -> LOAD N, DUP
            if (i + 1 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ')):

                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]

                if var1 == var2:
                    result.extend((lines[i], f"{indent}DUP"))
                    i += 2
                    continue

            # NEW: Pattern: LOAD a, LOAD b, ADD_I64 -> LOAD2_ADD_I64 a b (before multi-LOAD combining!)
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'ADD_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SUB_I64 -> LOAD2_SUB_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SUB_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, MUL_I64 -> LOAD2_MUL_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'MUL_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LT -> LOAD2_CMP_LT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GT -> LOAD2_CMP_GT a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GT'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_LE -> LOAD2_CMP_LE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_LE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_GE -> LOAD2_CMP_GE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_GE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_EQ -> LOAD2_CMP_EQ a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_EQ'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, CMP_NE -> LOAD2_CMP_NE a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'CMP_NE'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, DIV_I64 -> LOAD2_DIV_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'DIV_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_DIV_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, MOD_I64 -> LOAD2_MOD_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'MOD_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_MOD_I64 {var1} {var2}")
                i += 3
                continue

            # Float LOAD2 operations
            # NEW: Pattern: LOAD a, LOAD b, ADD_F64 -> LOAD2_ADD_F64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'ADD_F64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_ADD_F64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SUB_F64 -> LOAD2_SUB_F64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SUB_F64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_SUB_F64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, MUL_F64 -> LOAD2_MUL_F64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'MUL_F64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_MUL_F64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, DIV_F64 -> LOAD2_DIV_F64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'DIV_F64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_DIV_F64 {var1} {var2}")
                i += 3
                continue

            # Bitwise LOAD2 operations
            # NEW: Pattern: LOAD a, LOAD b, AND_I64 -> LOAD2_AND_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'AND_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_AND_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, OR_I64 -> LOAD2_OR_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'OR_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_OR_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, XOR_I64 -> LOAD2_XOR_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'XOR_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_XOR_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SHL_I64 -> LOAD2_SHL_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SHL_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_SHL_I64 {var1} {var2}")
                i += 3
                continue

            # NEW: Pattern: LOAD a, LOAD b, SHR_I64 -> LOAD2_SHR_I64 a b
            if (i + 2 < len(lines) and
                line.startswith('LOAD ') and
                lines[i+1].strip().startswith('LOAD ') and
                lines[i+2].strip() == 'SHR_I64'):
                var1 = line.split()[1]
                var2 = lines[i+1].strip().split()[1]
                result.append(f"{indent}LOAD2_SHR_I64 {var1} {var2}")
                i += 3
                continue

            # Pattern: Multiple consecutive LOAD instructions -> LOAD with multiple args
            # Look for sequences of LOAD instructions
            # BUT: Don't combine 2 LOADs if followed by fusable operation (let LOAD2+OP handle it)
            if line.startswith('LOAD '):
                loads = [line.strip().split()[1]]
                j = i + 1
                while j < len(lines) and lines[j].strip().startswith('LOAD '):
                    loads.append(lines[j].strip().split()[1])
                    j += 1

                # Check if this is exactly 2 LOADs followed by a fusable operation
                fusable_ops = {'ADD_I64', 'SUB_I64', 'MUL_I64', 'CMP_LT', 'CMP_GT', 'CMP_LE', 'CMP_GE', 'CMP_EQ', 'CMP_NE'}
                if len(loads) == 2 and j < len(lines) and lines[j].strip() in fusable_ops:
                    # Don't combine - let it fall through to LOAD2+OP patterns below
                    pass
                elif len(loads) >= 2:
                    # Combine multiple LOADs
                    result.append(f"{indent}LOAD {' '.join(loads)}")
                    i = j
                    continue

            # Pattern: LOAD a b (multi-LOAD), OP -> LOAD2_OP a b (must run AFTER multi-LOAD combining)
            # Check if this is a multi-arg LOAD followed by fused-capable operation
            stripped_line = line.strip()
            if stripped_line.startswith('LOAD ') and len(stripped_line.split()) == 3:  # LOAD with exactly 2 args
                parts = stripped_line.split()
                var1, var2 = parts[1], parts[2]

                # Check what operation follows
                if i + 1 < len(lines):
                    next_op = lines[i+1].strip()

                    # Integer arithmetic
                    if next_op == 'ADD_I64':
                        result.append(f"{indent}LOAD2_ADD_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SUB_I64':
                        result.append(f"{indent}LOAD2_SUB_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'MUL_I64':
                        result.append(f"{indent}LOAD2_MUL_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'DIV_I64':
                        result.append(f"{indent}LOAD2_DIV_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'MOD_I64':
                        result.append(f"{indent}LOAD2_MOD_I64 {var1} {var2}")
                        i += 2
                        continue

                    # Comparisons
                    elif next_op == 'CMP_LT':
                        result.append(f"{indent}LOAD2_CMP_LT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GT':
                        result.append(f"{indent}LOAD2_CMP_GT {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_LE':
                        result.append(f"{indent}LOAD2_CMP_LE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_GE':
                        result.append(f"{indent}LOAD2_CMP_GE {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_EQ':
                        result.append(f"{indent}LOAD2_CMP_EQ {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'CMP_NE':
                        result.append(f"{indent}LOAD2_CMP_NE {var1} {var2}")
                        i += 2
                        continue

                    # Float arithmetic
                    elif next_op == 'ADD_F64':
                        result.append(f"{indent}LOAD2_ADD_F64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SUB_F64':
                        result.append(f"{indent}LOAD2_SUB_F64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'MUL_F64':
                        result.append(f"{indent}LOAD2_MUL_F64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'DIV_F64':
                        result.append(f"{indent}LOAD2_DIV_F64 {var1} {var2}")
                        i += 2
                        continue

                    # Bitwise operations
                    elif next_op == 'AND_I64':
                        result.append(f"{indent}LOAD2_AND_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'OR_I64':
                        result.append(f"{indent}LOAD2_OR_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'XOR_I64':
                        result.append(f"{indent}LOAD2_XOR_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SHL_I64':
                        result.append(f"{indent}LOAD2_SHL_I64 {var1} {var2}")
                        i += 2
                        continue
                    elif next_op == 'SHR_I64':
                        result.append(f"{indent}LOAD2_SHR_I64 {var1} {var2}")
                        i += 2
                        continue

            # Pattern: Alternating LOAD/STORE sequence -> FUSED_LOAD_STORE
            # Must start with LOAD and alternate, can end with either LOAD or STORE
            if line.startswith('LOAD '):
                # Collect alternating LOAD/STORE operations
                ops = []
                j = i
                expect_load = True
                
                while j < len(lines):
                    curr_line = lines[j].strip()
                    if expect_load and curr_line.startswith('LOAD '):
                        ops.append(curr_line.split()[1])
                        expect_load = False
                        j += 1
                    elif not expect_load and curr_line.startswith('STORE '):
                        ops.append(curr_line.split()[1])
                        expect_load = True
                        j += 1
                    else:
                        break

                # Merge if we have at least one complete pair (2 ops) or more
                if len(ops) >= 2:
                    indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '
                    args = ' '.join(ops)
                    result.append(f"{indent}FUSED_LOAD_STORE {args}")
                    i = j
                    continue

            # Pattern: STORE N, LOAD N (same var) -> DUP, STORE N
            if (line.startswith('STORE ') and i + 1 < len(lines) and 
                lines[i+1].strip().startswith('LOAD ')):
                store_var = line.split()[1]
                load_var = lines[i+1].strip().split()[1]
                if store_var == load_var:
                    indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '
                    result.append(f"{indent}DUP")
                    result.append(lines[i])
                    i += 2
                    continue

            # Pattern: Alternating STORE/LOAD sequence -> FUSED_STORE_LOAD
            # Must start with STORE and alternate, can end with either STORE or LOAD
            # Skip if first pair is STORE N, LOAD N (same var) - handled above
            if line.startswith('STORE '):
                # Collect alternating STORE/LOAD operations
                ops = []
                j = i
                expect_store = True
                
                while j < len(lines):
                    curr_line = lines[j].strip()
                    if expect_store and curr_line.startswith('STORE '):
                        ops.append(curr_line.split()[1])
                        expect_store = False
                        j += 1
                    elif not expect_store and curr_line.startswith('LOAD '):
                        ops.append(curr_line.split()[1])
                        expect_store = True
                        j += 1
                    else:
                        break

                # Only merge if:
                # 1. We have at least 2 pairs (4 ops) OR
                # 2. We have 1 pair but STORE and LOAD are different vars OR
                # 3. We have more than 2 ops total
                should_fuse = len(ops) >= 4 or (len(ops) >= 2 and ops[0] != ops[1]) or len(ops) > 2
                
                if should_fuse:
                    indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '
                    args = ' '.join(ops)
                    result.append(f"{indent}FUSED_STORE_LOAD {args}")
                    i = j
                    continue

            # Pattern: LOAD a, LOAD b, SWAP -> LOAD b, LOAD a (reduce SWAP usage)
            # This is actually worse, so skip it

            # Pattern: DUP, LOAD N -> LOAD N, DUP (may enable other optimizations)
            # Skip for now as it's context-dependent

            result.append(lines[i])
            i += 1

        return result

    def remove_dead_stores(self, lines: List[str]) -> List[str]:
        """Remove stores to variables that are never loaded (basic pass)"""
        # Build label map first
        label_map = {}
        for i, line in enumerate(lines):
            if line.strip().startswith('LABEL '):
                label_name = line.strip().split()[1]
                label_map[label_name] = i

        # For now, just return as-is (dead store elimination is complex with control flow)
        # TODO: Implement proper liveness analysis
        return lines

    def optimize_negation(self, lines: List[str]) -> List[str]:
        """Optimize negation patterns to use NEG instruction"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: CONST_I64 -1, MUL_I64 -> NEG
            if (i + 1 < len(lines) and
                line.startswith('CONST_I64 -1') and
                lines[i+1].strip() == 'MUL_I64'):
                result.append(f"{indent}NEG")
                i += 2
                continue

            # Pattern: CONST_I64 0, SWAP, SUB_I64 -> NEG
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 0') and
                lines[i+1].strip() == 'SWAP' and
                lines[i+2].strip() == 'SUB_I64'):
                result.append(f"{indent}NEG")
                i += 3
                continue

            # Pattern: CONST_F64 -1, MUL_F64 -> NEG
            if (i + 1 < len(lines) and
                line.startswith('CONST_F64 -1') and
                lines[i+1].strip() == 'MUL_F64'):
                result.append(f"{indent}NEG")
                i += 2
                continue

            result.append(lines[i])
            i += 1

        return result

    def optimize_jumps(self, lines: List[str]) -> List[str]:
        """Optimize jump chains and remove redundant jumps"""
        # Build label map
        label_map = {}
        for i, line in enumerate(lines):
            stripped = line.strip()
            if stripped.startswith('LABEL '):
                label_name = stripped.split()[1]
                label_map[label_name] = i

        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: JUMP label immediately followed by LABEL label -> just LABEL
            if (i + 1 < len(lines) and
                line.startswith('JUMP ') and
                lines[i+1].strip().startswith('LABEL ')):

                jump_target = line.split()[1]
                label_name = lines[i+1].strip().split()[1]

                if jump_target == label_name:
                    # Skip the JUMP, keep the LABEL
                    i += 1
                    continue

            # Pattern: JUMP_IF_FALSE/TRUE followed immediately by unconditional JUMP
            # Can sometimes be optimized to invert the condition
            # This is complex and context-dependent, skip for now

            result.append(lines[i])
            i += 1

        return result

    def algebraic_simplifications(self, lines: List[str]) -> List[str]:
        """Apply algebraic simplifications (identity operations)"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: CONST_I64 0, ADD_I64 -> POP (x + 0 = x, but value already on stack)
            # Actually, x + 0 = x means we can just remove both instructions
            if (i + 1 < len(lines) and
                line == 'CONST_I64 0' and
                lines[i+1].strip() == 'ADD_I64'):
                # Skip both - result is already on stack
                i += 2
                continue

            # Pattern: CONST_I64 0, SUB_I64 -> nothing (x - 0 = x)
            if (i + 1 < len(lines) and
                line == 'CONST_I64 0' and
                lines[i+1].strip() == 'SUB_I64'):
                i += 2
                continue

            # Pattern: CONST_I64 1, MUL_I64 -> nothing (x * 1 = x)
            if (i + 1 < len(lines) and
                line == 'CONST_I64 1' and
                lines[i+1].strip() == 'MUL_I64'):
                i += 2
                continue

            # Pattern: CONST_I64 0, MUL_I64 -> POP, CONST_I64 0 (x * 0 = 0)
            if (i + 1 < len(lines) and
                line == 'CONST_I64 0' and
                lines[i+1].strip() == 'MUL_I64'):
                result.append(f"{indent}POP")
                result.append(f"{indent}CONST_I64 0")
                i += 2
                continue

            # Pattern: CONST_I64 1, DIV_I64 -> nothing (x / 1 = x)
            if (i + 1 < len(lines) and
                line == 'CONST_I64 1' and
                lines[i+1].strip() == 'DIV_I64'):
                i += 2
                continue

            # Float equivalents
            # Pattern: CONST_F64 0.0, ADD_F64 -> nothing (x + 0.0 = x)
            if (i + 1 < len(lines) and
                line in ['CONST_F64 0.0', 'CONST_F64 0'] and
                lines[i+1].strip() == 'ADD_F64'):
                i += 2
                continue

            # Pattern: CONST_F64 0.0, SUB_F64 -> nothing (x - 0.0 = x)
            if (i + 1 < len(lines) and
                line in ['CONST_F64 0.0', 'CONST_F64 0'] and
                lines[i+1].strip() == 'SUB_F64'):
                i += 2
                continue

            # Pattern: CONST_F64 1.0, MUL_F64 -> nothing (x * 1.0 = x)
            if (i + 1 < len(lines) and
                line in ['CONST_F64 1.0', 'CONST_F64 1'] and
                lines[i+1].strip() == 'MUL_F64'):
                i += 2
                continue

            # Pattern: CONST_F64 0.0, MUL_F64 -> POP, CONST_F64 0.0 (x * 0.0 = 0.0)
            if (i + 1 < len(lines) and
                line in ['CONST_F64 0.0', 'CONST_F64 0'] and
                lines[i+1].strip() == 'MUL_F64'):
                result.append(f"{indent}POP")
                result.append(f"{indent}CONST_F64 0.0")
                i += 2
                continue

            # Pattern: CONST_F64 1.0, DIV_F64 -> nothing (x / 1.0 = x)
            if (i + 1 < len(lines) and
                line in ['CONST_F64 1.0', 'CONST_F64 1'] and
                lines[i+1].strip() == 'DIV_F64'):
                i += 2
                continue

            # Logical identities
            # Pattern: CONST_BOOL true, AND -> nothing (x AND true = x)
            if (i + 1 < len(lines) and
                line == 'CONST_BOOL true' and
                lines[i+1].strip() == 'AND'):
                i += 2
                continue

            # Pattern: CONST_BOOL false, AND -> POP, CONST_BOOL false (x AND false = false)
            if (i + 1 < len(lines) and
                line == 'CONST_BOOL false' and
                lines[i+1].strip() == 'AND'):
                result.append(f"{indent}POP")
                result.append(f"{indent}CONST_BOOL false")
                i += 2
                continue

            # Pattern: CONST_BOOL false, OR -> nothing (x OR false = x)
            if (i + 1 < len(lines) and
                line == 'CONST_BOOL false' and
                lines[i+1].strip() == 'OR'):
                i += 2
                continue

            # Pattern: CONST_BOOL true, OR -> POP, CONST_BOOL true (x OR true = true)
            if (i + 1 < len(lines) and
                line == 'CONST_BOOL true' and
                lines[i+1].strip() == 'OR'):
                result.append(f"{indent}POP")
                result.append(f"{indent}CONST_BOOL true")
                i += 2
                continue

            # Pattern: NOT, NOT -> nothing (double negation cancels out)
            if (i + 1 < len(lines) and
                line == 'NOT' and
                lines[i+1].strip() == 'NOT'):
                i += 2
                continue

            result.append(lines[i])
            i += 1

        return result

    def optimize_comparison_inversions(self, lines: List[str]) -> List[str]:
        """Optimize comparison inversions (NOT after comparison -> inverted comparison)"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: CMP_LT, NOT -> CMP_GE
            if (i + 1 < len(lines) and
                line == 'CMP_LT' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_GE")
                i += 2
                continue

            # Pattern: CMP_GT, NOT -> CMP_LE
            if (i + 1 < len(lines) and
                line == 'CMP_GT' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_LE")
                i += 2
                continue

            # Pattern: CMP_LE, NOT -> CMP_GT
            if (i + 1 < len(lines) and
                line == 'CMP_LE' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_GT")
                i += 2
                continue

            # Pattern: CMP_GE, NOT -> CMP_LT
            if (i + 1 < len(lines) and
                line == 'CMP_GE' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_LT")
                i += 2
                continue

            # Pattern: CMP_EQ, NOT -> CMP_NE
            if (i + 1 < len(lines) and
                line == 'CMP_EQ' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_NE")
                i += 2
                continue

            # Pattern: CMP_NE, NOT -> CMP_EQ
            if (i + 1 < len(lines) and
                line == 'CMP_NE' and
                lines[i+1].strip() == 'NOT'):
                result.append(f"{indent}CMP_EQ")
                i += 2
                continue

            # Also handle LOAD2_CMP variants
            # Pattern: LOAD2_CMP_LT, NOT -> LOAD2_CMP_GE
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_LT ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_LT ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_GE {args}")
                i += 2
                continue

            # Pattern: LOAD2_CMP_GT, NOT -> LOAD2_CMP_LE
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_GT ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_GT ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_LE {args}")
                i += 2
                continue

            # Pattern: LOAD2_CMP_LE, NOT -> LOAD2_CMP_GT
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_LE ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_LE ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_GT {args}")
                i += 2
                continue

            # Pattern: LOAD2_CMP_GE, NOT -> LOAD2_CMP_LT
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_GE ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_GE ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_LT {args}")
                i += 2
                continue

            # Pattern: LOAD2_CMP_EQ, NOT -> LOAD2_CMP_NE
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_EQ ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_EQ ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_NE {args}")
                i += 2
                continue

            # Pattern: LOAD2_CMP_NE, NOT -> LOAD2_CMP_EQ
            if (i + 1 < len(lines) and
                line.startswith('LOAD2_CMP_NE ') and
                lines[i+1].strip() == 'NOT'):
                args = line.split('LOAD2_CMP_NE ', 1)[1]
                result.append(f"{indent}LOAD2_CMP_EQ {args}")
                i += 2
                continue

            result.append(lines[i])
            i += 1

        return result

    def constant_folding(self, lines: List[str]) -> List[str]:
        """Fold constant expressions at compile time"""
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if line else '  '

            # Pattern: CONST_I64 A, CONST_I64 B, ADD_I64 -> CONST_I64 (A+B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'ADD_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a + b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, SUB_I64 -> CONST_I64 (A-B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'SUB_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a - b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, MUL_I64 -> CONST_I64 (A*B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'MUL_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a * b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, DIV_I64 -> CONST_I64 (A//B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'DIV_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    if b != 0:
                        result.append(f"{indent}CONST_I64 {a // b}")
                        i += 3
                        continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, MOD_I64 -> CONST_I64 (A%B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'MOD_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    if b != 0:
                        result.append(f"{indent}CONST_I64 {a % b}")
                        i += 3
                        continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_F64 A, CONST_F64 B, ADD_F64 -> CONST_F64 (A+B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip().startswith('CONST_F64 ') and
                lines[i+2].strip() == 'ADD_F64'):
                try:
                    a = float(line.split()[1])
                    b = float(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_F64 {a + b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_F64 A, CONST_F64 B, SUB_F64 -> CONST_F64 (A-B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip().startswith('CONST_F64 ') and
                lines[i+2].strip() == 'SUB_F64'):
                try:
                    a = float(line.split()[1])
                    b = float(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_F64 {a - b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_F64 A, CONST_F64 B, MUL_F64 -> CONST_F64 (A*B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip().startswith('CONST_F64 ') and
                lines[i+2].strip() == 'MUL_F64'):
                try:
                    a = float(line.split()[1])
                    b = float(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_F64 {a * b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_F64 A, CONST_F64 B, DIV_F64 -> CONST_F64 (A/B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_F64 ') and
                lines[i+1].strip().startswith('CONST_F64 ') and
                lines[i+2].strip() == 'DIV_F64'):
                try:
                    a = float(line.split()[1])
                    b = float(lines[i+1].strip().split()[1])
                    if b != 0.0:
                        result.append(f"{indent}CONST_F64 {a / b}")
                        i += 3
                        continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_BOOL A, CONST_BOOL B, AND -> CONST_BOOL (A and B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip().startswith('CONST_BOOL ') and
                lines[i+2].strip() == 'AND'):
                try:
                    a = line.split()[1] == 'true'
                    b = lines[i+1].strip().split()[1] == 'true'
                    result.append(f"{indent}CONST_BOOL {'true' if (a and b) else 'false'}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_BOOL A, CONST_BOOL B, OR -> CONST_BOOL (A or B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip().startswith('CONST_BOOL ') and
                lines[i+2].strip() == 'OR'):
                try:
                    a = line.split()[1] == 'true'
                    b = lines[i+1].strip().split()[1] == 'true'
                    result.append(f"{indent}CONST_BOOL {'true' if (a or b) else 'false'}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_BOOL A, NOT -> CONST_BOOL (!A)
            if (i + 1 < len(lines) and
                line.startswith('CONST_BOOL ') and
                lines[i+1].strip() == 'NOT'):
                try:
                    a = line.split()[1] == 'true'
                    result.append(f"{indent}CONST_BOOL {'false' if a else 'true'}")
                    i += 2
                    continue
                except (ValueError, IndexError):
                    pass

            # Bitwise constant folding
            # Pattern: CONST_I64 A, CONST_I64 B, AND_I64 -> CONST_I64 (A & B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'AND_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a & b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, OR_I64 -> CONST_I64 (A | B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'OR_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a | b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, XOR_I64 -> CONST_I64 (A ^ B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'XOR_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    result.append(f"{indent}CONST_I64 {a ^ b}")
                    i += 3
                    continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, SHL_I64 -> CONST_I64 (A << B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'SHL_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    if 0 <= b < 64:  # Prevent shift overflow
                        result.append(f"{indent}CONST_I64 {a << b}")
                        i += 3
                        continue
                except (ValueError, IndexError):
                    pass

            # Pattern: CONST_I64 A, CONST_I64 B, SHR_I64 -> CONST_I64 (A >> B)
            if (i + 2 < len(lines) and
                line.startswith('CONST_I64 ') and
                lines[i+1].strip().startswith('CONST_I64 ') and
                lines[i+2].strip() == 'SHR_I64'):
                try:
                    a = int(line.split()[1])
                    b = int(lines[i+1].strip().split()[1])
                    if 0 <= b < 64:  # Prevent shift overflow
                        result.append(f"{indent}CONST_I64 {a >> b}")
                        i += 3
                        continue
                except (ValueError, IndexError):
                    pass

            result.append(lines[i])
            i += 1

        return result

    def optimize_switch_cases(self, lines: List[str]) -> List[str]:
        """Optimize switch-case patterns by:
        1. Detecting repeated LOAD var, CMP_EQ_CONST, JUMP_IF_TRUE sequences
        2. Consolidating redundant LOADs (cache the switch value)
        3. Using SWITCH_JUMP_TABLE for dense integer switches
        4. Using DUP to cache switch value for sparse switches
        """
        result: List[str] = []
        i = 0

        while i < len(lines):
            # Try to detect switch pattern: repeated sequences of:
            # LOAD <var>
            # CMP_EQ_CONST <value>
            # JUMP_IF_TRUE <label>

            if i + 2 < len(lines):
                line = lines[i].strip()

                # Look for first case comparison
                if (line.startswith('LOAD ') and
                    lines[i+1].strip().startswith('CMP_EQ_CONST ') and
                    lines[i+2].strip().startswith('JUMP_IF_TRUE ')):

                    switch_var = line.split()[1]
                    cases: List[Tuple[int, str]] = []  # (value, label)
                    j = i

                    # Collect all consecutive case comparisons on the same variable
                    while (j + 2 < len(lines) and
                           lines[j].strip() == f'LOAD {switch_var}' and
                           lines[j+1].strip().startswith('CMP_EQ_CONST ') and
                           lines[j+2].strip().startswith('JUMP_IF_TRUE ')):
                        try:
                            const_val = int(lines[j+1].strip().split()[1])
                            label = lines[j+2].strip().split()[1]
                            cases.append((const_val, label))
                        except ValueError:
                            break
                        j += 3

                    # Check for default case (JUMP without condition)
                    default_label = None
                    if j < len(lines) and lines[j].strip().startswith('JUMP '):
                        default_label = lines[j].strip().split()[1]
                        j += 1

                    # If we found 3+ cases, try to optimize
                    if len(cases) >= 3:
                        indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '

                        # Check if this is a dense switch (consecutive integers)
                        sorted_cases = sorted(cases, key=lambda x: x[0])
                        min_val = sorted_cases[0][0]
                        max_val = sorted_cases[-1][0]
                        is_dense = (max_val - min_val == len(sorted_cases) - 1)

                        # For dense switches with 5+ cases, use SWITCH_JUMP_TABLE
                        if is_dense and len(sorted_cases) >= 5 and default_label:
                            # Emit SWITCH_JUMP_TABLE instruction
                            result.append(f"{indent}LOAD {switch_var}")
                            
                            # Build case labels array in order from min to max
                            case_labels = []
                            for val in range(min_val, max_val + 1):
                                # Find the label for this value
                                for case_val, case_label in sorted_cases:
                                    if case_val == val:
                                        case_labels.append(case_label)
                                        break
                            
                            # Format: SWITCH_JUMP_TABLE min max label1 label2 ... labelN default
                            labels_str = ' '.join(case_labels)
                            result.append(f"{indent}SWITCH_JUMP_TABLE {min_val} {max_val} {labels_str} {default_label}")
                        else:
                            # For non-dense or small switches, use DUP optimization
                            result.append(f"{indent}LOAD {switch_var}")

                            for idx, (const_val, label) in enumerate(cases):
                                # DUP unless this is the last case and there's no default
                                if idx < len(cases) - 1 or default_label:
                                    result.append(f"{indent}DUP")
                                result.append(f"{indent}CMP_EQ_CONST {const_val}")
                                result.append(f"{indent}JUMP_IF_TRUE {label}")

                            # Add default jump if present
                            if default_label:
                                result.append(f"{indent}POP")
                                result.append(f"{indent}JUMP {default_label}")

                        i = j
                        continue

            result.append(lines[i])
            i += 1

        return result

    def fuse_similar_case_bodies(self, lines: List[str]) -> List[str]:
        """Detect and optimize repetitive case bodies that follow the same pattern.
        
        Example pattern to detect:
          LABEL case_15:
            LOAD 1
            ADD_CONST_I64 2
            STORE 1
            JUMP switch_end3
          LABEL case_26:
            LOAD 1
            ADD_CONST_I64 3
            STORE 1
            JUMP switch_end3
        
        All cases do: var += constant, then jump to same end label.
        We can replace this with a computed offset based on the case number.
        """
        result: List[str] = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()

            # Look for consecutive case labels with simple LOAD/ADD_CONST/STORE/JUMP pattern
            if line.startswith('LABEL case_'):
                # Collect consecutive similar case bodies
                cases = []
                j = i
                
                while j < len(lines) and lines[j].strip().startswith('LABEL case_'):
                    label = lines[j].strip()
                    
                    # Extract case number from label (e.g., "case_15" -> 1, "case_26" -> 2)
                    try:
                        label_suffix = label.split('case_')[1].rstrip(':')
                        # Get the first digit as the case value
                        case_num = int(label_suffix[0]) if label_suffix[0].isdigit() else None
                        if case_num is None:
                            break
                    except (ValueError, IndexError):
                        break
                    
                    # Check if next lines match pattern: LOAD var, ADD_CONST_I64 N, STORE var, JUMP end
                    if j + 4 < len(lines):
                        body_line1 = lines[j + 1].strip()
                        body_line2 = lines[j + 2].strip()
                        body_line3 = lines[j + 3].strip()
                        body_line4 = lines[j + 4].strip()
                        
                        if (body_line1.startswith('LOAD ') and
                            body_line2.startswith('ADD_CONST_I64 ') and
                            body_line3.startswith('STORE ') and
                            body_line4.startswith('JUMP ') and not body_line4.startswith('JUMP_IF')):
                            
                            var = body_line1.split()[1]
                            add_val = int(body_line2.split()[1])
                            store_var = body_line3.split()[1]
                            end_label = body_line4.split()[1]
                            
                            if var == store_var:
                                cases.append({
                                    'case_num': case_num,
                                    'label': label,
                                    'var': var,
                                    'add_val': add_val,
                                    'end_label': end_label
                                })
                                j += 5
                                continue
                    
                    break
                
                # If we found 3+ repetitive cases with the same pattern, optimize
                if len(cases) >= 3:
                    # Check if all cases use same variable and end label
                    first_var = cases[0]['var']
                    first_end = cases[0]['end_label']
                    
                    if all(c['var'] == first_var and c['end_label'] == first_end for c in cases):
                        # Check if add values follow a pattern: case N adds N+1
                        pattern_match = all(c['add_val'] == c['case_num'] + 1 for c in cases)
                        
                        if pattern_match:
                            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                            
                            # Try to find the switch value variable by looking backwards
                            switch_var = None
                            for k in range(i-1, max(0, i-20), -1):
                                prev = lines[k].strip()
                                if prev.startswith('SWITCH_JUMP_TABLE '):
                                    # Switch value is on the stack (from previous LOAD)
                                    for m in range(k-1, max(0, k-5), -1):
                                        if lines[m].strip().startswith('LOAD '):
                                            switch_var = lines[m].strip().split()[1]
                                            break
                                    break
                            
                            if switch_var:
                                indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                                
                                # Create a single label for the optimized code
                                optimized_label = f"switch_opt_{self.label_counter}"
                                self.label_counter += 1
                                
                                # Emit the optimized label and code
                                result.append(f"{indent}LABEL {optimized_label}")
                                result.append(f"{indent}LOAD {switch_var}")
                                result.append(f"{indent}ADD_CONST_I64 1")
                                result.append(f"{indent}LOAD {first_var}")
                                result.append(f"{indent}ADD_I64")
                                result.append(f"{indent}STORE {first_var}")
                                result.append(f"{indent}JUMP {first_end}")
                                
                                # Now we need to update the SWITCH_JUMP_TABLE to point to this label
                                # Go back and find the SWITCH_JUMP_TABLE instruction
                                for k in range(len(result) - 1, max(0, len(result) - 20), -1):
                                    if result[k].strip().startswith('SWITCH_JUMP_TABLE '):
                                        # Replace all case labels in SWITCH_JUMP_TABLE with optimized_label
                                        parts = result[k].split()
                                        min_val = parts[1]
                                        max_val = parts[2]
                                        # Replace case labels (skip min, max, keep default at end)
                                        num_cases = int(max_val) - int(min_val) + 1
                                        default_label = parts[3 + num_cases]  # Last label is default
                                        
                                        # All cases point to optimized label, except default
                                        new_labels = [optimized_label] * num_cases + [default_label]
                                        result[k] = f"{indent}SWITCH_JUMP_TABLE {min_val} {max_val} {' '.join(new_labels)}"
                                        break
                                
                                # Skip all the case bodies we optimized
                                i = j
                                continue

            result.append(lines[i])
            i += 1

        return result

    def cache_loaded_values(self, lines: List[str]) -> List[str]:
        """Cache loaded values using DUP to avoid redundant LOADs.
        
        This optimization is conservative and avoids DUP when there are
        type-converting or stack-consuming operations between the LOADs.
        """
        result: List[str] = []
        i = 0
        
        # Instructions that consume stack and may change types - not safe for DUP optimization
        unsafe_for_dup = {
            'BUILTIN_STR', 'TO_STR', 'TO_INT', 'TO_FLOAT', 'TO_BOOL',
            'BUILTIN_LEN', 'LIST_POP', 'LIST_GET', 'STRUCT_GET',
            'ADD_STR', 'ADD_I64', 'ADD_F64', 'SUB_I64', 'SUB_F64',
            'MUL_I64', 'MUL_F64', 'DIV_I64', 'DIV_F64', 'MOD_I64',
            'CMP_EQ', 'CMP_NE', 'CMP_LT', 'CMP_GT', 'CMP_LE', 'CMP_GE',
            'AND', 'OR', 'NOT', 'NEG'
        }
        
        while i < len(lines):
            line = lines[i].strip()
            
            # Only optimize LOAD instructions
            if line.startswith('LOAD '):
                var = line.split()[1]
                found_dup = False
                
                # Look ahead for duplicate LOAD of same variable (max 6 instructions)
                for j in range(i + 1, min(i + 6, len(lines))):
                    next_line = lines[j].strip()
                    
                    # Stop if we hit a control flow change or variable modification
                    if (next_line.startswith('LABEL ') or 
                        next_line.startswith('JUMP') or
                        next_line == f'STORE {var}' or
                        next_line.startswith('STORE_CONST')):
                        break
                    
                    # Stop if we hit an unsafe operation
                    if any(next_line.startswith(unsafe) for unsafe in unsafe_for_dup):
                        break
                    
                    # Found duplicate LOAD - check if it's safe to use DUP
                    if next_line == f'LOAD {var}':
                        # Double-check: ensure no unsafe operations between
                        has_unsafe = False
                        for k in range(i + 1, j):
                            check_line = lines[k].strip()
                            if any(check_line.startswith(unsafe) for unsafe in unsafe_for_dup):
                                has_unsafe = True
                                break
                        
                        if not has_unsafe:
                            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())]
                            result.append(lines[i])  # Original LOAD
                            result.append(f"{indent}DUP")  # Add DUP
                            
                            # Copy instructions between original and duplicate LOAD
                            for k in range(i + 1, j):
                                result.append(lines[k])
                            
                            # Skip the duplicate LOAD and continue from next instruction
                            i = j + 1
                            found_dup = True
                        break
                
                # If no duplicate found or unsafe, just copy the line
                if not found_dup:
                    result.append(lines[i])
                    i += 1
            else:
                # Not a LOAD, just copy it
                result.append(lines[i])
                i += 1
        
        return result

    def eliminate_redundant_continue_labels(self, lines: List[str]) -> List[str]:
        """Eliminate redundant loop continuation labels."""
        jumped_labels = set()
        for line in lines:
            stripped = line.strip()
            if stripped.startswith('JUMP '):
                parts = stripped.split()
                if len(parts) >= 2:
                    jumped_labels.add(parts[1])
            elif stripped.startswith('JUMP_IF_TRUE ') or stripped.startswith('JUMP_IF_FALSE '):
                parts = stripped.split()
                if len(parts) >= 2:
                    jumped_labels.add(parts[1])
        
        result: List[str] = []
        i = 0
        while i < len(lines):
            line = lines[i].strip()
            
            if line.startswith('LABEL for_continue'):
                label_name = line.split()[1]
                
                if label_name not in jumped_labels and i + 2 < len(lines):
                    next1 = lines[i + 1].strip()
                    next2 = lines[i + 2].strip()
                    
                    if next1.startswith('INC_LOCAL ') and next2.startswith('JUMP for_start'):
                        i += 1
                        continue
            
            result.append(lines[i])
            i += 1
        
        return result

    def optimize_list_creation(self, lines: List[str]) -> List[str]:
        """Optimize patterns of LIST_NEW followed by many CONST+LIST_APPEND sequences.
        
        Pattern: LIST_NEW, CONST_I64 v1, LIST_APPEND, CONST_I64 v2, LIST_APPEND, ...
        Replace with: LIST_NEW_I64 count v1 v2 v3 ...
        
        This drastically reduces bytecode size for const functions returning large lists.
        """
        result = []
        i = 0

        while i < len(lines):
            line = lines[i].strip()
            indent = lines[i][:len(lines[i]) - len(lines[i].lstrip())] if lines[i] else '  '

            # Check for LIST_NEW followed by CONST+APPEND pattern
            if line == 'LIST_NEW':
                # Collect all the appends
                values_i64 = []
                values_f64 = []
                values_str = []
                values_bool = []
                value_type = None
                j = i + 1

                # Scan for consecutive CONST + LIST_APPEND pairs
                while j + 1 < len(lines):
                    const_line = lines[j].strip()
                    append_line = lines[j + 1].strip()

                    # Must be CONST followed by LIST_APPEND
                    if not append_line == 'LIST_APPEND':
                        break

                    # Extract constant value based on type
                    if const_line.startswith('CONST_I64 '):
                        val = const_line.split('CONST_I64 ', 1)[1]
                        if value_type is None:
                            value_type = 'i64'
                        if value_type == 'i64':
                            values_i64.append(val)
                        else:
                            break  # Mixed types, stop optimization
                    elif const_line.startswith('CONST_F64 '):
                        val = const_line.split('CONST_F64 ', 1)[1]
                        if value_type is None:
                            value_type = 'f64'
                        if value_type == 'f64':
                            values_f64.append(val)
                        else:
                            break
                    elif const_line.startswith('CONST_STR '):
                        val = const_line.split('CONST_STR ', 1)[1]
                        if value_type is None:
                            value_type = 'str'
                        if value_type == 'str':
                            values_str.append(val)
                        else:
                            break
                    elif const_line.startswith('CONST_BOOL '):
                        val = const_line.split('CONST_BOOL ', 1)[1]
                        if value_type is None:
                            value_type = 'bool'
                        if value_type == 'bool':
                            values_bool.append(val)
                        else:
                            break
                    else:
                        # Not a const, stop
                        break

                    j += 2  # Skip CONST and LIST_APPEND

                # If we found at least 3 appends (to make it worthwhile), use optimized instruction
                if value_type and (len(values_i64) >= 3 or len(values_f64) >= 3 or 
                                   len(values_str) >= 3 or len(values_bool) >= 3):
                    if value_type == 'i64':
                        result.append(f"{indent}LIST_NEW_I64 {len(values_i64)} {' '.join(values_i64)}")
                    elif value_type == 'f64':
                        result.append(f"{indent}LIST_NEW_F64 {len(values_f64)} {' '.join(values_f64)}")
                    elif value_type == 'str':
                        result.append(f"{indent}LIST_NEW_STR {len(values_str)} {' '.join(values_str)}")
                    elif value_type == 'bool':
                        result.append(f"{indent}LIST_NEW_BOOL {len(values_bool)} {' '.join(values_bool)}")
                    i = j  # Skip all the processed lines
                    continue

            result.append(lines[i])
            i += 1
        
        return result

    def optimize_if_else_select(self, lines: List[str]) -> List[str]:
        """
        Optimize if-else patterns where both branches assign to the same variable.
        
        Pattern:
            <condition>
            JUMP_IF_FALSE else_label
            [statements]
            <value_expr>
            STORE var
            JUMP end_label
            LABEL else_label
            [statements]
            <value_expr>
            STORE var
            LABEL end_label
        
        Optimized to:
            <condition>
            <true_value_expr>
            <false_value_expr>
            SELECT
            STORE var
        """
        result: List[str] = []
        i = 0
        
        while i < len(lines):
            line = lines[i].strip()
            
            # Look for JUMP_IF_FALSE
            if line.startswith('JUMP_IF_FALSE '):
                parts = line.split()
                if len(parts) != 2:
                    result.append(lines[i])
                    i += 1
                    continue
                    
                else_label = parts[1]
                jump_idx = i
                
                # Scan if-branch
                j = i + 1
                if_store_idx = -1
                if_store_var = None
                end_label = None
                
                while j < len(lines):
                    curr = lines[j].strip()
                    
                    if curr.startswith('LABEL '):
                        break
                    
                    if curr.startswith('JUMP '):
                        end_label = curr.split()[1] if len(curr.split()) > 1 else None
                        j += 1
                        break
                    
                    if curr.startswith('STORE '):
                        if_store_var = curr.split()[1] if len(curr.split()) > 1 else None
                        if_store_idx = j
                    elif curr.startswith('STORE_CONST_'):
                        parts_sc = curr.split()
                        if len(parts_sc) >= 2:
                            if_store_var = parts_sc[1]
                            if_store_idx = j
                    
                    j += 1
                
                if if_store_idx == -1 or end_label is None:
                    result.append(lines[i])
                    i += 1
                    continue
                
                # Check for else label
                if j >= len(lines) or lines[j].strip() != f"LABEL {else_label}":
                    result.append(lines[i])
                    i += 1
                    continue
                
                # Scan else-branch
                k = j + 1
                else_store_idx = -1
                else_store_var = None
                
                while k < len(lines):
                    curr = lines[k].strip()
                    
                    if curr.startswith('LABEL '):
                        break
                    
                    if curr.startswith('STORE '):
                        else_store_var = curr.split()[1] if len(curr.split()) > 1 else None
                        else_store_idx = k
                    elif curr.startswith('STORE_CONST_'):
                        parts_sc = curr.split()
                        if len(parts_sc) >= 2:
                            else_store_var = parts_sc[1]
                            else_store_idx = k
                    
                    k += 1
                
                if else_store_idx == -1 or if_store_var != else_store_var:
                    result.append(lines[i])
                    i += 1
                    continue
                
                # Check for end label
                if k >= len(lines) or lines[k].strip() != f"LABEL {end_label}":
                    result.append(lines[i])
                    i += 1
                    continue
                
                # Successfully matched! Build optimized code
                indent = lines[if_store_idx][:len(lines[if_store_idx]) - len(lines[if_store_idx].lstrip())]
                
                # Check if there are any statements between jump and store (in if branch)
                # or between else label and store (in else branch)
                # For STORE_CONST, the constant value is part of the STORE so there's no separate CONST
                # but there could still be other instructions before it
                if_expr_count = 0
                for idx in range(jump_idx + 1, if_store_idx):
                    if lines[idx].strip():  # Non-empty line
                        if_expr_count += 1
                
                else_expr_count = 0
                for idx in range(j + 1, else_store_idx):
                    if lines[idx].strip():  # Non-empty line
                        else_expr_count += 1
                
                # If STORE_CONST, we expect 0 expression lines (value is in STORE itself)
                # If regular STORE, we expect exactly 1 expression line (the CONST/LOAD)
                # Anything else means side effects
                if lines[if_store_idx].strip().startswith('STORE_CONST_'):
                    if_has_side_effects = (if_expr_count > 0)
                else:
                    if_has_side_effects = (if_expr_count != 1)
                
                if lines[else_store_idx].strip().startswith('STORE_CONST_'):
                    else_has_side_effects = (else_expr_count > 0)
                else:
                    else_has_side_effects = (else_expr_count != 1)
                
                # Only optimize if no side effects in branches
                if if_has_side_effects or else_has_side_effects:
                    result.append(lines[i])
                    i += 1
                    continue
                
                # Find condition start
                cond_start = jump_idx - 1
                while cond_start >= len(result):
                    prev = lines[cond_start].strip()
                    if (prev.startswith('LABEL ') or prev.startswith('JUMP ') or
                        prev.startswith('.local ') or prev.startswith('STORE ') or
                        prev.startswith('STORE_CONST_')):
                        cond_start += 1
                        break
                    cond_start -= 1
                
                if cond_start < len(result):
                    cond_start = len(result)
                
                # Add condition
                for idx in range(cond_start, jump_idx):
                    result.append(lines[idx])
                
                # Add true value
                # If STORE is STORE_CONST_*, extract the constant and push it
                if lines[if_store_idx].strip().startswith('STORE_CONST_'):
                    # Extract constant from STORE_CONST_I64/F64/etc var val
                    parts = lines[if_store_idx].strip().split()
                    if len(parts) >= 3:
                        const_type = parts[0].replace('STORE_CONST_', 'CONST_')
                        const_val = parts[2]
                        result.append(f"{indent}{const_type} {const_val}")
                else:
                    # Normal case: copy instructions before STORE
                    for idx in range(jump_idx + 1, if_store_idx):
                        result.append(lines[idx])
                
                # Add false value
                # If STORE is STORE_CONST_*, extract the constant and push it
                if lines[else_store_idx].strip().startswith('STORE_CONST_'):
                    # Extract constant from STORE_CONST_I64/F64/etc var val
                    parts = lines[else_store_idx].strip().split()
                    if len(parts) >= 3:
                        const_type = parts[0].replace('STORE_CONST_', 'CONST_')
                        const_val = parts[2]
                        result.append(f"{indent}{const_type} {const_val}")
                else:
                    # Normal case: copy instructions before STORE
                    for idx in range(j + 1, else_store_idx):
                        result.append(lines[idx])
                
                # Add SELECT and STORE
                result.append(f"{indent}SELECT")
                result.append(f"{indent}STORE {if_store_var}")
                
                # Skip to after end label
                i = k + 1
                continue
            
            result.append(lines[i])
            i += 1
        
        return result



