"""
Unified Configuration Manager Implementation
===========================================

This module implements the unified configuration management service that consolidates
all configuration-related services using the strategy pattern. It replaces multiple
specialized configuration services with a single, extensible service.

Consolidates:
- ProjectConfigService
- AgentConfigService
- EnvironmentConfigService
- UserConfigService
- SystemConfigService
- And other configuration-related services

Features:
- Strategy-based configuration handling for different config types
- Configuration validation and schema enforcement
- Configuration merging with multiple strategies
- Hot-reload support for dynamic configuration updates
- Version control and rollback capabilities
"""

import json
from pathlib import Path
from typing import Any, Dict, List, Optional, Union

from claude_mpm.core.logging_utils import get_logger

from .interfaces import (
    ConfigurationResult,
    IConfigurationService,
    IUnifiedService,
    ServiceCapability,
    ServiceMetadata,
)
from .strategies import ConfigStrategy, StrategyContext, get_strategy_registry


class UnifiedConfigManager(IConfigurationService, IUnifiedService):
    """
    Unified configuration management service using strategy pattern.

    This service consolidates all configuration operations through a
    pluggable strategy system, providing consistent configuration
    management across different configuration types.
    """

    def __init__(self):
        """Initialize unified configuration manager."""
        self._logger = get_logger(f"{__name__}.UnifiedConfigManager")
        self._registry = get_strategy_registry()
        self._configs: Dict[str, Dict[str, Any]] = {}
        self._config_schemas: Dict[str, Dict[str, Any]] = {}
        self._config_versions: Dict[str, List[Dict[str, Any]]] = {}
        self._metrics = {
            "total_loads": 0,
            "total_saves": 0,
            "validation_errors": 0,
            "merge_operations": 0,
            "rollbacks": 0,
        }
        self._initialized = False
        self._hot_reload_enabled = False

    def get_metadata(self) -> ServiceMetadata:
        """
        Get service metadata.

        Returns:
            ServiceMetadata: Service metadata
        """
        return ServiceMetadata(
            name="UnifiedConfigManager",
            version="1.0.0",
            capabilities={
                ServiceCapability.ASYNC_OPERATIONS,
                ServiceCapability.VALIDATION,
                ServiceCapability.HOT_RELOAD,
                ServiceCapability.VERSIONING,
                ServiceCapability.ROLLBACK,
                ServiceCapability.METRICS,
                ServiceCapability.HEALTH_CHECK,
            },
            dependencies=["StrategyRegistry", "LoggingService"],
            description="Unified service for all configuration management",
            tags={"configuration", "unified", "strategy-pattern"},
            deprecated_services=[
                "ProjectConfigService",
                "AgentConfigService",
                "EnvironmentConfigService",
                "UserConfigService",
                "SystemConfigService",
            ],
        )

    async def initialize(self) -> bool:
        """
        Initialize the service.

        Returns:
            bool: True if initialization successful
        """
        try:
            self._logger.info("Initializing UnifiedConfigManager")

            # Register default strategies
            self._register_default_strategies()

            # Load default configurations
            await self._load_default_configs()

            # Initialize hot reload if enabled
            if self._hot_reload_enabled:
                await self._start_hot_reload()

            self._initialized = True
            self._logger.info("UnifiedConfigManager initialized successfully")
            return True

        except Exception as e:
            self._logger.error(f"Failed to initialize: {e!s}")
            return False

    async def shutdown(self) -> None:
        """Gracefully shutdown the service."""
        self._logger.info("Shutting down UnifiedConfigManager")

        # Stop hot reload if running
        if self._hot_reload_enabled:
            await self._stop_hot_reload()

        # Save current configurations
        await self._save_all_configs()

        # Clear configurations
        self._configs.clear()
        self._config_schemas.clear()
        self._config_versions.clear()

        self._initialized = False
        self._logger.info("UnifiedConfigManager shutdown complete")

    def health_check(self) -> Dict[str, Any]:
        """
        Perform health check.

        Returns:
            Dict[str, Any]: Health status
        """
        strategies = self._registry.list_strategies(ConfigStrategy)

        return {
            "service": "UnifiedConfigManager",
            "status": "healthy" if self._initialized else "unhealthy",
            "initialized": self._initialized,
            "registered_strategies": len(strategies),
            "loaded_configs": len(self._configs),
            "hot_reload_enabled": self._hot_reload_enabled,
            "metrics": self.get_metrics(),
        }

    def get_metrics(self) -> Dict[str, Any]:
        """
        Get service metrics.

        Returns:
            Dict[str, Any]: Service metrics
        """
        error_rate = 0.0
        total_ops = self._metrics["total_loads"] + self._metrics["total_saves"]
        if total_ops > 0:
            error_rate = (self._metrics["validation_errors"] / total_ops) * 100

        return {
            **self._metrics,
            "error_rate": error_rate,
            "loaded_configs": len(self._configs),
            "total_versions": sum(len(v) for v in self._config_versions.values()),
        }

    def reset(self) -> None:
        """Reset service to initial state."""
        self._logger.info("Resetting UnifiedConfigManager")
        self._configs.clear()
        self._config_schemas.clear()
        self._config_versions.clear()
        self._metrics = {
            "total_loads": 0,
            "total_saves": 0,
            "validation_errors": 0,
            "merge_operations": 0,
            "rollbacks": 0,
        }

    def load_config(
        self, source: Union[str, Path, Dict[str, Any]]
    ) -> ConfigurationResult:
        """
        Load configuration from source.

        Args:
            source: Configuration source

        Returns:
            ConfigurationResult: Loaded configuration
        """
        self._metrics["total_loads"] += 1

        try:
            # Determine config type
            config_type = self._determine_config_type(source)

            # Select configuration strategy
            context = StrategyContext(
                target_type=config_type,
                operation="load",
                parameters={"source": source},
            )

            strategy = self._registry.select_strategy(ConfigStrategy, context)

            if not strategy:
                self._metrics["validation_errors"] += 1
                return ConfigurationResult(
                    success=False,
                    validation_errors=[
                        f"No strategy available for config type: {config_type}"
                    ],
                )

            # Load configuration using strategy
            self._logger.info(
                f"Loading configuration from {source} using {strategy.metadata.name}"
            )

            config = strategy.load(source)

            # Validate configuration
            validation_errors = strategy.validate(config)
            if validation_errors:
                self._metrics["validation_errors"] += 1
                return ConfigurationResult(
                    success=False,
                    config=config,
                    validation_errors=validation_errors,
                    source=str(source),
                )

            # Apply defaults
            config_with_defaults = self._apply_strategy_defaults(strategy, config)

            # Store configuration
            config_id = self._generate_config_id(source, config_type)
            self._configs[config_id] = config_with_defaults

            # Store schema
            self._config_schemas[config_id] = strategy.get_schema()

            # Add to version history
            self._add_to_version_history(config_id, config_with_defaults)

            return ConfigurationResult(
                success=True,
                config=config_with_defaults,
                validation_errors=[],
                applied_defaults=self._get_applied_defaults(
                    config, config_with_defaults
                ),
                source=str(source),
            )

        except Exception as e:
            self._logger.error(f"Failed to load configuration: {e!s}")
            self._metrics["validation_errors"] += 1
            return ConfigurationResult(
                success=False,
                validation_errors=[f"Load failed: {e!s}"],
            )

    def save_config(
        self, config: Dict[str, Any], target: Union[str, Path]
    ) -> ConfigurationResult:
        """
        Save configuration to target.

        Args:
            config: Configuration to save
            target: Target location

        Returns:
            ConfigurationResult: Save result
        """
        self._metrics["total_saves"] += 1

        try:
            # Determine config type from target
            config_type = self._determine_config_type(target)

            # Select configuration strategy
            context = StrategyContext(
                target_type=config_type,
                operation="save",
                parameters={"target": target},
            )

            strategy = self._registry.select_strategy(ConfigStrategy, context)

            if not strategy:
                return ConfigurationResult(
                    success=False,
                    validation_errors=[
                        f"No strategy available for config type: {config_type}"
                    ],
                )

            # Validate configuration before saving
            validation_errors = strategy.validate(config)
            if validation_errors:
                self._metrics["validation_errors"] += 1
                return ConfigurationResult(
                    success=False,
                    config=config,
                    validation_errors=validation_errors,
                )

            # Transform configuration for saving
            transformed_config = strategy.transform(config)

            # Save configuration
            self._logger.info(f"Saving configuration to {target}")
            target_path = Path(target)
            target_path.parent.mkdir(parents=True, exist_ok=True)

            # Save based on file extension
            if target_path.suffix == ".json":
                with open(target_path, "w") as f:
                    json.dump(transformed_config, f, indent=2)
            elif target_path.suffix in [".yaml", ".yml"]:
                # Would use yaml library here
                with open(target_path, "w") as f:
                    f.write(str(transformed_config))
            else:
                with open(target_path, "w") as f:
                    f.write(str(transformed_config))

            return ConfigurationResult(
                success=True,
                config=transformed_config,
                validation_errors=[],
                source=str(target),
            )

        except Exception as e:
            self._logger.error(f"Failed to save configuration: {e!s}")
            return ConfigurationResult(
                success=False,
                validation_errors=[f"Save failed: {e!s}"],
            )

    def validate_config(self, config: Dict[str, Any]) -> List[str]:
        """
        Validate configuration.

        Args:
            config: Configuration to validate

        Returns:
            List[str]: Validation errors
        """
        errors = []

        # Basic validation
        if not config:
            errors.append("Configuration is empty")

        # Try to find matching strategy for validation
        config_type = config.get("type", "generic")
        context = StrategyContext(
            target_type=config_type,
            operation="validate",
        )

        strategy = self._registry.select_strategy(ConfigStrategy, context)
        if strategy:
            strategy_errors = strategy.validate(config)
            errors.extend(strategy_errors)
        else:
            errors.append(f"No validation strategy for type: {config_type}")

        return errors

    def merge_configs(
        self, *configs: Dict[str, Any], strategy: str = "deep"
    ) -> Dict[str, Any]:
        """
        Merge multiple configurations.

        Args:
            *configs: Configurations to merge
            strategy: Merge strategy

        Returns:
            Dict[str, Any]: Merged configuration
        """
        self._metrics["merge_operations"] += 1

        if not configs:
            return {}

        if len(configs) == 1:
            return configs[0].copy()

        # Implement merge based on strategy
        if strategy == "deep":
            return self._deep_merge(*configs)
        if strategy == "shallow":
            return self._shallow_merge(*configs)
        if strategy == "override":
            return self._override_merge(*configs)
        self._logger.warning(f"Unknown merge strategy: {strategy}, using deep")
        return self._deep_merge(*configs)

    def get_config_value(
        self, key: str, default: Any = None, config: Optional[Dict[str, Any]] = None
    ) -> Any:
        """
        Get configuration value by key.

        Args:
            key: Configuration key (dot notation)
            default: Default value
            config: Optional config dict

        Returns:
            Any: Configuration value
        """
        if config is None:
            # Use first loaded config as default
            if self._configs:
                config = next(iter(self._configs.values()))
            else:
                return default

        # Support dot notation
        keys = key.split(".")
        value = config

        for k in keys:
            if isinstance(value, dict):
                value = value.get(k)
                if value is None:
                    return default
            else:
                return default

        return value

    def set_config_value(
        self, key: str, value: Any, config: Optional[Dict[str, Any]] = None
    ) -> ConfigurationResult:
        """
        Set configuration value by key.

        Args:
            key: Configuration key (dot notation)
            value: Value to set
            config: Optional config dict

        Returns:
            ConfigurationResult: Result of set operation
        """
        if config is None:
            if not self._configs:
                return ConfigurationResult(
                    success=False,
                    validation_errors=["No configuration loaded"],
                )
            # Use first loaded config
            config_id = next(iter(self._configs.keys()))
            config = self._configs[config_id]

        # Support dot notation
        keys = key.split(".")
        current = config

        # Navigate to parent of target key
        for k in keys[:-1]:
            if k not in current:
                current[k] = {}
            current = current[k]

        # Set the value
        old_value = current.get(keys[-1])
        current[keys[-1]] = value

        # Validate updated configuration
        validation_errors = self.validate_config(config)

        if validation_errors:
            # Rollback change
            if old_value is None:
                del current[keys[-1]]
            else:
                current[keys[-1]] = old_value

            return ConfigurationResult(
                success=False,
                config=config,
                validation_errors=validation_errors,
            )

        return ConfigurationResult(
            success=True,
            config=config,
            validation_errors=[],
        )

    def get_schema(self) -> Dict[str, Any]:
        """
        Get configuration schema.

        Returns:
            Dict[str, Any]: Configuration schema
        """
        # Return first available schema or generic schema
        if self._config_schemas:
            return next(iter(self._config_schemas.values()))

        # Return generic schema
        return {
            "type": "object",
            "properties": {},
            "additionalProperties": True,
        }

    def apply_defaults(self, config: Dict[str, Any]) -> Dict[str, Any]:
        """
        Apply default values.

        Args:
            config: Configuration

        Returns:
            Dict[str, Any]: Configuration with defaults
        """
        # Determine config type
        config_type = config.get("type", "generic")

        # Get appropriate strategy
        context = StrategyContext(
            target_type=config_type,
            operation="defaults",
        )

        strategy = self._registry.select_strategy(ConfigStrategy, context)
        if strategy:
            return self._apply_strategy_defaults(strategy, config)

        return config

    # Private helper methods

    def _register_default_strategies(self) -> None:
        """Register default configuration strategies."""
        # Default strategies would be registered here
        self._logger.debug("Default strategies registered")

    async def _load_default_configs(self) -> None:
        """Load default configurations."""
        # Implementation would load default configs
        self._logger.debug("Default configurations loaded")

    async def _save_all_configs(self) -> None:
        """Save all loaded configurations."""
        # Implementation would save all configs
        self._logger.debug("All configurations saved")

    async def _start_hot_reload(self) -> None:
        """Start hot reload monitoring."""
        self._logger.info("Hot reload started")

    async def _stop_hot_reload(self) -> None:
        """Stop hot reload monitoring."""
        self._logger.info("Hot reload stopped")

    def _determine_config_type(self, source: Any) -> str:
        """Determine configuration type from source."""
        if isinstance(source, dict):
            return source.get("type", "generic")

        if isinstance(source, (str, Path)):
            path = Path(source)
            # Infer from filename
            if "project" in path.name.lower():
                return "project"
            if "agent" in path.name.lower():
                return "agent"
            if "env" in path.name.lower():
                return "environment"

        return "generic"

    def _generate_config_id(self, source: Any, config_type: str) -> str:
        """Generate configuration ID."""
        import hashlib

        source_str = str(source)
        return hashlib.md5(f"{config_type}:{source_str}".encode()).hexdigest()[:8]

    def _apply_strategy_defaults(
        self, strategy: ConfigStrategy, config: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Apply defaults using strategy."""
        schema = strategy.get_schema()
        return self._apply_schema_defaults(config, schema)

    def _apply_schema_defaults(
        self, config: Dict[str, Any], schema: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Apply defaults from schema."""
        result = config.copy()

        if "properties" in schema:
            for key, prop_schema in schema["properties"].items():
                if key not in result and "default" in prop_schema:
                    result[key] = prop_schema["default"]

        return result

    def _get_applied_defaults(
        self, original: Dict[str, Any], with_defaults: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Get dictionary of applied defaults."""
        applied = {}
        for key, value in with_defaults.items():
            if key not in original:
                applied[key] = value
        return applied

    def _add_to_version_history(self, config_id: str, config: Dict[str, Any]) -> None:
        """Add configuration to version history."""
        if config_id not in self._config_versions:
            self._config_versions[config_id] = []

        version_entry = {
            "timestamp": self._get_timestamp(),
            "config": config.copy(),
        }

        self._config_versions[config_id].append(version_entry)

        # Keep only last 10 versions
        if len(self._config_versions[config_id]) > 10:
            self._config_versions[config_id] = self._config_versions[config_id][-10:]

    def _deep_merge(self, *configs: Dict[str, Any]) -> Dict[str, Any]:
        """Deep merge configurations."""
        result = {}

        for config in configs:
            for key, value in config.items():
                if (
                    key in result
                    and isinstance(result[key], dict)
                    and isinstance(value, dict)
                ):
                    result[key] = self._deep_merge(result[key], value)
                else:
                    result[key] = value

        return result

    def _shallow_merge(self, *configs: Dict[str, Any]) -> Dict[str, Any]:
        """Shallow merge configurations."""
        result = {}
        for config in configs:
            result.update(config)
        return result

    def _override_merge(self, *configs: Dict[str, Any]) -> Dict[str, Any]:
        """Override merge - last config wins."""
        return configs[-1].copy() if configs else {}

    def _get_timestamp(self) -> str:
        """Get current timestamp."""
        from datetime import datetime

        return datetime.now().isoformat()
